/**
 * Copyright (C) 2003-2023, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdf.function.json;


import com.foxit.sdk.PDFException;
import com.foxit.sdk.common.DateTime;
import com.foxit.sdk.common.fxcrt.RectF;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.BorderInfo;
import com.foxit.sdk.pdf.annots.Markup;
import com.foxit.sdk.pdf.objects.PDFArray;
import com.foxit.sdk.pdf.objects.PDFDictionary;
import com.foxit.sdk.pdf.objects.PDFObject;
import com.foxit.sdk.pdf.objects.PDFStream;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;

class JSAnnotUtil {

    public static String type2String(Annot annot) {
        try {
            int type = annot.getType();
            switch (type) {
                case Annot.e_Note:
                    return "text";
                case Annot.e_Link:
                    return "link";
                case Annot.e_FreeText:
                    return "freetext";
                case Annot.e_Line:
                    return "line";
                case Annot.e_Square:
                    return "square";
                case Annot.e_Circle:
                    return "circle";
                case Annot.e_Polygon:
                    return "polygon";
                case Annot.e_PolyLine:
                    return "polyline";
                case Annot.e_Highlight:
                    return "highlight";
                case Annot.e_Underline:
                    return "underline";
                case Annot.e_Squiggly:
                    return "squiggly";
                case Annot.e_StrikeOut:
                    return "strikeout";
                case Annot.e_Stamp:
                    return "stamp";
                case Annot.e_Caret:
                    return "caret";
                case Annot.e_Ink:
                    return "ink";
                case Annot.e_PSInk:
                    return "psink";
                case Annot.e_FileAttachment:
                    return "fileattachment";
                case Annot.e_Sound:
                    return "sound";
                case Annot.e_Movie:
                    return "movie";
                case Annot.e_Widget:
                    return "widget";
                case Annot.e_Screen:
                    return "screen";
                case Annot.e_PrinterMark:
                    return "printermark";
                case Annot.e_TrapNet:
                    return "trapnet";
                case Annot.e_Watermark:
                    return "watermark";
                case Annot.e_3D:
                    return "3D";
                case Annot.e_Popup:
                    return "popup";
                case Annot.e_Redact:
                    return "redact";
                case Annot.e_RichMedia:
                    return "richmedia";
                default:
                    return "unknown";
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return "unknown";
    }

    public static String flags2String(Annot annot) {
        try {
            int flags = annot.getFlags();
            StringBuilder builder = new StringBuilder();
            if ((flags & Annot.e_FlagInvisible) == Annot.e_FlagInvisible) {
                builder.append("invisible");
                builder.append(",");
            }
            if ((flags & Annot.e_FlagHidden) == Annot.e_FlagHidden) {
                builder.append("hidden");
                builder.append(",");
            }
            if ((flags & Annot.e_FlagPrint) == Annot.e_FlagPrint) {
                builder.append("print");
                builder.append(",");
            }
            if ((flags & Annot.e_FlagNoZoom) == Annot.e_FlagNoZoom) {
                builder.append("nozoom");
                builder.append(",");
            }
            if ((flags & Annot.e_FlagNoRotate) == Annot.e_FlagNoRotate) {
                builder.append("norotate");
                builder.append(",");
            }
            if ((flags & Annot.e_FlagNoView) == Annot.e_FlagNoView) {
                builder.append("noview");
                builder.append(",");
            }
            if ((flags & Annot.e_FlagReadOnly) == Annot.e_FlagReadOnly) {
                builder.append("readonly");
                builder.append(",");
            }
            if ((flags & Annot.e_FlagLocked) == Annot.e_FlagLocked) {
                builder.append("locked");
                builder.append(",");
            }
            if ((flags & Annot.e_FlagToggleNoView) == Annot.e_FlagToggleNoView) {
                builder.append("togglenoview");
                builder.append(",");
            }
            if ((flags & Annot.e_FlagLockedContents) == Annot.e_FlagLockedContents) {
                builder.append("lockedcontents");
                builder.append(",");
            }

            String flagsStr = builder.toString();
            if (flagsStr.contains(",") && flagsStr.split(",").length == 1)
                return flagsStr.substring(0, flagsStr.length() - 1);
            else
                return flagsStr;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return "";
    }

    public static String borderStyle2String(Annot annot) {
        try {
            int style = annot.getBorderInfo().getStyle();
            switch (style) {
                case BorderInfo.e_Solid:
                    return "solid";
                case BorderInfo.e_Dashed:
                    return "dashed";
                case BorderInfo.e_UnderLine:
                    return "underline";
                case BorderInfo.e_Beveled:
                    return "beveled";
                case BorderInfo.e_Inset:
                    return "inset";
                case BorderInfo.e_Cloudy:
                    return "cloudy";
                default:
                    return "none";
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return "none";
    }

    public static String makeupEndingStyle(int style) {
        switch (style) {
            case Markup.e_EndingStyleNone:
                return "None";
            case Markup.e_EndingStyleSquare:
                return "Square";
            case Markup.e_EndingStyleCircle:
                return "Circle";
            case Markup.e_EndingStyleDiamond:
                return "Diamond";
            case Markup.e_EndingStyleOpenArrow:
                return "OpenArrow";
            case Markup.e_EndingStyleClosedArrow:
                return "ClosedArrow";
            case Markup.e_EndingStyleButt:
                return "Butt";
            case Markup.e_EndingStyleROpenArrow:
                return "ROpenArrow";
            case Markup.e_EndingStyleRClosedArrow:
                return "RClosedArrow";
            case Markup.e_EndingStyleSlash:
                return "Slash";
            default:
                return "None";
        }
    }

    public static String rect2String(Annot annot) {
        try {
            RectF rectF = annot.getRect();
            return rectF.getLeft() +
                    "," +
                    rectF.getBottom() +
                    "," +
                    rectF.getRight() +
                    "," +
                    rectF.getTop();
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return "";
    }

    public static String colorConvertor(int color) {
        int r = (color & 0xff0000) >> 16;
        int g = (color & 0xff00) >> 8;
        int b = color & 0xff;

        StringBuilder sb = new StringBuilder();
        String R = Integer.toHexString(r);
        String G = Integer.toHexString(g);
        String B = Integer.toHexString(b);
        R = R.length() == 1 ? "0" + R : R;
        G = G.length() == 1 ? "0" + G : G;
        B = B.length() == 1 ? "0" + B : B;
        sb.append("#");
        sb.append(R);
        sb.append(G);
        sb.append(B);
        return sb.toString();
    }

    public static String convertFromNumberToHex(int color) {
        String hexCode = String.format("#%06X", 0xffffff & color);
        return hexCode;
    }

    public static int[] convertFromNumberToRGBA(int color) {
        int a = color >>> 24;
        int r = (color & 0xff0000) >> 16;
        int g = (color & 0xff00) >> 8;
        int b = color & 0xff;
        return new int[]{r, g, b, a};
    }

    private static final int MICROSECONDS_PER_MINUTE = 60000;
    private static final int MICROSECONDS_PER_HOUR = 3600000;

    public static String formatDocumentDate(String pattern, DateTime dateTime) {
        if (isZero(dateTime)) {
            return "0";
        }
        Date date = documentDateToJavaDate(dateTime);
        DateFormat dateFormat = new SimpleDateFormat(pattern, Locale.getDefault());
        return dateFormat.format(date);
    }

    public static String formatDocumentDate(DateTime dateTime) {
        if (isZero(dateTime)) {
            return "D:19700101080000+08'00'";
        }
        return String.format(Locale.getDefault(),"D:%d%02d%02d%02d%02d%02d+%02d'%02d'",
                dateTime.getYear(),
                dateTime.getMonth(),
                dateTime.getDay(),
                dateTime.getHour(),
                dateTime.getMinute(),
                dateTime.getSecond(),
                dateTime.getUtc_hour_offset(),
                dateTime.getUtc_minute_offset());
    }



    public static boolean isZero(DateTime dateTime) {
        return dateTime.getYear() == 0 &&
                dateTime.getMonth() == 0 &&
                dateTime.getDay() == 0 &&
                dateTime.getHour() == 0 &&
                dateTime.getMinute() == 0 &&
                dateTime.getSecond() == 0 &&
                dateTime.getUtc_hour_offset() == 0 &&
                dateTime.getUtc_minute_offset() == 0;
    }

    public static Date documentDateToJavaDate(final DateTime dateTime) {
        if (dateTime == null)
            return null;
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.YEAR, dateTime.getYear());
        calendar.set(Calendar.MONTH, dateTime.getMonth() - 1);
        calendar.set(Calendar.DAY_OF_MONTH, dateTime.getDay());
        calendar.set(Calendar.HOUR_OF_DAY, dateTime.getHour());
        calendar.set(Calendar.MINUTE, dateTime.getMinute());
        calendar.set(Calendar.SECOND, dateTime.getSecond());
        int rawOffset = dateTime.getUtc_minute_offset() * MICROSECONDS_PER_MINUTE + dateTime.getUtc_hour_offset() * MICROSECONDS_PER_HOUR - TimeZone.getDefault().getRawOffset();
        return new Date(calendar.getTimeInMillis() - rawOffset);
    }

    public static Object retriveDict(JSONObject json, PDFObject root) {
        try {
            JSONObject jsonObject = new JSONObject();
            PDFDictionary dict = null;
            char[] hexCode = {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};
            switch (root.getType()) {
                case 1: // boolean:
                    return root.getWideString();
                case 2:// number;
                    if (Math.abs(root.getInteger() - root.getFloat()) < 0.001) {
                        return root.getInteger();
                    }
                    return root.getFloat();
                case 3: // string:
                    byte[] buffer = root.getString();
                    StringBuilder sb = new StringBuilder();
                    sb.append("<");
                    for (byte ch : buffer) {
                        sb.append(hexCode[(ch & 0xf0) >> 4]);
                        sb.append(hexCode[ch & 0x0f]);
                    }
                    sb.append(">");
                    return sb.toString();
                case 4: // name:c
                    return "/" + root.getName();
                case 5: { // array:
                    JSONArray jsonArray = new JSONArray();
                    PDFArray pdfArr = root.getArray();
                    int size = pdfArr.getElementCount();
                    for (int i = 0; i < size; i++) {
                        jsonArray.put(retriveDict(json, pdfArr.getElement(i)));
                    }
                    return jsonArray;
                }
                case 7: { // e_Stream
                    PDFStream stream = root.getStream();
                    dict = stream.getDictionary();

                    int size = stream.getDataSize(true);
                    if (size > 0) {
                        byte[] temp = new byte[size];
                        stream.getData(true, size, temp);
                        StringBuilder rawData = new StringBuilder();
                        for (byte ch : temp) {
                            rawData.append(hexCode[(ch & 0xf0) >> 4]);
                            rawData.append(hexCode[ch & 0x0f]);
                        }
                        jsonObject.put("stream", rawData.toString());
                    }
                }
                case 6: {// dict:
                    if (dict == null) dict = root.getDict();
                    if (dict == null) return jsonObject;

                    PDFObject type = dict.getElement("Type");
                    if (type != null && type.getWideString().equals("OCG")) {
                        return new JSONObject();
                    }

                    long pos = dict.moveNext(0);
                    while (pos != 0) {
                        String key2 = dict.getKey(pos);

                        PDFObject obj = dict.getElement(key2);
                        jsonObject.put(key2, retriveDict(json, obj));
                        pos = dict.moveNext(pos);
                    }
                    return jsonObject;
                }
                case 8:
                    return null;
                case 9: {// reference
                    int objNumber = root.getDirectObject().getObjNum();
                    String objNote = objNumber + "R";
                    json.put(objNumber + "", retriveDict(json, root.getDirectObject()));
                    return objNote;
                }
            }
        } catch (JSONException | PDFException e) {
            e.printStackTrace();
        }
        return null;
    }

}
