/**
 * Copyright (C) 2003-2023, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdf.function.annotation;

import android.content.Context;
import android.widget.Toast;

import com.foxit.pdf.function.Common;
import com.foxit.pdf.function.json.JSAnnot;
import com.foxit.pdf.function.json.JSCaret;
import com.foxit.pdf.function.json.JSCircle;
import com.foxit.pdf.function.json.JSFileAttachment;
import com.foxit.pdf.function.json.JSFreeText;
import com.foxit.pdf.function.json.JSInk;
import com.foxit.pdf.function.json.JSLine;
import com.foxit.pdf.function.json.JSMarkupAnnot;
import com.foxit.pdf.function.json.JSNote;
import com.foxit.pdf.function.json.JSPolyLine;
import com.foxit.pdf.function.json.JSPolygon;
import com.foxit.pdf.function.json.JSRedact;
import com.foxit.pdf.function.json.JSSound;
import com.foxit.pdf.function.json.JSSquare;
import com.foxit.pdf.function.json.JSStamp;
import com.foxit.pdf.function.json.JSTextMarkupAnnot;
import com.foxit.pdf.function_demo.R;
import com.foxit.sdk.PDFException;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.common.Font;
import com.foxit.sdk.common.Library;
import com.foxit.sdk.common.Path;
import com.foxit.sdk.common.fxcrt.FloatArray;
import com.foxit.sdk.common.fxcrt.PointF;
import com.foxit.sdk.common.fxcrt.PointFArray;
import com.foxit.sdk.common.fxcrt.RectF;
import com.foxit.sdk.pdf.FileSpec;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.actions.Action;
import com.foxit.sdk.pdf.actions.URIAction;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.BorderInfo;
import com.foxit.sdk.pdf.annots.Caret;
import com.foxit.sdk.pdf.annots.Circle;
import com.foxit.sdk.pdf.annots.DefaultAppearance;
import com.foxit.sdk.pdf.annots.FileAttachment;
import com.foxit.sdk.pdf.annots.FreeText;
import com.foxit.sdk.pdf.annots.Highlight;
import com.foxit.sdk.pdf.annots.Ink;
import com.foxit.sdk.pdf.annots.Line;
import com.foxit.sdk.pdf.annots.Link;
import com.foxit.sdk.pdf.annots.Markup;
import com.foxit.sdk.pdf.annots.Note;
import com.foxit.sdk.pdf.annots.PolyLine;
import com.foxit.sdk.pdf.annots.Polygon;
import com.foxit.sdk.pdf.annots.Popup;
import com.foxit.sdk.pdf.annots.QuadPoints;
import com.foxit.sdk.pdf.annots.QuadPointsArray;
import com.foxit.sdk.pdf.annots.Square;
import com.foxit.sdk.pdf.annots.Squiggly;
import com.foxit.sdk.pdf.annots.Stamp;
import com.foxit.sdk.pdf.annots.StrikeOut;
import com.foxit.sdk.pdf.annots.Underline;

import org.json.JSONArray;
import org.json.JSONObject;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;

public class Annotation {
    private final Context mContext;

    private final String inputFile;
    private final String outputFile;
    private final String outputJsonFile;

    public Annotation(Context context) {
        mContext = context;
        inputFile = Common.getFixFolder() + "annotation_input.pdf";
        outputFile = Common.getOutputFilesFolder(Common.ANNOTATION) + "annotation_add.pdf";
        outputJsonFile = Common.getOutputFilesFolder(Common.ANNOTATION) + "annotation_add.json";
    }

    public void addAnnotation() {
        PDFDoc doc = Common.loadPDFDoc(mContext, inputFile, null);
        if (doc == null) return;
        addAnnotation(doc);
        Common.saveDFDoc(mContext, doc, outputFile);
        exportToJSON(doc);
    }

    private void addAnnotation(PDFDoc doc) {
        try {
            PDFPage page = doc.getPage(0);
            // Add line annotation 
            // No special intent, as a common line.
            Annot annot = page.addAnnot(Annot.e_Line, new RectF(0, 650, 100, 750));
            Line line = new Line(annot);
            line.setStartPoint(new PointF(20, 650));
            line.setEndPoint(new PointF(100, 740));
            // Intent, as line arrow.
            line.setContent("A line arrow annotation");
            line.setIntent("LineArrow");
            line.setSubject("Arrow");
            line.setTitle("Foxit SDK");
            line.setCreationDateTime(Common.getCurrentDateTime());
            line.setModifiedDateTime(Common.getCurrentDateTime());
            line.setUniqueID(Common.randomUUID(null));
            // Appearance should be reset.
            line.resetAppearanceStream();

            line = new Line(page.addAnnot(Annot.e_Line, new RectF(0, 650, 100, 760)));
            // Set foxit RGB color
            line.setBorderColor(0x00FF00);
            line.setStartPoint(new PointF(10, 650));
            line.setEndPoint(new PointF(100, 750));
            line.setContent("A common line.");
            line.setLineStartStyle(Markup.e_EndingStyleSquare);
            line.setLineEndStyle(Markup.e_EndingStyleOpenArrow);
            // Show text in line
            line.enableCaption(true);
            line.setCaptionOffset(new PointF(0, 5));

            line.setSubject("Line");
            line.setTitle("Foxit SDK");
            line.setCreationDateTime(Common.getCurrentDateTime());
            line.setModifiedDateTime(Common.getCurrentDateTime());
            line.setUniqueID(Common.randomUUID(null));
            // Appearance should be reset.
            line.resetAppearanceStream();

            // Add circle annotation
            annot = page.addAnnot(Annot.e_Circle, new RectF(100, 650, 200, 750));
            Circle circle = new Circle(annot);
            circle.setInnerRect(new RectF(120, 660, 160, 740));
            circle.setSubject("Circle");
            circle.setTitle("Foxit SDK");
            circle.setCreationDateTime(Common.getCurrentDateTime());
            circle.setModifiedDateTime(Common.getCurrentDateTime());
            circle.setUniqueID(Common.randomUUID(null));
            // Appearance should be reset.
            circle.resetAppearanceStream();

            // Add square annotation
            annot = page.addAnnot(Annot.e_Square, new RectF(200, 650, 300, 750));
            Square square = new Square(annot);
            square.setFillColor(0x00FF00);
            square.setInnerRect(new RectF(220, 660, 260, 740));
            square.setSubject("Square");
            square.setTitle("Foxit SDK");
            square.setCreationDateTime(Common.getCurrentDateTime());
            square.setModifiedDateTime(Common.getCurrentDateTime());
            square.setUniqueID(Common.randomUUID(null));
            // Appearance should be reset.
            square.resetAppearanceStream();

            // Add polygon annotation, as cloud.
            annot = page.addAnnot(Annot.e_Polygon, new RectF(300, 650, 500, 750));
            Polygon polygon = new Polygon(annot);
            polygon.setIntent("PolygonCloud");
            polygon.setFillColor(0x0000FF);
            PointFArray vertexe_array = new PointFArray();
            vertexe_array.add(new PointF(335, 665));
            vertexe_array.add(new PointF(365, 665));
            vertexe_array.add(new PointF(385, 705));
            vertexe_array.add(new PointF(365, 740));
            vertexe_array.add(new PointF(335, 740));
            vertexe_array.add(new PointF(315, 705));
            polygon.setVertexes(vertexe_array);
            polygon.setSubject("Cloud");
            polygon.setTitle("Foxit SDK");
            polygon.setCreationDateTime(Common.getCurrentDateTime());
            polygon.setModifiedDateTime(Common.getCurrentDateTime());
            polygon.setUniqueID(Common.randomUUID(null));
            // Appearance should be reset.
            polygon.resetAppearanceStream();

            // Add polygon annotation, with dashed border.
            BorderInfo borderinfo = new BorderInfo();
            borderinfo.setCloud_intensity(2.0f);
            borderinfo.setWidth(2.0f);
            borderinfo.setStyle(BorderInfo.e_Dashed);
            borderinfo.setDash_phase(3.0f);

            FloatArray floatArray = new FloatArray();
            floatArray.add(2.0f);
            floatArray.add(2.0f);
            borderinfo.setDashes(floatArray);

            annot = page.addAnnot(Annot.e_Polygon, new RectF(400, 650, 500, 750));
            polygon = new Polygon(annot);
            polygon.setFillColor(0x0000FF);
            polygon.setBorderInfo(borderinfo);
            vertexe_array = new PointFArray();
            vertexe_array.add(new PointF(435, 665));
            vertexe_array.add(new PointF(465, 665));
            vertexe_array.add(new PointF(485, 705));
            vertexe_array.add(new PointF(465, 740));
            vertexe_array.add(new PointF(435, 740));
            vertexe_array.add(new PointF(415, 705));

            polygon.setVertexes(vertexe_array);
            polygon.setSubject("Polygon");
            polygon.setTitle("Foxit SDK");
            polygon.setCreationDateTime(Common.getCurrentDateTime());
            polygon.setModifiedDateTime(Common.getCurrentDateTime());
            polygon.setUniqueID(Common.randomUUID(null));
            // Appearance should be reset.
            polygon.resetAppearanceStream();

            // Add polyline annotation 
            annot = page.addAnnot(Annot.e_PolyLine, new RectF(500, 650, 600, 700));
            PolyLine polyline = new PolyLine(annot);
            vertexe_array = new PointFArray();
            vertexe_array.add(new PointF(515, 705));
            vertexe_array.add(new PointF(535, 740));
            vertexe_array.add(new PointF(565, 740));
            vertexe_array.add(new PointF(585, 705));
            vertexe_array.add(new PointF(565, 665));
            vertexe_array.add(new PointF(535, 665));
            polyline.setVertexes(vertexe_array);
            polyline.setSubject("PolyLine");
            polyline.setTitle("Foxit SDK");
            polyline.setCreationDateTime(Common.getCurrentDateTime());
            polyline.setModifiedDateTime(Common.getCurrentDateTime());
            polyline.setUniqueID(Common.randomUUID(null));
            // Appearance should be reset.
            polyline.resetAppearanceStream();

            // Add freetext annotation, as type writer
            annot = page.addAnnot(Annot.e_FreeText, new RectF(10, 550, 200, 600));
            FreeText freetext = new FreeText(annot);
            // Set default appearance
            DefaultAppearance default_ap = new DefaultAppearance();
            default_ap.setFlags(
                    DefaultAppearance.e_FlagFont | DefaultAppearance.e_FlagFontSize | DefaultAppearance.e_FlagTextColor);
            default_ap.setFont(new Font(Font.e_StdIDHelvetica));
            default_ap.setText_size(12.0f);
            default_ap.setText_color(0x000000);
            // Set default appearance for form.
            freetext.setDefaultAppearance(default_ap);
            freetext.setAlignment(Constants.e_AlignmentLeft);
            freetext.setIntent("FreeTextTypewriter");
            freetext.setContent("A typewriter annotation");
            freetext.setSubject("FreeTextTypewriter");
            freetext.setTitle("Foxit SDK");
            freetext.setCreationDateTime(Common.getCurrentDateTime());
            freetext.setModifiedDateTime(Common.getCurrentDateTime());
            freetext.setUniqueID(Common.randomUUID(null));
            // Appearance should be reset.
            freetext.resetAppearanceStream();

            // Add freetext annotation, as call-out
            annot = page.addAnnot(Annot.e_FreeText, new RectF(300, 550, 400, 600));
            freetext = new FreeText(annot);
            // Set default appearance
            default_ap = new DefaultAppearance();
            default_ap.setFlags(
                    DefaultAppearance.e_FlagFont | DefaultAppearance.e_FlagFontSize | DefaultAppearance.e_FlagTextColor);
            default_ap.setFont(new Font(Font.e_StdIDHelveticaB));
            default_ap.setText_size(12.0f);
            default_ap.setText_color(0x000000);
            // Set default appearance for form.
            freetext.setDefaultAppearance(default_ap);
            freetext.setAlignment(Constants.e_AlignmentCenter);
            freetext.setIntent("FreeTextCallout");
            PointFArray callout_points = new PointFArray();
            callout_points.add(new PointF(250, 540));
            callout_points.add(new PointF(280, 570));
            callout_points.add(new PointF(300, 570));

            freetext.setCalloutLinePoints(callout_points);
            freetext.setCalloutLineEndingStyle(Markup.e_EndingStyleOpenArrow);
            freetext.setContent("A callout annotation.");
            freetext.setSubject("FreeTextCallout");
            freetext.setTitle("Foxit SDK");
            freetext.setCreationDateTime(Common.getCurrentDateTime());
            freetext.setModifiedDateTime(Common.getCurrentDateTime());
            freetext.setUniqueID(Common.randomUUID(null));
            // Appearance should be reset.
            freetext.resetAppearanceStream();

            // Add freetext annotation, as text box
            annot = page.addAnnot(Annot.e_FreeText, new RectF(450, 550, 550, 600));
            freetext = new FreeText(annot);
            // Set default appearance
            default_ap = new DefaultAppearance();
            default_ap.setFlags(
                    DefaultAppearance.e_FlagFont | DefaultAppearance.e_FlagFontSize | DefaultAppearance.e_FlagTextColor);
            default_ap.setFont(new Font(Font.e_StdIDHelveticaI));
            default_ap.setText_size(12.0f);
            default_ap.setText_color(0x000000);
            // Set default appearance for form.
            freetext.setDefaultAppearance(default_ap);
            freetext.setAlignment(Constants.e_AlignmentCenter);
            freetext.setContent("A text box annotation.");
            freetext.setSubject("Textbox");
            freetext.setTitle("Foxit SDK");
            freetext.setCreationDateTime(Common.getCurrentDateTime());
            freetext.setModifiedDateTime(Common.getCurrentDateTime());
            freetext.setUniqueID(Common.randomUUID(null));
            // Appearance should be reset.
            freetext.resetAppearanceStream();

            // Add highlight annotation
            Highlight highlight = new Highlight(page.addAnnot(Annot.e_Highlight, new RectF(10, 450, 100, 550)));
            highlight.setContent("Highlight");
            QuadPoints quad_points = new QuadPoints();
            quad_points.setFirst(new PointF(10, 500));
            quad_points.setSecond(new PointF(90, 500));
            quad_points.setThird(new PointF(10, 480));
            quad_points.setFourth(new PointF(90, 480));
            QuadPointsArray quad_points_array = new QuadPointsArray();
            quad_points_array.add(quad_points);
            highlight.setQuadPoints(quad_points_array);
            highlight.setSubject("Highlight");
            highlight.setTitle("Foxit SDK");
            highlight.setCreationDateTime(Common.getCurrentDateTime());
            highlight.setModifiedDateTime(Common.getCurrentDateTime());
            highlight.setUniqueID(Common.randomUUID(null));
            // Appearance should be reset.
            highlight.resetAppearanceStream();

            // Add underline annotation
            Underline underline = new Underline(page.addAnnot(Annot.e_Underline, new RectF(100, 450, 200, 550)));
            quad_points = new QuadPoints();
            quad_points.setFirst(new PointF(110, 500));
            quad_points.setSecond(new PointF(190, 500));
            quad_points.setThird(new PointF(110, 480));
            quad_points.setFourth(new PointF(190, 480));
            quad_points_array = new QuadPointsArray();
            quad_points_array.add(quad_points);
            underline.setQuadPoints(quad_points_array);
            underline.setSubject("Underline");
            underline.setTitle("Foxit SDK");
            underline.setCreationDateTime(Common.getCurrentDateTime());
            underline.setModifiedDateTime(Common.getCurrentDateTime());
            underline.setUniqueID(Common.randomUUID(null));
            // Appearance should be reset.
            underline.resetAppearanceStream();

            // Add squiggly annotation
            Squiggly squiggly = new Squiggly(page.addAnnot(Annot.e_Squiggly, new RectF(200, 450, 300, 550)));
            squiggly.setIntent("Squiggly");

            quad_points = new QuadPoints();
            quad_points.setFirst(new PointF(210, 500));
            quad_points.setSecond(new PointF(290, 500));
            quad_points.setThird(new PointF(210, 480));
            quad_points.setFourth(new PointF(290, 480));
            quad_points_array = new QuadPointsArray();
            quad_points_array.add(quad_points);
            squiggly.setQuadPoints(quad_points_array);
            squiggly.setSubject("Squiggly");
            squiggly.setTitle("Foxit SDK");
            squiggly.setCreationDateTime(Common.getCurrentDateTime());
            squiggly.setModifiedDateTime(Common.getCurrentDateTime());
            squiggly.setUniqueID(Common.randomUUID(null));
            // Appearance should be reset.
            squiggly.resetAppearanceStream();

            // Add strikeout annotation
            StrikeOut strikeout = new StrikeOut(page.addAnnot(Annot.e_StrikeOut, new RectF(300, 450, 400, 550)));
            quad_points = new QuadPoints();
            quad_points.setFirst(new PointF(310, 500));
            quad_points.setSecond(new PointF(390, 500));
            quad_points.setThird(new PointF(310, 480));
            quad_points.setFourth(new PointF(390, 480));
            quad_points_array = new QuadPointsArray();
            quad_points_array.add(quad_points);
            strikeout.setQuadPoints(quad_points_array);
            strikeout.setSubject("StrikeOut");
            strikeout.setTitle("Foxit SDK");
            strikeout.setCreationDateTime(Common.getCurrentDateTime());
            strikeout.setModifiedDateTime(Common.getCurrentDateTime());
            strikeout.setUniqueID(Common.randomUUID(null));
            // Appearance should be reset.
            strikeout.resetAppearanceStream();

            // Add caret annotation
            Caret caret = new Caret(page.addAnnot(Annot.e_Caret, new RectF(400, 450, 420, 470)));
            caret.setInnerRect(new RectF(410, 450, 430, 470));
            caret.setContent("Caret annotation");
            caret.setSubject("Caret");
            caret.setTitle("Foxit SDK");
            caret.setCreationDateTime(Common.getCurrentDateTime());
            caret.setModifiedDateTime(Common.getCurrentDateTime());
            caret.setUniqueID(Common.randomUUID(null));
            // Appearance should be reset.
            caret.resetAppearanceStream();

            // Add note annotation
            Note note = new Note(page.addAnnot(Annot.e_Note, new RectF(10, 350, 50, 400)));
            note.setIconName("Comment");
            note.setSubject("Note");
            note.setTitle("Foxit SDK");
            note.setContent("Note annotation.");
            note.setCreationDateTime(Common.getCurrentDateTime());
            note.setModifiedDateTime(Common.getCurrentDateTime());
            note.setUniqueID(Common.randomUUID(null));
            // Add popup to note annotation
            Popup popup = new Popup(page.addAnnot(Annot.e_Popup, new RectF(300, 450, 500, 550)));
            popup.setBorderColor(0x00FF00);
            popup.setOpenStatus(false);
            popup.setModifiedDateTime(Common.getCurrentDateTime());
            note.setPopup(popup);

            // Add reply annotation to note annotation
            Note reply = note.addReply();
            reply.setContent("reply");
            reply.setModifiedDateTime(Common.getCurrentDateTime());
            reply.setTitle("Foxit SDK");
            reply.setUniqueID(Common.randomUUID(null));
            reply.resetAppearanceStream();

            // Add state annotation to note annotation
            Note state = new Note(note.addStateAnnot("Foxit SDK", Markup.e_StateModelReview, Markup.e_StateAccepted));
            state.setContent("Accepted set by Foxit SDK");
            state.setUniqueID(Common.randomUUID(null));
            state.resetAppearanceStream();
            // Appearance should be reset.
            note.resetAppearanceStream();

            // Add ink annotation
            Ink ink = new Ink(page.addAnnot(Annot.e_Ink, new RectF(100, 350, 200, 450)));
            Path inklist = new Path();
            float width = 100;
            float height = 100;
            float out_width = Math.min(width, height) * 2 / 3.f;
            float inner_width = (float) (out_width * (Math.sin(18.f / 180.f * 3.14f)) / (Math.sin(36.f / 180.f * 3.14f)));
            PointF center = new PointF(150, 400);
            float x = out_width;
            float y = 0;
            inklist.moveTo(new PointF(center.getX() + x, center.getY() + y));
            for (int i = 0; i < 5; i++) {
                x = (float) (out_width * Math.cos(72.f * i / 180.f * 3.14f));
                y = (float) (out_width * Math.sin(72.f * i / 180.f * 3.14f));
                inklist.lineTo(new PointF(center.getX() + x, center.getY() + y));

                x = (float) (inner_width * Math.cos((72.f * i + 36) / 180.f * 3.14f));
                y = (float) (inner_width * Math.sin((72.f * i + 36) / 180.f * 3.14f));
                inklist.lineTo(new PointF(center.getX() + x, center.getY() + y));
            }
            inklist.lineTo(new PointF(center.getX() + out_width, center.getY() + 0));
            inklist.closeFigure();
            ink.setInkList(inklist);
            ink.setSubject("Ink");
            ink.setTitle("Foxit SDK");
            ink.setContent("Note annotation.");
            ink.setCreationDateTime(Common.getCurrentDateTime());
            ink.setModifiedDateTime(Common.getCurrentDateTime());
            ink.setUniqueID(Common.randomUUID(null));
            // Appearance should be reset.
            ink.resetAppearanceStream();

            // Add file attachment annotation
            String attachment_file = Common.getFixFolder() + "AboutFoxit.pdf";
            FileAttachment file_attachment = new FileAttachment(
                    page.addAnnot(Annot.e_FileAttachment, new RectF(280, 350, 300, 380)));
            file_attachment.setIconName("Graph");

            FileSpec file_spec = new FileSpec(page.getDocument());
            file_spec.setFileName("attachment.pdf");
            file_spec.setCreationDateTime(Common.getCurrentDateTime());
            file_spec.setDescription("The original file");
            file_spec.setModifiedDateTime(Common.getCurrentDateTime());
            file_spec.embed(attachment_file);
            file_attachment.setFileSpec(file_spec);
            file_attachment.setSubject("File Attachment");
            file_attachment.setTitle("Foxit SDK");
            // Appearance should be reset.
            file_attachment.resetAppearanceStream();

            // Add link annotation
            Link link = new Link(page.addAnnot(Annot.e_Link, new RectF(350, 350, 380, 400)));
            link.setHighlightingMode(Annot.e_HighlightingToggle);

            // Add action for link annotation
            Action action = Action.create(page.getDocument(), Action.e_TypeURI);
            URIAction uriAction = new URIAction(action);
            uriAction.setTrackPositionFlag(true);
            uriAction.setURI("www.foxitsoftware.com");
            link.setAction(uriAction);
            // Appearance should be reset.
            link.resetAppearanceStream();


            // Set icon provider for annotation to Foxit PDF SDK.
            CustomIconProvider icon_provider = new CustomIconProvider(mContext);
            Library.setAnnotIconProviderCallback(icon_provider);

            // Add common stamp annotation.
            Library.setActionCallback(null);
            icon_provider.setUseDynamicStamp(false);
            annot = page.addAnnot(Annot.e_Stamp, new RectF(110, 150, 200, 250));
            Stamp static_stamp = new Stamp(annot);
            static_stamp.setIconName("Approved");
            // Appearance should be reset.
            static_stamp.resetAppearanceStream();

            // Add dynamic stamp annotation.
            CustomActionCallback action_callback = new CustomActionCallback(mContext);
            Library.setActionCallback(action_callback);
            icon_provider.setUseDynamicStamp(true);
            annot = page.addAnnot(Annot.e_Stamp, new RectF(10, 150, 100, 250));
            Stamp dynamic_stamp = new Stamp(annot);
            dynamic_stamp.setIconName("Approved");
            // Appearance should be reset.
            dynamic_stamp.resetAppearanceStream();
        } catch (PDFException e) {
            e.printStackTrace();
            Toast.makeText(mContext, mContext.getString(R.string.fx_add_annot_error, e.getMessage()), Toast.LENGTH_LONG).show();
        }
    }

    private void exportToJSON(PDFDoc doc) {
        if (doc == null) return;

        try {
            int pageCount = doc.getPageCount();

            JSONArray jsonArray = new JSONArray();
            for (int i = 0; i < pageCount; i++) {
                PDFPage page = doc.getPage(i);
                int annotCount = page.getAnnotCount();
                for (int j = 0; j < annotCount; j++) {
                    Annot annot = page.getAnnot(j);
                    if (annot == null || annot.isEmpty()) continue;

                    JSONObject jsonObject = exportAnnotToJSON(doc, annot);
                    if (jsonObject != null) {
                        jsonArray.put(jsonObject);
                    }
                }
            }

            String annotJson = jsonArray.toString().replace("\\/", "/");
            File txtFile = new File(outputJsonFile);
            FileWriter fileWriter = new FileWriter(txtFile);
            fileWriter.write(annotJson);
            fileWriter.flush();
            fileWriter.close();
        } catch (PDFException | IOException e) {
            e.printStackTrace();
        }
    }

    private JSONObject exportAnnotToJSON(PDFDoc doc, Annot annot) {
        try {
            int type = annot.getType();
            if (type == Annot.e_Widget || type == Annot.e_Popup || type == Annot.e_Watermark)
                return null;

            switch (type) {
                case Annot.e_Caret:
                    return JSCaret.exportToJSON(doc, annot);
                case Annot.e_Circle:
                    return JSCircle.exportToJSON(doc, annot);
                case Annot.e_FileAttachment:
                    return JSFileAttachment.exportToJSON(doc, annot);
                case Annot.e_FreeText:
                    return JSFreeText.exportToJSON(doc, annot);
                case Annot.e_Ink:
                    return JSInk.exportToJSON(doc, annot);
                case Annot.e_Line:
                    return JSLine.exportToJSON(doc, annot);
                case Annot.e_Note:
                    return JSNote.exportToJSON(doc, annot);
                case Annot.e_Polygon:
                    return JSPolygon.exportToJSON(doc, annot);
                case Annot.e_PolyLine:
                    return JSPolyLine.exportToJSON(doc, annot);
//                case Annot.e_Popup:
//              return      JSPopup.exportToJSON(annot);
                case Annot.e_Redact:
                    return JSRedact.exportToJSON(doc, annot);
                case Annot.e_Sound:
                    return JSSound.exportToJSON(doc, annot);
                case Annot.e_Square:
                    return JSSquare.exportToJSON(doc, annot);
                case Annot.e_Stamp:
                    return JSStamp.exportToJSON(doc, annot);
                case Annot.e_Highlight:
                case Annot.e_Underline:
                case Annot.e_StrikeOut:
                case Annot.e_Squiggly:
                    return JSTextMarkupAnnot.exportToJSON(doc, annot);
                default:
                    if (annot.isMarkup())
                        return JSMarkupAnnot.exportToJSON(doc, annot);
                    else
                        return JSAnnot.exportToJSON(annot);
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return null;
    }

}
