/**
 * Copyright (C) 2003-2023, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdf.function;


import android.content.Context;
import android.widget.Toast;

import com.foxit.pdf.function_demo.R;
import com.foxit.sdk.PDFException;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.common.Font;
import com.foxit.sdk.common.Image;
import com.foxit.sdk.common.Path;
import com.foxit.sdk.common.fxcrt.Matrix2D;
import com.foxit.sdk.common.fxcrt.PointF;
import com.foxit.sdk.common.fxcrt.RectF;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.graphics.GraphicsObject;
import com.foxit.sdk.pdf.graphics.ImageObject;
import com.foxit.sdk.pdf.graphics.PathObject;
import com.foxit.sdk.pdf.graphics.TextObject;
import com.foxit.sdk.pdf.graphics.TextState;

import java.io.FileWriter;

public class GraphicsObjects {

    private Context mContext;

    public GraphicsObjects(Context context) {
        this.mContext = context;
    }

    public void addGraphicsObjects() {
        String inputPath = Common.getFixFolder() + "graphics_objects.pdf";
        String outputPath = Common.getOutputFilesFolder(Common.GRAPHICS_OBJECTS) + "graphics_objects.pdf";

        PDFDoc doc = Common.loadPDFDoc(mContext, inputPath, null);
        if (doc == null) return;
        try {
            // Get original shading objects from PDF page.
            PDFPage original_page = Common.loadPage(mContext, doc, 0, PDFPage.e_ParsePageNormal);
            if (original_page == null) return;
            long position = original_page.getFirstGraphicsObjectPosition(GraphicsObject.e_TypeShading);
            if (position == 0) return;

            GraphicsObject black_pieces = original_page.getGraphicsObject(position);

            position = original_page.getNextGraphicsObjectPosition(position, GraphicsObject.e_TypeShading);
            GraphicsObject white_pieces = original_page.getGraphicsObject(position);

            // Add a new PDF page and insert text objects.
            PDFPage page = doc.insertPage(0, PDFPage.e_SizeLetter);
            addTextObjects(page);

            // Add a new PDF page and insert image objects.
            page = doc.insertPage(1, PDFPage.e_SizeLetter);
            String image_file = Common.getFixFolder() + "image/sdk.png";
            addImageObjects(page, image_file);

            // Add a new PDF page and insert path objects, and copy shading objects.
            page = doc.insertPage(2, PDFPage.e_SizeLetter);
            addPathObjects(page, black_pieces, white_pieces);

            boolean ret = doc.saveAs(outputPath, PDFDoc.e_SaveFlagNoOriginal);
            if (ret)
                Toast.makeText(mContext,Common.getSuccessInfo(mContext, Common.getOutputFilesFolder(Common.GRAPHICS_OBJECTS)), Toast.LENGTH_LONG).show();
            else
                Toast.makeText(mContext, mContext.getString(R.string.fx_add_graphics_failed), Toast.LENGTH_SHORT).show();
        } catch (Exception e) {
            Toast.makeText(mContext, mContext.getString(R.string.fx_add_graphics_failed), Toast.LENGTH_SHORT).show();
        }
    }

    private void addTextObjects(PDFPage page) throws PDFException {
        long position = page.getLastGraphicsObjectPosition(GraphicsObject.e_TypeText);
        TextObject text_object = TextObject.create();
        if (text_object == null) {
            Toast.makeText(mContext, mContext.getString(R.string.fx_add_graphics_failed), Toast.LENGTH_SHORT).show();
            return;
        }
        text_object.setFillColor(0xFFFF7F00);

        // Prepare text state
        TextState state = new TextState();
        state.setFont_size(80.0f);
        state.setFont(new Font("Simsun", Font.e_StylesSmallCap, Font.e_CharsetGB2312, 0));
        state.setTextmode(TextState.e_ModeFill);
        text_object.setTextState(page, state, false, 750);

        // Set text.
        text_object.setText("Foxit Software");
        long last_position = page.insertGraphicsObject(position, text_object);

        RectF rect = text_object.getRect();
        float offset_x = (page.getWidth() - (rect.getRight() - rect.getLeft())) / 2;
        float offset_y = page.getHeight() * 0.8f - (rect.getTop() - rect.getBottom()) / 2;
        text_object.transform(new Matrix2D(1, 0, 0, 1, offset_x, offset_y), false);

        // Generate content
        page.generateContent();

        // Clone a text object from the old text object.
        text_object = text_object.clone().getTextObject();
        state.setFont(new Font(Font.e_StdIDTimes));
        state.setFont_size(48.0f);
        state.setTextmode(TextState.e_ModeFillStrokeClip);

        text_object.setTextState(page, state, true, 750);
        text_object.setText("www.foxitsoftware.com");

        text_object.setFillColor(0xFFAAAAAA);
        text_object.setStrokeColor(0xFFF68C21);
        page.insertGraphicsObject(last_position, text_object);

        rect = text_object.getRect();
        offset_x = (page.getWidth() - (rect.getRight() - rect.getLeft())) / 2;
        offset_y = page.getHeight() * 0.618f - (rect.getTop() - rect.getBottom()) / 2;
        text_object.transform(new Matrix2D(1, 0, 0, 1, offset_x, offset_y), false);

        // Generate content again after transformation.
        page.generateContent();

        //Show how to get the characters' information of text object.
        try {
            int text_object_charcount = text_object.getCharCount();
            FileWriter file_writer = new FileWriter(Common.getOutputFilesFolder(Common.GRAPHICS_OBJECTS) + "text_objects_info.txt", false);
            file_writer.write("The new text object has " + text_object_charcount + "characters.\r\n");

            for (int i = 0; i < text_object_charcount; i++) {
                //The character's position.
                PointF char_pos = text_object.getCharPos(i);
                //The character's width.
                float width = text_object.getCharWidthByIndex(i);
                //The character's height.
                float height = text_object.getCharHeightByIndex(i);

                file_writer.write("The position of the " + i + " characters is (" + char_pos.getX() + "," + char_pos.getY() + ").The width and height of the characters is (" + width + "," + height + ").\r\n");
            }
            file_writer.close();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void addPieces(PDFPage page, GraphicsObject orignal_pieces, RectF dst_rect) throws PDFException {
        long position = page.getFirstGraphicsObjectPosition(GraphicsObject.e_TypeAll);
        GraphicsObject pieces = orignal_pieces.clone();

        RectF piece_rect = pieces.getRect();

        // Calculates the transformation matrix between dst_rect and  piece_rect.
        float a = (dst_rect.getRight() - dst_rect.getLeft()) / (piece_rect.getRight() - piece_rect.getLeft());
        float d = (dst_rect.getTop() - dst_rect.getBottom()) / (piece_rect.getTop() - piece_rect.getBottom());
        float e = dst_rect.getLeft() - piece_rect.getLeft() * a;
        float f = dst_rect.getTop() - piece_rect.getTop() * d;

        // Transform rect.
        pieces.transform(new Matrix2D(a, 0, 0, d, e, f), false);
        page.insertGraphicsObject(position, pieces);

        page.generateContent();
    }

    private void addPathObjects(PDFPage page, GraphicsObject black_pieces, GraphicsObject white_pieces) throws PDFException {
        long position = page.getLastGraphicsObjectPosition(GraphicsObject.e_TypePath);
        PathObject path_object = PathObject.create();
        if (path_object == null) {
            Toast.makeText(mContext, mContext.getString(R.string.fx_add_graphics_failed), Toast.LENGTH_SHORT).show();
            return;
        }
        Path path = new Path();
        float page_width = page.getWidth();
        float page_height = page.getHeight();

        float width = Math.min(page_width, page_height) / 20.0f;
        float start_x = (page_width - width * 18.0f) / 2.0f;
        float start_y = (page_height - width * 18.0f) / 2.0f;

        // Draw a chess board
        for (int i = 0; i < 19; i++) {
            float x1 = start_x;
            float y1 = i * width + start_y;

            float x2 = start_x + 18 * width;
            path.moveTo(new PointF(x1, y1));
            path.lineTo(new PointF(x2, y1));

            x1 = i * width + start_x;
            y1 = start_y;

            float y2 = 18 * width + start_y;
            path.moveTo(new PointF(x1, y1));
            path.lineTo(new PointF(x1, y2));
        }

        int[] star = {3, 9, 15};
        for (int i = 0; i < 3; i++) {
            for (int j = 0; j < 3; j++) {
                RectF rect = new RectF(start_x + star[i] * width - width / 12, start_y + star[j] * width - width / 12,
                        start_x + star[i] * width + width / 12, start_y + star[j] * width + width / 12);
                path.appendEllipse(rect);
            }
        }

        path_object.setPathData(path);
        path_object.setFillColor(0xFF000000);
        path_object.setFillMode(Constants.e_FillModeAlternate);
        path_object.setStrokeState(true);
        path_object.setStrokeColor(0xFF000000);

        page.insertGraphicsObject(position, path_object);
        page.generateContent();

        // Draw pieces
        PointF[][] pieces_vector = {{new PointF(3, 3), new PointF(3, 7), new PointF(3, 15), new PointF(13, 2),
                new PointF(13, 16), new PointF(13, 17), new PointF(15, 16), new PointF(16, 16)},
                {new PointF(11, 16), new PointF(12, 14), new PointF(14, 4), new PointF(14, 15),
                        new PointF(15, 3), new PointF(15, 9), new PointF(15, 15), new PointF(16, 15)}};
        for (int k = 0; k < 2; k++) {
            for (int i = 0; i < pieces_vector[k].length; i++) {
                int x = (int) pieces_vector[k][i].getX();
                int y = (int) pieces_vector[k][i].getY();
                addPieces(page, (k % 2 != 0) ? white_pieces : black_pieces,
                        new RectF(start_x + x * width - width / 2.f, start_y + y * width - width / 2.f,
                                start_x + x * width + width / 2.f, start_y + y * width + width / 2.f));
            }
        }
    }

    private void addImageObjects(PDFPage page, String image_file) throws PDFException {
        long position = page.getLastGraphicsObjectPosition(GraphicsObject.e_TypeImage);
        Image image = new Image(image_file);
        ImageObject image_object = ImageObject.create(page.getDocument());
        if (image_object == null) {
            Toast.makeText(mContext, mContext.getString(R.string.fx_add_graphics_failed), Toast.LENGTH_SHORT).show();
            return;
        }
        image_object.setImage(image, 0);

        float width = image.getWidth();
        float height = image.getHeight();

        float page_width = page.getWidth();
        float page_height = page.getHeight();

        // Please notice the matrix value.
        image_object.setMatrix(new Matrix2D(width, 0, 0, height, (page_width - width) / 2.0f, (page_height - height) / 2.0f));

        page.insertGraphicsObject(position, image_object);
        page.generateContent();
    }

}
