/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdfreader.fragment;


import android.app.Activity;
import android.content.Intent;
import android.content.res.Configuration;
import android.os.Bundle;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.widget.LinearLayout;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentManager;
import androidx.fragment.app.FragmentTransaction;

import com.foxit.App;
import com.foxit.home.HomeFragment;
import com.foxit.home.MainActivity;
import com.foxit.home.R;
import com.foxit.pdfreader.MultiTabView;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.home.IHomeModule;
import com.foxit.uiextensions.modules.signature.SignatureToolHandler;
import com.foxit.uiextensions.pdfreader.config.ReadStateConfig;
import com.foxit.uiextensions.utils.AppFileUtil;

public class PDFReaderTabsFragment extends Fragment implements UIExtensionsManager.OnFinishListener {
    public static final String FRAGMENT_NAME = "READER_FRAGMENT";
    public static final String SINGLE_DOC_TAG = "SINGLE_DOC_TAG";

    private FragmentManager mFragmentManager;

    private String filter = App.FILTER_DEFAULT;
    private String filePath;

    public static PDFReaderTabsFragment newInstance(String filter) {
        PDFReaderTabsFragment fragment = new PDFReaderTabsFragment();
        Bundle args = new Bundle();
        args.putString(HomeFragment.BUNDLE_KEY_FILTER, filter);
        fragment.setArguments(args);
        return fragment;
    }

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, @Nullable ViewGroup container, Bundle savedInstanceState) {
        return inflater.inflate(R.layout.activity_reader, container, false);
    }

    @Override
    public void onActivityCreated(@Nullable Bundle savedInstanceState) {
        super.onActivityCreated(savedInstanceState);
        if (getArguments() != null) {
            filter = getArguments().getString(HomeFragment.BUNDLE_KEY_FILTER);
        }
        mFragmentManager = getActivity().getSupportFragmentManager();
        App.instance().getTabsManager(filter).setFragmentManager(mFragmentManager);
        getActivity().getWindow().setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_STATE_ALWAYS_HIDDEN);

        handleIntent(getActivity().getIntent());
    }

    private void handleIntent(Intent intent) {
        if (intent != null) {
            String path = AppFileUtil.getFilePath(App.instance().getApplicationContext(), intent, IHomeModule.FILE_EXTRA);
            if (path != null) {
                changeReaderState(MainActivity.READER_STATE_READ);
                openDocument(intent);
            }
        }
    }

    public boolean onKeyDown(Activity activity, int keyCode, KeyEvent event) {
        BaseFragment currentFrag = App.instance().getTabsManager(filter).getCurrentFragment();

        if (App.instance().isMultiTab()) {
            if (keyCode == KeyEvent.KEYCODE_BACK) {
                UIExtensionsManager uiExtensionsManager = currentFrag.mUiExtensionsManager;

                if (uiExtensionsManager != null
                        && uiExtensionsManager.backToNormalState()) {
                    return true;
                } else {
                    if (uiExtensionsManager != null && uiExtensionsManager.getState() == ReadStateConfig.STATE_COMPARE) {
                        MultiTabView tabView = App.instance().getMultiTabView(filter);
                        tabView.removeTab(currentFrag.getPath());
                        return true;
                    }

                    hideFragment(currentFrag);
                    changeReaderState(MainActivity.READER_STATE_HOME);
                    return true;
                }
            } else {
                return false;
            }
        }

        return currentFrag != null && currentFrag.mUiExtensionsManager != null && currentFrag.mUiExtensionsManager.onKeyDown(getActivity(), keyCode, event);
    }


    @Override
    public void onFinish() {
        BaseFragment currentFrag = App.instance().getTabsManager(filter).getCurrentFragment();
        if (App.instance().isMultiTab()) {
            if (currentFrag != null && !currentFrag.isOpenSuccess) {
                hideFragment(currentFrag);
                changeReaderState(MainActivity.READER_STATE_HOME);
            }
        } else {
            hideFragment(currentFrag);
            App.instance().getTabsManager(filter).setFilePath(null);
            changeReaderState(MainActivity.READER_STATE_HOME);
        }
    }

    private void changeReaderState(int state) {
        if (getActivity() != null)
            ((MainActivity) getActivity()).changeReaderState(state);
    }

    private void hideFragment(Fragment fragment) {
        if (fragment.isVisible()) {
            FragmentTransaction fragmentTransaction = mFragmentManager.beginTransaction();
            fragmentTransaction.hide(fragment).commit();
        }
    }

    private void removeFragment(BaseFragment fragment) {
        if (fragment != null) {
            FragmentTransaction fragmentTransaction = mFragmentManager.beginTransaction();
            fragmentTransaction.remove(fragment).commitAllowingStateLoss();
        }
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        if (App.instance().isMultiTab()) {
            App.instance().getMultiTabView(filter).unregisterTabEventListener(mTabEventListener);
        }
        App.instance().onBack();
        mTabEventListener = null;
    }

    public void openDocument(Intent intent) {
        if (!App.instance().checkLicense()) {
            openEmptyView();
            return;
        }
        String oldPath = App.instance().getTabsManager(filter).getFilePath();
        filePath = AppFileUtil.getFilePath(App.instance().getApplicationContext(), intent, IHomeModule.FILE_EXTRA);
        App.instance().getTabsManager(filter).setFilePath(filePath);
        PDFReaderFragment fragment = (PDFReaderFragment) App.instance().getTabsManager(filter).getFragmentMap().get(filePath);
        boolean needReset = oldPath != null && !filePath.equals(oldPath) && fragment != null && fragment.isOpenSuccess;

        if (App.instance().isMultiTab()) {
            openMultiDocument(false);
        } else {
            openSingleDocument();
        }
        if (needReset) {
            resetTabView(true);
        }
    }

    private void openSingleDocument() {
        filePath = App.instance().getTabsManager(filter).getFilePath();
        FragmentTransaction fragmentTransaction = mFragmentManager.beginTransaction();
        PDFReaderFragment fragment = (PDFReaderFragment) mFragmentManager.findFragmentByTag(SINGLE_DOC_TAG);
        if (fragment == null) {
            fragment = new PDFReaderFragment();
            fragment.setPath(filePath);
            fragment.setOnFinishListener(this);
            fragment.filter = filter;
        } else {
            fragment.setPath(filePath);
            fragment.setOnFinishListener(this);
            fragment.filter = filter;
            fragment.openDocument();
        }

        if (!fragment.isAdded()) {
            fragmentTransaction.add(R.id.reader_container, fragment, SINGLE_DOC_TAG);
        }

        fragmentTransaction.show(fragment).commitAllowingStateLoss();
        App.instance().getTabsManager(filter).addFragment(filePath, fragment);
        App.instance().getTabsManager(filter).setCurrentFragment(fragment);
    }

    private void openMultiDocument(boolean isRemoveCurFragment) {
        App.instance().getMultiTabView(filter).registerTabEventListener(mTabEventListener);

        filePath = App.instance().getTabsManager(filter).getFilePath();
        FragmentTransaction fragmentTransaction = mFragmentManager.beginTransaction();

        PDFReaderFragment fragment = (PDFReaderFragment) App.instance().getTabsManager(filter).getFragmentMap().get(filePath);
        if (fragment == null) {
            fragment = new PDFReaderFragment();
            App.instance().getTabsManager(filter).addFragment(filePath, fragment);
            fragment.setPath(filePath);
            fragment.setOnFinishListener(this);
            fragment.filter = filter;
        } else {
            if (!fragment.isOpenSuccess) {
                App.instance().getTabsManager(filter).removeFragment(filePath);
                fragmentTransaction.remove(fragment);

                fragment = new PDFReaderFragment();
                App.instance().getTabsManager(filter).addFragment(filePath, fragment);
                fragment.setPath(filePath);
                fragment.setOnFinishListener(this);
                fragment.filter = filter;
            }
        }

        if (!fragment.isAdded()) {
            fragmentTransaction.add(R.id.reader_container, fragment);
        }

        Fragment currentFragment = App.instance().getTabsManager(filter).getCurrentFragment();
        if (currentFragment != null && !currentFragment.equals(fragment)) {
            if (isRemoveCurFragment) {
                fragmentTransaction.remove(currentFragment);
            } else {
                fragmentTransaction.hide(currentFragment);
            }
        }

        fragmentTransaction.show(fragment).commitAllowingStateLoss();
        App.instance().getTabsManager(filter).setCurrentFragment(fragment);
    }

    /**
     * when App license is valid, it should open a empty view use Fragment also.
     */
    private void openEmptyView() {
        FragmentTransaction fragmentTransaction = mFragmentManager.beginTransaction();
        fragmentTransaction.replace(R.id.reader_container, new EmptyViewFragment());
        fragmentTransaction.commitAllowingStateLoss();
    }

    private MultiTabView.ITabEventListener mTabEventListener = new MultiTabView.ITabEventListener() {
        @Override
        public void onTabChanged(MultiTabView.TabInfo oldTabInfo, MultiTabView.TabInfo newTabInfo) {
            PDFReaderFragment fragment = (PDFReaderFragment) App.instance().getTabsManager(filter).getFragmentMap().get(oldTabInfo.tabTarget);
            changeViewerState(fragment);
            filePath = newTabInfo.tabTarget;
            App.instance().getTabsManager(filter).setFilePath(filePath);

            PDFReaderFragment newfragment = (PDFReaderFragment) App.instance().getTabsManager(filter).getFragmentMap().get(filePath);
            newfragment.mUiExtensionsManager.getDocumentManager().resetActionCallback();

            openMultiDocument(false);
            resetTabView(false);
        }

        @Override
        public void onTabRemoved(final MultiTabView.TabInfo removedTab, final MultiTabView.TabInfo showTab) {
            final PDFReaderFragment fragment = (PDFReaderFragment) App.instance().getTabsManager(filter).getFragmentMap().get(removedTab.tabTarget);
            if (removedTab.tabTarget.equals(App.instance().getTabsManager(filter).getFilePath())) {
                changeViewerState(fragment);

                fragment.doClose(new BaseFragment.IFragmentEvent() {
                    @Override
                    public void onRemove() {
                        App.instance().getMultiTabView(filter).removeTab(removedTab);

                        if (showTab != null) {
                            filePath = showTab.tabTarget;
                            App.instance().getTabsManager(filter).setFilePath(filePath);

                            BaseFragment currentFrag = App.instance().getTabsManager(filter).getCurrentFragment();
                            if (currentFrag != null && currentFrag.mUiExtensionsManager.getState() == ReadStateConfig.STATE_COMPARE) {
                                removeFragment(fragment);
                                App.instance().getTabsManager(filter).setCurrentFragment(null);
                                changeReaderState(MainActivity.READER_STATE_HOME);
                            } else {
                                openMultiDocument(true);
                            }

                            resetTabView(true);
                            App.instance().getTabsManager(filter).getFragmentMap().remove(removedTab.tabTarget);
                        } else {
                            App.instance().getTabsManager(filter).setFilePath(null);

                            // only one tab
                            removeFragment(fragment);
                            App.instance().getTabsManager(filter).setCurrentFragment(null);
                            App.instance().getTabsManager(filter).clearFragment();

                            changeReaderState(MainActivity.READER_STATE_HOME);
                        }
                    }
                });
            } else {
                fragment.doClose(new BaseFragment.IFragmentEvent() {
                    @Override
                    public void onRemove() {
                        App.instance().getMultiTabView(filter).removeTab(removedTab);
                        removeFragment(fragment);
                        App.instance().getTabsManager(filter).getFragmentMap().remove(removedTab.tabTarget);
                        App.instance().getMultiTabView(filter).refreshTopBar(showTab.tabTarget);
                    }
                });
            }
        }
    };

    private void resetTabView(boolean needRefresh) {
        if (needRefresh) {
            App.instance().getMultiTabView(filter).refreshTopBar(App.instance().getTabsManager(filter).getFilePath());
        }
        PDFReaderFragment fragment = (PDFReaderFragment) App.instance().getTabsManager(filter).getFragmentMap().get(App.instance().getTabsManager(filter).getFilePath());
        int h = fragment.mUiExtensionsManager.getMainFrame().getTopToolbar().getContentView().getHeight();
        LinearLayout.LayoutParams params = new LinearLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, 2 * h / 3);
        params.topMargin = -10;
        ViewGroup parent = (ViewGroup) App.instance().getMultiTabView(filter).getTabView().getParent();
        if (parent != null) {
            parent.removeView(App.instance().getMultiTabView(filter).getTabView());
        }
        fragment.mUiExtensionsManager.getMainFrame().getTopActionView().addView(App.instance().getMultiTabView(filter).getTabView(), params);
    }

    private void changeViewerState(PDFReaderFragment fragment) {
        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) fragment.getPdfViewCtrl().getUIExtensionsManager();
        uiExtensionsManager.triggerDismissMenuEvent();
        uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
        uiExtensionsManager.exitPanZoomMode();
        ToolHandler toolHandler = uiExtensionsManager.getCurrentToolHandler();
        if (toolHandler != null) {
            uiExtensionsManager.setCurrentToolHandler(null);
            if (toolHandler instanceof SignatureToolHandler) {
                fragment.getPdfViewCtrl().invalidate();
            }
        }
    }

    @Override
    public void onConfigurationChanged(Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
        Activity activity = getActivity();
        if (activity == null)
            return;

        BaseFragment currentFrag = App.instance().getTabsManager(filter).getCurrentFragment();
        if (App.instance().isMultiTab()) {
            if (currentFrag != null && currentFrag.isOpenSuccess) {
                App.instance().getMultiTabView(filter).refreshTopBar(currentFrag.getPath());
            }
        }
    }

    public void handleActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        BaseFragment currentFrag = App.instance().getTabsManager(filter).getCurrentFragment();
        if (currentFrag == null) return;
        currentFrag.handleActivityResult(requestCode, resultCode, data);
    }

    public void handleRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);
        BaseFragment currentFrag = App.instance().getTabsManager(filter).getCurrentFragment();
        if (currentFrag == null) return;
        currentFrag.handleRequestPermissionsResult(requestCode, permissions, grantResults);
    }

}
