/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.utils;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.res.ColorStateList;
import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.graphics.Matrix;
import android.graphics.Point;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.Shader.TileMode;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.net.ConnectivityManager;
import android.net.Uri;
import android.os.Build;
import android.text.TextUtils;
import android.view.Gravity;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.ViewGroup;
import android.view.inputmethod.InputMethodManager;
import android.webkit.WebView;
import android.widget.FrameLayout;
import android.widget.ImageView;
import android.widget.TextView;
import android.widget.ZoomButtonsController;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.fxcrt.Matrix2D;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.PageBasicInfo;
import com.foxit.sdk.pdf.actions.Destination;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.controls.dialog.UIDialog;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.controls.filebrowser.imp.FileThumbnail;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.theme.ThemeConfig;

import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.util.Date;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.foxit.sdk.common.Constants.e_ErrDeviceLimitation;
import static com.foxit.sdk.common.Constants.e_ErrFormat;
import static com.foxit.sdk.common.Constants.e_ErrInvalidLicense;
import static com.foxit.sdk.common.Constants.e_ErrNoComparisonModuleRight;
import static com.foxit.sdk.common.Constants.e_ErrNoComplianceModuleRight;
import static com.foxit.sdk.common.Constants.e_ErrNoConnectedPDFModuleRight;
import static com.foxit.sdk.common.Constants.e_ErrNoConversionModuleRight;
import static com.foxit.sdk.common.Constants.e_ErrNoOCRModuleRight;
import static com.foxit.sdk.common.Constants.e_ErrNoOptimizerModuleRight;
import static com.foxit.sdk.common.Constants.e_ErrNoRMSModuleRight;
import static com.foxit.sdk.common.Constants.e_ErrNoRedactionModuleRight;
import static com.foxit.sdk.common.Constants.e_ErrNoRights;
import static com.foxit.sdk.common.Constants.e_ErrNoXFAModuleRight;
import static com.foxit.sdk.common.Constants.e_ErrRightsExpired;
import static com.foxit.uiextensions.utils.AppResource.getDrawableByFileName;

public class AppUtil {
    public static final int ALERT_OK = 1;
    public static final int ALERT_CANCEL = 2;

    public static boolean isEmailFormatForRMS(String userId) {
        Pattern emailPattern = Pattern.compile("[a-zA-Z0-9!#$%&'*+/=?^_`{|}~-]+(?:\\.[a-zA-Z0-9!#$%&'*+/=?^_`{|}~-]+)*@(?:[a-zA-Z0-9](?:[a-zA-Z0-9-]*[a-zA-Z0-9])?\\.)+[a-zA-Z0-9](?:[a-zA-Z0-9-]*[a-zA-Z0-9])?");
        Matcher matcher = emailPattern.matcher(userId);
        return matcher.find();
    }

    private static long sLastTimeMillis;

    public static boolean isFastDoubleClick() {
        long currentTimeMillis = System.currentTimeMillis();
        long delta = currentTimeMillis - sLastTimeMillis;
        if (Math.abs(delta) < 500) {
            return true;
        }
        sLastTimeMillis = currentTimeMillis;
        return false;
    }

    public static void showSoftInput(final View editText) {
        if (editText == null) return;
        editText.requestFocus();
        editText.post(new Runnable() {
            @Override
            public void run() {
                InputMethodManager imm = (InputMethodManager) editText.getContext().getSystemService(Context.INPUT_METHOD_SERVICE);
                imm.showSoftInput(editText, 0);
            }
        });
    }

    public static void dismissInputSoft(final View editText) {
        if (editText == null) return;
        InputMethodManager imm = (InputMethodManager) editText.getContext().getSystemService(Context.INPUT_METHOD_SERVICE);
        imm.hideSoftInputFromWindow(editText.getWindowToken(), 0);
    }

    public static void fixBackgroundRepeat(View view) {
        Drawable bg = view.getBackground();
        if (bg != null) {
            if (bg instanceof BitmapDrawable) {
                BitmapDrawable bmp = (BitmapDrawable) bg;
                bmp.mutate();
                bmp.setTileModeXY(TileMode.REPEAT, TileMode.REPEAT);
            }
        }
    }

    public static void openUrl(final Activity act, String url) {
        final String myurl = url;

        final UITextEditDialog dialog = new UITextEditDialog(act, UIDialog.NO_INPUT);
        dialog.setTitle(act.getApplicationContext().getString(R.string.security_warning));
        dialog.getPromptTextView().setText(String.format(AppResource.getString(act, R.string.open_external_link_confirm_content), url));
        dialog.getOKButton().setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                Uri uri;
                if (myurl.toLowerCase().startsWith("http://") || myurl.toLowerCase().startsWith("https://")) {
                    uri = Uri.parse(myurl);
                } else {
                    uri = Uri.parse("http://" + myurl);
                }
                Intent it = new Intent(Intent.ACTION_VIEW, uri);
                act.startActivity(it);
                dialog.dismiss();
            }
        });

        dialog.getCancelButton().setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                dialog.dismiss();
            }
        });

        dialog.show();
    }

    public static void mailTo(Activity act, String uri) {
        if (isEmpty(uri) || isFastDoubleClick()) return;
        Intent intent = new Intent(Intent.ACTION_SENDTO);
        if (uri.startsWith("mailto:")) {
            intent.setData(Uri.parse(uri));
        } else {
            intent.setData(Uri.parse("mailto:" + uri));
        }
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        act.startActivity(Intent.createChooser(intent, ""));
    }

    public static boolean isEmpty(CharSequence str) {
        return str == null || str.length() == 0;
    }

    public static boolean isEqual(CharSequence str1, CharSequence str2) {
        if (str1 == null && str2 == null)
            return true;
        if (str1 == null || str2 == null)
            return false;
        return str1.equals(str2);
    }

    public static void alert(Activity act, final String title, final String prompt, final int buttons) {
        final UITextEditDialog dlg = new UITextEditDialog(act, UIDialog.NO_INPUT);
        dlg.setTitle(title);
        dlg.getPromptTextView().setText(prompt);
        if ((buttons & ALERT_OK) == 0)
            dlg.getOKButton().setVisibility(View.GONE);
        if ((buttons & ALERT_CANCEL) == 0)
            dlg.getCancelButton().setVisibility(View.GONE);
        dlg.show();

        dlg.getOKButton().setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                dlg.dismiss();
            }
        });

        dlg.getCancelButton().setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                dlg.dismiss();
            }
        });
        dlg.setOnDismissListener(new DialogInterface.OnDismissListener() {
            @Override
            public void onDismiss(DialogInterface dialog) {
                dlg.dismiss();
            }
        });
    }

    private static int getStartPos(String allData, String entry) {
        int startPos = allData.indexOf(entry);
        if (startPos > 0 && !allData.substring(startPos - 1, startPos).contentEquals(",")) {
            return getStartPos(allData.substring(startPos + entry.length()), entry);
        }
        return startPos;
    }

    public static String getEntryName(String allData, String entry) {
        if (allData == null || entry == null)
            return null;
        //CN=X,O=X,OU=X,E=X,C=X
        String name = null;

        int startPos = getStartPos(allData, entry);//allData.indexOf(entry);
        if (startPos < 0) {
            return "";
        }
        int endPos = 0;
        if (entry.contentEquals("C=")) {
            endPos = allData.length();
        } else {
            endPos = allData.indexOf(",", startPos);
        }

        if (endPos == -1) {
            endPos = allData.length();
        }

        name = allData.substring(startPos + entry.length(), endPos);

        return name;
    }

    public static String getFileName(String filePath) {
        int index = filePath.lastIndexOf('/');
        return (index < 0) ? filePath : filePath.substring(index + 1);
    }


    public static String fileSizeToString(long size) {
        float fsize = size;
        char[] unit = {'B', 'K', 'M'};
        for (int i = 0; i < unit.length; i++) {
            if (fsize < 1024 || i == unit.length - 1) {
                BigDecimal b = new BigDecimal(fsize);
                fsize = b.setScale(2, BigDecimal.ROUND_HALF_UP).floatValue();
                return String.valueOf(fsize) + unit[i];
            }
            fsize /= 1024;
        }
        return "";
    }

    public static String getFileFolder(String filePath) {
        int index = filePath.lastIndexOf('/');
        if (index < 0) return "";
        return filePath.substring(0, index);
    }

    public static boolean isBlank(String str) {
        int strLen;
        if (str != null && (strLen = str.length()) != 0) {
            for (int i = 0; i < strLen; ++i) {
                if (!Character.isWhitespace(str.charAt(i))) {
                    return false;
                }
            }
            return true;
        } else {
            return true;
        }
    }

    /**
     * Returns {@code o} if non-null, or throws {@code NullPointerException}.
     */
    public static <T> T requireNonNull(T o) {
        if (o == null) {
            throw new NullPointerException();
        }
        return o;
    }

    /**
     * Returns {@code o} if non-null, or throws {@code NullPointerException}.
     */
    public static <T> T requireNonNull(T o, String detailMessage) {
        if (o == null) {
            throw new NullPointerException(detailMessage);
        }
        return o;
    }

    public static void setWebViewZoomControlButtonGone(WebView view) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.HONEYCOMB) {
            view.getSettings().setDisplayZoomControls(false);
            return;
        }
        Class classType;
        Field field;
        try {
            classType = WebView.class;
            field = classType.getDeclaredField("mZoomButtonsController");
            field.setAccessible(true);
            ZoomButtonsController mZoomButtonsController = new ZoomButtonsController(view);
            mZoomButtonsController.getZoomControls().setVisibility(View.GONE);
            try {
                field.set(view, mZoomButtonsController);
            } catch (IllegalArgumentException e) {
                e.printStackTrace();
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            }
        } catch (SecurityException e) {
            e.printStackTrace();
        } catch (NoSuchFieldException e) {
            e.printStackTrace();
        }
    }

    public static Rect toRect(com.foxit.sdk.common.fxcrt.RectI rectI) {
        return new Rect(rectI.getLeft(), rectI.getTop(), rectI.getRight(), rectI.getBottom());
    }

    public static RectF toRectF(com.foxit.sdk.common.fxcrt.RectI rect) {
        return new RectF(rect.getLeft(), rect.getTop(), rect.getRight(), rect.getBottom());
    }

    public static RectF toRectF(com.foxit.sdk.common.fxcrt.RectF rectF) {
        return new RectF(rectF.getLeft(), rectF.getTop(), rectF.getRight(), rectF.getBottom());
    }

    public static com.foxit.sdk.common.fxcrt.RectF toFxRectF(RectF rectF) {
        return new com.foxit.sdk.common.fxcrt.RectF(rectF.left, rectF.bottom, rectF.right, rectF.top);
    }

    public static PointF toPointF(com.foxit.sdk.common.fxcrt.PointF pointF) {
        return new PointF(pointF.getX(), pointF.getY());
    }

    public static com.foxit.sdk.common.fxcrt.PointF toFxPointF(PointF pointF) {
        return new com.foxit.sdk.common.fxcrt.PointF(pointF.x, pointF.y);
    }

    public static com.foxit.sdk.common.fxcrt.PointF toFxPointF(float x, float y) {
        return new com.foxit.sdk.common.fxcrt.PointF(x, y);
    }

    // in pdf coordinate
    public static void unionFxRectF(RectF lRect, RectF rRect) {
        if ((rRect.left < rRect.right) && (rRect.bottom < rRect.top)) {
            if ((lRect.left < lRect.right) && (lRect.bottom < lRect.top)) {
                if (lRect.left > rRect.left)
                    lRect.left = rRect.left;
                if (lRect.top < rRect.top)
                    lRect.top = rRect.top;
                if (lRect.right < rRect.right)
                    lRect.right = rRect.right;
                if (lRect.bottom > rRect.bottom)
                    lRect.bottom = rRect.bottom;
            } else {
                lRect.left = rRect.left;
                lRect.top = rRect.top;
                lRect.right = rRect.right;
                lRect.bottom = rRect.bottom;
            }
        }
    }

    public static boolean containsFxPoint(RectF srcRectF, float x, float y){
        return srcRectF.left < srcRectF.right && srcRectF.top > srcRectF.bottom  // check for empty first
                && x >= srcRectF.left && x < srcRectF.right && y <= srcRectF.top && y > srcRectF.bottom;
    }

    public static RectF toGlobalVisibleRectF(View rootView, RectF viewRectF) {
        int[] location = new int[2];
        rootView.getLocationInWindow(location);
        int x = location[0];
        int y = location[1] /*- AppDisplay.getStatusBarHeight()*/;
        return new RectF(viewRectF.left + x, viewRectF.top + y, viewRectF.right + x, viewRectF.bottom + y);
    }

    public static String getMessage(Context context, int errCode) {
        String message = AppResource.getString(context.getApplicationContext(), R.string.rv_document_open_failed);
        switch (errCode) {
            case e_ErrFormat:
                message = AppResource.getString(context.getApplicationContext(), R.string.rv_format_error);
                break;
            case e_ErrNoConnectedPDFModuleRight:
            case e_ErrNoXFAModuleRight:
            case e_ErrNoRedactionModuleRight:
            case e_ErrNoRMSModuleRight:
            case e_ErrNoOCRModuleRight:
            case e_ErrNoComparisonModuleRight:
            case e_ErrNoComplianceModuleRight:
            case e_ErrNoOptimizerModuleRight:
            case e_ErrNoConversionModuleRight:
            case e_ErrInvalidLicense:
                message = AppResource.getString(context.getApplicationContext(), R.string.rv_invalid_license);
                break;
            case e_ErrDeviceLimitation:
                message = AppResource.getString(context.getApplicationContext(), R.string.rv_deivce_limitation);
                break;
            case e_ErrNoRights:
                message = AppResource.getString(context.getApplicationContext(), R.string.rv_no_rights);
                break;
            case e_ErrRightsExpired:
                message = AppResource.getString(context.getApplicationContext(), R.string.rv_rights_expired);
                break;
            default:
                break;
        }
        return message;
    }

    public static Matrix2D toMatrix2D(Matrix matrix) {
        if (matrix == null) return null;
        float[] values = new float[9];
        matrix.getValues(values);
        return new Matrix2D(values[0], values[3], values[1], values[4], values[2], values[5]);
    }

    public static void normalizePDFRect(RectF rectF) {
        if (rectF.left > rectF.right) {
            float tmp = rectF.left;
            rectF.left = rectF.right;
            rectF.right = tmp;
        }

        if (rectF.top < rectF.bottom) {
            float tmp = rectF.top;
            rectF.top = rectF.bottom;
            rectF.bottom = tmp;
        }

        if (rectF.left == rectF.right) rectF.right += 1;
        if (rectF.top == rectF.bottom) rectF.top += 1;
    }

    public static boolean isEmptyPDFRect(RectF rectF) {
        return rectF.left >= rectF.right || rectF.top <= rectF.bottom;
    }

    public static PointF getDestinationPoint(PDFDoc doc, Destination destination) {
        if (doc == null || destination == null || destination.isEmpty()) {
            return null;
        }

        PointF pt = new PointF(0, 0);
        try {
            switch (destination.getZoomMode()) {
                case Destination.e_ZoomXYZ:
                    pt.x = destination.getLeft();
                    pt.y = destination.getTop();
                    break;
                case Destination.e_ZoomFitHorz:
                case Destination.e_ZoomFitBHorz:
                    pt.y = destination.getTop();
                    break;
                case Destination.e_ZoomFitVert:
                case Destination.e_ZoomFitBVert:
                    pt.x = destination.getLeft();
                    break;
                case Destination.e_ZoomFitRect:
                    pt.x = destination.getLeft();
                    pt.y = destination.getBottom();
                    break;
                default:
                    PageBasicInfo pageInfo = doc.getPageBasicInfo(destination.getPageIndex(doc));
                    pt.x = 0;
                    pt.y = pageInfo.getHeight();
                    break;
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return pt;
    }

    public static RectF calculateRect(RectF desRectF, RectF srcRectF) {
        if (srcRectF.isEmpty()) return desRectF;
        int count = 0;
        if (desRectF.left == srcRectF.left && desRectF.top == srcRectF.top) count++;
        if (desRectF.right == srcRectF.right && desRectF.top == srcRectF.top) count++;
        if (desRectF.left == srcRectF.left && desRectF.bottom == srcRectF.bottom) count++;
        if (desRectF.right == srcRectF.right && desRectF.bottom == srcRectF.bottom) count++;
        RectF tmpDesRect = new RectF(desRectF);
        if (count == 2) {
            tmpDesRect.union(srcRectF);
            RectF rectF = new RectF(tmpDesRect);
            tmpDesRect.intersect(srcRectF);
            rectF.intersect(tmpDesRect);
            return rectF;
        } else if (count == 3 || count == 4) {
            return tmpDesRect;
        } else {
            tmpDesRect.union(srcRectF);
            return tmpDesRect;
        }
    }

    public static int timeCompare(Date d1, Date d2) {
        if (d1 == null || d2 == null) return 0;
        if (d1.after(d2)) {
            return 1;
        } else if (d1.before(d2)) {
            return -1;
        } else {
            return 0;
        }
    }

    public static void removeViewFromParent(View view) {
        if (view != null && view.getParent() != null) {
            ViewGroup vg = (ViewGroup) view.getParent();
            vg.removeView(view);
        }
    }

    public static void upperCaseTextView(TextView textView) {
        if (textView == null || TextUtils.isEmpty(textView.getText())) return;
        textView.setText(textView.getText().toString().toUpperCase());
    }

    /**
     * set thumbnail or icon for the current document on title view
     *
     * @param context   context
     * @param imageView target image view
     * @param path      image path
     * @param callback  get thumbnail call back
     */
    public static void setThumbnailForImageView(Context context, ImageView imageView, String path, FileThumbnail.ThumbnailCallback callback) {
        if (AppUtil.isEmpty(path)) {
            imageView.setImageResource(R.drawable.fb_file_pdf);
            return;
        }
        String extension = path.substring(path.lastIndexOf(".") + 1).toLowerCase();
        if (extension != null && extension.equals("ofd")) {
            Bitmap bitmap = FileThumbnail.getInstance(context).getThumbnail(path, callback);
            if (bitmap != null) {
                imageView.setBackground(AppResource.getDrawable(context, R.drawable.fb_file_pdf_bg));
                imageView.setImageBitmap(bitmap);
            }
            return;
        }
        int drawableId = getDrawableByFileName(path);
        if (drawableId == -1) {
            Bitmap bitmap = FileThumbnail.getInstance(context).getThumbnail(path, callback);
            if (bitmap == null) {
                imageView.setImageResource(R.drawable.fb_file_pdf);
            } else {
                imageView.setBackground(AppResource.getDrawable(context, R.drawable.fb_file_pdf_bg));
                imageView.setImageBitmap(bitmap);
            }
        } else {
            imageView.setImageResource(drawableId);
        }
    }

    public static ColorStateList getSelectedButtonColorStateList(Context context) {
        int disabled = AppResource.getColor(context, R.color.p1);
        int selected = ThemeConfig.getInstance(context).getPrimaryColor();
        int normal = AppResource.getColor(context, R.color.p1);
        return AppResource.createColorStateList(selected, disabled, normal);
    }

    public static void setSelectedButtonState(boolean isSelected, ImageView button) {
        if (null == button) return;
        View parentView = (View) button.getParent();
        if (null == parentView) return;
        parentView.setSelected(isSelected);
        FrameLayout.LayoutParams params = (FrameLayout.LayoutParams) button.getLayoutParams();
        if (params == null) return;
        params.gravity = isSelected ? Gravity.END | Gravity.CENTER_VERTICAL : Gravity.START | Gravity.CENTER_VERTICAL;
        button.setLayoutParams(params);
    }

    public static float pointsToInches(float points) {
        // 1 Inch = 72 Points = 2.54 Centimeter
        return points / 72;
    }

    public static float pointsToCM(float points) {
        // 1 Inch = 72 Points = 2.54 Centimeter
        return points / 72 * 2.54f;
    }

    public static float pointsToMM(float points) {
        // 1 Inch = 72 Points = 2.54 Centimeter
        return points / 72 * 25.4f;
    }

    public static Point getThumbnailBackgroundSize(PDFViewCtrl pdfView) {
        if (pdfView == null) return null;
        float dpi = pdfView.getContext().getResources().getDisplayMetrics().densityDpi;
        if (dpi == 0) {
            dpi = 240;
        }
        float scale;
        try {
            PDFPage page = pdfView.getDoc().getPage(0);
            float width = page.getWidth();
            float height = page.getHeight();
            scale = width > height ? height / width : width / height;
//                mPDFView.getDoc().closePage(0);
        } catch (PDFException e) {
            scale = 0.8f;
        }

        int thumbnailHeight = (int) (dpi * 0.8f / scale) > dpi ? (int) dpi : (int) (dpi * 0.8f / scale);
        return new Point((int) (dpi * 0.8f), thumbnailHeight);
    }

    public static boolean isDarkMode(Context context) {
        int applicationUiMode = context.getResources().getConfiguration().uiMode;
        int systemMode = applicationUiMode & Configuration.UI_MODE_NIGHT_MASK;
        return systemMode == Configuration.UI_MODE_NIGHT_YES;
    }

    @SuppressLint("StaticFieldLeak")
    private static Context mApplicationContext = null;

    public static void initContext(Context applicationContext) {
        mApplicationContext = applicationContext;
    }

    public static Context getApplicationContext() {
        return mApplicationContext;
    }

    public static boolean isWhiteColorRange(int color) {
        int a = (color & 0xff000000) >>> 24;
        int r = (color & 0x00ff0000) >> 16;
        int g = (color & 0x0000ff00) >> 8;
        int b = color & 0x000000ff;

        return (a > 214 && r > 214 & g > 214 & b > 214);
    }

    public static String getToolbarItemString(Context context, int type) {
        String str = "";
        switch (type) {
            //Editor
            case ToolConstants.Text:
                str = AppResource.getString(context, R.string.edit_tool_text_type);
                break;
            case ToolConstants.Image:
                str = AppResource.getString(context, R.string.screen_annot_image);
                break;
            case ToolConstants.Audio:
                str = AppResource.getString(context, R.string.screen_annot_audio);
                break;
            case ToolConstants.Video:
                str = AppResource.getString(context, R.string.screen_annot_video);
                break;
            case ToolConstants.Link:
                str = AppResource.getString(context, R.string.edit_tool_link_type);
                break;
            case ToolConstants.CreateSignature:
                str = AppResource.getString(context, R.string.esign_create_title);
                break;
            case ToolConstants.ManageSignatures:
                str = AppResource.getString(context, R.string.esign_manage_title);
                break;
            case ToolConstants.ApplyAllSignatures:
                str = AppResource.getString(context, R.string.esign_apply_all_title);
                break;
            case ToolConstants.RequestSignature:
                str = AppResource.getString(context, R.string.esign_request_title);
                break;
            case ToolConstants.DocumentStaus:
                str = AppResource.getString(context, R.string.esign_document_status_title);
                break;
            case ToolConstants.SwitchAccount:
                str = AppResource.getString(context, R.string.esign_switch_account_title);
                break;
            //Fill & sign
            case ToolConstants.FillSignTypeText:
                str = AppResource.getString(context, R.string.fx_string_add_text);
                break;
            case ToolConstants.FillSignTypeComboText:
                str = AppResource.getString(context, R.string.fill_sign_tool_comb_field_type);
                break;
            case ToolConstants.FillSignTypeProFile:
                str = AppResource.getString(context, R.string.fillsign_profile_title);
                break;
            case ToolConstants.FillSignTypeCheckMark:
                str = AppResource.getString(context, R.string.fill_sign_tool_add_checkmark_type);
                break;
            case ToolConstants.FillSignTypeCrossMark:
                str = AppResource.getString(context, R.string.fill_sign_tool_addx_type);
                break;
            case ToolConstants.FillSignTypeDot:
                str = AppResource.getString(context, R.string.fill_sign_tool_add_not_type);
                break;
            case ToolConstants.FillSignTypeLine:
                str = AppResource.getString(context, R.string.fill_sign_tool_add_line_type);
                break;
            case ToolConstants.FillSignTypeRectangle:
                str = AppResource.getString(context, R.string.fill_sign_tool_add_rounded_rectangle_type);
                break;
            case ToolConstants.DigitalSignature:
                str = AppResource.getString(context, R.string.sg_signer_title);
                break;
            //Form
            case ToolConstants.TextField:
                str = AppResource.getString(context, R.string.form_tool_text_type);
                break;
            case ToolConstants.CheckBox:
                str = AppResource.getString(context, R.string.form_tool_check_type);
                break;
            case ToolConstants.RadioButton:
                str = AppResource.getString(context, R.string.form_tool_radio_type);
                break;
            case ToolConstants.ComboBox:
                str = AppResource.getString(context, R.string.form_tool_combo_type);
                break;
            case ToolConstants.ListBox:
                str = AppResource.getString(context, R.string.form_tool_list_type);
                break;
            case ToolConstants.SignatureField:
                str = AppResource.getString(context, R.string.form_tool_signature_type);
                break;
            case ToolConstants.ImageField:
                str = AppResource.getString(context, R.string.form_tool_image_type);
                break;
            //View
            case ToolConstants.TypeSinglePage:
                str = AppResource.getString(context, R.string.bottom_bar_view_setting);
                break;
            case ToolConstants.TypeDay:
                str = AppResource.getString(context, R.string.view_day_mode);
                break;
            case ToolConstants.TypeNight:
                str = AppResource.getString(context, R.string.ml_day_night);
                break;
            case ToolConstants.TypePageColor:
                str = AppResource.getString(context, R.string.view_setting_page_color);
                break;
            case ToolConstants.TypeFitPage:
                str = AppResource.getString(context, R.string.rd_fitpage_topbar_title);
                break;
            case ToolConstants.TypeFitWidth:
                str = AppResource.getString(context, R.string.rd_fitwidth_topbar_title);
                break;
            case ToolConstants.TypeReflow:
                str = AppResource.getString(context, R.string.rd_reflow_topbar_title);
                break;
            case ToolConstants.TypeCrop:
                str = AppResource.getString(context, R.string.rd_crop_topbar_title);
                break;
            case ToolConstants.TypeTts:
                str = AppResource.getString(context, R.string.rd_tts_speak);
                break;
            case ToolConstants.TypeAutoFlip:
                str = AppResource.getString(context, R.string.view_setting_auto_flip);
                break;
            case ToolConstants.TypeRotateView:
                str = AppResource.getString(context, R.string.rd_rotateview_topbar_title);
                break;
            case ToolConstants.TypePanZoom:
                str = AppResource.getString(context, R.string.rd_panzoom_topbar_title);
                break;
            //Protect
            case ToolConstants.ProtectTypeRedaction:
                str = AppResource.getString(context, R.string.fx_string_redaction);
                break;
            case ToolConstants.ProtectTypeStandard:
                str = AppResource.getString(context, R.string.menu_more_file_encryption);
                break;
            case ToolConstants.ProtectTypeTrustCert:
                str = AppResource.getString(context, R.string.menu_more_item_trust_certificate);
                break;
            case ToolConstants.ProtectTypeStandRemove:
            case ToolConstants.ProtectTypeCertRemove:
                str = AppResource.getString(context, R.string.rv_doc_encrpty_standard_remove);
                break;
            case ToolConstants.ProtectTypeCert:
                str = AppResource.getString(context, R.string.menu_more_item_cert_protect);
                break;
            case ToolConstants.ProtectTypeRMS:
                str = AppResource.getString(context, R.string.menu_more_item_rms);
                break;
            case ToolConstants.ProtectTypeAIP:
                str = AppResource.getString(context, R.string.menu_more_item_aip);
                break;

        }

        return str;
    }


    public static String getVersionName(Context context) {
        String versionName = "";
        try {
            PackageManager pm = context.getPackageManager();
            PackageInfo pi = pm.getPackageInfo(context.getPackageName(), 0);
            versionName += pi.versionName;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return versionName;
    }

    // check if the network is available
    public static boolean isNetworkAvailable(Context context) {
        ConnectivityManager manager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        boolean isMobileAvailable = (manager.getNetworkInfo(ConnectivityManager.TYPE_MOBILE) != null)
                && (manager.getNetworkInfo(ConnectivityManager.TYPE_MOBILE).isConnected());
        boolean isWifiAvilable = (manager.getNetworkInfo(ConnectivityManager.TYPE_WIFI) != null)
                && (manager.getNetworkInfo(ConnectivityManager.TYPE_WIFI).isConnected());
        boolean isWiMAXAvilable = (manager.getNetworkInfo(ConnectivityManager.TYPE_WIMAX) != null)
                && (manager.getNetworkInfo(ConnectivityManager.TYPE_WIMAX).isConnected());
        return isMobileAvailable || isWifiAvilable || isWiMAXAvilable;
    }

}

