/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.utils;

import android.content.Context;
import android.content.res.ColorStateList;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.PorterDuff;
import android.graphics.drawable.Drawable;
import android.os.Build;

import androidx.annotation.CheckResult;
import androidx.annotation.ColorInt;
import androidx.annotation.Nullable;
import androidx.core.graphics.drawable.DrawableCompat;

import com.foxit.uiextensions.R;

import java.util.Locale;

public class AppResource {
    public static String getString(Context context, int id) {
        return context.getApplicationContext().getString(id);
    }

    public static String getString(Context context, int resId, Object... formatArgs) {
        return context.getApplicationContext().getString(resId, formatArgs);
    }

    public static int getDimensionPixelSize(Context context, int id) {
        if (context == null || context.getResources() == null)
            return 0;
        return context.getResources().getDimensionPixelSize(id);
    }

    public static float getDimension(Context context, int id) {
        if (context == null || context.getResources() == null)
            return 0;
        return context.getResources().getDimension(id);
    }

    public static Drawable getDrawable(Context context, int id) {
        return context.getResources().getDrawable(id);
    }

    public static Drawable getDrawable(Context context, int id, @Nullable Resources.Theme theme) {
        if (Build.VERSION.SDK_INT >= 21) {
            //Build.VERSION_CODES.LOLLIPOP
            return context.getResources().getDrawable(id, theme);
        } else {
            return context.getResources().getDrawable(id);
        }
    }

    public static int getColor(Context context, int id) {
        return getColor(context, id, null);
    }

    public static int getColor(Context context, int id, @Nullable Resources.Theme theme) {
        if (Build.VERSION.SDK_INT >= 23) {
            // Build.VERSION_CODES.M
            return context.getResources().getColor(id, theme);
        } else {
            return context.getResources().getColor(id);
        }
    }

    public static ColorStateList createColorStateList(int pressed,
                                                      int selected,
                                                      int checked,
                                                      int disabled,
                                                      int normal) {
        int[] colors = new int[]{pressed,
                selected,
                checked,
                disabled,
                normal};
        int[][] states = new int[colors.length][];
        states[0] = new int[]{android.R.attr.state_pressed};
        states[1] = new int[]{android.R.attr.state_selected, android.R.attr.state_enabled};
        states[2] = new int[]{android.R.attr.state_checked};
        states[3] = new int[]{-android.R.attr.state_enabled};
        states[4] = new int[]{};
        return new ColorStateList(states, colors);
    }

    public static ColorStateList createColorStateList(int selected,
                                                      int disabled,
                                                      int normal) {
        int[] colors = new int[]{
                selected,
                disabled,
                normal};
        int[][] states = new int[colors.length][];

        states[0] = new int[]{android.R.attr.state_selected, android.R.attr.state_enabled};
        states[1] = new int[]{-android.R.attr.state_enabled};
        states[2] = new int[]{};
        return new ColorStateList(states, colors);
    }

    public static ColorStateList createColorStateList(Context context,
                                                      int disabled,
                                                      int normal) {
        int[] colors = new int[]{
                normal,
                disabled};
        int[][] states = new int[colors.length][];

        states[0] = new int[]{android.R.attr.state_enabled};
        states[1] = new int[]{-android.R.attr.state_enabled};
        return new ColorStateList(states, colors);
    }


    @CheckResult
    @Nullable
    public static Drawable createTintedDrawable(@Nullable Drawable drawable, @ColorInt int color) {
        if (drawable == null) return null;
        drawable = DrawableCompat.wrap(drawable).mutate();
        DrawableCompat.setTintMode(drawable, PorterDuff.Mode.SRC_IN);
        DrawableCompat.setTint(drawable, color);
        return drawable;
    }

    @CheckResult
    @Nullable
    public static Drawable createTintedDrawable(@Nullable Drawable drawable, @Nullable ColorStateList sl) {
        if (drawable == null) return null;
        drawable = DrawableCompat.wrap(drawable).mutate();
        DrawableCompat.setTintList(drawable, sl);
        return drawable;
    }

    /**
     * get drawable resource based on the file path name
     * @param name file path name
     * @return drawable resource id
     */
    public static int getDrawableByFileName(String name) {
        if (name == null || name.length() == 0) {
            return R.drawable.fb_file_other;
        }
        String extension = name.substring(name.lastIndexOf(".") + 1).toLowerCase(Locale.ROOT);
        if (extension.length() == 0) {
            return R.drawable.fb_file_other;
        } else if (extension.equals("ofd")) {
            return R.drawable.fb_file_ofd;
        } else if (extension.equals("ppdf")) {
            return R.drawable.fb_file_ppdf;
        } else if (extension.equals("doc") || extension.equals("docx")) {
            return R.drawable.fb_file_doc;
        } else if (extension.equals("xls") || extension.equals("xlsx")) {
            return R.drawable.fb_file_xls;
        } else if (extension.equals("jpg")
                || extension.equals("jpeg")
                || extension.equals("png")
                || extension.equals("bmp")
                || extension.equals("gif")) {
            return R.drawable.fb_file_picture;
        } else if (extension.equals("txt")) {
            return R.drawable.fb_file_txt;
        } else if (extension.equals("html") || extension.equals("xml")) {
            return R.drawable.fb_file_html;
        } else if (extension.equals("pdf")) {
            return -1;
        } else if (extension.equals("ppt")) {
            return R.drawable.fb_file_ppt;
        } else if (extension.equals("zip") || extension.equals("gz") || extension.equals("rar")) {
            return R.drawable.fb_file_zip;
        }
        return R.drawable.fb_file_other;
    }

    public static Bitmap getBitmap(Context context, int id){
        return BitmapFactory.decodeResource(context.getResources(),id);
    }

    public static Bitmap getBitmapFromDrawable(Context context, int id) {
        Bitmap bitmap;
        Drawable drawable = getDrawable(context,id);
        bitmap = Bitmap.createBitmap(drawable.getIntrinsicWidth(), drawable.getIntrinsicHeight(), Bitmap.Config.ARGB_8888);
        Canvas canvas = new Canvas(bitmap);
        drawable.setBounds(0, 0, canvas.getWidth(), canvas.getHeight());
        drawable.draw(canvas);
        return bitmap;
    }
}