/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.security.trustcertificate;


import android.app.Activity;
import android.content.Context;
import android.graphics.Rect;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.ImageView;
import android.widget.TextView;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.common.UIBtnImageView;
import com.foxit.uiextensions.browser.treeview.TreeNode;
import com.foxit.uiextensions.controls.dialog.UIDialog;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.controls.dialog.sheetmenu.ISheetMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UIBottomSheetMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UISheetMenu;
import com.foxit.uiextensions.security.certificate.CertificateFileInfo;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.SystemUiHelper;

import androidx.fragment.app.FragmentActivity;

public class ItemCertViewerHolder extends TreeNode.BaseNodeViewHolder<CertificateFileInfo> implements View.OnClickListener {

    private View mItemView;
    private ImageView toggleView;

    private TreeNode mNode;
    private CertificateFileInfo mFileInfo;
    private TrustCertificatePresenter mPresenter;
    private PDFViewCtrl mPDFViewCtrl;
    private ISheetMenu mSheetMenu;
    private OnSheetMenuListener mSheetMenuListener;

    public ItemCertViewerHolder(Context context, PDFViewCtrl pdfViewCtrl) {
        super(context);
        mPresenter = new TrustCertificatePresenter(context, pdfViewCtrl);
        mPDFViewCtrl = pdfViewCtrl;
    }

    @Override
    public View createNodeView(TreeNode node, CertificateFileInfo value) {
        mFileInfo = value;
        mNode = node;

        LayoutInflater inflater = LayoutInflater.from(mContext);
        mItemView = inflater.inflate(R.layout.cert_viewer_item, null, false);
        TextView tvTitle = mItemView.findViewById(R.id.tv_certificate_title);
        TextView tvDate = mItemView.findViewById(R.id.tv_certificate_date);
        UIBtnImageView ivMore = mItemView.findViewById(R.id.iv_cert_more);
        ivMore.setColorStateList(ThemeUtil.getPrimaryIconColor(mContext));
        toggleView = mItemView.findViewById(R.id.iv_toggle_icon);

        tvTitle.setText(value.subject);
        tvDate.setText(AppDmUtil.formatStrDate(AppDmUtil.FORMAT_MMM_DD_YYYY_HH_MM_SS,
                AppDmUtil.FORMAT_MMM_DD_YYYY_HH_MM,
                value.validTo));
        if (node.isLeaf()) {
            if (node.getParent() != null && node.getParent().getValue() != null)
                toggleView.setVisibility(View.INVISIBLE);
            else
                toggleView.setVisibility(View.GONE);
        } else
            toggleView.setVisibility(View.VISIBLE);

        ivMore.setOnClickListener(this);
        toggleView.setOnClickListener(this);
        return mItemView;
    }

    @Override
    public void toggle(boolean active) {
        toggleView.setImageResource(active ? R.drawable.rd_collapse_arrow : R.drawable.rd_expand_arrow);
    }

    @Override
    public void onClick(final View v) {
        int id = v.getId();
        if (id == R.id.iv_cert_more) {
            boolean isPad = AppDisplay.isPad();
            mItemView.setSelected(!isPad);
            final UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPDFViewCtrl.getUIExtensionsManager();
            mSheetMenu = UISheetMenu.newInstance((FragmentActivity) uiExtensionsManager.getAttachedActivity());
            if (isPad)
                mSheetMenu.setWidth(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_pad_more_menu_width));
            mSheetMenu.addSheetItem(ISheetMenu.INFO_SIGNATURE_MENU, true);
            if (!mFileInfo.isTrustCert)
                mSheetMenu.addSheetItem(ISheetMenu.TRUST_SIGNATURE_MENU);
            mSheetMenu.setSheetItemClickListener(new UIBottomSheetMenu.OnSheetItemClickListener() {
                @Override
                public void onItemClick(int type) {
                    if (ISheetMenu.INFO_SIGNATURE_MENU == type) {
                        mSheetMenu.dismiss();
                        mPresenter.viewCertInfo(mFileInfo);
                    } else if (ISheetMenu.TRUST_SIGNATURE_MENU == type) {
                        mSheetMenu.dismiss();

                        Activity activity = uiExtensionsManager.getAttachedActivity();
                        final UITextEditDialog dialog = new UITextEditDialog(activity, UIDialog.NO_INPUT);
                        dialog.setTitle(AppResource.getString(mContext, R.string.menu_more_item_trust_certificate));
                        dialog.getPromptTextView().setText(AppResource.getString(mContext, R.string.rv_add_trust_certificate_prompt));
                        dialog.getOKButton().setOnClickListener(new View.OnClickListener() {
                            @Override
                            public void onClick(View v) {
                                mPresenter.addTrustCert(mFileInfo, new Event.Callback() {
                                    @Override
                                    public void result(Event event, boolean success) {
                                        dialog.dismiss();
                                        if (success) {
                                            mFileInfo.isTrustCert = true;
                                        }
                                    }
                                });
                            }
                        });
                        dialog.show();
                    }
                }
            });
            mSheetMenu.setOnSheetDismissListener(new ISheetMenu.OnSheetDismissListener() {
                @Override
                public void onDismiss(ISheetMenu sheetMenu) {
                    mItemView.setSelected(false);
                    if (mSheetMenuListener != null)
                        mSheetMenuListener.onSheetMenuDismiss(mSheetMenu, v);
                }
            });

            Rect rect = new Rect();
            v.getGlobalVisibleRect(rect);
            if (SystemUiHelper.getInstance().isStatusBarShown(uiExtensionsManager.getAttachedActivity()))
                rect.offset(0, SystemUiHelper.getInstance().getStatusBarHeight(uiExtensionsManager.getAttachedActivity()));
            if (mSheetMenuListener != null)
                mSheetMenu.setAutoResetSystemUiOnShow(mSheetMenuListener.isFullScreen());
            mSheetMenu.show(uiExtensionsManager.getRootView(), rect);
            if (mSheetMenuListener != null)
                mSheetMenuListener.onSheetMenuShow(mSheetMenu, v);
        } else if (id == R.id.iv_toggle_icon) {
            getTreeView().toggleNode(mNode);
        }
    }

    ItemCertViewerHolder setSheetMenuListener(OnSheetMenuListener sheetListener) {
        mSheetMenuListener = sheetListener;
        return this;
    }

    interface OnSheetMenuListener {
        void onSheetMenuShow(ISheetMenu sheetMenu, View v);

        void onSheetMenuDismiss(ISheetMenu sheetMenu, View v);

        boolean isFullScreen();
    }

}
