/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.security.digitalsignature;

import android.app.Activity;
import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.graphics.Bitmap;
import android.graphics.RectF;
import android.text.TextUtils;
import android.widget.Toast;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.DateTime;
import com.foxit.sdk.common.Progressive;
import com.foxit.sdk.common.Renderer;
import com.foxit.sdk.common.fxcrt.Matrix2D;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.Signature;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.modules.signature.SignatureInkItem;
import com.foxit.uiextensions.security.certificate.CertificateDataSupport;
import com.foxit.uiextensions.security.certificate.CertificateFileInfo;
import com.foxit.uiextensions.security.certificate.CertificateFragment;
import com.foxit.uiextensions.security.certificate.CertificateSupport;
import com.foxit.uiextensions.security.certificate.CertificateViewSupport;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppSQLite;
import com.foxit.uiextensions.utils.AppStorageManager;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.SystemUiHelper;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Calendar;
import java.util.TimeZone;

public class DigitalSignatureUtil implements IDigitalSignatureUtil {
    private static final String DB_TABLE_DSG_PFX = "_pfx_dsg_cert";
    private static final String PUBLISHER = "publisher";
    private static final String ISSUER = "issuer";
    private static final String SERIALNUMBER = "serial_number";
    private static final String FILEPATH = "file_path";
    private static final String CHANGEFILEPATH = "file_change_path";
    private static final String FILENAME = "file_name";
    private static final String PASSWORD = "password";
    public static final int FULLPERMCODE = 0xf3c;
    public DigitalSignatureSecurityHandler mSecurityHandler;
    public Context mContext;
    public PDFViewCtrl mPdfViewCtrl;
    public CertificateSupport mCertSupport;
    public CertificateViewSupport mViewSupport;
    public CertificateFileInfo mFileInfo;

    public DigitalSignatureUtil(Context context, PDFViewCtrl pdfViewCtrl) {
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mCertSupport = new CertificateSupport(mContext);
        mViewSupport = new CertificateViewSupport(context, pdfViewCtrl, mCertSupport);
        mSecurityHandler = new DigitalSignatureSecurityHandler(mContext, mPdfViewCtrl, this);
    }

    public CertificateDataSupport getCertDataSupport() {
        return mViewSupport.getDataSupport();
    }

    public CertificateSupport getCertSupport() {
        return mViewSupport.getCertSupport();
    }

    @Override
    public void addCertSignature(final String docPath, final String certPath, final Bitmap bitmap, final RectF rectF, final int pageIndex, final SignatureInkItem inkItem, final IDigitalSignatureCreateCallBack callBack) {
        AppThreadManager.getInstance().startThread(new Runnable() {
            @Override
            public void run() {
                Cursor cursor = AppSQLite.getInstance(mContext).select(DB_TABLE_DSG_PFX, null, null, null, null, null, null);
                if (cursor != null) {
                    while (cursor.moveToNext()) {
                        String newPath = certPath + "x";
                        if (newPath.equals(cursor.getString(cursor.getColumnIndex(CHANGEFILEPATH)))) {
                            copyFile(mContext, newPath, cursor.getString(cursor.getColumnIndex(FILEPATH)));
                            final CertificateFileInfo info = new CertificateFileInfo();
                            info.serialNumber = cursor.getString(cursor.getColumnIndex(SERIALNUMBER));
                            info.issuer = cursor.getString(cursor.getColumnIndex(ISSUER));
                            info.publisher = cursor.getString(cursor.getColumnIndex(PUBLISHER));
                            info.filePath = cursor.getString(cursor.getColumnIndex(FILEPATH));
                            info.fileName = cursor.getString(cursor.getColumnIndex(FILENAME));
                            info.password = cursor.getString(cursor.getColumnIndex(PASSWORD));
                            info.isCertFile = false;
                            info.permCode = FULLPERMCODE;
                            AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
                                @Override
                                public void run() {
                                    DigitalSignatureUtil.this.creatDSGSign(bitmap, docPath, pageIndex, inkItem, rectF, callBack, info);
                                }
                            });
                            break;
                        }
                    }
                    cursor.close();
                }
            }
        });

    }

    public CertificateFileInfo getCertInfo(String certPath) {
        CertificateFileInfo info = null;
        Cursor cursor = AppSQLite.getInstance(mContext).select(DB_TABLE_DSG_PFX, null, null, null, null, null, null);
        if (cursor != null) {
            while (cursor.moveToNext()) {
                String newPath = certPath + "x";
                if (newPath.equals(cursor.getString(cursor.getColumnIndex(CHANGEFILEPATH)))) {
                    info = new CertificateFileInfo();
                    info.serialNumber = cursor.getString(cursor.getColumnIndex(SERIALNUMBER));
                    info.issuer = cursor.getString(cursor.getColumnIndex(ISSUER));
                    info.publisher = cursor.getString(cursor.getColumnIndex(PUBLISHER));
                    info.filePath = cursor.getString(cursor.getColumnIndex(FILEPATH));
                    info.fileName = cursor.getString(cursor.getColumnIndex(FILENAME));
                    info.password = cursor.getString(cursor.getColumnIndex(PASSWORD));
                    info.isCertFile = false;
                    info.permCode = FULLPERMCODE;
                    break;
                }
            }
            cursor.close();
        }
        return info;
    }

    @Override
    public void addCertSignature(final String docPath, final String certPath, final Signature signature, final RectF rectF, final int pageIndex, final boolean isCustom, final SignatureInkItem inkItem, final IDigitalSignatureCreateCallBack callBack) {
        AppThreadManager.getInstance().startThread(new Runnable() {
            @Override
            public void run() {
                Cursor cursor = AppSQLite.getInstance(mContext).select(DB_TABLE_DSG_PFX, null, null, null, null, null, null);
                if (cursor != null) {
                    while (cursor.moveToNext()) {
                        String newPath = certPath + "x";
                        if (newPath.equals(cursor.getString(cursor.getColumnIndex(CHANGEFILEPATH)))) {

                            copyFile(mContext, newPath, cursor.getString(cursor.getColumnIndex(FILEPATH)));
                            final CertificateFileInfo info = new CertificateFileInfo();
                            info.serialNumber = cursor.getString(cursor.getColumnIndex(SERIALNUMBER));
                            info.issuer = cursor.getString(cursor.getColumnIndex(ISSUER));
                            info.publisher = cursor.getString(cursor.getColumnIndex(PUBLISHER));
                            info.filePath = cursor.getString(cursor.getColumnIndex(FILEPATH));
                            info.fileName = cursor.getString(cursor.getColumnIndex(FILENAME));
                            info.password = cursor.getString(cursor.getColumnIndex(PASSWORD));
                            info.isCertFile = false;
                            info.permCode = FULLPERMCODE;
                            AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
                                @Override
                                public void run() {
                                    DigitalSignatureUtil.this.creatDSGSign(docPath, pageIndex, signature, rectF, isCustom, inkItem, callBack, info);
                                }
                            });
                            break;
                        }
                    }
                    cursor.close();
                }
            }
        });

    }

    public void dismissPfxDialog() {
        if (mViewSupport != null) {
            mViewSupport.dismissPfxDialog();
        }
    }

    @Override
    public void addCertList(final IDigitalSignatureCallBack callBack) {
        Activity act = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAttachedActivity();
        addCertList(!SystemUiHelper.getInstance().isStatusBarShown(act), callBack);
    }

    @Override
    public void addCertList(final boolean isFullScreen, final IDigitalSignatureCallBack callBack) {
        AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
            @Override
            public void run() {
                mViewSupport.showAllPfxFileDialog(isFullScreen, false, true, true, new CertificateFragment.ICertDialogCallback() {
                    @Override
                    public void result(boolean succeed, Object result, Bitmap forSign) {
                        if (succeed) {
                            if (result == null) return;
                            mFileInfo = (CertificateFileInfo) result;

                            String filePath = mContext.getFilesDir() + "/DSGCert";
                            File file = new File(filePath);
                            if (!file.exists()) {
                                file.mkdirs();
                            }
                            String newCertPath = filePath + File.separator + mFileInfo.fileName;
                            String changeCertPath = newCertPath + "x";
                            copyFile(mContext, mFileInfo.filePath, newCertPath);
                            ContentValues values = new ContentValues();
                            values.put(ISSUER, mFileInfo.issuer);
                            values.put(PUBLISHER, mFileInfo.publisher);
                            values.put(SERIALNUMBER, mFileInfo.serialNumber);
                            values.put(FILEPATH, newCertPath);
                            values.put(CHANGEFILEPATH, changeCertPath);
                            values.put(FILENAME, mFileInfo.fileName);
                            values.put(PASSWORD, mFileInfo.password);
                            Cursor cursor = AppSQLite.getInstance(mContext).select(DB_TABLE_DSG_PFX, null, null, null, null, null, null);

                            AppSQLite.getInstance(mContext).insert(DB_TABLE_DSG_PFX, values);
                            if (callBack != null) {
                                callBack.onCertSelect(newCertPath, mFileInfo);
                            }
                        } else {
                            mViewSupport.dismissPfxDialog();
                            if (callBack != null) {
                                callBack.onCertSelect(null, null);
                            }
                        }
                    }
                });
            }
        });
    }

    public void creatDSGSign(Bitmap bitmap, final String docPath, final int pageIndex, SignatureInkItem inkItem, final RectF rectF, final IDigitalSignatureCreateCallBack callBack, final CertificateFileInfo info) {

        mSecurityHandler.addSignature(docPath, info, bitmap, pageIndex, inkItem, rectF, new Event.Callback() {
            @Override
            public void result(Event event, boolean success) {
                if (mPdfViewCtrl.getUIExtensionsManager() instanceof UIExtensionsManager) {
                    ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).onDigitalSignatureSigned(success);
                }
                if (success) {
                    Toast.makeText(mContext, AppResource.getString(mContext, R.string.dsg_sign_succeed), Toast.LENGTH_SHORT).show();
                } else {
                    Toast.makeText(mContext, AppResource.getString(mContext, R.string.dsg_sign_failed), Toast.LENGTH_SHORT).show();
                }
                callBack.onCreateFinish(success);
            }
        });

    }

    public void creatDSGSign(final String docPath, final int pageIndex, Signature signature, final RectF rectF, boolean isCustom, SignatureInkItem inkItem, final IDigitalSignatureCreateCallBack callBack, final CertificateFileInfo info) {

        mSecurityHandler.addSignature(docPath, info, pageIndex, signature, isCustom, inkItem, new Event.Callback() {
            @Override
            public void result(Event event, boolean success) {
                if (mPdfViewCtrl.getUIExtensionsManager() instanceof UIExtensionsManager) {
                    ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).onDigitalSignatureSigned(success);
                }
                if (success) {
                    Toast.makeText(mContext, AppResource.getString(mContext, R.string.dsg_sign_succeed), Toast.LENGTH_SHORT).show();
                } else {
                    Toast.makeText(mContext, AppResource.getString(mContext, R.string.dsg_sign_failed), Toast.LENGTH_SHORT).show();
                }
                callBack.onCreateFinish(success);
            }
        });

    }


    public static void copyFile(Context context, String oldPath, String newPath) {
        InputStream inStream = null;
        FileOutputStream fs = null;
        try {
            int byteread;
            File file = new File(newPath);
            if (!file.exists()) {
                try {
                    inStream = new FileInputStream(oldPath);
                } catch (Exception e) {
                    inStream = context.getContentResolver().openInputStream(AppFileUtil.toDocumentUriFromPath(oldPath));
                }
                fs = new FileOutputStream(newPath);
                byte[] buffer = new byte[1444];
                while ((byteread = inStream.read(buffer)) != -1) {
                    fs.write(buffer, 0, byteread);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                if (inStream != null) {
                    inStream.close();
                }

                if (fs != null) {
                    fs.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    public void createSignature(RectF desRect, SignatureInkItem inkItem, Bitmap bitmap, Bitmap tempBitmap,boolean isNewDate) {
        String path = AppStorageManager.getInstance(mContext).getCacheDir() + File.separator;
        try {
            PDFDoc doc = new PDFDoc();
            PDFPage pdfPage = doc.insertPage(0, Math.abs(desRect.width()), Math.abs(desRect.height()));
            CertificateFileInfo fileInfo = getCertInfo(inkItem.getDsgPath());
            boolean isSignInfo = inkItem != null && !TextUtils.isEmpty(inkItem.getTitle());
            String filter = "Adobe.PPKLite";
            String subfilter = "adbe.pkcs7.detached";
            String dn = "";
            String location = "";
            String reason = "";
            String contactInfo = "";
            String signer = "";
            String text = "text";
            long state = 0;
            //set current time to dateTime.
            DateTime dateTime = new DateTime();
            Calendar c = Calendar.getInstance();
            TimeZone timeZone = c.getTimeZone();
            int offset = timeZone.getRawOffset();
            int tzHour = offset / (3600 * 1000);
            int tzMinute = (offset / (1000 * 60)) % 60;
            int year = c.get(Calendar.YEAR);
            int month = c.get(Calendar.MONTH) + 1;
            int day = c.get(Calendar.DATE);
            int hour = c.get(Calendar.HOUR_OF_DAY);
            int minute = c.get(Calendar.MINUTE);
            int second = c.get(Calendar.SECOND);
            if(isNewDate){
                dateTime.set(year, month, day, hour, minute, second, 0, (short) tzHour, tzMinute);
            }else{
                try {
                    if(!TextUtils.isEmpty(inkItem.getDate())){
                        String date = inkItem.getDate();
                        year = Integer.parseInt(date.substring(0,4));
                        month = Integer.parseInt(date.substring(5,7));
                        day = Integer.parseInt(date.substring(8,10));
                        hour = Integer.parseInt(date.substring(11,13));
                        minute = Integer.parseInt(date.substring(14,16));
                        second = Integer.parseInt(date.substring(17,19));
                        tzHour = Integer.parseInt(date.substring(20,22));
                        tzMinute = Integer.parseInt(date.substring(23,25));
                    }
                    dateTime.set(year, month, day, hour, minute, second, 0, (short) tzHour, tzMinute);
                }catch (Exception e){
                    dateTime.set(year, month, day, hour, minute, second, 0, (short) tzHour, tzMinute);
                }
            }
            Signature signature = null;
            RectF rectF = new RectF();
            rectF.set(0, pdfPage.getHeight(), pdfPage.getWidth(), 0);
            if (isSignInfo) {
                signature = pdfPage.addSignature(AppUtil.toFxRectF(rectF));
                signature.setBitmap(bitmap);
            }
            if (isSignInfo) {
                if (inkItem.isNameOpen()) {
                    signer = inkItem.getName();
                }
                if (inkItem.isDistinguishedNameOpen()) {
                    dn = inkItem.getDistinguishedName();
                }
                if (inkItem.isReasonOpen()) {
                    String textReason = getReason(inkItem.getReason());
                    reason = textReason;
                }
                if (inkItem.isLocationOpen()) {
                    String textLocation = "";
                    if (!TextUtils.isEmpty(inkItem.getLocation())) {
                        textLocation = inkItem.getLocation();
                    }
                    location = textLocation;
                }

                signature.setFilter(filter);
                signature.setSubFilter(subfilter);
                signature.setKeyValue(Signature.e_KeyNameDN, dn);
                signature.setKeyValue(Signature.e_KeyNameLocation, location);
                signature.setKeyValue(Signature.e_KeyNameReason, reason);
                signature.setKeyValue(Signature.e_KeyNameContactInfo, contactInfo);
                signature.setKeyValue(Signature.e_KeyNameSigner, signer);
                signature.setKeyValue(Signature.e_KeyNameText, text);
                signature.setSignTime(dateTime);
//                PDFDictionary dictionary = signature.getSignatureDict();
//                dictionary.setAtString("M", AppDmUtil.currentDateToDocumentDateString());

                StringBuffer stringBuffer = new StringBuffer();
                if (inkItem.isLabels()) {
                    if (((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAPPInfoProvider() != null) {
                        stringBuffer.append(((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAPPInfoProvider().getAppName());
                        stringBuffer.append(" ");
                        stringBuffer.append(mContext.getApplicationContext().getString(R.string.appearance_version));
                    }
                }
                if (!TextUtils.isEmpty(inkItem.getVersion())) {
                    stringBuffer.append(inkItem.getVersion());
                } else {
                    if (((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAPPInfoProvider() != null) {
                        stringBuffer.append(((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAPPInfoProvider().getAppVersion());
                    }
                }
                signature.setKeyValue(Signature.e_KeyNameProducer, stringBuffer.toString());
            }
            long flags;
            flags = Signature.e_APFlagBitmap;
            if ((inkItem == null || TextUtils.isEmpty(inkItem.getTitle()))) {
                flags = Signature.e_APFlagBitmap;
            } else {
                if (inkItem.isNameOpen()) {
                    flags = flags | Signature.e_APFlagSigner;
                }
                if (inkItem.isReasonOpen()) {
                    flags = flags | Signature.e_APFlagReason;
                }
                if (inkItem.isLocationOpen()) {
                    flags = flags | Signature.e_APFlagLocation;
                }
                if (inkItem.isDistinguishedNameOpen()) {
                    flags = flags | Signature.e_APFlagDN;
                }
                if (inkItem.isDateOpen()) {
                    flags = flags | Signature.e_APFlagSigningTime;
                }
                if (inkItem.isLabels()) {
                    flags = flags | Signature.e_APFlagLabel;
                }
                if (inkItem.isLogo()) {
                    flags = flags | Signature.e_APFlagFoxitEditorFlag;
                }
                if (inkItem.isVersionOpen()) {
                    flags = flags | Signature.e_APFlagProducer;
                }
            }
            signature.setAppearanceFlags((int) flags);
            Progressive progressive = null;
            try {

                path = path + "tempSignature.pdf";
                File file = new File(path);
                if (file.exists()) {
                    file.delete();
                }
                file.createNewFile();
                progressive = signature.startSign(fileInfo.filePath, fileInfo.password.getBytes(), Signature.e_DigestSHA1, path, null, null);
            } catch (Exception ignored) {
                return;
            }
            int progress = Progressive.e_ToBeContinued;
            while (progress == Progressive.e_ToBeContinued) {
                assert progressive != null;
                progress = progressive.resume();
            }
            if (progress == Progressive.e_Error) {
            }

            state = signature.getState();
            if (state != Signature.e_StateSigned || !signature.isSigned()) ;

            PDFDoc docSignature = new PDFDoc(path);
            docSignature.load(null);
            PDFPage pdfPageSignature = docSignature.getPage(0);
            pdfPageSignature.startParse(PDFPage.e_ParsePageNormal, null, false);
            Matrix2D matrix = pdfPage.getDisplayMatrix(0, 0, (int) tempBitmap.getWidth(), (int) tempBitmap.getHeight(), 0);
            Renderer render = new Renderer(tempBitmap, true);
            render.startRender(pdfPageSignature, matrix, null);
            docSignature.delete();
            File file = new File(path);
            if (file.exists()) {
                file.delete();
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private String getReason(int reason){
        if(reason == 0){
            return mContext.getResources().getString(R.string.reason_author);
        }else if(reason == 1){
            return mContext.getResources().getString(R.string.reason_review);
        }else if(reason == 2){
            return mContext.getResources().getString(R.string.reason_approve);
        }else if(reason == 3){
            return mContext.getResources().getString(R.string.reason_approve_legally_binding);
        }else if(reason == 4){
            return mContext.getResources().getString(R.string.reason_accuracy_integrity);
        }else if(reason == 5){
            return mContext.getResources().getString(R.string.reason_sign_terms);
        }else if(reason == 6){
            return mContext.getResources().getString(R.string.reason_specified);
        }else{
            return mContext.getResources().getString(R.string.reason_author);
        }
    }
}