/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.security.certificate;

import android.content.Context;
import android.content.DialogInterface;
import android.content.res.Configuration;
import android.text.InputType;
import android.view.KeyEvent;
import android.view.View;
import android.view.inputmethod.EditorInfo;
import android.widget.EditText;

import androidx.fragment.app.FragmentActivity;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;

import java.util.ArrayList;
import java.util.List;

public class CertificateViewSupport {

    private List<CertificateFileInfo> mCertInfos;

    private Context mContext;
    private CertificateSupport mCertSupport;
    private CertificateDataSupport mDataSupport;
    public static final int MESSAGE_UPDATE = 0x11;
    public static final int MESSAGE_FINISH = 0x12;
    public static final int MESSAGE_HIDEALLPFXFILEDLG = 0x13;
    private boolean mDoEncrypt;
    private PDFViewCtrl mPdfViewCtrl;

    public CertificateViewSupport(Context context, PDFViewCtrl pdfViewCtrl, CertificateSupport support) {
        mCertInfos = new ArrayList<CertificateFileInfo>();
        mDataSupport = new CertificateDataSupport(context);

        mCertSupport = support;
        mPdfViewCtrl = pdfViewCtrl;
        mContext = context;
    }

    public CertificateDataSupport getDataSupport() {
        return mDataSupport;
    }

    public CertificateSupport getCertSupport() {
        return mCertSupport;
    }

    CertificateFragment mCertDialog;

    public void showAllPfxFileDialog(boolean isFullScreen,
                                     boolean autoResetSystemUi,
                                     boolean isOnlyPFX,
                                     boolean x,
                                     final CertificateFragment.ICertDialogCallback callback) {
        if (mPdfViewCtrl.getUIExtensionsManager() == null) return;
        Context context = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAttachedActivity();
        if (context == null) return;
        FragmentActivity act = (FragmentActivity) context;

        mCertDialog = new CertificateFragment(act);
        mCertDialog.setAutoResetSystemUiOnDismiss(autoResetSystemUi);
        if (isOnlyPFX) {
            if (x) {
                mCertDialog.init(this, callback, CertificateFragment.CERLIST_TYPE_SIGNATURE);
            } else {
                mCertDialog.init(this, callback, CertificateFragment.CERLIST_TYPE_DECRYPT);
            }
        } else {
            mCertDialog.init(this, callback, CertificateFragment.CERLIST_TYPE_ENCRYPT);
        }
        mDoEncrypt = !isOnlyPFX;
        if (!isFullScreen)
            mCertDialog.setFullScreenWithStatusBar();
        mCertDialog.showDialog(isFullScreen);
        mCertDialog.setCanceledOnTouchOutside(false);

    }


    public void dismissPfxDialog() {
        if (mPdfViewCtrl.getUIExtensionsManager() == null) return;

        if (mCertDialog != null && mCertDialog.isShowing()) {
            mCertDialog.dismiss();
        }
    }

    private UITextEditDialog mPasswordDialog;

    void showPasswordDialog(final CertificateFileInfo info, final CertificateFragment.ICertDialogCallback callback) {
        if (mPasswordDialog != null) {
            mPasswordDialog.dismiss();
        }
        if (mPdfViewCtrl.getUIExtensionsManager() == null) return;
        Context context = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAttachedActivity();
        if (context == null) return;
        mPasswordDialog = new UITextEditDialog(context);
        mPasswordDialog.getDialog().setCanceledOnTouchOutside(false);
        mPasswordDialog.getInputEditText().setInputType(InputType.TYPE_CLASS_TEXT | InputType.TYPE_TEXT_VARIATION_PASSWORD);
        if (AppDisplay.isPad()) { // SDKRD-8881
            mPasswordDialog.getInputEditText().setImeOptions(EditorInfo.IME_FLAG_NO_EXTRACT_UI);
        }
        mPasswordDialog.setTitle(AppResource.getString(mContext.getApplicationContext(), R.string.rv_password_dialog_title));
        mPasswordDialog.getPromptTextView().setText(AppResource.getString(mContext.getApplicationContext(), R.string.rv_security_certlist_inputpasswd));
        mPasswordDialog.show();
        AppUtil.showSoftInput(mPasswordDialog.getInputEditText());
        mPasswordDialog.getOKButton().setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                EditText editText = mPasswordDialog.getInputEditText();
                String psd = editText.getText().toString();
                info.certificateInfo = mCertSupport.verifyPassword(info.filePath, psd);
                if (info.certificateInfo != null) {
                    info.password = psd;
                    info.issuer = info.certificateInfo.issuer;
                    info.publisher = info.certificateInfo.publisher;
                    info.serialNumber = info.certificateInfo.serialNumber;
                    info.emailAddress = info.certificateInfo.emailAddress;
                    if (!mCertInfos.contains(info)) {
                        updateInfo(info);
                    }
                    mPasswordDialog.dismiss();
                    mPasswordDialog = null;
                    if (callback != null) {
                        callback.result(true, null, null);
                    }
                } else {
                    editText.setText("");
                    editText.setFocusable(true);
                    mPasswordDialog.getPromptTextView().setText(AppResource.getString(mContext.getApplicationContext(), R.string.rv_security_certlist_invalidpasswd));
                }
            }
        });

        mPasswordDialog.getCancelButton().setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                AppUtil.dismissInputSoft(mPasswordDialog.getInputEditText());
                mPasswordDialog.dismiss();
                mPasswordDialog = null;
                if (callback != null) {
                    callback.result(false, null, null);
                }
            }
        });

        mPasswordDialog.getDialog().setOnKeyListener(new DialogInterface.OnKeyListener() {
            @Override
            public boolean onKey(DialogInterface dialog, int keyCode, KeyEvent event) {
                EditText editText = mPasswordDialog.getInputEditText();
                if (keyCode == KeyEvent.KEYCODE_BACK) {
                    mPasswordDialog.dismiss();
                    mPasswordDialog = null;
                    if (callback != null) {
                        callback.result(false, null, null);
                    }
                    return true;
                } else if (keyCode == KeyEvent.KEYCODE_ENTER && event.getAction() == KeyEvent.ACTION_DOWN) {
                    String psd = editText.getText().toString();
                    editText.setFocusable(true);
                    info.certificateInfo = mCertSupport.verifyPassword(info.filePath, psd);
                    if (info.certificateInfo != null) {
                        info.password = psd;
                        info.issuer = info.certificateInfo.issuer;
                        info.publisher = info.certificateInfo.publisher;
                        info.serialNumber = info.certificateInfo.serialNumber;
                        if (!mCertInfos.contains(info)) {
                            updateInfo(info);
                        }
                        mPasswordDialog.dismiss();
                        mPasswordDialog = null;
                        if (callback != null) {
                            callback.result(true, null, null);
                        }
                    } else {
                        editText.setText("");
                        AppUtil.showSoftInput(editText);
                        mPasswordDialog.getPromptTextView().setText(AppResource.getString(mContext.getApplicationContext(), R.string.rv_security_certlist_invalidpasswd));
                    }
                }
                return false;
            }
        });
    }

    CertificateDetailDialog mDetailDialog;

    public void showPermissionDialog(final CertificateFileInfo info) {
        if (mPdfViewCtrl.getUIExtensionsManager() == null) return;
        Context context = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAttachedActivity();
        if (context == null) return;
        FragmentActivity act = (FragmentActivity) context;

        int dlgType = CertificateDetailDialog.PERMDLG_TYPE_ENCRYPT;
        if (!mDoEncrypt) {
            dlgType = CertificateDetailDialog.PERMDLG_TYPE_DECRYPT;
            mDetailDialog = new CertificateDetailDialog(act, true);
        } else {
            mDetailDialog = new CertificateDetailDialog(act, false);
        }
        mDetailDialog.init(dlgType, info);
        mDetailDialog.showDialog();
    }

    private void updateInfo(CertificateFileInfo info) {
        if (info.isCertFile) {
            mDataSupport.insertCert(info.issuer, info.publisher, info.serialNumber, info.filePath, info.fileName);
        } else {
            mDataSupport.insertPfx(info.issuer, info.publisher, info.serialNumber, info.filePath, info.fileName, info.password);
        }
    }

    public void onConfigurationChanged() {

        if (mDetailDialog != null && mDetailDialog.isShowing()) {
            mDetailDialog.resetWH();
            mDetailDialog.showDialog();
        } else if (mCertDialog != null && mCertDialog.isShowing()) {
            mCertDialog.resetWH();
            mCertDialog.showDialog();
        } else if (mPasswordDialog != null && mPasswordDialog.isShowing()) {
            mPasswordDialog.setHeight(mPasswordDialog.getDialogHeight());
            mPasswordDialog.show();
        }
    }
    public void onConfigurationChanged(Configuration newConfig) {
        if(mDetailDialog!=null){
            mDetailDialog.dismiss();
            mDetailDialog=null;
        }
        if(mCertDialog!=null){
            mCertDialog.dismiss();
            mCertDialog=null;
        }
        if(mPasswordDialog!=null){
            mPasswordDialog.dismiss();
            mPasswordDialog=null;
        }
    }
}

