/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.security.certificate;

import static com.foxit.uiextensions.utils.AppDmUtil.javaDateToDocumentDate;

import android.content.Context;
import android.net.Uri;
import android.text.TextUtils;

import com.foxit.uiextensions.security.CertUtil;
import com.foxit.uiextensions.security.ICertificateSupport;
import com.foxit.uiextensions.security.PfxCrypto;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppUtil;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateFactory;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.CertificateParsingException;
import java.security.cert.X509Certificate;
import java.util.Date;


public class CertificateSupport implements ICertificateSupport {

    private Context mContext;

    public CertificateSupport(Context context) {
        mContext = context;
    }

    private static boolean matchUsage(boolean[] keyUsage, int usage) {
        if (usage == 0 || keyUsage == null)
            return true;
        for (int i = 0; i < Math.min(keyUsage.length, 32); i++) {
            if ((usage & (1 << i)) != 0 && !keyUsage[i])
                return false;
        }
        return true;
    }

//    private void generateCertificateInfo(String keyStorePath, String keyStorePassword, CertificateInfo info) throws Exception {
//
//        KeyStore keyStore = null;
//        if (keyStorePath.toLowerCase(Locale.ROOT).endsWith(".pfx") || keyStorePath.toLowerCase(Locale.ROOT).endsWith(".p12"))
//            keyStore = KeyStore.getInstance("PKCS12");
//        else
//            keyStore = KeyStore.getInstance("JKS");
//        InputStream fis = null;
//        try {
//            File file = new File(keyStorePath);
//            if (file.canRead()) {
//                fis = new FileInputStream(keyStorePath);
//            } else {
//                Uri uri = AppFileUtil.toDocumentUriFromPath(keyStorePath);
//                fis = mContext.getContentResolver().openInputStream(uri);
//            }
//            keyStore.load(fis, keyStorePassword.toCharArray());
//        } finally {
//            if (fis != null)
//                fis.close();
//        }
//        Enumeration aliases = keyStore.aliases();
//        String keyAlias = null;
//        if (aliases != null) {
//            while (aliases.hasMoreElements()) {
//                keyAlias = (String) aliases.nextElement();
//                Certificate[] certs = keyStore.getCertificateChain(keyAlias);
//                if (certs == null || certs.length == 0)
//                    continue;
//                X509Certificate cert = (X509Certificate) certs[0];
//
//                Date beforeDate = cert.getNotBefore();
//                info.startDate = AppDmUtil.formatDocumentDate(AppDmUtil.FORMAT_YYYY_MM_DD_HH_MM_SS, javaDateToDocumentDate(beforeDate));
//                Date afterDate = cert.getNotAfter();
//                info.expiringDate = AppDmUtil.formatDocumentDate(AppDmUtil.FORMAT_YYYY_MM_DD_HH_MM_SS, javaDateToDocumentDate(afterDate));
//                info.issuerUniqueID = "";
//                boolean[] uniqueid = cert.getIssuerUniqueID();
//                if (uniqueid != null) {
//                    for (int i = 0; i < uniqueid.length; i++) {
//                        info.issuerUniqueID += uniqueid[i];
//                    }
//                }
//                info.identity = "XXXX";
//                String algName = cert.getSigAlgName();
//                BigInteger bigInteger = cert.getSerialNumber();
//                if (bigInteger.compareTo(BigInteger.ZERO) < 0) {
//                    bigInteger = new BigInteger(1, bigInteger.toByteArray());
//                }
//                info.serialNumber = bigInteger.toString(16).toUpperCase();
//                info.issuer = AppUtil.getEntryName(cert.getIssuerDN().getName(), "CN=");
//                info.emailAddress = AppUtil.getEntryName(cert.getIssuerDN().getName(), "E=");
//                info.publisher = AppUtil.getEntryName(cert.getIssuerDN().getName(), "CN=");
//
//                StringBuilder sb = new StringBuilder();
//                sb.append(AppUtil.getEntryName(cert.getSubjectDN().getName(), "CN="));
//                String temp = AppUtil.getEntryName(cert.getSubjectDN().getName(), "E=");
//                if (!TextUtils.isEmpty(temp)) {
//                    sb.append(" <");
//                    sb.append(temp);
//                    sb.append("> ");
//                }
//                info.name = sb.toString();
//                info.subject = AppUtil.getEntryName(cert.getSubjectDN().getName(), "CN=");
//                info.keyUsage = cert.getKeyUsage();
//                int usageCode = 0;
//                if (info.keyUsage != null) {
//                    for (int i = 0; i < info.keyUsage.length; i++) {
//                        usageCode = (info.keyUsage[i] ? (usageCode | 1 << i) : usageCode);
//                    }
//                }
//                info.usageCode = usageCode;
//
//                if (matchUsage(cert.getKeyUsage(), 1)) {
//                    try {
//                        cert.checkValidity();
//                    } catch (CertificateExpiredException e) {
//                        info.expired = true;
//                    } catch (CertificateNotYetValidException e) {
//                        info.expired = false;
//                    }
//                    break;
//                }
//            }
//        }
//    }

    private void generateCertificateInfo(X509Certificate cert, CertificateInfo info) {
        info.publisher = CertUtil.getCommonName(cert);
        info.serialNumber = cert.getSerialNumber().toString(16);
        info.issuer = CertUtil.getOrganizationName(cert);
        info.emailAddress = CertUtil.getEmailFromCert(cert);
        Date beforeDate = cert.getNotBefore();
        info.startDate = AppDmUtil.formatDocumentDate(AppDmUtil.FORMAT_MMM_DD_YYYY_HH_MM_SS, javaDateToDocumentDate(beforeDate));
        Date afterDate = cert.getNotAfter();
        info.expiringDate = AppDmUtil.formatDocumentDate(AppDmUtil.FORMAT_MMM_DD_YYYY_HH_MM_SS, javaDateToDocumentDate(afterDate));
        info.issuerUniqueID = "";
        boolean[] uniqueid = cert.getIssuerUniqueID();
        if (uniqueid != null) {
            for (int i = 0; i < uniqueid.length; i++) {
                info.issuerUniqueID += uniqueid[i];
            }
        }
        //check local system time
        try {
            cert.checkValidity();
        } catch (CertificateExpiredException e) {
            info.expired = true;
        } catch (CertificateNotYetValidException e) {
            info.expired = false;
        }

        info.identity = "XXXX";
        info.keyUsage = cert.getKeyUsage();
        if (info.keyUsage != null) {
            String str = "";
            for (int i = 0; i < info.keyUsage.length; i++) {
                str += (info.keyUsage[i] ? 1 : 0) + " ";
            }
        }

        StringBuilder sb = new StringBuilder();
        sb.append(AppUtil.getEntryName(cert.getSubjectDN().getName(), "CN="));
        String temp = AppUtil.getEntryName(cert.getSubjectDN().getName(), "E=");
        if (!TextUtils.isEmpty(temp)) {
            sb.append(" <");
            sb.append(temp);
            sb.append("> ");
        }
        info.name = sb.toString();
        info.subject = AppUtil.getEntryName(cert.getSubjectDN().getName(), "CN=");
        int usageCode = 0;
        if (info.keyUsage != null) {
            for (int i = 0; i < info.keyUsage.length; i++) {
                usageCode = (info.keyUsage[i] ? (usageCode | 1 << i) : usageCode);
            }
        }
        info.usageCode = usageCode;
    }

    public CertificateInfo verifyPwd(String filePath, String password) {
        CertificateInfo info = null;
        try {
            FileInputStream fis = new FileInputStream(filePath);
            PfxCrypto mcrypto = new PfxCrypto(fis);
            mcrypto.load(password);
            X509Certificate certificate = mcrypto.getX509Certificate();
            info = new CertificateInfo();
            generateCertificateInfo(certificate, info);
            mcrypto.reset();

            fis.close();
        } catch (CertificateEncodingException e) {
            e.printStackTrace();
        } catch (CertificateParsingException e) {
            e.printStackTrace();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return info;
    }

    public CertificateInfo verifyPassword(String filePath, String password) {
//        CertificateInfo info = null;
//        boolean ret = false;
//        InputStream stream = null;
//        try {
//            KeyStore var2 = KeyStore.getInstance("PKCS12", "BC");
//            File file = new File(filePath);
//            if (file.canRead()) {
//                stream = new FileInputStream(filePath);
//            } else {
//                Uri uri = AppFileUtil.toDocumentUriFromPath(filePath);
//                stream = mContext.getContentResolver().openInputStream(uri);
//            }
//            var2.load(stream, password.toCharArray());
//            Enumeration var3 = var2.aliases();
//            ret = true;
//            info = new CertificateInfo();
//
//            generateCertificateInfo(filePath, password, info);
//        } catch (Exception e) {
//            e.printStackTrace();
//        } finally {
//            try {
//                if (stream != null) {
//                    stream.close();
//                }
//            } catch (IOException e) {
//                e.printStackTrace();
//            }
//        }

        return verifyPwd(filePath,password);
    }

    public CertificateInfo getCertificateInfo(String filePath) {
        CertificateInfo info = null;
        try {
            InputStream is;
            File file = new File(filePath);
            if (file.canRead()) {
                is = new FileInputStream(filePath);
            } else {
                Uri uri = AppFileUtil.toDocumentUriFromPath(filePath);
                is = mContext.getContentResolver().openInputStream(uri);
            }
            try {
                CertificateFactory factory = CertificateFactory.getInstance("X.509");
                X509Certificate certificate = (X509Certificate) factory.generateCertificate(is);
                info = new CertificateInfo();
            } finally {
                is.close();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return info;
    }
}
