/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.pdfreader.tab;


import android.content.Context;
import android.content.res.ColorStateList;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffColorFilter;
import android.graphics.Rect;
import android.os.Handler;
import android.text.TextUtils;
import android.util.TypedValue;
import android.view.Gravity;
import android.view.View;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

import androidx.core.widget.TextViewCompat;
import androidx.fragment.app.FragmentActivity;
import androidx.recyclerview.widget.DefaultItemAnimator;
import androidx.recyclerview.widget.ItemTouchHelper;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import com.foxit.uiextensions.IPermissionProvider;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.controls.dialog.UIPopoverWin;
import com.foxit.uiextensions.controls.dialog.sheetmenu.IActionMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.ISheetMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.SheetItemBean;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UIActionMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UISheetMenu;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.controls.toolbar.IBaseItem;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.impl.BaseItemImpl;
import com.foxit.uiextensions.controls.toolbar.impl.TabItemImpl;
import com.foxit.uiextensions.controls.toolbar.impl.TopBarImpl;
import com.foxit.uiextensions.pdfreader.MainCenterItemBean;
import com.foxit.uiextensions.theme.DynamicColorProvider;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppSharedPreferences;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

public class TabViewControl implements ITabViewControl {
    private static final int TOP_PHONE_CENTER_POP_ITEM = 10000;
    private static final int TOP_PAD_CENTER_POP_ITEM = 10001;

    private final Context mContext;
    private final UIExtensionsManager mUiExtensionsManager;
    private final TopBarImpl mTabBar;
    private ISheetMenu mPhoneExpandMenu;
    private IActionMenu mPadExpandMenu;
    private TabMenuAdapter mTabMenuAdapter;
    private ITabSelectedListener mTabSelectedListener;
    private ITabRemovedListener mTabRemovedListener;
    private final Map<Integer, IBaseItem> mTabItems;
    private final List<TabMenuAdapter.TabItemBean> mOnToolItems;
    private final List<TabMenuAdapter.TabItemBean> mAllToolItems;

    private IBaseItem mCurrentTabItem;
    private IBaseItem mExpandItem;
    private TextView mTabTitleView;
    private boolean mShouldLoadMenuByTabBar = false;
    // Some modules are loaded in threads, when switching tabs, need to determine whether the modules have been fully loaded.
    private boolean mAsynInitTaskFinished = false;
    private RelativeLayout mTextDesBtn_ly;
    private ImageView mTextDesBtn;
    private TextView mTextDesTV;
    private LinearLayout mTextDes_ly_phone;
    private LinearLayout mTextDes_ly_pad;

    public TabViewControl(Context context, UIExtensionsManager uiExtensionsManager) {
        mContext = context;
        mUiExtensionsManager = uiExtensionsManager;
        mTabBar = (TopBarImpl) uiExtensionsManager.getMainFrame().getTopToolbar();
        mTabItems = new LinkedHashMap<>();
        mOnToolItems = new ArrayList<>();
        mAllToolItems = new ArrayList<>();
    }

    @Override
    public boolean addTabs(List<MainCenterItemBean> items) {
        mTabItems.clear();
        mOnToolItems.clear();
        mAllToolItems.clear();

        for (MainCenterItemBean centerItemBean : items) {
            if (centerItemBean.type == ToolbarItemConfig.ITEM_HOME_TAB) {
                centerItemBean.iconId = R.drawable.top_home_menu;
                centerItemBean.name = AppResource.getString(mContext, R.string.fx_topbar_home);
            }
            if (centerItemBean.type == ToolbarItemConfig.ITEM_EDIT_TAB) {
                centerItemBean.iconId = R.drawable.top_edit_menu;
                centerItemBean.name = AppResource.getString(mContext, R.string.fx_string_edit);
            }
            if (centerItemBean.type == ToolbarItemConfig.ITEM_COMMENT_TAB) {
                centerItemBean.iconId = R.drawable.top_comment_menu;
                centerItemBean.name = AppResource.getString(mContext, R.string.fx_string_comment);
            }
            if (centerItemBean.type == ToolbarItemConfig.ITEM_DRAWING_TAB) {
                centerItemBean.iconId = R.drawable.top_drawing_menu;
                centerItemBean.name = AppResource.getString(mContext, R.string.mt_drawing);
            }
            if (centerItemBean.type == ToolbarItemConfig.ITEM_VIEW_TAB) {
                centerItemBean.iconId = R.drawable.top_view_menu;
                centerItemBean.name = AppResource.getString(mContext, R.string.rd_bar_setting);
            }
            if (centerItemBean.type == ToolbarItemConfig.ITEM_FORM_TAB) {
                centerItemBean.iconId = R.drawable.top_form_menu;
                centerItemBean.name = AppResource.getString(mContext, R.string.menu_more_group_form);
            }
            if (centerItemBean.type == ToolbarItemConfig.ITEM_FILLSIGN_TAB) {
                centerItemBean.iconId = R.drawable.top_sign_menu;
                centerItemBean.name = AppResource.getString(mContext, R.string.fx_topbar_fillsign);
            }
            if (centerItemBean.type == ToolbarItemConfig.ITEM_PROTECT_TAB) {
                centerItemBean.iconId = R.drawable.top_protect_menu;
                centerItemBean.name = AppResource.getString(mContext, R.string.menu_more_group_protect);
            }
        }

        for (MainCenterItemBean centerItemBean : items) {
            int type = centerItemBean.type;
            if (type == ToolbarItemConfig.ITEM_VIEW_TAB && !AppDisplay.isPad())
                continue;
            IBaseItem tabItem = createTabItem(type, centerItemBean.iconId, centerItemBean.name);
            mTabItems.put(type, tabItem);

            TabMenuAdapter.TabItemBean tabItemBean = new TabMenuAdapter.TabItemBean();
            tabItemBean.mViewType = TabConstants.Tab_Content;
            tabItemBean.mTabItem = tabItem;
            if (centerItemBean.position == 0) {
                mOnToolItems.add(tabItemBean);
            }
            mAllToolItems.add(tabItemBean);
        }

        if (mTabMenuAdapter != null) {
            List<TabMenuAdapter.TabItemBean> moreItems = new ArrayList<>();
            for (TabMenuAdapter.TabItemBean itemBean : mAllToolItems) {
                if (!mOnToolItems.contains(itemBean)) {
                    moreItems.add(itemBean);
                }
            }
            mTabMenuAdapter.initItems(mOnToolItems, moreItems);
        }


        if (AppDisplay.isPad()) {
            for (TabMenuAdapter.TabItemBean tabItemBean : mOnToolItems) {
                IBaseItem tabItem = tabItemBean.mTabItem;
                if (tabItem == null) continue;

                boolean ret = mTabBar.addTabItem(tabItem);
                if (!ret)
                    break;
            }

            mExpandItem = mTabBar.addExpandItem();
            mExpandItem.setTag(TOP_PAD_CENTER_POP_ITEM);
            mExpandItem.setImageTintList(ColorStateList.valueOf(AppResource.getColor(mContext, R.color.ui_color_top_bar_icon)));
            mExpandItem.setItemLayoutListener(new IBaseItem.IItemLayoutListener() {
                @Override
                public void onItemLayout(int l, int t, int r, int b) {
                    if (mPadExpandMenu != null && mPadExpandMenu.isShowing()) {
                        Rect rect = new Rect();
                        mExpandItem.getContentView().getGlobalVisibleRect(rect);
                        mPadExpandMenu.show(mUiExtensionsManager.getRootView(), rect, UIPopoverWin.ARROW_TOP, 0);
                    }
                }
            });

            mExpandItem.setOnItemClickListener(new IBaseItem.OnItemClickListener() {
                @Override
                public void onClick(IBaseItem item, View v) {
                    mExpandItem.setImageResource(R.drawable.top_pop_collapse_icon);
                    mUiExtensionsManager.getMainFrame().showToolbars();
                    mUiExtensionsManager.stopHideToolbarsTimer();

                    if (mPadExpandMenu == null)
                        mPadExpandMenu = createPadExpandMenu();
                    mPadExpandMenu.setOnDismissListener(new IActionMenu.OnActionMenuDismissListener() {
                        @Override
                        public void onDismiss(IActionMenu actionMenu) {
                            mExpandItem.setImageResource(R.drawable.top_pop_expand_icon);
                            mUiExtensionsManager.startHideToolbarsTimer();
                        }
                    });
                    Rect rect = new Rect();
                    mExpandItem.getContentView().getGlobalVisibleRect(rect);
                    mPadExpandMenu.show(mUiExtensionsManager.getRootView(), rect, UIPopoverWin.ARROW_TOP, 0);
                }
            });
        } else {
            addPopItem();
        }
        return true;
    }

    @Override
    public void sortTabs(List<MainCenterItemBean> centerItemBeans) {
        if (AppDisplay.isPad()) {
            if (centerItemBeans == null) return;
            if (mTabMenuAdapter == null) return;

            List<TabMenuAdapter.TabItemBean> onTools = mTabMenuAdapter.getOnToolItems();
            for (int i = 0; i < onTools.size(); i++) {
                TabMenuAdapter.TabItemBean tabItemBean = onTools.get(i);
                if (tabItemBean.mTabItem == null) continue;

                for (MainCenterItemBean centerItemBean : centerItemBeans) {
                    if (tabItemBean.mTabItem.getTag() == centerItemBean.type) {
                        centerItemBean.position = 0;
                        centerItemBeans.remove(centerItemBean);
                        centerItemBeans.add(centerItemBean);
                        break;
                    }
                }
            }

            List<TabMenuAdapter.TabItemBean> moreTools = mTabMenuAdapter.getMoreToolItems();
            for (int i = 0; i < moreTools.size(); i++) {
                TabMenuAdapter.TabItemBean tabItemBean = moreTools.get(i);
                if (tabItemBean.mTabItem == null) continue;

                for (MainCenterItemBean centerItemBean : centerItemBeans) {
                    if (tabItemBean.mTabItem.getTag() == centerItemBean.type) {
                        centerItemBean.position = 1;
                        centerItemBeans.remove(centerItemBean);
                        centerItemBeans.add(centerItemBean);
                        break;
                    }
                }
            }
        }
    }

    private IBaseItem createTabItem(int tabId, int iconId, String name) {
        final TabItemImpl tabItem = new TabItemImpl(mContext, name);
        tabItem.setHighlightImageResource(iconId);
        tabItem.setTag(tabId);
        tabItem.setBackgroundColorProvider(new DynamicColorProvider() {
            @Override
            public int getColor(boolean isDarkMode) {
                int color = 0;
                if (ThemeConfig.getInstance(mContext).getAccentColorProvider() != null) {
                    color  = ThemeConfig.getInstance(mContext).getAccentColorProvider().getColor(AppUtil.isDarkMode(mContext));
                } else {
                    if (AppUtil.isDarkMode(mContext)) {
                        color = ThemeConfig.getInstance(mContext).getPrimaryColor();
                    }
                }
                return color;
            }
        });
        tabItem.setOnItemClickListener(new IBaseItem.OnItemClickListener() {
            @Override
            public void onClick(IBaseItem item, View v) {
                onMenuItemClick(item.getTag());
            }
        });
        return tabItem;
    }

    private IActionMenu createPadExpandMenu() {
        if (mPadExpandMenu == null) {
            mPadExpandMenu = UIActionMenu.newInstance((FragmentActivity) mUiExtensionsManager.getAttachedActivity());
            mPadExpandMenu.setContentView(getTabMenuView());
            mPadExpandMenu.setWidth(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_pad_top_menu_width));
        }
        return mPadExpandMenu;
    }

    private ColorStateList getSelectedButtonColorStateList() {
        int disabled = AppResource.getColor(mContext, R.color.p1);
        int selected = ThemeConfig.getInstance(mContext).getPrimaryColor();
        int normal = AppResource.getColor(mContext, R.color.p1);
        return AppResource.createColorStateList(selected, disabled, normal);
    }

    private View.OnClickListener mClickListener = new View.OnClickListener() {
        @Override
        public void onClick(View v) {
            mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_ContinuousScrolling);
            boolean mode = !AppSharedPreferences.getInstance(mContext).getBoolean("Btn_TestDes_mode", "key_mode", true);
            AppSharedPreferences.getInstance(mContext).setBoolean("Btn_TestDes_mode", "key_mode", mode);
            setSelectedButtonState(mode, mTextDesBtn_ly, mTextDesBtn);
            mUiExtensionsManager.getMainFrame().setToolBar(mUiExtensionsManager.getMainFrame().getCurToolbar());
        }
    };

    private void setSelectedButtonState(boolean isSelected, RelativeLayout switchLayout, ImageView switchIcon) {
        if (switchLayout == null || switchIcon == null) return;
        switchLayout.setSelected(isSelected);
        RelativeLayout.LayoutParams params = (RelativeLayout.LayoutParams) switchIcon.getLayoutParams();
        if (params == null) return;
        params.removeRule(isSelected ? RelativeLayout.ALIGN_PARENT_LEFT : RelativeLayout.ALIGN_PARENT_RIGHT);
        params.addRule(isSelected ? RelativeLayout.ALIGN_PARENT_RIGHT : RelativeLayout.ALIGN_PARENT_LEFT, RelativeLayout.TRUE);
        switchIcon.setLayoutParams(params);
    }

    private View getTabMenuView() {
        View tabMenuView = View.inflate(mContext, R.layout.common_recyclerview_layout, null);
        tabMenuView.setPadding(0, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_8dp), 0, 0);
        RecyclerView recyclerView = tabMenuView.findViewById(R.id.rd_recyclerview_list);

        mTextDes_ly_pad = tabMenuView.findViewById(R.id.test_des_pad_ly);
        mTextDes_ly_pad.setVisibility(View.VISIBLE);
        mTextDesBtn_ly = tabMenuView.findViewById(R.id.text_des_btn_ll_pad);
        mTextDesTV = tabMenuView.findViewById(R.id.text_des_tv);
        mTextDesBtn_ly.setOnClickListener(mClickListener);
        ThemeUtil.setTintList(mTextDesTV, ThemeUtil.getToolbarTextColor(mContext));
        ThemeUtil.setBackgroundTintList(mTextDesBtn_ly, getSelectedButtonColorStateList());
        mTextDesBtn = tabMenuView.findViewById(R.id.text_des_btn_pad);
        setSelectedButtonState(AppSharedPreferences.getInstance(mContext).getBoolean("Btn_TestDes_mode", "key_mode", true),
                mTextDesBtn_ly, mTextDesBtn);

        List<TabMenuAdapter.TabItemBean> onToolsItems = new ArrayList<>();
        List<TabMenuAdapter.TabItemBean> moreToolsItems = new ArrayList<>();
        if (mShouldLoadMenuByTabBar) {
            mShouldLoadMenuByTabBar = false;
            resetMenuItemsByTabBar(mAllToolItems, onToolsItems, moreToolsItems);
        } else {
            int count = mTabBar.getItemsCount(BaseBar.TB_Position.Position_CENTER);
            for (int i = 0; i < count; i++) {
                TabMenuAdapter.TabItemBean tabItemBean = mAllToolItems.get(i);
                onToolsItems.add(tabItemBean);
            }

            for (int i = count; i < mAllToolItems.size(); i++) {
                TabMenuAdapter.TabItemBean tabItemBean = mAllToolItems.get(i);
                moreToolsItems.add(tabItemBean);
            }
        }

        mTabMenuAdapter = new TabMenuAdapter(mContext, onToolsItems, moreToolsItems);
        recyclerView.setAdapter(mTabMenuAdapter);
        mTabMenuAdapter.setTabMenuChangedListener(new ITabMenuChangedListener() {
            @Override
            public void onItemMoved(int position, TabMenuAdapter.TabItemBean tabItem) {
                mTabBar.moveTabItem(position, tabItem.mTabItem);
                resetData();
            }

            @Override
            public void onItemRemoved(int position, TabMenuAdapter.TabItemBean tabItem) {
                mTabBar.removeTabItem(tabItem.mTabItem);
                resetData();

                if (mTabRemovedListener != null) {
                    mTabRemovedListener.onTabRemoved(tabItem.mTabItem);
                }
            }

            @Override
            public void onItemInserted(int position, TabMenuAdapter.TabItemBean tabItem) {
                mTabBar.addTabItem(position, tabItem.mTabItem);

                if (mCurrentTabItem != null && !mTabBar.containsTabItam(mCurrentTabItem)) {
                    resetCurItemPosition(tabItem.mTabItem);
                } else {
                    int count = mTabBar.getItemsCount(BaseBar.TB_Position.Position_CENTER);
                    if (!mTabBar.containsTabItam(tabItem.mTabItem)) {
                        if (count == 1
                                && mTabBar.getItemByIndex(BaseBar.TB_Position.Position_CENTER, 0) == mCurrentTabItem) {

                            if (count != mTabItems.size()) {
                                mTabMenuAdapter.changeMoreToolsPosition(count + 1);
                            }
                        } else {
                            resetInsertItemPosition(true, tabItem.mTabItem);
                        }
                    } else {
                        if (count != mTabItems.size()) {
                            mTabMenuAdapter.changeMoreToolsPosition(count + 1);
                        }
                    }
                }
                resetData();
            }
        });
        mTabMenuAdapter.setTabMenuClickListener(new TabMenuAdapter.ITabMenuClickListener() {
            @Override
            public void onOnToolClick(int position, TabMenuAdapter.TabItemBean tabItemBean) {
                if (mPadExpandMenu != null && mPadExpandMenu.isShowing()) {
                    mPadExpandMenu.dismiss();
                }
                onMenuItemClick(tabItemBean.mTabItem.getTag());
            }

            @Override
            public void onMoreToolClick(int position, TabMenuAdapter.TabItemBean tabItemBean) {
                if (mPadExpandMenu != null && mPadExpandMenu.isShowing()) {
                    mPadExpandMenu.dismiss();
                }

                mTabBar.addTabItem(tabItemBean.mTabItem);
                if (!mTabBar.containsTabItam(tabItemBean.mTabItem)) {
                    resetInsertItemPosition(false, tabItemBean.mTabItem);
                }
                resetData();
                onMenuItemClick(tabItemBean.mTabItem.getTag());
            }

            @Override
            public void checkSubscriptionPermission(int tabId, Event.Callback callback) {
                if (mPadExpandMenu != null && mPadExpandMenu.isShowing()) {
                    mPadExpandMenu.dismiss();
                }
                TabViewControl.this.checkSubscriptionPermission(tabId, callback);
            }
        });
        LinearLayoutManager linearLayoutManager = new LinearLayoutManager(mContext, LinearLayoutManager.VERTICAL, false);
        recyclerView.setLayoutManager(linearLayoutManager);
        recyclerView.setItemAnimator(new DefaultItemAnimator());

        ItemTouchHelper.Callback callback = new TabMenuItemTouchHelper(mTabMenuAdapter);
        ItemTouchHelper itemTouchHelper = new ItemTouchHelper(callback);
        itemTouchHelper.attachToRecyclerView(recyclerView);
        return tabMenuView;
    }

    private void checkSubscriptionPermission(int tabId, final Event.Callback callback) {
        if (tabId == ToolbarItemConfig.ITEM_EDIT_TAB && mUiExtensionsManager.getPermissionProvider() != null) {
            mUiExtensionsManager.getPermissionProvider().checkPermission(IPermissionProvider.FUNCTION_EDIT,
                    new IPermissionProvider.IPermissionState() {
                        @Override
                        public void onState(int state) {
                            if (callback != null)
                                callback.result(null, state == IPermissionProvider.PERMISSION_STATE_SHOW);
                        }
                    });
        } else if (tabId == ToolbarItemConfig.ITEM_FORM_TAB && mUiExtensionsManager.getPermissionProvider() != null) {
            mUiExtensionsManager.getPermissionProvider().checkPermission(IPermissionProvider.FUNCTION_FORM_DESIGN,
                    new IPermissionProvider.IPermissionState() {
                        @Override
                        public void onState(int state) {
                            if (callback != null)
                                callback.result(null, state == IPermissionProvider.PERMISSION_STATE_SHOW);
                        }
                    });
        } else {
            if (callback != null)
                callback.result(null, true);
        }
    }

    private void addPopItem() {
        mTabTitleView = new TextView(mContext);
        if (mCurrentTabItem != null)
            mTabTitleView.setText(mCurrentTabItem.getText());
        mTabTitleView.setTextSize(TypedValue.COMPLEX_UNIT_PX,
                AppResource.getDimensionPixelSize(mContext, R.dimen.ux_text_size_15sp));
//        mTabTitleView.setTypeface(Typeface.defaultFromStyle(Typeface.BOLD));
        mTabTitleView.setGravity(Gravity.CENTER_VERTICAL);
        mTabTitleView.setCompoundDrawablesWithIntrinsicBounds(null, null,
                AppResource.getDrawable(mContext, R.drawable.top_pop_up_down), null);
        mTabTitleView.setEllipsize(TextUtils.TruncateAt.MIDDLE);
        mTabTitleView.setMaxLines(1);
        mTabTitleView.setGravity(Gravity.CENTER);
        updateTabTitleTheme();;
        mTabTitleView.setPadding(AppDisplay.dp2px(12), AppDisplay.dp2px(5), AppDisplay.dp2px(4), AppDisplay.dp2px(5));

        final IBaseItem titleItem = new BaseItemImpl(mContext, mTabTitleView);
        titleItem.setTag(TOP_PHONE_CENTER_POP_ITEM);
        titleItem.setItemLayoutListener(new IBaseItem.IItemLayoutListener() {
            @Override
            public void onItemLayout(int l, int t, int r, int b) {
                if (mPhoneExpandMenu != null && mPhoneExpandMenu.isShowing()) {
                    Rect rect = new Rect();
                    titleItem.getContentView().getGlobalVisibleRect(rect);
                    mPhoneExpandMenu.update(mUiExtensionsManager.getRootView(), rect, UIPopoverWin.ARROW_TOP);
                }
            }
        });

        titleItem.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mUiExtensionsManager.getPDFViewCtrl().getDoc() == null) return;
                if (mPhoneExpandMenu == null) {
                    initPhoneMenu();
                }
                mUiExtensionsManager.getMainFrame().showToolbars();
                mUiExtensionsManager.stopHideToolbarsTimer();
                Rect rect = new Rect();
                titleItem.getContentView().getGlobalVisibleRect(rect);
                mPhoneExpandMenu.show(mUiExtensionsManager.getRootView(), rect, UIPopoverWin.ARROW_TOP, 0);

                mPhoneExpandMenu.setOnSheetDismissListener(new ISheetMenu.OnSheetDismissListener() {
                    @Override
                    public void onDismiss(ISheetMenu sheetMenu) {
                        mUiExtensionsManager.startHideToolbarsTimer();
                    }
                });
                mPhoneExpandMenu.setSheetItemClickListener(new ISheetMenu.OnSheetItemClickListener() {
                    @Override
                    public void onItemClick(int type) {
                        if (mPhoneExpandMenu != null && mPhoneExpandMenu.isShowing()) {
                            mPhoneExpandMenu.dismiss();
                        }
                        onMenuItemClick(type);
                    }
                });
            }
        });
        titleItem.setSelected(true);
        mTabBar.addView(titleItem, BaseBar.TB_Position.Position_CENTER);
        mTabBar.setAddTabItemHandler(new TopBarImpl.AddTabItemHandler() {
            @Override
            public void addTabItemHandler() {
                try {
                    new Handler().postDelayed(new Runnable() {
                        @Override
                        public void run() {
                            titleViewMaxWidth();
                        }
                    }, 300);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        });
    }

    private void initPhoneMenu() {
        if (mPhoneExpandMenu == null) {
            mPhoneExpandMenu = UISheetMenu.newInstance((FragmentActivity) mUiExtensionsManager.getAttachedActivity());
            mPhoneExpandMenu.setMode(ISheetMenu.SELECTED_MODE);
            mPhoneExpandMenu.setTitle(AppResource.getString(mContext, R.string.fx_topbar_tools));
            mTextDes_ly_phone = ((UISheetMenu)mPhoneExpandMenu).getTextDesLayout();
            mTextDes_ly_phone.setVisibility(View.VISIBLE);

            mTextDesTV = mTextDes_ly_phone.findViewById(R.id.sheet_menu_text_des_tv);
            mTextDesBtn_ly = mTextDes_ly_phone.findViewById(R.id.sheet_menu_text_des_btn_ll);
            mTextDesBtn = mTextDes_ly_phone.findViewById(R.id.sheet_menu_text_des_btn);
            ThemeUtil.setTintList(mTextDesTV, ThemeUtil.getToolbarTextColor(mContext));
            ThemeUtil.setBackgroundTintList(mTextDesBtn_ly, getSelectedButtonColorStateList());
            mTextDesBtn_ly.setOnClickListener(mClickListener);
            setSelectedButtonState(AppSharedPreferences.getInstance(mContext).getBoolean("Btn_TestDes_mode", "key_mode", true),
                    mTextDesBtn_ly, mTextDesBtn);
        }
        List<SheetItemBean> sheetItems = new ArrayList<>();
        for (IBaseItem tabItem : mTabItems.values()) {
            SheetItemBean sheetItem = new SheetItemBean();
            sheetItem.name = tabItem.getText();
            sheetItem.iconResId = tabItem.getHighlightImageResource();
            sheetItem.selected = tabItem.isSelected();
            sheetItem.type = tabItem.getTag();
            sheetItem.enabled = tabItem.isEnable();
            sheetItems.add(sheetItem);
        }

        mPhoneExpandMenu.setCustomSheetItem(sheetItems);
    }

    private void onMenuItemClick(final int type) {
        final IBaseItem tabItem = mTabItems.get(type);
        if (tabItem == null) return;

        final Runnable runnable = new Runnable() {
            @Override
            public void run() {
                if (mCurrentTabItem == tabItem) return;

                if (mCurrentTabItem != null) {
                    mCurrentTabItem.setSelected(false);
                    if (mTabSelectedListener != null)
                        mTabSelectedListener.onTabUnselected(mCurrentTabItem.getTag());
                }
                tabItem.setSelected(true);
                mCurrentTabItem = tabItem;
                if (mTabMenuAdapter != null)
                    mTabMenuAdapter.notifyUpdateData();
                if (mTabTitleView != null)
                    mTabTitleView.setText(tabItem.getText());
                if (mTabSelectedListener != null)
                    mTabSelectedListener.onTabSelected(tabItem.getTag());
            }
        };

        checkSubscriptionPermission(type, new Event.Callback() {
            @Override
            public void result(Event event, boolean success) {
                if (success) {
                    runnable.run();
                } else {
                    if (mPhoneExpandMenu != null) {
                        mPhoneExpandMenu.setSheetItemSelected(mCurrentTabItem.getTag(), true);
                    }
                }
            }
        });
    }

    @Override
    public void setCurrentTab(int tabId) {
        IBaseItem tabItem = mTabItems.get(tabId);
        setCurrentTab(tabItem);
    }

    @Override
    public void setCurrentTab(final IBaseItem tabItem) {
        if (mCurrentTabItem != tabItem) {
            if (tabItem == null) {
                if (mTabMenuAdapter != null) {
                    List<TabMenuAdapter.TabItemBean> items = mTabMenuAdapter.getAllToolItems();
                    for (int i = 0; i < items.size(); i++) {
                        TabMenuAdapter.TabItemBean tabItemBean = items.get(i);
                        if (tabItemBean.mViewType == TabConstants.Tab_Content) {
                            if (mCurrentTabItem == tabItemBean.mTabItem) {
                                tabItemBean.mTabItem.setSelected(false);
                                mTabMenuAdapter.notifyItemChanged(i);
                                break;
                            }
                        }
                    }
                }
                mCurrentTabItem.setSelected(false);
                mCurrentTabItem = null;
            } else {
                if (mAsynInitTaskFinished) {
                    if (AppDisplay.isPad()) {
                        if (!mTabBar.containsTabItam(tabItem)) {
                            mTabBar.addTabItem(tabItem);
                            if (!mTabBar.containsTabItam(tabItem)) {
                                resetInsertItemPosition(false, tabItem);
                            }
                            resetData();
                        }
                    }
                    tabItem.performClick();
                }
            }
        }
    }

    @Override
    public IBaseItem getCurrentTab() {
        return mCurrentTabItem;
    }

    @Override
    public int getTabCount() {
        return mTabItems.size();
    }

    @Override
    public IBaseItem getTabItem(int tabId) {
        return mTabItems.get(tabId);
    }

    @Override
    public List<IBaseItem> getTabItems() {
        return new ArrayList<>(mTabItems.values());
    }

    @Override
    public int getOnTabCount() {
        if (mOnToolItems != null) {
            return mOnToolItems.size();
        }
        return -1;
    }

    @Override
    public IBaseItem getOnTabItem(int index) {
        if (index < mOnToolItems.size()) {
            return mOnToolItems.get(index).mTabItem;
        }
        return null;
    }

    @Override
    public void removeTab(int tabId) {
        if (mTabItems == null || mTabItems.size() == 0) return;

        IBaseItem tabItem = mTabItems.get(tabId);
        if (tabItem != null) {
            mTabItems.remove(tabId);
            for (TabMenuAdapter.TabItemBean tabItemBean : mAllToolItems) {
                if (tabItemBean.mTabItem.getTag() == tabId) {
                    mOnToolItems.remove(tabItemBean);
                    mAllToolItems.remove(tabItemBean);
                    break;
                }
            }
            if (AppDisplay.isPad()) {
                if (mTabMenuAdapter != null)
                    mTabMenuAdapter.removeItem(tabId);
            } else {
                if (mPhoneExpandMenu != null) {
                    mPhoneExpandMenu.removeSheetItem(tabId);
                }
            }
            mTabBar.removeTabItem(tabItem);

            if (tabItem == mCurrentTabItem) {
                mCurrentTabItem = null;
            }

            if (mTabItems.size() == 0) {
                if (AppDisplay.isPad()) {
                    mTabBar.removeTabItem(mExpandItem);
                } else {
                    mTabBar.removeItemByTag(TOP_PHONE_CENTER_POP_ITEM);
                }
                if (mTabRemovedListener != null)
                    mTabRemovedListener.onTabRemoved(tabItem);
            } else {
                if (mCurrentTabItem == null) {
                    for (Integer tabID : mTabItems.keySet()) {
                        if (tabID != null) {
                            setCurrentTab(tabID);
                            break;
                        }
                    }
                }
                if (AppDisplay.isPad()) {
                    updateLayout();
                }
            }
        }
    }

    @Override
    public void setTabSelectedListener(ITabSelectedListener tabSelectedListener) {
        mTabSelectedListener = tabSelectedListener;
    }

    @Override
    public void setTabRemovedListener(ITabRemovedListener tabRemovedListener) {
        mTabRemovedListener = tabRemovedListener;
    }

    @Override
    public void updateTheme() {
        if (mPhoneExpandMenu != null) {
            if (mPhoneExpandMenu.isShowing())
                mPhoneExpandMenu.dismiss();
            mPhoneExpandMenu.updateTheme();
        }
        if (mPadExpandMenu != null) {
            mPadExpandMenu.dismiss();
            mPadExpandMenu = null;
        }
        if (mExpandItem != null) {
            mExpandItem.setImageTintList(ColorStateList.valueOf(AppResource.getColor(mContext, R.color.ui_color_top_bar_icon)));
        }
        if (mTabTitleView != null) {
            updateTabTitleTheme();
        }
        for (IBaseItem tabItem : mTabItems.values()) {
            tabItem.setTextColor(ThemeUtil.getTopBarTextColorStateList(mContext));
            tabItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
            if (AppDisplay.isPad() && tabItem.isSelected()) {
                tabItem.setSelected(true);
            }
        }
        if (mTextDes_ly_phone != null) {
            mTextDes_ly_phone.setBackgroundColor(AppResource.getColor(mContext, R.color.b1));
            mTextDes_ly_phone.findViewById(R.id.text_des_line).setBackgroundColor(AppResource.getColor(mContext, R.color.p2));
        }
        if (mTextDes_ly_pad != null) {
            mTextDes_ly_pad.setBackgroundColor(AppResource.getColor(mContext, R.color.b1));
            mTextDes_ly_pad.findViewById(R.id.text_des_line).setBackgroundColor(AppResource.getColor(mContext, R.color.p2));
        }

        if (mTextDesBtn_ly != null) {
            ThemeUtil.setTintList(mTextDesTV, ThemeUtil.getToolbarTextColor(mContext));
            ThemeUtil.setBackgroundTintList(mTextDesBtn_ly, getSelectedButtonColorStateList());
        }


    }

    @Override
    public void updateLayout() {
        if (AppDisplay.isPad()) {
            mTabBar.resetTabLayout(getOnToolItems());

            if (mCurrentTabItem != null && !mTabBar.containsTabItam(mCurrentTabItem)) {
                resetCurItemPosition(null);
                resetData();
            } else {
                if (mTabMenuAdapter != null) {
                    List<TabMenuAdapter.TabItemBean> onToolItems = new ArrayList<>();
                    int count = mTabBar.getItemsCount(BaseBar.TB_Position.Position_CENTER);
                    for (int i = 0; i < count; i++) {
                        TabMenuAdapter.TabItemBean tabItemBean = mAllToolItems.get(i);
                        onToolItems.add(tabItemBean);
                    }
                    List<TabMenuAdapter.TabItemBean> moreToolItems = new ArrayList<>();
                    for (int i = count; i < mAllToolItems.size(); i++) {
                        TabMenuAdapter.TabItemBean tabItemBean = mAllToolItems.get(i);
                        moreToolItems.add(tabItemBean);
                    }
                    mTabMenuAdapter.initItems(onToolItems, moreToolItems);
                    mTabMenuAdapter.notifyUpdateData();
                }
            }
        }
    }

    @Override
    public void release() {
        mTabItems.clear();
        mPadExpandMenu = null;
        mPhoneExpandMenu = null;
//        mCurrentTabItem = null;
    }

    @Override
    public boolean isShowingTabPop() {
        return (mPhoneExpandMenu != null && mPhoneExpandMenu.isShowing())
                || (mPadExpandMenu != null && mPadExpandMenu.isShowing());
    }

    @Override
    public void dismissTabPop() {
        if (mPadExpandMenu != null) {
            mPadExpandMenu.dismiss();
            mPadExpandMenu = null;
        }
        if (mPhoneExpandMenu != null) {
            mPhoneExpandMenu.dismiss();
            mPhoneExpandMenu = null;
        }
    }

    private List<IBaseItem> getOnToolItems() {
        List<IBaseItem> tabItems = new ArrayList<>();
        for (TabMenuAdapter.TabItemBean tabItemBean : mOnToolItems) {
            tabItems.add(tabItemBean.mTabItem);
        }
        return tabItems;
    }

    private void resetCurItemPosition(IBaseItem insertItem) {
        reservedCurItem(insertItem);

        if (mTabMenuAdapter == null) {
            mShouldLoadMenuByTabBar = true;
            return;
        }
        List<TabMenuAdapter.TabItemBean> onToolItems = new ArrayList<>();
        List<TabMenuAdapter.TabItemBean> moreToolItems = new ArrayList<>();
        resetMenuItemsByTabBar(mTabMenuAdapter.getAllToolItems(), onToolItems, moreToolItems);
        mTabMenuAdapter.initItems(onToolItems, moreToolItems);
        mTabMenuAdapter.notifyUpdateData();
    }

    private void resetMenuItemsByTabBar(List<TabMenuAdapter.TabItemBean> allItems,
                                        List<TabMenuAdapter.TabItemBean> onToolItems,
                                        List<TabMenuAdapter.TabItemBean> moreToolItems) {
        int count = mTabBar.getItemsCount(BaseBar.TB_Position.Position_CENTER);
        for (int i = 0; i < allItems.size(); i++) {
            TabMenuAdapter.TabItemBean tabItemBean = allItems.get(i);
            if (tabItemBean.mViewType != TabConstants.Tab_Content) continue;

            for (int j = 0; j < count; j++) {
                IBaseItem onItem = mTabBar.getItemByIndex(BaseBar.TB_Position.Position_CENTER, j);
                if (tabItemBean.mTabItem.getTag() == onItem.getTag()) {
                    onToolItems.add(tabItemBean);
                    break;
                }
            }
            if (!onToolItems.contains(tabItemBean)) {
                moreToolItems.add(tabItemBean);
            }
        }
    }

    private void reservedCurItem(IBaseItem insertItem) {
        if (mCurrentTabItem != null && !mTabBar.containsTabItam(mCurrentTabItem)) {
            int count = mTabBar.getItemsCount(BaseBar.TB_Position.Position_CENTER);
            if (count > 0) {
                int removeIndex = count - 1;
                IBaseItem tabItem = mTabBar.getItemByIndex(BaseBar.TB_Position.Position_CENTER, removeIndex);
                if (count == 1) {
                    mTabBar.removeTabItem(tabItem);
                    mTabBar.addTabItem(mCurrentTabItem);
                } else {
                    if (insertItem != null && !mTabBar.containsTabItam(insertItem)) {
                        mTabBar.removeTabItem(tabItem);
                        mTabBar.addTabItem(insertItem);
                    } else {
                        if (tabItem == insertItem) {
                            tabItem = mTabBar.getItemByIndex(BaseBar.TB_Position.Position_CENTER, count - 2);
                        }
                        mTabBar.removeTabItem(tabItem);
                    }
                    mTabBar.addTabItem(mCurrentTabItem);
                    reservedCurItem(insertItem);
                }
            }
        }
    }

    private void resetInsertItemPosition(boolean keepCurItem, IBaseItem insertItem) {
        reservedInsertItem(keepCurItem, insertItem);
        if (mTabMenuAdapter == null) {
            mShouldLoadMenuByTabBar = true;
            return;
        }

        List<TabMenuAdapter.TabItemBean> onToolItems = new ArrayList<>();
        List<TabMenuAdapter.TabItemBean> moreToolItems = new ArrayList<>();
        resetMenuItemsByTabBar(mTabMenuAdapter.getAllToolItems(), onToolItems, moreToolItems);
        mTabMenuAdapter.initItems(onToolItems, moreToolItems);
        mTabMenuAdapter.notifyUpdateData();
    }

    private void reservedInsertItem(boolean keepCurItem, IBaseItem insertItem) {
        if (insertItem != null && !mTabBar.containsTabItam(insertItem)) {
            int count = mTabBar.getItemsCount(BaseBar.TB_Position.Position_CENTER);
            if (count > 0) {
                int removeIndex = count - 1;
                IBaseItem tabItem = mTabBar.getItemByIndex(BaseBar.TB_Position.Position_CENTER, removeIndex);
                if (keepCurItem && count == 1) {
                    if (tabItem == mCurrentTabItem) return;

                    mTabBar.removeTabItem(tabItem);
                    mTabBar.addTabItem(insertItem);
                } else {
                    if (keepCurItem && tabItem == mCurrentTabItem) {
                        tabItem = mTabBar.getItemByIndex(BaseBar.TB_Position.Position_CENTER, count - 2);
                    }
                    mTabBar.removeTabItem(tabItem);
                    mTabBar.addTabItem(insertItem);
                    reservedInsertItem(keepCurItem, insertItem);
                }
            }
        }
    }

    private void resetData() {
        if (mTabMenuAdapter == null) return;

        mOnToolItems.clear();
        mOnToolItems.addAll(mTabMenuAdapter.getOnToolItems());
        mAllToolItems.clear();
        mAllToolItems.addAll(mTabMenuAdapter.getOnToolItems());
        mAllToolItems.addAll(mTabMenuAdapter.getMoreToolItems());
    }

    @Override
    public void asynInitTaskFinished() {
        mAsynInitTaskFinished = true;
        this.updateTabTitleTheme();
        if (getOnTabCount() > 0) {
            IBaseItem tabItem = getOnTabItem(0);
            if (mUiExtensionsManager.getPermissionProvider() != null) {
                if (tabItem.getTag() == ToolbarItemConfig.ITEM_EDIT_TAB || tabItem.getTag() == ToolbarItemConfig.ITEM_FORM_TAB) {
                    return;
                }
            }
            setCurrentTab(tabItem);
        }
    }

    public void titleViewMaxWidth() {
        int maxWidth = AppDisplay.getScreenWidth() - Math.max(mTabBar.getLeftWidth(), mTabBar.getRightWidth()) * 2 - AppDisplay.dp2px(12);
        for (int i = 0; i < mUiExtensionsManager.getMainFrame().getCenterItems().size(); i++) {
            if (mTabBar.getCenterLayout().getChildAt(i) instanceof LinearLayout) {
                LinearLayout linearLayout = (LinearLayout) mTabBar.getCenterLayout().getChildAt(0);
                if (linearLayout.getChildAt(0) instanceof TextView) {
                    TextView tv = (TextView) linearLayout.getChildAt(0);
                        tv.setMaxWidth(maxWidth);
                    }
                }
        }
    }

    private void updateTabTitleTheme(){
        if (mTabTitleView == null) {
            return;
        }

        mTabTitleView.setTextColor(ThemeUtil.getTopBarTextColorStateList(mContext));
        TextViewCompat.setCompoundDrawableTintList(mTabTitleView, ThemeUtil.getTopBarTextColorStateList(mContext));
        mTabTitleView.setBackground(AppResource.getDrawable(mContext, R.drawable.toolbar_center_layout_bg));
        if(ThemeConfig.getInstance(mContext).getAccentColorProvider() != null){
            int color = ThemeConfig.getInstance(mContext).getAccentColorProvider().getColor(AppUtil.isDarkMode(mContext));
            mTabTitleView.getBackground().setColorFilter(color, PorterDuff.Mode.SRC_IN);
        } else {
            if (AppUtil.isDarkMode(mContext)) {
                mTabTitleView.getBackground().setColorFilter(new PorterDuffColorFilter(ThemeConfig.getInstance(mContext).getPrimaryColor(), PorterDuff.Mode.SRC_IN));
            }
        }
    }
}
