/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.tts;


import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffXfermode;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.Typeface;
import android.graphics.drawable.ColorDrawable;
import android.media.AudioManager;
import android.os.Build;
import android.os.Handler;
import android.speech.tts.TextToSpeech;
import android.speech.tts.UtteranceProgressListener;
import android.speech.tts.Voice;
import android.util.Log;
import android.util.SparseArray;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.Task;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.TextPage;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.multimedia.PhoneStateBroadCastReceiver;
import com.foxit.uiextensions.annots.multimedia.sound.SoundModule;
import com.foxit.uiextensions.annots.textmarkup.TextSelector;
import com.foxit.uiextensions.controls.dialog.MatchDialog;
import com.foxit.uiextensions.controls.dialog.UIMatchDialog;
import com.foxit.uiextensions.controls.dialog.UIPopoverWin;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.controls.dialog.sheetmenu.IActionMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.ISheetMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.SheetItemBean;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UIActionMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UISheetMenu;
import com.foxit.uiextensions.controls.menu.IMenuGroup;
import com.foxit.uiextensions.controls.menu.IMenuItem;
import com.foxit.uiextensions.controls.menu.MenuViewImpl;
import com.foxit.uiextensions.controls.propertybar.IViewSettingsWindow;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.controls.toolbar.IBaseItem;
import com.foxit.uiextensions.controls.toolbar.drag.UIDragView;
import com.foxit.uiextensions.controls.toolbar.impl.BaseBarImpl;
import com.foxit.uiextensions.controls.toolbar.impl.BaseItemImpl;
import com.foxit.uiextensions.controls.toolbar.impl.BottomBarImpl;
import com.foxit.uiextensions.controls.toolbar.impl.TopBarImpl;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.event.PageEventListener;
import com.foxit.uiextensions.pdfreader.ILayoutChangeListener;
import com.foxit.uiextensions.pdfreader.ILifecycleEventListener;
import com.foxit.uiextensions.pdfreader.IStateChangeListener;
import com.foxit.uiextensions.pdfreader.config.AppBuildConfig;
import com.foxit.uiextensions.pdfreader.config.ReadStateConfig;
import com.foxit.uiextensions.pdfreader.impl.LifecycleEventListener;
import com.foxit.uiextensions.pdfreader.impl.MainFrame;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppSharedPreferences;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.IResult;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.text.Collator;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Set;

import androidx.fragment.app.FragmentActivity;

public class TTSModule implements Module {
    private Context mContext;
    private PDFViewCtrl mPdfViewCtrl;
    private UIExtensionsManager mUiExtensionsManager;

    private TextToSpeech mTTS;
    private HashMap<String, String> mTTSUtteranceIdMap = new HashMap<>();
    private float mTtsSpeechRate;
    private float mTtsPitch = 1.0f;

    Boolean mSupportVoices;
    ArrayList<TtsVoiceGroup> mLanguageList = new ArrayList<>();
    private Voice mCurrentVoice;
    private Locale mCurrentLocale = Locale.US;
    private String mCurrentVoiceName;
    private boolean mShowVoicePrompt = true;

    private ArrayList<TTSInfo> mTtsRectInfoList = new ArrayList<>();
    private TTSInfo mCurTtsRectInfo = null;
    private ArrayList<String> mCurTtsStrList = new ArrayList<>();
    private ArrayList<Locale> mCurTtsLocalLanguageList = new ArrayList<>();

    private ArrayList<SearchTTSInfoTask> mSearchTextTaskArrays = new ArrayList<>();
    private SparseArray<SparseArray<ArrayList<TTSInfo>>> mTTSInfoArrays = new SparseArray<>();

    //    private BottomSheetDialog mSpeedRatesSheetDialog;
//    private PropertyBar mSpeedPropertyBar;
//
//    private View mSpeedRateView;
    private IViewSettingsWindow mSettingWindow;
    private BaseBar mTtsTopBar;
    private BaseBar mTtsBottomBar;

    private IBaseItem mTtsBackItem;
    private IBaseItem mTtsTitleItem;
    private IBaseItem mTtsPreviousPageItem;
    private IBaseItem mTtsNextPageItem;
    private IBaseItem mSpeedItem;
    private IBaseItem mTtsStartItem;
    private IBaseItem mTtsStopItem;
    private IBaseItem mTtsCycleItem;
    private IBaseItem mQuitItem;

    private boolean mHasInitTts = false;
    private boolean mSupperTts = false;
    private boolean mIsSpeaking = false;
    private boolean mIsPause = false;
    private boolean mIsCycleSpeak = false;
    private boolean mIsExit = true;
    private boolean mIsPad = false;

    private int mCurTtsRectPageIndex = -1;

    public TTSModule(Context context, ViewGroup parent, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = (UIExtensionsManager) uiExtensionsManager;
    }

    @Override
    public String getName() {
        return Module.MODULE_NAME_TTS;
    }

    @Override
    public boolean loadModule() {
        mTtsSpeechRate = 1.0f;
        mSettingWindow = mUiExtensionsManager.getMainFrame().getSettingWindow();
        mIsPad = AppDisplay.isPad();

        if (mTTS == null)
            mTTS = new TextToSpeech(mContext, mTextToSpeechOnInitListener);

        mUiExtensionsManager.registerModule(this);
        mUiExtensionsManager.registerLayoutChangeListener(mLayoutChangeListener);
        mUiExtensionsManager.registerStateChangeListener(mStatusChangeListener);
        mUiExtensionsManager.registerLifecycleListener(mRDLifecycleEventListener);
        mUiExtensionsManager.registerThemeEventListener(mThemeEventListener);
        mUiExtensionsManager.getPhoneStateBroadCaseReceiver().registerListener(mPhoneStateListener);
        mPdfViewCtrl.registerDocEventListener(mDocEventListener);
        mPdfViewCtrl.registerDrawEventListener(mDrawEventListener);
        mPdfViewCtrl.registerPageEventListener(mPageEventListener);
        return true;
    }

    @Override
    public boolean unloadModule() {
        mUiExtensionsManager.unregisterLayoutChangeListener(mLayoutChangeListener);
        mUiExtensionsManager.unregisterStateChangeListener(mStatusChangeListener);
        mUiExtensionsManager.unregisterLifecycleListener(mRDLifecycleEventListener);
        mUiExtensionsManager.unregisterThemeEventListener(mThemeEventListener);
        mUiExtensionsManager.getPhoneStateBroadCaseReceiver().unregisterListener(mPhoneStateListener);
        mPdfViewCtrl.unregisterDocEventListener(mDocEventListener);
        mPdfViewCtrl.unregisterDrawEventListener(mDrawEventListener);
        releaseTTS();
        mSettingWindow = null;
        mUiExtensionsManager = null;
        return true;
    }

    private boolean mPhonePause;
    private PhoneStateBroadCastReceiver.IPhoneStateListener mPhoneStateListener = new PhoneStateBroadCastReceiver.IPhoneStateListener() {
        @Override
        public void onOutGoingCall() {
            if (mTtsStartItem != null && !mIsPause) {
                mPhonePause = true;
                onStartItemClicked();
            }
        }

        @Override
        public void onIncomingCall() {
            if (mTtsStartItem != null && !mIsPause) {
                mPhonePause = true;
                onStartItemClicked();
            }
        }

        @Override
        public void onCallIdle() {
            if (mTtsStartItem != null && mPhonePause) {
                mPhonePause = false;
                onStartItemClicked();
            }
        }
    };

    private void initBars() {
        if (!AppDisplay.isPad()) {
            initTopBar();
        }
        initBottomBar();

        addSpeedRateItems();

        mCurrentTabItem = mSpeedDlgItems.get(convertSpeedRateValueToType(mTtsSpeechRate));
        mCurrentTabItem.setSelected(true);

        if (!AppDisplay.isPad()) {
            RelativeLayout.LayoutParams ttsTopLp = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
            ttsTopLp.addRule(RelativeLayout.ALIGN_PARENT_TOP);
            mUiExtensionsManager.getMainFrame().getContentView().addView(mTtsTopBar.getContentView(), ttsTopLp);

            RelativeLayout.LayoutParams ttsBottomLp = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
            ttsBottomLp.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
            mUiExtensionsManager.getMainFrame().getContentView().addView(mTtsBottomBar.getContentView(), ttsBottomLp);

            mTtsTopBar.getContentView().setVisibility(View.INVISIBLE);
            mTtsBottomBar.getContentView().setVisibility(View.INVISIBLE);
        }
    }

    private void initTopBar() {
        mTtsTopBar = new TopBarImpl(mContext);
        mTtsTopBar.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());

        mTtsBackItem = new BaseItemImpl(mContext);
        mTtsBackItem.setText(R.string.fx_string_close);
        mTtsBackItem.setTextSize(AppDisplay.px2dp(mContext.getResources().getDimensionPixelOffset(R.dimen.ux_text_size_16sp)));
        mTtsBackItem.setTextColor(ThemeConfig.getInstance(mContext).getPrimaryColor());
        mTtsBackItem.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (AppUtil.isFastDoubleClick()) {
                    return;
                }
                mAutoSpeak = false;
                if (mLastReadState == ReadStateConfig.STATE_FILLSIGN)
                    mUiExtensionsManager.changeState(ReadStateConfig.STATE_FILLSIGN);
                else
                    mUiExtensionsManager.changeState(ReadStateConfig.STATE_NORMAL);
            }
        });

        mTtsTitleItem = new BaseItemImpl(mContext);
        mTtsTitleItem.setText(AppResource.getString(mContext.getApplicationContext(), R.string.rd_tts_speak));
        mTtsTitleItem.setTextSize(AppDisplay.px2dp(mContext.getResources().getDimensionPixelOffset(R.dimen.ux_text_size_16sp)));
        mTtsTitleItem.setTextColor(AppResource.getColor(mContext, R.color.t4));
        mTtsTitleItem.setTypeface(Typeface.defaultFromStyle(Typeface.BOLD));

        mTtsTopBar.addView(mTtsBackItem, BaseBar.TB_Position.Position_LT);
        mTtsTopBar.addView(mTtsTitleItem, BaseBar.TB_Position.Position_CENTER);
        mTtsTopBar.setMiddleButtonCenter(true);
        mTtsTopBar.setStartMargin(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_text_size_16sp));
    }

    private void initBottomBar() {
        boolean isPad = AppDisplay.isPad();
        if (isPad) {
            mTtsBottomBar = new BaseBarImpl(mContext);
            mTtsBottomBar.setInterceptTouch(false);
            mQuitItem = new BaseItemImpl(mContext, R.drawable.ic_tool_bar_item_logout);
            mQuitItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
        } else {
            mTtsBottomBar = new BottomBarImpl(mContext);
        }
        mTtsBottomBar.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
        if (isPad) {
            if (AppDevice.isChromeOs(mPdfViewCtrl.getAttachedActivity()))
                mTtsBottomBar.setItemInterval(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_bottombar_button_space_phone));
            else
                mTtsBottomBar.setItemInterval(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_bottombar_button_space_pad));
        } else
            mTtsBottomBar.setItemInterval(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_speech_bottombar_button_space_phone));
        mTtsBottomBar.setAutoCompressItemsInterval(true);

        mTtsPreviousPageItem = new BaseItemImpl(mContext, R.drawable.tts_pre_page);
        mTtsStartItem = new BaseItemImpl(mContext, R.drawable.tts_start);
        mTtsStopItem = new BaseItemImpl(mContext, R.drawable.tts_stop);

        if (isSupportVoiceList()) {
            mSpeedItem = new BaseItemImpl(mContext, R.drawable.rd_item_more);
        } else {
            mSpeedItem = new BaseItemImpl(mContext, getSpeechRateImageRes(mTtsSpeechRate));
        }

        mTtsNextPageItem = new BaseItemImpl(mContext, R.drawable.tts_next_page);
        mTtsCycleItem = new BaseItemImpl(mContext, R.drawable.tts_cycle);

        mTtsPreviousPageItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
        mTtsStartItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
        mTtsStopItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
        mSpeedItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
        mTtsNextPageItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
        mTtsCycleItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));

        addBottomBarItemListener();

        mTtsBottomBar.addView(mTtsCycleItem, BaseBar.TB_Position.Position_CENTER);
        mTtsBottomBar.addView(mTtsPreviousPageItem, BaseBar.TB_Position.Position_CENTER);
        mTtsBottomBar.addView(mTtsStartItem, BaseBar.TB_Position.Position_CENTER);
//        mTtsBottomBar.addView(mTtsStopItem, BaseBar.TB_Position.Position_CENTER);
        mTtsBottomBar.addView(mTtsNextPageItem, BaseBar.TB_Position.Position_CENTER);
        mTtsBottomBar.addView(mSpeedItem, BaseBar.TB_Position.Position_CENTER);
        if (isPad) {
            mTtsBottomBar.addView(mQuitItem, BaseBar.TB_Position.Position_CENTER);
        }
    }

//    private String _getSpeechRateString(float rate) {
//        if (rate == 1 || rate == 2) {
//            return (int) rate + "X";
//        } else {
//            return rate + "X";
//        }
//    }

    private int getSpeechRateImageRes(float rate) {
        int resId = R.drawable.tts_speed_rate_1x;
        if (rate == 0.5) {
            resId = R.drawable.tts_speed_rate_05x;
        } else if (rate == 0.75) {
            resId = R.drawable.tts_speed_rate_075x;
        } else if (rate == 1) {
            resId = R.drawable.tts_speed_rate_1x;
        } else if (rate == 1.25) {
            resId = R.drawable.tts_speed_rate_125x;
        } else if (rate == 1.5) {
            resId = R.drawable.tts_speed_rate_15x;
        } else if (rate == 2) {
            resId = R.drawable.tts_speed_rate_2x;
        }
        return resId;
    }

    private void addBottomBarItemListener() {
        mTtsPreviousPageItem.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                final int pageIndex = mCurTtsRectPageIndex - 1;
                PDFPage page = mUiExtensionsManager.getDocumentManager().getPage(pageIndex, false);
                if (page == null || page.isEmpty())
                    return;

                cancelSearchTask();
                final int startIndex = 0;
                final ArrayList<TTSInfo> ttsInfos = getTTSInfosFromCache(pageIndex, startIndex);
                if (ttsInfos != null) {
                    startSpeak(ttsInfos, pageIndex);
                } else {
                    final SearchTTSInfoTask searchTask = new SearchTTSInfoTask(mPdfViewCtrl, page, startIndex, new IResult<ArrayList<TTSInfo>, Object, Object>() {
                        @Override
                        public void onResult(boolean success, ArrayList<TTSInfo> ttsInfos, Object p2, Object p3) {
                            if (success && ttsInfos != null) {
                                saveTTSInfosToCache(pageIndex, startIndex, ttsInfos);
                                startSpeak(ttsInfos, pageIndex);
                            }
                        }
                    });
                    mPdfViewCtrl.addTask(searchTask);
                    mSearchTextTaskArrays.add(searchTask);
                }
            }
        });

        mTtsStartItem.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                onStartItemClicked();
            }
        });

        mTtsStopItem.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                if (AppUtil.isFastDoubleClick()) {
                    return;
                }
                cancelSearchTask();
                stopTts();
            }
        });

        mSpeedItem.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (AppUtil.isFastDoubleClick()) {
                    return;
                }
                if (isSupportVoiceList()) {
                    showMorePopup();
                } else {
                    showSpeedPopup();
                }
            }
        });

        mTtsNextPageItem.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                final int pageIndex = mCurTtsRectPageIndex + 1;
                PDFPage page = mUiExtensionsManager.getDocumentManager().getPage(pageIndex, false);
                if (page == null || page.isEmpty())
                    return;

                cancelSearchTask();
                final int startIndex = 0;
                final ArrayList<TTSInfo> ttsInfos = getTTSInfosFromCache(pageIndex, startIndex);
                if (ttsInfos != null) {
                    startSpeak(ttsInfos, pageIndex);
                } else {
                    SearchTTSInfoTask searchTask = new SearchTTSInfoTask(mPdfViewCtrl, page, startIndex, new IResult<ArrayList<TTSInfo>, Object, Object>() {
                        @Override
                        public void onResult(boolean success, ArrayList<TTSInfo> ttsInfos, Object p2, Object p3) {
                            if (success && ttsInfos != null) {
                                saveTTSInfosToCache(pageIndex, startIndex, ttsInfos);
                                startSpeak(ttsInfos, pageIndex);
                            }
                        }
                    });
                    mPdfViewCtrl.addTask(searchTask);
                    mSearchTextTaskArrays.add(searchTask);
                }
            }
        });

        mTtsCycleItem.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mIsCycleSpeak = !mIsCycleSpeak;
                setAllTtsItemByState();
            }
        });
        if (AppDisplay.isPad()) {
            mQuitItem.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    if (AppUtil.isFastDoubleClick()) {
                        return;
                    }
                    mAutoSpeak = false;
                    if (mLastReadState == ReadStateConfig.STATE_FILLSIGN)
                        mUiExtensionsManager.changeState(ReadStateConfig.STATE_FILLSIGN);
                    else
                        mUiExtensionsManager.changeState(ReadStateConfig.STATE_NORMAL);
                }
            });
        }
    }

    public boolean isSupperTts() {
        if (mHasInitTts)
            return mSupperTts;
        else
            return true;
    }

    public boolean isSupportVoiceList() {
        return mSupportVoices != null && mSupportVoices;
    }

    public boolean isSelectedCurrentVoice() {
        return mCurrentVoice != null;
    }

    private boolean mTsChangeStatus;
    private boolean mbOnlySelect;

    public void speakFromTs(final TTSInfo info) {
        mUiExtensionsManager.requestPhoneStatePermission(AppResource.getString(mContext, R.string.audio_fun_name), new Event.Callback() {
            @Override
            public void result(Event event, boolean success) {
                if (mTTS != null && mTTS.isSpeaking()) {
                    mTTS.stop();
                    resetState();
                    resetAllTtsItem();
                    mPdfViewCtrl.invalidate();
                }

                if (mUiExtensionsManager.getState() != ReadStateConfig.STATE_TTS) {
                    mTsChangeStatus = true;
                    mUiExtensionsManager.changeState(ReadStateConfig.STATE_TTS);
                    MainFrame mainFrame = (MainFrame) mUiExtensionsManager.getMainFrame();
                    if (!mainFrame.isToolbarsVisible() && !mainFrame.isShowFullScreenUI())
                        mainFrame.showToolbars();
                }
                mCurTtsRectPageIndex = info.mPageIndex;
                mTtsRectInfoList.clear();

                setAllTtsItemByState();
                parsingTtsRectInfo(info.mText);
                mCurTtsRectInfo = info;
                speakStringAfterParsing();
                mbOnlySelect = true;
            }
        });

    }

    public void speakFromTp(final TTSInfo info) {
        mUiExtensionsManager.requestPhoneStatePermission(AppResource.getString(mContext, R.string.audio_fun_name), new Event.Callback() {
            @Override
            public void result(Event event, boolean success) {
                if (mTTS != null && mTTS.isSpeaking()) {
                    mTTS.stop();
                    resetState();
                    resetAllTtsItem();
                    mPdfViewCtrl.invalidate();
                }

                if (mUiExtensionsManager.getState() != ReadStateConfig.STATE_TTS) {
                    mTsChangeStatus = true;
                    mUiExtensionsManager.changeState(ReadStateConfig.STATE_TTS);
                    MainFrame mainFrame = (MainFrame) mUiExtensionsManager.getMainFrame();
                    if (!mainFrame.isToolbarsVisible() && !mainFrame.isShowFullScreenUI())
                        mainFrame.showToolbars();
                }
                final int pageIndex = info.mPageIndex;
                PDFPage page = mUiExtensionsManager.getDocumentManager().getPage(pageIndex, false);
                if (page == null || page.isEmpty()) return;

                cancelSearchTask();
                final int startIndex = info.mStart;
                final ArrayList<TTSInfo> infos = getTTSInfosFromCache(pageIndex, startIndex);
                if (infos != null) {
                    if (infos.size() != 0) {
                        mTtsRectInfoList.addAll(infos);
                        mCurTtsRectPageIndex = pageIndex;

                        if (mTtsRectInfoList.size() > 0) {
                            mCurTtsRectInfo = mTtsRectInfoList.remove(0);
                            parsingTtsRectInfo(mCurTtsRectInfo.mText);
                            speakStringAfterParsing();
                            setAllTtsItemByState();
                            mPdfViewCtrl.invalidate();
                        }
                    }
                } else {
                    SearchTTSInfoTask searchTask = new SearchTTSInfoTask(mPdfViewCtrl, page, startIndex, new IResult<ArrayList<TTSInfo>, Object, Object>() {
                        @Override
                        public void onResult(boolean success, ArrayList<TTSInfo> ttsInfos, Object p2, Object p3) {
                            if (success && ttsInfos != null) {
                                if (ttsInfos.size() != 0) {
                                    mTtsRectInfoList.addAll(ttsInfos);
                                    mCurTtsRectPageIndex = pageIndex;

                                    if (mTtsRectInfoList.size() > 0) {
                                        mCurTtsRectInfo = mTtsRectInfoList.remove(0);
                                        parsingTtsRectInfo(mCurTtsRectInfo.mText);
                                        speakStringAfterParsing();
                                        setAllTtsItemByState();
                                        mPdfViewCtrl.invalidate();
                                    }
                                }
                                saveTTSInfosToCache(pageIndex, startIndex, ttsInfos);
                            }
                        }
                    });
                    mPdfViewCtrl.addTask(searchTask);
                    mSearchTextTaskArrays.add(searchTask);
                }
            }
        });
    }

    private void setAllTtsItemByState() {
        if (mIsSpeaking) {
            if (mIsPause) {
                mTtsStartItem.setImageResource(R.drawable.tts_start);
                muteAudioFocus(mContext, false);
            } else {
                mTtsStartItem.setImageResource(R.drawable.tts_pause);
                muteAudioFocus(mContext, true);
            }
            mTtsPreviousPageItem.setEnable(mCurTtsRectPageIndex > 0);
            mTtsNextPageItem.setEnable(mPdfViewCtrl.getDoc() != null && mCurTtsRectPageIndex < mPdfViewCtrl.getPageCount() - 1);

            mTtsStartItem.setEnable(true);
            mTtsStopItem.setEnable(true);
        } else {
            mTtsStartItem.setImageResource(R.drawable.tts_start);
            mTtsPreviousPageItem.setEnable(false);
            mTtsNextPageItem.setEnable(false);
            mTtsStopItem.setEnable(false);
            muteAudioFocus(mContext, false);
        }
        mTtsCycleItem.setChecked(mIsCycleSpeak);
    }

    private void onStartItemClicked() {
        if (mIsSpeaking && !mIsPause) {
            mIsPause = true;
            mTTSUtteranceIdMap.clear();
            mTTS.stop();
            setAllTtsItemByState();
            mPdfViewCtrl.invalidate();
        } else if (mbOnlySelect) {
            if (mCurTtsRectInfo != null && (mCurTtsStrList.size() == 0 || mCurTtsStrList.size() != mCurTtsLocalLanguageList.size())) {
                parsingTtsRectInfo(mCurTtsRectInfo.mText);
            }
            speakStringAfterParsing();
        } else if (mIsPause && mCurTtsRectInfo != null) {
            parsingTtsRectInfo(mCurTtsRectInfo.mText);
            speakStringAfterParsing();
        } else {
            startSpeak();
        }
    }

    private void stopTts() {
        mTtsRectInfoList.clear();
        mIsExit = true;
        if (mTTS != null)
            mTTS.stop();

        mbOnlySelect = false;
        mAutoSpeak = false;
        mTTSUtteranceIdMap.clear();
        mIsSpeaking = false;
        mIsPause = false;
        mCurTtsRectInfo = null;
        mCurTtsStrList.clear();
        mCurTtsLocalLanguageList.clear();
        mCurTtsRectPageIndex = -1;
        setAllTtsItemByState();
        mPdfViewCtrl.invalidate();
    }

    private boolean muteAudioFocus(Context context, boolean bStopOtherMusic) {
        if (context == null) {
            return false;
        }
        int result;
        AudioManager am = (AudioManager) context.getSystemService(Context.AUDIO_SERVICE);
        if (bStopOtherMusic) {
            result = am.requestAudioFocus(null, AudioManager.STREAM_MUSIC, AudioManager.AUDIOFOCUS_GAIN_TRANSIENT);
        } else {
            result = am.abandonAudioFocus(null);
        }
        return result == AudioManager.AUDIOFOCUS_REQUEST_GRANTED;
    }

    public boolean onKeyBack() {
        if (mUiExtensionsManager.getState() == ReadStateConfig.STATE_TTS) {
            mAutoSpeak = false;
            if (mLastReadState == ReadStateConfig.STATE_FILLSIGN)
                mUiExtensionsManager.changeState(ReadStateConfig.STATE_FILLSIGN);
            else
                mUiExtensionsManager.changeState(ReadStateConfig.STATE_NORMAL);
            return true;
        }
        return false;
    }

    private void parsingTtsRectInfo(String ttsString) {
        if (!AppUtil.isEmpty(ttsString)) {
            mCurTtsStrList.clear();
            mCurTtsLocalLanguageList.clear();

            if (isSupportVoiceList() && isSelectedCurrentVoice()) { // MOBRD-8179
                mCurTtsStrList.add(ttsString);
                mCurTtsLocalLanguageList.add(mCurrentLocale);
                return;
            }

            ArrayList<Integer> codePageList = new ArrayList<>();
            TTSUtils.splitSentenceByLanguage(ttsString, mCurTtsStrList, codePageList);
            for (int codePage : codePageList) {
                mCurTtsLocalLanguageList.add(getLocaleLanguageByPageCode(codePage));
            }
        }
    }

    private Locale getLocaleLanguageByPageCode(int pageCode) {
        Locale locale;
        switch (pageCode) {
            case 874:// Thai
                locale = new Locale("th", "TH", "TH");
                break;
            case 932:// Shift JIS
                locale = getCJKLanguage(Locale.JAPANESE);
                break;
            case 936:// Chinese Simplified (GBK)
                locale = getCJKLanguage(Locale.CHINA);
                break;
            case 949:// Korean
                locale = getCJKLanguage(Locale.KOREAN);
                break;
            case 950:// Big5
                locale = Locale.TAIWAN;
                break;
//            case 1250:// Eastern European
//                locale = Locale.Eastern_European;
//                break;
            case 1251:// Cyrillic
                locale = new Locale("hr", "HR", "HR");
                break;
            case 1253:// Greek
                locale = new Locale("el", "GR", "GR");
                break;
            case 1254:// Turkish
                locale = new Locale("tr", "TR", "TR");
                break;
            case 1255:// Hebrew
                locale = new Locale("he", "IL", "IL");
                break;
            case 1256:// Arabic
                locale = new Locale("ar");
                break;
//            case 1257:// Baltic
//                locale = Locale.Baltic;
//                break;
            case 1258:// Vietnamese vi_VN
                locale = new Locale("vi", "VN", "VN");
                break;
//            case 1361://Johab
//                locale = Locale.Johab;
//                break;
            case 10001:// Mac Shift Jis
                locale = Locale.JAPANESE;
                break;
            case 10003:// Mac Korean
                locale = Locale.KOREAN;
                break;
            case 10008:// // Mac Chinese Simplified (GBK)
                locale = Locale.CHINA;
                break;
            case 10002:// Mac Big5
                locale = Locale.TAIWAN;
                break;
            case 10005:// Mac Hebrew
                locale = new Locale("he", "IL", "IL");
                break;
            case 10004:// Mac Arabic
                locale = new Locale("ar");
                break;
            case 10006:// Mac Greek
                locale = new Locale("el", "GR", "GR");
                break;
            case 10081:// Mac Turkish
                locale = new Locale("tr", "TR", "TR");
                break;
            case 10021:// Mac Thai
                locale = new Locale("th", "TH", "TH");
                break;
//            case 10029:// Mac Eastern European (Latin 2)
//                locale = Locale.Eastern_European;
//                break;
            case 10007:// Mac Cyrillic
                locale = new Locale("hr", "HR", "HR");
                break;
            default:
                return getDefaultLocaleLanguage();
        }
        if (locale != null && mTTS != null) {
            int ret = mTTS.isLanguageAvailable(locale);
            if (ret == TextToSpeech.LANG_MISSING_DATA || ret == TextToSpeech.LANG_NOT_SUPPORTED) {
                return getDefaultLocaleLanguage();
            } else {
                return locale;
            }
        } else {
            return getDefaultLocaleLanguage();
        }
    }

    private Locale getCJKLanguage(Locale defaultLocale) {// MOBRD-3335, for CJK Unified Ideographs
        Context context = mUiExtensionsManager.getAttachedActivity().getApplicationContext();
        String defaultLanguage = context.getResources().getConfiguration().locale.getLanguage();
        Locale locale;
        if (defaultLanguage.contains(Locale.CHINA.getLanguage())) {
            locale = Locale.CHINA;
        } else if (defaultLanguage.contains(Locale.JAPANESE.getLanguage())) {
            locale = Locale.JAPANESE;
        } else if (defaultLanguage.contains(Locale.KOREAN.getLanguage())) {
            locale = Locale.KOREAN;
        } else {
            locale = defaultLocale;
        }
        return locale;
    }

    private Locale getDefaultLocaleLanguage() {
        if (mTTS != null) {
            Locale locale = mUiExtensionsManager.getAttachedActivity().getApplicationContext().getResources().getConfiguration().locale;
            if (AppBuildConfig.SDK_VERSION >= 18) {
                locale = mTTS.getDefaultLanguage();
            }
            int ret = mTTS.isLanguageAvailable(locale);
            if (ret != TextToSpeech.LANG_MISSING_DATA && ret != TextToSpeech.LANG_NOT_SUPPORTED) {
                return locale;
            }
        }
        return Locale.US;
    }

    private boolean speakStringAfterParsing() {
        if (mTTS == null) {
            return false;
        }
        if (mCurTtsStrList.size() != 0 && mCurTtsStrList.size() == mCurTtsLocalLanguageList.size()) {
            int ret = mTTS.setLanguage(mCurTtsLocalLanguageList.remove(0));
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP && mCurrentVoice != null) {
                mTTS.setVoice(mCurrentVoice);
            }
            if (ret == TextToSpeech.LANG_MISSING_DATA || ret == TextToSpeech.LANG_NOT_SUPPORTED) {
                mCurTtsStrList.remove(0);
                return false;
            }
            speak(mCurTtsStrList.remove(0));
            return true;
        } else {
            return false;
        }
    }

    private void speak(String things) {
        mIsSpeaking = true;
        mIsPause = false;
        mIsExit = false;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.ICE_CREAM_SANDWICH_MR1) {
            mTTS.setOnUtteranceProgressListener(new UtteranceProgressListener() {
                @Override
                public void onStart(String utteranceId) {
                    AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
                        @Override
                        public void run() {
                            setAllTtsItemByState();
                            mPdfViewCtrl.invalidate();
                        }
                    });
                }

                @Override
                public void onDone(String utteranceId) {
                    if (!AppUtil.isEmpty(utteranceId) && utteranceId.equals(mTTSUtteranceIdMap.get(TextToSpeech.Engine.KEY_PARAM_UTTERANCE_ID))) {
                        AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
                            @Override
                            public void run() {
                                mTTSUtteranceIdMap.clear();
                                if (mIsExit) {
                                    mIsSpeaking = false;
                                    mIsPause = false;
                                    mCurTtsRectInfo = null;
                                    mCurTtsStrList.clear();
                                    mCurTtsLocalLanguageList.clear();
                                    mCurTtsRectPageIndex = -1;
                                } else {
                                    speakNextLine();
                                }
                                setAllTtsItemByState();
                                mPdfViewCtrl.invalidate();
                            }
                        });
                    }
                }

                @Override
                public void onError(String utteranceId) {
                    if (!AppUtil.isEmpty(utteranceId) && utteranceId.equals(mTTSUtteranceIdMap.get(TextToSpeech.Engine.KEY_PARAM_UTTERANCE_ID))) {
                        AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
                            @Override
                            public void run() {
                                mTTSUtteranceIdMap.clear();
                                if (mIsExit) {
                                    mIsSpeaking = false;
                                    mIsPause = false;
                                    mCurTtsRectInfo = null;
                                    mCurTtsStrList.clear();
                                    mCurTtsLocalLanguageList.clear();
                                    mCurTtsRectPageIndex = -1;
                                } else {
                                    speakNextLine();
                                }
                                setAllTtsItemByState();
                                mPdfViewCtrl.invalidate();
                            }
                        });
                    }
                }
            });
        }
        mTTSUtteranceIdMap.put(TextToSpeech.Engine.KEY_PARAM_UTTERANCE_ID, "UniqueID:" + Math.random());
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            mTTS.speak(things, TextToSpeech.QUEUE_FLUSH, null, mTTSUtteranceIdMap.get(TextToSpeech.Engine.KEY_PARAM_UTTERANCE_ID));
        } else {
            mTTS.speak(things, TextToSpeech.QUEUE_FLUSH, mTTSUtteranceIdMap);
        }
    }

    private void speakNextLine() {
        if (mIsPause) {
            return;
        }
        if (speakStringAfterParsing()) {
            return;
        }
        if (mTtsRectInfoList.size() != 0) {
            mCurTtsRectInfo = mTtsRectInfoList.remove(0);
            parsingTtsRectInfo(mCurTtsRectInfo.mText);
            speakStringAfterParsing();
        } else {
            if (mbOnlySelect) {
                if (mIsCycleSpeak) {
                    mCurTtsRectPageIndex = mCurTtsRectInfo.mPageIndex;
                    parsingTtsRectInfo(mCurTtsRectInfo.mText);
                    speakStringAfterParsing();
                } else {
                    mIsSpeaking = false;
                    mIsPause = false;
                    mIsExit = true;
                    mCurTtsRectInfo = null;
                    mCurTtsStrList.clear();
                    mCurTtsLocalLanguageList.clear();
                    mCurTtsRectPageIndex = -1;
                    mbOnlySelect = false;
                }
                return;
            }

            if (mIsCycleSpeak) {
                final int pageIndex = mCurTtsRectPageIndex;
                PDFPage page = mUiExtensionsManager.getDocumentManager().getPage(pageIndex, false);
                if (page == null || page.isEmpty()) return;

                cancelSearchTask();
                final int startIndex = 0;
                final ArrayList<TTSInfo> ttsInfos = getTTSInfosFromCache(pageIndex, startIndex);
                if (ttsInfos != null) {
                    if (ttsInfos.size() != 0) {
                        mTtsRectInfoList.addAll(ttsInfos);
                        mCurTtsRectInfo = mTtsRectInfoList.remove(0);
                        parsingTtsRectInfo(mCurTtsRectInfo.mText);
                        speakStringAfterParsing();
                    } else {
                        mIsSpeaking = false;
                        mIsPause = false;
                        mIsExit = true;
                        mCurTtsRectInfo = null;
                        mCurTtsStrList.clear();
                        mCurTtsLocalLanguageList.clear();
                        mCurTtsRectPageIndex = -1;
                    }
                } else {
                    SearchTTSInfoTask searchTask = new SearchTTSInfoTask(mPdfViewCtrl, page, startIndex, new IResult<ArrayList<TTSInfo>, Object, Object>() {
                        @Override
                        public void onResult(boolean success, ArrayList<TTSInfo> ttsInfos, Object p2, Object p3) {
                            if (success && ttsInfos != null) {
                                if (ttsInfos.size() != 0) {
                                    mTtsRectInfoList.addAll(ttsInfos);
                                    mCurTtsRectInfo = mTtsRectInfoList.remove(0);
                                    parsingTtsRectInfo(mCurTtsRectInfo.mText);
                                    speakStringAfterParsing();
                                } else {
                                    mIsSpeaking = false;
                                    mIsPause = false;
                                    mIsExit = true;
                                    mCurTtsRectInfo = null;
                                    mCurTtsStrList.clear();
                                    mCurTtsLocalLanguageList.clear();
                                    mCurTtsRectPageIndex = -1;
                                }

                                saveTTSInfosToCache(pageIndex, startIndex, ttsInfos);
                            }
                        }
                    });
                    mPdfViewCtrl.addTask(searchTask);
                    mSearchTextTaskArrays.add(searchTask);
                }
            } else {
                final int pageIndex = mCurTtsRectPageIndex + 1;
                PDFPage page = mUiExtensionsManager.getDocumentManager().getPage(pageIndex, false);
                if (page == null || page.isEmpty()) {
                    mIsSpeaking = false;
                    mIsPause = false;
                    mIsExit = true;
                    mCurTtsRectInfo = null;
                    mCurTtsStrList.clear();
                    mCurTtsLocalLanguageList.clear();
                    mCurTtsRectPageIndex = -1;
                    return;
                }

                cancelSearchTask();
                final int startIndex = 0;
                final ArrayList<TTSInfo> infos = getTTSInfosFromCache(pageIndex, startIndex);
                if (infos != null) {
                    if (infos.size() != 0) {
                        mCurTtsRectPageIndex = pageIndex;
                        endEditStatus();
                        mPdfViewCtrl.gotoPage(pageIndex, 0, 0);
                        mTtsRectInfoList.addAll(infos);
                        mCurTtsRectInfo = mTtsRectInfoList.remove(0);
                        parsingTtsRectInfo(mCurTtsRectInfo.mText);
                        speakStringAfterParsing();
                    } else {
                        mCurTtsRectPageIndex = pageIndex;
                        speakNextLine();
                    }
                } else {
                    SearchTTSInfoTask searchTask = new SearchTTSInfoTask(mPdfViewCtrl, page, startIndex, new IResult<ArrayList<TTSInfo>, Object, Object>() {
                        @Override
                        public void onResult(boolean success, ArrayList<TTSInfo> infos, Object p2, Object p3) {
                            if (success && infos != null) {
                                if (infos.size() != 0) {
                                    mCurTtsRectPageIndex = pageIndex;
                                    endEditStatus();
                                    mPdfViewCtrl.gotoPage(pageIndex, 0, 0);
                                    mTtsRectInfoList.addAll(infos);
                                    mCurTtsRectInfo = mTtsRectInfoList.remove(0);
                                    parsingTtsRectInfo(mCurTtsRectInfo.mText);
                                    speakStringAfterParsing();
                                } else {
                                    mCurTtsRectPageIndex = pageIndex;
                                    speakNextLine();
                                }
                                saveTTSInfosToCache(pageIndex, startIndex, infos);
                            }
                        }
                    });
                    mPdfViewCtrl.addTask(searchTask);
                    mSearchTextTaskArrays.add(searchTask);
                }
            }
        }
    }

    private void endEditStatus() {
        if (mUiExtensionsManager.getCurrentToolHandler() != null) {
            mUiExtensionsManager.setCurrentToolHandler(null);
        }
        if (mUiExtensionsManager.getDocumentManager().getCurrentAnnot() != null) {
            mUiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
        }
    }

    private void registerViewSettingListener() {
        if (mSettingWindow == null) return;
        mSettingWindow.registerListener(mMulChangeTtsListener);
    }

    private void unRegisterViewSettingListener() {
        if (mSettingWindow == null) return;
        mSettingWindow.unRegisterListener(mMulChangeTtsListener);
    }

    private UIDragView mDragView;
    private boolean mAutoSpeak = false;
    private int mLastReadState;
    private final IViewSettingsWindow.IValueChangeListener mMulChangeTtsListener = new IViewSettingsWindow.IValueChangeListener() {

        @Override
        public void onValueChanged(int type, Object value) {
            if (!(value instanceof Boolean)) return;
            final MainFrame mainFrame = (MainFrame) ((UIExtensionsManager) mUiExtensionsManager).getMainFrame();
            if (type == IViewSettingsWindow.TYPE_TTS) {
                mainFrame.hideSettingWindow();

                if ((boolean) value) {
                    mUiExtensionsManager.requestPhoneStatePermission(AppResource.getString(mContext, R.string.audio_fun_name), new Event.Callback() {
                        @Override
                        public void result(Event event, boolean success) {
                            if (AppDisplay.isPad()) {
                                if (mTtsBottomBar == null) {
                                    initPaint();
                                    initBars();
                                }
                                mDragView = createDrawView();
                                mDragView.setVisible(!mainFrame.isShowFullScreenUI());
                            }
                            mAutoSpeak = true;
                            mLastReadState = mUiExtensionsManager.getState();
                            mUiExtensionsManager.changeState(ReadStateConfig.STATE_TTS);
                            if (!mainFrame.isToolbarsVisible() && !mainFrame.isShowFullScreenUI())
                                mainFrame.showToolbars();

                            if(isMaskShowing()) {
                                mUiExtensionsManager.stopHideToolbarsTimer();;
                            }
                        }
                    });
                } else {
                    if (mDragView != null) {
                        mDragView.removeDragView();
                        mDragView = null;
                    }
                    mAutoSpeak = false;
                    if (mLastReadState == ReadStateConfig.STATE_FILLSIGN)
                        mUiExtensionsManager.changeState(ReadStateConfig.STATE_FILLSIGN);
                    else
                        mUiExtensionsManager.changeState(ReadStateConfig.STATE_NORMAL);
                }
            }
        }

        @Override
        public int getType() {
            return IViewSettingsWindow.TYPE_TTS;
        }

    };

    private final ILayoutChangeListener mLayoutChangeListener = new ILayoutChangeListener() {
        @Override
        public void onLayoutChange(View v, int newWidth, int newHeight, int oldWidth, int oldHeight) {
            if (mSpeedSheet != null && mSpeedSheet.isShowing()) {
                if (newWidth != oldWidth || newHeight != oldHeight) {
                    new Handler().postDelayed(new Runnable() {
                        @Override
                        public void run() {
                            Rect rect = new Rect();
                            mSpeedItem.getContentView().getGlobalVisibleRect(rect);
                            int pos;
                            if (rect.bottom < AppDisplay.getScreenHeight() / 2) {
                                pos = UIPopoverWin.ARROW_TOP;
                            } else {
                                pos = UIPopoverWin.ARROW_BOTTOM;
                            }
                            initSpeedRateDialog();
                            mSpeedSheet.update(mUiExtensionsManager.getRootView(), rect, pos);
                        }
                    }, 200);
                }
            }
        }
    };

    private final IStateChangeListener mStatusChangeListener = new IStateChangeListener() {
        @Override
        public void onStateChanged(int oldState, int newState) {
            onStatusChanged(oldState, newState);
            if (mSettingWindow == null) {
                mSettingWindow = mUiExtensionsManager.getMainFrame().getSettingWindow();
            }
            if ((newState == ReadStateConfig.STATE_TTS && oldState != ReadStateConfig.STATE_TTS) || mAI_speak) {
                final Runnable speanRunnable = new Runnable() {
                    @Override
                    public void run() {
                        Log.d("suyu", "AI-speak, "+ mAutoSpeak);
                        mAI_speak = false;
                        SoundModule soundModule = (SoundModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_SOUND);
                        if (soundModule != null && soundModule.getAudioPlayView() != null) {
                            soundModule.getAudioPlayView().release();
                        }

                        initTTS();
                        mSettingWindow.setProperty(IViewSettingsWindow.TYPE_TTS, true);
                    }
                };

                boolean isFirstOpen = AppSharedPreferences.getInstance(mContext).getBoolean("FIRST_OPEN_TTS", true);
                if (isFirstOpen) {
                    AppSharedPreferences.getInstance(mContext).setBoolean("FIRST_OPEN_TTS", false);

                    showMaskPromptDlg(new IResult<Void, Void, Void>() {
                        @Override
                        public void onResult(boolean success, Void p1, Void p2, Void p3) {
                            mShowVoicePrompt = false;
                            mTtsBottomBar.getContentView().setVisibility(View.VISIBLE);
                            mUiExtensionsManager.getMainFrame().showToolbars();
                            speanRunnable.run();
                        }
                    });

                    mTtsBottomBar.getContentView().setVisibility(View.INVISIBLE);
                } else {
                    speanRunnable.run();
                }
            } else if (newState != ReadStateConfig.STATE_TTS && oldState == ReadStateConfig.STATE_TTS) {
                cancelSearchTask();
                resetAllTtsItem();
                stopTts();
                mSettingWindow.setProperty(IViewSettingsWindow.TYPE_TTS, false);
            }
        }
    };

    public void stopSpeak() {
        cancelSearchTask();
        stopTts();
    }

    private void initTTS() {
        if (mTsChangeStatus) {
            mTsChangeStatus = false;
            mTtsStartItem.setEnable(true);
        } else {
            if (mTTS != null) {
                mTTS.shutdown();
                mTTS = null;
            }
            resetState();
            resetAllTtsItem();
            AppThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
                @Override
                public void run() {
                    if (mUiExtensionsManager.getAttachedActivity() == null) return;
                    mTTS = new TextToSpeech(mContext, mTextToSpeechOnInitListener);
                }
            }, 200);
        }
    }

    private final ILifecycleEventListener mRDLifecycleEventListener = new LifecycleEventListener() {
        @Override
        public void onHiddenChanged(boolean hidden) {
            if (mIsPad && hidden)
                onKeyBack();
        }
    };

    private final PDFViewCtrl.IDocEventListener mDocEventListener = new DocEventListener() {
        @Override
        public void onDocWillOpen() {
        }

        @Override
        public void onDocOpened(PDFDoc document, int errCode) {
            if (errCode == Constants.e_ErrSuccess) {
                boolean enabled;
                enabled = isSupperTts()
                        && document != null
                        && !mPdfViewCtrl.isDynamicXFA()
                        && mUiExtensionsManager.getDocumentManager().canCopy();

                if (mTtsStartItem != null)
                    mTtsStartItem.setEnable(enabled);
                if (mSettingWindow == null) {
                    mSettingWindow = mUiExtensionsManager.getMainFrame().getSettingWindow();
                }
                mSettingWindow.enableBar(IViewSettingsWindow.TYPE_TTS, enabled);
                registerViewSettingListener();

                if (mUiExtensionsManager != null && mUiExtensionsManager.getState() == ReadStateConfig.STATE_TTS) {
                    mUiExtensionsManager.changeState(ReadStateConfig.STATE_NORMAL);
                }
            }
        }

        @Override
        public void onDocWillClose(PDFDoc document) {
        }

        @Override
        public void onDocClosed(PDFDoc document, int errCode) {
            cancelSearchTask();
            unRegisterViewSettingListener();
            mTTSInfoArrays.clear();
            mSettingWindow = null;
        }

    };

    private RectF tmpRectF;
    private Paint mPaint = new Paint();

    private void initPaint() {
        mPaint.setAntiAlias(true);
        mPaint.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.MULTIPLY));
        mPaint.setColor(AppDmUtil.calColorByMultiply(0x73C1E1, 150));
    }

    private final PDFViewCtrl.IDrawEventListener mDrawEventListener = new PDFViewCtrl.IDrawEventListener() {


        @Override
        public void onDraw(int pageIndex, Canvas canvas) {
            if ((mCurTtsRectInfo != null && mCurTtsRectPageIndex == pageIndex)) {
                for (RectF rectF : mCurTtsRectInfo.mRects) {
                    tmpRectF = new RectF(rectF);
                    mPdfViewCtrl.convertPdfRectToPageViewRect(tmpRectF, tmpRectF, pageIndex);
                    canvas.drawRect(tmpRectF, mPaint);
                }
            }
        }
    };

    private final TextToSpeech.OnInitListener mTextToSpeechOnInitListener = new TextToSpeech.OnInitListener() {
        @Override
        public void onInit(int status) {
            mHasInitTts = true;

            boolean enabled;
            if (status == TextToSpeech.SUCCESS) {
                try {
                    if (mSupportVoices == null) {
                        AppThreadManager.getInstance().startThread(new Runnable() {
                            @Override
                            public void run() {
                                if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.LOLLIPOP) {
                                    Set<Voice> voices = null;
                                    try {
                                        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.LOLLIPOP) {
                                            voices = mTTS.getVoices();
                                            if (voices != null && voices.size() > 0) {
                                                for (Voice voice : voices) {
                                                    Locale locale = voice.getLocale();
                                                    String language = locale.getDisplayLanguage();

                                                    TtsVoiceGroup group = findVoiceGroup(language);
                                                    if (group == null) {
                                                        group = new TtsVoiceGroup();
                                                        group.language = language;
                                                        addVoiceGroup(group);
                                                    }

                                                    group.addVoice(voice);

                                                    if (AppUtil.isEqual(voice.getName(), mCurrentVoiceName)) {
                                                        mCurrentVoice = voice;
                                                        mCurrentLocale = voice.getLocale();
                                                    }
                                                }
                                            }
                                        }

                                        try {
                                            Collections.sort(mLanguageList, new TtsLanguageComparator());
                                        } catch (Exception e) {
                                            e.printStackTrace();
                                        }
                                    } catch (Exception e) {

                                    } finally {
                                        if (voices == null || voices.size() == 0) {
                                            mSupportVoices = false;
                                        } else {
                                            mSupportVoices = true;
                                        }
                                    }
                                } else {
                                    mSupportVoices = false;
                                }
                            }
                        });
                    }

                    mTTS.setPitch(mTtsPitch);
                    mTTS.setSpeechRate(mTtsSpeechRate);
                    int ret = mTTS.isLanguageAvailable(Locale.US);
                    if (ret == TextToSpeech.LANG_MISSING_DATA || ret == TextToSpeech.LANG_NOT_SUPPORTED) {
                    } else {
                        ret = mTTS.setLanguage(Locale.US);
                    }
                    if (ret == TextToSpeech.LANG_MISSING_DATA || ret == TextToSpeech.LANG_NOT_SUPPORTED) {
                        mSupperTts = false;
                        enabled = false;
                    } else {
                        mSupperTts = true;
                        enabled = true;
                    }
                } catch (Exception e) {
                    mSupperTts = false;
                    enabled = false;
                }
            } else {
                mSupperTts = false;
                enabled = false;
            }

            if (mTtsStartItem != null) {
                mTtsStartItem.setEnable(enabled);
                if (mSettingWindow == null) {
                    mSettingWindow = mUiExtensionsManager.getMainFrame().getSettingWindow();
                }
                mSettingWindow.enableBar(IViewSettingsWindow.TYPE_TTS, enabled);
            }

            if (mAutoSpeak) {
                mAutoSpeak = false;
                startSpeak();
            }
        }
    };

    private boolean mAI_speak;
    public void AI_startSpeak() {
        mUiExtensionsManager.requestPhoneStatePermission(AppResource.getString(mContext, R.string.audio_fun_name), new Event.Callback() {
            @Override
            public void result(Event event, boolean success) {
                if (mTtsStartItem == null) {
                    initPaint();
                    initBars();
                }
                mAI_speak = true;
                mAutoSpeak = true;
                mUiExtensionsManager.changeState(ReadStateConfig.STATE_TTS);
            }
        });
    }

    public boolean isSpeaking() {
        return !mIsPause;
    }
    public void startSpeak() {
        final int pageIndex = mPdfViewCtrl.getCurrentPage();
        PDFPage page = mUiExtensionsManager.getDocumentManager().getPage(pageIndex, false);
        if (page == null || page.isEmpty()) return;

        cancelSearchTask();
        final int startIndex = 0;
        final ArrayList<TTSInfo> infos = getTTSInfosFromCache(pageIndex, startIndex);
        if (infos != null) {
            if (infos.size() != 0) {
                mTtsRectInfoList.addAll(infos);
                mCurTtsRectPageIndex = pageIndex;
                mCurTtsRectInfo = mTtsRectInfoList.remove(0);
                while ((mCurTtsRectInfo == null || AppUtil.isEmpty(mCurTtsRectInfo.mText)) && mTtsRectInfoList.size() != 0) {
                    mCurTtsRectInfo = mTtsRectInfoList.remove(0);
                }
                if (mCurTtsRectInfo == null || AppUtil.isEmpty(mCurTtsRectInfo.mText)) {
                    if (mIsCycleSpeak) {
                        mIsSpeaking = false;
                        mIsPause = false;
                        mIsExit = true;
                        mCurTtsRectPageIndex = -1;
                        mCurTtsRectInfo = null;
                        mCurTtsStrList.clear();
                        mCurTtsLocalLanguageList.clear();
                        setAllTtsItemByState();
                        return;
                    } else {
                        speakNextLine();
                        setAllTtsItemByState();
                        mPdfViewCtrl.invalidate();
                        return;
                    }
                }

                parsingTtsRectInfo(mCurTtsRectInfo.mText);
                speakStringAfterParsing();
                setAllTtsItemByState();
                mPdfViewCtrl.invalidate();
            } else {
                if (!mIsCycleSpeak) {
                    mCurTtsRectPageIndex = pageIndex;
                    speakNextLine();
                    setAllTtsItemByState();
                    mPdfViewCtrl.invalidate();
                }
            }
        } else {
            SearchTTSInfoTask searchTask = new SearchTTSInfoTask(mPdfViewCtrl, page, startIndex, new IResult<ArrayList<TTSInfo>, Object, Object>() {
                @Override
                public void onResult(boolean success, ArrayList<TTSInfo> infos, Object p2, Object p3) {
                    if (success && infos != null) {
                        if (infos.size() != 0) {
                            mTtsRectInfoList.addAll(infos);
                            mCurTtsRectPageIndex = pageIndex;
                            mCurTtsRectInfo = mTtsRectInfoList.remove(0);
                            while ((mCurTtsRectInfo == null || AppUtil.isEmpty(mCurTtsRectInfo.mText)) && mTtsRectInfoList.size() != 0) {
                                mCurTtsRectInfo = mTtsRectInfoList.remove(0);
                            }
                            if (mCurTtsRectInfo == null || AppUtil.isEmpty(mCurTtsRectInfo.mText)) {
                                if (mIsCycleSpeak) {
                                    mIsSpeaking = false;
                                    mIsPause = false;
                                    mIsExit = true;
                                    mCurTtsRectPageIndex = -1;
                                    mCurTtsRectInfo = null;
                                    mCurTtsStrList.clear();
                                    mCurTtsLocalLanguageList.clear();
                                    setAllTtsItemByState();
                                    return;
                                } else {
                                    speakNextLine();
                                    setAllTtsItemByState();
                                    mPdfViewCtrl.invalidate();
                                    return;
                                }
                            }

                            parsingTtsRectInfo(mCurTtsRectInfo.mText);
                            speakStringAfterParsing();
                            setAllTtsItemByState();
                            mPdfViewCtrl.invalidate();
                        } else {
                            if (!mIsCycleSpeak) {
                                mCurTtsRectPageIndex = pageIndex;
                                speakNextLine();
                                setAllTtsItemByState();
                                mPdfViewCtrl.invalidate();
                            }
                        }
                        saveTTSInfosToCache(pageIndex, startIndex, infos);
                    }
                }
            });
            mPdfViewCtrl.addTask(searchTask);
            mSearchTextTaskArrays.add(searchTask);
        }
    }

    private void onStatusChanged(int oldState, int newState) {
        if (mPdfViewCtrl.getDoc() == null) {
            return;
        }
        if (mUiExtensionsManager.getState() == ReadStateConfig.STATE_TTS) {
            if (mTtsBottomBar == null) {
                initPaint();
                initBars();
            }
            if (AppDisplay.isPad()) {
                if (mDragView == null)
                    mDragView = createDrawView();
                MainFrame mainFrame = (MainFrame) ((UIExtensionsManager) mUiExtensionsManager).getMainFrame();
                mDragView.setVisible(!mainFrame.isShowFullScreenUI());
            } else {
                boolean isToolbarsVisible = mUiExtensionsManager.getMainFrame().isToolbarsVisible();
                if (isToolbarsVisible) {
                    mTtsTopBar.getContentView().startAnimation(mUiExtensionsManager.getMainFrame().getTopbarShowAnimation());
                    mTtsBottomBar.getContentView().startAnimation(mUiExtensionsManager.getMainFrame().getBottombarShowAnimation());

                    mTtsTopBar.getContentView().setVisibility(View.VISIBLE);
                    mTtsBottomBar.getContentView().setVisibility(View.VISIBLE);
                } else {
                    if (mTtsTopBar.getContentView().getVisibility() == View.VISIBLE) {
                        mTtsTopBar.getContentView().startAnimation(mUiExtensionsManager.getMainFrame().getTopbarHideAnimation());
                        mTtsTopBar.getContentView().setVisibility(View.INVISIBLE);
                    }

                    if (mTtsBottomBar.getContentView().getVisibility() == View.VISIBLE) {
                        mTtsBottomBar.getContentView().startAnimation(mUiExtensionsManager.getMainFrame().getBottombarHideAnimation());
                        mTtsBottomBar.getContentView().setVisibility(View.INVISIBLE);
                    }
                }
                mPdfViewCtrl.offsetScrollBoundary(0, isToolbarsVisible ? mTtsTopBar.getContentView().getHeight() : 0,
                        0, isToolbarsVisible ? -mTtsBottomBar.getContentView().getHeight() : 0);
                mPdfViewCtrl.postPageContainer();
            }
        } else {
            if (AppDisplay.isPad()) {
                if (mDragView != null) {
                    mDragView.removeDragView();
                    mDragView = null;
                }
            } else {
                if (mTtsBottomBar != null) {
                    mTtsBottomBar.getContentView().setVisibility(View.INVISIBLE);
                    mTtsTopBar.getContentView().setVisibility(View.INVISIBLE);
                }
            }
        }
    }

    private void resetState() {
        mbOnlySelect = false;
        mIsSpeaking = false;
        mIsPause = false;
        mIsExit = true;
        mIsCycleSpeak = false;
        mTtsPitch = 1.0f;
        mCurTtsRectInfo = null;
        mCurTtsStrList.clear();
        mTTSUtteranceIdMap.clear();
        mCurTtsLocalLanguageList.clear();
        mTtsRectInfoList.clear();
        mCurTtsRectPageIndex = -1;
    }

    private void resetAllTtsItem() {
        mTtsStartItem.setImageResource(R.drawable.tts_start);
        mTtsPreviousPageItem.setEnable(false);
        mTtsNextPageItem.setEnable(false);
        mTtsStartItem.setEnable(false);
        mTtsStopItem.setEnable(false);
        mTtsCycleItem.setChecked(mIsCycleSpeak);
    }

    private UIMatchDialog mMaskPromptDlg;
    private UIActionMenu mMoreAction;
    private UIActionMenu mLanguageListAction;
    private UIActionMenu mVoiceListAction;
    private ISheetMenu mSpeedSheet;
    private IBaseItem mCurrentTabItem;
    private List<IBaseItem> mSpeedDlgItems = new ArrayList<>();

    private final static int SPEED_RATE_0_5_X = 0;
    private final static int SPEED_RATE_0_75_X = 1;
    private final static int SPEED_RATE_1_X = 2;
    private final static int SPEED_RATE_1_25_X = 3;
    private final static int SPEED_RATE_1_5_X = 4;
    private final static int SPEED_RATE_2_X = 5;

    private void setCurrentTabItem(IBaseItem item) {
        mCurrentTabItem.setSelected(false);
        item.setSelected(true);
        mCurrentTabItem = item;
    }

    private void addItem(int tag, String text) {
        final IBaseItem centerItem = new BaseItemImpl(mContext, text);
        centerItem.setTag(tag);
        centerItem.setOnItemClickListener(new IBaseItem.OnItemClickListener() {
            @Override
            public void onClick(IBaseItem item, View v) {
                setCurrentTabItem(centerItem);
            }
        });

        if (!mSpeedDlgItems.contains(centerItem))
            mSpeedDlgItems.add(centerItem);
    }

    private void addSpeedRateItems() {
        addItem(SPEED_RATE_0_5_X, "0.5X");
        addItem(SPEED_RATE_0_75_X, "0.75X");
        addItem(SPEED_RATE_1_X, "1X");
        addItem(SPEED_RATE_1_25_X, "1.25X");
        addItem(SPEED_RATE_1_5_X, "1.5X");
        addItem(SPEED_RATE_2_X, "2X");
    }

    private int convertSpeedRateValueToType(float value) {
        int type = SPEED_RATE_1_X;
        if (value == 0.5) {
            type = SPEED_RATE_0_5_X;
        } else if (value == 0.75) {
            type = SPEED_RATE_0_75_X;
        } else if (value == 1) {
            type = SPEED_RATE_1_X;
        } else if (value == 1.25) {
            type = SPEED_RATE_1_25_X;
        } else if (value == 1.5) {
            type = SPEED_RATE_1_5_X;
        } else if (value == 2) {
            type = SPEED_RATE_2_X;
        }
        return type;
    }

    private int ConvertSpeechRateTypeToImageRes(int type) {
        int resId = R.drawable.tts_speed_rate_1x;
        if (type == SPEED_RATE_0_5_X) {
            resId = R.drawable.tts_speed_rate_05x;
        } else if (type == SPEED_RATE_0_75_X) {
            resId = R.drawable.tts_speed_rate_075x;
        } else if (type == SPEED_RATE_1_X) {
            resId = R.drawable.tts_speed_rate_1x;
        } else if (type == SPEED_RATE_1_25_X) {
            resId = R.drawable.tts_speed_rate_125x;
        } else if (type == SPEED_RATE_1_5_X) {
            resId = R.drawable.tts_speed_rate_15x;
        } else if (type == SPEED_RATE_2_X) {
            resId = R.drawable.tts_speed_rate_2x;
        }
        return resId;
    }

    private float ConvertSpeechRateTypeToValue(int type) {
        float value = 1.0f;
        if (type == SPEED_RATE_0_5_X) {
            value = 0.5f;
        } else if (type == SPEED_RATE_0_75_X) {
            value = 0.75f;
        } else if (type == SPEED_RATE_1_X) {
            value = 1.0f;
        } else if (type == SPEED_RATE_1_25_X) {
            value = 1.25f;
        } else if (type == SPEED_RATE_1_5_X) {
            value = 1.5f;
        } else if (type == SPEED_RATE_2_X) {
            value = 2.0f;
        }
        return value;
    }

    private void startSpeak(final ArrayList<TTSInfo> infos, final int pageIndex) {
        if (infos != null && infos.size() != 0) {
            if (mTTS.isSpeaking()) {
                mTTSUtteranceIdMap.clear();
                mTTS.stop();
            }
            AppThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
                @Override
                public void run() {
                    if (!mIsExit) {
                        mIsSpeaking = false;
                        mTtsRectInfoList.clear();
                        mCurTtsRectPageIndex = pageIndex;
                        endEditStatus();
                        mPdfViewCtrl.gotoPage(pageIndex, 0, 0);
                        mTtsRectInfoList.addAll(infos);
                        mCurTtsRectInfo = mTtsRectInfoList.remove(0);
                        parsingTtsRectInfo(mCurTtsRectInfo.mText);
                        speakStringAfterParsing();
                    }
                }
            }, 200);
        } else {
            if (mTTS.isSpeaking()) {
                mTTSUtteranceIdMap.clear();
                mTTS.stop();
            }

            mCurTtsRectInfo = null;
            mIsSpeaking = false;
            mTtsRectInfoList.clear();
            mCurTtsStrList.clear();
            mCurTtsRectPageIndex = pageIndex;
            endEditStatus();
            mPdfViewCtrl.gotoPage(pageIndex, 0, 0);
            setAllTtsItemByState();
            mPdfViewCtrl.invalidate();
        }
    }

    private ArrayList<TTSInfo> getTTSInfosFromCache(int pageIndex, int startIndex) {
        final SparseArray<ArrayList<TTSInfo>> pageTTSInfos = mTTSInfoArrays.get(pageIndex);
        if (pageTTSInfos != null && pageTTSInfos.get(startIndex) != null) {
            return pageTTSInfos.get(startIndex);
        }
        return null;
    }

    private void saveTTSInfosToCache(int pageIndex, int startIndex, ArrayList<TTSInfo> ttsInfos) {
        SparseArray<ArrayList<TTSInfo>> speakInfoArrays = mTTSInfoArrays.get(pageIndex);
        if (speakInfoArrays == null) {
            speakInfoArrays = new SparseArray<>();
            mTTSInfoArrays.put(pageIndex, speakInfoArrays);
        }
        speakInfoArrays.put(startIndex, ttsInfos);
    }

    private class SearchTTSInfoTask extends Task {
        private ArrayList<TTSInfo> ttsInfos;
        private PDFViewCtrl pdfViewCtrl;
        private PDFPage pdfPage;
        private int startIndex;
        private boolean success;
        private boolean cancel;

        SearchTTSInfoTask(PDFViewCtrl viewCtrl, PDFPage page, int startIndex, final IResult<ArrayList<TTSInfo>, Object, Object> result) {
            super(new CallBack() {
                @Override
                public void result(Task task) {
                    SearchTTSInfoTask searchTask = (SearchTTSInfoTask) task;
                    result.onResult(searchTask.success, searchTask.ttsInfos, null, null);
                }
            });
            this.pdfViewCtrl = viewCtrl;
            this.pdfPage = page;
            this.startIndex = startIndex;
        }

        @Override
        protected void execute() {
            try {
                if (cancel || mPdfViewCtrl == null || mPdfViewCtrl.getDoc() == null) {
                    success = false;
                    return;
                }

                ttsInfos = new ArrayList<>();
                if (startIndex < 0)
                    startIndex = 0;

                StringBuilder sb = new StringBuilder();
                TextPage textPage = new TextPage(pdfPage, TextPage.e_ParseTextNormal);
                int textCount = textPage.getCharCount() - startIndex;
                int contentIndex = 0;
                for (int i = contentIndex; i < textCount; i++) {
                    if (cancel || mPdfViewCtrl == null || mPdfViewCtrl.getDoc() == null) {
                        success = false;
                        return;
                    }

                    String content = textPage.getChars(startIndex + i, 1);
                    sb.append(content);

                    char[] charArrays = content.toCharArray();
                    boolean isEnd = false;
                    if (charArrays.length > 0)
                        isEnd = TTSUtils.isEndPunct(charArrays[charArrays.length - 1]);

                    if (isEnd || i == textCount - 1) {
                        String str = sb.toString().replaceAll("[\r\n]", "");
                        if (isEnd && str.length() <= 1) {
                            //reset index
                            contentIndex = i + 1;
                            sb.setLength(0);
                            continue;
                        }

                        TTSInfo ttsInfo = new TTSInfo();
                        //get string
                        if (isSupportVoiceList() && isSelectedCurrentVoice())
                            ttsInfo.mText = str;
                        else
                            ttsInfo.mText = TTSUtils.changePunctsToEn(str);

                        //get RectF
                        TextSelector ts = new TextSelector(pdfViewCtrl);
                        ts.computeSelected(pdfPage, startIndex + contentIndex, startIndex + i);
                        ttsInfo.mRects.addAll(ts.getRectFList());
                        ttsInfos.add(ttsInfo);

                        //reset index
                        contentIndex = i + 1;
                        sb.setLength(0);
                    }
                }

                success = !cancel && mPdfViewCtrl != null && mPdfViewCtrl.getDoc() != null;
            } catch (PDFException e) {
                success = false;
                e.printStackTrace();
            }
        }

        void cancelTask() {
            cancel = true;
        }
    }

    private void cancelSearchTask() {
        for (SearchTTSInfoTask searchTask : mSearchTextTaskArrays) {
            searchTask.cancelTask();
        }
        mSearchTextTaskArrays.clear();
    }

    private IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            updateTheme();
        }
    };

    public void setDragToolbarVisible(boolean visible) {
        if (mDragView != null)
            mDragView.setVisible(visible);
    }

    private void updateTheme() {
        if (mTtsBottomBar == null) return;
        boolean isPad = AppDisplay.isPad();
        initPaint();
        mUiExtensionsManager.getMainFrame().getContentView().removeView(mTtsBottomBar.getContentView());
        mTtsBottomBar.removeAllItems();
        if (mSpeedSheet != null) {
            mSpeedSheet.dismiss();
            mSpeedSheet = null;
        }
        if (isPad) {
            mTtsBottomBar = new BaseBarImpl(mContext);
            mTtsBottomBar.setInterceptTouch(false);
            mTtsBottomBar.setItemInterval(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_bottombar_button_space_pad));
            mQuitItem = mQuitItem.cloneItem();
        } else {
            mUiExtensionsManager.getMainFrame().getContentView().removeView(mTtsTopBar.getContentView());
            mTtsTopBar.removeAllItems();

            mTtsTopBar = new TopBarImpl(mContext);
            mTtsTopBar.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
            mTtsTopBar.setMiddleButtonCenter(true);
            mTtsTopBar.setStartMargin(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_text_size_16sp));

            mTtsBackItem = mTtsBackItem.cloneItem();
            mTtsTitleItem = mTtsTitleItem.cloneItem();
            mTtsTitleItem.setTextColor(AppResource.getColor(mContext, R.color.t4));
            mTtsTopBar.addView(mTtsBackItem, BaseBar.TB_Position.Position_LT);
            mTtsTopBar.addView(mTtsTitleItem, BaseBar.TB_Position.Position_CENTER);

            RelativeLayout.LayoutParams ttsTopLp = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
            ttsTopLp.addRule(RelativeLayout.ALIGN_PARENT_TOP);
            mUiExtensionsManager.getMainFrame().getContentView().addView(mTtsTopBar.getContentView(), ttsTopLp);

            mTtsBottomBar = new BottomBarImpl(mContext);
            mTtsBottomBar.setItemInterval(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_speech_bottombar_button_space_phone));
            RelativeLayout.LayoutParams ttsBottomLp = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
            ttsBottomLp.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
            mUiExtensionsManager.getMainFrame().getContentView().addView(mTtsBottomBar.getContentView(), ttsBottomLp);
        }
        mTtsBottomBar.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
        mTtsBottomBar.setAutoCompressItemsInterval(true);

        mTtsPreviousPageItem = mTtsPreviousPageItem.cloneItem();
        mTtsStartItem = mTtsStartItem.cloneItem();
        mTtsStopItem = mTtsStopItem.cloneItem();
        mTtsNextPageItem = mTtsNextPageItem.cloneItem();
        mTtsCycleItem = mTtsCycleItem.cloneItem();
        mSpeedItem = mSpeedItem.cloneItem();

        mTtsBottomBar.addView(mTtsCycleItem, BaseBar.TB_Position.Position_CENTER);
        mTtsBottomBar.addView(mTtsPreviousPageItem, BaseBar.TB_Position.Position_CENTER);
        mTtsBottomBar.addView(mTtsStartItem, BaseBar.TB_Position.Position_CENTER);
//        mTtsBottomBar.addView(mTtsStopItem, BaseBar.TB_Position.Position_CENTER);
        mTtsBottomBar.addView(mTtsNextPageItem, BaseBar.TB_Position.Position_CENTER);
        mTtsBottomBar.addView(mSpeedItem, BaseBar.TB_Position.Position_CENTER);
        if (isPad)
            mTtsBottomBar.addView(mQuitItem, BaseBar.TB_Position.Position_CENTER);

        if (mUiExtensionsManager.getState() == ReadStateConfig.STATE_TTS) {
            if (AppDisplay.isPad()) {
                mDragView.removeDragView();
                mDragView = null;
                mDragView = createDrawView();
                MainFrame mainFrame = (MainFrame) mUiExtensionsManager.getMainFrame();
                mDragView.setVisible(!mainFrame.isShowFullScreenUI());
            } else {
                mTtsTopBar.getContentView().setVisibility(View.VISIBLE);
                mTtsBottomBar.getContentView().setVisibility(View.VISIBLE);
            }
        } else {
            if (isPad) {
                if (mDragView != null) {
                    mDragView.removeDragView();
                    mDragView = null;
                }
            } else {
                mTtsTopBar.getContentView().setVisibility(View.INVISIBLE);
                mTtsBottomBar.getContentView().setVisibility(View.INVISIBLE);
            }
        }
    }

    private UIDragView createDrawView() {
        if (mDragView == null) {
            int width = 0;
            if (AppDevice.isChromeOs(((UIExtensionsManager) mUiExtensionsManager).getAttachedActivity()))
                width = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_bottombar_button_space_phone) * 5 +
                        AppResource.getDimensionPixelSize(mContext, R.dimen.ux_tool_icon_size) * 6 +
                        AppDisplay.dp2px(+20);
            else
                width = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_bottombar_button_space_pad) * 5 +
                        AppResource.getDimensionPixelSize(mContext, R.dimen.ux_tool_icon_size) * 6 +
                        AppDisplay.dp2px(+20);
            mDragView = new UIDragView(mTtsBottomBar.getContentView(), mUiExtensionsManager, width);
        }
        return mDragView;
    }

    public void setTtsSpeechRate(float ttsSpeechRate) {
        mTtsSpeechRate = ttsSpeechRate;

        if (mSpeedItem != null && !isSupportVoiceList())
            mSpeedItem.setImageResource(getSpeechRateImageRes(mTtsSpeechRate));

        if (mCurrentTabItem != null) {
            mCurrentTabItem.setSelected(false);
            mCurrentTabItem = mSpeedDlgItems.get(convertSpeedRateValueToType(mTtsSpeechRate));
            mCurrentTabItem.setSelected(true);
        }

        if (mHasInitTts) {
            mTTS.setSpeechRate(mTtsSpeechRate);
        }
    }

    public float getTtsSpeechRate() {
        return mTtsSpeechRate;
    }

    public void setCurrentVoiceName(String name) {
        if (AppUtil.isEmpty(name)) {
            mCurrentLocale = getDefaultLocaleLanguage();
        }
        mCurrentVoiceName = name;
    }

    public String getCurrentVoiceName() {
        if (mCurrentVoice != null) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                return mCurrentVoice.getName();
            }
        }
        return "";
    }

    public void setShowVoicePrompt(boolean showVoicePrompt) {
        mShowVoicePrompt = showVoicePrompt;
    }

    public boolean getShowVoicePrompt() {
        return mShowVoicePrompt;
    }

    private final PDFViewCtrl.IPageEventListener mPageEventListener = new PageEventListener() {
        @Override
        public void onPagesRemoved(boolean success, int[] pageIndexes) {
            if (mIsSpeaking && mQuitItem != null) {
                if (success && pageIndexes != null && pageIndexes.length > 1) {
                    for (int pageIndex : pageIndexes) {
                        if (pageIndex == mCurTtsRectPageIndex) {
                            mPdfViewCtrl.post(new Runnable() {
                                @Override
                                public void run() {
                                    mQuitItem.performClick();
                                }
                            });
                            break;
                        }

                    }
                }
            }

        }
    };

    private void showMaskPromptDlg(IResult<Void, Void, Void> result) {
        mUiExtensionsManager.stopHideToolbarsTimer();
        initMaskPromptDlg(result);

        mMaskPromptDlg.showDialog();
    }

    TextView mVoiceSelectiontext;
    ImageView mRateSelectionIcon;
    TextView mRateSelectionIconText;
    private void initMoreDialog() {
        if (mMoreAction == null) {
            MenuViewImpl menuView = new MenuViewImpl(mUiExtensionsManager.getAttachedActivity());
            menuView.setTitleBarVisible(false);

            IMenuGroup menuGroup = menuView.addGroup("");
            menuGroup.setHeaderTitleVisible(false);

            View voiceSelectView = View.inflate(mUiExtensionsManager.getAttachedActivity(), R.layout.tts_more_item, null);
            ((ImageView)voiceSelectView.findViewById(R.id.tts_more_item_icon)).setImageResource(R.drawable.tts_voice_selection);
            ((TextView)voiceSelectView.findViewById(R.id.tts_more_item_caption)).setText(R.string.tts_voice_selection);
            mVoiceSelectiontext = (TextView)voiceSelectView.findViewById(R.id.tts_more_item_selection_text);
            IMenuItem voiceSelectItem = menuGroup.addItemAt(0, voiceSelectView);

            View speedSelectView = View.inflate(mUiExtensionsManager.getAttachedActivity(), R.layout.tts_more_item, null);
            ((ImageView)speedSelectView.findViewById(R.id.tts_more_item_icon)).setImageResource(R.drawable.tts_speech_rate);
            ((TextView)speedSelectView.findViewById(R.id.tts_more_item_caption)).setText(R.string.tts_speech_rate);
            mRateSelectionIconText = (TextView)speedSelectView.findViewById(R.id.tts_more_item_selection_text);
            //mRateSelectionIconText.setVisibility(View.GONE);
            mRateSelectionIcon = (ImageView)speedSelectView.findViewById(R.id.tts_more_item_selection_icon);
            //mRateSelectionIcon.setVisibility(View.VISIBLE);
            IMenuItem speedSelectItem = menuGroup.addItemAt(1, speedSelectView);

            voiceSelectView.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View view) {
                    mMoreAction.dismiss();
                    showLanguageListPopup();
                }
            });

            speedSelectView.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View view) {
                    mMoreAction.dismiss();
                    showSpeedPopup();
                }
            });


            mMoreAction = (UIActionMenu) UIActionMenu.newInstance((FragmentActivity) mUiExtensionsManager.getAttachedActivity());
            mMoreAction.setAutoResetSystemUiOnDismiss(false);
            mMoreAction.setAutoResetSystemUiOnShow(false);

            mMoreAction.setContentView(menuView.getContentView());

            if (AppDisplay.isPad()) {
                menuView.getContentView().setBackgroundResource(R.drawable.dlg_bg_4circle_corner_10dp_white);
                int width = Math.min(AppDisplay.getScreenWidth(), AppDisplay.getScreenHeight()) * 2 / 5;
                mMoreAction.setWidth(width);
            }
        }

        mVoiceSelectiontext.setText(mCurrentLocale.getDisplayLanguage());

        String rate;
        if ((int)(mTtsSpeechRate * 10) % 10 == 0) {
            rate = String.valueOf((int)mTtsSpeechRate);
        } else {
            rate = String.valueOf(mTtsSpeechRate);
        }
        mRateSelectionIconText.setText(rate + "x");
        mRateSelectionIcon.setImageResource(getSpeechRateImageRes(mTtsSpeechRate));
    }

    private void showMorePopup() {
        mUiExtensionsManager.stopHideToolbarsTimer();
        initMoreDialog();

        Rect rect = new Rect();
        mSpeedItem.getContentView().getGlobalVisibleRect(rect);
        int pos;
        if (rect.bottom < AppDisplay.getScreenHeight() / 2) {
            pos = UIPopoverWin.ARROW_TOP;
        } else {
            pos = UIPopoverWin.ARROW_BOTTOM;
        }

        mMoreAction.show(mUiExtensionsManager.getRootView(), rect, pos, 0);
    }

    private void initLanguageListDialog(final IResult<TtsVoiceGroup, Void, Void> onClickResult) {
        if (mLanguageListAction == null) {
            MenuViewImpl menuView = new MenuViewImpl(mUiExtensionsManager.getAttachedActivity());
            menuView.setTitleBarVisible(false);

            IMenuGroup menuGroup = menuView.addGroup("");
            if (AppDisplay.isPad()) {
                menuGroup.setHeaderTitleHeight(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_8dp));
                menuGroup.setHeaderTitleVisible(true);
            } else {
                menuGroup.setHeaderTitleVisible(false);
            }

            for (int i = 0; i < mLanguageList.size(); i ++) {
                final TtsVoiceGroup voiceGroup = mLanguageList.get(i);

                View languageItemView = View.inflate(mUiExtensionsManager.getAttachedActivity(), R.layout.tts_more_item, null);
                ((TextView) languageItemView.findViewById(R.id.tts_more_item_caption)).setText(voiceGroup.language);
                ((ImageView) languageItemView.findViewById(R.id.tts_more_item_icon)).setVisibility(View.GONE);
                ((TextView) languageItemView.findViewById(R.id.tts_more_item_selection_text)).setVisibility(View.GONE);

                IMenuItem voiceItem = menuGroup.addItem(languageItemView);

                languageItemView.setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View view) {
                        onClickResult.onResult(true, voiceGroup, null, null);
                        //showVoiceListPopup(voiceGroup);
                    }
                });
            }

            mLanguageListAction = (UIActionMenu) UIActionMenu.newInstance((FragmentActivity) mUiExtensionsManager.getAttachedActivity());
            mLanguageListAction.setAutoResetSystemUiOnDismiss(false);
            mLanguageListAction.setAutoResetSystemUiOnShow(false);

            mLanguageListAction.setContentView(menuView.getContentView());

            if (AppDisplay.isPad()) {
                menuView.getContentView().setBackgroundResource(R.drawable.dlg_bg_4circle_corner_10dp_white);
                int width = Math.min(AppDisplay.getScreenWidth(), AppDisplay.getScreenHeight()) * 2 / 5;
                mLanguageListAction.setWidth(width);
            }
        }
    }

    private void showLanguageListPopup() {
        mUiExtensionsManager.stopHideToolbarsTimer();

        if (AppDisplay.isPad()) {
            final UIMatchDialog dlg = new UIMatchDialog(mUiExtensionsManager.getAttachedActivity());

            initLanguageListDialog(new IResult<TtsVoiceGroup, Void, Void>() {
                @Override
                public void onResult(boolean success, final TtsVoiceGroup voiceGroup, Void p2, Void p3) {
                    dlg.dismiss();

                    dlg.setOnDLDismissListener(new MatchDialog.DismissListener() {
                        @Override
                        public void onDismiss() {
                            mUiExtensionsManager.startHideToolbarsTimer();
                            showVoiceListPopup(voiceGroup);
                            mLanguageListAction = null;
                        }
                    });
                }
            });

            dlg.setTitle(AppResource.getString(mContext, R.string.tts_voice_selection));
            dlg.setBackButtonStyle(MatchDialog.TEXT_BACK);
            dlg.setBackButtonText(AppResource.getString(mContext, R.string.fx_string_close));
            dlg.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
            dlg.setContentView(mLanguageListAction.getContentView());

            dlg.showDialog();

            dlg.setOnDLDismissListener(new MatchDialog.DismissListener() {
                @Override
                public void onDismiss() {
                    mUiExtensionsManager.startHideToolbarsTimer();
                    mLanguageListAction = null;
                }
            });

        } else {
            initLanguageListDialog(new IResult<TtsVoiceGroup, Void, Void>() {
                @Override
                public void onResult(boolean success, TtsVoiceGroup voiceGroup, Void p2, Void p3) {
                    mLanguageListAction.dismiss();
                    showVoiceListPopup(voiceGroup);
                }
            });

            Rect rect = new Rect();
            mSpeedItem.getContentView().getGlobalVisibleRect(rect);
            int pos;
            if (rect.bottom < AppDisplay.getScreenHeight() / 2) {
                pos = UIPopoverWin.ARROW_TOP;
            } else {
                pos = UIPopoverWin.ARROW_BOTTOM;
            }

            mLanguageListAction.show(mUiExtensionsManager.getRootView(), rect, pos, 0);

            mLanguageListAction.setOnDismissListener(new IActionMenu.OnActionMenuDismissListener() {
                @Override
                public void onDismiss(IActionMenu actionMenu) {
                    mUiExtensionsManager.startHideToolbarsTimer();
                    mLanguageListAction = null;
                }
            });
        }
    }

    void addVoiceGroup(TtsVoiceGroup group) {
        mLanguageList.add(group);
    }

    private View initVoiceListDialog(TtsVoiceGroup group) {
        //if (mVoiceListAction == null) {
        try {
            Collections.sort(group.voiceList, new TtsVoiceComparator());
        } catch (Exception e) {
            e.printStackTrace();
        }

        MenuViewImpl menuView = new MenuViewImpl(mUiExtensionsManager.getAttachedActivity());
        menuView.setTitleBarVisible(false);
        //menuView.setTitle(group.language);

        final IMenuGroup menuGroup = menuView.addGroup("");
        if (AppDisplay.isPad()) {
            menuGroup.setHeaderTitleVisible(true);
            menuGroup.setHeaderTitle(group.language);
            menuGroup.setHeaderTitleHeight(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_32dp));
        } else {
            menuGroup.setHeaderTitleVisible(false);
        }

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            for (int i = 0; i < group.voiceList.size(); i ++) {
                final Voice voice = group.voiceList.get(i);

                final View voiceItemView = View.inflate(mUiExtensionsManager.getAttachedActivity(), R.layout.tts_more_item, null);
                voiceItemView.findViewById(R.id.tts_more_item_icon).setVisibility(View.GONE);
                ((TextView) voiceItemView.findViewById(R.id.tts_more_item_caption)).setText(String.format("%s - %s", voice.getLocale().getDisplayName(), voice.getName()));
                voiceItemView.findViewById(R.id.tts_more_item_selection_text).setVisibility(View.GONE);

                if (voice.isNetworkConnectionRequired()) {
                    TextView selectionTV = voiceItemView.findViewById(R.id.tts_more_item_netwrok);
                    selectionTV.setVisibility(View.VISIBLE);
                    selectionTV.setText(AppResource.getString(mContext, R.string.tts_need_network));
                    selectionTV.setTextColor(AppResource.getColor(mContext, R.color.t2));
                    if (voice != mCurrentVoice)
                        selectionTV.setPadding(0, 0, (int)AppResource.getDimension(mContext, R.dimen.ux_margin_16dp), 0);
                } else {
                    ((TextView) voiceItemView.findViewById(R.id.tts_more_item_selection_text)).setVisibility(View.GONE);
                }

                ImageView selectedIcon = (ImageView) voiceItemView.findViewById(R.id.tts_more_item_arrow);
                selectedIcon.setSelected(true);
                ThemeUtil.setTintList(selectedIcon, ThemeUtil.getPrimaryIconColor(mContext));
                if (voice == mCurrentVoice) {
                    selectedIcon.setVisibility(View.VISIBLE);
                    selectedIcon.setImageResource(R.drawable.rd_select_icon);
                } else {
                    selectedIcon.setVisibility(View.GONE);
                }

                IMenuItem voiceItem = menuGroup.addItem(voiceItemView);

                voiceItemView.setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View view) {
                        final Runnable clickResult = new Runnable() {
                            @Override
                            public void run() {
                                //mVoiceListAction.dismiss();
                                List<IMenuItem> items = menuGroup.getItems();
                                for (IMenuItem item : items) {
                                    item.getContentView().findViewById(R.id.tts_more_item_arrow).setVisibility(View.GONE);
                                    item.getContentView().findViewById(R.id.tts_more_item_selection_text).setPadding(
                                            0, 0, (int)AppResource.getDimension(mContext, R.dimen.ux_margin_16dp), 0);
                                }

                                if (voice == mCurrentVoice) {
                                    mCurrentVoice = null;
                                    mCurrentLocale = getDefaultLocaleLanguage();
                                    mCurrentVoiceName = "";
                                } else {
                                    mCurrentVoice = voice;
                                    mCurrentLocale = voice.getLocale();
                                    mCurrentVoiceName = voice.getName();

                                    voiceItemView.findViewById(R.id.tts_more_item_selection_text).setPadding(0, 0, 0, 0);
                                    voiceItemView.findViewById(R.id.tts_more_item_arrow).setVisibility(View.VISIBLE);
                                    ((ImageView)voiceItemView.findViewById(R.id.tts_more_item_arrow)).setImageResource(R.drawable.rd_select_icon);
                                }

                                if (isSpeaking()) {
                                    onStartItemClicked();
                                    mTTS.setLanguage(mCurrentLocale);
                                    if (mCurrentVoice != null) {
                                        mTTS.setVoice(mCurrentVoice);
                                    }
                                    onStartItemClicked();
                                }
                            }
                        };

                        if (voice != mCurrentVoice && voice.isNetworkConnectionRequired()) {
                            final UITextEditDialog needNetDlg = new UITextEditDialog(mUiExtensionsManager.getAttachedActivity());
                            //needNetDlg.getDialog().setCanceledOnTouchOutside(false);
                            needNetDlg.setTitle(AppResource.getString(mContext, R.string.tts_prompt));
                            needNetDlg.getPromptTextView().setText(AppResource.getString(mContext, R.string.tts_network_prompt));
                            needNetDlg.getInputEditText().setVisibility(View.GONE);
                            needNetDlg.getCancelButton().setVisibility(View.GONE);

                            needNetDlg.show();

                            needNetDlg.getOKButton().setOnClickListener(new View.OnClickListener() {
                                @Override
                                public void onClick(View v) {
                                    needNetDlg.dismiss();
                                    clickResult.run();
                                }
                            });
                        } else {
                            clickResult.run();
                        }
                    }
                });
            }
        }

        mVoiceListAction = (UIActionMenu) UIActionMenu.newInstance((FragmentActivity) mUiExtensionsManager.getAttachedActivity());
        mVoiceListAction.setAutoResetSystemUiOnDismiss(false);
        mVoiceListAction.setAutoResetSystemUiOnShow(false);
        mVoiceListAction.setTitleBarView(group.language, new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mVoiceListAction.dismiss();
            }
        });
        mVoiceListAction.setBackItemColor(ThemeUtil.getPrimaryIconColor(mContext));

        mVoiceListAction.setContentView(menuView.getContentView());

        if (AppDisplay.isPad()) {
            menuView.getContentView().setBackgroundResource(R.drawable.dlg_bg_4circle_corner_10dp_white);
            int width = Math.min(AppDisplay.getScreenWidth(), AppDisplay.getScreenHeight()) * 2 / 5;
            mVoiceListAction.setWidth(width);
        }
        //}

        return menuView.getContentView();
    }

    private void showVoiceListPopup(TtsVoiceGroup group) {
        initVoiceListDialog(group);

        if (AppDisplay.isPad()) {
            UIMatchDialog dlg = new UIMatchDialog(mUiExtensionsManager.getAttachedActivity());

            dlg.setTitle(AppResource.getString(mContext, R.string.tts_voice_selection));
            //dlg.setBackButtonStyle(MatchDialog.TEXT_BACK);
            //dlg.setBackButtonText(AppResource.getString(mContext, R.string.fx_string_close));
            dlg.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
            dlg.setContentView(mVoiceListAction.getContentView());

            dlg.showDialog();

            dlg.setOnDLDismissListener(new MatchDialog.DismissListener() {
                @Override
                public void onDismiss() {
                    showLanguageListPopup();
                }
            });

        } else {
            mVoiceListAction.setOnDismissListener(new IActionMenu.OnActionMenuDismissListener() {
                @Override
                public void onDismiss(IActionMenu actionMenu) {
                    showLanguageListPopup();
                }
            });

            Rect rect = new Rect();
            mSpeedItem.getContentView().getGlobalVisibleRect(rect);
            int pos;
            if (rect.bottom < AppDisplay.getScreenHeight() / 2) {
                pos = UIPopoverWin.ARROW_TOP;
            } else {
                pos = UIPopoverWin.ARROW_BOTTOM;
            }

            mVoiceListAction.show(mUiExtensionsManager.getRootView(), rect, pos, 0);
        }
    }

    private void initSpeedRateDialog() {
        if (mSpeedSheet == null) {
            mSpeedSheet = UISheetMenu.newInstance((FragmentActivity) mUiExtensionsManager.getAttachedActivity());
            if (AppDisplay.isPad()) {
                mSpeedSheet.getContentView().setBackgroundResource(R.drawable.dlg_bg_4circle_corner_10dp_white);
                int width = Math.min(AppDisplay.getScreenWidth(), AppDisplay.getScreenHeight()) * 2 / 5;
                mSpeedSheet.setWidth(width);
            }
            mSpeedSheet.setMode(ISheetMenu.SELECTED_MODE);

            List<SheetItemBean> sheetItems = new ArrayList<>();
            for (IBaseItem centerItem : mSpeedDlgItems) {
                SheetItemBean sheetItem = new SheetItemBean();
                sheetItem.name = centerItem.getText();
                sheetItem.selected = centerItem.isSelected();
                sheetItem.type = centerItem.getTag();
                sheetItem.enabled = centerItem.isEnable();
                sheetItems.add(sheetItem);
            }
            mSpeedSheet.setCustomSheetItem(sheetItems);
        }
    }

    private void showSpeedPopup() {
        mUiExtensionsManager.stopHideToolbarsTimer();
        initSpeedRateDialog();

        mSpeedSheet.setSheetItemClickListener(new ISheetMenu.OnSheetItemClickListener() {
            @Override
            public void onItemClick(int type) {
                for (IBaseItem centerItem : mSpeedDlgItems) {
                    if (centerItem.getTag() == type) {
                        centerItem.performClick();
                        if (!isSupportVoiceList()) {
                            mSpeedItem.setImageResource(ConvertSpeechRateTypeToImageRes(type));
                        }
                        mTtsSpeechRate = ConvertSpeechRateTypeToValue(type);
                        mTTS.setSpeechRate(mTtsSpeechRate);
                        break;
                    }
                }
            }
        });

        if (AppDisplay.isPad()) {
            UIMatchDialog dlg = new UIMatchDialog(mUiExtensionsManager.getAttachedActivity());
            dlg.setTitle(AppResource.getString(mContext, R.string.tts_speech_rate));
            dlg.setBackButtonStyle(MatchDialog.TEXT_BACK);
            dlg.setBackButtonText(AppResource.getString(mContext, R.string.fx_string_close));
            //dlg.setStyle(MatchDialog.DLG_TITLE_STYLE_BG_DEFAULT);
            dlg.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
            dlg.setContentView(mSpeedSheet.getContentView());
            ((ViewGroup)mSpeedSheet.getContentView().getParent()).setPadding(0,
                    AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_8dp),
                    0, 0);

            dlg.showDialog();

            dlg.setOnDLDismissListener(new MatchDialog.DismissListener() {
                @Override
                public void onDismiss() {
                    mUiExtensionsManager.startHideToolbarsTimer();
                    mSpeedSheet = null;
                }
            });
        } else {
            mSpeedSheet.setTitle(AppResource.getString(mContext, R.string.tts_speech_rate));
            mSpeedSheet.setOnSheetDismissListener(new ISheetMenu.OnSheetDismissListener() {
                @Override
                public void onDismiss(ISheetMenu sheetMenu) {
                    mUiExtensionsManager.startHideToolbarsTimer();
                    mSpeedSheet = null;
                }
            });

            Rect rect = new Rect();
            mSpeedItem.getContentView().getGlobalVisibleRect(rect);
            int pos;
            if (rect.bottom < AppDisplay.getScreenHeight() / 2) {
                pos = UIPopoverWin.ARROW_TOP;
            } else {
                pos = UIPopoverWin.ARROW_BOTTOM;
            }

            mSpeedSheet.show(mUiExtensionsManager.getRootView(), rect, pos, 0);
        }
    }

    private void initMaskPromptDlg(final IResult<Void, Void, Void> result) {
        if (mMaskPromptDlg == null) {

            mMaskPromptDlg = new UIMatchDialog(mUiExtensionsManager.getAttachedActivity(), R.style.rd_dialog_fullscreen_style);

            mMaskPromptDlg.setTitleBarVisiable(false);
            mMaskPromptDlg.getWindow().setBackgroundDrawable(new ColorDrawable(AppResource.getColor(mContext, R.color.ux_color_mask_background)));
            mMaskPromptDlg.setBackgroundColor(AppResource.getColor(mContext, R.color.ux_color_translucent));

            View maskPromptView;
            if (AppDisplay.isPad()) {
                maskPromptView = View.inflate(mContext, R.layout.tts_mask_prompt_pad, null);
                if (AppUtil.isDarkMode(mContext)) {
                    ((ImageView)maskPromptView.findViewById(R.id.tts_toolbar_iv_src)).setImageResource(R.drawable.tts_prompt_toolbar_dark);
                    maskPromptView.findViewById(R.id.tts_voice_and_speech_ll).setBackgroundResource(R.drawable.tts_prompt_voice_speech_bg_dark);
                }
            } else {
                maskPromptView = View.inflate(mContext, R.layout.tts_mask_prompt, null);
            }

            mMaskPromptDlg.setContentView(maskPromptView);
        }

        mMaskPromptDlg.getRootView().findViewById(R.id.tts_language_voice_button).setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mMaskPromptDlg.dismiss();
            }
        });

        mMaskPromptDlg.setOnDLDismissListener(new MatchDialog.DismissListener() {
            @Override
            public void onDismiss() {
                result.onResult(true, null, null, null);
            }
        });
    }

    private boolean isMaskShowing(){
        return mMaskPromptDlg != null && mMaskPromptDlg.isShowing();
    }

    TtsVoiceGroup findVoiceGroup(String language) {
        for (int i = 0; i < mLanguageList.size(); i ++) {
            if (AppUtil.isEqual(language, mLanguageList.get(i).language)) {
                return mLanguageList.get(i);
            }
        }
        return null;
    }

    private void releaseTTS() {
        if (mTTS != null) {
            mTTS.shutdown();
        }
        mTTS = null;
        mIsSpeaking = false;
        mIsPause = false;
        mIsExit = true;
        mTtsRectInfoList.clear();
        mCurTtsRectInfo = null;
        mCurTtsStrList.clear();
        mCurTtsLocalLanguageList.clear();
        mCurTtsRectPageIndex = -1;
    }

    class TtsVoiceGroup {
        String language;
        ArrayList<Voice> voiceList = new ArrayList<>();

        void addVoice(Voice voice) {
            voiceList.add(voice);
        }
    }

    class TtsLanguageComparator implements Comparator<TtsVoiceGroup> {

        @Override
        public int compare(TtsVoiceGroup lhs, TtsVoiceGroup rhs) {
            try {
                Locale sysLocale = getDefaultLocaleLanguage();
                Comparator<Object> comparator = Collator.getInstance(sysLocale);

                return comparator.compare(lhs.language, rhs.language);
            }catch (Exception ignored){
            }
            return 0;
        }
    }

    class TtsVoiceComparator implements Comparator<Voice> {

        @Override
        public int compare(Voice lhs, Voice rhs) {
            try {
                Locale sysLocale = getDefaultLocaleLanguage();
                Comparator<Object> comparator = Collator.getInstance(sysLocale);

                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                    return comparator.compare(lhs.getName(), rhs.getName());
                }
                return 0;
            }catch (Exception ignored){
            }
            return 0;
        }
    }
}
