/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.thumbnail.createpage;


import android.content.Context;
import android.graphics.drawable.Drawable;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import android.widget.ImageView;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.annots.common.UIFillView;
import com.foxit.uiextensions.browser.adapter.SuperAdapter;
import com.foxit.uiextensions.browser.adapter.viewholder.SuperViewHolder;
import com.foxit.uiextensions.modules.panel.bean.BaseBean;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.NonNull;

public class PageColorAdapter extends SuperAdapter<PageColorAdapter.PageColorBean> {

    private List<PageColorBean> mDefaultColors;
    private List<PageColorBean> mItems;
    private OnPageColorItemClickListener mItemClickListener;

    private PageColorBean mCustomColor;
    private int mLastSelectedPosition = -1;
    private int mCurrentColor;

    public PageColorAdapter(Context context) {
        super(context);
        mItems = new ArrayList<>();
        mDefaultColors = new ArrayList<>();
    }

    @Override
    public void notifyUpdateData() {
        notifyDataSetChanged();
    }

    @Override
    public PageColorBean getDataItem(int position) {
        return mItems.get(position);
    }

    @Override
    public int getItemCount() {
        return mItems.size();
    }

    @NonNull
    @Override
    public SuperViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        return new PageColorAdapter.PageColorViewHolder(LayoutInflater.from(getContext()).inflate(R.layout.thumbnail_page_color_item, parent, false));
    }

    private class PageColorViewHolder extends SuperViewHolder {
        private final ImageView mCheckView;
        private final UIFillView mFillView;

        public PageColorViewHolder(View itemView) {
            super(itemView);
            mCheckView = itemView.findViewById(R.id.thumbnail_color_item_check_view);
            mFillView = itemView.findViewById(R.id.thumbnail_fill_color_item);
        }

        @Override
        public void bind(BaseBean data, int position) {
            PageColorBean colorBean = (PageColorBean) data;
            mFillView.setFillDrawale(colorBean.mFillDrawable);
            mFillView.setFillColorFilter(colorBean.mFillColor);
            if (colorBean.mBorderDrawable != null) {
                mFillView.setBorderDrawale(colorBean.mBorderDrawable);
                mFillView.setBorderWidth(colorBean.mBorderWidth);
            } else {
                mFillView.setBorderDrawale(null);
                mFillView.setBorderWidth(0);
            }
            FrameLayout.LayoutParams layoutParams = (FrameLayout.LayoutParams) mFillView.getLayoutParams();
            if (colorBean.mType == PageColorBean.DEFAULT_COLOR) {
                layoutParams.width = AppDisplay.dp2px(24);
                layoutParams.height = AppDisplay.dp2px(24);
            } else {
                layoutParams.width = AppDisplay.dp2px(30);
                layoutParams.height = AppDisplay.dp2px(30);
            }
            mFillView.setLayoutParams(layoutParams);

            mCheckView.setVisibility(colorBean.mSelected ? View.VISIBLE : View.INVISIBLE);
            mCheckView.setColorFilter(ThemeConfig.getInstance(getContext()).getPrimaryColor());
        }

        @Override
        public void onClick(View v) {
            int position = getAdapterPosition();
            if (mLastSelectedPosition == position) return;
            if (mLastSelectedPosition != -1) {
                PageColorBean lastColorBean = mItems.get(mLastSelectedPosition);
                lastColorBean.mSelected = false;
                notifyItemChanged(mLastSelectedPosition);
            }

            PageColorBean colorBean = mItems.get(position);
            if (colorBean.mType == PageColorBean.CUSTOM_COLOR && colorBean.mBorderDrawable == null) {
                setCustomColor(mCurrentColor);
            }
            colorBean.mSelected = true;
            notifyItemChanged(position);
            mLastSelectedPosition = position;

            if (mItemClickListener != null)
                mItemClickListener.onItemClick(colorBean.mType, colorBean.mFillColor);
        }
    }

    static class PageColorBean extends BaseBean {
        public static final int DEFAULT_COLOR = 0;
        public static final int CUSTOM_COLOR = 1;

        public int mType = DEFAULT_COLOR;
        public int mBorderWidth;
        public int mFillColor;
        public boolean mSelected;

        public Drawable mBorderDrawable;
        public Drawable mFillDrawable;
    }

    public void setColor(int color) {
        mCurrentColor = color;
        if (mLastSelectedPosition != -1) {
            PageColorBean lastPageColor = mItems.get(mLastSelectedPosition);
            lastPageColor.mSelected = false;
            notifyItemChanged(mLastSelectedPosition);
            mLastSelectedPosition = -1;
        }

        for (int i = 0; i < mDefaultColors.size(); i++) {
            PageColorBean colorBean = mDefaultColors.get(i);
            if (color == mDefaultColors.get(i).mFillColor) {
                colorBean.mSelected = true;
                mLastSelectedPosition = i;
                break;
            }
        }

        int customIndex = mItems.indexOf(mCustomColor);
        if (mLastSelectedPosition != -1 && mLastSelectedPosition != customIndex) {
            mCustomColor.mFillColor = 0x00000000;
            mCustomColor.mBorderDrawable = null;
            mCustomColor.mSelected = false;
            mCustomColor.mFillDrawable = AppResource.getDrawable(getContext(), R.drawable.thumbnail_custom_none_fill_color_bg);
        } else {
            mLastSelectedPosition = customIndex;
            setCustomColor(color);
        }
        notifyItemChanged(mLastSelectedPosition);
    }

    public void setCustomColor(int color) {
        if (mCustomColor.mBorderDrawable == null) {
            mCustomColor.mBorderWidth = AppResource.getDimensionPixelSize(getContext(), R.dimen.ux_margin_6dp);
            mCustomColor.mBorderDrawable = AppResource.getDrawable(getContext(), R.drawable.thumbnail_custom_border_color_bg);
        }
        mCustomColor.mFillColor = color;
        mCustomColor.mSelected = true;
        mCustomColor.mFillDrawable = AppResource.getDrawable(getContext(), R.drawable.thumbnail_custom_fill_color_bg);
    }

    public void initData() {
        PageColorBean color1 = new PageColorBean();
        color1.mBorderWidth = AppResource.getDimensionPixelSize(getContext(), R.dimen.ux_item_border_unselected_width);
        color1.mFillColor = 0xFFFFFFFF;
        color1.mBorderDrawable = AppResource.getDrawable(getContext(), R.drawable.thumbnail_default_border_color_bg);
        color1.mFillDrawable = AppResource.getDrawable(getContext(), R.drawable.thumbnail_default_fill_color_bg);

        PageColorBean color2 = new PageColorBean();
        color2.mFillColor = 0xFFFFFFBF;
        color2.mFillDrawable = AppResource.getDrawable(getContext(), R.drawable.thumbnail_default_fill_color_bg);

        PageColorBean color3 = new PageColorBean();
        color3.mFillColor = 0xFFDDDDFF;
        color3.mFillDrawable = AppResource.getDrawable(getContext(), R.drawable.thumbnail_default_fill_color_bg);

        PageColorBean color4 = new PageColorBean();
        color4.mFillColor = 0xFF669933;
        color4.mFillDrawable = AppResource.getDrawable(getContext(), R.drawable.thumbnail_default_fill_color_bg);

        PageColorBean color5 = new PageColorBean();
        color5.mFillColor = 0xFFD8D8D8;
        color5.mFillDrawable = AppResource.getDrawable(getContext(), R.drawable.thumbnail_default_fill_color_bg);

        PageColorBean color6 = new PageColorBean();
        color6.mFillColor = 0xFFB0B0B0;
        color6.mFillDrawable = AppResource.getDrawable(getContext(), R.drawable.thumbnail_default_fill_color_bg);

        PageColorBean color7 = new PageColorBean();
        color7.mFillColor = 0xFF000000;
        color7.mFillDrawable = AppResource.getDrawable(getContext(), R.drawable.thumbnail_default_fill_color_bg);

        mCustomColor = new PageColorBean();
        mCustomColor.mBorderWidth = AppResource.getDimensionPixelSize(getContext(), R.dimen.ux_margin_6dp);
        mCustomColor.mFillColor = 0xFFFFFFFF;
        mCustomColor.mType = PageColorBean.CUSTOM_COLOR;
        mCustomColor.mBorderDrawable = AppResource.getDrawable(getContext(), R.drawable.thumbnail_custom_border_color_bg);
        mCustomColor.mFillDrawable = AppResource.getDrawable(getContext(), R.drawable.thumbnail_custom_fill_color_bg);

        mItems.add(color1);
        mItems.add(color2);
        mItems.add(color3);
        mItems.add(color4);
        mItems.add(color5);
        mItems.add(color6);
        mItems.add(color7);
        mItems.add(mCustomColor);

        mDefaultColors.add(color1);
        mDefaultColors.add(color2);
        mDefaultColors.add(color3);
        mDefaultColors.add(color4);
        mDefaultColors.add(color5);
        mDefaultColors.add(color6);
        mDefaultColors.add(color7);
    }

    public void setItemClickListener(OnPageColorItemClickListener clickListener) {
        mItemClickListener = clickListener;
    }

    public interface OnPageColorItemClickListener {
        void onItemClick(int type, int color);
    }

}
