/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.thumbnail;

import android.app.Activity;
import android.content.Context;
import android.view.View;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.controls.dialog.AppDialogManager;
import com.foxit.uiextensions.controls.dialog.BaseDialogFragment;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.controls.toolbar.IBaseItem;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.impl.BaseItemImpl;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.pdfreader.config.ReadStateConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.UIToast;

import androidx.fragment.app.FragmentActivity;

/** All the page thumbnails are placed on a list view, and this is convenient for the user to add/edit/delete pages on this view.. */
public class ThumbnailModule implements Module {
    private final Context mContext;
    private final PDFViewCtrl mPdfView;
    private PDFViewCtrl.UIExtensionsManager mUiExtensionsManager;
    private ThumbnailSupport mSupport;
    private boolean mShareItemVisible = false;

    public ThumbnailModule(Context context, PDFViewCtrl pdfView, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        mContext = context;
        mPdfView = pdfView;
        mUiExtensionsManager = uiExtensionsManager;
    }

    public void show() {
        showThumbnailDialog();
    }

    @Override
    public String getName() {
        return Module.MODULE_NAME_THUMBNAIL;
    }

    @Override
    public boolean loadModule() {
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            initThumbnailSupportFragment();
            final UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            IBaseItem thumbnailItem = new BaseItemImpl(mContext, R.drawable.toolbar_thumbnail_icon);
            thumbnailItem.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    if (AppUtil.isFastDoubleClick())
                        return;
                    show();
                    uiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_Thumbnial);
                }
            });
            if(AppDisplay.isPad()) {
                thumbnailItem.setImageTintList(ThemeUtil.getTopBarIconColorStateList(mContext));
            }else{
                thumbnailItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
            }
            thumbnailItem.setId(R.id.id_bottom_bar_thumbnail);
            if (AppDisplay.isPad()) {
                thumbnailItem.setTag(ToolbarItemConfig.ITEM_TOPBAR_THUMBNAIL);
                uiExtensionsManager.getMainFrame().getTopToolbar().addView(thumbnailItem, BaseBar.TB_Position.Position_RB);
            } else {
                thumbnailItem.setTag(ToolbarItemConfig.ITEM_BOTTOMBAR_THUMBNAIL);
                uiExtensionsManager.getMainFrame().getBottomToolbar().addView(thumbnailItem, BaseBar.TB_Position.Position_CENTER);
            }
            uiExtensionsManager.registerModule(this);
            uiExtensionsManager.registerThemeEventListener(mThemeEventListener);
        }
        mPdfView.registerDocEventListener(mDocEventListener);
        return true;
    }

    @Override
    public boolean unloadModule() {
        ((UIExtensionsManager) mUiExtensionsManager).unregisterThemeEventListener(mThemeEventListener);
        mPdfView.unregisterDocEventListener(mDocEventListener);
        return true;
    }

    private void showThumbnailDialog() {
        if (mUiExtensionsManager == null) return;
        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
        Activity activity = uiExtensionsManager.getAttachedActivity();
        if (activity == null) return;

        if (!(activity instanceof FragmentActivity)) {
            UIToast.getInstance(mContext).show(mContext.getApplicationContext().getString(R.string.the_attached_activity_is_not_fragmentActivity));
            return;
        }

        if (uiExtensionsManager.getState() == ReadStateConfig.STATE_PANZOOM)
            uiExtensionsManager.changeState(ReadStateConfig.STATE_NORMAL);
        if (uiExtensionsManager.getCurrentToolHandler() != null)
            uiExtensionsManager.setCurrentToolHandler(null);
        if (uiExtensionsManager.getDocumentManager().getCurrentAnnot() != null)
            uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
        uiExtensionsManager.triggerDismissMenuEvent();

        FragmentActivity act = (FragmentActivity) activity;
        ThumbnailSupport support = (ThumbnailSupport) act.getSupportFragmentManager().findFragmentByTag("ThumbnailSupport");
        if (support == null) {
            initThumbnailSupportFragment();
            support = mSupport;
        }
        support.setShareItemVisible(mShareItemVisible);
        if (!support.isShowing()) {
            AppDialogManager.getInstance().showAllowManager(support, act.getSupportFragmentManager(), "ThumbnailSupport", null);
        }
    }

    public ThumbnailSupport getThumbnailSupport() {
        initThumbnailSupportFragment();
        return mSupport;
    }


    private IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            if (mSupport != null){
                mSupport.onThemeColorChanged(type, color);
            }
            if (mSupport != null && mSupport.isShowing()) {
                mSupport.updateTheme();
                AppDialogManager.getInstance().dismiss(mSupport);
            }
        }
    };

    private void initThumbnailSupportFragment() {
        if (mSupport != null) return;
        mSupport = new ThumbnailSupport();
        mSupport.init(mPdfView, mContext);
        mSupport.setDismissListener(new BaseDialogFragment.DismissListener() {
            @Override
            public void onDismiss() {
                mSupport = null;
            }
        });
    }

    public void setShareItemVisible(boolean visible) {
        if (mSupport == null) return;
        mShareItemVisible = visible;
        mSupport.setShareItemVisible(visible);
    }

    private final PDFViewCtrl.IDocEventListener mDocEventListener = new DocEventListener() {

        @Override
        public void onDocOpened(PDFDoc document, int errCode) {
            if (mSupport != null && mSupport.isShowing()){
                AppDialogManager.getInstance().dismiss(mSupport);
            }
        }

    };
}



