/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.thumbnail;

import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.res.ColorStateList;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.Point;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffColorFilter;
import android.graphics.RectF;
import android.net.Uri;
import android.os.Build;
import android.text.Editable;
import android.text.InputType;
import android.text.TextWatcher;
import android.util.SparseArray;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.inputmethod.InputMethodManager;
import android.widget.CheckBox;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.Task;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.common.Progressive;
import com.foxit.sdk.common.Range;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.multimedia.screen.multimedia.MultimediaModule;
import com.foxit.uiextensions.annots.multimedia.screen.multimedia.MultimediaToolHandler;
import com.foxit.uiextensions.controls.dialog.FxProgressDialog;
import com.foxit.uiextensions.controls.dialog.MatchDialog;
import com.foxit.uiextensions.controls.dialog.UIDialog;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.controls.dialog.fileselect.UIFileSelectDialog;
import com.foxit.uiextensions.controls.dialog.fileselect.UIFolderSelectDialog;
import com.foxit.uiextensions.home.local.LocalModule;
import com.foxit.uiextensions.modules.doc.docinfo.DocInfoModule;
import com.foxit.uiextensions.modules.pagenavigation.LogicalPageHelper;
import com.foxit.uiextensions.modules.thumbnail.createpage.CreatePageBean;
import com.foxit.uiextensions.pdfreader.config.ActRequestCode;
import com.foxit.uiextensions.pdfreader.impl.MainFrame;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppIntentUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppStorageManager;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.UIToast;
import com.google.android.material.card.MaterialCardView;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.core.content.FileProvider;
import androidx.recyclerview.widget.RecyclerView;

public class ThumbnailAdapter extends RecyclerView.Adapter<ThumbnailAdapter.ThumbViewHolder> implements ThumbnailItemTouchCallback.ItemTouchAdapter {
    private static final int REMOVE_PAGE = 0;
    private static final int ROTATE_PAGE_CW = 1;
    private static final int ROTATE_PAGE_ACW = 2;
    private static final int INSERT_FROM_DOCUMENT = 3;
    private static final int EXPORT_PAGE = 4;
    private static final int COPY_PAGES = 5;
    private static final int INSERT_PAGES = 6;

    List<ThumbnailItem> mThumbnailList;
    final SparseArray<ThumbnailItem> mSelectedArray;
    final ArrayList<ThumbnailItem> mCacheList;
    final private ArrayList<DrawThumbnailTask> mTaskList;

    private Context mContext;
    private final PDFViewCtrl mPDFViewCtrl;
    private final UIExtensionsManager mUIExtensionsManager;

    private PDFDoc mTempPasteDoc;
    private ThumbnailItem mPasteItem;

    private boolean mIsEditing = false;
    int mMoveToPosition = -1;

    private IThumbnailSupport mSupport;
    private int mTasksMax;
    private int mBitmapsMax;
    int mCurrentPage;
    int mCurPagePosition;

    private int mTabPosition;

    public ThumbnailAdapter(int tabPosition, IThumbnailSupport support, List<ThumbnailItem> items) {
        mSupport = support;
        mTabPosition = tabPosition;

        if (mSupport != null)
            mContext = mSupport.getContext() == null ? AppUtil.getApplicationContext() : mSupport.getContext();
        else
            mContext = AppUtil.getApplicationContext();
        mPDFViewCtrl = support.getPDFView();
        mUIExtensionsManager = (UIExtensionsManager) mPDFViewCtrl.getUIExtensionsManager();
        mCurrentPage = mPDFViewCtrl.getCurrentPage();

        mThumbnailList = items;
        mSelectedArray = new SparseArray<>();
        mCacheList = new ArrayList<>();
        mTaskList = new ArrayList<>();
        mCacheFolder_share = AppFileUtil.getAppCacheDir(mContext) + "/thumbnail_share/";
    }

    int getInsertIndex() {
        int insertPositon = mCurPagePosition >= 0
                ? Math.min(mCurPagePosition + 1, getItemCount()) : getItemCount();
        int insertIndex = mThumbnailList.get(insertPositon - 1).getPageIndex() + 1;
        return insertIndex;
    }

    public void setCacheSize(int tasksMax, int bitmapsMax) {
        mTasksMax = tasksMax;
        mBitmapsMax = bitmapsMax;
    }

    public boolean isSelectedAll() {
        return mSelectedArray.size() > 0 && mThumbnailList.size() == mSelectedArray.size();
    }

    public void selectAll(boolean isSelect) {
        mSelectedArray.clear();
        for (int i = 0; i < mThumbnailList.size(); i++) {
            ThumbnailItem thumbnailItem = mThumbnailList.get(i);
            updateSelectListInfo(i, thumbnailItem, isSelect);
        }
        notifyDataSetChanged();
    }

    private void addTask(DrawThumbnailTask task) {
        synchronized (mTaskList) {
            if (mTaskList.size() >= mTasksMax) {
                DrawThumbnailTask oldTask = null;
                int position = task.getPosition();
                for (DrawThumbnailTask thumbnailTask : mTaskList) {
                    if (!mSupport.isThumbnailItemVisible(mTabPosition, thumbnailTask.getThumbnailItem())) {
                        if (oldTask == null) {
                            oldTask = thumbnailTask;
                        } else {
                            if ((Math.abs(oldTask.getPosition() - position)) < Math.abs(thumbnailTask.getPosition() - position))
                                oldTask = thumbnailTask;
                        }
                        break;
                    }
                }
                if (oldTask == null) {
                    oldTask = mTaskList.get(0);
                }

                mPDFViewCtrl.removeTask(oldTask);
                mTaskList.remove(oldTask);
                oldTask.getThumbnailItem().resetRending(false);
            }
            mTaskList.add(task);
            mPDFViewCtrl.addTask(task);
        }
    }

    private void removeTask(DrawThumbnailTask task) {
        synchronized (mTaskList) {
            mTaskList.remove(task);
        }
    }

    public void updateCacheListInfo(ThumbnailItem value, boolean add) {
        if (add) {
            if (mCacheList.contains(value))
                return;
            if (mCacheList.size() >= mBitmapsMax) {
                ThumbnailItem item = mCacheList.get(0);
                item.setBitmap(null);
                mCacheList.remove(0);
            }
            mCacheList.add(value);
        } else {
            if (mCacheList.contains(value)) {
                mCacheList.remove(value);
                value.setBitmap(null);
            }
        }
    }

    public int getSelectedItemCount() {
        return mSelectedArray.size();
    }

    public int[] getSelectedItemIndexes() {
        int size = mSelectedArray.size();
        if (size == 0) return new int[]{};
        int[] indexes = new int[size];
        for (int i = 0; i < size; i++) {
            indexes[i] = mSelectedArray.get(mSelectedArray.keyAt(i)).getPageIndex();
        }
        return indexes;
    }

    public void updateSelectListInfo(int position, ThumbnailItem item, boolean select) {
        if (select) {
            if (mSelectedArray.indexOfValue(item) < 0)
                mSelectedArray.append(position, item);
        } else {
            mSelectedArray.remove(position);
        }
        item.setSelected(select);
    }

    @NonNull
    @Override
    public ThumbViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        View itemView = LayoutInflater.from(parent.getContext()).inflate(R.layout.thumbnail_view, parent, false);
        return new ThumbViewHolder(itemView);
    }

    @Override
    public void onBindViewHolder(final ThumbViewHolder holder, int position) {
        ThumbnailItem thumbnailItem = mThumbnailList.get(position);
        holder.drawThumbnail(thumbnailItem, position);
    }

    @Override
    public int getItemCount() {
        return mThumbnailList.size();
    }

    public ThumbnailItem getThumbnailItem(int position) {
        if (position < 0 || position > getItemCount())
            return null;
        return mThumbnailList.get(position);
    }

    @Override
    public void onMove(int fromPosition, int toPosition) {
        if (mUIExtensionsManager.getDocumentManager().isXFA()) {
            UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.xfa_not_support_move_toast));
            return;
        }
        swapData(fromPosition, toPosition);

        mMoveToPosition = toPosition;
        notifyItemMoved(fromPosition, toPosition);
    }

    void swapData(int fromPosition, int toPosition) {
        if (fromPosition < toPosition) {
            for (int i = fromPosition; i < toPosition; i++) {
                swap(i, i + 1);
            }
        } else {
            for (int i = fromPosition; i > toPosition; i--) {
                swap(i, i - 1);
            }
        }
    }

    private void swap(int dst, int src) {
        Collections.swap(mThumbnailList, dst, src);
    }

    public void release() {
        if (mSelectedArray != null) {
            int selectedSize = mSelectedArray.size();
            for (int i = 0; i < selectedSize; i++) {
                mSelectedArray.valueAt(i).setSelected(false);
            }
            mSelectedArray.clear();
        }
        if (mCacheList != null) {
            for (ThumbnailItem item : mCacheList) {
                item.setBitmap(null);
            }
            mCacheList.clear();
        }

        if (mThumbnailList != null) {
            mThumbnailList.clear();
        }

        for (int i = 0; i < mTaskList.size(); i++) {
            DrawThumbnailTask task = mTaskList.get(i);
            mPDFViewCtrl.removeTask(task);
        }
        mTaskList.clear();
        mSupport = null;
    }

    public void removeSelectedPages() {
        if (mSelectedArray.size() == 0)
            return;
        mSupport.getProgressDialog().setTips(AppResource.getString(mContext, R.string.rv_page_delete));
        mSupport.getProgressDialog().show();

        final SparseArray<ThumbnailItem> tmpSelectList = mSelectedArray.clone();
        doRemovePages(tmpSelectList, new EditThumbnailCallback() {
            @Override
            public void onProgressChanged(int type, int progress) {
            }

            @Override
            public void result(boolean success, PDFDoc srcDoc, PDFDoc dstDoc) {
                if (mPDFViewCtrl.getPageCount() == 1)
                    ((MainFrame) mUIExtensionsManager.getMainFrame()).resetPageLayout();
                tmpSelectList.clear();
                mSupport.getProgressDialog().dismiss();
                notifyDataSetChanged();
            }
        });
    }

    //remove items form all item list and cache list but not selected item list,remove pages from doc.
    private void doRemovePages(final SparseArray<ThumbnailItem> itemList, final EditThumbnailCallback callback) {
        int removeSize = itemList.size();
        for (int i = 0; i < removeSize; i++) {
            if (itemList.valueAt(i).isRendering()) {
                if (callback != null) {
                    callback.result(true, null, null);
                }
                return;
            }
        }
        EditThumbnailTask editThumbnailTask = new EditThumbnailTask(REMOVE_PAGE, itemList, callback);
        mPDFViewCtrl.addTask(editThumbnailTask);
    }

    private void doRotatePages(final SparseArray<ThumbnailItem> itemList, final boolean isClockWise, final EditThumbnailCallback callback) {
        int rotateSize = itemList.size();
        for (int i = 0; i < rotateSize; i++) {
            if (itemList.valueAt(i).isRendering()) {
                if (callback != null) {
                    callback.result(true, null, null);
                }
                return;
            }
        }
        EditThumbnailTask editThumbnailTask = new EditThumbnailTask(isClockWise ? ROTATE_PAGE_CW : ROTATE_PAGE_ACW, itemList, callback);
        mPDFViewCtrl.addTask(editThumbnailTask);
    }

    public void copyPages(final Event.Callback callback) {
        if (mTempPasteDoc != null) {
            mTempPasteDoc.delete();
            mTempPasteDoc = null;
        }

        final FxProgressDialog progressDialog = mSupport.getProgressDialog();
        progressDialog.setTips(AppResource.getString(mContext, R.string.fx_string_copying));
        progressDialog.show();
        EditThumbnailTask editThumbnailTask = new EditThumbnailTask(mSelectedArray, null, new EditThumbnailCallback() {
            @Override
            public void onProgressChanged(int type, int progress) {
            }

            @Override
            public void result(boolean success, PDFDoc srcDoc, PDFDoc dstDoc) {
                if (success)
                    mTempPasteDoc = dstDoc;
                if (callback != null)
                    callback.result(null, success);
                progressDialog.dismiss();
            }
        });
        mPDFViewCtrl.addTask(editThumbnailTask);
    }

    private void pastePages(final PDFDoc pdfdoc) {
        if (pdfdoc == null) return;
        int pasteIndex = mThumbnailList.indexOf(mPasteItem);
        int dstIndex = 0;
        if (pasteIndex == 0) {
            if (mThumbnailList.size() == 1) {
                dstIndex = mPDFViewCtrl.getPageCount();
            } else {
                dstIndex = mThumbnailList.get(pasteIndex + 1).getPageIndex();
                if (dstIndex > 0)
                    dstIndex -= 1;
            }
        } else
            dstIndex = mThumbnailList.get(pasteIndex - 1).getPageIndex() + 1;

        if (pasteIndex >= 0) {
            mThumbnailList.remove(pasteIndex);
            mPasteItem = null;
        }
        try {
            final FxProgressDialog progressDialog = mSupport.getProgressDialog();
            progressDialog.setTips(AppResource.getString(mContext, R.string.fx_string_copying));
            progressDialog.show();
            int[] ranges = {0, pdfdoc.getPageCount()};
            final int lastPageCount = mPDFViewCtrl.getPageCount();

            doImportPages(dstIndex, ranges, pdfdoc, new EditThumbnailCallback() {
                @Override
                public void onProgressChanged(int type, int progress) {
                }

                @Override
                public void result(boolean success, PDFDoc srcDoc, PDFDoc dstDoc) {
                    if (success && lastPageCount == 1)
                        ((MainFrame) mUIExtensionsManager.getMainFrame()).resetPageLayout();
                    notifyDataSetChanged();
                    progressDialog.dismiss();
                    mSupport.switchPasteState(false);
                    mSupport.setSelectViewMode(isSelectedAll());
                }
            });

        } catch (Exception ignored) {
        }
    }

    void reset() {
        removePasteItem();
        if (mTempPasteDoc != null) {
            mTempPasteDoc.delete();
            mTempPasteDoc = null;
        }
        if (mSelectedArray != null) {
            int selectedSize = mSelectedArray.size();
            for (int i = 0; i < selectedSize; i++) {
                mSelectedArray.valueAt(i).setSelected(false);
            }
            mSelectedArray.clear();
        }
    }

    int getPasteItemIndex() {
        if (mPasteItem == null) return -1;
        return mThumbnailList.indexOf(mPasteItem);
    }

    void insertPasteItem(int insertPosition) {
        mPasteItem = new ThumbnailItem(mSupport.getThumbnailBackgroundSize(), mPDFViewCtrl);
        mThumbnailList.add(insertPosition, mPasteItem);
        notifyItemInserted(insertPosition);
    }

    void removePasteItem() {
        if (mPasteItem == null) return;
        int pasteItemIndex = getPasteItemIndex();
        if (pasteItemIndex >= 0) {
            mThumbnailList.remove(pasteItemIndex);
            notifyItemRemoved(pasteItemIndex);
            mPasteItem = null;
        }
    }

    void importDocument(final int dstIndex, final String filepath, String password, final Event.Callback callback) {
        try {
            final PDFDoc doc = new PDFDoc(filepath);
            int errorCode;
            if (password == null) {
                errorCode = doc.load(null);
            } else {
                errorCode = doc.load(password.getBytes());
                if (doc.getPasswordType() != PDFDoc.e_PwdOwner) {
                    errorCode = doc.checkPassword(password.getBytes()) == PDFDoc.e_PwdOwner ? doc.load(password.getBytes()) : Constants.e_ErrPassword;
                }
            }

            if (errorCode == Constants.e_ErrSuccess) {
                if (doc.isWrapper() && doc.getWrapperType() != PDFDoc.e_WrapperNone) {
                    if (callback != null)
                        callback.result(null, false);
                    UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.rv_tips_import_rms_files));
                    return;
                }

                final FxProgressDialog progressDialog = mSupport.getProgressDialog();
                progressDialog.setTips(AppResource.getString(mContext, R.string.rv_page_import));
                progressDialog.show();

                int[] ranges = {0, doc.getPageCount()};
                final int lastPageCount = mPDFViewCtrl.getPageCount();

                doImportPages(dstIndex, ranges, doc, new EditThumbnailCallback() {
                    @Override
                    public void onProgressChanged(int type, int progress) {
                    }

                    @Override
                    public void result(boolean success, PDFDoc srcDoc, PDFDoc dstDoc) {
                        if (success && lastPageCount == 1)
                            ((MainFrame) mUIExtensionsManager.getMainFrame()).resetPageLayout();
                        doc.delete();
                        progressDialog.dismiss();
                        notifyDataSetChanged();
                        if (callback != null)
                            callback.result(null, success);
                    }
                });
            } else {
                if (errorCode == Constants.e_ErrPassword) {
                    String tips;
                    if (password != null && password.trim().length() > 0) {
                        tips = AppResource.getString(mContext, R.string.rv_tips_password_error);
                    } else {
                        tips = AppResource.getString(mContext, R.string.rv_import_protected_files);
                    }
                    final UITextEditDialog uiTextEditDialog = new UITextEditDialog(mSupport.getActivity());
                    uiTextEditDialog.getDialog().setCanceledOnTouchOutside(false);
                    uiTextEditDialog.getInputEditText().setInputType(InputType.TYPE_CLASS_TEXT | InputType.TYPE_TEXT_VARIATION_PASSWORD);
                    uiTextEditDialog.setTitle(AppResource.getString(mContext, R.string.rv_password_dialog_title));
                    uiTextEditDialog.getPromptTextView().setText(tips);
                    uiTextEditDialog.show();
                    AppUtil.showSoftInput(uiTextEditDialog.getInputEditText());
                    uiTextEditDialog.getOKButton().setOnClickListener(new View.OnClickListener() {
                        @Override
                        public void onClick(View v) {
                            uiTextEditDialog.dismiss();
                            InputMethodManager inputManager = (InputMethodManager) v.getContext().getSystemService(Context.INPUT_METHOD_SERVICE);
                            inputManager.hideSoftInputFromWindow(v.getWindowToken(), 0);
                            String pw = uiTextEditDialog.getInputEditText().getText().toString();
                            importDocument(dstIndex, filepath, pw, callback);
                        }
                    });

                    uiTextEditDialog.getCancelButton().setOnClickListener(new View.OnClickListener() {
                        @Override
                        public void onClick(View v) {
                            uiTextEditDialog.dismiss();
                            if (callback != null)
                                callback.result(null, false);
                            InputMethodManager inputManager = (InputMethodManager) v.getContext().getSystemService(Context.INPUT_METHOD_SERVICE);
                            inputManager.hideSoftInputFromWindow(v.getWindowToken(), 0);
                        }
                    });

                    uiTextEditDialog.getDialog().setOnKeyListener(new DialogInterface.OnKeyListener() {
                        @Override
                        public boolean onKey(DialogInterface dialog, int keyCode, KeyEvent event) {
                            if (keyCode == KeyEvent.KEYCODE_BACK) {
                                uiTextEditDialog.getDialog().cancel();
                                if (callback != null)
                                    callback.result(null, false);
                                return true;
                            }
                            return false;
                        }
                    });
                    uiTextEditDialog.show();
                } else {
                    if (callback != null)
                        callback.result(null, false);
                    UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.rv_page_import_error));
                }
            }
        } catch (PDFException e) {
            if (callback != null)
                callback.result(null, false);
            UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.rv_page_import_error));
        }
    }

    //import pages form sd-card special file
    public void importPagesFromSpecialFile(final int dstIndex) {
        importPagesFromSpecialFile(dstIndex, null, null);
    }

    public void importPagesFromSpecialFile(final int dstIndex, final DialogInterface.OnShowListener showListener, final Event.Callback callback) {
        final UIFileSelectDialog dialog = mSupport.getFileSelectDialog();
        dialog.init(null, true);
        dialog.setListener(new MatchDialog.DialogListener() {
            @Override
            public void onResult(long btType) {
            }

            @Override
            public void onBackClick() {
            }

            @Override
            public void onTitleRightButtonClick() {
                String filepath = dialog.getSelectedFiles().get(0).path;
                if (!AppUtil.isEmpty(filepath)) {
                    AppFileUtil.updateIsExternalStorageManager();
                    filepath = AppFileUtil.getAdaptedFilePath(mContext, filepath);
                    importDocument(dstIndex, filepath, null, callback);
                }
                dialog.dismiss();
            }
        });

        if (showListener != null) {
            dialog.setOnShowListener(new DialogInterface.OnShowListener() {
                @Override
                public void onShow(DialogInterface dialog) {
                    showListener.onShow(dialog);
                }
            });
        }
        dialog.resetWH();
        dialog.showDialog();
    }

    private void doImportPages(int dstIndex, int[] pageRanges, final PDFDoc srcDoc, EditThumbnailCallback callback) {
        try {
            if (srcDoc != null && srcDoc.isXFA()) {
                UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.rv_xfapage_import_error));
                if (callback != null)
                    callback.result(false, srcDoc, mPDFViewCtrl.getDoc());
                return;
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }

        EditThumbnailTask editThumbnailTask = new EditThumbnailTask(dstIndex, pageRanges, srcDoc, callback);
        mPDFViewCtrl.addTask(editThumbnailTask);
    }

    private void doInsertPages(int dstIndex, int[] pageRanges, final CreatePageBean pageBean, EditThumbnailCallback callback) {
        EditThumbnailTask editThumbnailTask = new EditThumbnailTask(dstIndex, pageRanges, pageBean, callback);
        mPDFViewCtrl.addTask(editThumbnailTask);
    }

    private void doCopyPages(int[] pageRanges, final PDFDoc srcDoc, EditThumbnailCallback callback) {
        EditThumbnailTask editThumbnailTask = new EditThumbnailTask(pageRanges, srcDoc, callback);
        mPDFViewCtrl.addTask(editThumbnailTask);
    }

    private void doExtractPages(SparseArray<ThumbnailItem> itemList, String path, final Event.Callback callback) {
        final FxProgressDialog progressDialog = mSupport.getProgressDialog();
        progressDialog.setTips(AppResource.getString(mContext, R.string.rv_page_extract));
        progressDialog.show();

        AppFileUtil.updateIsExternalStorageManager();
        final String targetPath = AppFileUtil.getAdaptedFilePath(mContext, path);
        EditThumbnailTask editThumbnailTask = new EditThumbnailTask(itemList, targetPath, new EditThumbnailCallback() {
            @Override
            public void onProgressChanged(int type, int progress) {
            }

            @Override
            public void result(boolean success, PDFDoc srcDoc, PDFDoc dstDoc) {
                progressDialog.dismiss();
                if (!success)
                    UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.rv_page_extract_error));
                else
                    UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.thumbnail_extract_success));

                if (callback != null)
                    callback.result(null, success);
                LocalModule module = (LocalModule) mUIExtensionsManager.getModuleByName(Module.MODULE_NAME_LOCAL);
                if (module != null && targetPath != null)
                    module.updateThumbnail(targetPath);
                notifyDataSetChanged();
            }
        });
        mPDFViewCtrl.addTask(editThumbnailTask);
    }

    private void doExtractPagesSilence(SparseArray<ThumbnailItem> itemList, String path, final Event.Callback callback) {
        final FxProgressDialog progressDialog = mSupport.getProgressDialog();
        progressDialog.show();

        AppFileUtil.updateIsExternalStorageManager();
        final String targetPath = AppFileUtil.getAdaptedFilePath(mContext, path);
        com.foxit.uiextensions.modules.thumbnail.ThumbnailAdapter.EditThumbnailTask editThumbnailTask = new com.foxit.uiextensions.modules.thumbnail.ThumbnailAdapter.EditThumbnailTask(itemList, targetPath, new com.foxit.uiextensions.modules.thumbnail.ThumbnailAdapter.EditThumbnailCallback() {
            @Override
            public void onProgressChanged(int type, int progress) {
            }

            @Override
            public void result(boolean success, PDFDoc srcDoc, PDFDoc dstDoc) {
                progressDialog.dismiss();
                if (callback != null)
                    callback.result(null, success);
                notifyDataSetChanged();
            }
        });
        mPDFViewCtrl.addTask(editThumbnailTask);
    }

    private void showInputFileNameDialog(final String fileFolder, final Event.Callback callback) {
        final String filePath = AppFileUtil.getFileDuplicateName("");
        final String fileName = AppFileUtil.getFileNameWithoutExt(filePath);

        final UITextEditDialog rmDialog = new UITextEditDialog(mSupport.getActivity());
        rmDialog.setPattern("[/\\:*?<>|\"\n\t]");
        rmDialog.setTitle(AppResource.getString(mContext, R.string.fx_rename_file));
        rmDialog.setLengthFilters(UITextEditDialog.MAX_FILE_NAME_LENGTH);
        rmDialog.getPromptTextView().setVisibility(View.GONE);
        rmDialog.getInputEditText().setText(fileName);
        rmDialog.getInputEditText().selectAll();
        rmDialog.getInputEditText().setMaxLines(6);
        rmDialog.show();
        AppUtil.showSoftInput(rmDialog.getInputEditText());

        final View okButton = rmDialog.getOKButton();
        okButton.setEnabled(false);
        final EditText editText = rmDialog.getInputEditText();
        editText.addTextChangedListener(new TextWatcher() {
            @Override
            public void beforeTextChanged(CharSequence charSequence, int i, int i1, int i2) {
            }

            @Override
            public void onTextChanged(CharSequence charSequence, int i, int i1, int i2) {
            }

            @Override
            public void afterTextChanged(Editable editable) {
                if (editText.getText().toString().trim().equals("")) {
                    okButton.setEnabled(false);
                } else
                    okButton.setEnabled(true);
            }
        });

        okButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                rmDialog.dismiss();
                String inputName = rmDialog.getInputEditText().getText().toString();
                String newPath = fileFolder + "/" + inputName;
                newPath += ".pdf";
                File file = new File(newPath);
                if (file.exists()) {
                    Module module = mUIExtensionsManager.getModuleByName(Module.MODULE_NAME_DOCINFO);
                    if (module == null) {
                        UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.docinfo_module_not_regisetered_toast));
                        return;
                    }
                    String currentPath = ((DocInfoModule) module).getFilePath();
                    if (!newPath.contentEquals(currentPath))
                        showAskReplaceDialog(fileFolder, newPath, callback);
                    else {
                        UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.cinflicted_with_current_doc_toast));
                        showInputFileNameDialog(fileFolder, callback);
                    }

                } else {
                    doExtractPages(mSelectedArray, newPath, callback);
                }
            }
        });

        rmDialog.getCancelButton().setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                rmDialog.dismiss();
                if (callback != null)
                    callback.result(null, false);
            }
        });

        rmDialog.getDialog().setOnKeyListener(new DialogInterface.OnKeyListener() {
            @Override
            public boolean onKey(DialogInterface dialog, int keyCode, KeyEvent event) {
                if (keyCode == KeyEvent.KEYCODE_BACK) {
                    rmDialog.getDialog().cancel();
                    if (callback != null)
                        callback.result(null, false);
                    return true;
                }
                return false;
            }
        });
    }

    private void showAskReplaceDialog(final String fileFolder, final String newPath, final Event.Callback callback) {
        final UITextEditDialog rmDialog = new UITextEditDialog(mSupport.getActivity(), UIDialog.NO_INPUT);
        rmDialog.setTitle(AppResource.getString(mContext, R.string.fx_string_warning_title));
        rmDialog.getPromptTextView().setText(AppResource.getString(mContext, R.string.fx_string_filereplace_warning));
        rmDialog.show();

        rmDialog.getOKButton().setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                rmDialog.dismiss();
                File file = new File(newPath);
                if (file.delete()) {
                    doExtractPages(mSelectedArray, newPath, callback);
                } else {
                    UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.the_file_can_not_replace_toast, file.getPath()));
                }
            }
        });

        rmDialog.getCancelButton().setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                rmDialog.dismiss();
                showInputFileNameDialog(fileFolder, callback);
            }
        });
    }

    void rotateSelectedPages(boolean isClockWise) {
        if (mSelectedArray.size() == 0) return;
        mSupport.getProgressDialog().setTips(AppResource.getString(mContext, R.string.fx_string_processing));
        mSupport.getProgressDialog().show();
        doRotatePages(mSelectedArray, isClockWise, new EditThumbnailCallback() {
            @Override
            public void onProgressChanged(int type, int progress) {
            }

            @Override
            public void result(boolean success, PDFDoc srcDoc, PDFDoc dstDoc) {
                mSupport.getProgressDialog().dismiss();
                notifyDataSetChanged();
            }
        });
    }

    boolean movePage(int fromPosition, int toPosition) {
        if (mPDFViewCtrl.movePage(fromPosition, toPosition)) {
            mUIExtensionsManager.getDocumentManager().setDocModified(true);
            return true;
        }
        return false;
    }

    public void insertPage(int index, CreatePageBean pageBean) {
        insertPage(index, pageBean, null);
    }

    public void insertPage(int index, CreatePageBean pageBean, final Event.Callback callback) {
        final FxProgressDialog progressDialog = mSupport.getProgressDialog();
        progressDialog.setTips(AppResource.getString(mContext, R.string.fx_string_processing));
        int[] ranges = {index, pageBean.getPageCounts()};
        final int lastPageCount = mPDFViewCtrl.getPageCount();

        doInsertPages(index, ranges, pageBean, new EditThumbnailCallback() {
            @Override
            public void onProgressChanged(int type, int progress) {
            }

            @Override
            public void result(boolean success, PDFDoc srcDoc, PDFDoc dstDoc) {
                if (success && lastPageCount == 1)
                    ((MainFrame) mUIExtensionsManager.getMainFrame()).resetPageLayout();
                progressDialog.dismiss();
                notifyDataSetChanged();

                if (callback != null)
                    callback.result(null, success);
            }
        });
        progressDialog.show();
    }

    public boolean importPagesFromDCIM(final int dstIndex, String imagePath) {
        return mPDFViewCtrl.addImagePage(dstIndex, imagePath);
    }

    public boolean importPagesFromCamera(final int dstIndex, String imagePath) {
        return mPDFViewCtrl.addImagePage(dstIndex, imagePath);
    }

    public void importPagesFromVideo(final int dstIndex, final String videoPath, final Event.Callback callback) {
        final FxProgressDialog progressDialog = mSupport.getProgressDialog();
        progressDialog.setTips(AppResource.getString(mContext, R.string.fx_string_processing));

        CreatePageBean createPageBean = new CreatePageBean();
        int[] ranges = {dstIndex, createPageBean.getPageCounts()};
        doInsertPages(dstIndex, ranges, createPageBean, new EditThumbnailCallback() {
            @Override
            public void onProgressChanged(int type, int progress) {
            }

            @Override
            public void result(boolean success, PDFDoc srcDoc, PDFDoc dstDoc) {
                try {
                    PDFPage page = mPDFViewCtrl.getDoc().getPage(dstIndex);
                    MultimediaModule module = (MultimediaModule) mUIExtensionsManager.getModuleByName(Module.MODULE_NAME_MEDIA);
                    MultimediaToolHandler toolHandler = module.getVideoToolHandler();

                    float pageWidth = page.getWidth();
                    float pageHeight = page.getHeight();

                    Bitmap thumbnail = AppFileUtil.getVideoThumbnail(videoPath);
                    if (thumbnail != null) {
                        float thumbnailWidth = thumbnail.getWidth();
                        float thumbnailHeight = thumbnail.getHeight();

                        float widthScale = thumbnailWidth / pageWidth;
                        float heightScale = thumbnailHeight / pageHeight;
                        float scale;
                        float newWidth;
                        float newHeight;
                        if (widthScale > heightScale) {
                            scale = widthScale;
                            newWidth = pageWidth;
                            newHeight = thumbnailHeight / widthScale;
                        } else {
                            scale = heightScale;
                            newWidth = thumbnailWidth / heightScale;
                            newHeight = pageHeight;
                        }
                        page.setSize(newWidth, newHeight);

                        Matrix matrix = new Matrix();
                        matrix.postScale(scale, scale);
                        Bitmap _thumbnail = Bitmap.createBitmap(thumbnail, 0, 0, (int) thumbnailWidth, (int) thumbnailHeight, matrix, true);
                        thumbnail.recycle();

                        RectF rectF = new RectF(0, newHeight, newWidth, 0);
                        toolHandler.createAnnot(dstIndex, rectF, videoPath, _thumbnail, new Event.Callback() {
                            @Override
                            public void result(Event event, boolean success) {
                                progressDialog.dismiss();
                                if (callback != null)
                                    callback.result(event, success);
                            }
                        });
                    }
                } catch (Exception ignored) {
                    progressDialog.dismiss();
                    if (callback != null)
                        callback.result(null, false);
                }
            }
        });
        progressDialog.show();
    }

    void extractPages(final Event.Callback callback) {
        final UIFolderSelectDialog dialog = mSupport.getFolderSelectDialog();
        dialog.setListener(new MatchDialog.DialogListener() {
            @Override
            public void onResult(long btType) {
            }

            @Override
            public void onBackClick() {
            }

            @Override
            public void onTitleRightButtonClick() {
                String fileFolder = dialog.getCurrentPath();
                if (!AppStorageManager.getInstance(mContext).checkDirectoryPermission(fileFolder))
                    return;
                showInputFileNameDialog(fileFolder, callback);
                dialog.dismiss();
            }
        });
        dialog.setFileFilter(null);
        dialog.resetWH();
        dialog.showDialog();
        dialog.notifyDataSetChanged();
    }

    private String mCacheFolder_share;
    public void cleanCacheFolder() {
        File folder = new File(mCacheFolder_share);
        if (folder.exists())
            AppFileUtil.deleteFolder(folder, true);
    }

    public void sharePages() {
        File folder = new File(mCacheFolder_share);
        if (!folder.exists())
            folder.mkdirs();

        final String path = folder.getAbsolutePath() + File.separator + AppFileUtil.getFileName(mPDFViewCtrl.getFilePath());
        doExtractPagesSilence(mSelectedArray, path, new Event.Callback() {
            @Override
            public void result(Event event, boolean success) {
                if (success) {
                    File file = new File(path);
                    Uri uri = Uri.fromFile(file);
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP_MR1) {
                        //通过FileProvider创建一个content类型的Uri
                        uri = FileProvider.getUriForFile(mUIExtensionsManager.getAttachedActivity() ,
                                AppIntentUtil.getFileProviderName(mContext), file);
                    }

                    final Intent intent = new Intent(Intent.ACTION_SEND);
                    intent.setType("application/pdf");
                    intent.putExtra(Intent.EXTRA_STREAM, uri);
                    intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                    mUIExtensionsManager.getAttachedActivity().startActivityForResult(
                            Intent.createChooser(intent, AppResource.getString(mContext, R.string.fx_string_share)), ActRequestCode.REQ_FILE_FROM_THUMB_SHARE);
                }
            }
        });

    }

    class EditThumbnailTask extends Task {
        private final int mType;
        private final SparseArray<ThumbnailItem> tmpItemLists;
        private int mInsertPosition;
        private int[] mImportRanges;
        private PDFDoc mSrcDoc;
        private PDFDoc mDstDoc;

        private EditThumbnailCallback mEditCallback;
        private CreatePageBean mPageBean;
        private String mExtractPath;

        private boolean mSuccess = false;

        EditThumbnailTask(int type, SparseArray<ThumbnailItem> itemLists, final EditThumbnailCallback callback) {
            super(new CallBack() {
                @Override
                public void result(Task task) {
                    if (callback != null) {
                        EditThumbnailTask task1 = ((EditThumbnailTask) task);
                        callback.result(task1.mSuccess, task1.mSrcDoc, task1.mSrcDoc);
                    }
                }
            });
            mSrcDoc = mPDFViewCtrl.getDoc();
            mType = type;
            tmpItemLists = itemLists;
            mEditCallback = callback;
        }

        EditThumbnailTask(int position, int[] ranges, PDFDoc doc, final EditThumbnailCallback callback) {
            super(new CallBack() {
                @Override
                public void result(Task task) {
                    if (callback != null) {
                        EditThumbnailTask task1 = ((EditThumbnailTask) task);
                        callback.result(task1.mSuccess, task1.mSrcDoc, task1.mSrcDoc);
                    }
                }
            });
            mType = INSERT_FROM_DOCUMENT;
            tmpItemLists = null;
            mInsertPosition = position;
            mImportRanges = ranges;
            mSrcDoc = doc;
            mEditCallback = callback;
        }

        EditThumbnailTask(int[] ranges, PDFDoc doc, final EditThumbnailCallback callback) {
            super(new CallBack() {
                @Override
                public void result(Task task) {
                    if (callback != null) {
                        EditThumbnailTask task1 = ((EditThumbnailTask) task);
                        callback.result(task1.mSuccess, task1.mSrcDoc, task1.mSrcDoc);
                    }
                }
            });
            mType = COPY_PAGES;
            tmpItemLists = null;
            mInsertPosition = ranges[ranges.length - 2] + 1;
            mImportRanges = ranges;
            mSrcDoc = doc;
            mEditCallback = callback;
        }

        EditThumbnailTask(int position, int[] ranges, CreatePageBean pageBean, final EditThumbnailCallback callback) {
            super(new CallBack() {
                @Override
                public void result(Task task) {
                    if (callback != null) {
                        EditThumbnailTask task1 = ((EditThumbnailTask) task);
                        callback.result(task1.mSuccess, task1.mSrcDoc, task1.mSrcDoc);
                    }
                }
            });
            mType = INSERT_PAGES;
            tmpItemLists = null;
            mInsertPosition = position;
            mImportRanges = ranges;
            mPageBean = pageBean;
            mEditCallback = callback;
        }

        EditThumbnailTask(SparseArray<ThumbnailItem> itemLists, String path, final EditThumbnailCallback callback) {
            super(new CallBack() {
                @Override
                public void result(Task task) {
                    if (callback != null) {
                        EditThumbnailTask task1 = ((EditThumbnailTask) task);
                        callback.result(task1.mSuccess, task1.mSrcDoc, task1.mDstDoc);
                    }
                }
            });
            mSrcDoc = mPDFViewCtrl.getDoc();
            mType = EXPORT_PAGE;
            tmpItemLists = itemLists;
            mExtractPath = path;
            mEditCallback = callback;
        }

        @Override
        protected void execute() {
            mIsEditing = true;
            switch (mType) {
                case REMOVE_PAGE:
                case ROTATE_PAGE_CW:
                case ROTATE_PAGE_ACW:
                    editSelectedPages();
                    break;
                case INSERT_FROM_DOCUMENT:
                    insertPages();
                    break;
                case EXPORT_PAGE:
                    extractPages();
                    break;
                case COPY_PAGES:
                    copyPages();
                    break;
                case INSERT_PAGES:
                    createPages();
                    break;
                default://not support now
                    break;
            }
            mIsEditing = false;
            if (mSuccess && mType != EXPORT_PAGE) {
                mUIExtensionsManager.getDocumentManager().setDocModified(true);
            }
        }

        private void editSelectedPages() {
            if (mType == REMOVE_PAGE) {
                int[] pageIndexes = new int[tmpItemLists.size()];
                for (int i = 0; i < tmpItemLists.size(); i++) {
                    pageIndexes[i] = tmpItemLists.valueAt(i).getPageIndex();
                }
                mSuccess = mPDFViewCtrl.removePages(pageIndexes);
            } else {
                int tmpSize = tmpItemLists.size();
                for (int i = 0; i < tmpSize; i++) {
                    ThumbnailItem item = tmpItemLists.valueAt(i);
                    switch (mType) {
                        case ROTATE_PAGE_CW: {
                            int rotation = item.getRotation();
                            mSuccess = item.setRotation(rotation < 3 ? rotation + 1 : 3 - rotation);
                        }
                        break;
                        case ROTATE_PAGE_ACW: {
                            int rotation = item.getRotation();
                            mSuccess = item.setRotation(rotation > 0 ? rotation - 1 : 3 + rotation);
                        }
                        break;
                        default:
                            break;
                    }
                }
            }
        }

        private void insertPages() {
            int flags = PDFDoc.e_ImportFlagNormal;
            try {
                flags = mSrcDoc.isEncrypted() ? PDFDoc.e_ImportFlagNormal : PDFDoc.e_ImportFlagShareStream;
            } catch (PDFException e) {
                e.printStackTrace();
            }
            mSuccess = mPDFViewCtrl.insertPages(mInsertPosition, flags, null, mSrcDoc, mImportRanges);
        }

        private void createPages() {
            float width = mPageBean.getWidth();
            float height = mPageBean.getHeight();
            if (Constants.e_Rotation90 == mPageBean.getPageDirection() ||
                    Constants.e_Rotation270 == mPageBean.getPageDirection()) {
                width = mPageBean.getHeight();
                height = mPageBean.getWidth();
            }
            mSuccess = mPDFViewCtrl.insertPages(mInsertPosition,
                    width,
                    height,
                    mPageBean.getPageStyle(),
                    mPageBean.getPageColor(),
                    Constants.e_Rotation0,
                    mPageBean.getPageCounts());
        }

        private void copyPages() {
            int flags = PDFDoc.e_ImportFlagNormal;
            try {
                flags = mSrcDoc.isEncrypted() ? PDFDoc.e_ImportFlagNormal : PDFDoc.e_ImportFlagShareStream;
            } catch (PDFException e) {
                e.printStackTrace();
            }
            mSuccess = mPDFViewCtrl.insertPages(mInsertPosition, flags, null, mSrcDoc, mImportRanges);
        }

        private void extractPages() {
            try {
                mDstDoc = new PDFDoc();
                ArrayList<Boolean> extractFlagList = new ArrayList<>();
                for (int i = 0; i < mPDFViewCtrl.getPageCount(); i++) {
                    extractFlagList.add(false);
                }

                int tmpSize = tmpItemLists.size();
                for (int i = 0; i < tmpSize; i++) {
                    int index = tmpItemLists.valueAt(i).getPageIndex();
                    extractFlagList.set(index, true);
                }
                ArrayList<Integer> rangeList = new ArrayList<>();

                int lastIndex = -1;
                int count = 0;
                for (int i = 0; i < extractFlagList.size(); i++) {
                    if (extractFlagList.get(i)) {
                        if (lastIndex == -1)
                            lastIndex = i;
                        count++;
                    } else {
                        if (lastIndex == -1) {
                            count = 0;
                        } else {
                            rangeList.add(lastIndex);
                            rangeList.add(count);
                            lastIndex = -1;
                            count = 0;
                        }
                    }
                }
                if (lastIndex != -1) {
                    rangeList.add(lastIndex);
                    rangeList.add(count);
                }

                Range ranges = new Range();
                for (int i = 0; i < rangeList.size(); i += 2) {
                    ranges.addSegment(rangeList.get(i), rangeList.get(i) + rangeList.get(i + 1) - 1, Range.e_All);
                }
                Progressive importProgressive = mDstDoc.startImportPages(mDstDoc.getPageCount(), mSrcDoc, PDFDoc.e_ImportFlagShareStream, null, ranges, null);
                int importState = Progressive.e_ToBeContinued;
                while (importState == Progressive.e_ToBeContinued) {
                    if (mEditCallback != null)
                        mEditCallback.onProgressChanged(mType, importProgressive.getRateOfProgress());
                    importState = importProgressive.resume();
                    if (mEditCallback != null)
                        mEditCallback.onProgressChanged(mType, importProgressive.getRateOfProgress());
                }

                if (AppUtil.isEmpty(mExtractPath)) {
                    mSuccess = true;
                } else {
                    Progressive progressive = mDstDoc.startSaveAs(mExtractPath, PDFDoc.e_SaveFlagIncremental, null);
                    int state = Progressive.e_ToBeContinued;
                    while (state == Progressive.e_ToBeContinued) {
                        state = progressive.resume();
                    }
                    mDstDoc.delete();
                    mSuccess = (state == Progressive.e_Finished);
                    if (mSuccess) {
                        AppStorageManager storageManager = AppStorageManager.getInstance(mContext);
                        String target = storageManager.toExternalPathFromScopedCache(mExtractPath);
                        if (target != null) {
                            mSuccess = storageManager.copyDocument(mExtractPath, target, true);
                        }
                    }
                }
            } catch (PDFException e) {
                mSuccess = false;
            }
        }
    }

    interface EditThumbnailCallback {
        void onProgressChanged(int type, int progress);

        void result(boolean success, PDFDoc srcDoc, PDFDoc dstDoc);
    }

    public class ThumbViewHolder extends RecyclerView.ViewHolder {
        private final RelativeLayout mPasteRootView;
        private final TextView mPasteView;
        private final ImageView mClostPasteView;
        private final ImageView mBookmarkView;
        private final TextView mIndexView;
        private final TextView mSelectedCount;
        private final ImageView mImageView;
        private final CheckBox mSelectView;
        private final MaterialCardView mContentCardView;
        protected Bitmap mThumbnailBitmap;

        public ThumbViewHolder(View itemView) {
            super(itemView);
            mPasteRootView = itemView.findViewById(R.id.thumbnail_item_paste);
            mPasteView = itemView.findViewById(R.id.thumbnail_item_paste_tv);
            mClostPasteView = itemView.findViewById(R.id.thumbnail_item_close_paste);
            mContentCardView = itemView.findViewById(R.id.thumbnail_root_view);
            mBookmarkView = itemView.findViewById(R.id.thumbnail_item_bookmark);
            mIndexView = itemView.findViewById(R.id.thumbnail_page_text);
            mImageView = itemView.findViewById(R.id.item_image);
            mSelectView = itemView.findViewById(R.id.thumbnail_select_view);
            mSelectedCount = itemView.findViewById(R.id.thumbnail_select_count);
            ThemeUtil.setBackgroundTintList(mSelectedCount, getPrimaryTintList(mContext));
        }

        public void updateImageView() {
            mImageView.setImageBitmap(getThumbnailBitmap());
            mImageView.invalidate();
        }

        public Bitmap getThumbnailBitmap() {
            if (mThumbnailBitmap == null)
                mThumbnailBitmap = Bitmap.createBitmap(mSupport.getThumbnailBackgroundSize().x, mSupport.getThumbnailBackgroundSize().y, Bitmap.Config.RGB_565);
            return mThumbnailBitmap;
        }

        protected void blank(ThumbnailItem item) {
            Point size = item.getSize();
            if (size.x == 0 || size.y == 0) return;
            Bitmap bitmap = getThumbnailBitmap();
            bitmap.eraseColor(AppResource.getColor(mContext, R.color.ux_grid_item_bg, null));
            mImageView.setImageBitmap(bitmap);
            mImageView.invalidate();
        }

        public void changeSelectView(boolean selected) {
            if (mSupport.isEditMode()) {
                mSelectView.setVisibility(View.VISIBLE);
                mSelectView.setChecked(selected);
                mSelectView.getBackground().setColorFilter(
                        new PorterDuffColorFilter(AppResource.getColor(mContext, R.color.b1), PorterDuff.Mode.SRC_IN));
                ThemeUtil.setTintList(mSelectView, ThemeUtil.getCheckboxColor(mContext));
            } else {
                mSelectView.setVisibility(View.GONE);
            }
        }

        public void drawThumbnail(final ThumbnailItem item, int position) {
            if (item.itemType == ThumbnailItem.PASTE_ITEM) {
                RelativeLayout.LayoutParams pasteParams = (RelativeLayout.LayoutParams) mPasteRootView.getLayoutParams();
                pasteParams.width = item.getBackgroundSize().x;
                pasteParams.height = item.getBackgroundSize().y;
                mPasteRootView.setLayoutParams(pasteParams);

                mContentCardView.setVisibility(View.GONE);
                mPasteRootView.setVisibility(View.VISIBLE);
                mSelectedCount.setVisibility(View.GONE);
                mSelectedCount.setTextColor(AppResource.getColor(mContext, R.color.t1));
                ThemeUtil.setTintList(mClostPasteView, ThemeUtil.getItemIconColor(mContext));
                ThemeUtil.setBackgroundTintList(mPasteView, getPrimaryTintList(mContext));
                mPasteView.getBackground().setAlpha(50);
                mPasteView.setTextColor(AppResource.getColor(mContext, R.color.t4));
                int pasteBgColor = ThemeConfig.getInstance(mContext).getPrimaryColor();
                mPasteRootView.getBackground().setColorFilter(new PorterDuffColorFilter(pasteBgColor, PorterDuff.Mode.SRC_IN));

                mPasteView.setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        pastePages(mTempPasteDoc);
                    }
                });
                mClostPasteView.setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        removePasteItem();
                        mSupport.switchPasteState(false);
                    }
                });
            } else {
                mContentCardView.setVisibility(View.VISIBLE);
                mPasteRootView.setVisibility(View.GONE);
                mBookmarkView.setVisibility(item.isReadingBookmark() ? View.VISIBLE : View.GONE);
                mBookmarkView.setColorFilter(ThemeConfig.getInstance(mContext).getPrimaryColor());

                changeSelectView(item.isSelected());
                if (item.isDragging()) {
                    mSelectedCount.setVisibility(View.VISIBLE);
                    mSelectedCount.setText(String.format("%s", mSelectedArray.size()));
                } else {
                    mSelectedCount.setVisibility(View.GONE);
                }
                if (mCurrentPage == item.getPageIndex()) {
                    mContentCardView.setStrokeColor(ThemeConfig.getInstance(mContext).getPrimaryColor());
                    mContentCardView.setStrokeWidth(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_item_border_selected_width));
                    mIndexView.setBackground(AppResource.getDrawable(mContext, R.drawable.thumbnail_textview_background_current));
                    ThemeUtil.setBackgroundTintList(mIndexView, getPrimaryTintList(mContext));
                } else {
                    mContentCardView.setStrokeColor(AppResource.getColor(mContext, R.color.p1));
                    mContentCardView.setStrokeWidth(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_item_border_unselected_width));
                    mIndexView.setBackground(AppResource.getDrawable(mContext, R.drawable.thumbnail_textview_background_normal));
                    ThemeUtil.setBackgroundTintList(mIndexView, null);
                }
                mContentCardView.setCardBackgroundColor(AppResource.getColor(mContext, R.color.ux_grid_item_bg));
                mIndexView.getBackground().setAlpha(125);
                String indexStr;
                if (LogicalPageHelper.getInstance().isSupportLogicalPage()) {
                    indexStr = LogicalPageHelper.getInstance().getLogicalPageNumber(item.getPageIndex(), mPDFViewCtrl.getDoc());
                } else {
                    indexStr = String.format("%s", item.getPageIndex() + 1);
                }
                mIndexView.setText(indexStr);
                mIndexView.setTextColor(AppResource.getColor(mContext, R.color.t1));
                if (item.getBitmap() != null && !item.needRecompute()) {
                    mImageView.setImageBitmap(item.getBitmap());
                    mImageView.invalidate();
                } else {
                    blank(item);
                    if (item.isRendering() && !item.needRecompute()) {
                        return;
                    }
                    if (!mIsEditing) {
                        final DrawThumbnailTask drawThumbnailTask = new DrawThumbnailTask(position, item, new DrawThumbnailCallback() {
                            @Override
                            public void result(ThumbnailItem item, DrawThumbnailTask task, Bitmap bitmap) {
                                if (mSupport == null) return;
                                ThumbViewHolder viewHolder = mSupport.getViewHolderByItem(mTabPosition, item);
                                if (bitmap == null || viewHolder == null)
                                    return;
                                Bitmap vhBitmap = viewHolder.getThumbnailBitmap();
                                Canvas canvas = new Canvas(vhBitmap);
                                canvas.drawBitmap(bitmap, null, item.getRect(), new Paint());
                                item.setBitmap(Bitmap.createBitmap(vhBitmap));

                                viewHolder.updateImageView();
                                updateCacheListInfo(item, true);
                                removeTask(task);
                            }
                        });
                        addTask(drawThumbnailTask);
                    }
                }
            }
        }

        private ColorStateList getPrimaryTintList(Context context) {
            int normal = ThemeConfig.getInstance(context).getPrimaryColor();
            return AppResource.createColorStateList(normal, normal, normal, normal, normal);
        }
    }
}