/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.textselect;

import android.content.ClipData;
import android.content.ClipboardManager;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.PointF;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffXfermode;
import android.graphics.Rect;
import android.graphics.RectF;
import android.view.MotionEvent;
import android.view.View;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.common.Range;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.TextPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotEventListener;
import com.foxit.uiextensions.annots.common.UIMagnifierView;
import com.foxit.uiextensions.annots.redaction.RedactModule;
import com.foxit.uiextensions.annots.textmarkup.TextMarkupContentAbs;
import com.foxit.uiextensions.annots.textmarkup.TextSelector;
import com.foxit.uiextensions.controls.propertybar.AnnotMenu;
import com.foxit.uiextensions.controls.propertybar.MenuItem;
import com.foxit.uiextensions.controls.propertybar.imp.AnnotMenuImpl;
import com.foxit.uiextensions.modules.tts.TTSInfo;
import com.foxit.uiextensions.modules.tts.TTSModule;
import com.foxit.uiextensions.modules.tts.TTSUtils;
import com.foxit.uiextensions.pdfreader.impl.MainFrame;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.UIToast;

import java.util.ArrayList;

public class TextSelectToolHandler implements ToolHandler {
    private static final int HANDLE_AREA = 10;

    private Context mContext;
    private PDFViewCtrl mPdfViewCtrl;
    final TextSelector mSelectInfo;
    private int mCurrentIndex;
    private RectF mTmpRect;
    private Paint mPaint;
    private Bitmap mHandlerBitmap;
    boolean mIsEdit;
    public AnnotMenu mAnnotationMenu;
    private ArrayList<Integer> mText;
    private ArrayList<Integer> mTTS;

    private AnnotEventListener mAnnotListener;
    private RectF mMenuBox;
    private boolean mNoFirstRedact = false;
    private ArrayList<SelectExtentsionMenuItem> mExtMenuItems = new ArrayList<>();

    private UIMagnifierView mMagnifierView;
    private int FINGER_AREA;

    protected AnnotMenu getAnnotationMenu() {
        return mAnnotationMenu;
    }

    private UIExtensionsManager.ToolHandlerChangedListener mHandlerChangedListener = new UIExtensionsManager.ToolHandlerChangedListener() {
        @Override
        public void onToolHandlerChanged(ToolHandler lastTool, ToolHandler currentTool) {

            if (currentTool != null && mIsEdit) {
                RectF rectF = new RectF(mSelectInfo.getBbox());
                mSelectInfo.clear();
                mPdfViewCtrl.convertPdfRectToPageViewRect(mSelectInfo.getBbox(), rectF, mCurrentIndex);
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectF, rectF, mCurrentIndex);
                RectF rF = AppUtil.calculateRect(rectF, mTmpRect);
                Rect rect = new Rect();
                rF.roundOut(rect);
                rect.top -= mHandlerBitmap.getHeight();
                rect.bottom += mHandlerBitmap.getHeight();
                rect.left -= mHandlerBitmap.getWidth() / 2;
                rect.right += mHandlerBitmap.getWidth() / 2;
                mPdfViewCtrl.invalidate(rect);
                mIsEdit = false;
                mAnnotationMenu.dismiss();
                if(mMagnifierView != null){
                    mMagnifierView.setVisibility(View.GONE);
                }
            }
        }
    };

    protected UIExtensionsManager.ToolHandlerChangedListener getHandlerChangedListener() {
        return mHandlerChangedListener;
    }

    @Override
    public String getType() {
        return ToolHandler.TH_TYPE_TEXTSELECT;
    }


    public String getCurrentSelectedText() {
        PDFPage page = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getPage(mCurrentIndex, false);
        if (page != null)
            return mSelectInfo.getText(page);
        return mSelectInfo.getContents();
    }

    public int getCurrentIndex() {
        return mCurrentIndex;
    }

    public RectF getCurrentSelectedRect() {
        return mSelectInfo.getBbox();
    }


    @Override
    public void onActivate() {
    }

    @Override
    public void onDeactivate() {
        RectF rectF = new RectF(mSelectInfo.getBbox());
        mSelectInfo.clear();
        if (!mPdfViewCtrl.isPageVisible(mCurrentIndex)) return;
        mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, mCurrentIndex);
        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectF, rectF, mCurrentIndex);
        RectF rF = AppUtil.calculateRect(rectF, mTmpRect);
        Rect rect = new Rect();
        rF.roundOut(rect);
        getInvalidateRect(rect);
        mPdfViewCtrl.invalidate(rect);
        mAnnotationMenu.dismiss();
        mIsEdit = false;
    }

    public TextSelectToolHandler(Context context, PDFViewCtrl pdfViewCtrl) {
        mPdfViewCtrl = pdfViewCtrl;
        mContext = context;
        mSelectInfo = new TextSelector(pdfViewCtrl);
        FINGER_AREA = (int) (AppDisplay.getFingerArea() / 4);
        mPaint = new Paint();
        mPaint.setAntiAlias(true);
        mPaint.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.MULTIPLY));

        mTmpRect = new RectF();
        mHandlerBitmap = BitmapFactory.decodeResource(mContext.getResources(), R.drawable.rv_textselect_handler);
        mAnnotationMenu = new AnnotMenuImpl(context, mPdfViewCtrl);

        mIsEdit = false;
        mText = new ArrayList<>();
        mTTS = new ArrayList<>();

        mAnnotListener = new AnnotEventListener() {
            @Override
            public void onAnnotAdded(PDFPage page, Annot annot) {
            }

            @Override
            public void onAnnotWillDelete(PDFPage page, Annot annot) {
            }

            @Override
            public void onAnnotDeleted(PDFPage page, Annot annot) {

            }

            @Override
            public void onAnnotModified(PDFPage page, Annot annot) {
            }

            @Override
            public void onAnnotChanged(Annot lastAnnot, Annot currentAnnot) {
                if (currentAnnot != null && mIsEdit) {
                    RectF rectF = new RectF(mSelectInfo.getBbox());
                    mSelectInfo.clear();
                    if (!mPdfViewCtrl.isPageVisible(mCurrentIndex)) return;
                    mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, mCurrentIndex);
                    mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectF, rectF, mCurrentIndex);
                    RectF rF = AppUtil.calculateRect(rectF, mTmpRect);
                    Rect rect = new Rect();
                    rF.roundOut(rect);
                    getInvalidateRect(rect);
                    mPdfViewCtrl.invalidate(rect);
                    mIsEdit = false;
                    mAnnotationMenu.dismiss();
                }
            }
        };
        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().registerAnnotEventListener(mAnnotListener);
        initMagnifierView();
    }

    public void reloadMenu() {
        mText.clear();
        mTTS.clear();

        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        if (uiExtensionsManager.getDocumentManager().canCopy()) {
            mText.add(AnnotMenu.AM_BT_COPY);
        }

        if (uiExtensionsManager.getDocumentManager().canAddAnnot()
                && uiExtensionsManager.isEnableModification()
                && uiExtensionsManager.getDocumentManager().withAddPermission()) {

            if (uiExtensionsManager.getAnnotHandlerByType(Annot.e_Highlight) != null)
                mText.add(AnnotMenu.AM_BT_HIGHLIGHT);
            if (uiExtensionsManager.getAnnotHandlerByType(Annot.e_Underline) != null)
                mText.add(AnnotMenu.AM_BT_UNDERLINE);
            if (uiExtensionsManager.getConfig().modules.annotations.isLoadStrikeout
                    && uiExtensionsManager.getAnnotHandlerByType(Annot.e_StrikeOut) != null)
                mText.add(AnnotMenu.AM_BT_STRIKEOUT);
            if (uiExtensionsManager.getAnnotHandlerByType(Annot.e_Squiggly) != null)
                mText.add(AnnotMenu.AM_BT_SQUIGGLY);
            if (uiExtensionsManager.getConfig().modules.annotations.isLoadRedaction
                    && uiExtensionsManager.getAnnotHandlerByType(Annot.e_Redact) != null)
                mText.add(AnnotMenu.AM_BT_REDACT);
        }

        if (uiExtensionsManager.getDocumentManager().canCopy()
                && uiExtensionsManager.getModuleByName(Module.MODULE_NAME_TTS) != null
                && ((TTSModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_TTS)).isSupperTts()) {
            mText.add(AnnotMenu.AM_BT_TTS);
        }

        mTTS.add(AnnotMenu.AM_BT_TTS_STRING);
        mTTS.add(AnnotMenu.AM_BT_TTS_START);
    }

    int[] enSeparatorList = {0, 10, 13, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 58, 59, 60, 61, 62, 63, 64}; //{0，\n, \r, space, !, ", #, $, %, &, ', (, ), *, +, ,, -, ., /, :, ;, <, =, >, ?, @}

    private void findEnWord(int pageIndex, TextSelector info, int index) {
        info.setStart(index);
        info.setEnd(index);
        try {
            PDFPage page = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getPage(pageIndex, false);
            if (page == null || page.isEmpty()) return;
            TextPage textPage = new TextPage(page, TextPage.e_ParseTextNormal);

            String charInfo;
            for (; info.getStart() >= 0; info.setStart(info.getStart() - 1)) {
                charInfo = textPage.getChars(info.getStart(), 1);
                if (charInfo == null || charInfo.isEmpty()) {
                    info.setStart(info.getStart() + 1);
                    break;
                }
                int i;
                for (i = 0; i < enSeparatorList.length; i++) {
                    if (enSeparatorList[i] == charInfo.charAt(0)) {
                        break;
                    }
                }
                if (i != enSeparatorList.length) {
                    info.setStart(info.getStart() + 1);
                    break;
                }
            }

            if (info.getStart() < 0) {
                info.setStart(0);
            }

            for (; ; info.setEnd(info.getEnd() + 1)) {
                charInfo = textPage.getChars(info.getEnd(), 1);
                if (charInfo == null || charInfo.isEmpty()) {
                    info.setEnd(info.getEnd() - 1);
                    break;
                }
                int i;
                for (i = 0; i < enSeparatorList.length; i++) {
                    if (enSeparatorList[i] == charInfo.charAt(0)) {
                        break;
                    }
                }
                if (i != enSeparatorList.length) {
                    info.setEnd(info.getEnd() - 1);
                    break;
                }
            }
//            if (charInfo == null || charInfo.isEmpty()) {
//                info.setEnd(info.getEnd() - 1);
//            }
        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
        }
    }

    int[] chPassList = {0, 10, 13, 32, 8230, 12290, 65281, 65292, 65306, 65307, 65311}; // {0, \n, \r, space, ..., 。, ！, ，, ：, ；, ？}

    private void findChWord(int pageIndex, TextSelector info, int index) {
        info.setStart(index);
        info.setEnd(index);
        info.setStart(info.getStart() - 1);
        info.setEnd(info.getEnd() + 1);
        try {
            PDFPage page = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getPage(pageIndex, false);
            if (page == null || page.isEmpty()) return;
            TextPage textPage = new TextPage(page, TextPage.e_ParseTextNormal);

            String charinfo;
            charinfo = textPage.getChars(index, 1);

            for (; info.getStart() >= 0; info.setStart(info.getStart() - 1)) {
                charinfo = textPage.getChars(info.getStart(), 1);
                if (charinfo == null || charinfo.isEmpty()) {
                    info.setStart(info.getStart() + 1);
                    break;
                }

                int i;
                for (i = 0; i < chPassList.length; i++) {
                    if (chPassList[i] == charinfo.charAt(0)) {
                        break;
                    }
                }
                if (i != chPassList.length) {
                    info.setStart(info.getStart() + 1);
                    break;
                }
            }
            if (info.getStart() < 0) {
                info.setStart(0);
            }

            for (; info.getEnd() >= 0; info.setEnd(info.getEnd() + 1)) {
                charinfo = textPage.getChars(info.getEnd(), 1);
                if (charinfo == null || charinfo.isEmpty()) {
                    info.setEnd(info.getEnd() - 1);
                    break;
                }

                int i;
                for (i = 0; i < chPassList.length; i++) {
                    if (chPassList[i] == charinfo.charAt(0)) {
                        break;
                    }
                }
                if (i != chPassList.length) {
                    info.setEnd(info.getEnd() - 1);
                    break;
                }
            }
//            if (charinfo == null || charinfo.isEmpty()) {
//                info.setEnd(info.getEnd() - 1);
//            }
        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
        }
    }

    private int mCtrlPoint = 0;

    public void uninit() {
        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().unregisterAnnotEventListener(mAnnotListener);
        removeMagnifierView();
    }

    @Override
    public boolean onTouchEvent(int pageIndex, MotionEvent motionEvent) {
        return handleOnTouchEvent(pageIndex, motionEvent);
    }

    public static boolean noSelectRedact(UIExtensionsManager extensionsManager) {
        RedactModule module = (RedactModule) extensionsManager.getModuleByName(Module.MODULE_NAME_REDACT);
        if (module == null)return true;
        return module.noSelectRedact();
    }

    @Override
    public boolean onLongPress(final int pageIndex, MotionEvent motionEvent) {
        if (mMagnifierView != null && !mPdfViewCtrl.isDynamicXFA() && !mIsEdit){
            mCtrlPoint = 2;
            mPdfViewCtrl.capturePageViewOnTouch(motionEvent);
        }
        final PointF pointF = AppAnnotUtil.getPageViewPoint(mPdfViewCtrl, pageIndex, motionEvent);
        try {
            final UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager());
            if (uiExtensionsManager.getDocumentManager().getCurrentAnnot() != null) {
                uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
                return false;
            }

            if (mIsEdit) {
                RectF rectF = new RectF(mSelectInfo.getBbox());
                mSelectInfo.clear();
                mCurrentIndex = pageIndex;
                mPdfViewCtrl.convertPdfRectToPageViewRect(mSelectInfo.getBbox(), rectF, mCurrentIndex);
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectF, rectF, mCurrentIndex);
                RectF rF = AppUtil.calculateRect(rectF, mTmpRect);
                Rect rect = new Rect();
                rF.roundOut(rect);
                getInvalidateRect(rect);
                mPdfViewCtrl.invalidate(rect);
                mIsEdit = false;
                mAnnotationMenu.dismiss();
                return true;
            }
            if (mAnnotationMenu.isShowing()) {
                mAnnotationMenu.dismiss();
            }

            mCurrentIndex = pageIndex;
            PointF pointPdfView = new PointF();
            mPdfViewCtrl.convertPageViewPtToPdfPt(pointF, pointPdfView, mCurrentIndex);
            final PDFPage page = uiExtensionsManager.getDocumentManager().getPage(mCurrentIndex, false);
            if (page == null || page.isEmpty())
                return false;

            final TextPage textPage = new TextPage(page, TextPage.e_ParseTextNormal);
            int index = textPage.getIndexAtPos(pointPdfView.x, pointPdfView.y, 10);

            if (index == -1) {
                return true;
            }
            String info = textPage.getChars(index, 1);
            if (info.length() == 0) return false;
            if (index >= 0) {
                reloadMenu();
                if (mText.size() == 0 && mExtMenuItems.size() == 0)
                    return false;

                Range range = textPage.getWordAtPos(pointPdfView.x, pointPdfView.y, 10);
                if(!range.isEmpty() && range.getSegmentCount() == 1) {
                    mSelectInfo.setStart(range.getSegmentStart(0));
                    mSelectInfo.setEnd(range.getSegmentEnd(0));
                } else {
                    if ((info.charAt(0) >= 65 && info.charAt(0) <= 90) || (info.charAt(0) >= 97 && info.charAt(0) <= 122)) {
                        findEnWord(mCurrentIndex, mSelectInfo, index);
                    } else {
                        findChWord(mCurrentIndex, mSelectInfo, index);
                    }
                }
                mSelectStartIndex = mSelectInfo.getStart();
                mSelectEndIndex = mSelectInfo.getEnd();
                mSelectInfo.computeSelected(page, mSelectInfo.getStart(), mSelectInfo.getEnd());
                invalidateTouch(mCurrentIndex, mSelectInfo);
                mIsEdit = true;
            } else {
                mIsEdit = false;
            }
            if (mSelectInfo.getRectFList().size() == 0) {
                mIsEdit = false;
            }
            if (mIsEdit) {
                if (!mPdfViewCtrl.isDynamicXFA()) {
                    mMagnifierView.onTouchEvent(motionEvent);
                    mMagnifierView.setVisibility(View.VISIBLE);
                }

                mMenuBox = new RectF(mSelectInfo.getBbox());
                mPdfViewCtrl.convertPdfRectToPageViewRect(mMenuBox, mMenuBox, mCurrentIndex);
                mMenuBox.inset(-10, -10);
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mMenuBox, mMenuBox, mCurrentIndex);
                if (mText.size() > 0) {
                    mAnnotationMenu.setMenuItems(mText);
                }
                if (mExtMenuItems.size() > 0) {
                    ArrayList<MenuItem> menuItems = new ArrayList<>();
                    for (int i = 0; i < mExtMenuItems.size(); i ++) {
                        menuItems.add(mExtMenuItems.get(i).getMenuItem());
                    }
                    mAnnotationMenu.setExtensionMenuItems(menuItems);
                }

                if (noSelectRedact((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager())) {
                    mAnnotationMenu.show(mMenuBox);
                }

                mAnnotationMenu.setListener(new AnnotMenu.ClickListener() {
                    @Override
                    public void onAMClick(final int btType) {
                        if (btType == AnnotMenu.AM_BT_COPY) {
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_LongPress_CopyText);
                            ClipboardManager clipboard = (ClipboardManager) mContext.getSystemService(Context.CLIPBOARD_SERVICE);
                            clipboard.setPrimaryClip(ClipData.newPlainText(null, mSelectInfo.getText(page)));
                            AppAnnotUtil.toastAnnotCopy(mContext);
                            RectF rectF = new RectF(mSelectInfo.getBbox());
                            mSelectInfo.clear();
                            mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, mCurrentIndex);
                            mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectF, rectF, mCurrentIndex);
                            RectF rF = AppUtil.calculateRect(rectF, mTmpRect);
                            Rect rect = new Rect();
                            rF.roundOut(rect);
                            getInvalidateRect(rect);
                            mPdfViewCtrl.invalidate(rect);
                            mIsEdit = false;
                            mAnnotationMenu.dismiss();
                            return;
                        }

                        if (btType == AnnotMenu.AM_BT_TTS) {
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_LongPress_Speak);
                            mIsEdit = true;
                            mAnnotationMenu.setMenuItems(mTTS);
                            mAnnotationMenu.setExtensionMenuItems(new ArrayList<MenuItem>());
                            mAnnotationMenu.setListener(new AnnotMenu.ClickListener() {
                                @Override
                                public void onAMClick(int btType) {
                                    TTSModule ttsModule = (TTSModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_TTS);

                                    mIsEdit = false;
                                    mAnnotationMenu.dismiss();
                                    TTSInfo ttsInfo = new TTSInfo();
                                    if (ttsModule.isSupportVoiceList() && ttsModule.isSelectedCurrentVoice())
                                        ttsInfo.mText = mSelectInfo.getText(page);
                                    else
                                        ttsInfo.mText = TTSUtils.changePunctsToEn(mSelectInfo.getText(page));
                                    ttsInfo.mStart = mSelectInfo.getStart();
                                    RectF rectF = new RectF(mSelectInfo.getBbox());
                                    ttsInfo.mRect = rectF;
                                    ttsInfo.mPageIndex = pageIndex;

                                    ArrayList<RectF> seletRectFList = mSelectInfo.getRectFList();
                                    int rectSize = seletRectFList.size();
                                    for (int i = 0; i < rectSize; i++) {
                                        RectF selectRect = seletRectFList.get(i);
                                        ttsInfo.mRects.add(new RectF(selectRect.left, selectRect.top, selectRect.right, selectRect.bottom));
                                    }

                                    mSelectInfo.clear();
                                    if (mPdfViewCtrl.isPageVisible(mCurrentIndex)) {
                                        mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, mCurrentIndex);
                                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectF, rectF, mCurrentIndex);
                                        RectF rF = AppUtil.calculateRect(rectF, mTmpRect);
                                        Rect rect = new Rect();
                                        rF.roundOut(rect);
                                        getInvalidateRect(rect);
                                        mPdfViewCtrl.invalidate(rect);
                                    }

                                    if (btType == AnnotMenu.AM_BT_TTS_STRING) {
                                        if (!AppUtil.isEmpty(ttsInfo.mText)) {
                                            ttsModule.speakFromTs(ttsInfo);
                                        }

                                    } else if (btType == AnnotMenu.AM_BT_TTS_START) {
                                        if (!AppUtil.isEmpty(ttsInfo.mText)) {
                                            ttsModule.speakFromTp(ttsInfo);
                                        }
                                    }
                                }
                            });

                            mAnnotationMenu.dismiss();
                            mAnnotationMenu.show(mMenuBox);
                            return;
                        }

                        if (btType == AnnotMenu.AM_BT_HIGHLIGHT) {
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_LongPress_Highlight);
                            uiExtensionsManager.getDocumentManager().addAnnot(page, new TextMarkupContentAbs() {
                                @Override
                                public TextSelector getTextSelector() {
                                    mSelectInfo.setContents(mSelectInfo.getText(page));
                                    return mSelectInfo;
                                }

                                @Override
                                public int getPageIndex() {
                                    return mCurrentIndex;
                                }

                                @Override
                                public int getType() {
                                    return Annot.e_Highlight;
                                }

                                @Override
                                public int getFillColor() {
                                    return 0;
                                }

                                @Override
                                public String getIntent() {
                                    return null;
                                }

                                @Override
                                public String getAuthor() {
                                    return null;
                                }

                            }, true, mAddResult);

                        } else if (btType == AnnotMenu.AM_BT_UNDERLINE) {
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_LongPress_Underline);
                            uiExtensionsManager.getDocumentManager().addAnnot(page, new TextMarkupContentAbs() {
                                @Override
                                public TextSelector getTextSelector() {
                                    mSelectInfo.setContents(mSelectInfo.getText(page));
                                    return mSelectInfo;
                                }

                                @Override
                                public int getPageIndex() {
                                    return mCurrentIndex;
                                }

                                @Override
                                public int getType() {
                                    return Annot.e_Underline;
                                }

                                @Override
                                public int getFillColor() {
                                    return 0;
                                }

                                @Override
                                public String getIntent() {
                                    return null;
                                }

                                @Override
                                public String getAuthor() {
                                    return null;
                                }
                            }, true, mAddResult);

                        } else if (btType == AnnotMenu.AM_BT_STRIKEOUT) {
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_LongPress_Strikeout);
                            uiExtensionsManager.getDocumentManager().addAnnot(page, new TextMarkupContentAbs() {
                                @Override
                                public TextSelector getTextSelector() {
                                    mSelectInfo.setContents(mSelectInfo.getText(page));
                                    return mSelectInfo;
                                }

                                @Override
                                public int getPageIndex() {
                                    return mCurrentIndex;
                                }

                                @Override
                                public int getType() {
                                    return Annot.e_StrikeOut;
                                }

                                @Override
                                public int getFillColor() {
                                    return 0;
                                }

                                @Override
                                public String getIntent() {
                                    return null;
                                }

                                @Override
                                public String getAuthor() {
                                    return null;
                                }

                            }, true, mAddResult);

                        } else if (btType == AnnotMenu.AM_BT_SQUIGGLY) {
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_LongPress_Squiggly);
                            uiExtensionsManager.getDocumentManager().addAnnot(page, new TextMarkupContentAbs() {
                                @Override
                                public TextSelector getTextSelector() {
                                    mSelectInfo.setContents(mSelectInfo.getText(page));
                                    return mSelectInfo;
                                }

                                @Override
                                public int getPageIndex() {
                                    return mCurrentIndex;
                                }

                                @Override
                                public int getType() {
                                    return Annot.e_Squiggly;
                                }

                                @Override
                                public int getFillColor() {
                                    return 0;
                                }

                                @Override
                                public String getIntent() {
                                    return null;
                                }

                                @Override
                                public String getAuthor() {
                                    return null;
                                }

                            }, true, mAddResult);
                        } else if (btType == AnnotMenu.AM_BT_REDACT) {
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_LongPress_Redaction);
                            redactCurrentSelection(uiExtensionsManager, page);
                        } else {
                            for (int i = 0; i < mExtMenuItems.size(); i ++) {
                                if (mExtMenuItems.get(i).getMenuItem().getType() == btType) {
                                    if (btType == 201) {
                                        showAiSubMenuItem();
                                        return;
                                    }
                                    RectF bbox = mSelectInfo.getBbox();
                                    PointF point = new PointF(bbox.right, bbox.bottom);
                                    PointF p = new PointF(point.x, point.y);
                                    mPdfViewCtrl.convertPdfPtToPageViewPt(p, p, mCurrentIndex);
                                    if (mExtMenuItems.get(i).getSelectListener() != null) {
                                        mExtMenuItems.get(i).getSelectListener().onSelected(mExtMenuItems.get(i).getMenuItem(), mCurrentIndex, p);
                                    }

                                    RectF rectF = new RectF(mSelectInfo.getBbox());
                                    mSelectInfo.clear();
                                    mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, mCurrentIndex);
                                    mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectF, rectF, mCurrentIndex);
                                    RectF rF = AppUtil.calculateRect(rectF, mTmpRect);
                                    Rect rect = new Rect();
                                    rF.roundOut(rect);
                                    getInvalidateRect(rect);
                                    mPdfViewCtrl.invalidate(rect);
                                }

                            }

                            for (int i = 0; i < mAiSubItems.size(); i ++) {
                                if (mAiSubItems.get(i).getMenuItem().getType() == btType) {
                                    RectF bbox = mSelectInfo.getBbox();
                                    PointF point = new PointF(bbox.right, bbox.bottom);
                                    PointF p = new PointF(point.x, point.y);
                                    mPdfViewCtrl.convertPdfPtToPageViewPt(p, p, mCurrentIndex);
                                    if (mAiSubItems.get(i).getSelectListener() != null) {
                                        mAiSubItems.get(i).getSelectListener().onSelected(mAiSubItems.get(i).getMenuItem(), mCurrentIndex, p);
                                    }

                                    RectF rectF = new RectF(mSelectInfo.getBbox());
                                    mSelectInfo.clear();
                                    mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, mCurrentIndex);
                                    mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectF, rectF, mCurrentIndex);
                                    RectF rF = AppUtil.calculateRect(rectF, mTmpRect);
                                    Rect rect = new Rect();
                                    rF.roundOut(rect);
                                    getInvalidateRect(rect);
                                    mPdfViewCtrl.invalidate(rect);
                                }
                            }
                        }
                        mIsEdit = false;
                        mAnnotationMenu.dismiss();
                    }
                });
            }
        } catch (PDFException exception) {
            if (exception.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
                return true;
            }
        }
        return true;
    }

    private void redactCurrentSelection(UIExtensionsManager uiExtensionsManager, final PDFPage page) {
        if (AppAnnotUtil.hasModuleLicenseRight(Constants.e_ModuleNameRedaction)) {
            uiExtensionsManager.getDocumentManager().addAnnot(page, new TextMarkupContentAbs() {
                @Override
                public TextSelector getTextSelector() {
                    mSelectInfo.setContents(mSelectInfo.getText(page));
                    return mSelectInfo;
                }

                @Override
                public int getPageIndex() {
                    return mCurrentIndex;
                }

                @Override
                public int getType() {
                    return Annot.e_Redact;
                }

                @Override
                public int getFillColor() {
                    return 0;
                }

                @Override
                public String getIntent() {
                    return null;
                }

                @Override
                public String getAuthor() {
                    return null;
                }

            }, true, mAddResult);
        } else {
            String message = AppUtil.getMessage(mContext, Constants.e_ErrNoRedactionModuleRight);
            UIToast.getInstance(mContext).show(message);
        }
    }

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent) {
        if (mIsEdit) {
            RectF rectF = new RectF(mSelectInfo.getBbox());
            mSelectInfo.clear();
            mPdfViewCtrl.convertPdfRectToPageViewRect(mSelectInfo.getBbox(), rectF, mCurrentIndex);
            mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectF, rectF, mCurrentIndex);
            RectF rF = AppUtil.calculateRect(rectF, mTmpRect);
            Rect rect = new Rect();
            rF.roundOut(rect);
            getInvalidateRect(rect);
            mPdfViewCtrl.invalidate(rect);
            mIsEdit = false;
            mAnnotationMenu.dismiss();

            return true;
        } else {
            return false;
        }
    }

    @Override
    public boolean isContinueAddAnnot() {
        return false;
    }

    @Override
    public void setContinueAddAnnot(boolean continueAddAnnot) {
    }

    private Event.Callback mAddResult = new Event.Callback() {
        @Override
        public void result(Event event, boolean success) {
            mSelectInfo.clear();
        }
    };

    protected TextSelector getSelectInfo() {
        return mSelectInfo;
    }

    @Override
    public void onDraw(int pageIndex, Canvas canvas) {
        if (mCurrentIndex != pageIndex) return;
        if (mSelectInfo != null) {
            mPaint.setColor(((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getSelectionHighlightColor());
            Rect clipRect = canvas.getClipBounds();
            for (RectF rect : mSelectInfo.getRectFList()) {
                RectF tmp = new RectF(rect);
                mPdfViewCtrl.convertPdfRectToPageViewRect(rect, tmp, mCurrentIndex);
                Rect r = new Rect();
                tmp.round(r);
                if (r.intersect(clipRect)) {
                    canvas.save();
                    canvas.drawRect(r, mPaint);
                    canvas.restore();
                }
            }
            if (mSelectInfo.getRectFList().size() > 0) {
                RectF start = new RectF(mSelectInfo.getRectFList().get(0));
                RectF end = new RectF(mSelectInfo.getRectFList().get(mSelectInfo.getRectFList().size() - 1));

                mPdfViewCtrl.convertPdfRectToPageViewRect(start, start, mCurrentIndex);
                mPdfViewCtrl.convertPdfRectToPageViewRect(end, end, mCurrentIndex);

                canvas.drawBitmap(mHandlerBitmap, start.left - mHandlerBitmap.getWidth(), start.top - mHandlerBitmap.getHeight(), null);
                canvas.drawBitmap(mHandlerBitmap, end.right, end.bottom, null);

                mPaint.setARGB(255, 76, 121, 164);
                canvas.drawLine(start.left, start.top - 1, start.left, start.bottom + 1, mPaint);
                canvas.drawLine(end.right, end.top - 1, end.right, end.bottom + 1, mPaint);
            }
        }

    }

    private PointF mLastDownPoint = new PointF();
    private int mSelectStartIndex;
    private int mSelectEndIndex;

    private boolean handleOnTouchEvent(int pageIndex, MotionEvent motionEvent) {
        PointF devPt = new PointF(motionEvent.getX(), motionEvent.getY());
        PointF point = new PointF();
        mPdfViewCtrl.convertDisplayViewPtToPageViewPt(devPt, point, pageIndex);
        int action = motionEvent.getActionMasked();
        switch (action) {
            case MotionEvent.ACTION_DOWN:
                if (mLastDownPoint == null)
                    mLastDownPoint = new PointF();
                mLastDownPoint.set(point);
                mCtrlPoint = getControlPoint(mCurrentIndex, point);
                if (mCtrlPoint != 0) {
                    mSelectStartIndex = mSelectInfo.getStart();
                    mSelectEndIndex = mSelectInfo.getEnd();
                    if (mMagnifierView != null){
                        mMagnifierView.onTouchEvent(motionEvent);
                        mMagnifierView.setVisibility(View.VISIBLE);
                    }
                    return true;
                }
                return false;
            case MotionEvent.ACTION_MOVE:
                if (AppDmUtil.distanceOfTwoPoints(mLastDownPoint,point) <= 10)
                    break;
                mAnnotationMenu.dismiss();
                if (mCtrlPoint != 0) {
                    mNoFirstRedact = true;
                    onSelectMove(pageIndex, point, mSelectInfo);
                    invalidateTouch(mCurrentIndex, mSelectInfo);
                    if (mMagnifierView != null){
                        mMagnifierView.onTouchEvent(motionEvent);
                    }
                    return true;
                }
                return false;
            case MotionEvent.ACTION_CANCEL:
            case MotionEvent.ACTION_UP:
                if (mMagnifierView != null){
                    mMagnifierView.setVisibility(View.GONE);
                }
                if (mIsEdit) {
                    mText.clear();
                    reloadMenu();
                    if (mText.size() == 0 && mExtMenuItems.size() == 0) return false;
                    if (mText.size() > 0) {
                        mAnnotationMenu.setMenuItems(mText);
                    }
                    if (mExtMenuItems.size() > 0) {
                        ArrayList<MenuItem> menuItems = new ArrayList<>();
                        for (int i = 0; i < mExtMenuItems.size(); i ++) {
                            menuItems.add(mExtMenuItems.get(i).getMenuItem());
                        }
                        mAnnotationMenu.setExtensionMenuItems(menuItems);
                    }
                    mMenuBox = new RectF();
                    mPdfViewCtrl.convertPdfRectToPageViewRect(mSelectInfo.getBbox(), mMenuBox, mCurrentIndex);
                    mMenuBox.inset(-10, -10);
                    mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mMenuBox, mMenuBox, mCurrentIndex);
                    UIExtensionsManager extensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
                    if (noSelectRedact(extensionsManager)) {
                        mAnnotationMenu.show(mMenuBox);
                    } else if(mNoFirstRedact) {
                        UIExtensionsManager manager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
                        redactCurrentSelection(manager, manager.getDocumentManager().getPage(mCurrentIndex, false));
                        mNoFirstRedact = false;
                    }
                    return false;
                }
                break;
            default:
                break;
        }

        return false;
    }

    private void invalidateTouch(int pageIndex, TextSelector selectInfo) {
        if (selectInfo == null) return;
        RectF rectF = new RectF();
        mPdfViewCtrl.convertPdfRectToPageViewRect(mSelectInfo.getBbox(), rectF, pageIndex);
        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectF, rectF, pageIndex);
        RectF rF = AppUtil.calculateRect(rectF, mTmpRect);
        Rect rect = new Rect();
        rF.roundOut(rect);
        getInvalidateRect(rect);
        mPdfViewCtrl.invalidate(rect);
        mTmpRect.set(rectF);
    }

    private int isControlPoint(int pageIndex, PointF point) {
        if (mSelectInfo != null && mSelectInfo.getRectFList().size() > 0) {
            RectF mStart = new RectF(mSelectInfo.getRectFList().get(0));
            RectF mEnd = new RectF(mSelectInfo.getRectFList().get(mSelectInfo.getRectFList().size() - 1));
            mPdfViewCtrl.convertPdfRectToPageViewRect(mStart, mStart, pageIndex);
            mPdfViewCtrl.convertPdfRectToPageViewRect(mEnd, mEnd, pageIndex);

            RectF startHandler = new RectF(mStart.left - mHandlerBitmap.getWidth(), mStart.top - mHandlerBitmap.getHeight(),
                    mStart.left, mStart.top);
            RectF endHandler = new RectF(mEnd.right, mEnd.bottom,
                    mEnd.right + mHandlerBitmap.getWidth(), mEnd.bottom + mHandlerBitmap.getHeight());
            startHandler.inset(-HANDLE_AREA, -HANDLE_AREA);
            endHandler.inset(-HANDLE_AREA, -HANDLE_AREA);
            if (startHandler.contains(point.x, point.y))
                return 1;
            if (endHandler.contains(point.x, point.y))
                return 2;
        }
        return 0;

    }

    private boolean onSelectMove(int pageIndex, PointF point, TextSelector selectInfo) {
        if (selectInfo == null || mCurrentIndex != pageIndex) return false;
        try {
            PDFPage page = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getPage(mCurrentIndex, false);
            if (page == null) return false;
            TextPage textPage = new TextPage(page, TextPage.e_ParseTextNormal);
            mPdfViewCtrl.convertPageViewPtToPdfPt(point, point, mCurrentIndex);
            int index = textPage.getIndexAtPos(point.x, point.y, FINGER_AREA);
            if (index < 0) return false;
            if (mCtrlPoint == 1) {
                if (index <= mSelectEndIndex) {
                    selectInfo.setStart(index);
                    selectInfo.setEnd(mSelectEndIndex);
                } else {
                    selectInfo.setStart(mSelectEndIndex);
                    selectInfo.setEnd(index);
                }
            } else if (mCtrlPoint == 2) {
                if (index >= mSelectStartIndex) {
                    selectInfo.setStart(mSelectStartIndex);
                    selectInfo.setEnd(index);
                } else {
                    selectInfo.setStart(index);
                    selectInfo.setEnd(mSelectStartIndex);
                }
            }
            selectInfo.computeSelected(page, selectInfo.getStart(), selectInfo.getEnd());
        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
                return false;
            }
        }
        return true;
    }

    private int getControlPoint(int pageIndex,PointF point){
        if (mSelectInfo != null && mSelectInfo.getRectFList().size() > 0) {
            RectF start = new RectF(mSelectInfo.getRectFList().get(0));
            RectF end = new RectF(mSelectInfo.getRectFList().get(mSelectInfo.getRectFList().size() - 1));
            mPdfViewCtrl.convertPdfRectToPageViewRect(start, start, pageIndex);
            mPdfViewCtrl.convertPdfRectToPageViewRect(end, end, pageIndex);

            RectF startHandler = new RectF(start.left - mHandlerBitmap.getWidth(), start.top - mHandlerBitmap.getHeight(),
                    start.left, start.top);
            RectF endHandler = new RectF(end.right, end.bottom,
                    end.right + mHandlerBitmap.getWidth(), end.bottom + mHandlerBitmap.getHeight());
            startHandler.inset(-FINGER_AREA, -FINGER_AREA);
            endHandler.inset(-FINGER_AREA, -FINGER_AREA);

            start.set(start.left, start.top, start.left, start.bottom);
            end.set(end.right, end.top, end.right, end.bottom);
            start.inset(-FINGER_AREA, -FINGER_AREA);
            end.inset(-FINGER_AREA, -FINGER_AREA);

            if (start.contains(point.x, point.y) || startHandler.contains(point.x,point.y))
                return 1;
            if (end.contains(point.x, point.y) || endHandler.contains(point.x,point.y))
                return 2;
        }
        return 0;
    }

    public void dismissMenu() {
        if (mIsEdit) {
            RectF rectF = new RectF(mSelectInfo.getBbox());
            mSelectInfo.clear();
            int pageIndex = mPdfViewCtrl.getCurrentPage();
            if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, pageIndex);
                RectF rF = AppUtil.calculateRect(rectF, mTmpRect);
                Rect rect = new Rect();
                rF.roundOut(rect);
                getInvalidateRect(rect);
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectF, rectF, pageIndex);
                mPdfViewCtrl.invalidate(rect);
            }
            mIsEdit = false;
            mAnnotationMenu.dismiss();
        }
    }

    public void getInvalidateRect(Rect rect) {
        rect.top -= mHandlerBitmap.getHeight();
        rect.bottom += mHandlerBitmap.getHeight();
        rect.left -= mHandlerBitmap.getWidth() / 2;
        rect.right += mHandlerBitmap.getWidth() / 2;
        rect.inset(-20, -20);
    }

    public void onDrawForAnnotMenu(Canvas canvas) {
        if (!mPdfViewCtrl.isPageVisible(mCurrentIndex)) {
            return;
        }
        if (!mIsEdit) {
            return;
        }

        RectF bboxRect = new RectF(mSelectInfo.getBbox());
        mPdfViewCtrl.convertPdfRectToPageViewRect(bboxRect, bboxRect, mCurrentIndex);
        bboxRect.inset(-10, -10);

        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(bboxRect, bboxRect, mCurrentIndex);
        mAnnotationMenu.update(bboxRect);
    }

    protected boolean onKeyBack() {
        if (mIsEdit) {
            mIsEdit = false;
            mAnnotationMenu.dismiss();

            RectF rectF = new RectF(mSelectInfo.getBbox());
            mSelectInfo.clear();
            if (!mPdfViewCtrl.isPageVisible(mCurrentIndex))
                return true;
            mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, mCurrentIndex);
            RectF rF = AppUtil.calculateRect(rectF, mTmpRect);
            Rect rect = new Rect();
            rF.roundOut(rect);
            getInvalidateRect(rect);
            mPdfViewCtrl.invalidate(rect);

            return true;
        }
        return false;
    }

    private ArrayList<SelectExtentsionMenuItem> mAiSubItems = new ArrayList<>();
    public void addAiSubMenuIItem(ArrayList<SelectExtentsionMenuItem> list) {
        mAiSubItems.clear();
        for (SelectExtentsionMenuItem seItem : list) {
            mAiSubItems.add(seItem);
        }
    }

    public void showAiSubMenuItem() {
        mAnnotationMenu.setMenuItems(new ArrayList<Integer>());
        ArrayList<MenuItem> menuList = new ArrayList<>();
        for (SelectExtentsionMenuItem sItem : mAiSubItems) {
            menuList.add(sItem.getMenuItem());
        }
        mAnnotationMenu.setExtensionMenuItems(menuList);
        mAnnotationMenu.dismiss();
        mAnnotationMenu.show(mMenuBox);
    }

    public void addExtensionMenuItem(SelectExtentsionMenuItem menuItem) {
        if (!mExtMenuItems.contains(menuItem)) {
            mExtMenuItems.add(menuItem);
        }
    }

    public void removeExtensionMenuItem(SelectExtentsionMenuItem menuItem) {
        mExtMenuItems.remove(menuItem);
    }

    private void initMagnifierView() {
        MainFrame mainFrame = (MainFrame) ((UIExtensionsManager)mPdfViewCtrl.getUIExtensionsManager()).getMainFrame();
        if (mMagnifierView == null){
            mMagnifierView = new UIMagnifierView(mContext.getApplicationContext());
        }
        mMagnifierView.setTargetView(mPdfViewCtrl);
        mMagnifierView.setVisibility(View.GONE);
        mainFrame.getContentView().addView(mMagnifierView);
    }

    private void removeMagnifierView() {
        if (mMagnifierView != null){
            MainFrame mainFrame = (MainFrame) ((UIExtensionsManager)mPdfViewCtrl.getUIExtensionsManager()).getMainFrame();
            mainFrame.getContentView().removeView(mMagnifierView);
            mMagnifierView.setTargetView(null);
        }
    }
}
