/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.snapshot;

import android.content.Context;
import android.graphics.Bitmap;
import android.net.Uri;

import androidx.annotation.NonNull;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppStorageManager;
import com.foxit.uiextensions.utils.AppUtil;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Calendar;

/**  Take snapshot on the view control..*/
public class SnapshotPresenter implements SnapshotContract.Presenter {

    private SnapshotContract.View view;
    private Context mContext;

    private final static String PATH =  AppFileUtil.getSDPath() + "/FoxitSDK/Images";
    private static String SAVE_PATH = "";

    public SnapshotPresenter(Context context, @NonNull SnapshotContract.View view) {
        this.view = AppUtil.requireNonNull(view);
        this.view.setPresenter(this);
        this.mContext = context;
    }

    @Override
    public void save() {
        Bitmap result = view.getBitmap();
        FileOutputStream fo = null;
        String path = generateFilePath();

        if (!AppFileUtil.needScopedStorageAdaptation()){
            try {
                mkdir();
                File file = new File(path);
                file.createNewFile();
                fo = new FileOutputStream(file);
                result.compress(Bitmap.CompressFormat.PNG, 100, fo);
            } catch (FileNotFoundException fnfe) {
                fnfe.printStackTrace();
                view.showToast(mContext.getApplicationContext().getString(R.string.failed_to_save_snapshot));
                view.dismiss();
                throw new RuntimeException(mContext.getApplicationContext().getString(R.string.failed_to_save_snapshot));
            } catch (IOException ioe) {
                ioe.printStackTrace();
                view.showToast(mContext.getApplicationContext().getString(R.string.failed_to_save_snapshot));
                view.dismiss();
                throw new RuntimeException(mContext.getApplicationContext().getString(R.string.failed_to_save_snapshot));
            } catch (Exception e){
                e.printStackTrace();
                view.showToast(mContext.getApplicationContext().getString(R.string.failed_to_save_snapshot));
                view.dismiss();
                throw new RuntimeException(mContext.getApplicationContext().getString(R.string.failed_to_save_snapshot));
            } finally {
                if (fo != null)
                    try {
                        fo.flush();
                        fo.close();
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
            }
        }else {
            try {
                Uri uri = AppFileUtil.toDocumentUriFromPath(path);
                AppFileUtil.createNewDocument(uri,false);
                fo = new FileOutputStream(mContext.getContentResolver().openFileDescriptor(uri, "w").getFileDescriptor());
                result.compress(Bitmap.CompressFormat.PNG, 100, fo);
            } catch (Exception e) {
                e.printStackTrace();
                view.showToast(mContext.getApplicationContext().getString(R.string.failed_to_save_snapshot));
                view.dismiss();
                return;
            }finally {
                if (fo != null)
                    try {
                        fo.flush();
                        fo.close();
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
            }
        }
        view.showToast(mContext.getApplicationContext().getString(R.string.menu_more_screen_saved_toast_content, path));
        view.dismiss();
    }

    public static void setSavePath(String path) {
        SAVE_PATH = path;
    }

    private String getSavePath() {
        if (!AppUtil.isEmpty(SAVE_PATH)) {
            return SAVE_PATH;
        }
        if (AppFileUtil.needScopedStorageAdaptation()){
            return AppStorageManager.getInstance(mContext).getDefaultFolder() + "/Images";
        }
        return PATH;
    }

    private String generateFilePath(){
        Calendar calendar = Calendar.getInstance();
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyMMddHHmmss");

        StringBuffer sb = new StringBuffer();
        sb.append(getSavePath() + "/"+"snapshot-");
        sb.append(simpleDateFormat.format(calendar.getTime()));
        sb.append(".png");

        return sb.toString();
    }

    private boolean mkdir(){
        File file = new File(getSavePath());
        if (!file.exists()){
            return file.mkdirs();
        }
        return false;
    }
}

