/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.signature.appearance;


import android.app.Activity;
import android.content.Context;
import android.content.DialogInterface;
import android.content.pm.ActivityInfo;
import android.graphics.Bitmap;
import android.graphics.BitmapShader;
import android.graphics.Canvas;
import android.graphics.ComposeShader;
import android.graphics.Paint;
import android.graphics.PorterDuff;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.Shader;
import android.text.Editable;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewTreeObserver;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.Toast;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.controls.dialog.UIDialog;
import com.foxit.uiextensions.controls.dialog.UIMatchDialog;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.modules.signature.SignatureConstants;
import com.foxit.uiextensions.modules.signature.SignatureDataUtil;
import com.foxit.uiextensions.modules.signature.SignatureInkItem;
import com.foxit.uiextensions.security.ICertificateSupport;
import com.foxit.uiextensions.security.certificate.CertificateFileInfo;
import com.foxit.uiextensions.security.digitalsignature.DigitalSignatureModule;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;

import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

public class SignatureAppearanceDialog extends UIMatchDialog {
    private Context mContext;
    private ViewGroup mParent;
    private PDFViewCtrl mPdfViewCtrl;
    private UIExtensionsManager mUIExtensionsManager;
    private ViewGroup mContentView;
    private LinearLayout mSignLl;
    private ImageView mSignBitmapIv;

    private LinearLayout mTitleStyleLl;
    private TextView mTitleStyleTv;
    private ImageView mTitleStyleNextIv;
    private LinearLayout mLocationLl;
    private TextView mLocationTv;
    private LinearLayout mReasonLl;
    private TextView mReasonTv;
    private TextView mCreateTypeTv;
    private View mViewFill;
    private View mViewFillBottom;
    private RecyclerView mOptionalRecyclerView;
    private TextView mOptionalGroupTv;
    private TextView mOptionalDoneTv;
    private ISignAppearanceCallback mSignAppearanceCallback;
    private SignatureOptionalAdapter mOptionalAdapter;
    private LinearLayoutManager mLayoutManager;
    private boolean mIsCreateTitle = false;
    private SignatureAppearanceTitleDialog mSignAppearanceTitleDialog;
    private SignatureAppearanceLocationDialog mSignAppearanceLocationDialog;
    private SignatureAppearanceReasonDialog mSignAppearanceReasonDialog;
    private UITextEditDialog mCreateTitleDialog;
    private List<SignatureOptionalBean> mOptionalList = new ArrayList<SignatureOptionalBean>();
    private SignatureInkItem mSignInkItem;
    private SignatureInkItem mCopySignInkItem;
    private SignatureInkItem mCreateNewTitleInkItem;
    private ICertificateSupport.CertificateInfo mInfo;

    private Bitmap mBitmap;
    private String mEditedTitle;

    public SignatureAppearanceDialog(Context context,
                                     ViewGroup parent,
                                     PDFViewCtrl pdfViewCtrl,
                                     SignatureInkItem inkItem
    ) {
        super(context);
        mContext = context.getApplicationContext();
        mParent = parent;
        mPdfViewCtrl = pdfViewCtrl;
        mUIExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        mSignInkItem = inkItem;

        if (!AppDisplay.isPad()) {
            mUIExtensionsManager.getAttachedActivity().setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_SENSOR_PORTRAIT);
        }
        initStyle();
        initView();

        setOnDLDismissListener(new DismissListener() {
            @Override
            public void onDismiss() {
                mInfo = null;
                mUIExtensionsManager.unregisterThemeEventListener(mThemeColorChangedListener);
            }
        });
    }

    @Override
    public void onBackPressed() {
        mIsBackDismiss = false;
        cancelUseCurrent();
    }

    private void initStyle() {
        setBackButtonStyle(TEXT_BACK);
        setBackButtonText(AppResource.getString(mContext, R.string.fx_string_cancel));
        setTitle(AppResource.getString(mContext, R.string.appearance_preview));
        setTitlePosition(BaseBar.TB_Position.Position_CENTER);
        setRightButtonVisible(View.VISIBLE);
        setRightButtonText(AppResource.getString(mContext, R.string.fx_string_save));
        setStyle(DLG_TITLE_STYLE_BG_DEFAULT);
        setListener(new DialogListener() {
            @Override
            public void onResult(long btType) {
            }

            @Override
            public void onBackClick() {
                if (mIsCreateTitle) {
                    cancelCreateTitle();
                } else {
                    mIsBackDismiss = false;
                    cancelUseCurrent();
                }
            }

            @Override
            public void onTitleRightButtonClick() {
                saveSign();
            }
        });
    }

    private void initView() {
        mContentView = (ViewGroup) View.inflate(mContext, R.layout.sign_appearance_layout, null);
        mSignLl = mContentView.findViewById(R.id.sign_appearance_signature_ll);
        mSignBitmapIv = mContentView.findViewById(R.id.sign_appearance_signature_bitmap);
        mTitleStyleLl = mContentView.findViewById(R.id.sign_appearance_title_style_ll);
        mTitleStyleTv = mContentView.findViewById(R.id.sign_appearance_title_style);
        mTitleStyleNextIv = mContentView.findViewById(R.id.sign_appearance_title_style_next);
        mLocationLl = mContentView.findViewById(R.id.sign_appearance_location_ll);
        mLocationTv = mContentView.findViewById(R.id.sign_appearance_location);
        mReasonLl = mContentView.findViewById(R.id.sign_appearance_reason_ll);
        mReasonTv = mContentView.findViewById(R.id.sign_appearance_reason);
        mCreateTypeTv = mContentView.findViewById(R.id.sign_appearance_create_type);
        mViewFill = mContentView.findViewById(R.id.sign_appearance_view_fill);
        mViewFillBottom = mContentView.findViewById(R.id.sign_appearance_view_fill_bottom);
        mOptionalGroupTv = mContentView.findViewById(R.id.sign_appearance_optional_group);
        mOptionalRecyclerView = mContentView.findViewById(R.id.sign_appearance_optional_rv);
        mOptionalAdapter = new SignatureOptionalAdapter(mContext, mPdfViewCtrl);
        mOptionalRecyclerView.setAdapter(mOptionalAdapter);
        mOptionalAdapter.setOnItemClickCallback(onItemClickCallback);
        mLayoutManager = new LinearLayoutManager(mContext, LinearLayoutManager.VERTICAL, false);
        mOptionalRecyclerView.setLayoutManager(mLayoutManager);
        mOptionalDoneTv = mContentView.findViewById(R.id.sign_appearance_done);
        setContentView(mContentView);
        mTitleStyleLl.setOnClickListener(mOnClickListener);
        mLocationLl.setOnClickListener(mOnClickListener);
        mReasonLl.setOnClickListener(mOnClickListener);
        mCreateTypeTv.setOnClickListener(mOnClickListener);
        mOptionalDoneTv.setOnClickListener(mOnClickListener);
    }

    public void loadData() {
        mCopySignInkItem = mSignInkItem.clone();
        mUIExtensionsManager.registerThemeEventListener(mThemeColorChangedListener);
        final DigitalSignatureModule dsgModule = (DigitalSignatureModule) mUIExtensionsManager.getModuleByName(Module.MODULE_NAME_DIGITALSIGNATURE);
        if (dsgModule != null) {
            CertificateFileInfo fileInfo = dsgModule.getDigitalSignatureUtil().getCertInfo(mCopySignInkItem.getDsgPath());
            ICertificateSupport.CertificateInfo info = dsgModule.getDigitalSignatureUtil().getCertSupport().verifyPwd(fileInfo.filePath, fileInfo.password);
            mInfo = info;
        }
        String publisher = mInfo.publisher == null ? "":mInfo.publisher;

        if (!SignatureDataUtil.getDataCount(mContext, SignatureConstants.getTitleTableName())) {
            SignatureInkItem standardTitleItem = new SignatureInkItem();
            standardTitleItem.setTitle("appearance_standard_style");
            standardTitleItem.setNameOpen(1);
            standardTitleItem.setName(publisher);
            standardTitleItem.setDistinguishedName(getDn());
            standardTitleItem.setDistinguishedNameOpen(1);
            standardTitleItem.setReasonOpen(1);
            standardTitleItem.setReason(0);
            standardTitleItem.setLocationOpen(1);
            standardTitleItem.setLocation("");
            standardTitleItem.setDateOpen(1);
            standardTitleItem.setVersionOpen(1);
            standardTitleItem.setLabels(0);
            if (mUIExtensionsManager.getAPPInfoProvider() != null) {
                standardTitleItem.setVersion(mUIExtensionsManager.getAPPInfoProvider().getAppVersion());
            }
            SignatureDataUtil.insertTitleData(mContext, standardTitleItem);
        }
        SignatureInkItem titleInkItem;
        //create
        if (mSignInkItem.getKey() == null) {
            String recentTitle = SignatureDataUtil.getRecentTitleKey(mContext);
            String standardStyle = "appearance_standard_style";
            if (TextUtils.isEmpty(recentTitle)) {
                recentTitle = standardStyle;
            }
            if (recentTitle.equals("appearance_standard_style")) {
                mTitleStyleTv.setText(mContext.getApplicationContext().getString(R.string.appearance_standard_style));
            } else {
                mTitleStyleTv.setText(recentTitle);
            }
            titleInkItem = SignatureDataUtil.getTitleByKey(mContext, recentTitle);

            setReasonAndLocation(titleInkItem);
            setReasonAndLocationText(titleInkItem);
        } else {
            //edit
            if (TextUtils.isEmpty(mSignInkItem.getTitle())) {
                String recentTitle = SignatureDataUtil.getRecentTitleKey(mContext);
                String standardStyle = "appearance_standard_style";
                if (TextUtils.isEmpty(recentTitle)) {
                    recentTitle = standardStyle;
                }
                if (recentTitle.equals("appearance_standard_style")) {
                    mTitleStyleTv.setText(mContext.getApplicationContext().getString(R.string.appearance_standard_style));
                } else {
                    mTitleStyleTv.setText(recentTitle);
                }
                titleInkItem = SignatureDataUtil.getTitleByKey(mContext, recentTitle);
                setReasonAndLocation(titleInkItem);
                setReasonAndLocationText(titleInkItem);
            } else {
                String standardStyle = "appearance_standard_style";
                if (standardStyle.equals(mCopySignInkItem.getTitle())) {
                    mTitleStyleTv.setText(mContext.getApplicationContext().getString(R.string.appearance_standard_style));
                } else {
                    mEditedTitle = mCopySignInkItem.getTitle();
                    mTitleStyleTv.setText(mCopySignInkItem.getTitle());
                }
                titleInkItem = SignatureDataUtil.getTitleByKey(mContext, mCopySignInkItem.getTitle());
                setReasonAndLocationText(mCopySignInkItem);
            }
        }
        setTitleStyle(titleInkItem);
        setSignDynamicInfo(titleInkItem);

        Bitmap bitmap = mCopySignInkItem.getBitmap();
        Rect bitmapRect = mCopySignInkItem.getRect();
        int t = bitmapRect.top;
        int b = bitmapRect.bottom;
        int l = bitmapRect.left;
        int r = bitmapRect.right;
        int[] pixels = new int[bitmapRect.width() * bitmapRect.height()];
        bitmap.getPixels(pixels, 0, r - l, l, t, r - l, b - t);
//        for (int i = 0; i < pixels.length; i++) {
//            if (0xFFFFFFFF == pixels[i]) {
//                pixels[i] = 0x0;
//            }
//        }
        mBitmap = Bitmap.createBitmap(pixels, bitmapRect.width(), bitmapRect.height(), Bitmap.Config.ARGB_8888);
        mSignLl.getViewTreeObserver().addOnGlobalLayoutListener(new ViewTreeObserver.OnGlobalLayoutListener() {
            @Override
            public void onGlobalLayout() {
                renderSignature(mCopySignInkItem);
                mSignLl.getViewTreeObserver().removeOnGlobalLayoutListener(this);
            }
        });

    }

    private String getDn() {
        if (mInfo != null) {
            String issuer = mInfo.issuer;
            if (TextUtils.isEmpty(issuer)) {
                issuer = "\"\"";
            }
            String publisher = mInfo.publisher;
            if (TextUtils.isEmpty(publisher)) {
                publisher = "\"\"";
            }
            String emailAddress = mInfo.emailAddress;
            if (TextUtils.isEmpty(emailAddress)) {
                emailAddress = "\"\"";
            }
            return "CN=" + publisher + ",OU=" + issuer + ",E=" + emailAddress;
        } else {
            return "";
        }
    }

    private void setSignDynamicInfo(SignatureInkItem inkItem) {
        if (inkItem.isNameOpen()) {
            String publisher = mInfo.publisher == null ? "":mInfo.publisher;
            mCopySignInkItem.setName(publisher);
        }
        if (inkItem.isVersionOpen() && mUIExtensionsManager.getAPPInfoProvider() != null) {
            mCopySignInkItem.setVersion(mUIExtensionsManager.getAPPInfoProvider().getAppVersion());
        }
    }

    private void setTitleStyle(SignatureInkItem inkItem) {
        mCopySignInkItem.setTitle(inkItem.getTitle());
        mCopySignInkItem.setNameOpen(inkItem.isNameOpen() ? 1 : 0);
        mCopySignInkItem.setDistinguishedNameOpen(inkItem.isDistinguishedNameOpen() ? 1 : 0);
        if (inkItem.isDistinguishedNameOpen()) {
            mCopySignInkItem.setDistinguishedName(getDn());
        }
        mCopySignInkItem.setDateOpen(inkItem.isDateOpen() ? 1 : 0);
        mCopySignInkItem.setVersionOpen(inkItem.isVersionOpen() ? 1 : 0);
        mCopySignInkItem.setLogo(inkItem.isLogo() ? 1 : 0);
        mCopySignInkItem.setLabels(inkItem.isLabels() ? 1 : 0);
    }

    private void setReasonAndLocation(SignatureInkItem inkItem) {
        mCopySignInkItem.setReasonOpen(inkItem.isReasonOpen() ? 1 : 0);
        if (inkItem.isReasonOpen()) {
            mCopySignInkItem.setReason(inkItem.getReason());
        } else {
            mCopySignInkItem.setReason(0);
        }
        mCopySignInkItem.setLocationOpen(inkItem.isLocationOpen() ? 1 : 0);
        if (inkItem.isLocationOpen()) {
            if (!TextUtils.isEmpty(inkItem.getLocation())) {
                mCopySignInkItem.setLocation(inkItem.getLocation());
            } else {
                mCopySignInkItem.setLocation("");
            }
        } else {
            mCopySignInkItem.setLocation("");
        }
    }

    private void setReasonAndLocationText(SignatureInkItem inkItem) {
        if (inkItem.isReasonOpen()) {
            String  textReason = getReason(inkItem.getReason());
            mReasonTv.setText(textReason);
        } else {
            mReasonTv.setText(mContext.getApplicationContext().getString(R.string.fx_string_close));
        }
        if (inkItem.isLocationOpen()) {
            if (!TextUtils.isEmpty(inkItem.getLocation())) {
                mLocationTv.setText(inkItem.getLocation());
            } else {
                mLocationTv.setText("");
            }
        } else {
            mLocationTv.setText(mContext.getApplicationContext().getString(R.string.fx_string_close));
        }
    }

    private void renderSignature(SignatureInkItem inkItem) {
        final DigitalSignatureModule dsgModule = (DigitalSignatureModule) mUIExtensionsManager.getModuleByName(Module.MODULE_NAME_DIGITALSIGNATURE);
        if (dsgModule != null && inkItem != null && !TextUtils.isEmpty(inkItem.getTitle())) {
            Rect rect = new Rect();
            mSignLl.getDrawingRect(rect);
            RectF desRect = new RectF(rect);
            Bitmap tempBitmap = Bitmap.createBitmap((int) desRect.width(), (int) desRect.height(), Bitmap.Config.ARGB_8888);
            RectF pageRect = new RectF();
            pageRect.set(0, 0, AppDisplay.px2dp(rect.width()), AppDisplay.px2dp(rect.height()));
            dsgModule.getDigitalSignatureUtil().createSignature(pageRect, inkItem, mBitmap, tempBitmap, true);
            mSignBitmapIv.setImageBitmap(tempBitmap);
            tempBitmap = null;
        }
    }

    private IThemeEventListener mThemeColorChangedListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            dismiss();
        }
    };

    public void setSignCallback(ISignAppearanceCallback signAppearanceCallback) {
        mSignAppearanceCallback = signAppearanceCallback;
    }

    public interface ISignAppearanceCallback {
        void onDismiss(SignatureInkItem inkItem);

        void onSaveSign(SignatureInkItem inkItem);
    }

    private View.OnClickListener mOnClickListener = new View.OnClickListener() {

        @Override
        public void onClick(View v) {
            if (AppUtil.isFastDoubleClick()) return;
            int id = v.getId();
            if (id == R.id.sign_appearance_title_style_ll) {
                showAppearanceTitleDialog();
            } else if (id == R.id.sign_appearance_location_ll) {
                showAppearanceLocationDialog();
            } else if (id == R.id.sign_appearance_reason_ll) {
                showAppearanceReasonDialog();
            } else if (id == R.id.sign_appearance_create_type) {
                createTitle();
            } else if (id == R.id.sign_appearance_done) {
                doneTitle();
            }
        }
    };
    private SignatureOptionalAdapter.OnItemClickCallback onItemClickCallback = new SignatureOptionalAdapter.OnItemClickCallback() {
        @Override
        public void onItemClick(int position, SignatureOptionalBean item) {
            if (item.getName().equals(mContext.getResources().getString(R.string.appearance_optional_labels))) {
                if (item.isSelect()) {
                    mCreateNewTitleInkItem.setLabels(1);
                } else {
                    mCreateNewTitleInkItem.setLabels(0);
                }
            }
            optionalRender(item);
            renderSignature(mCreateNewTitleInkItem);
        }
    };

    private void optionalRender(SignatureOptionalBean item) {
        if (item.getName().equals(mContext.getResources().getString(R.string.appearance_optional_name))) {
            if (item.isSelect()) {
                mCreateNewTitleInkItem.setNameOpen(1);
                    String publisher = mInfo.publisher == null ? "":mInfo.publisher;
                    mCreateNewTitleInkItem.setName(publisher);

            } else {
                mCreateNewTitleInkItem.setNameOpen(0);

            }
        } else if (item.getName().equals(mContext.getResources().getString(R.string.appearance_optional_foxit_version))) {
            if (item.isSelect()) {
                mCreateNewTitleInkItem.setVersionOpen(1);
                if (mUIExtensionsManager.getAPPInfoProvider() != null) {
                    mCreateNewTitleInkItem.setVersion(mUIExtensionsManager.getAPPInfoProvider().getAppVersion());
                }
            } else {
                mCreateNewTitleInkItem.setVersionOpen(0);
            }
        } else if (item.getName().equals(mContext.getResources().getString(R.string.appearance_optional_dn))) {
            if (item.isSelect()) {
                mCreateNewTitleInkItem.setDistinguishedNameOpen(1);
                mCreateNewTitleInkItem.setDistinguishedName(getDn());

            } else {
                mCreateNewTitleInkItem.setDistinguishedNameOpen(0);

            }
        } else if (item.getName().equals(mContext.getResources().getString(R.string.appearance_optional_date))) {
            if (item.isSelect()) {
                mCreateNewTitleInkItem.setDateOpen(1);
            } else {
                mCreateNewTitleInkItem.setDateOpen(0);
            }
        } else if (item.getName().equals(mContext.getResources().getString(R.string.appearance_optional_logo))) {
            if (item.isSelect()) {
                mCreateNewTitleInkItem.setLogo(1);
            } else {
                mCreateNewTitleInkItem.setLogo(0);
            }
        }
    }

    private void createTitle() {
        Context context = mUIExtensionsManager.getAttachedActivity();
        if (context == null) return;
        mCreateNewTitleInkItem = new SignatureInkItem();
        mCreateNewTitleInkItem.setDsgPath(mCopySignInkItem.getDsgPath());
        mCreateTitleDialog = new UITextEditDialog(context, UIDialog.NO_INPUT);
        mCreateTitleDialog.setTitle(mContext.getApplicationContext().getString(R.string.appearance_type_create));
        mCreateTitleDialog.getPromptTextView().setText(mContext.getApplicationContext().getString(R.string.appearance_warn_type));
        final EditText editText = mCreateTitleDialog.getInputEditText();
        editText.setHint(mContext.getApplicationContext().getString(R.string.appearance_hint_type));
        mCreateTitleDialog.setEditVisible(true);
        editText.addTextChangedListener(new TextWatcher() {
            @Override
            public void beforeTextChanged(CharSequence charSequence, int i, int i1, int i2) {

            }

            @Override
            public void onTextChanged(CharSequence charSequence, int i, int i1, int i2) {
                if(TextUtils.isEmpty(editText.getText().toString().trim())){
                    mCreateTitleDialog.getOKButton().setEnabled(false);
                }else{
                    mCreateTitleDialog.getOKButton().setEnabled(true);
                }
            }

            @Override
            public void afterTextChanged(Editable editable) {
                if(TextUtils.isEmpty(editText.getText().toString())){
                    mCreateTitleDialog.getOKButton().setEnabled(false);
                }
            }
        });

        mCreateTitleDialog.getOKButton().setOnClickListener(new View.OnClickListener() {

            @Override
            public void onClick(View v) {
                if (AppUtil.isFastDoubleClick())
                    return;
                if (TextUtils.isEmpty(editText.getText().toString().trim())) {
                    return;
                }
                if (SignatureDataUtil.isExistKey(mContext, SignatureConstants.getTitleTableName(), SignatureConstants.SG_TITLE_NAME, editText.getText().toString())) {
                    Toast.makeText(mContext, mContext.getApplicationContext().getString(R.string.appearance_type_exists), Toast.LENGTH_SHORT).show();
                    return;
                }
                mIsCreateTitle = true;
                setTitle(AppResource.getString(mContext, R.string.appearance_type));
                mIsBackDismiss = true;
                setRightButtonEnable(false);
                mRightTextItem.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
                mCreateTypeTv.setVisibility(View.GONE);
                mViewFill.setVisibility(View.GONE);
                mViewFillBottom.setVisibility(View.VISIBLE);
                mTitleStyleNextIv.setVisibility(View.INVISIBLE);
                mOptionalGroupTv.setVisibility(View.VISIBLE);
                mOptionalRecyclerView.setVisibility(View.VISIBLE);
                mOptionalDoneTv.setVisibility(View.VISIBLE);
                mOptionalList.clear();
                mOptionalList.add(new SignatureOptionalBean(mContext.getResources().getString(R.string.appearance_optional_name), false));
                mOptionalList.add(new SignatureOptionalBean(mContext.getResources().getString(R.string.appearance_optional_foxit_version), false));
                mOptionalList.add(new SignatureOptionalBean(mContext.getResources().getString(R.string.appearance_optional_dn), false));
                mOptionalList.add(new SignatureOptionalBean(mContext.getResources().getString(R.string.appearance_optional_date), false));
                mOptionalList.add(new SignatureOptionalBean(mContext.getResources().getString(R.string.appearance_optional_labels), false));
                mOptionalList.add(new SignatureOptionalBean(mContext.getResources().getString(R.string.appearance_optional_logo), false));
                mOptionalAdapter.setOptionalList(mOptionalList);
                mOptionalAdapter.notifyUpdateData();
                mTitleStyleLl.setClickable(false);
                mTitleStyleTv.setText(editText.getText().toString());
                mLocationTv.setText(mContext.getResources().getString(R.string.fx_string_close));
                mReasonTv.setText(mContext.getResources().getString(R.string.fx_string_close));
                mCreateNewTitleInkItem.setTitle(editText.getText().toString());
                mCreateTitleDialog.dismiss();
                renderSignature(mCreateNewTitleInkItem);
            }
        });
        AppUtil.showSoftInput(mCreateTitleDialog.getInputEditText());
        mCreateTitleDialog.getCancelButton().setOnClickListener(new View.OnClickListener() {

            @Override
            public void onClick(View v) {
                if(mCreateTitleDialog!=null) {
                    mCreateTitleDialog.dismiss();
                }
            }
        });
        mCreateTitleDialog.setOnDismissListener(new DialogInterface.OnDismissListener() {
            @Override
            public void onDismiss(DialogInterface dialog) {
                mCreateTitleDialog = null;
            }
        });
        mCreateTitleDialog.getOKButton().setEnabled(false);
        mCreateTitleDialog.show();
    }


    private void doneTitle() {
        mIsCreateTitle = false;
        setTitle(AppResource.getString(mContext, R.string.appearance_preview));
        setRightButtonEnable(true);
        mTitleStyleLl.setClickable(true);
        mRightTextItem.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        mCreateTypeTv.setVisibility(View.VISIBLE);
        mViewFill.setVisibility(View.VISIBLE);
        mViewFillBottom.setVisibility(View.GONE);
        mTitleStyleNextIv.setVisibility(View.VISIBLE);
        mOptionalGroupTv.setVisibility(View.GONE);
        mOptionalRecyclerView.setVisibility(View.GONE);
        mOptionalDoneTv.setVisibility(View.GONE);
        setTitleStyle(mCreateNewTitleInkItem);
        setSignDynamicInfo(mCreateNewTitleInkItem);
        setReasonAndLocation(mCreateNewTitleInkItem);
        SignatureDataUtil.insertTitleData(mContext, mCreateNewTitleInkItem);
    }


    private void cancelCreateTitle() {
        mIsCreateTitle = false;
        setTitle(AppResource.getString(mContext, R.string.appearance_preview));
        setRightButtonEnable(true);
        mRightTextItem.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        mCreateTypeTv.setVisibility(View.VISIBLE);
        mViewFill.setVisibility(View.VISIBLE);
        mViewFillBottom.setVisibility(View.GONE);
        mTitleStyleNextIv.setVisibility(View.VISIBLE);
        mOptionalGroupTv.setVisibility(View.GONE);
        mOptionalRecyclerView.setVisibility(View.GONE);
        mOptionalDoneTv.setVisibility(View.GONE);
        mTitleStyleLl.setClickable(true);

        String standardStyle = "appearance_standard_style";
        if (standardStyle.equals(mCopySignInkItem.getTitle())) {
            standardStyle = mContext.getApplicationContext().getString(R.string.appearance_standard_style);
        } else {
            standardStyle = mCopySignInkItem.getTitle();
        }
        mTitleStyleTv.setText(standardStyle);
        setSignDynamicInfo(mCopySignInkItem);
        setReasonAndLocationText(mCopySignInkItem);
        renderSignature(mCopySignInkItem);
    }

    private void saveSign() {
        setSignDynamicInfo(mCopySignInkItem);
        mSignAppearanceCallback.onSaveSign(mCopySignInkItem);

    }

    private void cancelUseCurrent() {
        mSignAppearanceCallback.onDismiss(mSignInkItem);
    }

    public void showAppearanceTitleDialog() {
        Context context = mUIExtensionsManager.getAttachedActivity();
        if (context == null) return;
        mSignAppearanceTitleDialog = new SignatureAppearanceTitleDialog(context, mParent, mPdfViewCtrl, mCopySignInkItem.getTitle(),mEditedTitle);
        mSignAppearanceTitleDialog.setSelectorTitleCallback(new SignatureAppearanceTitleDialog.ISignSelectorTitleCallback() {
            @Override
            public void onSelectorTitle(String title) {
                SignatureInkItem titleInkItem = SignatureDataUtil.getTitleByKey(mContext, title);
                String standardStyle = "appearance_standard_style";
                if (standardStyle.equals(title)) {
                    standardStyle = mContext.getApplicationContext().getString(R.string.appearance_standard_style);
                } else {
                    standardStyle = title;
                }
                mTitleStyleTv.setText(standardStyle);
                setTitleStyle(titleInkItem);
                setSignDynamicInfo(titleInkItem);
                setReasonAndLocation(titleInkItem);
                setReasonAndLocationText(titleInkItem);
                renderSignature(mCopySignInkItem);
                mCreateNewTitleInkItem = null;
            }
        });
        mSignAppearanceTitleDialog.loadData();
        mSignAppearanceTitleDialog.showDialog();
    }

    public void showAppearanceLocationDialog() {
        Activity activity = mUIExtensionsManager.getAttachedActivity();
        boolean locationOpen = false;
        String location = "";
        if (mIsCreateTitle) {
            locationOpen = mCreateNewTitleInkItem.isLocationOpen();
            location = mCreateNewTitleInkItem.getLocation();
        } else {
            locationOpen = mCopySignInkItem.isLocationOpen();
            location = mCopySignInkItem.getLocation();
        }
        mSignAppearanceLocationDialog = new SignatureAppearanceLocationDialog(activity, mParent, mPdfViewCtrl, locationOpen, location);
        mSignAppearanceLocationDialog.setSwitchLocation(new SignatureAppearanceLocationDialog.ISignSwitchLocationCallback() {
            @Override
            public void switchLocation(boolean open, String location) {

                if (mIsCreateTitle) {
                    mCreateNewTitleInkItem.setLocationOpen(open ? 1 : 0);
                    mCreateNewTitleInkItem.setLocation(location);
                    renderSignature(mCreateNewTitleInkItem);
                } else {
                    mCopySignInkItem.setLocationOpen(open ? 1 : 0);
                    mCopySignInkItem.setLocation(location);
                    renderSignature(mCopySignInkItem);
                }
                if (open) {
                    if (TextUtils.isEmpty(location)) {
                        mLocationTv.setText("");
                    } else {
                        mLocationTv.setText(location);
                    }

                } else {
                    mLocationTv.setText(mContext.getResources().getString(R.string.fx_string_close));
                }
            }
        });
        mSignAppearanceLocationDialog.loadData();
        mSignAppearanceLocationDialog.showDialog();
    }

    public void showAppearanceReasonDialog() {
        Activity activity = mUIExtensionsManager.getAttachedActivity();
        boolean reasonOpen = false;
        int reason = 0;
        if (mIsCreateTitle) {
            reasonOpen = mCreateNewTitleInkItem.isReasonOpen();
            reason = mCreateNewTitleInkItem.getReason();
        } else {
            reasonOpen = mCopySignInkItem.isReasonOpen();
            reason = mCopySignInkItem.getReason();
        }
        mSignAppearanceReasonDialog = new SignatureAppearanceReasonDialog(activity, mParent, mPdfViewCtrl, reasonOpen, reason);
        mSignAppearanceReasonDialog.setSwitchReason(new SignatureAppearanceReasonDialog.ISignSwitchReasonCallback() {
            @Override
            public void switchReason(boolean open, int reason) {
                if (mIsCreateTitle) {
                    mCreateNewTitleInkItem.setReasonOpen(open ? 1 : 0);
                    mCreateNewTitleInkItem.setReason(reason);
                    renderSignature(mCreateNewTitleInkItem);
                } else {
                    mCopySignInkItem.setReasonOpen(open ? 1 : 0);
                    mCopySignInkItem.setReason(reason);
                    renderSignature(mCopySignInkItem);
                }
                if (open) {
                    mReasonTv.setText(getReason(reason));

                } else {
                    mReasonTv.setText(mContext.getResources().getString(R.string.fx_string_close));
                }
            }
        });
        mSignAppearanceReasonDialog.loadData();
        mSignAppearanceReasonDialog.showDialog();
    }

    public byte[] bitmapToByteArrayARGB8888(Bitmap bitmap) {

        ByteBuffer byteBuffer = ByteBuffer.allocate(bitmap.getByteCount());

        bitmap.copyPixelsToBuffer(byteBuffer);
//        byteBuffer.rewind();
        return byteBuffer.array();
    }

    private Bitmap removeWhiteBg(Bitmap bmp) {

        int format =  com.foxit.sdk.common.Bitmap.e_DIBArgb;
        int pitch = bmp.getRowBytes();
        byte[]  buffer = null;
        buffer = bitmapToByteArrayARGB8888(bmp);
        try {
            com.foxit.sdk.common.Bitmap bitmap = new com.foxit.sdk.common.Bitmap(bmp.getWidth(),bmp.getHeight(),format,buffer,pitch);
            Bitmap maskBitmap =  bitmap.fadeOutBackground(com.foxit.sdk.common.Bitmap.e_EnhanceAlgoGrayLevelTransformation, com.foxit.sdk.common.Bitmap.e_ThresholdAlgoFixedThreshold, 1.0f);
            Bitmap resultBitmap =  applyMask(bmp,maskBitmap);
            return  resultBitmap;

        } catch (PDFException e) {
            throw new RuntimeException(e);
        }

    }

    public Bitmap applyMask(Bitmap originalBitmap, Bitmap maskBitmap){

        Bitmap resultBitmap = Bitmap.createBitmap(originalBitmap.getWidth(), originalBitmap.getHeight(), Bitmap.Config.ARGB_8888);

        Canvas canvas = new Canvas(resultBitmap);
        Paint paint = new Paint(Paint.ANTI_ALIAS_FLAG);

        BitmapShader originalShader = new BitmapShader(originalBitmap, Shader.TileMode.CLAMP, Shader.TileMode.CLAMP);
        BitmapShader alphaShader = new BitmapShader(maskBitmap, Shader.TileMode.CLAMP, Shader.TileMode.CLAMP);


        paint.setShader(new ComposeShader(originalShader, alphaShader, PorterDuff.Mode.DST_IN));
        canvas.drawRect(0, 0, originalBitmap.getWidth(), originalBitmap.getHeight(), paint);
        return resultBitmap;
    }

    private String getReason(int reason){
        if(reason == 0){
            return mContext.getResources().getString(R.string.reason_author);
        }else if(reason == 1){
            return mContext.getResources().getString(R.string.reason_review);
        }else if(reason == 2){
            return mContext.getResources().getString(R.string.reason_approve);
        }else if(reason == 3){
            return mContext.getResources().getString(R.string.reason_approve_legally_binding);
        }else if(reason == 4){
            return mContext.getResources().getString(R.string.reason_accuracy_integrity);
        }else if(reason == 5){
            return mContext.getResources().getString(R.string.reason_sign_terms);
        }else if(reason == 6){
            return mContext.getResources().getString(R.string.reason_specified);
        }else{
            return mContext.getResources().getString(R.string.reason_author);
        }
    }
}
