/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.signature;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.view.View;
import android.view.ViewGroup;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.controls.toolbar.IBaseItem;
import com.foxit.uiextensions.controls.toolbar.impl.TopBarImpl;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.pdfreader.ILayoutChangeListener;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.SystemUiHelper;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import androidx.fragment.app.FragmentActivity;

/**
 * There are two kinds of signatures for SDK, one is the handwriting signature, which simulate the user writing on
 * pdf page, the other is the digital signature, which follows the industrial PKCS#7 standards.
 */
public class SignatureModule implements Module {

    private Context mContext;
    private ViewGroup mParent;
    private PDFViewCtrl mPdfViewCtrl;
    private TopBarImpl mSignatureViewTopBar;
    private SignatureToolHandler mToolHandler;
    private PDFViewCtrl.UIExtensionsManager mUiExtensionsManager;

    public SignatureModule(Context context, ViewGroup parent, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        mContext = context;
        mParent = parent;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = uiExtensionsManager;
    }

    public ToolHandler getToolHandler() {
        return mToolHandler;
    }

    @Override
    public String getName() {
        return Module.MODULE_NAME_PSISIGNATURE;
    }

    @Override
    public boolean loadModule() {
        mToolHandler = new SignatureToolHandler(mContext, mParent, mPdfViewCtrl);
        mSignatureViewTopBar = new TopBarImpl(mContext);
        mSignatureViewTopBar.setStartMargin(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_16dp));
        mSignatureViewTopBar.setEndMargin(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_16dp));
        mSignatureViewTopBar.setEndItemiInterval(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_8dp));
        mSignatureViewTopBar.setBackgroundColor(mContext.getResources().getColor(R.color.b2));
        mSignatureViewTopBar.setShowSolidLine(false);

        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            ((UIExtensionsManager) mUiExtensionsManager).registerToolHandler(mToolHandler);
            ((UIExtensionsManager) mUiExtensionsManager).registerModule(this);
            ((UIExtensionsManager) mUiExtensionsManager).registerLayoutChangeListener(mLayoutChangeListener);
        }
        mPdfViewCtrl.registerDrawEventListener(mDrawEventListener);
        mPdfViewCtrl.registerDocEventListener(mDocEventListener);
        return true;
    }

    @Override
    public boolean unloadModule() {
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            ((UIExtensionsManager) mUiExtensionsManager).unregisterToolHandler(mToolHandler);
            ((UIExtensionsManager) mUiExtensionsManager).unregisterLayoutChangeListener(mLayoutChangeListener);
        }
        mPdfViewCtrl.unregisterDrawEventListener(mDrawEventListener);
        mPdfViewCtrl.unregisterDocEventListener(mDocEventListener);
        return true;
    }

    private ILayoutChangeListener mLayoutChangeListener = new ILayoutChangeListener() {
        @Override
        public void onLayoutChange(View v, int newWidth, int newHeight, int oldWidth, int oldHeight) {
            mToolHandler.onLayoutChange(v, newWidth, newHeight, oldWidth, oldHeight);
        }
    };

    private final PDFViewCtrl.IDocEventListener mDocEventListener = new DocEventListener() {

        @Override
        public void onDocOpened(PDFDoc document, int errCode) {
            if (mToolHandler.isAddCertSignature()) {
                mPdfViewCtrl.updatePagesLayout();
                mToolHandler.gotoSignPage();
                AppThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
                    @Override
                    public void run() {
                        if (SystemUiHelper.getInstance().isFullScreen())
                            SystemUiHelper.getInstance().hideSystemUI(((UIExtensionsManager) mUiExtensionsManager).getAttachedActivity());
                    }
                },100);
            }
        }

        @Override
        public void onDocWillClose(PDFDoc document) {
            mToolHandler.clear();
        }

    };

    private final PDFViewCtrl.IDrawEventListener mDrawEventListener = new PDFViewCtrl.IDrawEventListener() {


        @Override
        public void onDraw(int pageIndex, Canvas canvas) {
            if (mToolHandler != null)
                mToolHandler.onDrawForControls(canvas);
        }
    };

    public boolean onKeyBack() {
        return mToolHandler.onKeyBack();
    }

    public void activeSign(IBaseItem item, boolean active) {
        if (active) {
            if (mToolHandler._onActivate()) {
                mToolHandler.showSignList();
            }
        } else {
            mToolHandler.onDeactivate();
            mPdfViewCtrl.invalidate();
        }
    }

    public BaseBar getSignatureViewTopBar() {
        return mSignatureViewTopBar;
    }

    public Bitmap getSignatureViewBitmap() {
        FragmentActivity act = (FragmentActivity) ((UIExtensionsManager) mUiExtensionsManager).getAttachedActivity();
        SignatureFragment signatureFragment = (SignatureFragment) act.getSupportFragmentManager().findFragmentByTag(SignatureFragment.TAG);
        if (signatureFragment != null)
            return signatureFragment.getSignatureBitmap();
        return null;
    }

    private ISigBitmapChangeCallback mBitmapChangeCallback;

    public void setBitmapChangeCallback(ISigBitmapChangeCallback callback) {
        mBitmapChangeCallback = callback;
    }

    public ISigBitmapChangeCallback getBitmapChangeCallback() {
        return mBitmapChangeCallback;
    }
}
