/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.signature;


import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffColorFilter;
import android.graphics.Rect;
import android.graphics.RectF;
import android.os.Handler;
import android.text.TextUtils;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.CheckBox;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.fragment.app.FragmentActivity;
import androidx.recyclerview.widget.LinearLayoutManager;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.browser.adapter.SuperAdapter;
import com.foxit.uiextensions.browser.adapter.viewholder.SuperViewHolder;
import com.foxit.uiextensions.controls.dialog.sheetmenu.ISheetMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UIBottomSheetMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UISheetMenu;
import com.foxit.uiextensions.modules.panel.bean.BaseBean;
import com.foxit.uiextensions.security.digitalsignature.DigitalSignatureModule;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.SystemUiHelper;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

class SignatureListAdapter extends SuperAdapter<SignatureInkItem> {
    public static final int CLICK_TYPE_CONTENT = 0;
    public static final int CLICK_TYPE_EDIT = 1;
    public static final int CLICK_TYPE_DELETE = 2;

    private OnItemClickCallback mOnItemClickCallback;
    private ISheetMenu mSheetMenu;
    private UIExtensionsManager mUIExtensionsManager;
    private ViewGroup mRootView;

    private List<SignatureInkItem> mSignList;
    private Map<String, WeakReference<Bitmap>> mCacheMap = new HashMap<>();
    private int mSelectedPosition = -1;
    private Context mContext;

    public SignatureListAdapter(Context context, PDFViewCtrl pdfViewCtrl) {
        super(context);
        mContext = context;
        mSignList = new ArrayList<>();
        mUIExtensionsManager = (UIExtensionsManager) pdfViewCtrl.getUIExtensionsManager();
    }

    @Override
    public void notifyUpdateData() {
        notifyDataSetChanged();
    }

    @Override
    public SignatureInkItem getDataItem(int position) {
        return mSignList.get(position);
    }

    @NonNull
    @Override
    public SuperViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        return new SignatureListViewHolder(LayoutInflater.from(getContext()).inflate(R.layout.sign_list_item, parent, false));
    }

    @Override
    public int getItemCount() {
        return mSignList.size();
    }

    class SignatureListViewHolder extends SuperViewHolder {
        private View mContentView;
        private TextView mGroupTitleTv;
        private CheckBox mItemCheckView;
        private ImageView mThumbnailIv;
        private ImageView mMoreIv;

        public SignatureListViewHolder(View itemView) {
            super(itemView);
            mContentView = itemView.findViewById(R.id.sign_list_content_item);
            mGroupTitleTv = itemView.findViewById(R.id.sign_list_group_name);
            mItemCheckView = itemView.findViewById(R.id.sign_list_item_selected);
            mThumbnailIv = itemView.findViewById(R.id.sign_list_child_item_bitmap);
            mMoreIv = itemView.findViewById(R.id.sign_list_more_iv);
        }

        @Override
        public void bind(BaseBean data, int position) {
            SignatureInkItem inkItem = (SignatureInkItem) data;
            if (inkItem.getFlag() == SignatureInkItem.FLAG_LABEL) {
                mGroupTitleTv.setVisibility(View.VISIBLE);
                mContentView.setVisibility(View.GONE);
                mGroupTitleTv.setText(inkItem.getTag());
                return;
            }

            mGroupTitleTv.setVisibility(View.GONE);
            mContentView.setVisibility(View.VISIBLE);
            if (inkItem.selected) {
                mContentView.setSelected(true);
                mItemCheckView.setChecked(true);
                mItemCheckView.setVisibility(View.VISIBLE);
            } else {
                mContentView.setSelected(false);
                mItemCheckView.setChecked(false);
                mItemCheckView.setVisibility(View.INVISIBLE);
            }

            ThemeUtil.setTintList(mItemCheckView, ThemeUtil.getCheckboxColor(getContext()));
            mItemCheckView.getBackground().setColorFilter(new
                    PorterDuffColorFilter(AppResource.getColor(getContext(), R.color.b1), PorterDuff.Mode.SRC_IN));

            mContentView.setSelected(!AppDisplay.isPad()
                    && mSelectedPosition == position);
            LinearLayout.LayoutParams lp = (LinearLayout.LayoutParams)mThumbnailIv.getLayoutParams();
            if(!TextUtils.isEmpty(inkItem.getTitle()) && inkItem.getDsgPath()!=null){
                lp.width = LinearLayout.LayoutParams.WRAP_CONTENT;
                lp.height = LinearLayout.LayoutParams.WRAP_CONTENT;
            }else{
                lp.width = AppDisplay.dp2px(100);
                lp.height = AppDisplay.dp2px(80);
            }
            mThumbnailIv.setLayoutParams(lp);
            WeakReference<Bitmap> reference = mCacheMap.get(inkItem.key);
            Bitmap bitmap = null;
            if (reference != null)
                bitmap = reference.get();
            if (bitmap == null)
                bitmap = getBitmap(inkItem);

            if (bitmap != null) {
                mThumbnailIv.setImageBitmap(bitmap);
            }
            mMoreIv.setOnClickListener(this);
            mMoreIv.setColorFilter(AppResource.getColor(getContext(), R.color.i3_light));
            mContentView.setOnClickListener(this);
        }

        @Override
        public void onClick(View v) {
            if (AppUtil.isFastDoubleClick()) {
                return;
            }
            int id = v.getId();
            int position = getAdapterPosition();
            SignatureInkItem item = mSignList.get(position);
            if (id == R.id.sign_list_content_item) {
                if (mOnItemClickCallback != null)
                    mOnItemClickCallback.onItemClick(CLICK_TYPE_CONTENT, position, item);
            } else if (id == R.id.sign_list_more_iv) {
                mSelectedPosition = position;
                if (mSelectedPosition >= 0)
                    notifyItemChanged(mSelectedPosition);
                showSheetMenu(position, item, v);
            }
        }
    }

    private Bitmap getBitmap(SignatureInkItem item) {
        Bitmap bmp = null;
        try {
            bmp = SignatureDataUtil.getScaleBmpByKey(getContext(), item.key,
                    item.getRect());
            if(!TextUtils.isEmpty(item.getTitle()) && bmp!=null && item.getDsgPath()!=null){
                Bitmap tempBitmap = renderSignature(item,bmp);
                if(tempBitmap!=null){
                    bmp = tempBitmap;
                }
            }
            WeakReference<Bitmap> reference = new WeakReference<>(bmp);
            mCacheMap.put(item.key, reference);
        } catch (OutOfMemoryError error) {
            error.printStackTrace();
        }
        return bmp;
    }

    private Bitmap renderSignature(SignatureInkItem inkItem,Bitmap bitmap) {
        final DigitalSignatureModule dsgModule = (DigitalSignatureModule) mUIExtensionsManager.getModuleByName(Module.MODULE_NAME_DIGITALSIGNATURE);
        if (dsgModule != null && inkItem != null && !TextUtils.isEmpty(inkItem.getTitle())) {
            Bitmap tempBitmap = Bitmap.createBitmap( AppDisplay.dp2px(308),
                    AppDisplay.dp2px(154), Bitmap.Config.ARGB_8888);
            RectF pageRect = new RectF();
            pageRect.set(0, 0, 308,154);
            dsgModule.getDigitalSignatureUtil().createSignature(pageRect, inkItem, bitmap, tempBitmap,false);
            return tempBitmap;
        }
       return null;
    }
    private void showSheetMenu(final int position, final SignatureInkItem inkItem, View view) {
        mSheetMenu = UISheetMenu.newInstance((FragmentActivity) mUIExtensionsManager.getAttachedActivity());
        if (AppDisplay.isPad())
            mSheetMenu.setWidth(AppResource.getDimensionPixelSize(getContext(), R.dimen.ux_pad_more_menu_width));
        List<Integer> items = new ArrayList<>();
        items.add(ISheetMenu.EDIT_MENU);
        items.add(ISheetMenu.DELETE_MENU);
        mSheetMenu.setSheetItems(items);
        mSheetMenu.setAutoResetSystemUiOnShow(false);
        mSheetMenu.setSheetItemClickListener(new UIBottomSheetMenu.OnSheetItemClickListener() {
            @Override
            public void onItemClick(int type) {
                dismissSheetMenu();
                if (AppUtil.isFastDoubleClick()) {
                    return;
                }

                if (type == ISheetMenu.EDIT_MENU) {
                    if (mOnItemClickCallback != null)
                        mOnItemClickCallback.onItemClick(CLICK_TYPE_EDIT, position, inkItem);
                } else if (type == ISheetMenu.DELETE_MENU) {
                    if (mOnItemClickCallback != null)
                        mOnItemClickCallback.onItemClick(CLICK_TYPE_DELETE, position, inkItem);
                }
            }
        });
        mSheetMenu.setOnSheetDismissListener(new ISheetMenu.OnSheetDismissListener() {
            @Override
            public void onDismiss(ISheetMenu sheetMenu) {
                if (mSelectedPosition != -1) {
                    int tempPosition = mSelectedPosition;
                    mSelectedPosition = -1;
                    if (tempPosition >= 0)
                        notifyItemChanged(tempPosition);
                }
            }
        });
        showSheetMenu(view);
    }

    private void showSheetMenu(View view) {
        Rect rect = new Rect();
        view.getGlobalVisibleRect(rect);

        int[] location = new int[2];
        view.getLocationOnScreen(location);
        int screen_x = location[0];
        int screen_y = location[1];
        if (AppDevice.isChromeOs(mUIExtensionsManager.getAttachedActivity())) {
            mUIExtensionsManager.getRootView().getLocationOnScreen(location);
            screen_x -= location[0];
            screen_y -= location[1];
            rect.set(screen_x, screen_y, screen_x + rect.width(), screen_y + rect.height());
        } else {
            view.getLocationInWindow(location);
            int window_x = location[0];
            int window_y = location[1];
            rect.offset(screen_x - window_x, screen_y - window_y);
        }
        if (!SystemUiHelper.getInstance().isFullScreenMode(mUIExtensionsManager.getAttachedActivity())
                && SystemUiHelper.getInstance().isFullScreen())
            rect.offset(0, -SystemUiHelper.getInstance().getStatusBarHeight(mUIExtensionsManager.getAttachedActivity()));
        mSheetMenu.show(mUIExtensionsManager.getRootView(), rect);
    }

    private void updateSheetMenu(final LinearLayoutManager layoutManager) {
        if (mSheetMenu != null && mSheetMenu.isShowing()) {
            new Handler().postDelayed(new Runnable() {
                @Override
                public void run() {
                    if (AppDisplay.isPad()) {
                        View view = layoutManager.findViewByPosition(mSelectedPosition);
                        if (view != null) {
                            View moreView = view.findViewById(R.id.sign_list_more_iv);
                            showSheetMenu(moreView);
                        } else {
                            dismissSheetMenu();
                        }
                    } else {
                        showSheetMenu(mUIExtensionsManager.getRootView());
                    }
                }
            }, 300);
        }
    }

    private void dismissSheetMenu() {
        if (mSheetMenu != null && mSheetMenu.isShowing()) {
            mSheetMenu.dismiss();
        }
    }

    void setSignList(List<SignatureInkItem> signList) {
        mSignList = signList;
    }

    void clear() {
        for (WeakReference<Bitmap> bitmapWeakReference : mCacheMap.values()) {
            bitmapWeakReference.clear();
        }
        mCacheMap.clear();
    }

    void setOnItemClickCallback(OnItemClickCallback callback) {
        mOnItemClickCallback = callback;
    }

    interface OnItemClickCallback {
        void onItemClick(int type, int position, SignatureInkItem inkItem);
    }

    void onLayoutChanged(final LinearLayoutManager layoutManager) {
        updateSheetMenu(layoutManager);
    }

}
