/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.signature;

import android.app.Activity;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.pm.ActivityInfo;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.os.Build;
import android.os.Bundle;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.ISystemPermissionProvider;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.controls.dialog.BaseDialogFragment;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.SystemUiHelper;


public class SignatureFragment extends BaseDialogFragment {
    public static final String TAG = "InkSignFragment";
    public static final String KEY_SCREEN_ORIENTATION = "ScreenOrientation";

    public interface SignatureInkCallback {
        void onSuccess(boolean isFromFragment, SignatureInkItem inkItem);

        void onBackPressed();
    }

    private Context mContext;
    private ViewGroup mParent;
    private PDFViewCtrl mPdfViewCtrl;
    private SignatureViewController mSupport;
    private SignatureInkCallback mCallback;
    private int mOrientation;
    private boolean mAttach;
    private SignatureInkItem mInkItem;
    private boolean mIsFromSignatureField = false;

    public boolean isAttached() {
        return mAttach;
    }

    public void setInkCallback(SignatureInkCallback callback) {
        this.mCallback = callback;
    }

    void setInkCallback(SignatureInkCallback callback, SignatureInkItem item) {
        this.mCallback = callback;
        mInkItem = item;
    }

    private boolean checkInit() {

        return mCallback != null;
    }

    public void init(Context context, ViewGroup parent, PDFViewCtrl pdfViewCtrl, boolean isFromSignatureField) {
        mContext = context;
        mParent = parent;
        mPdfViewCtrl = pdfViewCtrl;
        mIsFromSignatureField = isFromSignatureField;
    }

    @Override
    public void onAttach(Activity activity) {
        super.onAttach(activity);
        if (mPdfViewCtrl == null) {
            dismissAllowingStateLoss();
            return;
        }

        mOrientation = activity.getRequestedOrientation();
        activity.setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_SENSOR_LANDSCAPE);
        if (!checkInit()) {
            getActivity().getSupportFragmentManager().popBackStack();
            return;
        }
        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).registerThemeEventListener(mThemeChangedListener);
        if (mSupport == null) {
            mSupport = new SignatureViewController(mContext, mParent, mPdfViewCtrl, mCallback);
        }
        mAttach = true;
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        if (savedInstanceState != null) {
            mOrientation = savedInstanceState.getInt(KEY_SCREEN_ORIENTATION, mOrientation);
        }
    }

    @Override
    public void onSaveInstanceState(@NonNull Bundle outState) {
        super.onSaveInstanceState(outState);
        outState.putInt(KEY_SCREEN_ORIENTATION, mOrientation);
    }

    private boolean mCheckCreateView;

    @Override
    protected View onCreateView(LayoutInflater inflater, ViewGroup container) {
        if (mSupport == null) {
            getActivity().getSupportFragmentManager().popBackStack();
            return null;
        }
        ViewGroup view = (ViewGroup) mSupport.getView().getParent();
        if (view != null)
            view.removeView(mSupport.getView());

        this.getDialog().setOnKeyListener(new DialogInterface.OnKeyListener() {
            @Override
            public boolean onKey(DialogInterface dialog, int keyCode, KeyEvent event) {
                if (keyCode == KeyEvent.KEYCODE_BACK) {
                    mCallback.onBackPressed();
                    dismiss();
                    return true;
                }
                return false;
            }
        });

        mSupport.setIsFromSignatureField(mIsFromSignatureField);
        mSupport.setActivity(getActivity(), this);
        mCheckCreateView = true;
        int screenWidth = AppDisplay.getActivityWidth();
        int screenHeight = AppDisplay.getActivityHeight();
        if (!AppDevice.isChromeOs(getActivity()) && screenWidth < screenHeight) {
            screenWidth = AppDisplay.getActivityHeight();
            screenHeight = AppDisplay.getActivityWidth();
        }

        if (mInkItem == null) {
            mSupport.init(screenWidth, screenHeight);
        } else {
            mSupport.init(screenWidth,
                    screenHeight,
                    mInkItem);
        }
        return mSupport.getView();
    }

    @Override
    protected PDFViewCtrl getPDFViewCtrl() {
        return mPdfViewCtrl;
    }

    @Override
    protected void onLayoutChange(View v, int newWidth, int newHeight, int oldWidth, int oldHeight) {
    }

    @Override
    public void onConfigurationChanged(Configuration newConfig) {
        super.onConfigurationChanged(newConfig);

        int actW = AppDisplay.getActivityWidth();
        int actH = AppDisplay.getActivityHeight();
        if (!mCheckCreateView && actW > actH) {
            mCheckCreateView = true;
            if (mInkItem == null) {
                mSupport.init(actW, actH);
            } else {
                mSupport.init(actW,
                        actH,
                        mInkItem);
            }
        } else {
            if (AppDevice.isChromeOs(getActivity())) {
                mSupport.reDraw(actW, actH);
            }
        }
    }

    @Override
    public void onDetach() {
        super.onDetach();
        if (getActivity() !=null) {
            getActivity().setRequestedOrientation(mOrientation);
        }

        if (mSupport != null) {
            mSupport.unInit();
        }
        mAttach = false;
        mSupport = null;

        if (getPDFViewCtrl() != null) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) getPDFViewCtrl().getUIExtensionsManager();
            if (uiExtensionsManager != null) {
                uiExtensionsManager.unregisterThemeEventListener(mThemeChangedListener);
            }
        }
    }

    @Override
    public void onActivityResult(int requestCode, int resultCode, @Nullable Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if (SystemUiHelper.getInstance().isFullScreen() && getDialog() != null)
            SystemUiHelper.getInstance().hideNavigationBar(getDialog().getWindow());

        if (mSupport != null)
            mSupport.onActivityResult(requestCode, resultCode, data);
    }

    private int mPermissionRequestCode;
    private Event.Callback mPermissionCallback;

    public void requestPermissions(String permission, final String[] permissions, int requestCode, Event.Callback callback) {
        mPermissionCallback = callback;
        mPermissionRequestCode = requestCode;
        if (Build.VERSION.SDK_INT >= 23) {
            int permissionCode = ContextCompat.checkSelfPermission(mContext, permission);
            if (permissionCode != PackageManager.PERMISSION_GRANTED) {
                UIExtensionsManager UIExtManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
                if (UIExtManager.getSystemPermissionProvider() == null) {
                    requestPermissions(permissions, mPermissionRequestCode);
                } else {
                    UIExtManager.getSystemPermissionProvider().requestPermission(
                            true,
                            AppResource.getString(mContext, R.string.fx_string_signature),
                            permissions,
                            new ISystemPermissionProvider.IPermissionResult() {
                                @Override
                                public void onResult(boolean allowed) {
                                    if (allowed)
                                        requestPermissions(permissions, mPermissionRequestCode);
                                }
                            });
                }
                return;
            }
        }
        callback.result(null, true);
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);
        if (verifyPermissions(grantResults)) {
            if (mPermissionRequestCode == requestCode) {
                mPermissionCallback.result(null, true);
            }
        } else {
            mPermissionCallback.result(null, false);
        }
    }

    private boolean verifyPermissions(int[] grantResults) {
        for (int grantResult : grantResults) {
            if (grantResult != PackageManager.PERMISSION_GRANTED) {
                return false;
            }
        }
        return true;
    }

    public Bitmap getSignatureBitmap() {
        if (mSupport != null)
            return mSupport.getSignatureBitmap();
        return null;
    }

    private IThemeEventListener mThemeChangedListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            if (mSupport != null) {
                mSupport.updateTheme();
            }
        }
    };

}
