/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.signature;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Rect;
import android.text.TextUtils;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.utils.AppSQLite;
import com.foxit.uiextensions.utils.AppSQLite.FieldInfo;
import com.foxit.uiextensions.utils.AppSharedPreferences;
import com.foxit.uiextensions.utils.AppUtil;

import java.io.ByteArrayOutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.UUID;

public class SignatureDataUtil {
    private static final String TABLE_MODEL = SignatureConstants.getModelTableName();
    private static final String NAME = SignatureConstants.SG_DSG_NAME;
    private static final String LEFT = "_sg_left";
    private static final String TOP = "_sg_top";
    private static final String RIGHT = "_sg_right";
    private static final String BOTTOM = "_sg_bottom";
    private static final String BLOB = "_sg_bolb";
    private static final String COLOR = "_color";
    private static final String DIAMETER = "_diamter";
    private static final String DSG_PATH = SignatureConstants.SG_DSG_PATH_FIELD;
    private static final String TABLE_RECENT = SignatureConstants.getRecentTableName();
    private static final String[] COL = {NAME, BLOB};
    private static final String TABLE_TITLE = SignatureConstants.getTitleTableName();
    private static final String TABLE_RECENT_TITLE = SignatureConstants.getRecentTitleTableName();
    private static final String TITLE_NAME = SignatureConstants.SG_TITLE_NAME;
    private static final String TITLE_LOCATION = SignatureConstants.SG_TITLE_LOCATION;
    private static final String TITLE_OPEN_LOCATION = SignatureConstants.SG_TITLE_OPEN_LOCATION;
    private static final String TITLE_REASON = SignatureConstants.SG_TITLE_REASON;
    private static final String TITLE_OPEN_REASON = SignatureConstants.SG_TITLE_OPEN_REASON;
    private static final String TITLE_OPTIONAL_NAME = SignatureConstants.SG_TITLE_OPTIONAL_NAME;
    private static final String TITLE_OPTIONAL_VERSION = SignatureConstants.SG_TITLE_OPTIONAL_VERSION;
    private static final String TITLE_OPTIONAL_DN = SignatureConstants.SG_TITLE_OPTIONAL_DN;
    private static final String TITLE_OPTIONAL_DATE = SignatureConstants.SG_TITLE_OPTIONAL_DATE;
    private static final String TITLE_OPTIONAL_LABELS = SignatureConstants.SG_TITLE_OPTIONAL_LABELS;
    private static final String TITLE_OPTIONAL_LOGO = SignatureConstants.SG_TITLE_OPTIONAL_LOGO;


    private static final String CLEAR_BG = SignatureConstants.SG_CLEAR_BG;

    private static final String PIC = SignatureConstants.SG_PIC;
    private static boolean mInit = false;

    synchronized public static List<String> getModelKeys(Context context) {
        if (!checkInit(context)) return null;
        List<String> list = null;
        Cursor cursor = AppSQLite.getInstance(context).select(TABLE_MODEL, COL, null, null, null, null, "_id desc");
        if (cursor == null) return null;
        try {
            int count = cursor.getCount();
            if (count > 0) {
                list = new ArrayList<String>(count);
                while (cursor.moveToNext()) {
                    list.add(cursor.getString(cursor.getColumnIndex(NAME)));
                }
            }
        } catch (Exception e) {
        }
        cursor.close();
        return list;
    }


    synchronized public static Bitmap getScaleBmpByKey(Context context, String key, Rect rect) {
        if (!checkInit(context)) return null;
        Bitmap bitmap = null;
        Cursor cursor = AppSQLite.getInstance(context).select(TABLE_MODEL, null, NAME + "=?", new String[]{key}, null, null, null);
        if (cursor == null) return null;
        if (cursor.getCount() > 0) {
            cursor.moveToFirst();
            byte[] in = cursor.getBlob(cursor.getColumnIndex(BLOB));
            Bitmap bmp = BitmapFactory.decodeByteArray(in, 0, in.length);
            bitmap = bmp.copy(Bitmap.Config.ARGB_8888, true);
            if (bmp != null) {
                Rect bitmapRect = rect;
                int t = bitmapRect.top;
                int b = bitmapRect.bottom;
                int l = bitmapRect.left;
                int r = bitmapRect.right;
                int[] pixels = new int[bitmapRect.width() * bitmapRect.height()];
                bitmap.getPixels(pixels, 0, r - l, l, t, r - l, b - t);
                for (int i = 0; i < pixels.length; i++) {
                    if (0xFFFFFFFF == pixels[i]) {
                        pixels[i] = 0x0;
                    }
                }
                bitmap = Bitmap.createBitmap(pixels, bitmapRect.width(), bitmapRect.height(), Bitmap.Config.ARGB_8888);
                bmp.recycle();
                bmp = null;
            }
        }
        cursor.close();
        return bitmap;
    }

    synchronized public static HashMap<String, Object> getBitmapByKey(Context context, String key) {
        if (!checkInit(context)) return null;

        if (!AppSharedPreferences.getInstance(context).getBoolean("signature", "AmendSigKey", false)) {

            String defaultTitleName = getDefaultTitleKey(context);
            if (TextUtils.isEmpty(defaultTitleName)) {
                defaultTitleName = R.string.appearance_standard_style + "";
            }
            updateByTitleKey(context, TABLE_MODEL, TITLE_NAME, defaultTitleName);
            updateByTitleKey(context, TABLE_TITLE, TITLE_NAME, defaultTitleName);
            updateByTitleKey(context, TABLE_RECENT_TITLE, TITLE_NAME, defaultTitleName);
            int[] reasons = new int[]{R.string.reason_author, R.string.reason_review, R.string.reason_approve, R.string.reason_approve_legally_binding
                    , R.string.reason_accuracy_integrity, R.string.reason_sign_terms, R.string.reason_specified};
            for (int i = 0; i < reasons.length; i++) {
                updateByReason(context, TABLE_MODEL, TITLE_REASON, reasons[i]);
                updateByReason(context, TABLE_TITLE, TITLE_REASON, reasons[i]);
            }
            try {
                updateSigAndTitleByReason(context, TABLE_MODEL, TITLE_REASON);
                updateSigAndTitleByReason(context, TABLE_TITLE, TITLE_REASON);
            } catch (Exception e) {
                e.printStackTrace();
            }
            AppSharedPreferences.getInstance(context).setBoolean("signature", "AmendSigKey", true);
        }
        HashMap<String, Object> map = null;
        Cursor cursor = AppSQLite.getInstance(context).select(TABLE_MODEL, null, NAME + "=?", new String[]{key}, null, null, null);
        if (cursor == null) return null;
        try {
            if (cursor.getCount() > 0) {
                while (cursor.moveToNext()) {
                    byte[] in = cursor.getBlob(cursor.getColumnIndex(BLOB));
                    Bitmap bmp = BitmapFactory.decodeByteArray(in, 0, in.length);
                    if (bmp != null) {
                        map = new HashMap<String, Object>();
                        map.put("key", key);

                        int color = cursor.getInt(cursor.getColumnIndex(COLOR));
                        map.put("color", color);
                        map.put("bitmap", bmp);
                        float diameter = cursor.getFloat(cursor.getColumnIndex(DIAMETER));
                        map.put("diameter", diameter);
                        int l = cursor.getInt(cursor.getColumnIndex(LEFT));
                        int t = cursor.getInt(cursor.getColumnIndex(TOP));
                        int r = cursor.getInt(cursor.getColumnIndex(RIGHT));
                        int b = cursor.getInt(cursor.getColumnIndex(BOTTOM));
                        Rect rect = new Rect(l, t, r, b);
                        map.put("rect", rect);
                        int clearBg = cursor.getInt(cursor.getColumnIndex(CLEAR_BG));
                        map.put("clearBg", clearBg);
                        int pic = cursor.getInt(cursor.getColumnIndex(PIC));
                        map.put("pic", pic);

                        if (cursor.getColumnIndex(DSG_PATH) != -1) {
                            try {
                                map.put("dsgPath", cursor.getString(cursor.getColumnIndex(DSG_PATH)));

                                String title = cursor.getString(cursor.getColumnIndex(TITLE_NAME));
                                map.put("title", title);

                                String dn = cursor.getString(cursor.getColumnIndex(TITLE_OPTIONAL_DN));
                                map.put("dn", dn);
                                String location = cursor.getString(cursor.getColumnIndex(TITLE_LOCATION));
                                map.put("location", location);
                                int openLocation = cursor.getInt(cursor.getColumnIndex(TITLE_OPEN_LOCATION));
                                map.put("openLocation", openLocation);
                                int reason = cursor.getInt(cursor.getColumnIndex(TITLE_REASON));
                                map.put("reason", reason);
                                int openReason = cursor.getInt(cursor.getColumnIndex(TITLE_OPEN_REASON));
                                map.put("openReason", openReason);
                                String name = cursor.getString(cursor.getColumnIndex(TITLE_OPTIONAL_NAME));
                                map.put("name", name);
                                String date = cursor.getString(cursor.getColumnIndex(TITLE_OPTIONAL_DATE));
                                map.put("date", date);
                                String version = cursor.getString(cursor.getColumnIndex(TITLE_OPTIONAL_VERSION));
                                map.put("version", version);


                            } catch (Exception e) {
                                map.put("dsgPath", null);
                            }
                        } else {
                            map.put("dsgPath", null);
                        }

                        break;
                    }
                }
            }
        } catch (Exception e) {
        }
        cursor.close();
        return map;
    }

    synchronized public static boolean insertData(Context context, Bitmap bmp, Rect rect, int color, float diameter, String dsgPath,boolean isClearBg,boolean isPic) {
        if (!checkInit(context)) return false;
        ContentValues values = new ContentValues();
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        autoCompressBitmap(bmp, os);
        String key = UUID.randomUUID().toString();
        values.put(NAME, key);
        values.put(DIAMETER, diameter);
        values.put(COLOR, color);
        values.put(LEFT, rect.left);
        values.put(TOP, rect.top);
        values.put(RIGHT, rect.right);
        values.put(BOTTOM, rect.bottom);
        values.put(BLOB, os.toByteArray());
        values.put(DSG_PATH, dsgPath);
        values.put(CLEAR_BG, isClearBg);
        values.put(PIC, isPic);
        AppSQLite.getInstance(context).insert(TABLE_MODEL, values);
        insertRecent(context, key);
        return true;
    }

    synchronized public static boolean insertData(Context context, SignatureInkItem inkItem) {
        if (!checkInit(context)) return false;
        ContentValues values = new ContentValues();
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        autoCompressBitmap(inkItem.bitmap, os);
        String key = UUID.randomUUID().toString();
        values.put(NAME, key);
        values.put(DIAMETER, inkItem.diameter);
        values.put(COLOR, inkItem.color);
        values.put(LEFT, inkItem.rect.left);
        values.put(TOP, inkItem.rect.top);
        values.put(RIGHT, inkItem.rect.right);
        values.put(BOTTOM, inkItem.rect.bottom);
        values.put(BLOB, os.toByteArray());
        values.put(DSG_PATH, inkItem.dsgPath);
        values.put(TITLE_NAME, inkItem.title);
        values.put(TITLE_LOCATION, inkItem.location);
        values.put(TITLE_OPEN_LOCATION, inkItem.isLocationOpen);
        values.put(TITLE_REASON, inkItem.reason);
        values.put(TITLE_OPEN_REASON, inkItem.isReasonOpen);
        values.put(TITLE_OPTIONAL_NAME, inkItem.name);
        values.put(TITLE_OPTIONAL_VERSION, inkItem.version);
        values.put(TITLE_OPTIONAL_DN, inkItem.distinguishedName);
        values.put(TITLE_OPTIONAL_DATE, inkItem.date);
        values.put(CLEAR_BG, inkItem.isClearBg);
        values.put(PIC, inkItem.isPic);
        AppSQLite.getInstance(context).insert(TABLE_MODEL, values);
        insertRecent(context, key);
        return true;
    }

    private static void autoCompressBitmap(Bitmap bmp, ByteArrayOutputStream os) {
        if (bmp == null) return;
        int size = bmp.getByteCount();
        if (size > 2 * 1024 * 1024) {

            int compressCount = 0;
            while (size >= 2 * 1024 * 1024) {
                os.reset();
                if (compressCount == 0) {
                    bmp.compress(Bitmap.CompressFormat.PNG, 100, os);
                } else {
                    bmp.compress(Bitmap.CompressFormat.JPEG, 100 - (5 * compressCount - 1), os);
                }
                size = os.toByteArray().length;
                compressCount++;
                if (100 - (5 * compressCount - 1) == 5) {
                    break;
                }
            }
        } else {
            bmp.compress(Bitmap.CompressFormat.PNG, 100, os);
        }
    }

    synchronized public static boolean updateByKey(Context context, String key, Bitmap bmp, Rect rect, int color, float diameter, String dsgPath,boolean isClearBg,boolean isPic) {
        if (!checkInit(context)) return false;
        if (isExistKey(context, TABLE_MODEL, key)) {
            ContentValues values = new ContentValues();
            ByteArrayOutputStream os = new ByteArrayOutputStream();
            bmp.compress(Bitmap.CompressFormat.PNG, 100, os);
            values.put(BLOB, os.toByteArray());
            values.put(LEFT, rect.left);
            values.put(TOP, rect.top);
            values.put(RIGHT, rect.right);
            values.put(BOTTOM, rect.bottom);
            values.put(DIAMETER, diameter);
            values.put(COLOR, color);
            values.put(DSG_PATH, dsgPath);
            values.put(CLEAR_BG, isClearBg);
            values.put(PIC, isPic);
            AppSQLite.getInstance(context).update(TABLE_MODEL, values, NAME, new String[]{key});
            insertRecent(context, key);
        } else {
            insertData(context, bmp, rect, color, diameter, dsgPath,isClearBg,isPic);
        }
        return true;
    }

    synchronized public static boolean updateByKey(Context context, SignatureInkItem inkItem) {
        if (!checkInit(context)) return false;
        if (isExistKey(context, TABLE_MODEL, inkItem.key)) {
            ContentValues values = new ContentValues();
            ByteArrayOutputStream os = new ByteArrayOutputStream();
            inkItem.bitmap.compress(Bitmap.CompressFormat.PNG, 100, os);
            values.put(BLOB, os.toByteArray());
            values.put(LEFT, inkItem.rect.left);
            values.put(TOP, inkItem.rect.top);
            values.put(RIGHT, inkItem.rect.right);
            values.put(BOTTOM, inkItem.rect.bottom);
            values.put(DIAMETER, inkItem.diameter);
            values.put(COLOR, inkItem.color);
            values.put(DSG_PATH, inkItem.dsgPath);
            values.put(TITLE_NAME, inkItem.title);
            values.put(TITLE_LOCATION, inkItem.location);
            values.put(TITLE_OPEN_LOCATION, inkItem.isLocationOpen);
            values.put(TITLE_REASON, inkItem.reason);
            values.put(TITLE_OPEN_REASON, inkItem.isReasonOpen);
            values.put(TITLE_OPTIONAL_NAME, inkItem.name);
            values.put(TITLE_OPTIONAL_VERSION, inkItem.version);
            values.put(TITLE_OPTIONAL_DN, inkItem.distinguishedName);
            values.put(TITLE_OPTIONAL_DATE, inkItem.date);
            values.put(CLEAR_BG, inkItem.isClearBg);
            values.put(PIC, inkItem.isPic);
            AppSQLite.getInstance(context).update(TABLE_MODEL, values, NAME, new String[]{inkItem.key});
            insertRecent(context, inkItem.key);
        } else {
            insertData(context, inkItem);
        }
        return true;
    }

    synchronized public static boolean deleteSigByTitle(Context context, String title) {
        if (!checkInit(context)) return false;

        Cursor cursor = AppSQLite.getInstance(context).select(TABLE_MODEL, new String[]{NAME}, TITLE_NAME + "=?", new String[]{title}, null, null, "_id desc");
        if (cursor == null) return false;
        if (cursor.getCount() > 0) {
            List<String> temp = new ArrayList<String>();
            while (cursor.moveToNext()) {
                String key = cursor.getString(cursor.getColumnIndex(NAME));
                temp.add(key);
            }
            if (temp.size() > 0) {
                for (int i = 0; i < temp.size(); i++) {
                    deleteByKey(context, TABLE_MODEL, temp.get(i));
                }
            }
        }
        cursor.close();

        return true;
    }

    synchronized public static boolean deleteByKey(Context context, String table, String key) {
        if (!checkInit(context)) return false;
        if (!table.equals(TABLE_RECENT) && isExistKey(context, TABLE_RECENT, key)) {
            deleteByKey(context, TABLE_RECENT, key);
        }
        AppSQLite.getInstance(context).delete(table, NAME, new String[]{key});
        return true;
    }


    synchronized public static boolean insertRecent(Context context, String key) {
        if (!checkInit(context)) return false;
        if (isExistKey(context, TABLE_RECENT, key)) {
            deleteByKey(context, TABLE_RECENT, key);
        }
        ContentValues values = new ContentValues();
        values.put(NAME, key);
        AppSQLite.getInstance(context).insert(TABLE_RECENT, values);
        return true;
    }

    synchronized public static List<String> getRecentKeys(Context context) {
        if (!checkInit(context)) return null;
        List<String> list = null;
        Cursor cursor = AppSQLite.getInstance(context).select(TABLE_RECENT, new String[]{NAME}, null, null, null, null, "_id desc");
        if (cursor == null) return null;
        if (cursor.getCount() > 0) {
            list = new ArrayList<String>();
            List<String> temp = new ArrayList<String>();
            int count = 0;
            while (cursor.moveToNext()) {
                String key = cursor.getString(cursor.getColumnIndex(NAME));
                if (isExistKey(context, TABLE_MODEL, key))
                    list.add(key);
                else
                    temp.add(key);
            }
            if (temp.size() > 0) {
                for (int i = 0; i < temp.size(); i++) {
                    deleteByKey(context, TABLE_RECENT, temp.get(i));
                }
            }
        }
        cursor.close();
        return list;
    }


    synchronized public static HashMap<String, Object> getRecentData(Context context) {
        if (!checkInit(context)) return null;
        HashMap<String, Object> map = null;
        List<String> list = getRecentKeys(context);
        if (list != null && list.size() > 0) {
            map = getBitmapByKey(context, list.get(0));
        }
        return map;
    }

    synchronized public static HashMap<String, Object> getRecentNormalSignData(Context context) {
        if (!checkInit(context)) return null;
        HashMap<String, Object> map = null;
        List<String> list = getRecentKeys(context);
        if (list != null && list.size() > 0) {
            for (String key : list) {
                map = getBitmapByKey(context, key);
                if (map != null && map.get("dsgPath") == null /*|| AppUtil.isEmpty((String) map.get("dsgPath"))*/) {
                    insertRecent(context, key);
                    return map;
                }
            }
        }
        return null;
    }

    synchronized public static HashMap<String, Object> getRecentDsgSignData(Context context) {
        if (!checkInit(context)) return null;
        HashMap<String, Object> map = null;
        List<String> list = getRecentKeys(context);
        if (list != null && list.size() > 0) {
            for (String key : list) {
                map = getBitmapByKey(context, key);
                if (map != null && map.get("dsgPath") != null && !AppUtil.isEmpty((String) map.get("dsgPath"))) {
                    return map;
                }
            }
        }
        return null;
    }


    synchronized public static SignatureInkItem getTitleByKey(Context context, String
            key) {
        if (!checkInit(context)) return null;
        SignatureInkItem inkItem = null;
        Cursor cursor = AppSQLite.getInstance(context).select(TABLE_TITLE, null, TITLE_NAME + "=?", new String[]{key}, null, null, null);
        if (cursor == null) return null;
        if (cursor.getCount() > 0) {
            while (cursor.moveToNext()) {
                inkItem = new SignatureInkItem();
                inkItem.setTitle(cursor.getString(cursor.getColumnIndex(TITLE_NAME)));
                inkItem.setReason(cursor.getInt(cursor.getColumnIndex(TITLE_REASON)));
                inkItem.setReasonOpen(cursor.getInt(cursor.getColumnIndex(TITLE_OPEN_REASON)));
                inkItem.setLocation(cursor.getString(cursor.getColumnIndex(TITLE_LOCATION)));
                inkItem.setLocationOpen(cursor.getInt(cursor.getColumnIndex(TITLE_OPEN_LOCATION)));
                inkItem.setNameOpen(cursor.getInt(cursor.getColumnIndex(TITLE_OPTIONAL_NAME)));
                inkItem.setDistinguishedNameOpen(cursor.getInt(cursor.getColumnIndex(TITLE_OPTIONAL_DN)));
                inkItem.setDateOpen(cursor.getInt(cursor.getColumnIndex(TITLE_OPTIONAL_DATE)));
                inkItem.setVersionOpen(cursor.getInt(cursor.getColumnIndex(TITLE_OPTIONAL_VERSION)));
                inkItem.setLogo(cursor.getInt(cursor.getColumnIndex(TITLE_OPTIONAL_LOGO)));
                inkItem.setLabels(cursor.getInt(cursor.getColumnIndex(TITLE_OPTIONAL_LABELS)));
                break;
            }
        }
        cursor.close();
        return inkItem;
    }

    synchronized public static List<String> getTitleKeys(Context context) {
        if (!checkInit(context)) return null;
        List<String> list = null;
        Cursor cursor = AppSQLite.getInstance(context).select(TABLE_TITLE, new String[]{TITLE_NAME}, null, null, null, null, "_id desc");
        if (cursor == null) return null;
        int count = cursor.getCount();
        if (count > 0) {
            list = new ArrayList<String>(count);
            while (cursor.moveToNext()) {
                list.add(cursor.getString(cursor.getColumnIndex(TITLE_NAME)));
            }
        }
        cursor.close();
        return list;
    }

    synchronized public static boolean updateSigAndTitleByReason(Context context, String tableName, String filedName) {
        if (!checkInit(context)) return false;
        ContentValues values = new ContentValues();
        String[] reasons = new String[]{"6"};
        Cursor cursor = AppSQLite.getInstance(context).select(tableName, new String[]{filedName}, null, null, null, null, "_id desc");
        if (cursor == null) return false;
        int count = cursor.getCount();
        if (count > 0) {
            while (cursor.moveToNext()) {
                values.put(TITLE_REASON, 0);
                AppSQLite.getInstance(context).updateWhereGreaterThan(tableName, values, filedName, reasons);
            }
        }
        cursor.close();
        return true;
    }

    synchronized public static String getDefaultTitleKey(Context context) {
        if (!checkInit(context)) return null;
        String titleName = "";
        Cursor cursor = AppSQLite.getInstance(context).select(TABLE_TITLE, new String[]{TITLE_NAME}, null, null, null, null, "_id asc");
        if (cursor == null) return null;
        int count = cursor.getCount();
        if (count > 0) {
            while (cursor.moveToNext()) {
                titleName = cursor.getString(cursor.getColumnIndex(TITLE_NAME));
                break;
            }
        }
        cursor.close();
        return titleName;
    }

    synchronized public static String getRecentTitleKey(Context context) {
        if (!checkInit(context)) return null;
        String recentTitleName = "";
        Cursor cursor = AppSQLite.getInstance(context).select(TABLE_RECENT_TITLE, null, null, null, null, null, null);
        if (cursor == null) return null;
        int count = cursor.getCount();
        if (count > 0) {
            while (cursor.moveToNext()) {
                recentTitleName = cursor.getString(cursor.getColumnIndex(TITLE_NAME));
            }
        }
        cursor.close();
        return recentTitleName;
    }

    synchronized public static boolean updateByTitleKey(Context context, String tableName, String filedName, String key) {
        if (!checkInit(context)) return false;
        if (isExistKey(context, tableName, filedName, key)) {
            ContentValues values = new ContentValues();
            values.put(filedName, "appearance_standard_style");
            AppSQLite.getInstance(context).update(tableName, values, filedName, new String[]{key});

        }
        return true;
    }

    synchronized public static boolean updateByReason(Context context, String tableName, String filedName, int reason) {
        if (!checkInit(context)) return false;
        ContentValues values = new ContentValues();
        if (reason == R.string.reason_author) {
            values.put(TITLE_REASON, 0);
        } else if (reason == R.string.reason_review) {
            values.put(TITLE_REASON, 1);
        } else if (reason == R.string.reason_approve) {
            values.put(TITLE_REASON, 2);
        } else if (reason == R.string.reason_approve_legally_binding) {
            values.put(TITLE_REASON, 3);
        } else if (reason == R.string.reason_accuracy_integrity) {
            values.put(TITLE_REASON, 4);
        } else if (reason == R.string.reason_sign_terms) {
            values.put(TITLE_REASON, 5);
        } else if (reason == R.string.reason_specified) {
            values.put(TITLE_REASON, 6);
        } else {
            values.put(TITLE_REASON, 0);
        }

        AppSQLite.getInstance(context).update(tableName, values, filedName, new String[]{reason + ""});

        return true;
    }

    synchronized public static boolean insertTitleData(Context context, SignatureInkItem item) {
        if (!checkInit(context)) return false;
        ContentValues values = new ContentValues();
        if (!TextUtils.isEmpty(item.title)) {
            insertRecentTitle(context, item.title);
            values.put(TITLE_NAME, item.title);
        }
        values.put(TITLE_REASON, item.reason);

        if (!TextUtils.isEmpty(item.location)) {
            values.put(TITLE_LOCATION, item.location);
        } else {
            values.put(TITLE_LOCATION, "");
        }
        values.put(TITLE_OPTIONAL_NAME, item.isNameOpen);
        values.put(TITLE_OPTIONAL_DN, item.isDistinguishedNameOpen);
        values.put(TITLE_OPTIONAL_DATE, item.isDateOpen);
        values.put(TITLE_OPTIONAL_VERSION, item.isVersionOpen);
        values.put(TITLE_OPTIONAL_LOGO, item.isLogo);
        values.put(TITLE_OPEN_REASON, item.isReasonOpen);
        values.put(TITLE_OPEN_LOCATION, item.isLocationOpen);
        values.put(TITLE_OPTIONAL_LABELS, item.isLabels);
        AppSQLite.getInstance(context).insert(TABLE_TITLE, values);

        return true;
    }
    synchronized public static boolean deleteTitle(Context context, String titleName) {
        if (!checkInit(context)) return false;
        AppSQLite.getInstance(context).delete(TABLE_TITLE, TITLE_NAME, new String[]{titleName});
        return true;
    }


    synchronized public static boolean insertRecentTitle(Context context, String name) {
        if (!checkInit(context)) return false;
        deleteRecentTitle(context);
        ContentValues values = new ContentValues();
        values.put(TITLE_NAME, name);
        AppSQLite.getInstance(context).insert(TABLE_RECENT_TITLE, values);
        return true;
    }

    synchronized public static boolean deleteRecentTitle(Context context) {
        if (!checkInit(context)) return false;
        List<String> list = new ArrayList<>();
        Cursor cursor = AppSQLite.getInstance(context).select(TABLE_RECENT_TITLE, new String[]{TITLE_NAME}, null, null, null, null, "_id desc");
        if (cursor == null) return false;
        int count = cursor.getCount();
        if (count > 0) {
            while (cursor.moveToNext()) {
                list.add(cursor.getString(cursor.getColumnIndex(TITLE_NAME)));
            }

        }
        cursor.close();
        for (int i = 0; i < list.size(); i++) {
            AppSQLite.getInstance(context).delete(TABLE_RECENT_TITLE, TITLE_NAME, new String[]{list.get(i)});
        }
        return true;
    }

    synchronized public static boolean isExistKey(Context context, String table, String
            columnsName, String key) {
        return AppSQLite.getInstance(context).isRowExist(table, columnsName, new String[]{key});
    }

    synchronized public static boolean isExistKey(Context context, String table, String key) {
        return AppSQLite.getInstance(context).isRowExist(table, NAME, new String[]{key});
    }


    synchronized public static boolean getDataCount(Context context, String table) {
        if (!checkInit(context)) return false;
        Cursor cursor = AppSQLite.getInstance(context).select(table, null, null, null, null, null, "_id desc");

        return cursor.getCount() > 0;
    }

    synchronized private static boolean checkInit(Context context) {
        if (!AppSQLite.getInstance(context).isDBOpened()) {
            AppSQLite.getInstance(context).openDB();
        }
        if (!mInit) {
            mInit = init(context);
        }
        return mInit;
    }

    private static boolean init(Context context) {
        return createModelTable(context) && createRecentTable(context) && createTitleTable(context) && createRecentTitleTable(context);
    }

    private static boolean createModelTable(Context context) {
        ArrayList<FieldInfo> tableList = new ArrayList<FieldInfo>();
        tableList.add(new FieldInfo(NAME, "VARCHAR"));
        tableList.add(new FieldInfo(LEFT, "INTEGER"));
        tableList.add(new FieldInfo(TOP, "INTEGER"));
        tableList.add(new FieldInfo(RIGHT, "INTEGER"));
        tableList.add(new FieldInfo(BOTTOM, "INTEGER"));
        tableList.add(new FieldInfo(COLOR, "INTEGER"));
        tableList.add(new FieldInfo(DIAMETER, "FLOAT"));
        tableList.add(new FieldInfo(BLOB, "BLOB"));
        tableList.add(new FieldInfo(DSG_PATH, "VARCHAR"));
        tableList.add(new FieldInfo(TITLE_NAME, "VARCHAR"));
        tableList.add(new FieldInfo(TITLE_LOCATION, "VARCHAR"));
        tableList.add(new FieldInfo(TITLE_OPEN_LOCATION, "INTEGER"));
        tableList.add(new FieldInfo(TITLE_REASON, "INTEGER"));
        tableList.add(new FieldInfo(TITLE_OPEN_REASON, "INTEGER"));
        tableList.add(new FieldInfo(TITLE_OPTIONAL_NAME, "VARCHAR"));
        tableList.add(new FieldInfo(TITLE_OPTIONAL_VERSION, "VARCHAR"));
        tableList.add(new FieldInfo(TITLE_OPTIONAL_DN, "VARCHAR"));
        tableList.add(new FieldInfo(TITLE_OPTIONAL_DATE, "VARCHAR"));
        tableList.add(new FieldInfo(CLEAR_BG, "INTEGER"));
        tableList.add(new FieldInfo(PIC, "INTEGER"));
        return AppSQLite.getInstance(context).createTable(TABLE_MODEL, tableList);
    }

    private static boolean createRecentTable(Context context) {
        ArrayList<FieldInfo> tableList = new ArrayList<FieldInfo>();
        tableList.add(new FieldInfo(NAME, "VARCHAR"));
        return AppSQLite.getInstance(context).createTable(TABLE_RECENT, tableList);
    }


    private static boolean createTitleTable(Context context) {
        ArrayList<FieldInfo> tableList = new ArrayList<FieldInfo>();
        tableList.add(new FieldInfo(TITLE_NAME, "VARCHAR"));
        tableList.add(new FieldInfo(TITLE_LOCATION, "VARCHAR"));
        tableList.add(new FieldInfo(TITLE_OPEN_LOCATION, "INTEGER"));
        tableList.add(new FieldInfo(TITLE_REASON, "INTEGER"));
        tableList.add(new FieldInfo(TITLE_OPEN_REASON, "INTEGER"));
        tableList.add(new FieldInfo(TITLE_OPTIONAL_NAME, "INTEGER"));
        tableList.add(new FieldInfo(TITLE_OPTIONAL_VERSION, "INTEGER"));
        tableList.add(new FieldInfo(TITLE_OPTIONAL_DN, "INTEGER"));
        tableList.add(new FieldInfo(TITLE_OPTIONAL_DATE, "INTEGER"));
        tableList.add(new FieldInfo(TITLE_OPTIONAL_LABELS, "INTEGER"));
        tableList.add(new FieldInfo(TITLE_OPTIONAL_LOGO, "INTEGER"));
        return AppSQLite.getInstance(context).createTable(TABLE_TITLE, tableList);
    }

    private static boolean createRecentTitleTable(Context context) {
        ArrayList<FieldInfo> tableList = new ArrayList<FieldInfo>();
        tableList.add(new FieldInfo(TITLE_NAME, "VARCHAR"));
        return AppSQLite.getInstance(context).createTable(TABLE_RECENT_TITLE, tableList);
    }
}
