/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.panzoom;

import android.app.Activity;
import android.app.Dialog;
import android.content.Context;
import android.content.Intent;
import android.content.res.ColorStateList;
import android.content.res.Configuration;
import android.graphics.Canvas;
import android.graphics.Rect;
import android.graphics.Typeface;
import android.graphics.drawable.ClipDrawable;
import android.graphics.drawable.ColorDrawable;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.LayerDrawable;
import android.net.Uri;
import android.os.Build;
import android.provider.Settings;
import android.text.TextUtils;
import android.view.Gravity;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.ScaleGestureDetector;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.SeekBar;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotEventListener;
import com.foxit.uiextensions.annots.IRedactionEventListener;
import com.foxit.uiextensions.controls.propertybar.IViewSettingsWindow;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.controls.toolbar.IBaseItem;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.drag.UIDragView;
import com.foxit.uiextensions.controls.toolbar.impl.BaseItemImpl;
import com.foxit.uiextensions.controls.toolbar.impl.TopBarImpl;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.event.PageEventListener;
import com.foxit.uiextensions.modules.panzoom.floatwindow.FloatWindowUtil;
import com.foxit.uiextensions.modules.panzoom.floatwindow.receiver.HomeKeyReceiver;
import com.foxit.uiextensions.pdfreader.ILayoutChangeListener;
import com.foxit.uiextensions.pdfreader.ILifecycleEventListener;
import com.foxit.uiextensions.pdfreader.IMainFrame;
import com.foxit.uiextensions.pdfreader.IStateChangeListener;
import com.foxit.uiextensions.pdfreader.config.ReadStateConfig;
import com.foxit.uiextensions.pdfreader.impl.LifecycleEventListener;
import com.foxit.uiextensions.pdfreader.impl.MainFrame;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;

/**
 * This module provides a tool to help to magnify and move the PDF page, it's very useful for PDF page with large size.
 */
public class PanZoomModule implements Module {
    private Context mContext = null;
    private PDFViewCtrl mPdfViewCtrl = null;
    private ViewGroup mParent = null;
    private PDFViewCtrl.UIExtensionsManager mUiExtensionsManager;

    private IViewSettingsWindow mSettingWindow;

    private boolean mIsPanZoomModule;

    private BaseBar mPZTopBar;
    private View mPZBottomBar;
    private IBaseItem mBackItem;
    private IBaseItem mTitleItem;
    private SeekBar mSeekBarItem;
    private ImageView mZoomOutItem;//out(-)
    private ImageView mZoomInItem;//in(+)
    private ImageView mPrePageItem;
    private ImageView mNextPageItem;
    private ImageView mQuitItem;
    private float mScale = 1.0f;
    private float mMaxScale = 8.0f;
    private boolean mIsConfigurationChanged = false;
    private int mLastReadState;
    Dialog mPanZoomDialog;
    //    private FloatWindowController mFloatWindowController; suyu
    private PanZoomView mPanZoomView;

    public PanZoomModule(Context context, ViewGroup parent, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        mContext = context;
        mParent = parent;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = uiExtensionsManager;

//        mFloatWindowController = new FloatWindowController(context); suyu
    }

    private void _createPanZoomView() {
        mPanZoomView = new PanZoomView(mContext, mParent, mPdfViewCtrl);
    }

    @Override
    public String getName() {
        return MODULE_NAME_PANZOOM;
    }

    @Override
    public boolean loadModule() {
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;

            mSettingWindow = uiExtensionsManager.getMainFrame().getSettingWindow();
            uiExtensionsManager.registerModule(this);
            uiExtensionsManager.registerStateChangeListener(mStatusChangeListener);
            uiExtensionsManager.getDocumentManager().registerAnnotEventListener(mAnnotEventListener);
            uiExtensionsManager.getDocumentManager().registerRedactionEventListener(mRedactionEventListener);
            uiExtensionsManager.registerLayoutChangeListener(mLayoutChangeListener);
            uiExtensionsManager.registerConfigurationChangedListener(mConfigurationChangedListener);
            uiExtensionsManager.registerThemeEventListener(mThemeEventListener);
            ((UIExtensionsManager) mUiExtensionsManager).registerLifecycleListener(mLifecycleEventListener);
        }
        mPdfViewCtrl.registerDocEventListener(docEventListener);
        mPdfViewCtrl.registerPageEventListener(mPageListener);
        mPdfViewCtrl.registerDrawEventListener(mDrawEventListener);
        mPdfViewCtrl.registerScaleGestureEventListener(mScaleListener);

//        mFloatWindowController.registerDefaultHomeKeyReceiver();
//        mFloatWindowController.registerHomeKeyEventListener(mHomeKeyEventListener); suyu
        return true;
    }

    @Override
    public boolean unloadModule() {
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            ((UIExtensionsManager) mUiExtensionsManager).unregisterStateChangeListener(mStatusChangeListener);
            ((UIExtensionsManager) mUiExtensionsManager).getDocumentManager().unregisterAnnotEventListener(mAnnotEventListener);
            ((UIExtensionsManager) mUiExtensionsManager).getDocumentManager().unregisterRedactionEventListener(mRedactionEventListener);
            ((UIExtensionsManager) mUiExtensionsManager).unregisterLayoutChangeListener(mLayoutChangeListener);
            ((UIExtensionsManager) mUiExtensionsManager).unregisterConfigurationChangedListener(mConfigurationChangedListener);
            ((UIExtensionsManager) mUiExtensionsManager).unregisterThemeEventListener(mThemeEventListener);
            ((UIExtensionsManager) mUiExtensionsManager).unregisterLifecycleListener(mLifecycleEventListener);
        }
        mPdfViewCtrl.unregisterDocEventListener(docEventListener);
        mPdfViewCtrl.unregisterPageEventListener(mPageListener);
        mPdfViewCtrl.unregisterDrawEventListener(mDrawEventListener);
        mPdfViewCtrl.unregisterScaleGestureEventListener(mScaleListener);

//        mFloatWindowController.unregisterHomeKeyEventListener(mHomeKeyEventListener);
//        mFloatWindowController.unregisterDefaultHomeKeyReceiver(); suyu
        return true;
    }

    public boolean isInPanZoomMode() {
        return mIsPanZoomModule;
    }

    private IStateChangeListener mStatusChangeListener = new IStateChangeListener() {
        @Override
        public void onStateChanged(int oldState, int newState) {
            UIExtensionsManager extensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            if (oldState != ReadStateConfig.STATE_PANZOOM && ReadStateConfig.STATE_PANZOOM == newState) {
                if (!mIsPanZoomModule) {
                    mPanZoomChangeListener.onValueChanged(IViewSettingsWindow.TYPE_PAN_ZOOM, true);
                    return;
                }

                if (extensionsManager.getDocumentManager().getCurrentAnnot() != null)
                    extensionsManager.getDocumentManager().setCurrentAnnot(null);
                mIsPanZoomModule = true;
//                if (mFloatWindowController.getFloatWindow() == null)
//                    mFloatWindowController.startFloatWindowServer(); suyu
                if (mPanZoomView == null)
                    _createPanZoomView();
            } else if (ReadStateConfig.STATE_PANZOOM == oldState && ReadStateConfig.STATE_PANZOOM != newState) {
//                if (mFloatWindowController.getFloatWindow() != null) {
//                    ((PanZoomView) mFloatWindowController.getFloatWindow()).exit();
//                    mFloatWindowController.stopFloatWindowServer();
//                    FloatWindowUtil.getInstance().setParent(null);
//                } suyu
                if (mPanZoomView != null) {
                    mPanZoomView.exit();
                    mPanZoomView = null;
                }
                mIsPanZoomModule = false;
            }

            mSettingWindow = extensionsManager.getMainFrame().getSettingWindow();
            if (mSettingWindow != null) {
                mSettingWindow.setProperty(IViewSettingsWindow.TYPE_PAN_ZOOM, mIsPanZoomModule);
            }
            onStatusChanged();
        }
    };

    private void onStatusChanged() {
        if (mPdfViewCtrl.getDoc() == null) {
            return;
        }
        if (mPZBottomBar == null)
            initPanZoomBar();

        MainFrame mainFrame = (MainFrame) ((UIExtensionsManager) mUiExtensionsManager).getMainFrame();
        if (mIsPanZoomModule) {
            if (AppDisplay.isPad()) {
                if (mDragView == null)
                    mDragView = new UIDragView(mPZBottomBar, (UIExtensionsManager) mUiExtensionsManager);
                mDragView.setVisible(!mainFrame.isShowFullScreenUI());
            } else {
                boolean isToolbarsVisible = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getMainFrame().isToolbarsVisible();
                if (isToolbarsVisible) {
                    mPZBottomBar.setVisibility(View.VISIBLE);
                    mPZTopBar.getContentView().setVisibility(View.VISIBLE);
                } else {
                    mPZBottomBar.setVisibility(View.INVISIBLE);
                    mPZTopBar.getContentView().setVisibility(View.INVISIBLE);
                }
                mPdfViewCtrl.offsetScrollBoundary(0, isToolbarsVisible ? mPZTopBar.getContentView().getHeight() : 0,
                        0, isToolbarsVisible ? -mPZBottomBar.getHeight() : 0);
                mPdfViewCtrl.postPageContainer();
            }

            mScale = mPdfViewCtrl.getZoom();
            mSeekBarItem.setProgress((int) (mScale - 1 + 0.5f));
            resetNextPageItem();
            resetPrePageItem();
        } else {
            if (AppDisplay.isPad()) {
                if (mDragView != null) {
                    mDragView.removeDragView();
                    mDragView = null;
                }
            } else {
                mPZBottomBar.setVisibility(View.INVISIBLE);
                mPZTopBar.getContentView().setVisibility(View.INVISIBLE);
            }
        }
//        if (mFloatWindowController != null && mFloatWindowController.getFloatWindow() != null
//                && mPZBottomBar != null) {
//            mFloatWindowController.getFloatWindow().setVisibility(mPZBottomBar.getVisibility());
//        } suyu
        if (mPanZoomView != null && mPZBottomBar != null) {
            mPanZoomView.setVisibility(mPZBottomBar.getVisibility());
        }

    }

    private UIDragView mDragView;

    private void addBar() {
        if (!AppDisplay.isPad()) {
            removeBar();

            RelativeLayout.LayoutParams pzTopLp = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
            pzTopLp.addRule(RelativeLayout.ALIGN_PARENT_TOP);
            mParent.addView(mPZTopBar.getContentView(), pzTopLp);

            RelativeLayout.LayoutParams pzBottomLp = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
            pzBottomLp.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
            mParent.addView(mPZBottomBar, pzBottomLp);

            mPZTopBar.getContentView().setVisibility(View.INVISIBLE);
            mPZBottomBar.setVisibility(View.INVISIBLE);
        }
    }

    private void removeBar() {
        if (AppDisplay.isPad()) {
            if (mDragView != null) {
                mDragView.removeDragView();
                mDragView = null;
            }
        } else {
            //mParent.removeView(mPZBottomBar);
            //mParent.removeView(mPZTopBar.getContentView());
            AppUtil.removeViewFromParent(mPZBottomBar);
            if (mPZTopBar != null)
                AppUtil.removeViewFromParent(mPZTopBar.getContentView());
        }
    }

    private void initPanZoomBar() {
        removeBar();

        if (!AppDisplay.isPad()) {
            mPZTopBar = new TopBarImpl(mContext);
            mBackItem = new BaseItemImpl(mContext);
            mTitleItem = new BaseItemImpl(mContext);
            initItemsImgRes();

            mPZTopBar.addView(mBackItem, BaseBar.TB_Position.Position_LT);
            mPZTopBar.addView(mTitleItem, BaseBar.TB_Position.Position_CENTER);
            mPZTopBar.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
            mPZTopBar.setStartMargin(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_text_size_16sp));
            mPZTopBar.setMiddleButtonCenter(true);
        }
        mPZBottomBar = LayoutInflater.from(mContext).inflate(R.layout.pan_zoom_bottom_layout, null, false);
        mPZBottomBar.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
        mZoomOutItem = (ImageView) mPZBottomBar.findViewById(R.id.rd_panzoom_ll_zoomout);
        mZoomInItem = (ImageView) mPZBottomBar.findViewById(R.id.rd_panzoom_ll_zoomin);
        mSeekBarItem = (SeekBar) mPZBottomBar.findViewById(R.id.rd_panzoom_ll_zoom);
        mPrePageItem = (ImageView) mPZBottomBar.findViewById(R.id.rd_panzoom_ll_prevpage);
        mNextPageItem = (ImageView) mPZBottomBar.findViewById(R.id.rd_panzoom_ll_nextpage);
        mQuitItem = (ImageView) mPZBottomBar.findViewById(R.id.panzoom_quit);
        if (AppDisplay.isPad()) {
            Drawable backgroundDrawable = AppResource.getDrawable(mContext, R.drawable.progress_background);
            int color = ThemeConfig.getInstance(mContext).getPrimaryColor();
            ColorDrawable progressDrawable = new ColorDrawable(color);
            ClipDrawable clipDrawable = new ClipDrawable(progressDrawable, Gravity.LEFT, ClipDrawable.HORIZONTAL);

            Drawable[] drawables = new Drawable[]{backgroundDrawable, clipDrawable};
            //Create layer drawables with android pre-defined ids
            LayerDrawable layerDrawable = new LayerDrawable(drawables);
            layerDrawable.setId(0, android.R.id.background);
            layerDrawable.setId(1, android.R.id.progress);

            //Set to seek bar
            mSeekBarItem.setProgressDrawable(layerDrawable);
            mQuitItem.setVisibility(View.VISIBLE);
        }
        setSeekBarTint();
        ThemeUtil.setTintList(mZoomOutItem, ThemeUtil.getPrimaryIconColor(mContext));
        ThemeUtil.setTintList(mZoomInItem, ThemeUtil.getPrimaryIconColor(mContext));
        ThemeUtil.setTintList(mPrePageItem, ThemeUtil.getPrimaryIconColor(mContext));
        ThemeUtil.setTintList(mNextPageItem, ThemeUtil.getPrimaryIconColor(mContext));
        ThemeUtil.setTintList(mQuitItem, ThemeUtil.getPrimaryIconColor(mContext));
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            mZoomInItem.setForceDarkAllowed(false);
            mZoomOutItem.setForceDarkAllowed(false);
            mPrePageItem.setForceDarkAllowed(false);
            mNextPageItem.setForceDarkAllowed(false);
            mQuitItem.setForceDarkAllowed(false);
        }
        mZoomOutItem.setEnabled(false);
        mZoomInItem.setEnabled(false);
        initItemsOnClickListener();
    }

    private void initItemsImgRes() {
        mBackItem.setText(R.string.fx_string_close);
        mBackItem.setTag(ToolbarItemConfig.ITEM_TOPBAR_BACK);
        mBackItem.setTextSize(AppDisplay.px2dp(mContext.getResources().getDimensionPixelOffset(R.dimen.ux_text_size_16sp)));
        mBackItem.setTextColor(ThemeConfig.getInstance(mContext).getPrimaryColor());

        mTitleItem.setText(AppResource.getString(mContext.getApplicationContext(), R.string.rd_panzoom_topbar_title));
        mTitleItem.setTextSize(AppDisplay.px2dp(mContext.getResources().getDimensionPixelOffset(R.dimen.ux_text_size_16sp)));
        mTitleItem.setTextColor(AppResource.getColor(mContext, R.color.t4));
        mTitleItem.setTypeface(Typeface.defaultFromStyle(Typeface.BOLD));
        mTitleItem.setEllipsize(TextUtils.TruncateAt.END);
        mTitleItem.setMaxWidth(AppDisplay.getActivityWidth() / 3);
    }

    private void resetPrePageItem() {
        if (mPdfViewCtrl.isContinuous() && mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_FACING) {
            int currentPageIndex = mPdfViewCtrl.getCurrentPage();
            if (currentPageIndex > 2) {
                mPrePageItem.setEnabled(true);
            } else if (currentPageIndex == 2) {
                Rect preLeftPageRect = mPdfViewCtrl.getPageViewRect(currentPageIndex - 2);
                Rect preRightPageRect = mPdfViewCtrl.getPageViewRect(currentPageIndex - 1);
                int height = Math.max(preLeftPageRect.height(), preRightPageRect.height());
                if (height > mParent.getHeight() / 2) {
                    mPrePageItem.setEnabled(true);
                } else {
                    mPrePageItem.setEnabled(false);
                }
            } else {
                mPrePageItem.setEnabled(false);
            }
        } else if (mPdfViewCtrl.isContinuous() && mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_COVER) {
            int currentPageIndex = mPdfViewCtrl.getCurrentPage();
            if (currentPageIndex > 1) {
                mPrePageItem.setEnabled(true);
            } else if (currentPageIndex == 1) {
                Rect preRightPageRect = mPdfViewCtrl.getPageViewRect(currentPageIndex - 1);
                if (preRightPageRect.height() > mParent.getHeight() / 2) {
                    mPrePageItem.setEnabled(true);
                } else {
                    mPrePageItem.setEnabled(false);
                }
            } else {
                mPrePageItem.setEnabled(false);
            }
        } else {
            if (mPdfViewCtrl.getCurrentPage() == 0) {
                mPrePageItem.setEnabled(false);
            } else {
                mPrePageItem.setEnabled(true);
            }
        }
    }

    private void resetNextPageItem() {
        if (mPdfViewCtrl.isContinuous() && (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_FACING
                || mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_COVER)) {

            if (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_COVER && mPdfViewCtrl.getCurrentPage() == 0) {

                if (mPdfViewCtrl.getPageCount() >= 4) {
                    mNextPageItem.setEnabled(true);
                } else if (mPdfViewCtrl.getPageCount() >= 2) {
                    Rect nextLeftPageRect = mPdfViewCtrl.getPageViewRect(mPdfViewCtrl.getCurrentPage() + 1);
                    Rect nextRightPageRect = mPdfViewCtrl.getPageViewRect(mPdfViewCtrl.getCurrentPage() + 2);
                    int height = Math.max(nextLeftPageRect.height(), nextRightPageRect.height());
                    if (height > mParent.getHeight() / 2) {
                        mNextPageItem.setEnabled(true);
                    } else {
                        mNextPageItem.setEnabled(false);
                    }
                } else {
                    mNextPageItem.setEnabled(false);
                }
            } else {
                if (mPdfViewCtrl.getCurrentPage() + 4 >= mPdfViewCtrl.getPageCount()) {
                    Rect nextLeftPageRect = mPdfViewCtrl.getPageViewRect(mPdfViewCtrl.getCurrentPage() + 2);
                    Rect nextRightPageRect = mPdfViewCtrl.getPageViewRect(mPdfViewCtrl.getCurrentPage() + 3);
                    int height = Math.max(nextLeftPageRect.height(), nextRightPageRect.height());
                    if (height > mParent.getHeight() / 2) {
                        mNextPageItem.setEnabled(true);
                    } else {
                        mNextPageItem.setEnabled(false);
                    }
                } else {
                    mNextPageItem.setEnabled(true);
                }
            }
        } else if (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_FACING
                || (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_COVER && mPdfViewCtrl.getPageCount() > 2)) {
            if (mPdfViewCtrl.getCurrentPage() + 2 >= mPdfViewCtrl.getPageCount()) {
                mNextPageItem.setEnabled(false);
            } else {
                mNextPageItem.setEnabled(true);
            }
        } else {
            if (mPdfViewCtrl.getCurrentPage() + 1 == mPdfViewCtrl.getPageCount()) {
                mNextPageItem.setEnabled(false);
            } else {
                mNextPageItem.setEnabled(true);
            }
        }
    }

    private void initItemsOnClickListener() {
        mPrePageItem.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_FACING
                        || mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_COVER) {

                    if (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_COVER && mPdfViewCtrl.getCurrentPage() == 1) {
                        int pageIndex = 0;
                        if (mPdfViewCtrl.isContinuous()) {
                            float devOffsetX = 0;
                            Rect pageRect = mPdfViewCtrl.getPageViewRect(pageIndex);
                            float devOffsetY = (float) (mParent.getHeight() - pageRect.height()) / 2;
                            mPdfViewCtrl.gotoPage(pageIndex, devOffsetX, -devOffsetY);
                        } else {
                            mPdfViewCtrl.gotoPage(pageIndex);
                        }
                    } else if (mPdfViewCtrl.getCurrentPage() - 2 >= 0) {
                        int pageIndex = mPdfViewCtrl.getCurrentPage() - 2;
                        if (mPdfViewCtrl.isContinuous()) {
                            gotoPage(pageIndex);
                        } else {
                            mPdfViewCtrl.gotoPage(pageIndex);
                        }
                    }
                } else if ((mPdfViewCtrl.getCurrentPage() - 1) >= 0) {
                    mPdfViewCtrl.gotoPrevPage();
                }
                ((UIExtensionsManager) mUiExtensionsManager).resetHideToolbarsTimer();
            }
        });
        mNextPageItem.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_FACING
                        || mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_COVER) {

                    if (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_COVER &&
                            mPdfViewCtrl.getCurrentPage() == 0 &&
                            mPdfViewCtrl.getPageCount() >= 2) {

                        int pageIndex = mPdfViewCtrl.getCurrentPage() + 1;
                        if (mPdfViewCtrl.isContinuous()) {
                            gotoPage(pageIndex);
                        } else {
                            mPdfViewCtrl.gotoPage(pageIndex);
                        }
                    } else if (mPdfViewCtrl.getCurrentPage() + 2 < mPdfViewCtrl.getPageCount()) {
                        int pageIndex = mPdfViewCtrl.getCurrentPage() + 2;
                        if (mPdfViewCtrl.isContinuous()) {
                            gotoPage(pageIndex);
                        } else {
                            mPdfViewCtrl.gotoPage(pageIndex);
                        }
                    }
                } else if ((mPdfViewCtrl.getCurrentPage() + 1) < mPdfViewCtrl.getPageCount()) {
                    mPdfViewCtrl.gotoNextPage();
                }
                ((UIExtensionsManager) mUiExtensionsManager).resetHideToolbarsTimer();
            }
        });

        View.OnClickListener quitClickListener = new View.OnClickListener() {
            @Override
            public void onClick(View v) {
//                if (mFloatWindowController.getFloatWindow() != null) {
//                    ((PanZoomView) mFloatWindowController.getFloatWindow()).onBack();
//                    mFloatWindowController.stopFloatWindowServer();
//                    FloatWindowUtil.getInstance().setParent(null); suyu
                if (mPanZoomView != null) {
                    mPanZoomView.onBack();
                    mPanZoomView = null;
                }
                mIsPanZoomModule = false;

                if (mLastReadState == ReadStateConfig.STATE_FILLSIGN)
                    ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).changeState(ReadStateConfig.STATE_FILLSIGN);
                else
                    ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).changeState(ReadStateConfig.STATE_NORMAL);
//                }
            }
        };
        if (!AppDisplay.isPad()) {
            mBackItem.setOnClickListener(quitClickListener);
        } else {
            mQuitItem.setOnClickListener(quitClickListener);
        }

        mMaxScale = mPdfViewCtrl.getMaxZoomLimit();
        mSeekBarItem.setMax((int) (mMaxScale - 1));
        mSeekBarItem.setProgress((int) (mScale - 1 + 0.5f));
        mSeekBarItem.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {
            @Override
            public void onProgressChanged(SeekBar seekBar, int progress, boolean fromUser) {
                if (progress >= 0 && progress < mMaxScale && fromUser) {
                    mScale = progress + 1;
                    mPdfViewCtrl.setZoom(mScale);
                    ((UIExtensionsManager) mUiExtensionsManager).resetHideToolbarsTimer();
                }
            }

            @Override
            public void onStartTrackingTouch(SeekBar seekBar) {
            }

            @Override
            public void onStopTrackingTouch(SeekBar seekBar) {
                if (AppDisplay.isPad()) {
                    IViewSettingsWindow settingsWindow = ((UIExtensionsManager) mUiExtensionsManager).getMainFrame().getSettingWindow();
                    if (settingsWindow != null) {
                        settingsWindow.setProperty(IViewSettingsWindow.TYPE_FIT_WIDTH, mPdfViewCtrl.getZoomMode());
                        settingsWindow.setProperty(IViewSettingsWindow.TYPE_FIT_PAGE, mPdfViewCtrl.getZoomMode());
                    }
                }
            }
        });
    }

    private void gotoPage(int pageIndex) {
        Rect pageLeftRect = mPdfViewCtrl.getPageViewRect(pageIndex);
        Rect pageRightRect = mPdfViewCtrl.getPageViewRect(pageIndex + 1);
        int height = Math.max(pageLeftRect.height(), pageRightRect.height());

        float devOffsetX = 0;
        float devOffsetY = (float) (mParent.getHeight() - height) / 2;
        mPdfViewCtrl.gotoPage(pageIndex, devOffsetX, -devOffsetY);
    }

    private final PDFViewCtrl.IDocEventListener docEventListener = new DocEventListener() {
        @Override
        public void onDocWillOpen() {
            exit();
        }

        @Override
        public void onDocOpened(PDFDoc document, int errCode) {
            if (errCode != Constants.e_ErrSuccess) return;
            if (mPZBottomBar == null)
                initPanZoomBar();

            if (!mPdfViewCtrl.isDynamicXFA()) {
                addBar();
            }
            initValue();
            if (!initViewSettingValue()) return;
            registerViewSettingListener();
        }

        @Override
        public void onDocWillClose(PDFDoc document) {
        }

        @Override
        public void onDocClosed(PDFDoc document, int errCode) {
            removeBar();
            unRegisterViewSettingListener();
        }

    };

    private void initValue() {
        // set value with the value of automatic pan zoom setting from system.
        mIsPanZoomModule = false;
    }

    private boolean initViewSettingValue() {
        mSettingWindow = ((UIExtensionsManager) mUiExtensionsManager).getMainFrame().getSettingWindow();
        if (mPdfViewCtrl.isDynamicXFA()) {
            mSettingWindow.enableBar(IViewSettingsWindow.TYPE_PAN_ZOOM, false);
            return false;
        } else {
            mSettingWindow.enableBar(IViewSettingsWindow.TYPE_PAN_ZOOM, true);
        }
        mSettingWindow.setProperty(IViewSettingsWindow.TYPE_PAN_ZOOM, mIsPanZoomModule);
        if (mIsPanZoomModule) {
            ((UIExtensionsManager) mUiExtensionsManager).changeState(ReadStateConfig.STATE_PANZOOM);
        }
        return true;
    }

    private void registerViewSettingListener() {
        mSettingWindow.registerListener(mPanZoomChangeListener);
    }

    private void unRegisterViewSettingListener() {
        mSettingWindow.unRegisterListener(mPanZoomChangeListener);
    }

    private void applyPermission(Context context) {
        String property = FloatWindowUtil.getSystemProperty("ro.build.display.id");
        if (!TextUtils.isEmpty(property) && (property.contains("flyme") || property.toLowerCase().contains("flyme"))) {
            try {
                Intent intent = new Intent("com.meizu.safe.security.SHOW_APPSEC");
//              intent.setClassName("com.meizu.safe", "com.meizu.safe.security.AppSecActivity");
                intent.putExtra("packageName", context.getPackageName());
                intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                context.startActivity(intent);
            } catch (Exception e) {
                gotoOverlayPermissionActivity(context);
            }
        } else {
            gotoOverlayPermissionActivity(context);
        }
    }

    private void gotoOverlayPermissionActivity(Context context) {
        Intent intent = new Intent(Settings.ACTION_MANAGE_OVERLAY_PERMISSION);
        intent.setData(Uri.parse("package:" + context.getPackageName()));
        intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        context.startActivity(intent);
    }

    private IViewSettingsWindow.IValueChangeListener mPanZoomChangeListener = new IViewSettingsWindow.IValueChangeListener() {

        @Override
        public void onValueChanged(int type, Object value) {
            if (type == IViewSettingsWindow.TYPE_PAN_ZOOM) {
//                onStatusChanged();

                UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
                if ((Boolean) value) {
                    mLastReadState = uiExtensionsManager.getState();
                    mIsPanZoomModule = (boolean) value;

                    mMaxScale = mPdfViewCtrl.getMaxZoomLimit();
                    int maxScale = (int) (mMaxScale - 1);
                    if (mSeekBarItem != null && mSeekBarItem.getMax() != maxScale) {
                        mSeekBarItem.setMax(maxScale);
                    }

//                    FloatWindowUtil.getInstance().setContext(mContext);
//                    FloatWindowUtil.getInstance().setParent(mParent);
//                    FloatWindowUtil.getInstance().setPdfViewCtrl(mPdfViewCtrl);
//                    final Activity activity = uiExtensionsManager.getAttachedActivity();
//                    if (Build.VERSION.SDK_INT >= 23 && !Settings.canDrawOverlays(activity)) {
//                        String message = mContext.getApplicationContext().getString(R.string.not_window_permissions_tips);
//                        String title = "";
//                        mPanZoomDialog = new AlertDialog.Builder(activity).setCancelable(true).setTitle(title)
//                                .setMessage(message)
//                                .setPositiveButton(mContext.getApplicationContext().getString(R.string.fx_turn_on),
//                                        new DialogInterface.OnClickListener() {
//                                            @Override
//                                            public void onClick(DialogInterface dialog, int which) {
//                                                applyPermission(activity);
//                                                FloatWindowUtil.getInstance().setParent(null);
//                                                dialog.dismiss();
//                                            }
//                                        }).setNegativeButton(mContext.getApplicationContext().getString(R.string.fx_turn_later),
//                                        new DialogInterface.OnClickListener() {
//
//                                            @Override
//                                            public void onClick(DialogInterface dialog, int which) {
//                                                FloatWindowUtil.getInstance().setParent(null);
//                                                dialog.dismiss();
//                                            }
//                                        }).create();
//                        mPanZoomDialog.show();
//
//                        mIsPanZoomModule = false;
//                        if (mLastReadState == ReadStateConfig.STATE_FILLSIGN)
//                            uiExtensionsManager.changeState(ReadStateConfig.STATE_FILLSIGN);
//                        else
//                            uiExtensionsManager.changeState(ReadStateConfig.STATE_NORMAL);
//                        return;
//                    } suyu

//                    if (mFloatWindowController.getFloatWindow() == null)
//                        mFloatWindowController.startFloatWindowServer();  suyu
                    if (mPanZoomView == null)
                        _createPanZoomView();
                    uiExtensionsManager.triggerDismissMenuEvent();
                    uiExtensionsManager.changeState(ReadStateConfig.STATE_PANZOOM);
                    IMainFrame mainFrame = uiExtensionsManager.getMainFrame();
                    mainFrame.hideSettingWindow();
                    if (!mainFrame.isToolbarsVisible())
                        mainFrame.showToolbars();
                    if (mainFrame.getPanelManager().isShowingPanel())
                        mainFrame.getPanelManager().hidePanel();
                    if (uiExtensionsManager.getDocumentManager().getCurrentAnnot() != null) {
                        uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
                    }
                    mPdfViewCtrl.postDelayed(new Runnable() {
                        @Override
                        public void run() {
                            mPdfViewCtrl.postPageContainer();
                        }
                    }, 100);
                } else {
                    exit();
                }
            } else if (type == IViewSettingsWindow.TYPE_DAY) {
//                PanZoomView panZoomView = (PanZoomView) mFloatWindowController.getFloatWindow();
//                if (null != panZoomView) {
//                    reDrawPanZoomView(mPdfViewCtrl.getCurrentPage());
//                } suyu
                if (mPanZoomView != null) {
                    onDaynightModeChanged(mPdfViewCtrl.getCurrentPage());
                }
            }
        }

        @Override
        public int getType() {
            return IViewSettingsWindow.TYPE_PAN_ZOOM;
        }
    };

    public boolean onKeyDown(int keyCode, KeyEvent event) {
        if (keyCode == KeyEvent.KEYCODE_BACK) {
            return exit();
        }
        return false;
    }

    public boolean exit() {
        if (!mIsPanZoomModule) return false;
//        if (mFloatWindowController.getFloatWindow() != null) {
//            ((PanZoomView) mFloatWindowController.getFloatWindow()).exit();
//            mFloatWindowController.stopFloatWindowServer();
//            FloatWindowUtil.getInstance().setParent(null);
//        } suyu
        if (mPanZoomView != null) {
            mPanZoomView.exit();
            mPanZoomView = null;
        }

        mIsPanZoomModule = false;
        if (mLastReadState == ReadStateConfig.STATE_FILLSIGN)
            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).changeState(ReadStateConfig.STATE_FILLSIGN);
        else
            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).changeState(ReadStateConfig.STATE_NORMAL);
        return true;
    }

    private void onDaynightModeChanged(int pageIndex){
        if (mPanZoomView != null) {
            if (mPanZoomView.isPanZoomRectMoving()) return;
            mPanZoomView.reDrawPanZoomView(pageIndex);
            mPanZoomView.updateTheme();
        }
    }

    private void reDrawPanZoomView(int pageIndex) {
//        if (mFloatWindowController.getFloatWindow() != null) {
//            PanZoomView panZoomView = (PanZoomView) mFloatWindowController.getFloatWindow();
//            if (panZoomView.isPanZoomRectMoving()) return;
//            panZoomView.reDrawPanZoomView(pageIndex);
//        } suyu
        if (mPanZoomView != null) {
            if (mPanZoomView.isPanZoomRectMoving()) return;
            mPanZoomView.reDrawPanZoomView(pageIndex);
        }
    }

    private PDFViewCtrl.IPageEventListener mPageListener = new PageEventListener() {
        @Override
        public void onPageVisible(int index) {
        }

        @Override
        public void onPageInvisible(int index) {

        }

        @Override
        public void onPageChanged(int oldPageIndex, int curPageIndex) {
            if (mPZBottomBar == null)
                return;
            resetPrePageItem();
            resetNextPageItem();
            reDrawPanZoomView(curPageIndex);
        }

        @Override
        public void onPageJumped() {
            if (mPZBottomBar == null)
                return;
            resetNextPageItem();
            resetPrePageItem();
            reDrawPanZoomView(mPdfViewCtrl.getCurrentPage());
        }

        @Override
        public void onPagesWillRemove(int[] pageIndexes) {

        }

        @Override
        public void onPageWillMove(int index, int dstIndex) {

        }

        @Override
        public void onPagesWillRotate(int[] pageIndexes, int rotation) {

        }

        @Override
        public void onPagesRemoved(boolean success, int[] pageIndexes) {

        }

        @Override
        public void onPageMoved(boolean success, int index, int dstIndex) {

        }

        @Override
        public void onPagesRotated(boolean success, int[] pageIndexes, int rotation) {

        }

        @Override
        public void onPagesInserted(boolean success, int dstIndex, int[] pageRanges) {

        }

        @Override
        public void onPagesWillInsert(int dstIndex, int[] pageRanges) {

        }
    };

    private AnnotEventListener mAnnotEventListener = new AnnotEventListener() {
        @Override
        public void onAnnotAdded(PDFPage page, Annot annot) {
            try {
//                PanZoomView panZoomView = (PanZoomView) mFloatWindowController.getFloatWindow();
//                if (null != panZoomView && page.getIndex() == panZoomView.getCurPageIndex()) {
//                    reDrawPanZoomView(page.getIndex());
//                } suyu
                if (mPanZoomView != null && page.getIndex() == mPanZoomView.getCurPageIndex()) {
                    reDrawPanZoomView(page.getIndex());
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }

        @Override
        public void onAnnotWillDelete(PDFPage page, Annot annot) {

        }

        @Override
        public void onAnnotDeleted(PDFPage page, Annot annot) {
            try {
//                PanZoomView panZoomView = (PanZoomView) mFloatWindowController.getFloatWindow();
//                if (null != panZoomView && page.getIndex() == panZoomView.getCurPageIndex()) {
//                    reDrawPanZoomView(page.getIndex());
//                } suyu
                if (mPanZoomView != null && page.getIndex() == mPanZoomView.getCurPageIndex()) {
                    reDrawPanZoomView(page.getIndex());
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }

        @Override
        public void onAnnotModified(PDFPage page, Annot annot) {
            try {
//                PanZoomView panZoomView = (PanZoomView) mFloatWindowController.getFloatWindow();
//                if (null != panZoomView && page.getIndex() == panZoomView.getCurPageIndex()) {
//                    reDrawPanZoomView(page.getIndex());
//                } suyu
                if (mPanZoomView != null && page.getIndex() == mPanZoomView.getCurPageIndex()) {
                    reDrawPanZoomView(page.getIndex());
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }

        @Override
        public void onAnnotChanged(Annot lastAnnot, Annot currentAnnot) {

        }
    };

    private IRedactionEventListener mRedactionEventListener = new IRedactionEventListener() {
        @Override
        public void onAnnotWillApply(PDFPage page, Annot annot) {

        }

        @Override
        public void onAnnotApplied(PDFPage page, Annot annot) {
            try {
                reDrawPanZoomView(page.getIndex());
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
    };

    private PDFViewCtrl.IDrawEventListener mDrawEventListener = new PDFViewCtrl.IDrawEventListener() {
        @Override
        public void onDraw(int pageIndex, Canvas canvas) {
//            if (mFloatWindowController.getFloatWindow() != null) {
//                PanZoomView panZoomView = (PanZoomView) mFloatWindowController.getFloatWindow();
//                if (panZoomView.isPanZoomRectMoving()) return;
//                panZoomView.reCalculatePanZoomRect(pageIndex);
//                if (mSeekBarItem != null)
//                    mSeekBarItem.setProgress((int) (mPdfViewCtrl.getZoom() - 1 + 0.5));
//            } suyu
            if (mPanZoomView != null) {
                if (mPanZoomView.isPanZoomRectMoving()) return;
                mPanZoomView.reCalculatePanZoomRect(pageIndex);
                if (mSeekBarItem != null)
                    mSeekBarItem.setProgress((int) (mPdfViewCtrl.getZoom() - 1 + 0.5));
            }
        }
    };

    private PDFViewCtrl.IScaleGestureEventListener mScaleListener = new PDFViewCtrl.IScaleGestureEventListener() {
        @Override
        public boolean onScale(ScaleGestureDetector detector) {
            return false;
        }

        @Override
        public boolean onScaleBegin(ScaleGestureDetector detector) {
            return false;
        }

        @Override
        public void onScaleEnd(ScaleGestureDetector detector) {
            mScale = mPdfViewCtrl.getZoom();

            if (mSeekBarItem == null) return;
            mSeekBarItem.setProgress((int) (mScale - 1 + 0.5f));
            if (mPdfViewCtrl.isContinuous()) {

                if (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_FACING) {
                    if (mPdfViewCtrl.getCurrentPage() == 2)
                        resetPrePageItem();

                    if (mPdfViewCtrl.getPageCount() > 2 &&
                            mPdfViewCtrl.getCurrentPage() + 2 < mPdfViewCtrl.getPageCount() &&
                            mPdfViewCtrl.getCurrentPage() + 4 >= mPdfViewCtrl.getPageCount())
                        resetNextPageItem();
                } else if (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_COVER && mPdfViewCtrl.getCurrentPage() == 1) {
                    if (mPdfViewCtrl.getCurrentPage() == 1)
                        resetPrePageItem();

                    if (mPdfViewCtrl.getPageCount() >= 2 && mPdfViewCtrl.getCurrentPage() + 4 >= mPdfViewCtrl.getPageCount()) {
                        if (mPdfViewCtrl.getCurrentPage() == 0 || mPdfViewCtrl.getCurrentPage() + 2 < mPdfViewCtrl.getPageCount()) {
                            resetNextPageItem();
                        }
                    }

                }
            }
        }
    };

    private HomeKeyReceiver.IHomeKeyEventListener mHomeKeyEventListener = new HomeKeyReceiver.IHomeKeyEventListener() {
        @Override
        public void onHomeKeyPressed() {
            exit();
        }
    };

    public boolean onTouchEvent(int pageIndex, MotionEvent motionEvent) {
        if (mIsPanZoomModule) {
//            PanZoomView panZoomView = (PanZoomView) mFloatWindowController.getFloatWindow();
//            if (panZoomView != null) {
//                panZoomView.onTouchEvent(pageIndex, motionEvent);
//            } suyu
            if (mPanZoomView != null) {
                mPanZoomView.onTouchEvent(pageIndex, motionEvent);
            }
        }
        return false;
    }

    private ILayoutChangeListener mLayoutChangeListener = new ILayoutChangeListener() {
        @Override
        public void onLayoutChange(View v, int newWidth, int newHeight, int oldWidth, int oldHeight) {
//            if (mFloatWindowController.getFloatWindow() != null) {
//                if (mIsConfigurationChanged && (newWidth != oldWidth || newHeight != oldHeight)) {
//                    mIsConfigurationChanged = false;
//                    PanZoomView panZoomView = (PanZoomView) mFloatWindowController.getFloatWindow();
//                    panZoomView.onLayoutChange(v, newWidth, newHeight, oldWidth, oldHeight);
//                }
//            } suyu
            if (mPanZoomView != null) {
                if (mIsConfigurationChanged && (newWidth != oldWidth || newHeight != oldHeight)) {
                    mIsConfigurationChanged = false;
                    mPanZoomView.onLayoutChange(v, newWidth, newHeight, oldWidth, oldHeight);
                }
            }
        }
    };

    private UIExtensionsManager.ConfigurationChangedListener mConfigurationChangedListener = new UIExtensionsManager.ConfigurationChangedListener() {
        @Override
        public void onConfigurationChanged(Configuration newConfig) {
            mIsConfigurationChanged = true;
            if (mTitleItem != null) {
                mTitleItem.setMaxWidth(AppDisplay.getActivityWidth() / 3);
            }
        }
    };

    private ILifecycleEventListener mLifecycleEventListener = new LifecycleEventListener() {
        @Override
        public void onHiddenChanged(boolean hidden) {
            if (hidden) {
                onKeyDown(KeyEvent.KEYCODE_BACK, null);
            }
        }

        @Override
        public void onStop(Activity act) {
//            if (mIsPanZoomModule){
//                exit();
//            } suyu
        }

    };

    private IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            initPanZoomBar();
            if (AppDisplay.isPad()) {
                if (mDragView != null)
                    mDragView.removeDragView();
            } else {
                mParent.removeView(mPZBottomBar);
                mParent.removeView(mPZTopBar.getContentView());
            }
            if (mPanZoomDialog != null) {
                mPanZoomDialog.dismiss();
                mPanZoomDialog = null;
            }
            boolean preItemEnabled = mPrePageItem.isEnabled();
            boolean nextItemEnabled = mNextPageItem.isEnabled();
            if (mPdfViewCtrl.isDynamicXFA()) return;
            mPrePageItem.setEnabled(preItemEnabled);
            mNextPageItem.setEnabled(nextItemEnabled);
            addBar();
            if (((UIExtensionsManager) mUiExtensionsManager).getState() == ReadStateConfig.STATE_PANZOOM) {
                if (AppDisplay.isPad()) {
                    mDragView = new UIDragView(mPZBottomBar, (UIExtensionsManager) mUiExtensionsManager);
                } else {
                    if (((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getMainFrame().isToolbarsVisible()) {
                        mPZBottomBar.setVisibility(View.VISIBLE);
                        mPZTopBar.getContentView().setVisibility(View.VISIBLE);
                    } else {
                        mPZBottomBar.setVisibility(View.INVISIBLE);
                        mPZTopBar.getContentView().setVisibility(View.INVISIBLE);
                    }
                }

//                PanZoomView panZoomView = (PanZoomView) mFloatWindowController.getFloatWindow();
//                if (panZoomView != null)
//                    panZoomView.updateTheme(); suyu
                if (mPanZoomView != null) {
                    mPanZoomView.updateTheme();
                }
            } else {
                if (AppDisplay.isPad()) {
                    if (mDragView != null) {
                        mDragView.removeDragView();
                        mDragView = null;
                    }
                } else {
                    mPZBottomBar.setVisibility(View.INVISIBLE);
                    mPZTopBar.getContentView().setVisibility(View.INVISIBLE);
                }
            }
        }
    };

    private void setSeekBarTint() {
        if (mSeekBarItem == null) return;
        int color = ThemeConfig.getInstance(mContext).getPrimaryColor();
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            mSeekBarItem.setProgressTintList(ColorStateList.valueOf(color));
        } else {
            Drawable drawable = mSeekBarItem.getProgressDrawable();
            drawable = AppResource.createTintedDrawable(drawable, ColorStateList.valueOf(color));
            mSeekBarItem.setProgressDrawable(drawable);
        }
    }
}
