/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.panel.outline;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.content.res.ColorStateList;
import android.util.SparseBooleanArray;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.ViewGroup;
import android.widget.CheckBox;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.recyclerview.widget.DefaultItemAnimator;
import androidx.recyclerview.widget.ItemTouchHelper;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.Bookmark;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.actions.Destination;
import com.foxit.uiextensions.IInteractionEventListener;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.common.UIBtnImageView;
import com.foxit.uiextensions.browser.PullListView;
import com.foxit.uiextensions.browser.adapter.SuperAdapter;
import com.foxit.uiextensions.browser.adapter.viewholder.SuperViewHolder;
import com.foxit.uiextensions.controls.dialog.MatchDialog;
import com.foxit.uiextensions.controls.dialog.UIDialog;
import com.foxit.uiextensions.controls.dialog.UIMatchDialog;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.controls.panel.PanelSpec;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.event.PageEventListener;
import com.foxit.uiextensions.modules.panel.IPanelManager;
import com.foxit.uiextensions.modules.panel.PanelSearchView;
import com.foxit.uiextensions.modules.panel.bean.BaseBean;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.IResult;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * The module enable user to view all the outlines on the PDF document, and jump to the page associated with the specified outline.
 */
public class OutlineModule implements Module, PanelSpec {
    private Context mContext;

    private View mOutlineTopView;
    private PanelSearchView mSearchView;
    private View mContentView;
    private View mBackView;
    private UIBtnImageView mBackUpOutline;
    private RecyclerView mListView;
    private OutlineSupport.OutlineAdapter mOutlineAdapter;
    private TextView mNoInfoTv;
    private ImageView mNoInfoIv;
    private TextView mEditViewTv;
    private TextView mCloseView;
    private RelativeLayout mTopContainer;
    private PullListView mPullListView;
    private View mBottomView;
    private TextView mSelectedAllTv;
    private ImageView mDeleteIv;

    private OutlineSupport mOutlineSupport;
    private PDFViewCtrl mPdfViewCtrl;
    private UIExtensionsManager mUiExtensionsManager;
    private ArrayList<OutlineNode> mOutlineArr = new ArrayList<>();

    private boolean mIsNeedRefreshOutline = false;
    private int mLevel = 0;
    private int mState = OutlineSupport.STATE_NORMAL;
    private TextView mTitleView;
    private ViewGroup mRlContent;
    private UIBtnImageView mBackRoot;
    private TextView mAddView;

    private IInteractionEventListener mInteractionEventListener = new IInteractionEventListener() {
        @Override
        public boolean onKeyDown(Activity act, int keyCode, KeyEvent event) {
            if (mOutlineAdapter == null) return false;
            if (keyCode == KeyEvent.KEYCODE_BACK) {
                if (mOutlineAdapter.isSearchState()) {
                    switchSearchState(false);
                    return true;
                }
                if (mOutlineAdapter.isEditState()) {
                    switchEditState(false);
                    return true;
                }
            }
            return false;
        }
    };

    private IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            if (mOutlineTopView == null) return;
            dismissAllDialog();
            mSearchView.onThemeColorChanged();
            mCloseView.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
            mEditViewTv.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
            mNoInfoIv.setColorFilter(ThemeConfig.getInstance(mContext).getPrimaryColor());
            mNoInfoTv.setTextColor(AppResource.getColor(mContext, R.color.t2));
            if (mMoveNoOutlineView != null){
                ((TextView)mMoveNoOutlineView.findViewById(R.id.no_outline_tv)).setTextColor(AppResource.getColor(mContext, R.color.t2));
                ((ImageView)mMoveNoOutlineView.findViewById(R.id.no_outline_iv)).setColorFilter(ThemeConfig.getInstance(mContext).getPrimaryColor());
            }
            mOutlineAdapter.notifyUpdateData();
            mTopContainer.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
            mTitleView.setTextColor(AppResource.getColor(mContext, R.color.t4));
            mRlContent.setBackgroundColor(AppResource.getColor(mContext, R.color.b1));

            mSelectedAllTv.setTextColor(AppResource.getColor(mContext, R.color.fx_menu_text_selector));

            ThemeUtil.setTintList(mDeleteIv, ThemeUtil.getPrimaryIconColor(mContext));
            ThemeUtil.setTintList(mAddIv, ThemeUtil.getPrimaryIconColor(mContext));
            ThemeUtil.setTintList(mMoveIv, ThemeUtil.getPrimaryIconColor(mContext));

            mBottomView.setBackground(AppResource.getDrawable(mContext, R.drawable.bottom_menu_bg));
            mBackView.setBackgroundColor(AppResource.getColor(mContext, R.color.b1));
            mBackUpOutline.setColorStateList(ThemeUtil.getPrimaryIconColor(mContext));
            mBackRoot.setColorStateList(ThemeUtil.getPrimaryIconColor(mContext));

            mOutlineBottomView.setBackground(AppResource.getDrawable(mContext, R.drawable.bottom_menu_bg));
            mAddView.setTextColor(AppResource.getColor(mContext, R.color.fx_menu_text_selector));
        }
    };

    private PDFViewCtrl.IDocEventListener mDocEventListener = new DocEventListener() {
        @Override
        public void onDocWillOpen() {
        }

        @Override
        public void onDocOpened(PDFDoc pdfDoc, int errCode) {
            if (pdfDoc == null || errCode != Constants.e_ErrSuccess || mOutlineTopView == null)
                return;
            initData();

            IPanelManager panelManager = mUiExtensionsManager.getPanelManager();
            if (panelManager.isShowingPanel()
                    && panelManager.getPanelHost().getCurrentSpec() == OutlineModule.this) {
                panelManager.getPanelHost().setCurrentSpec(getPanelType());
            }
        }

        @Override
        public void onDocWillClose(PDFDoc pdfDoc) {
            mIsNeedRefreshOutline = false;
        }

        @Override
        public void onDocClosed(PDFDoc pdfDoc, int i) {
            if (mOutlineTopView == null) return;
            mLevel = 0;
            mOutlineArr.clear();
            mOutlineSupport.clear();
            mPullListView.hideHeaderView();
        }

    };

    private void initData() {
        if (mOutlineSupport != null) {
            mOutlineSupport.init();
        }
    }

    public OutlineModule(Context context, ViewGroup parent, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        mPdfViewCtrl = pdfViewCtrl;
        mContext = context;
        mUiExtensionsManager = (UIExtensionsManager) uiExtensionsManager;
    }

    @Override
    public String getName() {
        return Module.MODULE_NAME_OUTLINE;
    }

    @Override
    public boolean loadModule() {
        mUiExtensionsManager.registerThemeEventListener(mThemeEventListener);
        mUiExtensionsManager.registerInteractionListener(mInteractionEventListener);
        mUiExtensionsManager.registerModule(this);
        mPdfViewCtrl.registerDocEventListener(mDocEventListener);
        mPdfViewCtrl.registerPageEventListener(mPageEventListener);
        mUiExtensionsManager.getPanelManager().registerPanelEventListener(mPanelEventListener);
        return true;
    }

    private final IPanelManager.OnPanelEventListener mPanelEventListener = new IPanelManager.OnPanelEventListener() {
        @Override
        public void onPanelOpened() {
            if (mOutlineTopView != null) return;
            initView();
            initOutlineSupport();
            mIsNeedRefreshOutline = true;
        }

        @Override
        public void onPanelClosed() {

        }
    };

    public void AI_addOutline(String value, int pageindex) {
        if (mUiExtensionsManager.getPanelManager().getPanelHost().getSpec(getPanelType()) == null) {
            initView();
            initOutlineSupport();
            mIsNeedRefreshOutline = true;
        }

        addOutline(value, pageindex);
        mUiExtensionsManager.getDocumentManager().setDocModified(true);
    }

    @Override
    public boolean unloadModule() {
        mUiExtensionsManager.getPanelManager().removePanel(this);
        mUiExtensionsManager.getPanelManager().unregisterPanelEventListener(mPanelEventListener);

        mUiExtensionsManager.unregisterThemeEventListener(mThemeEventListener);
        mUiExtensionsManager.unregisterInteractionListener(mInteractionEventListener);
        mPdfViewCtrl.unregisterDocEventListener(mDocEventListener);
        mPdfViewCtrl.unregisterPageEventListener(mPageEventListener);
        mDocEventListener = null;
        mPageEventListener = null;
        mOutlineSupport = null;
        return true;
    }

    private void initOutlineSupport() {
        mOutlineSupport = new OutlineSupport(mContext, mPdfViewCtrl, mBackUpOutline) {

            @Override
            public void outlineBindingListView(OutlineAdapter adapter) {
                mOutlineAdapter = adapter;
                mListView.setAdapter(mOutlineAdapter);
            }

            @Override
            public void setShowOutline(ArrayList<OutlineNode> outlineList) {
                mOutlineArr.clear();
                mOutlineArr.addAll(outlineList);
            }

            @Override
            public void updateUI(int level, int state) {
                mLevel = level;
                mState = state;
                updateButtonState();
            }
        };

        mOutlineSupport.setItemClickListener(new OutlineSupport.OnItemClickListener() {
            @Override
            public void onItemClick(int position, OutlineNode outlineItem) {
                if (mOutlineAdapter.isEditState()) {
                    mDeleteIv.setEnabled(mOutlineAdapter.getEditItems().size() > 0);
                    mMoveIv.setEnabled(mDeleteIv.isEnabled());
                    if (mOutlineAdapter.isSelectedAll())
                        mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_deselect_all));
                    else
                        mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_select_all));
                }
            }
        });
    }

    private void initView() {
        mOutlineTopView = createTopView();
        mSearchView = createSearchView();
        mContentView = createContentView();
        mUiExtensionsManager.getPanelManager().addPanel(this);
    }

    private View createTopView() {
        View topView = View.inflate(mContext, R.layout.panel_topbar_layout, null);
        mTopContainer = topView.findViewById(R.id.panel_rl_top_container);
        mTopContainer.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
        mCloseView = topView.findViewById(R.id.panel_close_tv);
        mCloseView.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        if (AppDevice.isChromeOs(mPdfViewCtrl.getAttachedActivity())) {
            mCloseView.setVisibility(View.VISIBLE);
        } else {
            if (AppDisplay.isPad())
                mCloseView.setVisibility(View.INVISIBLE);
            else
                mCloseView.setVisibility(View.VISIBLE);
        }
        mCloseView.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mUiExtensionsManager.getPanelManager().isShowingPanel())
                    mUiExtensionsManager.getPanelManager().hidePanel();
            }
        });
        mTitleView = topView.findViewById(R.id.panel_title_tv);
        mTitleView.setText(AppResource.getString(mContext, R.string.rv_panel_phone_top_outline));

        mEditViewTv = topView.findViewById(R.id.panel_edit_tv);
        mEditViewTv.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        mEditViewTv.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mOutlineAdapter == null) return;
                switchEditState(!mOutlineAdapter.isEditState());
            }
        });
        return topView;
    }

    private String mLastSearchText;

    private PanelSearchView createSearchView() {
        final PanelSearchView searchView = new PanelSearchView(mContext);
        searchView.setSearchEventListener(new PanelSearchView.OnSearchEventListener() {
            @Override
            public void onQueryTextChange(String newText) {
                if (!AppUtil.isEqual(mLastSearchText, newText)) {
                    List<OutlineNode> searchItems = mOutlineAdapter.getSearchItems();
                    List<OutlineNode> outlineItems = mOutlineSupport.getAllOutLineItems();
                    searchItems.clear();
                    mLastSearchText = newText.toLowerCase();

                    if (AppUtil.isEmpty(mLastSearchText)) {
                        searchItems.addAll(outlineItems);
                    } else {
                        for (int index = 0; index < outlineItems.size(); index++) {
                            String content = outlineItems.get(index).mTitle.toLowerCase();
                            if (content.contains(mLastSearchText)) {
                                boolean inserted = false;
                                for (int j = 0; j < searchItems.size(); j++) {
                                    if (outlineItems.get(index).mPageIndex < searchItems.get(j).mPageIndex) {
                                        searchItems.add(j, outlineItems.get(index));
                                        inserted = true;
                                        break;
                                    }
                                }
                                if (!inserted) {
                                    searchItems.add(outlineItems.get(index));
                                }
                            }
                        }
                    }
                    updateButtonState();
                }
            }

            @Override
            public void onFocusChange(View v, boolean hasFocus) {
                if (hasFocus) {
                    switchSearchState(true);
                    mOutlineSupport.loadAllOutlines(new IResult<Void, Void, List<OutlineNode>>() {
                        @Override
                        public void onResult(boolean success, Void p1, Void p2, List<OutlineNode> outlineItems) {
                            if (AppUtil.isEmpty(searchView.getSearchText())) {
                                mOutlineAdapter.getSearchItems().clear();
                                mOutlineAdapter.getSearchItems().addAll(outlineItems);
                                updateButtonState();
                            }
                        }
                    });
                } else {
                    if (AppUtil.isEmpty(mSearchView.getSearchText())) {
                        switchSearchState(false);
                    } else {
                        updateButtonState();
                    }
                }
            }

            @Override
            public void onCancel() {
                if (mPullListView != null)
                    mPullListView.hideHeaderView();
                mState = OutlineSupport.STATE_NORMAL;
                mOutlineArr.clear();
                mOutlineArr.addAll(mOutlineSupport.getShowOutLineList());
                mOutlineSupport.cancelLoadAllOutlinesTask();
                switchSearchState(false);
            }
        });

        return searchView;
    }

    private UITextEditDialog mClearAllDialog;
    private View mOutlineBottomView;

    private View createContentView() {
        View contentView = View.inflate(mContext, R.layout.panel_outline_contentview, null);
        mRlContent = contentView.findViewById(R.id.panel_outline_content_root);
        mNoInfoTv = contentView.findViewById(R.id.panel_no_outline_tv);
        mNoInfoTv.setTextColor(AppResource.getColor(mContext, R.color.t2));
        mNoInfoIv = contentView.findViewById(R.id.panel_no_outline_iv);
        mNoInfoIv.setColorFilter(ThemeConfig.getInstance(mContext).getPrimaryColor());
        mBottomView = contentView.findViewById(R.id.outline_panel_bottom_tool_bar);
        mOutlineBottomView = contentView.findViewById(R.id.panel_outline_bottom);
        mDeleteIv = contentView.findViewById(R.id.panel_bottom_delete_iv);
        mAddIv = contentView.findViewById(R.id.panel_bottom_add_iv);
        mMoveIv = contentView.findViewById(R.id.panel_bottom_move_iv);
        ThemeUtil.setTintList(mDeleteIv, ThemeUtil.getPrimaryIconColor(mContext));
        ThemeUtil.setTintList(mAddIv, ThemeUtil.getPrimaryIconColor(mContext));
        ThemeUtil.setTintList(mMoveIv, ThemeUtil.getPrimaryIconColor(mContext));
        mDeleteIv.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                Context context = mUiExtensionsManager.getAttachedActivity();
                if (context == null) return;
                if (mClearAllDialog == null) {
                    mClearAllDialog = new UITextEditDialog(context, UIDialog.NO_INPUT);
                    mClearAllDialog.setTitle(AppResource.getString(mContext, R.string.menu_more_confirm));
                    mClearAllDialog.getOKButton().setOnClickListener(new View.OnClickListener() {
                        @Override
                        public void onClick(View v) {
                            mClearAllDialog.dismiss();
                            removeSelectedOutlines();
                        }
                    });
                }
                mClearAllDialog.getPromptTextView().setText(mOutlineAdapter.isSelectedAll() ?
                        AppResource.getString(mContext, R.string.rd_panel_clear_outline)
                        : AppResource.getString(mContext, R.string.rd_panel_delete_outline));
                mClearAllDialog.show();
            }
        });
        mSelectedAllTv = contentView.findViewById(R.id.panel_bottom_select_all_tv);
        mSelectedAllTv.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                boolean isSelectedAll = mOutlineAdapter.isSelectedAll();
                if (isSelectedAll) {
                    mDeleteIv.setEnabled(false);
                    mOutlineAdapter.selectAll(false);
                    mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_select_all));
                } else {
                    mDeleteIv.setEnabled(true);
                    mOutlineAdapter.selectAll(true);
                    mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_deselect_all));
                }
                mMoveIv.setEnabled(mDeleteIv.isEnabled());
            }
        });

        mPullListView = contentView.findViewById(R.id.panel_outline_lv);
        mPullListView.addHeaderView(mSearchView);

        mBackView = View.inflate(mContext, R.layout.panel_outline_back, null);
        mBackView.setVisibility(View.GONE);
        mBackUpOutline = mBackView.findViewById(R.id.back_up_level_list_iv);
        mBackUpOutline.setColorStateList(ThemeUtil.getPrimaryIconColor(mContext));
        mBackRoot = mBackView.findViewById(R.id.back_root_list_iv);
        mBackRoot.setColorStateList(ThemeUtil.getPrimaryIconColor(mContext));
        mPullListView.addTopView(mBackView);
        mBackRoot.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                mOutlineAdapter.getEditItems().clear();
                mOutlineSupport.gotoRootOutlineNode();
            }
        });
        mListView = mPullListView.getRecyclerView();
        mListView.setLayoutManager(new LinearLayoutManager(mContext, LinearLayoutManager.VERTICAL, false));
        mListView.setItemAnimator(new DefaultItemAnimator());
        mAddView = contentView.findViewById(R.id.panel_add_outline);
        mAddView.setTextColor(ThemeUtil.getEnableTextColor(mContext));
        View.OnClickListener addOutlineClickListener = new OnClickListener() {
            @Override
            public void onClick(View v) {
                showAddDialog();
            }
        };
        mAddView.setOnClickListener(addOutlineClickListener);
        mAddIv.setOnClickListener(addOutlineClickListener);
        mMoveIv.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                showMoveOutlineDialog();
            }
        });
        mAddIv.setVisibility(View.VISIBLE);
        mMoveIv.setVisibility(View.VISIBLE);
        mOutlineBottomView.setVisibility(View.VISIBLE);
        return contentView;
    }

    private void removeSelectedOutlines() {
        mOutlineSupport.removeBookmark(mOutlineAdapter.getEditItems(), new IResult<Void, Integer, ArrayList<OutlineNode>>() {
            @Override
            public void onResult(boolean success, Void p1, Integer level, ArrayList<OutlineNode> showOutlineItems) {
                mLevel = level;
                mOutlineArr.clear();
                mOutlineArr.addAll(showOutlineItems);
                if (level < 0 || mOutlineArr.isEmpty())
                    switchEditState(false);
                else
                    updateButtonState();
                mUiExtensionsManager.getDocumentManager().setDocModified(true);
            }
        });
    }

    private void updateButtonState() {
        if (mOutlineAdapter == null) return;

        boolean canAssemble = mUiExtensionsManager.getDocumentManager().canAssemble() && mUiExtensionsManager.isEnableModification();
        if (mOutlineAdapter.isSearchState()) {
            mOutlineAdapter.setSearchPattern(mSearchView.getSearchText().toLowerCase());

            if (mState == OutlineSupport.STATE_LOADING) {
//                mSearchView.getEditText().setEnabled(false);
                mNoInfoIv.setVisibility(View.GONE);
                mNoInfoTv.setVisibility(View.VISIBLE);
                mNoInfoTv.setText(mContext.getApplicationContext().getString(R.string.rv_panel_annot_loading_start));
            } else {
//                mSearchView.getEditText().setEnabled(true);
                mNoInfoTv.setText(mContext.getApplicationContext().getString(R.string.fx_no_search_result));
                mNoInfoTv.setVisibility(mOutlineAdapter.getSearchItems().size() == 0 ? View.VISIBLE : View.GONE);
                mNoInfoIv.setImageResource(R.drawable.panel_no_search_result);
                mNoInfoIv.setVisibility(mOutlineAdapter.getSearchItems().size() == 0 ? View.VISIBLE : View.GONE);
            }
        } else if (mOutlineArr.size() > 0) {
            mBackView.setVisibility(mLevel > 0 ? View.VISIBLE : View.GONE);
            mEditViewTv.setEnabled(canAssemble);
            mSelectedAllTv.setEnabled(mEditViewTv.isEnabled());
            mNoInfoTv.setVisibility(View.GONE);
            mNoInfoIv.setVisibility(View.GONE);

            if (mOutlineAdapter.isEditState()) {
                mPullListView.setScrollable(false);
                mDeleteIv.setEnabled(mOutlineAdapter.getEditItems().size() > 0);
                if (mOutlineAdapter.isSelectedAll())
                    mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_deselect_all));
                else
                    mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_select_all));
            } else {
                mPullListView.setScrollable(true);
            }
        } else {
            mBackView.setVisibility(mLevel > 0 ? View.VISIBLE : View.GONE);
            if (mState == OutlineSupport.STATE_LOADING) {
                mNoInfoIv.setVisibility(View.GONE);
                mNoInfoTv.setVisibility(View.VISIBLE);
                mNoInfoTv.setText(mContext.getApplicationContext().getString(R.string.rv_panel_annot_loading_start));
            } else {
                mNoInfoTv.setText(mContext.getApplicationContext().getString(R.string.rv_panel_outline_noinfo));
                mNoInfoIv.setImageResource(R.drawable.panel_no_outline);
                mNoInfoTv.setVisibility(mOutlineArr.size() == 0 ? View.VISIBLE : View.GONE);
                mNoInfoIv.setVisibility(mOutlineArr.size() == 0 ? View.VISIBLE : View.GONE);
            }
            mEditViewTv.setEnabled(false);
            mPullListView.setScrollable(false);
            mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_select_all));
            mSelectedAllTv.setEnabled(false);
            mMoveIv.setEnabled(false);
            mDeleteIv.setEnabled(false);
        }
        mOutlineAdapter.notifyDataSetChanged();
        mAddView.setVisibility(mOutlineAdapter.isSearchState() || mOutlineAdapter.isEditState() ? View.GONE : View.VISIBLE);
        mMoveIv.setEnabled(mDeleteIv.isEnabled());
        mAddView.setEnabled(canAssemble);
        mOutlineBottomView.setVisibility(View.VISIBLE);
    }

    private void switchSearchState(boolean toSearchState) {
        if (mOutlineAdapter == null) return;
        if (mOutlineAdapter.isSearchState() == toSearchState) return;

        mOutlineAdapter.switchSearchState(toSearchState);
        if (toSearchState) {
            mBackView.setVisibility(View.GONE);
            mPullListView.setScrollable(false);
            mUiExtensionsManager.getPanelManager().getPanelHost().setTabVisibility(false);
            mSearchView.onActionViewExpanded();
        } else {
            mPullListView.setScrollable(true);
            mUiExtensionsManager.getPanelManager().getPanelHost().setTabVisibility(true);
            mSearchView.onActionViewCollapsed();
        }
        updateButtonState();
    }

    public void switchEditState(boolean toEditState) {
        if (mOutlineAdapter == null) return;
        if (mOutlineAdapter.isEditState() == toEditState) return;
        if (toEditState && mOutlineAdapter.getItemCount() == 0) return;

        mDeleteIv.setEnabled(false);
        mOutlineAdapter.getEditItems().clear();
        mOutlineAdapter.switchEditState(toEditState);
        if (toEditState) {
            mEditViewTv.setText(AppResource.getString(mContext, R.string.fx_string_done));
            mPullListView.setScrollable(false);
            mPullListView.hideHeaderView();
            mBottomView.setVisibility(View.VISIBLE);
            mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_select_all));
        } else {
            mEditViewTv.setText(AppResource.getString(mContext, R.string.fx_string_edit));
            mPullListView.setScrollable(true);
            mBottomView.setVisibility(View.GONE);
        }
        updateButtonState();
        enableMovable(mListView, toEditState);
    }

    @Override
    public int getPanelType() {
        return PanelSpec.OUTLINE;
    }

    @Override
    public int getIcon() {
        return R.drawable.panel_tab_outline;
    }

    @Override
    public ColorStateList getIconTint() {
        return null;
    }

    @Override
    public View getTopToolbar() {
        return mOutlineTopView;
    }

    @Override
    public View getContentView() {
        return mContentView;
    }

    @Override
    public void onActivated() {
        if (mOutlineAdapter != null) {
            if (mOutlineAdapter.isEditState())
                switchEditState(false);
            if (mOutlineAdapter.isSearchState())
                switchSearchState(false);
        }

        if (mIsNeedRefreshOutline) {
            initData();
            mIsNeedRefreshOutline = false;
        } else {
            if (mOutlineSupport != null) {
                mOutlineSupport.updateUI(mLevel, mOutlineSupport.getCurrentState());
            }
        }
    }

    @Override
    public void onDeactivated() {
        if (mOutlineAdapter != null) {
            if (mOutlineAdapter.isEditState())
                switchEditState(false);
            if (mOutlineAdapter.isSearchState())
                switchSearchState(false);
        }
        dismissAllDialog();
    }

    private PDFViewCtrl.IPageEventListener mPageEventListener = new PageEventListener() {
        @Override
        public void onPageMoved(boolean success, int index, int dstIndex) {
            mIsNeedRefreshOutline = true;
        }

        @Override
        public void onPagesInserted(boolean success, int dstIndex, int[] range) {
            mIsNeedRefreshOutline = true;
        }

        @Override
        public void onPagesRemoved(boolean success, int[] pageIndexes) {
            mIsNeedRefreshOutline = true;
        }
    };

    private ItemTouchHelper mTouchHelper = null;
    private int mMoveFromPosition = -1;
    private int mMoveToPosition = -1;

    private void enableMovable(RecyclerView recyclerView, boolean enabled) {
        if (mOutlineAdapter == null) return;
        if (enabled) {
            if (mTouchHelper == null) {
                mTouchHelper = new ItemTouchHelper(new ItemTouchHelper.SimpleCallback(ItemTouchHelper.UP
                        | ItemTouchHelper.DOWN, 0) {

                    @Override
                    public boolean onMove(@NonNull RecyclerView recyclerView, @NonNull RecyclerView.ViewHolder viewHolder, @NonNull RecyclerView.ViewHolder target) {
                        int fromPosition = viewHolder.getAdapterPosition();
                        mMoveToPosition = target.getAdapterPosition();
                        if (mMoveFromPosition == -1)
                            mMoveFromPosition = fromPosition;
                        Collections.swap(mOutlineSupport.getShowOutLineList(), fromPosition, mMoveToPosition);
                        mOutlineAdapter.notifyItemMoved(fromPosition, mMoveToPosition);
                        return true;
                    }

                    @Override
                    public void onSwiped(@NonNull RecyclerView.ViewHolder viewHolder, int direction) {

                    }

                    @Override
                    public void onSelectedChanged(@Nullable RecyclerView.ViewHolder viewHolder, int actionState) {
                        super.onSelectedChanged(viewHolder, actionState);
                        if (actionState == 0 && mMoveFromPosition != mMoveToPosition) {
                            mOutlineSupport.moveOutline(mMoveFromPosition, mMoveToPosition);
                        }
                        mMoveFromPosition = -1;
                        mMoveToPosition = -1;
                    }
                });
            }
            mTouchHelper.attachToRecyclerView(recyclerView);
        } else if (mTouchHelper != null) {
            mTouchHelper.attachToRecyclerView(null);
        }
        if (mOutlineSupport != null){
            mOutlineSupport.setItemTouchHelper(mTouchHelper);
        }
    }

    private AddOutlineDialog mAddDialog;
    private AddOutlineDialog.OnAddOutlineListener mAddOutlineListener;
    private ImageView mAddIv;
    private ImageView mMoveIv;

    private void showAddDialog() {
        if (mAddDialog != null) {
            mAddDialog.dismiss();
        }
        mAddDialog = new AddOutlineDialog(mPdfViewCtrl.getAttachedActivity(), mPdfViewCtrl);
        mAddDialog.setOutlineValue("", mPdfViewCtrl.getCurrentPage(), true);
        if (mAddOutlineListener == null)
            mAddOutlineListener = new AddOutlineDialog.OnAddOutlineListener() {
                @Override
                public void onAddOutline(String name, int pageIndex) {
                    addOutline(name, pageIndex);
                    mUiExtensionsManager.getDocumentManager().setDocModified(true);
                }
            };
        mAddDialog.setOnAddOutlineListener(mAddOutlineListener);
        mAddDialog.show();
    }

    private void addOutline(String name, int pageIndex) {
        OutlineNode newOne = new OutlineNode();
        try {
            PDFDoc doc = mPdfViewCtrl.getDoc();
            OutlineNode previous = null;
            List<OutlineNode> editItems = mOutlineAdapter.getSortedEditItems();
            if (!editItems.isEmpty()) {
                previous = editItems.get(0);
            }
            OutlineNode currentParent = mOutlineSupport.getCurrentParentNode();
            Bookmark parent = mOutlineArr.isEmpty() ?
                    (currentParent == null ? getRootBookMark(doc) : currentParent.mBookmark)
                    : mOutlineArr.get(0).mParent.mBookmark;
            if (previous == null)
                previous = mOutlineArr.isEmpty() ? null : mOutlineArr.get(mOutlineArr.size() - 1);
            Bookmark bookmark = previous == null ? parent.insert(name, Bookmark.e_PosLastChild) : previous.mBookmark.insert(name, Bookmark.e_PosNextSibling);
            bookmark.setDestination(Destination.createFitPage(doc, pageIndex));
            newOne.mBookmark = bookmark;
            newOne.mParent = previous == null ?
                    (currentParent == null ? mOutlineSupport.createRootOutlineNode() : currentParent)
                    : previous.mParent;
            newOne.mLevel = previous == null ? 1 : previous.mLevel;
            newOne.mPosition = previous == null ? 1 : previous.mPosition + 1;
            newOne.mTitle = name;
            newOne.mPid = newOne.mParent.mId;
            newOne.mId = newOne.mPid + "-" + newOne.mPosition;
            newOne.mPageIndex = pageIndex;
            int index = mOutlineArr.size();
            if (previous != null) {
                index = mOutlineArr.contains(previous) ? mOutlineArr.indexOf(previous) + 1 : index;
            }
            mOutlineSupport.appendOutline(index, newOne);
            updateButtonState();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private Bookmark getRootBookMark(PDFDoc doc) {
        Bookmark root = null;
        if (doc != null) {
            try {
                root = doc.getRootBookmark();
                if (root.isEmpty()) {
                    root = doc.createRootBookmark();
                } else {
                    if (root.hasChild()) {
                        Bookmark firstBookmark = root.getFirstChild();
                        if (firstBookmark == null || firstBookmark.isEmpty()) {
                            doc.removeBookmark(root);
                            root = doc.createRootBookmark();
                        }
                    }
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
        return root;
    }

    private UIMatchDialog mMoveOutlineDialog = null;

    private void showMoveOutlineDialog() {
        Activity activity = mPdfViewCtrl.getAttachedActivity();
        if (activity == null) return;
        if (mMoveOutlineDialog != null){
            mMoveOutlineDialog.dismiss();
            mMoveOutlineDialog.release();
        }
        mMoveOutlineDialog = new UIMatchDialog(activity);
        mMoveOutlineDialog.setTitlePosition(BaseBar.TB_Position.Position_CENTER);
        mMoveOutlineDialog.setBackButtonStyle(MatchDialog.TEXT_BACK);
        mMoveOutlineDialog.setTitle(AppResource.getString(mContext, R.string.move_outline));
        mMoveOutlineDialog.getTitleBar().setMiddleButtonCenter(true);
        mMoveOutlineDialog.setRightButtonVisible(View.VISIBLE);
        mMoveOutlineDialog.setRightButtonText(AppResource.getString(mContext, R.string.fx_string_ok));
        createOutlineListView();
        mMoveRootNode = mOutlineSupport.createRootOutlineNode();
        updateMoveOutline(mMoveRootNode);
        mMoveOutlineDialog.setContentView(mMoveOutlineContentView);
        mMoveOutlineDialog.setListener(new MatchDialog.DialogListener() {
            @Override
            public void onResult(long btType) {
            }

            @Override
            public void onBackClick() {
            }

            @Override
            public void onTitleRightButtonClick() {
                mMoveOutlineAdapter.moveOutline();
                mMoveOutlineDialog.release();
                mMoveOutlineDialog.dismiss();
                mMoveOutlineDialog = null;
                mMoveOutlineAdapter.mLastSelectedIndex = -1;
                mMoveOutlineAdapter.mTargetOutline = null;
                mOutlineSupport.refreshCurrentList();
            }
        });
        mMoveOutlineDialog.show();
    }

    public void dismissAllDialog() {
        if (mClearAllDialog != null) {
            mClearAllDialog.dismiss();
            mClearAllDialog = null;
        }
        if (mMoveOutlineDialog != null) {
            mMoveOutlineDialog.dismiss();
            mMoveOutlineDialog.release();
            mMoveOutlineDialog = null;
        }

        if (mAddDialog != null){
            mAddDialog.dismiss();
        }
    }

    private RecyclerView mMoveOutlineListView;
    private List<OutlineNode> mMoveOutlineList = new ArrayList<>();
    private MoveOutlineAdapter mMoveOutlineAdapter;
    private View mMoveOutlineContentView;
    private View mMoveNoOutlineView;
    private View mMoveOutlineBackView;
    private OnClickListener mBackClickListener;
    private OutlineNode mMoveParentNode = null;
    private OutlineNode mMoveRootNode = null;

    private void createOutlineListView() {
        if (mMoveOutlineContentView == null){
            mMoveOutlineContentView = View.inflate(mContext, R.layout.move_outline_list_layout, null);
            mMoveOutlineListView = mMoveOutlineContentView.findViewById(R.id.move_outline_list);
            mMoveNoOutlineView = mMoveOutlineContentView.findViewById(R.id.no_outline_ll);
            ((TextView)mMoveNoOutlineView.findViewById(R.id.no_outline_tv)).setTextColor(AppResource.getColor(mContext, R.color.t2));
            ((ImageView)mMoveNoOutlineView.findViewById(R.id.no_outline_iv)).setColorFilter(ThemeConfig.getInstance(mContext).getPrimaryColor());
            mMoveOutlineListView.setLayoutManager(new LinearLayoutManager(mContext, LinearLayoutManager.VERTICAL, false));
            mMoveOutlineBackView = mMoveOutlineContentView.findViewById(R.id.outline_navigation_header);
            mBackClickListener = new OnClickListener() {
                @Override
                public void onClick(View v) {
                    if (v.getId() == R.id.back_root_list_iv){
                        mMoveParentNode = null;
                        mMoveOutlineBackView.setVisibility(View.GONE);
                        updateMoveOutlineListView(mMoveRootNode.mChildren);
                    }else if (v.getId() == R.id.back_up_level_list_iv){
                        updateMoveOutline(mMoveParentNode);
                        mMoveParentNode = mMoveParentNode.mParent;
                    }
                }
            };
            mMoveOutlineBackView.findViewById(R.id.back_root_list_iv).setOnClickListener(mBackClickListener);
            mMoveOutlineBackView.findViewById(R.id.back_up_level_list_iv).setOnClickListener(mBackClickListener);
        }
        mMoveOutlineContentView.setBackgroundColor(AppResource.getColor(mContext, R.color.b1));
        if (mMoveOutlineAdapter == null)
            mMoveOutlineAdapter = new MoveOutlineAdapter(mContext);
        mMoveOutlineListView.setAdapter(mMoveOutlineAdapter);
    }

    private void updateMoveOutline(OutlineNode outlineNode) {
        if (outlineNode == null) return;
        if (!outlineNode.mChildren.isEmpty()){
            outlineNode.mChildren.clear();
        }
        mOutlineSupport.getOutline(outlineNode, new IResult<Void, Void, OutlineNode>(){
            @Override
            public void onResult(boolean success, Void p1, Void p2, final OutlineNode node) {
                updateMoveOutlineListView(node.mChildren);
                mMoveOutlineBackView.setVisibility(!node.equals(mMoveRootNode) ? View.VISIBLE : View.GONE);
            }
        });
    }

    private void updateMoveOutlineListView(List<OutlineNode> list){
        mMoveOutlineList.clear();
        mMoveOutlineList.addAll(list);
        mMoveOutlineAdapter.notifyUpdateData();
        mMoveNoOutlineView.setVisibility(mMoveOutlineList.isEmpty() ? View.VISIBLE : View.GONE);
    }

    public class MoveOutlineAdapter extends SuperAdapter<OutlineNode>{

        private int mLastSelectedIndex = -1;
        private OutlineNode mTargetOutline;
        private SparseBooleanArray mEnabledList = new SparseBooleanArray();

        public MoveOutlineAdapter(Context context) {
            super(context);
        }

        @SuppressLint("NotifyDataSetChanged")
        @Override
        public void notifyUpdateData() {
            notifyDataSetChanged();
        }

        @Override
        public OutlineNode getDataItem(int position) {
            return mMoveOutlineList.get(position);
        }

        @NonNull
        @Override
        public SuperViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
            return new MoveOutlineViewHolder(LayoutInflater.from(getContext()).inflate(R.layout.rv_panel_outline_item, parent, false));
        }

        @Override
        public int getItemCount() {
            return mMoveOutlineList.size();
        }

        @Override
        public long getItemId(int position) {
            return position;
        }

        public void moveOutline() {
            List<OutlineNode> sortedEditItems = mOutlineAdapter.getSortedEditItems();
            if (!sortedEditItems.isEmpty()){
                OutlineNode first = sortedEditItems.get(0);
                OutlineNode toParent = mTargetOutline;
                if (mLastSelectedIndex != -1)
                    toParent = mTargetOutline.mParent;
                if (!first.mParent.equals(toParent)){
                    OutlineNode fromParent = first.mParent.mChildren.size() == sortedEditItems.size() ?
                            first.mParent.mParent : null;
                    mOutlineSupport.setMovedParents(fromParent, toParent);
                }
            }
            if (mLastSelectedIndex != -1){
                for (int i = 0; i < sortedEditItems.size(); i++) {
                    if (i == 0){
                        mOutlineSupport.moveBookmark(sortedEditItems.get(i), mTargetOutline, Bookmark.e_PosNextSibling);
                    }else {
                        mOutlineSupport.moveBookmark(sortedEditItems.get(i), sortedEditItems.get(i - 1), Bookmark.e_PosNextSibling);
                    }
                }
            }else if (mTargetOutline.mIsLeaf){
                for (int i = 0; i < sortedEditItems.size(); i++) {
                    if (i == 0){
                        mOutlineSupport.moveBookmark(sortedEditItems.get(i), mTargetOutline, Bookmark.e_PosFirstChild);
                    }else {
                        mOutlineSupport.moveBookmark(sortedEditItems.get(i), sortedEditItems.get(i - 1), Bookmark.e_PosNextSibling);
                    }
                }
            } else if (!mTargetOutline.mChildren.isEmpty()){
                for (int i = 0; i < sortedEditItems.size(); i++) {
                    if (i == 0){
                        mOutlineSupport.moveBookmark(sortedEditItems.get(i), mTargetOutline.mChildren.get(mTargetOutline.mChildren.size() - 1), Bookmark.e_PosNextSibling);
                    }else {
                        mOutlineSupport.moveBookmark(sortedEditItems.get(i), sortedEditItems.get(i - 1), Bookmark.e_PosNextSibling);
                    }
                }
            }
            ((UIExtensionsManager)mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setDocModified(true);
        }

        private class MoveOutlineViewHolder extends SuperViewHolder {
            private LinearLayout mItemView;
            private TextView mChapterTv;
            private TextView mChapterIndexTv;
            private CheckBox mChapterCheck;
            private ImageView mMoreIv;
            private boolean mIsSelected = false;

            public MoveOutlineViewHolder(View itemView) {
                super(itemView);
                mItemView = itemView.findViewById(R.id.panel_outline_item_container);
                mChapterTv = itemView.findViewById(R.id.panel_outline_item_chapter);
                mChapterIndexTv = itemView.findViewById(R.id.panel_outline_item_index);
                mChapterCheck = itemView.findViewById(R.id.rd_outline_item_checkbox);
                mMoreIv = itemView.findViewById(R.id.panel_outline_item_more);
                mMoreIv.setOnClickListener(this);
                mItemView.setOnClickListener(this);
            }

            @Override
            public void bind(BaseBean data, int position) {
                OutlineNode outlineItem = (OutlineNode) data;
                boolean enabled = !mOutlineAdapter.getEditItems().contains(outlineItem);
                if (position == 0)
                    mEnabledList.clear();
                if (enabled){
                    mEnabledList.append(position, true);
                }else {
                    mEnabledList.delete(position);
                }
                mChapterIndexTv.setText(String.valueOf(outlineItem.mPageIndex + 1));
                mChapterIndexTv.setVisibility(outlineItem.mPageIndex >= 0 ? View.VISIBLE : View.GONE);
                mChapterCheck.setVisibility(View.GONE);
                mMoreIv.setVisibility(enabled ? View.VISIBLE : View.INVISIBLE);
                mChapterTv.setText(outlineItem.mTitle);
                mItemView.setBackground(AppResource.getDrawable(mContext,R.drawable.rd_list_item_bg));
                ThemeUtil.setTintList(mMoreIv,ThemeUtil.getPrimaryIconColor(mContext));
                mChapterTv.setTextColor(AppResource.getColor(mContext,R.color.t4));
                mChapterIndexTv.setTextColor(AppResource.getColor(mContext,R.color.t3));
                updateSelectedState();
                mItemView.setEnabled(enabled);
                if (position == getItemCount() - 1){
                    if (mEnabledList.size() == 0){
                        mTargetOutline = null;
                        mMoveOutlineDialog.setRightButtonEnable(false);
                    }else {
                        if (mTargetOutline == null)
                            mTargetOutline = mMoveRootNode;
                        mMoveOutlineDialog.setRightButtonEnable(true);
                    }
                }
            }

            @Override
            public void onClick(View v) {
                int id = v.getId();
                int position = getAdapterPosition();
                OutlineNode outlineNode = mMoveOutlineList.get(position);
                if (id == R.id.panel_outline_item_more) {
                    if (mLastSelectedIndex != -1){
                        MoveOutlineViewHolder holder = (MoveOutlineViewHolder) mMoveOutlineListView.findViewHolderForAdapterPosition(mLastSelectedIndex);
                        if (holder != null){
                            holder.mIsSelected = false;
                            holder.updateSelectedState();
                        }
                    }
                    mLastSelectedIndex = -1;
                    mTargetOutline = outlineNode;
                    mMoveParentNode = outlineNode.mParent;
                    updateMoveOutline(outlineNode);
                    return;
                }
                mIsSelected = !mIsSelected;
                if (mIsSelected)
                    mTargetOutline = outlineNode;
                if (mLastSelectedIndex == position && !mIsSelected){
                    mLastSelectedIndex = -1;
                    mTargetOutline  =outlineNode.mParent;
                }
                if (mLastSelectedIndex != position && mIsSelected){
                    MoveOutlineViewHolder holder = (MoveOutlineViewHolder) mMoveOutlineListView.findViewHolderForAdapterPosition(mLastSelectedIndex);
                    if (holder != null){
                        holder.mIsSelected = false;
                        holder.updateSelectedState();
                    }
                    mLastSelectedIndex = position;
                }
                updateSelectedState();
            }

            void updateSelectedState(){
                mItemView.setSelected(mIsSelected);
            }
        }
    }
}
