/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.panel.outline;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.drawable.Drawable;
import android.text.Editable;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.util.Log;
import android.view.MotionEvent;
import android.view.View;
import android.view.inputmethod.EditorInfo;
import android.widget.Button;
import android.widget.EditText;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;
import android.widget.Toast;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.controls.dialog.UIDialog;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppTheme;
import com.foxit.uiextensions.utils.UIMarqueeTextView;
import com.foxit.uiextensions.utils.UIToast;

public class AddOutlineDialog extends UIDialog {

    Button mOkButton;
    Button mCancelButton;
    View mCuttingLine;
    View mTitleViewDivider;
    View mDivider;
    UIMarqueeTextView mTvTitle;

    int mTopHeight;
    int mBottomHeight;
    View mEditContentView;
    View mBottomView;
    private final Context mContext;
    private Drawable mClearDrawable;
    EditText mNameEditText;
    EditText mPageNumberEditText;
    View mInputNameLayout;
    View mInputPageNumberLayout;
    TextView mOutlineNameHeader;
    TextView mOutlinePageNumberHeader;
    private PDFViewCtrl mViewCtrl;

    public AddOutlineDialog(Context context, PDFViewCtrl viewCtrl) {
        this(context, viewCtrl, UIDialog.WITH_INPUT);
    }

    @SuppressLint("ClickableViewAccessibility")
    private AddOutlineDialog(Context context, PDFViewCtrl viewCtrl, @ShowType int showType) {
        super(context, R.layout.add_outline_dialog,
                AppTheme.getDialogTheme(),
                AppDisplay.getUITextEditDialogWidth());
        mViewCtrl = viewCtrl;
        mContext = context.getApplicationContext();
        mTopHeight = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_toolbar_height_phone);
        RelativeLayout.LayoutParams titleParams = (RelativeLayout.LayoutParams) mTitleView.getLayoutParams();
        titleParams.height = mTopHeight;
        mTitleView.setLayoutParams(titleParams);

        mBottomView = mContentView.findViewById(R.id.fx_dialog_bottom_ll);
        mBottomHeight = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_dialog_button_height);
        RelativeLayout.LayoutParams bottomParams = (RelativeLayout.LayoutParams) mBottomView.getLayoutParams();
        bottomParams.height = mBottomHeight;
        mBottomView.setLayoutParams(bottomParams);

        mTvTitle = mContentView.findViewById(R.id.fx_dialog_title);
        mTitleViewDivider = mContentView.findViewById(R.id.fx_dialog_title_divider);
        mEditContentView = mContentView.findViewById(R.id.fx_dialog_content);
        mDivider = mContentView.findViewById(R.id.fx_dialog_divider);
        mOutlineNameHeader = mContentView.findViewById(R.id.outline_name_header);
        mOutlinePageNumberHeader = mContentView.findViewById(R.id.outline_page_number_header);
        mNameEditText = mContentView.findViewById(R.id.outline_name_edit_text);
        mInputNameLayout = mContentView.findViewById(R.id.input_outline_name_layout);
        mPageNumberEditText = mContentView.findViewById(R.id.outline_page_number_edit_text);
        mInputPageNumberLayout = mContentView.findViewById(R.id.input_page_number_layout);
        if (AppDisplay.isPad()) { // SDKRD-9313
            mNameEditText.setImeOptions(EditorInfo.IME_FLAG_NO_EXTRACT_UI);
            mPageNumberEditText.setImeOptions(EditorInfo.IME_FLAG_NO_EXTRACT_UI);
        }
        initEditText(mNameEditText);
        initEditText(mPageNumberEditText);

        if (AppDisplay.isPad()) {
            usePadDimes();
        }

        mOkButton = mContentView.findViewById(R.id.fx_dialog_ok);
        mOkButton.setTextColor(ThemeUtil.getToolbarTextColor(mContext));
        mOkButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mAddOutlineListener != null){
                    int pageIndex = 0;
                    try{
                        pageIndex = Integer.parseInt(mPageNumberEditText.getText().toString()) - 1;
                    }catch (Exception e){
                        e.printStackTrace();
                    }
                    mAddOutlineListener.onAddOutline(mNameEditText.getText().toString(), pageIndex);
                }
                mDialog.dismiss();
            }
        });
        mCuttingLine = mContentView.findViewById(R.id.fx_dialog_button_cutting_line);
        mCancelButton = mContentView.findViewById(R.id.fx_dialog_cancel);
        mCancelButton.setTextColor(ThemeUtil.getToolbarTextColor(mContext));
        mCancelButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mDialog.dismiss();
            }
        });
        mDialog.getWindow().setBackgroundDrawableResource(android.R.color.transparent);
    }

    @SuppressLint("ClickableViewAccessibility")
    private void initEditText(final EditText editText) {
        mClearDrawable = editText.getCompoundDrawables()[2];
        if (mClearDrawable == null) {
            mClearDrawable = AppResource.getDrawable(mContext, R.drawable.rd_clear_search);
        }
        mClearDrawable.setBounds(0, 0, mClearDrawable.getIntrinsicWidth(), mClearDrawable.getIntrinsicHeight());
        editText.setOnTouchListener(new View.OnTouchListener() {
            @Override
            public boolean onTouch(View v, MotionEvent event) {
                if (event.getAction() == MotionEvent.ACTION_UP) {
                    if (editText.getCompoundDrawables()[2] != null) {
                        boolean isTouchRight = event.getX() > (editText.getWidth() - editText.getTotalPaddingRight()) && (event.getX() < ((editText.getWidth() - editText.getPaddingRight())));
                        if (isTouchRight) {
                            editText.setText("");
                        }
                    }
                }
                return false;
            }
        });
        editText.addTextChangedListener(new TextWatcher() {
            @Override
            public void beforeTextChanged(CharSequence s, int start, int count, int after) {
            }

            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {
                validateInput(editText);
            }

            @Override
            public void afterTextChanged(Editable s) {
            }
        });
    }

    public void validateInput(EditText editText) {
        String text = editText.getText().toString();
        if (!TextUtils.isEmpty(text)) {
            editText.setCompoundDrawables(null, null, mClearDrawable, null);
        } else {
            editText.setCompoundDrawables(null, null, null, null);
        }
        if (!TextUtils.isEmpty(mNameEditText.getText()) && !TextUtils.isEmpty(mPageNumberEditText.getText())) {
            int pageNumber = Integer.parseInt(mPageNumberEditText.getText().toString());
            if (pageNumber <= mViewCtrl.getPageCount()) {
                mOkButton.setEnabled(true);
            } else {
                mPageNumberEditText.selectAll();
                mOkButton.setEnabled(false);
                UIToast.getInstance(mContext).show(R.string.invalid_page_count, Toast.LENGTH_SHORT);
            }
        } else {
            mOkButton.setEnabled(false);
        }
    }

    public void setDarkModeColor() {
        mTvTitle.setTextColor(mContext.getResources().getColor(R.color.t4));
        mOutlineNameHeader.setTextColor(mContext.getResources().getColor(R.color.t4));
        mTitleViewDivider.setBackgroundColor(mContext.getResources().getColor(R.color.p3));
        mInputNameLayout.setBackground(mContext.getResources().getDrawable(R.drawable.edit_text_input_bg));
        mInputPageNumberLayout.setBackground(mContext.getResources().getDrawable(R.drawable.edit_text_input_bg));
        mNameEditText.setHintTextColor(mContext.getResources().getColor(R.color.t3));
        mNameEditText.setTextColor(mContext.getResources().getColor(R.color.t4));
        mPageNumberEditText.setHintTextColor(mContext.getResources().getColor(R.color.t3));
        mPageNumberEditText.setTextColor(mContext.getResources().getColor(R.color.t4));
        mDivider.setBackgroundColor(mContext.getResources().getColor(R.color.p2));
        mCancelButton.setTextColor(ThemeUtil.getToolbarTextColor(mContext));
        mCancelButton.setBackground(mContext.getResources().getDrawable(R.drawable.rd_list_item_bg));
        mCuttingLine.setBackgroundColor(mContext.getResources().getColor(R.color.p2));
        mOkButton.setTextColor(ThemeUtil.getToolbarTextColor(mContext));
        mOkButton.setBackground(mContext.getResources().getDrawable(R.drawable.rd_list_item_bg));
        mContentView.setBackground(mContext.getResources().getDrawable(R.drawable.dialog_radian));
    }

    private void usePadDimes() {
        try {
            ((LinearLayout.LayoutParams) mTitleView.getLayoutParams()).leftMargin = AppDisplay.dp2px(24);
            ((LinearLayout.LayoutParams) mTitleView.getLayoutParams()).rightMargin = AppDisplay.dp2px(24);
            ((LinearLayout.LayoutParams) mOutlineNameHeader.getLayoutParams()).leftMargin = AppDisplay.dp2px(24);
            ((LinearLayout.LayoutParams) mOutlineNameHeader.getLayoutParams()).rightMargin = AppDisplay.dp2px(24);
            ((LinearLayout.LayoutParams) mNameEditText.getLayoutParams()).leftMargin = AppDisplay.dp2px(24);
            ((LinearLayout.LayoutParams) mNameEditText.getLayoutParams()).rightMargin = AppDisplay.dp2px(24);
        } catch (Exception e) {
            if (e.getMessage() != null) {
                Log.e("UITextEditDialog", e.getMessage());
            } else {
                Log.e("UITextEditDialog", "usePadDimes_has_an_error");
            }
        }
    }

    @Override
    public void show() {
        if (mOkButton.getVisibility() == View.VISIBLE && mCancelButton.getVisibility() == View.VISIBLE) {
            mCuttingLine.setVisibility(View.VISIBLE);
            mOkButton.setBackgroundResource(R.drawable.dialog_right_button_background_selector);
            mCancelButton.setBackgroundResource(R.drawable.dialog_left_button_background_selector);
        } else {
            mCuttingLine.setVisibility(View.GONE);
            mOkButton.setBackgroundResource(R.drawable.dialog_button_background_selector);
            mCancelButton.setBackgroundResource(R.drawable.dialog_button_background_selector);
        }
        super.show();
    }

    public boolean isShowing() {
        return mDialog.isShowing();
    }

    public void setOutlineValue(String name, int pageIndex, boolean firstOnFocus) {
        mPageNumberEditText.setText(String.format(AppResource.getString(mContext, R.string.number_format_text), pageIndex + 1));
        mNameEditText.setText(name);
        if (firstOnFocus){
            mNameEditText.requestFocus();
        }else {
            mPageNumberEditText.requestFocus();
        }
        mNameEditText.setSelection(mNameEditText.getText().length());
    }

    public interface OnAddOutlineListener{
        void onAddOutline(String name, int pageIndex);
    }

    private OnAddOutlineListener mAddOutlineListener;

    public void setOnAddOutlineListener(OnAddOutlineListener addOutlineListener) {
        mAddOutlineListener = addOutlineListener;
    }

    public void changeToEditUI(){
        setTitle(R.string.edit_outline);
        mOutlineNameHeader.setText(R.string.input_new_outline_name);
        mOutlinePageNumberHeader.setText(R.string.input_new_outline_page);
    }
}