/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.panel.filespec;

import android.app.Activity;
import android.app.ProgressDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.res.ColorStateList;
import android.content.res.Configuration;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.view.KeyEvent;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.DefaultItemAnimator;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.Markup;
import com.foxit.sdk.pdf.annots.MarkupArray;
import com.foxit.uiextensions.IInteractionEventListener;
import com.foxit.uiextensions.IPermissionProvider;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotEventListener;
import com.foxit.uiextensions.annots.IFlattenEventListener;
import com.foxit.uiextensions.annots.IGroupEventListener;
import com.foxit.uiextensions.annots.IImportAnnotsEventListener;
import com.foxit.uiextensions.annots.IRedactionEventListener;
import com.foxit.uiextensions.annots.multiselect.GroupManager;
import com.foxit.uiextensions.controls.dialog.AppDialogManager;
import com.foxit.uiextensions.controls.dialog.FxProgressDialog;
import com.foxit.uiextensions.controls.dialog.UIDialog;
import com.foxit.uiextensions.controls.dialog.UIMatchDialog;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.controls.dialog.fileselect.UIFileSelectDialog;
import com.foxit.uiextensions.controls.filebrowser.FileBrowser;
import com.foxit.uiextensions.controls.filebrowser.imp.FileItem;
import com.foxit.uiextensions.controls.panel.PanelSpec;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.event.PageEventListener;
import com.foxit.uiextensions.modules.AutoFlipModule;
import com.foxit.uiextensions.modules.ReflowModule;
import com.foxit.uiextensions.modules.panel.IPanelManager;
import com.foxit.uiextensions.modules.panel.bean.FileBean;
import com.foxit.uiextensions.modules.tts.TTSModule;
import com.foxit.uiextensions.pdfreader.config.ReadStateConfig;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.SystemUiHelper;
import com.foxit.uiextensions.utils.UIToast;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.io.File;
import java.io.FileFilter;
import java.util.ArrayList;
import java.util.List;

public class FileSpecPanelModule implements Module, PanelSpec, FileSpecModuleCallback {
    private static final int ADD_ANNOT = 1;
    private static final int MODIFIED_ANNOT = 2;
    private static final int DELETTE_ANNOT = 3;
    private static final int GROUP_ANNOT = 4;
    private static final int UN_GROUP_ANNOT = 5;

    private PDFViewCtrl mPdfViewCtrl;
    private FileAttachmentAdapter mFileAttachmentAdapter;
    private UIExtensionsManager mUiExtensionsManager;
    private FxProgressDialog mProgressDialog;
    private UITextEditDialog mClearAllDialog;
    private LinearLayoutManager mLayoutManager;

    private Context mContext;
    private ViewGroup mParent;
    private View mTopBarView;
    private TextView mAddView;
    private View mBottomView;
    private TextView mSelectedAllTv;
    private ImageView mDeleteIv;

    private View mContentView;
    private View mNoInfoContainer;
    private ImageView mNoInfoIv;
    private TextView mNoInfoTv;
    private TextView mLoadingView;
    private FileSpecOpenView openView;
    private TextView mEditViewTv;
    private TextView mCloseView;
    private TextView mTitleView;
    private RelativeLayout mRlContainer;
    private RelativeLayout mRlTopContainer;

    private int mLastReadState;
    private boolean bDocClosed;
    private final boolean mIsLoadAnnotation;
    private boolean mShouldReloadData;
    private int mNightMode = -1;

    public FileSpecPanelModule(Context context, ViewGroup parent, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        if (context == null || pdfViewCtrl == null) {
            throw new NullPointerException();
        }
        mContext = context.getApplicationContext();
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = (UIExtensionsManager) uiExtensionsManager;
        mParent = parent;
        mFileAttachmentAdapter = new FileAttachmentAdapter(mContext, mPdfViewCtrl, this);
        mFileAttachmentAdapter.setOnItemClickListener(new FileAttachmentAdapter.OnItemClickListener() {
            @Override
            public void onItemClick(int position, FileBean fileBean) {
                if (mFileAttachmentAdapter.isEditState()) {
                    mDeleteIv.setEnabled(mFileAttachmentAdapter.getEditList().size() > 0);
                    if (mFileAttachmentAdapter.isSelectedAll())
                        mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_deselect_all));
                    else
                        mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_select_all));
                }
            }
        });
        mIsLoadAnnotation = mUiExtensionsManager.getConfig().modules.isLoadAnnotations;
    }

    private final UIExtensionsManager.ConfigurationChangedListener mConfigurationChangedListener = new UIExtensionsManager.ConfigurationChangedListener() {
        @Override
        public void onConfigurationChanged(Configuration newConfig) {
            if (mNightMode != (newConfig.uiMode & Configuration.UI_MODE_NIGHT_MASK)) {
                mNightMode = newConfig.uiMode & Configuration.UI_MODE_NIGHT_MASK;
                return;
            }
            if (mfileSelectDialog != null && mfileSelectDialog.isShowing()) {
                mfileSelectDialog.resetWH();
                mfileSelectDialog.showDialog(false);
            }

            if (mFileAttachmentAdapter != null) {
                mFileAttachmentAdapter.onConfigurationChanged(newConfig);
                mFileAttachmentAdapter.updateMenu(mLayoutManager);
            }
        }
    };

    @Override
    public boolean loadModule() {
        mUiExtensionsManager.registerModule(this);
        mNightMode = mContext.getResources().getConfiguration().uiMode & Configuration.UI_MODE_NIGHT_MASK;
        mPdfViewCtrl.registerDocEventListener(mDocEventListener);
        mPdfViewCtrl.registerPageEventListener(mPageEventListener);
        mPdfViewCtrl.registerRecoveryEventListener(recoveryEventListener);
        mUiExtensionsManager.getDocumentManager().registerAnnotEventListener(mAnnotEventListener);
        mUiExtensionsManager.getDocumentManager().registerImportedAnnotsEventListener(mImportAnnotEventListener);
        mUiExtensionsManager.getDocumentManager().registerFlattenEventListener(mFlattenEventListener);
        mUiExtensionsManager.getDocumentManager().registerRedactionEventListener(mRedactionEventListener);
        mUiExtensionsManager.getDocumentManager().registerGroupEventListener(mGroupEventListener);
        mUiExtensionsManager.registerConfigurationChangedListener(mConfigurationChangedListener);
        mUiExtensionsManager.registerInteractionListener(mInteractionEventListener);
        mUiExtensionsManager.registerThemeEventListener(mThemeEventListener);
        mUiExtensionsManager.getPanelManager().registerPanelEventListener(mPanelEventListener);
        return true;
    }

    private final IPanelManager.OnPanelEventListener mPanelEventListener = new IPanelManager.OnPanelEventListener() {
        @Override
        public void onPanelOpened() {
            if (mTopBarView == null) {
                initPanelView();
            }
            resetAddView();
        }

        @Override
        public void onPanelClosed() {

        }
    };

    private void initPanelView() {
        mTopBarView = createTopView();
        mContentView = createContentView();
        mUiExtensionsManager.getPanelManager().addPanel(this);
        updateUIState();
    }

    public void AI_addAttachment() {
        if (mContentView == null) {
            initPanelView();
        }
        mAddView.callOnClick();
    }

    private View createTopView() {
        View topView = View.inflate(mContext, R.layout.panel_topbar_layout, null);
        mRlTopContainer = topView.findViewById(R.id.panel_rl_top_container);
        mRlTopContainer.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
        mCloseView = topView.findViewById(R.id.panel_close_tv);
        mCloseView.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        if (AppDevice.isChromeOs(mPdfViewCtrl.getAttachedActivity())) {
            mCloseView.setVisibility(View.VISIBLE);
        } else {
            if (AppDisplay.isPad())
                mCloseView.setVisibility(View.INVISIBLE);
            else
                mCloseView.setVisibility(View.VISIBLE);
        }
        mCloseView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mUiExtensionsManager.getPanelManager().isShowingPanel())
                    mUiExtensionsManager.getPanelManager().hidePanel();
                if (mFileAttachmentAdapter.isEditState())
                    switchEditState(false);
            }
        });
        mTitleView = topView.findViewById(R.id.panel_title_tv);
        mTitleView.setText(AppResource.getString(mContext, R.string.rv_panel_title_attachment));

        mEditViewTv = topView.findViewById(R.id.panel_edit_tv);
        mEditViewTv.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        mEditViewTv.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                switchEditState(!mFileAttachmentAdapter.isEditState());
            }
        });
        return topView;
    }

    private View createContentView() {
        View contentView = View.inflate(mContext, R.layout.panel_filespec_content, null);
        mRlContainer = contentView.findViewById(R.id.rv_panel_attachment_layout);
        mNoInfoIv = contentView.findViewById(R.id.panel_filespec_noinfo_iv);
        mNoInfoTv = contentView.findViewById(R.id.panel_filespec_noinfo_tv);
        mNoInfoIv.setColorFilter(ThemeConfig.getInstance(mContext).getPrimaryColor());
        mNoInfoContainer = contentView.findViewById(R.id.panel_filespec_noinfo_container);
        mLoadingView = contentView.findViewById(R.id.rv_panel_filespec_loading);
        RecyclerView recyclerView = contentView.findViewById(R.id.rv_panel_filespec_list);
        recyclerView.setAdapter(mFileAttachmentAdapter);
        mLayoutManager = new LinearLayoutManager(mContext, LinearLayoutManager.VERTICAL, false);
        recyclerView.setLayoutManager(mLayoutManager);
        recyclerView.setItemAnimator(new DefaultItemAnimator());
        if (AppDisplay.isPad()) {
            recyclerView.addOnScrollListener(new RecyclerView.OnScrollListener() {
                @Override
                public void onScrolled(@NonNull RecyclerView recyclerView, int dx, int dy) {
                    mFileAttachmentAdapter.onScrolled(mLayoutManager);
                }
            });
        }

        mAddView = contentView.findViewById(R.id.panel_add_attachment);
        mAddView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                showFileSelectDialog();
            }
        });

        mBottomView = contentView.findViewById(R.id.attachment_panel_bottom_view);
        mDeleteIv = mBottomView.findViewById(R.id.panel_bottom_delete_iv);
        ThemeUtil.setTintList(mDeleteIv, ThemeUtil.getPrimaryIconColor(mContext));
        mDeleteIv.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mFileAttachmentAdapter.isSelectedAll()) {
                    Context context = mUiExtensionsManager.getAttachedActivity();
                    if (context == null) return;
                    if (mClearAllDialog == null) {
                        mClearAllDialog = new UITextEditDialog(context, UIDialog.NO_INPUT);
                        mClearAllDialog.setTitle(AppResource.getString(mContext, R.string.menu_more_confirm));
                        mClearAllDialog.getPromptTextView().setText(AppResource.getString(mContext, R.string.rd_panel_clear_attachments));
                    }
                    mClearAllDialog.getOKButton().setOnClickListener(new View.OnClickListener() {
                        @Override
                        public void onClick(View v) {
                            mClearAllDialog.dismiss();
                            mFileAttachmentAdapter.delete(mFileAttachmentAdapter.getEditList());
                            mDeleteIv.setEnabled(false);
                        }
                    });
                    mClearAllDialog.show();
                } else {
                    mFileAttachmentAdapter.delete(mFileAttachmentAdapter.getEditList());
                    mDeleteIv.setEnabled(false);
                }
            }
        });
        mSelectedAllTv = mBottomView.findViewById(R.id.panel_bottom_select_all_tv);
        mSelectedAllTv.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                boolean isSelectedAll = mFileAttachmentAdapter.isSelectedAll();
                if (isSelectedAll) {
                    mDeleteIv.setEnabled(false);
                    mFileAttachmentAdapter.selectAll(false);
                    mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_select_all));
                } else {
                    mDeleteIv.setEnabled(true);
                    mFileAttachmentAdapter.selectAll(true);
                    mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_deselect_all));
                }
            }
        });
        return contentView;
    }

    public boolean AI_deleteAttachments() {
        if (mContentView == null) {
            initPanelView();
        }

        mFileAttachmentAdapter.selectAll(true);
        if (mFileAttachmentAdapter.getList().size() > 0 && mFileAttachmentAdapter.getList().size() != mFileAttachmentAdapter.getCanEditItemCount())
            return false;
        mFileAttachmentAdapter.delete(mFileAttachmentAdapter.getEditList());
        mFileAttachmentAdapter.notifyUpdateData();
        return true;
    }

    private void switchEditState(boolean toEditState) {
        if (mFileAttachmentAdapter.isEditState() == toEditState) return;
        mFileAttachmentAdapter.getEditList().clear();
        mDeleteIv.setEnabled(false);
        mFileAttachmentAdapter.switchEditState(toEditState);
        if (toEditState) {
            hideAddView();
            mEditViewTv.setText(AppResource.getString(mContext, R.string.fx_string_done));
            mBottomView.setVisibility(View.VISIBLE);
            mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_select_all));
            mSelectedAllTv.setEnabled(mFileAttachmentAdapter.getCanEditItemCount() > 0);
        } else {
            resetAddView();
            mEditViewTv.setEnabled(mFileAttachmentAdapter.getCanEditItemCount() > 0);
            mEditViewTv.setText(AppResource.getString(mContext, R.string.fx_string_edit));
            mBottomView.setVisibility(View.GONE);
        }
        mFileAttachmentAdapter.notifyUpdateData();
    }

    private static final int MAX_ATTACHMENT_FILE_SIZE = 1024 * 1024 * 300;
    private UIFileSelectDialog mfileSelectDialog;

    private void showFileSelectDialog() {
        if (mfileSelectDialog != null && mfileSelectDialog.isShowing()) return;
        final Context context = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAttachedActivity();
        if (context == null) {
            return;
        }

        mfileSelectDialog = new UIFileSelectDialog(context);
        mfileSelectDialog.init(new FileFilter() {
            @Override
            public boolean accept(File pathname) {
                return !pathname.isHidden() && AppFileUtil.canRead(pathname);
            }
        }, false);
        mfileSelectDialog.setRightButtonEnable(false);
        mfileSelectDialog.setFileLimitListener(new FileBrowser.FileLimitListener() {
            @Override
            public int getLimitSize() {
                return MAX_ATTACHMENT_FILE_SIZE;
            }

            @Override
            public boolean exceedLimitSize() {
                String msg = AppResource.getString(mContext, R.string.annot_fat_filesizelimit_meg,
                        MAX_ATTACHMENT_FILE_SIZE / (1024 * 1024));
                UIToast.getInstance(mContext).showCustomToast(msg);
                return false;
            }
        });
        mfileSelectDialog.setListener(new UIMatchDialog.DialogListener() {
            @Override
            public void onResult(long btType) {
            }

            @Override
            public void onBackClick() {
            }

            @Override
            public void onTitleRightButtonClick() {
                mfileSelectDialog.dismiss();

                List<FileItem> fileItems = mfileSelectDialog.getSelectedFiles();
                if (mProgressDialog == null)
                    mProgressDialog = new FxProgressDialog(mUiExtensionsManager.getAttachedActivity(), AppResource.getString(mContext, R.string.fx_string_processing));
                mProgressDialog.show();
                for (FileItem fileItem : fileItems) {
                    String path = fileItem.path;
                    final String name = fileItem.name;
                    if (path == null || path.length() < 1) continue;
                    final String adapterPath = AppFileUtil.getAdaptedFilePath(context, path);
                    if (mFileAttachmentAdapter.hasName(name)) {
                        final UITextEditDialog dialog = new UITextEditDialog(mUiExtensionsManager.getAttachedActivity());
                        dialog.setEditVisible(false);
                        dialog.setTitle(AppResource.getString(context, R.string.fx_string_warning_title));
                        dialog.getPromptTextView().setText(AppResource.getString(context, R.string.rename_attachment_prompt));
                        dialog.setBottomItemWeight(2, 3);
                        dialog.getOKButton().setText(AppResource.getString(mContext, R.string.rename_attachment_ok_button_text));
                        dialog.getOKButton().setOnClickListener(new View.OnClickListener() {
                            @Override
                            public void onClick(View v) {
                                mFileAttachmentAdapter.add(rename(name), adapterPath);
                                dialog.dismiss();
                            }
                        });
                        dialog.show();
                    } else {
                        mFileAttachmentAdapter.add(name, adapterPath);
                    }
                }
                mProgressDialog.dismiss();
            }
        });
        mfileSelectDialog.setOnKeyListener(new DialogInterface.OnKeyListener() {
            @Override
            public boolean onKey(DialogInterface dialog, int keyCode, KeyEvent event) {
                if (keyCode == KeyEvent.KEYCODE_BACK) {
                    mfileSelectDialog.dismiss();
                }
                return true;
            }
        });

        mfileSelectDialog.showDialog(false);
    }

    private String rename(String name) {
        int lastIndex = name.lastIndexOf('.');
        if (lastIndex == -1) {
            lastIndex = name.length() - 1;
        }
        String oldName = name.substring(0, lastIndex);
        String copyName = oldName + "_Copy";
        name = name.replace(oldName, copyName);
        if (mFileAttachmentAdapter.hasName(name)) {
            return getDuplicateName(name);
        }
        return name;
    }

    private String getDuplicateName(String name) {
        String newName = name;
        while (true) {
            if (mFileAttachmentAdapter.hasName(newName)) {
                int lastIndex = newName.lastIndexOf('.');
                String ext = "";
                if (lastIndex != -1) {
                    ext = newName.substring(lastIndex);
                    newName = newName.substring(0, lastIndex);
                }

                int begin = 0;
                int end = newName.length() - 1;
                if (newName.charAt(end) == ')') {
                    for (int i = end - 1; i >= 0; i--) {
                        char c = newName.charAt(i);
                        if (c == '(') {
                            begin = i;
                            break;
                        }
                        if (c < '0' || c > '9')
                            break;
                    }
                }
                if (begin > 0 && end - begin < 32) {
                    String num = newName.substring(begin + 1, end);
                    int index = Integer.parseInt(num, 10) + 1;
                    newName = newName.substring(0, begin) + "(" + index + ")" + ext;
                    continue;
                }
                newName = newName + "(" + 1 + ")" + ext;
                continue;
            }
            break;
        }
        return newName;
    }

    private final PDFViewCtrl.IPageEventListener mPageEventListener = new PageEventListener() {

        @Override
        public void onPagesRemoved(boolean success, int[] pageIndexes) {
            if (mContentView == null) return;
            if (mFileAttachmentAdapter.isEmpty()) return;

            if (success) {
                for (int i = 0; i < pageIndexes.length; i++) {
                    mFileAttachmentAdapter.onPageRemoved(pageIndexes[i] - i);
                }
            }
        }

        @Override
        public void onPageMoved(boolean success, int index, int dstIndex) {
            if (mContentView == null) return;
            if (mFileAttachmentAdapter.isEmpty()) return;

            if (success) {
                if (mFileAttachmentAdapter.getCurrentStatus() == FileAttachmentPresenter.FileAttachmentViewer.STATUS_DONE) {
                    mFileAttachmentAdapter.onPageMoved(index, dstIndex);
                } else {
                    startSearchFileAttachment(false);
                }
            }
        }

        @Override
        public void onPagesInserted(boolean success, int dstIndex, int[] range) {
            if (mContentView == null) return;

            startSearchFileAttachment(false);
        }
    };

    private void startSearchFileAttachment(final boolean reLoadNameTree) {
        AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
            @Override
            public void run() {
                updateUIState();
                mFileAttachmentAdapter.initPDFNameTree(reLoadNameTree);
                mFileAttachmentAdapter.startSearchFileAttachment(mIsLoadAnnotation, reLoadNameTree);
            }
        });
    }

    private final PDFViewCtrl.IDocEventListener mDocEventListener = new DocEventListener() {
        @Override
        public void onDocWillOpen() {
        }

        @Override
        public void onDocOpened(PDFDoc document, int errCode) {
            if (errCode == Constants.e_ErrSuccess) {
                mShouldReloadData = true;
                bDocClosed = false;
            }
        }

        @Override
        public void onDocWillClose(PDFDoc document) {
            mShouldReloadData = false;
            mFileAttachmentAdapter.release();
        }

        @Override
        public void onDocClosed(PDFDoc document, int errCode) {
            bDocClosed = true;
        }

    };

    private final PDFViewCtrl.IRecoveryEventListener recoveryEventListener = new PDFViewCtrl.IRecoveryEventListener() {
        @Override
        public void onWillRecover() {

        }

        @Override
        public void onRecovered() {
            startSearchFileAttachment(true);
            if (mLoadingView == null) return;
            mLoadingView.setVisibility(View.VISIBLE);
            mNoInfoContainer.setVisibility(View.GONE);
        }
    };

    private final IInteractionEventListener mInteractionEventListener = new IInteractionEventListener() {
        @Override
        public boolean onKeyDown(Activity act, int keyCode, KeyEvent event) {
            if (keyCode == KeyEvent.KEYCODE_BACK) {
                if (mFileAttachmentAdapter.isEditState()) {
                    switchEditState(false);
                    return true;
                }
            }
            return false;
        }
    };

    private final IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            if (mTopBarView == null) return;
            if (mfileSelectDialog != null) {
                mfileSelectDialog.dismiss();
                mfileSelectDialog = null;
            }
            if (mClearAllDialog != null) {
                mClearAllDialog.dismiss();
                mClearAllDialog = null;
            }
            mFileAttachmentAdapter.dismissDialog();
            mCloseView.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
            mEditViewTv.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
            mNoInfoIv.setColorFilter(ThemeConfig.getInstance(mContext).getPrimaryColor());
            mNoInfoTv.setTextColor(AppResource.getColor(mContext, R.color.t2));
            mFileAttachmentAdapter.notifyUpdateData();
            mRlTopContainer.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
            mRlContainer.setBackgroundColor(AppResource.getColor(mContext, R.color.b1));
            mTitleView.setTextColor(AppResource.getColor(mContext, R.color.t4));
            mLoadingView.setTextColor(AppResource.getColor(mContext, R.color.t3));
            mAddView.setTextColor(AppResource.getColor(mContext, R.color.t4));
            mAddView.setBackground(AppResource.getDrawable(mContext, R.drawable.bottom_menu_bg));
            mBottomView.setBackground(AppResource.getDrawable(mContext, R.drawable.bottom_menu_bg));
            mSelectedAllTv.setTextColor(ThemeUtil.getToolbarTextColor(mContext));
            ThemeUtil.setTintList(mDeleteIv, ThemeUtil.getPrimaryIconColor(mContext));
        }
    };

    private final AnnotEventListener mAnnotEventListener = new AnnotEventListener() {
        @Override
        public void onAnnotAdded(PDFPage page, Annot annot) {
            try {
                if (annot.getType() == Annot.e_FileAttachment) {
                    Message message = Message.obtain();
                    message.what = ADD_ANNOT;
                    message.obj = annot;
                    mHandler.sendMessage(message);
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }

        @Override
        public void onAnnotWillDelete(PDFPage page, Annot annot) {
            try {
                if (GroupManager.getInstance().isGrouped(mPdfViewCtrl, annot)) {

                    MarkupArray markupArray = ((Markup) annot).getGroupElements();
                    if (markupArray.getSize() == 2) {
                        ArrayList<Annot> groupAnnots = new ArrayList<>();
                        for (int i = 0; i < markupArray.getSize(); i++) {
                            Annot groupAnnot = markupArray.getAt(i);
                            if (AppAnnotUtil.getAnnotUniqueID(groupAnnot).equals(AppAnnotUtil.getAnnotUniqueID(annot)))
                                continue;
                            groupAnnots.add(groupAnnot);
                        }

                        if (groupAnnots.size() == 1 && groupAnnots.get(0).getType() == Annot.e_FileAttachment) {
                            Message message = Message.obtain();
                            message.what = UN_GROUP_ANNOT;

                            Annot unGroupAnnot = groupAnnots.get(0);
                            FileBean fileBean = new FileBean();
                            fileBean.setPageIndex(unGroupAnnot.getPage().getIndex());
                            fileBean.setUuid(AppAnnotUtil.getAnnotUniqueID(unGroupAnnot));
                            fileBean.setCanDelete(!(AppAnnotUtil.isLocked(annot) || AppAnnotUtil.isReadOnly(annot)));
                            message.obj = fileBean;
                            mHandler.sendMessage(message);
                        }
                    }
                }

                if (annot.getType() == Annot.e_FileAttachment) {
                    Message message = Message.obtain();
                    message.what = DELETTE_ANNOT;
                    message.arg1 = annot.getPage().getIndex();
                    message.obj = AppAnnotUtil.getAnnotUniqueID(annot);
                    mHandler.sendMessage(message);
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }

        @Override
        public void onAnnotDeleted(PDFPage page, Annot annot) {

        }

        @Override
        public void onAnnotModified(PDFPage page, Annot annot) {
            try {
                if (annot.getType() == Annot.e_FileAttachment) {
                    Message message = Message.obtain();
                    message.what = MODIFIED_ANNOT;
                    message.obj = annot;
                    mHandler.sendMessage(message);
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }

        @Override
        public void onAnnotChanged(Annot lastAnnot, Annot currentAnnot) {

        }
    };

    private final IFlattenEventListener mFlattenEventListener = new IFlattenEventListener() {
        @Override
        public void onAnnotWillFlatten(PDFPage page, Annot annot) {
            try {
                if (GroupManager.getInstance().isGrouped(mPdfViewCtrl, annot)) {

                    MarkupArray markupArray = ((Markup) annot).getGroupElements();
                    if (markupArray.getSize() == 2) {
                        ArrayList<Annot> groupAnnots = new ArrayList<>();
                        for (int i = 0; i < markupArray.getSize(); i++) {
                            Annot groupAnnot = markupArray.getAt(i);
                            if (AppAnnotUtil.getAnnotUniqueID(groupAnnot).equals(AppAnnotUtil.getAnnotUniqueID(annot)))
                                continue;
                            groupAnnots.add(groupAnnot);
                        }

                        if (groupAnnots.size() == 1 && groupAnnots.get(0).getType() == Annot.e_FileAttachment) {
                            Message message = Message.obtain();
                            message.what = UN_GROUP_ANNOT;

                            Annot unGroupAnnot = groupAnnots.get(0);
                            FileBean fileBean = new FileBean();
                            fileBean.setPageIndex(unGroupAnnot.getPage().getIndex());
                            fileBean.setUuid(AppAnnotUtil.getAnnotUniqueID(unGroupAnnot));
                            fileBean.setCanDelete(!(AppAnnotUtil.isLocked(annot) || AppAnnotUtil.isReadOnly(annot)));
                            message.obj = fileBean;
                            mHandler.sendMessage(message);
                        }
                    }
                }

                if (annot.getType() == Annot.e_FileAttachment) {
                    Message message = Message.obtain();
                    message.what = DELETTE_ANNOT;
                    message.arg1 = annot.getPage().getIndex();
                    message.obj = AppAnnotUtil.getAnnotUniqueID(annot);
                    mHandler.sendMessage(message);
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }

        @Override
        public void onAnnotsFlattened(PDFDoc doc) {
            if (mContentView == null) return;
            if (mFileAttachmentAdapter.isEmpty()) return;

            startSearchFileAttachment(false);
        }
    };

    private final IRedactionEventListener mRedactionEventListener = new IRedactionEventListener() {

        @Override
        public void onAnnotsApplied(PDFDoc doc) {
            if (mContentView == null) return;
            if (mFileAttachmentAdapter.isEmpty()) return;

            startSearchFileAttachment(false);
        }

        @Override
        public void onAnnotApplied(PDFPage page, Annot annot) {
            if (mContentView == null) return;
            if (mFileAttachmentAdapter.isEmpty()) return;

            startSearchFileAttachment(false);
        }
    };

    private final IImportAnnotsEventListener mImportAnnotEventListener = new IImportAnnotsEventListener() {
        @Override
        public void onAnnotsImported() {
            if (mContentView == null) return;

            startSearchFileAttachment(false);
        }
    };

    private final IGroupEventListener mGroupEventListener = new IGroupEventListener() {
        @Override
        public void onAnnotGrouped(PDFPage page, List<Annot> groupAnnots) {
            updateFileBean(groupAnnots);
        }

        @Override
        public void onAnnotUnGrouped(PDFPage page, List<Annot> unGroupAnnots) {
            updateFileBean(unGroupAnnots);
        }

        private void updateFileBean(List<Annot> annots) {
            try {
                for (int i = 0; i < annots.size(); i++) {
                    Annot annot = annots.get(i);
                    if (annot != null
                            && !annot.isEmpty()
                            && annot.getType() == Annot.e_FileAttachment) {
                        Message message = Message.obtain();
                        message.what = GROUP_ANNOT;
                        message.obj = annot;
                        mHandler.sendMessage(message);
                    }
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
    };

    private final Handler mHandler = new Handler(Looper.getMainLooper()) {
        @Override
        public void handleMessage(Message msg) {
            super.handleMessage(msg);
            if (bDocClosed) return;

            switch (msg.what) {
                case ADD_ANNOT:
                    mFileAttachmentAdapter.add((Annot) msg.obj);
                    break;
                case MODIFIED_ANNOT:
                    mFileAttachmentAdapter.updateByOutside((Annot) msg.obj);
                    break;
                case DELETTE_ANNOT:
                    mFileAttachmentAdapter.removeItemBean(msg.arg1, (String) msg.obj);
                    break;
                case GROUP_ANNOT:
                    mFileAttachmentAdapter.groupAnnot((Annot) msg.obj);
                    break;
                case UN_GROUP_ANNOT:
                    mFileAttachmentAdapter.unGroupAnnot((FileBean) msg.obj);
                    break;
                default:
                    break;
            }
        }
    };

    public boolean onKeyDown(int keyCode, KeyEvent event) {
        if (openView == null) {
            return false;
        } else if (openView.getVisibility() == View.VISIBLE && keyCode == KeyEvent.KEYCODE_BACK) {
            openView.closeAttachment();
            openView.setVisibility(View.GONE);
            return true;
        }
        return false;
    }

    public boolean onKeyBack() {
        if (openView == null) {
            return false;
        } else if (openView.getVisibility() == View.VISIBLE) {
            openView.closeAttachment();
            openView.setVisibility(View.GONE);
            return true;
        }
        return false;
    }

    @Override
    public boolean unloadModule() {
        mUiExtensionsManager.getPanelManager().removePanel(this);
        mUiExtensionsManager.getPanelManager().unregisterPanelEventListener(mPanelEventListener);
        mPdfViewCtrl.unregisterDocEventListener(mDocEventListener);
        mPdfViewCtrl.unregisterPageEventListener(mPageEventListener);

        mUiExtensionsManager.getDocumentManager().unregisterAnnotEventListener(mAnnotEventListener);
        mUiExtensionsManager.getDocumentManager().unregisterImportedAnnotsEventListener(mImportAnnotEventListener);
        mUiExtensionsManager.getDocumentManager().unregisterFlattenEventListener(mFlattenEventListener);
        mUiExtensionsManager.getDocumentManager().unregisterRedactionEventListener(mRedactionEventListener);
        mUiExtensionsManager.getDocumentManager().unregisterGroupEventListener(mGroupEventListener);
        mUiExtensionsManager.unregisterConfigurationChangedListener(mConfigurationChangedListener);
        mUiExtensionsManager.unregisterInteractionListener(mInteractionEventListener);
        mUiExtensionsManager.unregisterThemeEventListener(mThemeEventListener);
        mParent = null;
        return true;
    }

    @Override
    public String getName() {
        return MODULE_NAME_FILE_PANEL;
    }

    @Override
    public int getPanelType() {
        return PanelSpec.ATTACHMENTS;
    }

    @Override
    public int getIcon() {
        return R.drawable.panel_tab_attachment;
    }

    @Override
    public ColorStateList getIconTint() {
        return null;
    }

    @Override
    public View getTopToolbar() {
        return mTopBarView;
    }

    @Override
    public View getContentView() {
        return mContentView;
    }

    @Override
    public void onActivated() {
        if (openView == null)
            openView = new FileSpecOpenView(mContext, mPdfViewCtrl, mParent);

        if (mFileAttachmentAdapter != null) {
            int status = mFileAttachmentAdapter.getCurrentStatus();
            switch (status) {
                case FileAttachmentPresenter.FileAttachmentViewer.STATUS_PREPARE:
                case FileAttachmentPresenter.FileAttachmentViewer.STATUS_CANCEL:
                    if (mShouldReloadData) {
                        mShouldReloadData = false;
                        startSearchFileAttachment(true);
                    }
                    break;
                case FileAttachmentPresenter.FileAttachmentViewer.STATUS_LOADING:
                    mFileAttachmentAdapter.setPause(false);
                    break;
                case FileAttachmentPresenter.FileAttachmentViewer.STATUS_PAUSED:
                    mFileAttachmentAdapter.reStart();
                    break;
                case FileAttachmentPresenter.FileAttachmentViewer.STATUS_DONE:
                case FileAttachmentPresenter.FileAttachmentViewer.STATUS_FAILED:
                default:
                    updateUIState(status);
                    break;
            }
        }
    }

    private void resetAddView() {
        if (mAddView == null) return;
        boolean enable = !mPdfViewCtrl.isDynamicXFA()
                && mUiExtensionsManager.canModifyContents()
                && mUiExtensionsManager.isEnableModification();
        mAddView.setVisibility(enable ? View.VISIBLE : View.GONE);
    }

    private void hideAddView() {
        mAddView.setVisibility(View.GONE);
    }

    @Override
    public void onDeactivated() {
        if (mFileAttachmentAdapter.isEditState())
            switchEditState(false);
        openView = null;

        if (mFileAttachmentAdapter.getCurrentStatus() == FileAttachmentPresenter.FileAttachmentViewer.STATUS_LOADING) {
            mFileAttachmentAdapter.setPause(true);
        }
    }

    @Override
    public void success(List<FileBean> fileList) {
        updateUIState();
    }

    @Override
    public void loading(int pageIndex, int total) {
        if (mContentView == null) return;

        mEditViewTv.setEnabled(false);
        mNoInfoContainer.setVisibility(View.GONE);
        mLoadingView.setVisibility(View.VISIBLE);
        mLoadingView.setText(AppResource.getString(mContext, R.string.rv_panel_annot_loading_pagenum, pageIndex, total));
    }

    @Override
    public void pause(int pageIndex) {
        updateUIState();
    }

    @Override
    public void fail(int errCode) {
        if (errCode == FileAttachmentPresenter.FileAttachmentViewer.DELETE) {
            UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.rv_panel_annot_failed));
        }
        updateUIState();
    }

    private void updateUIState() {
        updateUIState(mFileAttachmentAdapter.getCurrentStatus());
    }

    private void updateUIState(int state) {
        if (mContentView == null) return;

        switch (state) {
            case FileAttachmentPresenter.FileAttachmentViewer.STATUS_PREPARE:
            case FileAttachmentPresenter.FileAttachmentViewer.STATUS_CANCEL:
                mLoadingView.setText(AppResource.getString(mContext, R.string.rv_panel_annot_loading_start));
            case FileAttachmentPresenter.FileAttachmentViewer.STATUS_LOADING:
            case FileAttachmentPresenter.FileAttachmentViewer.STATUS_PAUSED:
                mEditViewTv.setEnabled(false);
                mLoadingView.setVisibility(View.VISIBLE);
                mNoInfoContainer.setVisibility(View.GONE);
                break;
            case FileAttachmentPresenter.FileAttachmentViewer.STATUS_DONE:
            case FileAttachmentPresenter.FileAttachmentViewer.STATUS_FAILED:
            default:
                mLoadingView.setVisibility(View.GONE);
                if (mFileAttachmentAdapter.isEmpty()) {
                    if (mFileAttachmentAdapter.isEditState())
                        switchEditState(false);
                    mEditViewTv.setEnabled(false);
                    mNoInfoContainer.setVisibility(View.VISIBLE);
                } else {
                    if (mFileAttachmentAdapter.getCanEditItemCount() > 0) {
                        if (mUiExtensionsManager.getPermissionProvider() != null) {
                            mUiExtensionsManager.getPermissionProvider().checkPermission(IPermissionProvider.FUNCTION_ATTACHMENT_MODIFY, new IPermissionProvider.IPermissionState() {
                                @Override
                                public void onState(int state) {
                                    mEditViewTv.setEnabled(state == 0);
                                }
                            });
                        } else {
                            mEditViewTv.setEnabled(true);
                        }
                    } else {
                        mEditViewTv.setEnabled(false);
                        if (mFileAttachmentAdapter.isEditState())
                            switchEditState(false);
                    }
                    mNoInfoContainer.setVisibility(View.GONE);
                }
                break;
        }
    }

    @Override
    public void showProgressDlg(int type) {
        String msg;
        if (type == FileAttachmentPresenter.FileAttachmentViewer.OPER_DELETE)
            msg = AppResource.getString(mContext, R.string.rv_panel_annot_deleting);
        else
            msg = AppResource.getString(mContext, R.string.fx_string_opening);
        showProgressDlg(msg);
    }

    @Override
    public void openDoc(String path, String filename) {
        openView.openAttachment(path, filename, new IAttachmentDocEvent() {
            @Override
            public void onAttachmentDocWillOpen() {
                showProgressDlg(AppResource.getString(mContext, R.string.fx_string_opening));
            }

            @Override
            public void onAttachmentDocOpened(PDFDoc document, int errCode) {
                if (AppDisplay.isPad()) {
                    mLastReadState = mUiExtensionsManager.getState();
                    if (mLastReadState == ReadStateConfig.STATE_REFLOW) {
                        ReflowModule reflowModule = (ReflowModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_REFLOW);
                        reflowModule.setDragToolbarVisiable(false);
                    } else if (mLastReadState == ReadStateConfig.STATE_TTS) {
                        TTSModule ttsModule = (TTSModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_TTS);
                        ttsModule.setDragToolbarVisible(false);
                    } else if (mLastReadState == ReadStateConfig.STATE_AUTOFLIP) {
                        AutoFlipModule autoFlipModule = (AutoFlipModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_AUTOFLIP);
                        autoFlipModule.setDragToolbarVisible(false);
                    }
                }
                if (SystemUiHelper.getInstance().isFullScreen())
                    SystemUiHelper.getInstance().hideSystemUI(mUiExtensionsManager.getAttachedActivity());
                dismissProgressDlg();
            }

            @Override
            public void onAttachmentDocWillClose() {
                if (AppDisplay.isPad() && mLastReadState != ReadStateConfig.STATE_NORMAL) {
                    if (mLastReadState == ReadStateConfig.STATE_REFLOW) {
                        ReflowModule reflowModule = (ReflowModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_REFLOW);
                        reflowModule.setDragToolbarVisiable(true);
                    } else if (mLastReadState == ReadStateConfig.STATE_TTS) {
                        TTSModule ttsModule = (TTSModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_TTS);
                        ttsModule.setDragToolbarVisible(true);
                    } else if (mLastReadState == ReadStateConfig.STATE_AUTOFLIP) {
                        AutoFlipModule autoFlipModule = (AutoFlipModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_AUTOFLIP);
                        autoFlipModule.setDragToolbarVisible(true);
                    }
                }
            }

            @Override
            public void onAttachmentDocClosed() {
            }
        });
        openView.setVisibility(View.VISIBLE);
    }

    @Override
    public void dismissProgressDlg(int type) {
        dismissProgressDlg();
    }

    private ProgressDialog mProgressDlg;

    private void showProgressDlg(String msg) {
        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();

        if (mProgressDlg == null && uiExtensionsManager.getAttachedActivity() != null) {
            mProgressDlg = new ProgressDialog(uiExtensionsManager.getAttachedActivity());
            mProgressDlg.setProgressStyle(ProgressDialog.STYLE_SPINNER);
            mProgressDlg.setCancelable(false);
            mProgressDlg.setIndeterminate(false);
        }

        if (mProgressDlg != null && !mProgressDlg.isShowing()) {
            mProgressDlg.setMessage(msg);
            AppDialogManager.getInstance().showAllowManager(mProgressDlg, null);
        }
    }

    private void dismissProgressDlg() {
        if (mProgressDlg != null && mProgressDlg.isShowing()) {
            AppDialogManager.getInstance().dismiss(mProgressDlg);
            mProgressDlg = null;
        }
    }

}
