/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.panel.bookmark;

import android.app.Activity;
import android.content.Context;
import android.content.res.ColorStateList;
import android.content.res.Configuration;
import android.graphics.PorterDuff;
import android.graphics.Rect;
import android.os.Build;
import android.os.Handler;
import android.text.Editable;
import android.text.TextWatcher;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.inputmethod.InputMethodManager;
import android.widget.Button;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.uiextensions.IInteractionEventListener;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.browser.PullRecyclerView;
import com.foxit.uiextensions.browser.behavior.NestedHeaderBehavior;
import com.foxit.uiextensions.controls.dialog.UIDialog;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.controls.dialog.sheetmenu.ISheetMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UIBottomSheetMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UISheetMenu;
import com.foxit.uiextensions.controls.panel.PanelSpec;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.controls.toolbar.IBaseItem;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.impl.BaseItemImpl;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.event.PageEventListener;
import com.foxit.uiextensions.modules.panel.IPanelManager;
import com.foxit.uiextensions.modules.panel.PanelSearchView;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.FragmentActivity;
import androidx.recyclerview.widget.DefaultItemAnimator;
import androidx.recyclerview.widget.ItemTouchHelper;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

/**
 * The module enable user to add or delete their own customized bookmark to the PDF document.
 */
public class ReadingBookmarkModule implements Module, PanelSpec {

    private static final int INPUT_MAX_LENGTH = 200;

    private ArrayList<IBaseItem> mMarkItemList;
    private Context mContext;
    private UITextEditDialog mDialog;
    private ReadingBookmarkAdapter mReadingBookmarkAdapter;
    private RecyclerView mReadingMarkRV;
    private LinearLayoutManager mLinearLayoutManager;

    private PDFViewCtrl mPdfViewCtrl;
    private View mTopBarView;
    private PanelSearchView mSearchView;
    private TextView mEditView;
    private TextView mCloseView;
    private View mBottomView;
    private TextView mSelectedAllTv;
    private ImageView mDeleteIv;
    private View mContentView;

    private View mNoReadingBookmarkContainer;
    private TextView mNoReadingBookmarkTv;
    private ImageView mNoReadingBookmarkIv;
    private PullRecyclerView mPullListView;

    private UIExtensionsManager mUiExtensionsManager;
    private IBaseItem mBookmarkAddButton;
    private ISheetMenu mSheetMenu;

    private boolean mIsNeedReloadData = false;
    private boolean mIsReadingBookmark = false;
    private boolean mIsPad;
    UITextEditDialog mRenameDlg;
    private RelativeLayout mRlTopContainer;
    private TextView mTitleView;

    public ReadingBookmarkModule(Context context, ViewGroup parent, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        if (context == null || pdfViewCtrl == null) {
            throw new NullPointerException();
        }
        mContext = context.getApplicationContext();
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = (UIExtensionsManager) uiExtensionsManager;
        mMarkItemList = new ArrayList<IBaseItem>();
        mIsPad = AppDisplay.isPad();

        if (mUiExtensionsManager.getConfig().uiSettings.fullscreen) {
            mFullScreenFavoriteItem = mUiExtensionsManager.getMainFrame().getContentView().findViewById(R.id.read_fullscreen_favorite);
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                mFullScreenFavoriteItem.setForceDarkAllowed(false);
            }
        }
    }

    public void changeMarkItemState(boolean mark) {
        mIsReadingBookmark = mark;
        for (IBaseItem item : mMarkItemList) {
            item.setSelected(mIsReadingBookmark);
        }
    }

    public void addMarkedItem(final IBaseItem item) {
        if (mMarkItemList.contains(item))
            return;
        mMarkItemList.add(item);

        View.OnClickListener listener = new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_Bookmark);
                mIsReadingBookmark = !isMarked(mPdfViewCtrl.getCurrentPage());
                if (mIsReadingBookmark) {
                    addMark(mPdfViewCtrl.getCurrentPage());
                } else {
                    removeMark(mPdfViewCtrl.getCurrentPage());
                }
                changeMarkItemState(mIsReadingBookmark);
                changeViewState(mReadingBookmarkAdapter.getItemCount() != 0);
                mUiExtensionsManager.getDocumentManager().setDocModified(true);
                if(AppDisplay.isPad()) {
                    item.setImageTintList(ThemeUtil.getTopBarIcon2ColorStateList(mContext));
                }else{
                    item.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
                }
            }
        };

        item.setOnClickListener(listener);
    }

    public void removeMarkedItem(IBaseItem item) {
        if (!mMarkItemList.contains(item)) return;
        mMarkItemList.remove(item);
    }

    private IInteractionEventListener mInteractionEventListener = new IInteractionEventListener() {
        @Override
        public boolean onKeyDown(Activity act, int keyCode, KeyEvent event) {
            if (keyCode == KeyEvent.KEYCODE_BACK) {
                if (mReadingBookmarkAdapter.isSearchState()) {
                    switchSearchState(false);
                    return true;
                }
                if (mReadingBookmarkAdapter.isEditState()) {
                    switchEditState(false);
                    return true;
                }
            }
            return false;
        }
    };

    private final UIExtensionsManager.ConfigurationChangedListener mConfigurationChangedListener = new UIExtensionsManager.ConfigurationChangedListener() {
        @Override
        public void onConfigurationChanged(Configuration newConfig) {
            updateMenu();
        }
    };

    private void updateMenu() {
        if (mSheetMenu != null && mSheetMenu.isShowing()) {
            if (mIsPad) {
                new Handler().postDelayed(new Runnable() {
                    @Override
                    public void run() {
                        View view = mLinearLayoutManager.findViewByPosition(mReadingBookmarkAdapter.getSelectedPosition());
                        if (view != null) {
                            View moreView = view.findViewById(R.id.rd_panel_item_more);
                            showSheetMenu(moreView);
                        } else {
                            mShouldUpdateMenu = true;

                            int scrollPosition = Math.min(mReadingBookmarkAdapter.getItemCount() - 1,
                                    mReadingBookmarkAdapter.getSelectedPosition() + 1);
                            mLinearLayoutManager.scrollToPosition(scrollPosition);
                        }
                    }
                }, 380);
            } else {
                showSheetMenu(mUiExtensionsManager.getRootView());
            }
        }
    }

    private boolean mShouldUpdateMenu = false;

    private void onScrolled(LinearLayoutManager layoutManager) {
        int selectedPosition = mReadingBookmarkAdapter.getSelectedPosition();
        if (selectedPosition == -1 || !mShouldUpdateMenu) return;

        mShouldUpdateMenu = false;
        View view = layoutManager.findViewByPosition(mReadingBookmarkAdapter.getSelectedPosition());
        if (view != null) {
            View moreView = view.findViewById(R.id.panel_item_fileattachment_more);
            showSheetMenu(moreView);
        } else {
            dismissSheetMenu();
        }
    }

    private IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            if(AppDisplay.isPad()){
                mBookmarkAddButton.setImageTintList(ThemeUtil.getTopBarIcon2ColorStateList(mContext));
            }else{
                mBookmarkAddButton.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
            }
            updateBookmarkItemDisabledColor(mBookmarkAddButton);
            updateFullScreenBookmarkDisabledColor();
            if (mTopBarView == null) return;
            if (mDialog != null) {
                mDialog.dismiss();
                mDialog = null;
            }
            if (mRenameDlg != null) {
                mRenameDlg.dismiss();
                mRenameDlg = null;
            }
            mSearchView.onThemeColorChanged();
            mCloseView.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
            mEditView.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
            mNoReadingBookmarkIv.setColorFilter(ThemeConfig.getInstance(mContext).getPrimaryColor());
            mReadingBookmarkAdapter.notifyUpdateData();
            mRlTopContainer.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
            mTitleView.setTextColor(AppResource.getColor(mContext, R.color.t4));
            mNoReadingBookmarkTv.setTextColor(AppResource.getColor(mContext, R.color.t2));
            readingMarkContent.setBackgroundColor(AppResource.getColor(mContext, R.color.b1));
            if (mSheetMenu != null) {
                mSheetMenu.dismiss();
                mSheetMenu = null;
            }
            mBottomView.setBackground(AppResource.getDrawable(mContext, R.drawable.bottom_menu_bg));
            mSelectedAllTv.setTextColor(ThemeUtil.getToolbarTextColor(mContext));
            ThemeUtil.setTintList(mDeleteIv, ThemeUtil.getPrimaryIconColor(mContext));
        }
    };

    private final PDFViewCtrl.IDocEventListener mDocEventListener = new DocEventListener() {
        @Override
        public void onDocWillOpen() {
        }

        @Override
        public void onDocOpened(PDFDoc pdfDoc, int errCode) {
            if (errCode != Constants.e_ErrSuccess) {
                return;
            }
            if (mPdfViewCtrl.isDynamicXFA()) {
                mBookmarkAddButton.setEnable(false);
            } else {
                boolean canAssemble = mUiExtensionsManager.getDocumentManager().canAssemble() && mUiExtensionsManager.isEnableModification();
                mBookmarkAddButton.setEnable(canAssemble);
                if (mUiExtensionsManager.getConfig().uiSettings.fullscreen && mFullScreenFavoriteItem != null) {
                    mFullScreenFavoriteItem.setEnabled(canAssemble);
                }
                if (mEditView != null)
                    mEditView.setEnabled(canAssemble);
            }
            prepareReadingBookMark();
            IPanelManager panelManager = mUiExtensionsManager.getPanelManager();
            if (panelManager.isShowingPanel()
                    && panelManager.getPanelHost().getCurrentSpec() == ReadingBookmarkModule.this) {
                panelManager.getPanelHost().setCurrentSpec(getPanelType());
            }
        }

        @Override
        public void onDocWillClose(PDFDoc pdfDoc) {
            if (mPullListView == null) return;
            mPullListView.setHeaderState(NestedHeaderBehavior.HIDE);
        }

        @Override
        public void onDocClosed(PDFDoc pdfDoc, int i) {
            if (mReadingBookmarkAdapter != null)
                mReadingBookmarkAdapter.release();
        }

    };

    public void prepareReadingBookMark() {
        if (mReadingBookmarkAdapter == null) return;

        mReadingBookmarkAdapter.initBookmarkList();
        remarkItemState(mPdfViewCtrl.getCurrentPage());
    }

    private final PDFViewCtrl.IPageEventListener mPageEventListener = new PageEventListener() {

        @Override
        public void onPageChanged(int oldPageIndex, int curPageIndex) {
            if (curPageIndex < 0) {
                curPageIndex = 0;
            }
            remarkItemState(curPageIndex);
        }

        @Override
        public void onPageMoved(boolean success, int index, int dstIndex) {
            mReadingBookmarkAdapter.onPageMoved(success, index, dstIndex);
            remarkItemState(mPdfViewCtrl.getCurrentPage());
        }

        @Override
        public void onPagesRemoved(boolean success, int[] pageIndexes) {
            for (int i = 0; i < pageIndexes.length; i++) {
                mReadingBookmarkAdapter.onPageRemoved(success, pageIndexes[i] - i);
            }
            mIsNeedReloadData = true;
            remarkItemState(mPdfViewCtrl.getCurrentPage());
        }

        @Override
        public void onPagesInserted(boolean success, int dstIndex, int[] range) {
            mReadingBookmarkAdapter.onPagesInsert(success, dstIndex, range);
            remarkItemState(mPdfViewCtrl.getCurrentPage());
        }
    };

    private ImageView mFullScreenFavoriteItem;

    public void remarkItemState(final int index) {
        mUiExtensionsManager.getAttachedActivity().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                boolean isMarked = isMarked(index);
                for (IBaseItem item : mMarkItemList) {
                    item.setSelected(isMarked);
                    if (AppDisplay.isPad()) {
                        item.setImageTintList(ThemeUtil.getTopBarIcon2ColorStateList(mContext));
                    }else{
                        item.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
                    }
                    updateBookmarkItemDisabledColor(item);
                }
                if (mFullScreenFavoriteItem != null) {
                    mFullScreenFavoriteItem.setSelected(isMarked);
                    updateFullScreenBookmarkDisabledColor();
                }
            }
        });
    }

    private void changeViewState(boolean enable) {
        if (mEditView == null) return;
        boolean canAssemble = mUiExtensionsManager.getDocumentManager().canAssemble() && mUiExtensionsManager.isEnableModification();
        mEditView.setEnabled(enable && canAssemble);
        if (enable) {
            if (mReadingBookmarkAdapter.isSearchState()) {
                mPullListView.setHeaderState(NestedHeaderBehavior.FIXED);
            } else if (mReadingBookmarkAdapter.isEditState()) {
                mPullListView.setHeaderState(NestedHeaderBehavior.HIDE);
            } else {
                mPullListView.setHeaderState(NestedHeaderBehavior.SHOW);
            }
            mNoReadingBookmarkContainer.setVisibility(View.GONE);
        } else {
            mPullListView.setHeaderState(NestedHeaderBehavior.HIDE);
            mNoReadingBookmarkTv.setText(AppResource.getString(mContext, R.string.rv_panel_readingbookmark_tips_noinfo));
            mNoReadingBookmarkIv.setImageResource(R.drawable.panel_no_bookmark);
            mNoReadingBookmarkContainer.setVisibility(View.VISIBLE);
        }
    }

    @Override
    public boolean loadModule() {
        mBookmarkAddButton = new BaseItemImpl(mContext, R.drawable.rd_bottom_bookmark_selector);
        mBookmarkAddButton.setForceDarkAllowed(false);
        mBookmarkAddButton.setId(R.id.id_bottom_bar_bookmark);
        if(AppDisplay.isPad()){
            mBookmarkAddButton.setImageTintList(ThemeUtil.getTopBarIcon2ColorStateList(mContext));
        }else{
            mBookmarkAddButton.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
        }
        addMarkedItem(mBookmarkAddButton);
        if (AppDisplay.isPad()) {
            mBookmarkAddButton.setTag(ToolbarItemConfig.ITEM_TOPBAR_BOOKMARK);
            mUiExtensionsManager.getMainFrame().getTopToolbar().addView(mBookmarkAddButton, BaseBar.TB_Position.Position_RB);
        } else {
            mBookmarkAddButton.setTag(ToolbarItemConfig.ITEM_BOTTOMBAR_BOOKMARK);
            mUiExtensionsManager.getMainFrame().getBottomToolbar().addView(mBookmarkAddButton, BaseBar.TB_Position.Position_CENTER);
        }

        mReadingBookmarkAdapter = new ReadingBookmarkAdapter(mContext, mPdfViewCtrl, new ReadingBookmarkAdapter.OnBookmarkItemClickListener() {
            @Override
            public void onMoreClick(int position, View view) {
                showSheetMenu(position, view);
            }

            @Override
            public void onItemClick(int position, ReadingBookmarkAdapter.ReadingBookmarkNode node) {
                if (AppUtil.isFastDoubleClick()) return;
                List<ReadingBookmarkAdapter.ReadingBookmarkNode> editNodes = mReadingBookmarkAdapter.getEditNodes();
                if (mReadingBookmarkAdapter.isEditState()) {
                    if (editNodes.contains(node)) {
                        editNodes.remove(node);
                    } else {
                        editNodes.add(node);
                    }
                    mReadingBookmarkAdapter.notifyItemChanged(position);
                    mDeleteIv.setEnabled(editNodes.size() > 0);
                    if (mReadingBookmarkAdapter.isSelectedAll())
                        mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_deselect_all));
                    else
                        mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_select_all));
                } else {
                    mPdfViewCtrl.gotoPage(node.getIndex());
                    if (mUiExtensionsManager.getPanelManager().isShowingPanel())
                        mUiExtensionsManager.getPanelManager().hidePanel();
                }
            }
        });

        mUiExtensionsManager.registerModule(this);
        mUiExtensionsManager.registerConfigurationChangedListener(mConfigurationChangedListener);
        mUiExtensionsManager.registerInteractionListener(mInteractionEventListener);
        mUiExtensionsManager.registerThemeEventListener(mThemeEventListener);
        mPdfViewCtrl.registerDocEventListener(mDocEventListener);
        mPdfViewCtrl.registerPageEventListener(mPageEventListener);
        mUiExtensionsManager.getPanelManager().registerPanelEventListener(mPanelEventListener);
        return true;
    }

    private final IPanelManager.OnPanelEventListener mPanelEventListener = new IPanelManager.OnPanelEventListener() {
        @Override
        public void onPanelOpened() {
            if (mTopBarView != null) return;
            initReadingBookmarkPanelView();
        }

        @Override
        public void onPanelClosed() {

        }
    };

    private void initReadingBookmarkPanelView() {
        mTopBarView = createTopView();
        mSearchView = createSearchView();
        mContentView = createContentView();
        mUiExtensionsManager.getPanelManager().addPanel(this);
    }

    private View createTopView() {
        View topBarView = View.inflate(mContext, R.layout.panel_topbar_layout, null);
        mRlTopContainer = topBarView.findViewById(R.id.panel_rl_top_container);
        mRlTopContainer.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
        mCloseView = topBarView.findViewById(R.id.panel_close_tv);
        mCloseView.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        mTitleView = topBarView.findViewById(R.id.panel_title_tv);
        mTitleView.setText(AppResource.getString(mContext, R.string.rv_panel_readingbookmark_title));
        mEditView = topBarView.findViewById(R.id.panel_edit_tv);
        mEditView.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        mEditView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                switchEditState(!mReadingBookmarkAdapter.isEditState());
            }
        });

        if (AppDevice.isChromeOs(mPdfViewCtrl.getAttachedActivity())) {
            mCloseView.setVisibility(View.VISIBLE);
        } else {
            if (mIsPad) {
                mCloseView.setVisibility(View.INVISIBLE);
            } else {
                mCloseView.setVisibility(View.VISIBLE);
            }
        }
        mCloseView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mUiExtensionsManager.getPanelManager().isShowingPanel()) {
                    mUiExtensionsManager.getPanelManager().hidePanel();
                }
            }
        });
        return topBarView;
    }

    private String mLastSearchText;

    private PanelSearchView createSearchView() {
        PanelSearchView searchView = new PanelSearchView(mContext);
        searchView.setSearchEventListener(new PanelSearchView.OnSearchEventListener() {
            @Override
            public void onQueryTextChange(String newText) {
                if (!AppUtil.isEqual(mLastSearchText, newText)) {
                    List<ReadingBookmarkAdapter.ReadingBookmarkNode> searchBookmarkNodes = mReadingBookmarkAdapter.getSearchNodes();
                    List<ReadingBookmarkAdapter.ReadingBookmarkNode> bookmarkNodes = mReadingBookmarkAdapter.getReadingBookmarkNodes();

                    mLastSearchText = newText.toLowerCase();
                    searchBookmarkNodes.clear();
                    if (AppUtil.isEmpty(mLastSearchText)) {
                        searchBookmarkNodes.addAll(bookmarkNodes);
                    } else {
                        for (int index = 0; index < bookmarkNodes.size(); index++) {
                            String content = bookmarkNodes.get(index).getTitle().toLowerCase();
                            if (content.contains(mLastSearchText)) {
                                boolean inserted = false;
                                for (int j = 0; j < searchBookmarkNodes.size(); j++) {
                                    if (bookmarkNodes.get(index).getIndex() < searchBookmarkNodes.get(j).getIndex()) {
                                        searchBookmarkNodes.add(j, bookmarkNodes.get(index));
                                        inserted = true;
                                        break;
                                    }
                                }
                                if (!inserted) {
                                    searchBookmarkNodes.add(bookmarkNodes.get(index));
                                }
                            }
                        }
                    }

                    updateUIState();
                }
            }

            @Override
            public void onFocusChange(View v, boolean hasFocus) {
                if (hasFocus) {
                    if (AppUtil.isEmpty(mSearchView.getSearchText())) {
                        mReadingBookmarkAdapter.getSearchNodes().clear();
                        mReadingBookmarkAdapter.getSearchNodes().addAll(mReadingBookmarkAdapter.getReadingBookmarkNodes());
                    }

                    switchSearchState(true);
                } else {
                    if (AppUtil.isEmpty(mSearchView.getSearchText())) {
                        switchSearchState(false);
                    }
                }
                updateUIState();
            }

            @Override
            public void onCancel() {
                mPullListView.hideHeaderView();
                switchSearchState(false);
            }
        });
        return searchView;
    }

    RelativeLayout readingMarkContent;

    private View createContentView() {
        View contentView = LayoutInflater.from(mContext).inflate(R.layout.panel_bookmark_contentview, null);

        mBottomView = contentView.findViewById(R.id.bookmark_panel_bottom_tool_bar);
        mDeleteIv = contentView.findViewById(R.id.panel_bottom_delete_iv);
        ThemeUtil.setTintList(mDeleteIv, ThemeUtil.getPrimaryIconColor(mContext));
        mDeleteIv.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mReadingBookmarkAdapter.isSelectedAll()) {
                    if (mPdfViewCtrl.getUIExtensionsManager() == null) return;
                    Context context = mUiExtensionsManager.getAttachedActivity();
                    if (context == null) return;
                    mDialog = new UITextEditDialog(context, UIDialog.NO_INPUT);
                    mDialog.setTitle(AppResource.getString(mContext, R.string.menu_more_confirm));
                    mDialog.getPromptTextView().setText(AppResource.getString(mContext, R.string.rd_panel_clear_readingbookmarks));
                    mDialog.getOKButton().setOnClickListener(new View.OnClickListener() {
                        @Override
                        public void onClick(View v) {
                            mDialog.dismiss();
                            mReadingBookmarkAdapter.clearAllNodes(new Event.Callback() {
                                @Override
                                public void result(Event event, boolean success) {
                                    if (success) {
                                        changeMarkItemState(false);
                                        switchEditState(false);
                                        mUiExtensionsManager.getDocumentManager().setDocModified(true);
                                    }
                                }
                            });
                        }
                    });
                    mDialog.show();
                } else {
                    List<ReadingBookmarkAdapter.ReadingBookmarkNode> editNodes = mReadingBookmarkAdapter.getEditNodes();
                    if (editNodes.size() > 0) {
                        for (int i = editNodes.size() - 1; i >= 0; i--) {
                            ReadingBookmarkAdapter.ReadingBookmarkNode bookmarkNode = editNodes.get(i);
                            deleteBookmark(bookmarkNode.getIndex());
                        }
                        if (mReadingBookmarkAdapter.getItemCount() == 0) {
                            switchEditState(false);
                        } else {
                            updateUIState();
                        }
                    }
                }
            }
        });
        mSelectedAllTv = contentView.findViewById(R.id.panel_bottom_select_all_tv);
        mSelectedAllTv.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                boolean isSelectedAll = mReadingBookmarkAdapter.isSelectedAll();
                if (isSelectedAll) {
                    mDeleteIv.setEnabled(false);
                    mReadingBookmarkAdapter.selectAll(false);
                    mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_select_all));
                } else {
                    mDeleteIv.setEnabled(true);
                    mReadingBookmarkAdapter.selectAll(true);
                    mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_deselect_all));
                }
            }
        });

        readingMarkContent = contentView.findViewById(R.id.panel_bookmark_content_root);
        mNoReadingBookmarkContainer = readingMarkContent.findViewById(R.id.panel_nobookmark_ll);
        mNoReadingBookmarkTv = readingMarkContent.findViewById(R.id.panel_nobookmark_tv);
        mNoReadingBookmarkIv = readingMarkContent.findViewById(R.id.panel_nobookmark_iv);
        mNoReadingBookmarkIv.setColorFilter(ThemeConfig.getInstance(mContext).getPrimaryColor());
        mPullListView = readingMarkContent.findViewById(R.id.panel_bookmark_lv);
        mPullListView.addHeaderView(mSearchView);
        mReadingMarkRV = mPullListView.getContentRV();
        mLinearLayoutManager = new LinearLayoutManager(mContext, LinearLayoutManager.VERTICAL, false);
        mReadingMarkRV.setLayoutManager(mLinearLayoutManager);
        mReadingMarkRV.setItemAnimator(new DefaultItemAnimator());
        mReadingMarkRV.setAdapter(mReadingBookmarkAdapter);
        if (AppDisplay.isPad()) {
            mReadingMarkRV.addOnScrollListener(new RecyclerView.OnScrollListener() {
                @Override
                public void onScrolled(@NonNull RecyclerView recyclerView, int dx, int dy) {
                    ReadingBookmarkModule.this.onScrolled(mLinearLayoutManager);
                }
            });
        }
        return contentView;
    }


    @Override
    public boolean unloadModule() {
        if (AppDisplay.isPad()) {
            mUiExtensionsManager.getMainFrame().getTopToolbar().removeItem(mBookmarkAddButton);
        } else {
            mUiExtensionsManager.getMainFrame().getBottomToolbar().removeItem(mBookmarkAddButton);
        }
        mMarkItemList.clear();

        mUiExtensionsManager.getPanelManager().removePanel(this);
        mUiExtensionsManager.getPanelManager().unregisterPanelEventListener(mPanelEventListener);
        mUiExtensionsManager.unregisterInteractionListener(mInteractionEventListener);
        mUiExtensionsManager.unregisterConfigurationChangedListener(mConfigurationChangedListener);
        mUiExtensionsManager.unregisterThemeEventListener(mThemeEventListener);
        mPdfViewCtrl.unregisterDocEventListener(mDocEventListener);
        mPdfViewCtrl.unregisterPageEventListener(mPageEventListener);
        return true;
    }

    @Override
    public String getName() {
        return MODULE_NAME_BOOKMARK;
    }

    @Override
    public void onActivated() {
        if (mReadingBookmarkAdapter.isEditState())
            switchEditState(false);
        if (mReadingBookmarkAdapter.isSearchState())
            switchSearchState(false);
        updateUIState();
        if (mIsNeedReloadData) {
            mReadingBookmarkAdapter.notifyDataSetChanged();
        }
    }

    @Override
    public void onDeactivated() {
        if (mReadingBookmarkAdapter.isEditState())
            switchEditState(false);
        if (mReadingBookmarkAdapter.isSearchState())
            switchSearchState(false);
    }

    @Override
    public View getTopToolbar() {
        return mTopBarView;
    }

    @Override
    public int getIcon() {
        return R.drawable.panel_tab_bookmark;
    }

    @Override
    public ColorStateList getIconTint() {
        return null;
    }

    @Override
    public int getPanelType() {
        return PanelSpec.BOOKMARKS;
    }

    @Override
    public View getContentView() {
        return mContentView;
    }

    private void updateUIState() {
        if (mReadingBookmarkAdapter.isSearchState()) {
            mNoReadingBookmarkTv.setText(AppResource.getString(mContext, R.string.fx_no_search_result));
            mNoReadingBookmarkIv.setImageResource(R.drawable.panel_no_search_result);
            mNoReadingBookmarkContainer.setVisibility(mReadingBookmarkAdapter.getItemCount() != 0 ? View.GONE : View.VISIBLE);
            mReadingBookmarkAdapter.setSearchPattern(mSearchView.getSearchText().toLowerCase());
        } else {
            changeViewState(mReadingBookmarkAdapter.getItemCount() != 0);
        }
        mReadingBookmarkAdapter.notifyUpdateData();
    }

    private void switchEditState(boolean toEditState) {
        if (mReadingBookmarkAdapter.isEditState() == toEditState) return;

        mReadingBookmarkAdapter.switchEditState(toEditState);
        mReadingBookmarkAdapter.getEditNodes().clear();
        mDeleteIv.setEnabled(false);
        if (toEditState) {
            mEditView.setText(AppResource.getString(mContext, R.string.fx_string_done));
            mPullListView.setHeaderState(NestedHeaderBehavior.HIDE);
            mBottomView.setVisibility(View.VISIBLE);
            mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_select_all));
        } else {
            mEditView.setText(AppResource.getString(mContext, R.string.fx_string_edit));
            mPullListView.setHeaderState(NestedHeaderBehavior.SHOW);
            mBottomView.setVisibility(View.GONE);
        }
        enableMovable(mReadingMarkRV, toEditState);
        updateUIState();
    }

    private void switchSearchState(boolean toSearchState) {
        if (mReadingBookmarkAdapter.isSearchState() == toSearchState) return;

        mReadingBookmarkAdapter.switchSearchState(toSearchState);
        if (toSearchState) {
            mPullListView.setHeaderState(NestedHeaderBehavior.FIXED);
            mUiExtensionsManager.getPanelManager().getPanelHost().setTabVisibility(false);
            mSearchView.onActionViewExpanded();
        } else {
            mPullListView.setHeaderState(NestedHeaderBehavior.SHOW);
            mUiExtensionsManager.getPanelManager().getPanelHost().setTabVisibility(true);
            mSearchView.onActionViewCollapsed();
        }
        updateUIState();
    }

    public boolean isMarked(int pageIndex) {
        return mReadingBookmarkAdapter.isMarked(pageIndex);
    }

    public void addMark(int pageIndex) {
        mReadingBookmarkAdapter.addBookmarkNode(0, pageIndex,
                AppResource.getString(mContext, R.string.fx_page_book_mark, pageIndex + 1));
        if (mSelectedAllTv == null) return;
        mSelectedAllTv.setText(mReadingBookmarkAdapter.isSelectedAll() ? AppResource.getString(mContext, R.string.fx_string_deselect_all) : AppResource.getString(mContext, R.string.fx_string_select_all));
    }

    public void removeMark(int index) {
        mReadingBookmarkAdapter.removeBookmarkNode(index);
        if (mSelectedAllTv == null) return;
        mSelectedAllTv.setText(mReadingBookmarkAdapter.isSelectedAll() ? AppResource.getString(mContext, R.string.fx_string_deselect_all) : AppResource.getString(mContext, R.string.fx_string_select_all));
    }

    private void showSheetMenu(final int position, View view) {
        mSheetMenu = UISheetMenu.newInstance((FragmentActivity) mUiExtensionsManager.getAttachedActivity());
        if (mIsPad)
            mSheetMenu.setWidth(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_pad_more_menu_width));
        List<Integer> items = new ArrayList<>();
        items.add(ISheetMenu.RENAME_MENU);
        items.add(ISheetMenu.DELETE_MENU);
        mSheetMenu.setSheetItems(items);
        mSheetMenu.setSheetItemClickListener(new UIBottomSheetMenu.OnSheetItemClickListener() {
            @Override
            public void onItemClick(int type) {
                dismissSheetMenu();
                if (type == ISheetMenu.RENAME_MENU) {
                    renameBookmark(position);
                } else if (type == ISheetMenu.DELETE_MENU) {
                    ReadingBookmarkAdapter.ReadingBookmarkNode node = mReadingBookmarkAdapter.getDataItem(position);
                    deleteBookmark(node.getIndex());
                }
            }
        });
        mSheetMenu.setOnSheetDismissListener(new ISheetMenu.OnSheetDismissListener() {
            @Override
            public void onDismiss(ISheetMenu sheetMenu) {
                mReadingBookmarkAdapter.resetSelectedPosition();
            }
        });
        showSheetMenu(view);
    }

    private void showSheetMenu(View view) {
        Rect rect = new Rect();
        view.getGlobalVisibleRect(rect);
        mSheetMenu.show(mUiExtensionsManager.getRootView(), rect);
    }

    private void dismissSheetMenu() {
        if (mSheetMenu != null && mSheetMenu.isShowing()) {
            mSheetMenu.dismiss();
        }
    }

    private void renameBookmark(final int position) {
        if (mUiExtensionsManager == null) return;

        Context context = mUiExtensionsManager.getAttachedActivity();
        if (context == null) return;
        mRenameDlg = new UITextEditDialog(context);
        mRenameDlg.getPromptTextView().setText(AppResource.getString(mContext, R.string.fx_rename_bookmark));
        mRenameDlg.setTitle(AppResource.getString(mContext, R.string.fx_renamebook_title));
        mRenameDlg.getDialog().setCanceledOnTouchOutside(false);
        mRenameDlg.setLengthFilters(INPUT_MAX_LENGTH, AppResource.getString(mContext, R.string.rv_panel_readingbookmark_tips_limited));

        final EditText renameDlgEt = mRenameDlg.getInputEditText();
        final Button renameDlgOk = mRenameDlg.getOKButton();
        Button renameDlgCancel = mRenameDlg.getCancelButton();

        final InputMethodManager inputManager = (InputMethodManager) renameDlgEt.getContext().getSystemService(Context.INPUT_METHOD_SERVICE);
        ReadingBookmarkAdapter.ReadingBookmarkNode node = mReadingBookmarkAdapter.getDataItem(position);
        renameDlgEt.setMaxLines(6);
        renameDlgEt.setText(node.getTitle());
        renameDlgEt.selectAll();
        renameDlgOk.setEnabled(false);
        renameDlgEt.addTextChangedListener(new TextWatcher() {

            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {
                renameDlgOk.setEnabled(renameDlgEt.getText().toString().trim().length() != 0);
            }

            @Override
            public void beforeTextChanged(CharSequence s, int start, int count, int after) {
            }

            @Override
            public void afterTextChanged(Editable s) {
                for (int i = s.length(); i > 0; i--) {
                    if (s.subSequence(i - 1, i).toString().equals("\n"))
                        s.replace(i - 1, i, "");
                }
            }
        });

        renameDlgEt.setOnKeyListener(new View.OnKeyListener() {

            @Override
            public boolean onKey(View v, int keyCode, KeyEvent event) {
                if (KeyEvent.KEYCODE_ENTER == keyCode && event.getAction() == KeyEvent.ACTION_DOWN) {
                    inputManager.hideSoftInputFromWindow(renameDlgEt.getWindowToken(), 0);
                    return true;
                }
                return false;
            }
        });

        renameDlgOk.setOnClickListener(new View.OnClickListener() {

            @Override
            public void onClick(View v) {
                String newContent = renameDlgEt.getText().toString().trim();
                mReadingBookmarkAdapter.updateBookmarkNode(position, newContent, AppDmUtil.currentDateToDocumentDate());
                mReadingBookmarkAdapter.notifyItemChanged(position);
                inputManager.hideSoftInputFromWindow(renameDlgEt.getWindowToken(), 0);
                mRenameDlg.dismiss();
                mUiExtensionsManager.getDocumentManager().setDocModified(true);
            }
        });

        renameDlgCancel.setOnClickListener(new View.OnClickListener() {

            @Override
            public void onClick(View v) {
                inputManager.hideSoftInputFromWindow(renameDlgEt.getWindowToken(), 0);
                mRenameDlg.dismiss();
            }
        });
        mRenameDlg.show();
        renameDlgEt.setFocusable(true);
        renameDlgEt.setFocusableInTouchMode(true);
        renameDlgEt.requestFocus();
        Timer timer = new Timer();
        timer.schedule(new TimerTask() {

            @Override
            public void run() {
                inputManager.showSoftInput(renameDlgEt, 0);
            }
        }, 300);
    }

    private void deleteBookmark(int pageIndex) {
        mReadingBookmarkAdapter.removeBookmarkNode(pageIndex);
        if (pageIndex == mPdfViewCtrl.getCurrentPage()) {
            changeMarkItemState(false);
        }
        changeViewState(mReadingBookmarkAdapter.getItemCount() != 0);
        mUiExtensionsManager.getDocumentManager().setDocModified(true);
    }

    private void updateBookmarkItemDisabledColor(IBaseItem item) {
        if (!item.isEnable() && item.isSelected() && item.getImageDrawable() != null) {
            if(AppDisplay.isPad()) {
                item.getImageDrawable()
                        .setColorFilter(AppResource.getColor(mContext, R.color.ui_color_top_bar_icon)
                                , PorterDuff.Mode.SRC_ATOP);
            }else{
                item.getImageDrawable()
                        .setColorFilter(ThemeConfig.getInstance(mContext).getPrimaryColor()
                                , PorterDuff.Mode.SRC_ATOP);
            }
            item.getImageDrawable().setAlpha(127);
        } else if (item.getImageDrawable() != null) {
            item.getImageDrawable().setColorFilter(null);
            item.getImageDrawable().setAlpha(255);
        }
    }

    private void updateFullScreenBookmarkDisabledColor() {
        if (mFullScreenFavoriteItem == null) return;
        if (!mFullScreenFavoriteItem.isEnabled() && mFullScreenFavoriteItem.isSelected()) {
            mFullScreenFavoriteItem.setColorFilter(ThemeConfig.getInstance(mContext).getPrimaryColor(), PorterDuff.Mode.SRC_ATOP);
            mFullScreenFavoriteItem.setAlpha(127);
        } else if (mFullScreenFavoriteItem.getColorFilter() != null) {
            mFullScreenFavoriteItem.setColorFilter(null);
            mFullScreenFavoriteItem.setAlpha(255);
        }
    }

    private ItemTouchHelper mTouchHelper = null;
    private int mMoveFromPosition = -1;
    private int mMoveToPosition = -1;

    private void enableMovable(RecyclerView recyclerView, boolean enabled) {
        if (mReadingBookmarkAdapter == null) return;

        if (enabled) {
            if (mTouchHelper == null) {
                mTouchHelper = new ItemTouchHelper(new ItemTouchHelper.SimpleCallback(ItemTouchHelper.UP
                        | ItemTouchHelper.DOWN, 0) {

                    @Override
                    public boolean onMove(@NonNull RecyclerView recyclerView,
                                          @NonNull RecyclerView.ViewHolder viewHolder, @NonNull RecyclerView.ViewHolder target) {
                        int fromPosition = viewHolder.getAdapterPosition();
                        mMoveToPosition = target.getAdapterPosition();
                        if (mMoveFromPosition == -1)
                            mMoveFromPosition = fromPosition;

                        Collections.swap(mReadingBookmarkAdapter.getReadingBookmarkNodes(), fromPosition, mMoveToPosition);
                        mReadingBookmarkAdapter.notifyItemMoved(fromPosition, mMoveToPosition);
                        return true;
                    }

                    @Override
                    public void onSwiped(@NonNull RecyclerView.ViewHolder viewHolder, int direction) {
                    }

                    @Override
                    public void onSelectedChanged(@Nullable RecyclerView.ViewHolder viewHolder, int actionState) {
                        super.onSelectedChanged(viewHolder, actionState);
                        if (actionState == 0 && mMoveFromPosition != mMoveToPosition) {

                            mReadingBookmarkAdapter.moveBookmarkNode(mMoveToPosition);
                            mUiExtensionsManager.getDocumentManager().setDocModified(true);
                        }
                        mMoveFromPosition = -1;
                        mMoveToPosition = -1;
                    }
                });
            }
            mTouchHelper.attachToRecyclerView(recyclerView);
        } else if (mTouchHelper != null) {
            mTouchHelper.attachToRecyclerView(null);
        }

        if (mReadingBookmarkAdapter != null) {
            mReadingBookmarkAdapter.setItemTouchHelper(mTouchHelper);
        }
    }

}
