/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.panel.bookmark;


import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.ProgressDialog;
import android.content.Context;
import android.text.Spannable;
import android.text.SpannableStringBuilder;
import android.text.style.ForegroundColorSpan;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.widget.CheckBox;
import android.widget.ImageView;
import android.widget.TextView;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.Task;
import com.foxit.sdk.common.DateTime;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.ReadingBookmark;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.browser.adapter.SuperAdapter;
import com.foxit.uiextensions.browser.adapter.viewholder.SuperViewHolder;
import com.foxit.uiextensions.modules.panel.bean.BaseBean;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.ItemTouchHelper;

public class ReadingBookmarkAdapter extends SuperAdapter<ReadingBookmarkAdapter.ReadingBookmarkNode> {
    private final List<ReadingBookmarkNode> mNodeList;
    private final List<ReadingBookmarkNode> mSearchNodeList;
    private final List<ReadingBookmarkNode> mEditNodeList;

    private final PDFViewCtrl mPDFViewCtrl;
    private final OnBookmarkItemClickListener mReadingbookMarkListener;

    private String mSearchPattern;
    private int mSelectedPosition = -1;
    private boolean mIsEditState = false;
    private boolean mIsSearchState = false;
    private final boolean mIsPad;
    private final Context mContext;

    public ReadingBookmarkAdapter(Context context, PDFViewCtrl viewCtrl, OnBookmarkItemClickListener listener) {
        super(context);
        mContext = context;
        mNodeList = new ArrayList<>();
        mSearchNodeList = new ArrayList<>();
        mEditNodeList = new ArrayList<>();
        mPDFViewCtrl = viewCtrl;
        mReadingbookMarkListener = listener;
        mIsPad = AppDisplay.isPad();
    }

    @Override
    public void notifyUpdateData() {
        notifyDataSetChanged();
    }

    @Override
    public ReadingBookmarkNode getDataItem(int position) {
        if (isSearchState())
            return mSearchNodeList.get(position);
        else
            return mNodeList.get(position);
    }

    @NonNull
    @Override
    public ReadingBookmarkViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        return new ReadingBookmarkViewHolder(LayoutInflater.from(getContext()).inflate(R.layout.rd_readingmark_item, parent, false));
    }

    @Override
    public int getItemCount() {
        if (isSearchState())
            return mSearchNodeList.size();
        else
            return mNodeList.size();
    }

    void switchSearchState(boolean toSearthState) {
        mIsSearchState = toSearthState;
    }

    boolean isSearchState() {
        return mIsSearchState;
    }

    void switchEditState(boolean toEditState) {
        mIsEditState = toEditState;
    }

    boolean isEditState() {
        return mIsEditState;
    }

    List<ReadingBookmarkNode> getSearchNodes() {
        return mSearchNodeList;
    }

    List<ReadingBookmarkNode> getEditNodes() {
        return mEditNodeList;
    }

    List<ReadingBookmarkNode> getReadingBookmarkNodes() {
        return mNodeList;
    }

    void setSearchPattern(String Pattern) {
        mSearchPattern = Pattern;
    }

    void addBookmarkNode(int index, int pageIndex, String title) {
        try {
            if (mPDFViewCtrl.getDoc() == null) return;

            ReadingBookmark readingBookmark = mPDFViewCtrl.getDoc().insertReadingBookmark(index, title, pageIndex);
            DateTime dateTime = AppDmUtil.currentDateToDocumentDate();
            readingBookmark.setDateTime(dateTime, false);
            mNodeList.add(index, new ReadingBookmarkNode(readingBookmark));
            notifyItemInserted(0);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    void removeBookmarkNode(int pageIndex) {
        if (mPDFViewCtrl.getDoc() == null) return;

        for (int position = 0; position < mNodeList.size(); position++) {
            if (mNodeList.get(position).getIndex() == pageIndex) {
                mEditNodeList.remove(mNodeList.get(position));
                mNodeList.remove(position);
                notifyItemRemoved(position);
                break;
            }
        }
        try {
            int nCount = mPDFViewCtrl.getDoc().getReadingBookmarkCount();
            for (int i = 0; i < nCount; i++) {
                ReadingBookmark readingMark = mPDFViewCtrl.getDoc().getReadingBookmark(i);
                if (readingMark.getPageIndex() == pageIndex) {
                    mPDFViewCtrl.getDoc().removeReadingBookmark(readingMark);
                    break;
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    void moveBookmarkNode( int toPosition) {
        PDFDoc doc = mPDFViewCtrl.getDoc();
        if (doc == null) return;

        try {
            ReadingBookmarkNode bookmarkNode = mNodeList.get(toPosition);
            if (bookmarkNode != null) {
                boolean ret = doc.removeReadingBookmark(bookmarkNode.mBookrmak);

                if (ret) {
                    ReadingBookmark readingBookmark = doc.insertReadingBookmark(toPosition, bookmarkNode.mTitle, bookmarkNode.mIndex);
                    DateTime dateTime = AppDmUtil.currentDateToDocumentDate();
                    readingBookmark.setDateTime(dateTime, false);
                    bookmarkNode.mBookrmak = readingBookmark;
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    boolean isSelectedAll() {
        return mEditNodeList.size() > 0 && mNodeList.size() == mEditNodeList.size();
    }

    void selectAll(boolean select) {
        mEditNodeList.clear();
        if (select)
            mEditNodeList.addAll(mNodeList);
        notifyDataSetChanged();
    }

    void clearAllNodes(final Event.Callback callback) {
        if (mPDFViewCtrl.getUIExtensionsManager() == null) return;
        final Context context = ((UIExtensionsManager) mPDFViewCtrl.getUIExtensionsManager()).getAttachedActivity();
        if (context == null) return;
        final ProgressDialog progressDialog = new ProgressDialog(context);
        progressDialog.setMessage(context.getApplicationContext().getString(R.string.rv_panel_annot_deleting));
        progressDialog.setCancelable(false);
        progressDialog.show();
        RemoveReadingBookmarkTask bookmarkTask = new RemoveReadingBookmarkTask(new Event.Callback() {
            @Override
            public void result(Event event, boolean success) {
                progressDialog.dismiss();
                notifyDataSetChanged();
                if (callback != null)
                    callback.result(null, success);
            }
        });
        mPDFViewCtrl.addTask(bookmarkTask);
    }

    void initBookmarkList() {
        try {
            if (mPDFViewCtrl.getDoc() == null) return;

            mNodeList.clear();
            int nCount = mPDFViewCtrl.getDoc().getReadingBookmarkCount();
            for (int i = 0; i < nCount; i++) {
                ReadingBookmark readingBookmark = mPDFViewCtrl.getDoc().getReadingBookmark(i);
                if (readingBookmark == null)
                    continue;
                ReadingBookmarkNode node = new ReadingBookmarkNode(readingBookmark);
                mNodeList.add(node);
            }
            notifyUpdateData();
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    void release() {
        mNodeList.clear();
    }

    void onPageRemoved(boolean success, int index) {
        if (!success || mPDFViewCtrl.getDoc() == null) return;

        ArrayList<ReadingBookmarkNode> invalidList = new ArrayList<>();
        for (ReadingBookmarkNode node : mNodeList) {
            if (node.getIndex() == index) {
                invalidList.add(node);
            } else if (node.getIndex() > index) {
                node.setIndex(node.getIndex() - 1);
            }
        }
        for (ReadingBookmarkNode node : invalidList) {
            mNodeList.remove(node);
            try {
                mPDFViewCtrl.getDoc().removeReadingBookmark(node.mBookrmak);
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
        invalidList.clear();
    }

    void onPageMoved(boolean success, int index, int dstIndex) {
        if (!success || mPDFViewCtrl.getDoc() == null) return;

        for (int i = 0; i < mNodeList.size(); i++) {
            ReadingBookmarkNode node = mNodeList.get(i);
            if (index < dstIndex) {
                if (node.getIndex() <= dstIndex && node.getIndex() > index) {
                    node.setIndex(node.getIndex() - 1);
                } else if (node.getIndex() == index) {
                    node.setIndex(dstIndex);
                }

            } else {
                if (node.getIndex() >= dstIndex && node.getIndex() < index) {
                    node.setIndex(node.getIndex() + 1);
                } else if (node.getIndex() == index) {
                    node.setIndex(dstIndex);
                }
            }
        }
    }

    void onPagesInsert(boolean success, int dstIndex, int[] range) {
        if (!success)
            return;
        int offsetIndex = 0;
        for (int i = 0; i < range.length / 2; i++) {
            offsetIndex += range[2 * i + 1];
        }
        updateReadingBookmarkItems(dstIndex, offsetIndex);
    }

    private void updateReadingBookmarkItems(int dstIndex, int offsetIndex) {
        for (int i = 0; i < mNodeList.size(); i++) {
            ReadingBookmarkNode node = mNodeList.get(i);
            if (node.getIndex() >= dstIndex) {
                node.setIndex(node.getIndex() + offsetIndex);
            }
        }
    }

    void updateBookmarkNode(int position, String title, DateTime dateTime) {
        mNodeList.get(position).setTitle(title);
        mNodeList.get(position).setModifiedDateTime(dateTime);
    }

    boolean isMarked(int index) {
        for (ReadingBookmarkNode node : mNodeList) {
            if (node.getIndex() == index)
                return true;
        }
        return false;
    }

    class RemoveReadingBookmarkTask extends Task {
        boolean ret;

        public RemoveReadingBookmarkTask(final Event.Callback callback) {
            super(new CallBack() {
                @Override
                public void result(Task task) {
                    callback.result(null, ((RemoveReadingBookmarkTask) task).ret);
                }
            });
        }

        @Override
        protected void execute() {
            if (mPDFViewCtrl.getDoc() == null) return;

            try {
                for (int i = 0; i < mNodeList.size(); i++) {
                    ReadingBookmark readingBookmark = mNodeList.get(i).mBookrmak;
                    if (readingBookmark != null && !readingBookmark.isEmpty())
                        mPDFViewCtrl.getDoc().removeReadingBookmark(readingBookmark);
                }
                mNodeList.clear();
                ret = true;
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }

        private void updateLayout() {
            if (mPDFViewCtrl.getUIExtensionsManager() == null) return;
            Activity activity = ((UIExtensionsManager) mPDFViewCtrl.getUIExtensionsManager()).getAttachedActivity();
            if (activity == null) return;
            activity.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    notifyDataSetChanged();
                }
            });
        }
    }

    class ReadingBookmarkViewHolder extends SuperViewHolder {
        private CheckBox mCheckBox;
        private TextView mTvContent;
        private ImageView mIvMore;
        private View mItemView;

        public ReadingBookmarkViewHolder(View itemView) {
            super(itemView);
            mItemView = itemView.findViewById(R.id.panel_readingmark_item_conotainer);
            mCheckBox = itemView.findViewById(R.id.rd_bookmark_item_checkbox);
            mTvContent = itemView.findViewById(R.id.rd_bookmark_item_content);
            mIvMore = itemView.findViewById(R.id.rd_panel_item_more);

            mIvMore.setOnClickListener(this);
            mItemView.setOnClickListener(this);
        }

        @Override
        public void bind(BaseBean data, int position) {
            ReadingBookmarkNode node = (ReadingBookmarkNode) data;
            if (isSearchState()) {
                mIvMore.setVisibility(View.GONE);
                mCheckBox.setVisibility(View.GONE);

                Object text = mTvContent.getText();
                SpannableStringBuilder spannable;

                if (text instanceof SpannableStringBuilder) {
                    spannable = (SpannableStringBuilder) text;
                    spannable.clearSpans();
                    spannable.clear();
                    spannable.append(node.getTitle());
                } else {
                    spannable = new SpannableStringBuilder(node.getTitle());
                }

                if (!AppUtil.isEmpty(mSearchPattern)) {
                    int start = node.getTitle().toLowerCase().indexOf(mSearchPattern, -1);
                    if (start != -1) {
                        spannable.setSpan(new ForegroundColorSpan(ThemeConfig.getInstance(getContext()).getPrimaryColor()),
                                start, start + mSearchPattern.length(),
                                Spannable.SPAN_EXCLUSIVE_INCLUSIVE
                        );
                    }
                }
                mTvContent.setText(spannable);
            } else if (isEditState()) {
                mIvMore.setVisibility(mNodeList.size() == 1 ? View.GONE : View.VISIBLE);
                mIvMore.setImageResource(R.drawable.ic_list_item_drag);
                mCheckBox.setVisibility(View.VISIBLE);
                mCheckBox.setChecked(mEditNodeList.contains(node));
                mTvContent.setText(node.getTitle());
                ThemeUtil.setTintList(mCheckBox, ThemeUtil.getCheckboxColor(getContext()));
            } else {
                mCheckBox.setVisibility(View.GONE);
                boolean canAssemble = ((UIExtensionsManager) mPDFViewCtrl.getUIExtensionsManager())
                        .getDocumentManager().canAssemble() && ((UIExtensionsManager) mPDFViewCtrl.getUIExtensionsManager()).isEnableModification();
                mIvMore.setEnabled(canAssemble);
                mIvMore.setVisibility(View.VISIBLE);
                mIvMore.setImageResource(R.drawable.rd_item_more);
                mTvContent.setText(node.getTitle());
            }
            ThemeUtil.setTintList(mIvMore, ThemeUtil.getEnableIconColor(mContext));
            mItemView.setSelected(!mIsPad && mSelectedPosition == position);

            mItemView.setBackground(AppResource.getDrawable(mContext, R.drawable.rd_list_item_bg));
            mTvContent.setTextColor(AppResource.getColor(mContext, R.color.t4));

            bindDragIv();
        }

        @SuppressLint("ClickableViewAccessibility")
        private void bindDragIv() {
            if (isEditState()) {
                mIvMore.setOnTouchListener(new View.OnTouchListener() {
                    @Override
                    public boolean onTouch(View v, MotionEvent event) {
                        int actionMask = event.getActionMasked();
                        if (actionMask == MotionEvent.ACTION_DOWN) {
                            if (mItemTouchHelper != null)
                                mItemTouchHelper.startDrag(ReadingBookmarkViewHolder.this);
                        }
                        return true;
                    }
                });
            } else {
                mIvMore.setOnTouchListener(null);
            }
        }

        @Override
        public void onClick(View v) {
            int id = v.getId();
            int position = getAdapterPosition();
            if (id == R.id.rd_panel_item_more) {
                if (AppUtil.isFastDoubleClick()) return;
                if (mReadingbookMarkListener != null) {
                    mReadingbookMarkListener.onMoreClick(position, v);
                }
                mSelectedPosition = position;
                if (mSelectedPosition >= 0)
                    notifyItemChanged(mSelectedPosition);
            } else if (id == R.id.panel_readingmark_item_conotainer) {
                if (mReadingbookMarkListener != null) {
                    ReadingBookmarkNode readingBookmarkNode;
                    if (isSearchState())
                        readingBookmarkNode = mSearchNodeList.get(position);
                    else
                        readingBookmarkNode = mNodeList.get(position);
                    mReadingbookMarkListener.onItemClick(position, readingBookmarkNode);
                }
            }
        }
    }

    int getSelectedPosition() {
        return mSelectedPosition;
    }

    void resetSelectedPosition() {
        if (mSelectedPosition != -1) {
            int tempPosition = mSelectedPosition;
            mSelectedPosition = -1;
            if (tempPosition >= 0)
                notifyItemChanged(tempPosition);
        }
    }

    private ItemTouchHelper mItemTouchHelper;

    public void setItemTouchHelper(ItemTouchHelper itemTouchHelper) {
        mItemTouchHelper = itemTouchHelper;
    }

    static class ReadingBookmarkNode extends BaseBean {
        private ReadingBookmark mBookrmak;
        private String mTitle;
        private int mIndex;
        private DateTime mModifiedDateTime;

        public ReadingBookmarkNode(ReadingBookmark bookmark) {
            mBookrmak = bookmark;
            try {
                mTitle = mBookrmak.getTitle();
                mIndex = mBookrmak.getPageIndex();
                mModifiedDateTime = mBookrmak.getDateTime(false) == null ? mBookrmak.getDateTime(true) : mBookrmak.getDateTime(false);
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }

        public void setTitle(String title) {
            mTitle = title;
            try {
                mBookrmak.setTitle(title);
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }

        public void setModifiedDateTime(DateTime dateTime) {
            mModifiedDateTime = dateTime;
            try {
                mBookrmak.setDateTime(mModifiedDateTime, false);
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }

        public DateTime getModifiedDateTime() {
            return mModifiedDateTime;
        }

        public String getTitle() {
            return mTitle;
        }

        public int getIndex() {
            return mIndex;
        }

        public void setIndex(int index) {
            mIndex = index;
            try {
                mBookrmak.setPageIndex(index);
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }

        public ReadingBookmarkNode cloneNode() {
            return new ReadingBookmarkNode(this.mBookrmak);
        }
    }

    interface OnBookmarkItemClickListener {
        void onMoreClick(int position, View view);

        void onItemClick(int position, ReadingBookmarkNode node);
    }

}
