/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.panel.annot;


import android.content.Context;
import android.content.res.Configuration;
import android.view.View;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.controls.dialog.UIMatchDialog;
import com.foxit.uiextensions.controls.dialog.sheetmenu.ISheetMenu;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.DefaultItemAnimator;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

class GroupListDialog extends UIMatchDialog {

    private PDFViewCtrl mPDFViewCtrl;
    private AnnotAdapter mGroupListAdapter;
    private LinearLayoutManager mLayoutManager;

    private List<AnnotNode> mNodeList;
    private List<AnnotNode> mGroupList;

    GroupListDialog(Context context, PDFViewCtrl pdfViewCtrl) {
        super(context);
        mPDFViewCtrl = pdfViewCtrl;
        mNodeList = new ArrayList<>();
        mGroupList = new ArrayList<>();

        initView();
        initStyle();
    }

    private void initView() {
        View view = View.inflate(mContext, R.layout.rd_recyclerview_layout, null);
        RecyclerView recyclerView = view.findViewById(R.id.rd_recyclerview_list);
        mLayoutManager = new LinearLayoutManager(mContext, LinearLayoutManager.VERTICAL, false);
        recyclerView.setLayoutManager(mLayoutManager);
        recyclerView.setItemAnimator(new DefaultItemAnimator());
        mGroupListAdapter = new AnnotAdapter(mContext, mPDFViewCtrl);
        recyclerView.setAdapter(mGroupListAdapter);
        if (AppDisplay.isPad()){
            recyclerView.addOnScrollListener(new RecyclerView.OnScrollListener() {
                @Override
                public void onScrolled(@NonNull RecyclerView recyclerView, int dx, int dy) {
                    mGroupListAdapter.onScrolled(mLayoutManager);
                }
            });
        }
        mGroupListAdapter.setSheetMenuClickListener(new AnnotAdapter.OnSheetMenuClickListener() {
            @Override
            public void onSheetMenuClick(boolean success, int type, AnnotNode node) {
                if (success) {
                    if (type == ISheetMenu.FLATTEN_MENU) {
                        dismiss();
                    } else if (type == ISheetMenu.DELETE_MENU) {
                        if (AppUtil.isEmpty(node.getGroupHeaderNM())) {
                            establishNodeList(mGroupList);//delete reply node
                            mGroupListAdapter.notifyUpdateData();
                        } else {
                            dismiss();// delete root node
                        }
                    } else if (type == ISheetMenu.REPLY_MENU) {
                        establishNodeList(mGroupList);
                        mGroupListAdapter.notifyUpdateData();
                    }
                }
            }
        });
        setContentView(view);

        final UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPDFViewCtrl.getUIExtensionsManager();
        uiExtensionsManager.registerConfigurationChangedListener(mConfigurationChangedListener);
        setOnDLDismissListener(new DismissListener() {
            @Override
            public void onDismiss() {
                uiExtensionsManager.unregisterConfigurationChangedListener(mConfigurationChangedListener);
            }
        });
    }

    private void initStyle() {
        setTitle(AppResource.getString(mContext, R.string.rd_sheet_menu_group_list));
        setBackButtonText(AppResource.getString(mContext, R.string.fx_string_close));
        setBackButtonVisible(View.VISIBLE);
        setBackButtonStyle(TEXT_BACK);
        setTitlePosition(BaseBar.TB_Position.Position_CENTER);
        setStyle(DLG_TITLE_STYLE_BG_DEFAULT);
    }

    void initData(List<List<AnnotNode>> pageNodes, List<AnnotNode> groupList) {
        mNodeList.clear();
        mGroupList = groupList;
        if (groupList.size() > 0) {
            establishNodeList(groupList);

            mGroupListAdapter.setPageNodes(pageNodes);
            mGroupListAdapter.setNodeList(mNodeList);
            mGroupListAdapter.setGroupDialogView(getRootView());
            mGroupListAdapter.setState(AnnotAdapter.GROUP_LIST_STATE);
            AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
                @Override
                public void run() {
                    mGroupListAdapter.notifyUpdateData();
                }
            });
        }
    }

    private void establishNodeList(List<AnnotNode> groupList) {
        mNodeList.clear();
        int annotCount = 0;
        for (AnnotNode an : groupList) {
            if (an.isRootNode() && !an.isRedundant()) {
                an.setFlag(AnnotNode.FLAG_CONTENT);
                mNodeList.add(an);
                annotCount++;
                establishNodeRoot(an);
            }
        }
        AnnotNode pageNode = new AnnotNode(groupList.get(0).getPageIndex());
        pageNode.counter = annotCount;
        pageNode.setFlag(AnnotNode.FLAG_LABLE);
        mNodeList.add(0, pageNode);
    }

    private void establishNodeRoot(AnnotNode parent) {
        if (parent.isLeafNode() || !parent.isExpanded()) return;

        for (AnnotNode child : parent.getChildren()) {
            child.setFlag(AnnotNode.FLAG_CONTENT);
            mNodeList.add(child);
            establishNodeRoot(child);
        }
    }

    private UIExtensionsManager.ConfigurationChangedListener mConfigurationChangedListener = new UIExtensionsManager.ConfigurationChangedListener() {
        @Override
        public void onConfigurationChanged(Configuration newConfig) {
            mGroupListAdapter.onConfigurationChanged(mLayoutManager);
        }
    };

}
