/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.panel.annot;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.view.View;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.Task;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.Markup;
import com.foxit.sdk.pdf.annots.MarkupArray;
import com.foxit.sdk.pdf.annots.Note;
import com.foxit.uiextensions.DocumentManager;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotEventListener;
import com.foxit.uiextensions.annots.IFlattenEventListener;
import com.foxit.uiextensions.annots.IGroupEventListener;
import com.foxit.uiextensions.annots.IImportAnnotsEventListener;
import com.foxit.uiextensions.annots.IRedactionEventListener;
import com.foxit.uiextensions.annots.multiselect.GroupManager;
import com.foxit.uiextensions.controls.dialog.FxProgressDialog;
import com.foxit.uiextensions.controls.dialog.UIDialog;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.utils.AnnotPermissionUtil;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.UIToast;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

class AnnotPanel implements View.OnClickListener {

    public static final int STATUS_LOADING = 1;
    public static final int STATUS_CANCEL = 2;
    public static final int STATUS_PAUSED = 3;
    public static final int STATUS_DONE = 4;
    public static final int STATUS_FAILED = 5;
    public static final int STATUS_DELETING = 6;

    private static final int DELETE_CAN = 0;
    private static final int DELETE_SRCAN = 1;
    private static final int DELETE_UNCAN = 2;

    private final PDFViewCtrl mPdfViewCtrl;
    private final Context mContext;
    private final AnnotAdapter mAdapter;
    private final List<AnnotNode> mDeleteTemps;

    private int mLoadedState;
    private int mLoadedIndex;

    private AnnotPanelModule mPanel;
    private FxProgressDialog mDeleteDialog;
    private UITextEditDialog mSRDeleteDialog;
    private Handler mHandle;

    AnnotPanel(AnnotPanelModule panelModule, Context context, PDFViewCtrl pdfViewCtrl) {
        mPanel = panelModule;
        mPdfViewCtrl = pdfViewCtrl;
        mContext = context;
        mDeleteTemps = new ArrayList<AnnotNode>();
        mAdapter = new AnnotAdapter(context, mPdfViewCtrl);
        mHandle = new Handler(Looper.getMainLooper()) {
            @Override
            public void handleMessage(Message msg) {
                switch (msg.what) {
                    case 1:
                        if (mAdapter != null) {
//                            mAdapter.updateAnnotationListFromCheck();
                        }
                        break;
                    case 2://onAnnotAdded()
                        mAdapter.notifyDataSetChanged();
                        mPanel.updateUIState();
                        break;
                    case 3://onAnnotModified()
                        Annot annot = (Annot) msg.obj;
                        mAdapter.updateNode(annot);
                        mAdapter.notifyDataSetChanged();
                        break;
                    case 4://onAnnotDeleted()
                        mAdapter.notifyDataSetChanged();
                        if (mLoadedState == STATUS_DONE && mAdapter.getItemCount() == 0) {
                            mPanel.updateUIState();
                        }
                        break;
                    default:
                        break;
                }
            }
        };
    }

    private void prepareDeleteNodeList(List<AnnotNode> selectedNodes) {
        mDeleteTemps.clear();
        for (AnnotNode node : selectedNodes) {
            if (node.getParent() != null && selectedNodes.contains(node.getParent())) continue;
            if (node.canDelete()) {
                mDeleteTemps.add(node);
            }
        }
    }

    void clearNodes() {
        if (mPdfViewCtrl.getUIExtensionsManager() == null) return;
        final Context context = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAttachedActivity();
        if (context == null) return;
        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setCurrentAnnot(null);

        if (mDeleteDialog == null)
            mDeleteDialog = new FxProgressDialog(context, context.getApplicationContext().getString(R.string.rv_panel_annot_deleting));
        resetDeleteDialog();

        Collections.sort(mAdapter.getEditItems());
        int checkStatus = checkDeleteStatus(mAdapter.getEditItems());
        if (checkStatus == DELETE_SRCAN) {
            if (mSRDeleteDialog == null || mSRDeleteDialog.getDialog().getOwnerActivity() == null) {
                mSRDeleteDialog = new UITextEditDialog(context, UIDialog.NO_INPUT);
                mSRDeleteDialog.getPromptTextView().setText(context.getApplicationContext().getString(R.string.rv_panel_annot_delete_tips));
                mSRDeleteDialog.setTitle(context.getApplicationContext().getString(R.string.cloud_delete_tv));
            }
            mSRDeleteDialog.getOKButton().setOnClickListener(new View.OnClickListener() {

                @Override
                public void onClick(View v) {
                    mSRDeleteDialog.dismiss();
                    mDeleteDialog.show();
                    prepareDeleteNodeList(mAdapter.getEditItems());
                    deleteItems();
                    mAdapter.clearSelectedNodes();
                }
            });
            mSRDeleteDialog.show();
        } else if (checkStatus == DELETE_UNCAN) {
            UIToast.getInstance(mContext).show(context.getApplicationContext().getString(R.string.rv_panel_annot_failed));
        } else {
            mDeleteDialog.show();
            prepareDeleteNodeList(mAdapter.getEditItems());
            deleteItems();
            mAdapter.clearSelectedNodes();
        }
        mPanel.updateUIState();
    }

    private boolean isPageLoaded(PDFPage page) {
        if (page == null) return false;
        try {
            int pageIndex = page.getIndex();
            return pageIndex < mLoadedIndex || mLoadedState == STATUS_DONE;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }


    private boolean mPause;

    public void setStatusPause(boolean status) {
        mPause = status;
    }

    public AnnotEventListener getAnnotEventListener() {
        return mAnnotEventListener;
    }

    private AnnotEventListener mAnnotEventListener = new AnnotEventListener() {
        @Override
        public void onAnnotAdded(PDFPage page, Annot annot) {
            if (mPanel.getTopToolbar() == null) return;
            //After adding an annotation, add a node that corresponds to the annotation in the Annotation list.
            if (page == null || annot == null || AppAnnotUtil.isSupportGroupElement(annot)) return;
            try {
                addNode(page, annot);
                if (annot.isMarkup())
                    addReplyNodes(page, (Markup) annot);
            } catch (PDFException e) {
                e.printStackTrace();
                return;
            }
            mAdapter.establishNodeList();
            mHandle.sendEmptyMessage(2);
        }

        @Override
        public void onAnnotWillDelete(PDFPage page, Annot annot) {
            if (mPanel.getTopToolbar() == null) return;
            removeNode(page, annot);
        }

        @Override
        public void onAnnotDeleted(PDFPage page, Annot annot) {
            if (mPanel.getTopToolbar() == null) return;
            //Refresh the page after the delete
            mAdapter.establishNodeList();
            mHandle.sendEmptyMessage(4);
        }

        @Override
        public void onAnnotModified(PDFPage page, Annot annot) {
            if (mPanel.getTopToolbar() == null) return;
            //After modifying an annotation, modify information of the node that corresponds to the annotation in the Annotation list.
            if (page == null || annot == null || AppAnnotUtil.isSupportGroupElement(annot)) return;
            try {
                if (!AppAnnotUtil.isSupportEditAnnot(annot) || (annot.getFlags() & Annot.e_FlagHidden) != 0 || !isPageLoaded(page))
                    return;

                Message message = Message.obtain();
                message.what = 3;
                message.obj = annot;
                mHandle.sendMessage(message);
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }

        @Override
        public void onAnnotChanged(Annot lastAnnot, Annot currentAnnot) {

        }
    };

    private void addReplyNodes(PDFPage page, Markup annot) {
        try {
            int replyCount = annot.getReplyCount();
            for (int i = 0; i < replyCount; i++) {
                Note note = annot.getReply(i);
                addNode(page, note);
                addReplyNodes(page, note);
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private void addNode(PDFPage page, Annot annot) {
        try {
            if (!AppAnnotUtil.isSupportEditAnnot(annot)
                    || (annot.getFlags() & Annot.e_FlagHidden) != 0
                    || !isPageLoaded(page))
                return;
            AnnotNode node = mAdapter.getAnnotNode(page, AppAnnotUtil.getAnnotUniqueID(annot));
            if (node == null) {
                Annot replyToAnnot = AppAnnotUtil.getReplyToAnnot(annot);
                node = new AnnotNode(page.getIndex(),
                        AppAnnotUtil.getAnnotObjNum(annot),
                        AppAnnotUtil.getAnnotUniqueID(annot),
                        AppAnnotUtil.getAnnotObjNum(replyToAnnot),
                        AppAnnotUtil.getAnnotUniqueID(replyToAnnot));
                if (annot.isMarkup())
                    node.setAuthor(((Markup) annot).getTitle());
                String typeString = AppAnnotUtil.getTypeString(annot);
                node.setType(typeString);
                node.setContent(annot.getContent());
                String modifiedDate = AppDmUtil.formatDocumentDate(AppDmUtil.FORMAT_MMM_DD_YYYY_HH_MM, annot.getModifiedDateTime());
                String createdDate = AppDmUtil.DEFAULT_MMM_DD_YYYY_HH_MM;
                if (annot.isMarkup())
                    createdDate = AppDmUtil.formatDocumentDate(AppDmUtil.FORMAT_MMM_DD_YYYY_HH_MM, ((Markup) annot).getCreationDateTime());
                if (modifiedDate == null || modifiedDate.equals(AppDmUtil.DEFAULT_MMM_DD_YYYY_HH_MM)) {
                    modifiedDate = createdDate;
                }
                node.setModifiedDate(modifiedDate);
                node.setCreationDate(createdDate);
                node.setApplyRedaction(!annot.isEmpty() && annot.getType() == Annot.e_Redact);

                boolean isLocked = AppAnnotUtil.isLocked(annot);
                boolean isReadOnly = AppAnnotUtil.isReadOnly(annot);
                node.setLocked(isLocked);
                node.setReadOnlyFlag(isReadOnly);
                node.setEditable(!isReadOnly);
                if (GroupManager.getInstance().isGrouped(mPdfViewCtrl, annot)) {
                    node.setDeletable(GroupManager.getInstance().canDelete(mPdfViewCtrl, annot));
                    node.setCanReply(GroupManager.getInstance().canReply(mPdfViewCtrl, annot));
                    node.setCanModifyContents(GroupManager.getInstance().contentsModifiable(mPdfViewCtrl, annot));
                    node.setCanComment(false);
                } else {
                    node.setDeletable(AppAnnotUtil.isSupportDeleteAnnot(annot) && !(isLocked || isReadOnly));
                    node.setCanReply(AppAnnotUtil.isAnnotSupportReply(annot) && !isReadOnly);
                    node.setCanModifyContents(AppAnnotUtil.contentsModifiable(typeString));
                    node.setCanComment(!isLocked && !isReadOnly);
                }
                DocumentManager documentManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager();
                node.setWithModificationPermission(AnnotPermissionUtil.canModifyAnnot(documentManager, annot));
                node.setWithDeletePermission(AnnotPermissionUtil.canDeleteAnnot(documentManager, annot));
                node.setWithReplyPermission(AnnotPermissionUtil.canReplyAnnot(documentManager, annot));
                if (annot.isMarkup())
                    node.setIntent(((Markup) annot).getIntent());
                mAdapter.addNode(node);
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private IFlattenEventListener mFlattenEventListener = new IFlattenEventListener() {
        @Override
        public void onAnnotWillFlatten(PDFPage page, Annot annot) {
            removeNode(page, annot);
        }

        @Override
        public void onAnnotFlattened(PDFPage page, Annot annot) {
            if (mPanel.getTopToolbar() == null) return;
            mAdapter.establishNodeList();
            mHandle.sendEmptyMessage(4);
        }

        @Override
        public void onAnnotsFlattened(PDFDoc doc) {
            if (mPanel.getTopToolbar() == null) return;
            prepare();
            startSearch(0);
        }
    };

    private IGroupEventListener mGroupEventListener = new IGroupEventListener() {

        @Override
        public void onAnnotGrouped(PDFPage page, List<Annot> groupAnnots) {
            if (mPanel.getTopToolbar() == null) return;
            updateAnnotNode(page, groupAnnots);
        }

        @Override
        public void onAnnotUnGrouped(PDFPage page, List<Annot> unGroupAnnots) {
            if (mPanel.getTopToolbar() == null) return;
            updateAnnotNode(page, unGroupAnnots);
        }

        private void updateAnnotNode(PDFPage page, List<Annot> groupAnnots) {
            for (int i = 0; i < groupAnnots.size(); i++) {
                Annot annot = groupAnnots.get(i);
                try {
                    if (!AppAnnotUtil.isSupportEditAnnot(annot) || (annot.getFlags() & Annot.e_FlagHidden) != 0 || !isPageLoaded(page))
                        return;
                    Message message = Message.obtain();
                    message.what = 3;
                    message.obj = annot;
                    mHandle.sendMessage(message);
                } catch (PDFException e) {
                    e.printStackTrace();
                }
            }
            mAdapter.resetCanEditAnnotCount();
        }
    };

    private IRedactionEventListener mRedactionEventListener = new IRedactionEventListener() {
        @Override
        public void onAnnotWillApply(PDFPage page, Annot annot) {
        }

        @Override
        public void onAnnotApplied(PDFPage page, Annot annot) {
            if (mPanel.getTopToolbar() == null) return;
            //reloaded the annot list
            prepare();
            startSearch(0);
        }

        @Override
        public void onAnnotsApplied(PDFDoc doc) {
            if (mPanel.getTopToolbar() == null) return;
            //reloaded the annot list
            prepare();
            startSearch(0);
        }

        @Override
        public void onAnnotsAdded(PDFDoc doc) {
            if (mPanel.getTopToolbar() == null) return;
            //reloaded the annot list
            prepare();
            startSearch(0);
        }

        @Override
        public void onAnnotsDeleted(PDFDoc doc) {
            if (mPanel.getTopToolbar() == null) return;
            //reloaded the annot list
            prepare();
            startSearch(0);
        }
    };

    IRedactionEventListener getRedactionEventListener() {
        return mRedactionEventListener;
    }

    IFlattenEventListener getFlattenEventListener() {
        return mFlattenEventListener;
    }

    IGroupEventListener getGroupEventListener() {
        return mGroupEventListener;
    }

    private IImportAnnotsEventListener mImportAnnotsListener = new IImportAnnotsEventListener() {
        @Override
        public void onAnnotsImported() {
            if (mPanel.getTopToolbar() == null) return;
            //reloaded the annot list
            prepare();
            startSearch(0);
        }
    };

    IImportAnnotsEventListener getImportAnnotsListener() {
        return mImportAnnotsListener;
    }

    private void removeNode(PDFPage page, Annot annot) {
        //After removing an annotation, remove the node that corresponds to the annotation in the Annotation list.
        if (page == null || annot == null || AppAnnotUtil.isSupportGroupElement(annot)) return;
        try {
            if (!AppAnnotUtil.isSupportEditAnnot(annot) || (annot.getFlags() & Annot.e_FlagHidden) != 0 || !isPageLoaded(page))
                return;

            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
            if (GroupManager.getInstance().isGrouped(mPdfViewCtrl, annot)) {
                MarkupArray markupArray = ((Markup) annot).getGroupElements();
                long markupSize = markupArray.getSize();
                if (markupSize > 0) {
                    ArrayList<Annot> groupAnnots = new ArrayList<>();
                    boolean canModifyContent = true;
                    boolean canReply = true;
                    boolean canDelete = true;

                    for (long i = 0; i < markupSize; i++) {
                        Annot groupAnnot = AppAnnotUtil.createAnnot(markupArray.getAt(i));
                        if (groupAnnot == null)
                            continue;
                        String annotNM = AppAnnotUtil.getAnnotUniqueID(groupAnnot);
                        if (!AppUtil.isEmpty(annotNM) && annotNM.equals(AppAnnotUtil.getAnnotUniqueID(annot)))
                            continue;

                        if (uiExtensionsManager.isLoadAnnotModule(groupAnnot)) {
                            if (canModifyContent)
                                canModifyContent = AppAnnotUtil.contentsModifiable(AppAnnotUtil.getTypeString(groupAnnot));
                            if (canReply)
                                canReply = AppAnnotUtil.isAnnotSupportReply(groupAnnot) && !AppAnnotUtil.isReadOnly(groupAnnot);
                            if (canDelete)
                                canDelete = AppAnnotUtil.isSupportDeleteAnnot(groupAnnot) && !(AppAnnotUtil.isLocked(groupAnnot) || AppAnnotUtil.isReadOnly(groupAnnot));

                            groupAnnots.add(groupAnnot);
                        }
                    }

                    String groupHeaderNM = AppAnnotUtil.getAnnotUniqueID(((Markup) annot).getGroupHeader());
                    if (!AppUtil.isEmpty(groupHeaderNM) && groupHeaderNM.equals(AppAnnotUtil.getAnnotUniqueID(annot))) {
                        if (groupAnnots.size() > 0)
                            groupHeaderNM = AppAnnotUtil.getAnnotUniqueID(groupAnnots.get(0));
                    }
                    if (AppUtil.isEmpty(groupHeaderNM)) {
                        groupHeaderNM = AppDmUtil.randomUUID(null);
                    }

                    for (int i = 0; i < groupAnnots.size(); i++) {
                        String annotNM = AppAnnotUtil.getAnnotUniqueID(groupAnnots.get(i));
                        AnnotNode node = mAdapter.getAnnotNode(page, annotNM);
                        if (node != null) {
                            node.setCanModifyContents(canModifyContent);
                            node.setCanReply(canReply);
                            node.setCanComment(false);
                            node.setDeletable(canDelete);
                            node.setGroupHeaderNM(groupHeaderNM);
                        }
                    }
                }
            }

            mAdapter.removeNode(page, AppAnnotUtil.getAnnotUniqueID(annot));
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void onClick(View v) {

    }

    private void resetDeleteDialog() {
        if (mDeleteDialog != null) {
            mDeleteDialog.dismiss();
        }
    }

    private void deleteItems() {
        final int size = mDeleteTemps.size();
        if (size == 0) {
            if (mLoadedState == STATUS_DELETING) {
                startSearch(mLoadedIndex);
            }
            resetDeleteDialog();
            mPanel.updateUIState();
            return;
        }
        if (mLoadedState != STATUS_DONE)
            mLoadedState = STATUS_DELETING;

        if (mPdfViewCtrl.getDoc() == null) return;
        if (((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getCurrentToolHandler() != null)
            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).setCurrentToolHandler(null);


        final AnnotNode node = mDeleteTemps.get(size - 1);
        if (node == null || node.isPageDivider()) {
            mDeleteTemps.remove(node);
            deleteItems();
            return;
        }
        if (!node.canDelete()) {
            node.setChecked(false);
            mDeleteTemps.remove(node);
            deleteItems();
            return;
        }
        mAdapter.removeNode(node, new AnnotAdapter.DeleteCallback() {

            @Override
            public void result(boolean success, AnnotNode n) {
                if (success) {
                    mDeleteTemps.remove(n);
                    deleteItems();
                } else {
                    resetDeleteDialog();
                }
            }
        });
    }

    private int checkDeleteStatus(List<AnnotNode> selectedNodes) {
        int status = DELETE_CAN;
        for (AnnotNode node : selectedNodes) {
            if (!node.canDelete()) {
                status = DELETE_UNCAN;
                AnnotNode parent = node.getParent();
                while (parent != null) {
                    if (parent.isChecked() && parent.canDelete()) {
                        status = DELETE_SRCAN;
                        break;
                    }
                    parent = parent.getParent();
                }
                if (status == DELETE_UNCAN) break;
            }
        }
        return status;
    }

    public int getCount() {
        return mAdapter.getItemCount();
    }

    public int getCurrentStatus() {
        return mLoadedState;
    }


    public AnnotAdapter getAdapter() {
        return mAdapter;
    }

    public void prepare() {
        mAdapter.preparePageNodes();
        AppThreadManager.getInstance().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                reset();
            }
        });
    }

    public void onDocWillClose() {
        reset();
        resetDeleteDialog();
        mSRDeleteDialog = null;
    }

    private void reset() {
        mLoadedState = STATUS_CANCEL;
        mPause = false;
        mLoadedIndex = 0;
        mPanel.updateUIState();
        mAdapter.clearNodes();
        mAdapter.notifyDataSetChanged();
    }

    void startSearch(final int pageIndex, final boolean continues) {
        mLoadedIndex = pageIndex;
        searchPage(pageIndex, new OnSearchPageEndListener() {
            @Override
            public void onResult(boolean success, ArrayList<AnnotNode> nodeList) {
                int pageCount = 0;
                try {
                    pageCount = mPdfViewCtrl.getPageCount();
                } catch (Exception e) {
                    mLoadedState = STATUS_FAILED;
                    mPanel.updateLoadedPage(pageIndex + 1, pageCount);
                    mPanel.updateUIState();
                    return;
                }
                //If search the current page  failed, then all subsequent pages will no longer continue to search.
                if (!success) {
                    mLoadedState = STATUS_FAILED;
                    mPanel.updateLoadedPage(pageIndex + 1, pageCount);
                    mPanel.updateUIState();
                    return;
                }
                if (mPause) {
                    mPanel.pauseSearch(pageIndex);
                    mLoadedState = STATUS_PAUSED;
                    return;
                }

                mPanel.updateLoadedPage(pageIndex + 1, pageCount);
                int nCount = nodeList.size();
                for (int i = 0; i < nCount; i++) {
                    AnnotNode annotNode = nodeList.get(i);
                    mAdapter.addNode(annotNode);
                }
                mAdapter.establishNodeList();
                if (continues && pageIndex < pageCount - 1) {
                    if (mLoadedState != STATUS_CANCEL) {
                        startSearch(pageIndex + 1);
                    }
                } else {
                    mLoadedState = STATUS_DONE;
                    mPanel.updateLoadedPage(0, 0);
                }
                mPanel.updateUIState();
            }
        });
    }

    void startSearch(int pageIndex) {
        startSearch(pageIndex, true);
    }

    public interface OnSearchPageEndListener {
        void onResult(boolean success, ArrayList<AnnotNode> list);
    }

    private void searchPage(int pageIndex, final OnSearchPageEndListener result) {
        if (mPdfViewCtrl.getDoc() == null) return;
        SearchPageTask searchPageTask = new SearchPageTask(mPdfViewCtrl, pageIndex, result);
        mPdfViewCtrl.addTask(searchPageTask);
    }

    class SearchPageTask extends Task {
        private int mPageIndex;
        private PDFViewCtrl mPdfView;
        private ArrayList<AnnotNode> mSearchResults;
        private boolean mSearchRet;

        public SearchPageTask(PDFViewCtrl pdfView, int pageIndex, final OnSearchPageEndListener onSearchPageEndListener) {
            super(new CallBack() {
                @Override
                public void result(Task task) {
                    SearchPageTask task1 = (SearchPageTask) task;
                    onSearchPageEndListener.onResult(task1.mSearchRet, task1.mSearchResults);
                }
            });
            mPdfView = pdfView;
            mPageIndex = pageIndex;
        }

        @Override
        protected void execute() {
            if (mSearchResults == null) {
                mSearchResults = new ArrayList<AnnotNode>();
            }
            mLoadedState = STATUS_LOADING;
            mSearchRet = searchPage();
        }

        private boolean searchPage() {
            try {
                if (mPdfView.getDoc() == null)
                    return false;
                PDFPage page = mPdfView.getDoc().getPage(mPageIndex);
                if (page == null || page.isEmpty()) return false;

                int nCount = page.getAnnotCount();
                if (nCount > 0) {
                    for (int i = 0; i < nCount; i++) {
                        Annot annot = AppAnnotUtil.createAnnot(page.getAnnot(i));
                        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
                        if (annot == null
                                || (annot.getFlags() & Annot.e_FlagHidden) != 0
                                || !AppAnnotUtil.isSupportEditAnnot(annot))
                            continue;

                        String typeString = AppAnnotUtil.getTypeString(annot);
                        if (!uiExtensionsManager.isLoadAnnotModule(typeString))
                            continue;

                        Annot replyToAnnot = AppAnnotUtil.getReplyToAnnot(annot);
                        //must getPage again.
                        AnnotNode node = new AnnotNode(mPageIndex,
                                AppAnnotUtil.getAnnotObjNum(annot),
                                AppAnnotUtil.getAnnotUniqueID(annot),
                                AppAnnotUtil.getAnnotObjNum(replyToAnnot),
                                AppAnnotUtil.getAnnotUniqueID(replyToAnnot));

                        String modifiedDate = AppDmUtil.DEFAULT_MMM_DD_YYYY_HH_MM;
                        try {
                            modifiedDate = AppDmUtil.formatDocumentDate(AppDmUtil.FORMAT_MMM_DD_YYYY_HH_MM, annot.getModifiedDateTime());
                        } catch (PDFException e) {
                            e.printStackTrace();
                        }
                        String creationDate = AppDmUtil.DEFAULT_MMM_DD_YYYY_HH_MM;
                        if (annot.isMarkup()) {
                            String title = "";
                            try {
                                title = ((Markup) annot).getTitle();
                            } catch (PDFException e) {
                                e.printStackTrace();
                            }
                            node.setAuthor(title);

                            String intent = "";
                            try {
                                intent = ((Markup) annot).getIntent();
                            } catch (PDFException e) {
                                e.printStackTrace();
                            }
                            node.setIntent(intent);

                            try {
                                creationDate = AppDmUtil.formatDocumentDate(AppDmUtil.FORMAT_MMM_DD_YYYY_HH_MM, ((Markup) annot).getCreationDateTime());
                            } catch (PDFException e) {
                                e.printStackTrace();
                            }
                        }
                        node.setType(typeString);
                        node.setContent(annot.getContent());

                        if (annot.getType() == Annot.e_Redact) {
                            node.setApplyRedaction(!annot.isEmpty());
                        }

                        if (modifiedDate == null || modifiedDate.equals(AppDmUtil.DEFAULT_MMM_DD_YYYY_HH_MM)) {
                            modifiedDate = creationDate;
                        }
                        node.setModifiedDate(modifiedDate);
                        node.setCreationDate(creationDate);

                        boolean isLocked = AppAnnotUtil.isLocked(annot);
                        boolean isReadOnly = AppAnnotUtil.isReadOnly(annot);
                        node.setLocked(isLocked);
                        node.setReadOnlyFlag(isReadOnly);
                        node.setEditable(!isReadOnly);
                        if (GroupManager.getInstance().isGrouped(mPdfViewCtrl, annot)) {
                            node.setDeletable(GroupManager.getInstance().canDelete(mPdfViewCtrl, annot));
                            node.setCanReply(GroupManager.getInstance().canReply(mPdfViewCtrl, annot));
                            node.setCanModifyContents(GroupManager.getInstance().contentsModifiable(mPdfViewCtrl, annot));
                            node.setCanComment(false);
                            node.setGroupHeaderNM(GroupManager.getInstance().getHeaderUniqueID(mPdfViewCtrl, annot));
                        } else {
                            if (annot.getType() == Annot.e_Redact) {
                                node.setDeletable(AppAnnotUtil.hasModuleLicenseRight(Constants.e_ModuleNameRedaction)
                                        && AppAnnotUtil.isSupportDeleteAnnot(annot)
                                        && !(isLocked || isReadOnly));
                            } else {
                                node.setDeletable(AppAnnotUtil.isSupportDeleteAnnot(annot)
                                        && !(isLocked || isReadOnly));
                            }
                            node.setCanReply(AppAnnotUtil.isAnnotSupportReply(annot) && !isReadOnly);
                            node.setCanModifyContents(AppAnnotUtil.contentsModifiable(typeString));
                            node.setCanComment(!isLocked && !isReadOnly);
                            node.setGroupHeaderNM("");
                        }
                        node.setWithDeletePermission(AnnotPermissionUtil.canDeleteAnnot(uiExtensionsManager.getDocumentManager(), annot));
                        node.setWithModificationPermission(AnnotPermissionUtil.canModifyAnnot(uiExtensionsManager.getDocumentManager(), annot));
                        node.setWithReplyPermission(AnnotPermissionUtil.canReplyAnnot(uiExtensionsManager.getDocumentManager(), annot));
                        mSearchResults.add(node);
                    }
                }

            } catch (PDFException e) {
                e.printStackTrace();
                return false;
            }
            return true;
        }
    }

}
