/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.panel.annot;


import android.content.Context;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.TextView;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.browser.adapter.SuperAdapter;
import com.foxit.uiextensions.browser.adapter.viewholder.SuperViewHolder;
import com.foxit.uiextensions.config.JsonConstants;
import com.foxit.uiextensions.config.modules.annotations.AnnotationsConfig;
import com.foxit.uiextensions.modules.panel.bean.BaseBean;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppResource;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.ArrayList;
import java.util.List;

import androidx.annotation.IntDef;
import androidx.annotation.NonNull;

public class AnnotFilterAdapter extends SuperAdapter<AnnotFilterAdapter.FilterNode> {

    private List<FilterNode> mFilterNodes;
    private List<FilterNode> mSelectedNodes;
    private Context mContext;

    public AnnotFilterAdapter(Context context) {
        super(context);
        mContext = context;
        mFilterNodes = new ArrayList<>();
        mSelectedNodes = new ArrayList<>();
    }

    void addFilterNode(FilterNode node) {
        mFilterNodes.add(node);
    }

    List<String> getSelectedTypes() {
        List<String> types = new ArrayList<>();
        for (FilterNode filterNode : mSelectedNodes) {
            types.add(filterNode.typeString);
        }
        return types;
    }

    @Override
    public void notifyUpdateData() {
        notifyDataSetChanged();
    }

    @Override
    public FilterNode getDataItem(int position) {
        return mFilterNodes.get(position);
    }

    @NonNull
    @Override
    public SuperViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        return new FilterViewHolder(LayoutInflater.from(getContext()).inflate(R.layout.panel_filter_item_layout, parent, false));
    }

    @Override
    public int getItemCount() {
        return mFilterNodes.size();
    }

    class FilterViewHolder extends SuperViewHolder {
        private ImageView mFilterIcon;
        private TextView mFilterName;
        private ImageView mCheck;
        private View mDividerView;
        private View mFilterContentView;

        public FilterViewHolder(View itemView) {
            super(itemView);
            mFilterIcon = itemView.findViewById(R.id.panel_filter_icon);
            mFilterName = itemView.findViewById(R.id.panel_filter_name);
            mCheck = itemView.findViewById(R.id.panel_filter_check);
            mDividerView = itemView.findViewById(R.id.panel_filter_type_divider);
            mFilterContentView = itemView.findViewById(R.id.panel_filter_content);

            mFilterContentView.setOnClickListener(this);
        }

        @Override
        public void bind(BaseBean data, int position) {
            FilterNode filterNote = (FilterNode) data;

            if (filterNote.type == FilterNode.TYPE_DIVIDER) {
                mDividerView.setVisibility(View.VISIBLE);
                mFilterContentView.setVisibility(View.GONE);
            } else {
                mDividerView.setVisibility(View.GONE);
                mFilterContentView.setVisibility(View.VISIBLE);
                mFilterName.setText(AppResource.getString(context, filterNote.descId));
                mFilterIcon.setImageResource(filterNote.iconId);
                ThemeUtil.setTintList(mFilterIcon, ThemeUtil.getEnableIconColor(mContext));
                mCheck.setColorFilter(ThemeConfig.getInstance(getContext()).getPrimaryColor());
                mCheck.setVisibility(filterNote.selected ? View.VISIBLE : View.INVISIBLE);
            }
        }

        @Override
        public void onClick(View v) {
            int id = v.getId();
            int position = getAdapterPosition();
            if (id == R.id.panel_filter_content) {
                FilterNode filterNode = mFilterNodes.get(position);
                if (mSelectedNodes.contains(filterNode)) {
                    mSelectedNodes.remove(filterNode);
                    filterNode.selected = false;
                } else {
                    mSelectedNodes.add(filterNode);
                    filterNode.selected = true;
                }
                notifyItemChanged(position);
            }
        }
    }

    static class FilterNode extends BaseBean {
        static final int TYPE_DIVIDER = 0;
        static final int TYPE_CONTENT = 1;

        @IntDef({TYPE_DIVIDER, TYPE_CONTENT})
        @Retention(RetentionPolicy.SOURCE)
        public @interface Type {
        }

        public int type;
        public int iconId;
        public int descId;
        public String typeString;
        public boolean selected;

        FilterNode(@Type int type) {
            this.type = type;
        }

        FilterNode(int iconId, int descId, String typeString) {
            this.type = TYPE_CONTENT;
            this.iconId = iconId;
            this.descId = descId;
            this.typeString = typeString;
        }
    }

    void addFilterNodes(AnnotationsConfig config) {
        mFilterNodes.clear();

        FilterNode filterNode;
        if (config.isLoadNote) {
            filterNode = new FilterNode(R.drawable.comment_tool_note_bg, R.string.fx_string_note, JsonConstants.TYPE_NOTE);
            addFilterNode(filterNode);
        }
        if (config.isLoadFileattachment) {
            filterNode = new FilterNode(R.drawable.comment_tool_attachment_bg, R.string.fx_string_fileattachment, JsonConstants.TYPE_ATTACHMENT);
            addFilterNode(filterNode);
        }
        if (config.isLoadStamp) {
            filterNode = new FilterNode(R.drawable.comment_tool_stamp_bg, R.string.annot_tool_display_name_stamp, JsonConstants.TYPE_STAMP);
            addFilterNode(filterNode);
        }
        if (config.isLoadTypewriter) {
            filterNode = new FilterNode(R.drawable.comment_tool_typewriter_bg, R.string.annot_tool_display_name_typewrite, JsonConstants.TYPE_TYPEWRITER);
            addFilterNode(filterNode);
        }
        if (config.isLoadCallout) {
            filterNode = new FilterNode(R.drawable.comment_tool_callout_bg, R.string.annot_tool_display_name_callout, JsonConstants.TYPE_CALLOUT);
            addFilterNode(filterNode);
        }
        if (config.isLoadTextbox) {
            filterNode = new FilterNode(R.drawable.comment_tool_textbox_bg, R.string.annot_tool_display_name_textbox, JsonConstants.TYPE_TEXTBOX);
            addFilterNode(filterNode);
        }
        if (config.isLoadHighlight) {
            filterNode = new FilterNode(R.drawable.comment_tool_highlight_bg, R.string.fx_string_highlight, JsonConstants.TYPE_HIGHLIGHT);
            addFilterNode(filterNode);
        }
        if (config.isLoadUnderline) {
            filterNode = new FilterNode(R.drawable.comment_tool_underline_bg, R.string.fx_string_underline, JsonConstants.TYPE_UNDERLINE);
            addFilterNode(filterNode);
        }
        if (config.isLoadStrikeout) {
            filterNode = new FilterNode(R.drawable.comment_tool_strikeout_bg, R.string.fx_string_strikeout, JsonConstants.TYPE_STRIKEOUT);
            addFilterNode(filterNode);
        }
        if (config.isLoadSquiggly) {
            filterNode = new FilterNode(R.drawable.comment_tool_squiggly_bg, R.string.fx_string_squiggly, JsonConstants.TYPE_SQUIGGLY);
            addFilterNode(filterNode);
        }
        if (config.isLoadReplaceText) {
            filterNode = new FilterNode(R.drawable.comment_tool_replace_text_bg, R.string.fx_string_replace_text, JsonConstants.TYPE_REPLACE);
            addFilterNode(filterNode);
        }
        if (config.isLoadInsertText) {
            filterNode = new FilterNode(R.drawable.comment_tool_insert_text_bg, R.string.fx_string_insert_text, JsonConstants.TYPE_CARET);
            addFilterNode(filterNode);
        }
        if (config.isLoadHighlight) {
            filterNode = new FilterNode(R.drawable.comment_tool_area_highlight_bg, R.string.fx_string_area_highlight, JsonConstants.TYPE_AREA_HIGHLIGHT);
            addFilterNode(filterNode);
        }

        if (config.isLoadMeasure) {
            filterNode = new FilterNode(R.drawable.comment_tool_line_measure_bg, R.string.fx_distance, JsonConstants.TYPE_LINEDIMENSION);
            addFilterNode(filterNode);
        }
        if (config.isLoadImage) {
            filterNode = new FilterNode(R.drawable.comment_tool_image_bg, R.string.screen_annot_image, JsonConstants.TYPE_SCREEN_IMAGE);
            addFilterNode(filterNode);
        }

        filterNode = new FilterNode(FilterNode.TYPE_DIVIDER);
        addFilterNode(filterNode);

        if (config.isLoadPencil) {
            filterNode = new FilterNode(R.drawable.drawing_tool_pencil_bg, R.string.fx_string_pencil, JsonConstants.TYPE_INK);
            addFilterNode(filterNode);
        }
        if (config.isLoadSquare) {
            filterNode = new FilterNode(R.drawable.drawing_tool_square, R.string.fx_string_rectangle, JsonConstants.TYPE_SQUARE);
            addFilterNode(filterNode);
        }
        if (config.isLoadCircle) {
            filterNode = new FilterNode(R.drawable.drawing_tool_oval, R.string.fx_string_oval, JsonConstants.TYPE_CIRCLE);
            addFilterNode(filterNode);
        }
        if (config.isLoadLine) {
            filterNode = new FilterNode(R.drawable.drawing_tool_line, R.string.fx_string_line, JsonConstants.TYPE_LINE);
            addFilterNode(filterNode);
        }
        if (config.isLoadArrow) {
            filterNode = new FilterNode(R.drawable.drawing_tool_line_arrow, R.string.fx_string_arrow, JsonConstants.TYPE_LINEARROW);
            addFilterNode(filterNode);
        }
        if (config.isLoadPolyLine) {
            filterNode = new FilterNode(R.drawable.drawing_tool_polygonline, R.string.fx_string_polyLine, JsonConstants.TYPE_POLYLINE);
            addFilterNode(filterNode);
        }
        if (config.isLoadPolygon) {
            filterNode = new FilterNode(R.drawable.drawing_tool_polygon, R.string.fx_string_polygon, JsonConstants.TYPE_POLYGON);
            addFilterNode(filterNode);
        }
        if (config.isLoadCloud) {
            filterNode = new FilterNode(R.drawable.drawing_tool_polygon_cloud, R.string.drawing_tool_cloud_type, JsonConstants.TYPE_POLYGONCLOUD);
            addFilterNode(filterNode);
        }
    }

}
