/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.panel.annot;

import android.app.Activity;
import android.app.ProgressDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.RectF;
import android.os.Handler;
import android.text.Spannable;
import android.text.SpannableStringBuilder;
import android.text.style.ForegroundColorSpan;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.CheckBox;
import android.widget.ImageView;
import android.widget.TextView;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.Task;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.Markup;
import com.foxit.uiextensions.DocumentManager;
import com.foxit.uiextensions.IPermissionProvider;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.common.UIAnnotReply;
import com.foxit.uiextensions.annots.common.UIBtnImageView;
import com.foxit.uiextensions.annots.multiselect.GroupManager;
import com.foxit.uiextensions.annots.multiselect.MultiSelectToolHandler;
import com.foxit.uiextensions.browser.adapter.SuperAdapter;
import com.foxit.uiextensions.browser.adapter.viewholder.SuperViewHolder;
import com.foxit.uiextensions.controls.dialog.AppDialogManager;
import com.foxit.uiextensions.controls.dialog.sheetmenu.ISheetMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UIBottomSheetMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UISheetMenu;
import com.foxit.uiextensions.modules.panel.bean.BaseBean;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AnnotPermissionUtil;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.SystemUiHelper;
import com.foxit.uiextensions.utils.UIToast;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.fragment.app.FragmentActivity;
import androidx.recyclerview.widget.LinearLayoutManager;


public class AnnotAdapter extends SuperAdapter<AnnotNode> {
    static final int NORMAL_STATE = 0;
    static final int SEARCH_STATE = 1;
    static final int EDIT_STATE = 2;
    static final int GROUP_LIST_STATE = 3;

    private Context mContext;
    private PDFViewCtrl mPdfViewCtrl;
    private UIExtensionsManager mUIExtensionsManager;
    private OnItemClickListener mItemListener;
    private OnSheetMenuClickListener mMenuClickListener;

    private ISheetMenu mSheetMenu;
    private ProgressDialog mProgressDialog;
    private GroupListDialog mGroupListDialog;
    private ViewGroup mGroupView;

    private List<String> mFilterTypes;
    private List<AnnotNode> mNodeList;
    private List<AnnotNode> mSearchItems;
    private List<AnnotNode> mEditItems;
    private List<AnnotNode> mSelectedGroupNodeList;
    private List<List<AnnotNode>> mPageNodesList;

    private String mSearchPattern;
    private boolean mIsPad;
    private boolean mDateChanged = false;
    private int mState;
    private int mSelectedPosition = -1;
    private int mCanEditAnnotCount = 0;

    interface DeleteCallback {
        void result(boolean success, AnnotNode node);
    }

    public AnnotAdapter(Context context, PDFViewCtrl pdfViewCtrl) {
        super(context);
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mUIExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        mNodeList = new ArrayList<>();
        mPageNodesList = new ArrayList<>();
        mSelectedGroupNodeList = new ArrayList<>();
        mSearchItems = new ArrayList<>();
        mEditItems = new ArrayList<>();

        mIsPad = AppDisplay.isPad();
        mState = NORMAL_STATE;
    }

    void setFilterTypes(List<String> filterTypes) {
        this.mFilterTypes = filterTypes;
    }

    boolean hasDataChanged() {
        return mDateChanged;
    }

    void resetDataChanged() {
        mDateChanged = false;
    }

    void setState(int state) {
        mState = state;
    }

    int getState() {
        return mState;
    }

    void setSearchPattern(String Pattern) {
        mSearchPattern = Pattern;
    }

    List<AnnotNode> getSearchItems() {
        return mSearchItems;
    }

    List<AnnotNode> getEditItems() {
        return mEditItems;
    }

    void preparePageNodes() {
        mPageNodesList.clear();
        for (int i = 0; i < mPdfViewCtrl.getPageCount(); i++) {
            mPageNodesList.add(new ArrayList<AnnotNode>());
        }
    }

    void setPageNodes(List<List<AnnotNode>> pageNodes) {
        mPageNodesList = pageNodes;
    }

    void setNodeList(List<AnnotNode> nodeList) {
        mNodeList = nodeList;
    }

    void setGroupDialogView(ViewGroup view) {
        mGroupView = view;
    }

    List<AnnotNode> getNodeList() {
        return mNodeList;
    }

    void establishNodeList() {
        mNodeList.clear();
        mCanEditAnnotCount = 0;
        for (int i = 0; i < mPageNodesList.size(); i++) {
            List<AnnotNode> nodes = mPageNodesList.get(i);
            if (nodes != null && nodes.size() > 0) {
                int index = mNodeList.size();
                boolean addPageNode = false;
                int count = 0;
                for (AnnotNode an : nodes) {
                    if (an.isRootNode() && !an.isRedundant()) {
                        if (mFilterTypes != null
                                && mFilterTypes.size() > 0
                                && !mFilterTypes.contains(an.getType()))
                            continue;

                        addPageNode = true;
                        an.setFlag(AnnotNode.FLAG_CONTENT);
                        mNodeList.add(an);
                        count++;
                        if (an.canDelete() && an.isWithDeletePermission())
                            mCanEditAnnotCount++;
                        establishNodeRoot(an);
                    }
                }
                if (getState() != SEARCH_STATE && addPageNode) {
                    AnnotNode pageNode = new AnnotNode(i);
                    pageNode.counter = count;
                    pageNode.setFlag(AnnotNode.FLAG_LABLE);
                    mNodeList.add(index, pageNode);
                }
            }
        }
    }

    void clearSelectedNodes() {
        for (AnnotNode node : mEditItems) {
            node.setChecked(false);
        }
        mEditItems.clear();
    }

    boolean isSelectedAll() {
        return mEditItems.size() > 0 && mEditItems.size() == getCanEditAnnotCount();
    }

    void selectAll(boolean selectAll) {
        mEditItems.clear();
        for (AnnotNode node : mNodeList) {
            if (node.isPageDivider() || node.isRedundant()) continue;

            if (node.isRootNode()) {
                if (selectAll) {
                    if (node.canDelete() && node.isWithDeletePermission()) {
                        node.setChecked(true);
                        mEditItems.add(node);
                    }
                    selectReplyNodes(true, node);
                } else {
                    node.setChecked(false);
                    selectReplyNodes(false, node);
                }
            }
        }
        if (selectAll)
            mCanEditAnnotCount = mEditItems.size();
        notifyUpdateData();
    }

    int getCanEditAnnotCount() {
        return mCanEditAnnotCount;
    }

    void resetCanEditAnnotCount() {
        int count = 0;
        for (AnnotNode node : mNodeList) {
            if (node.isPageDivider() || node.isRedundant()) continue;

            if (node.isRootNode()) {
                if (node.canDelete() && node.isWithDeletePermission())
                    count++;
                establishReplyNode(count, node);
            }
        }
        mCanEditAnnotCount = count;
    }

    private void establishReplyNode(int count, @NonNull AnnotNode parentNode) {
        if (parentNode.isLeafNode()) return;
        for (AnnotNode child : parentNode.getChildren()) {
            if (child.canDelete() && child.isWithDeletePermission())
                count++;
            establishReplyNode(count, child);
        }
    }

    private void selectReplyNodes(boolean select, AnnotNode parentNode) {
        if (parentNode.isLeafNode()) return;
        for (AnnotNode child : parentNode.getChildren()) {
            if (child.canDelete() && child.isWithDeletePermission()) {
                child.setChecked(select);
                if (select)
                    mEditItems.add(child);
            }
            selectReplyNodes(select, child);
        }
    }

    private void establishReplyList(AnnotNode node) {
        if (node.isExpanded()) {
            int index = mNodeList.indexOf(node);
            if (index == -1) return;
            List<AnnotNode> replyList = new ArrayList<>();
            establishReplyNode(replyList, node);
            mNodeList.addAll(index + 1, replyList);
        } else {
            removeReplyChildren(node);
        }
    }

    private void removeReplyChildren(AnnotNode node) {
        if (node.isLeafNode()) return;
        List<AnnotNode> childrenNodes = node.getChildren();
        mNodeList.removeAll(childrenNodes);
        for (AnnotNode childrenNode : childrenNodes) {
            removeReplyChildren(childrenNode);
        }
    }

    void establishReplyNode(List<AnnotNode> replyList, AnnotNode parent) {
        if (parent.isLeafNode() || !parent.isExpanded()) return;
        for (AnnotNode child : parent.getChildren()) {
            child.setFlag(AnnotNode.FLAG_CONTENT);
            replyList.add(child);
            establishReplyNode(replyList, child);
        }
    }

    private void establishNodeRoot(AnnotNode parent) {
        if (parent.isLeafNode()) return;

        for (AnnotNode child : parent.getChildren()) {
            child.setFlag(AnnotNode.FLAG_CONTENT);
            if (getState() == SEARCH_STATE || parent.isExpanded()) {
                mNodeList.add(child);
            }
            if (child.canDelete() && child.isWithDeletePermission())
                mCanEditAnnotCount++;
            establishNodeRoot(child);
        }
    }

    void clearPageNodes(int pageIndex) {
        List<AnnotNode> annotNodes = mPageNodesList.get(pageIndex);
        if (annotNodes != null)
            annotNodes.clear();
    }

    void clearNodes() {
        mNodeList.clear();
        for (int i = 0; i < mPageNodesList.size(); i++) {
            List<AnnotNode> nodes = mPageNodesList.get(i);
            if (nodes != null) {
                nodes.clear();
            }
            mPageNodesList.set(i, null);
        }
        notifyDataSetChanged();
    }

    void addNode(AnnotNode node) {
        List<AnnotNode> nodes = mPageNodesList.get(node.getPageIndex());
        if (nodes == null) {
            nodes = new ArrayList<>();
            mPageNodesList.set(node.getPageIndex(), nodes);
        }
        if (nodes.contains(node)) return;
        if (node.getReplyTo().equals("") && node.getUID().equals("")) {
            return;
        }
        boolean needFind = !node.getReplyTo().equals("");
        for (AnnotNode an : nodes) {
            if (needFind) {
                if (an.getUID().equals(node.getReplyTo())) {
                    node.setParent(an);
                    an.addChildNode(node);
                    needFind = false;
                    continue;
                }
            }
            if (!an.getReplyTo().equals("") && an.getReplyTo().equals(node.getUID())) {
                an.setParent(node);
                node.addChildNode(an);
            }
        }
        nodes.add(node);
    }

    void updateNode(Annot annot) {
        try {
            if (!AppAnnotUtil.isSupportEditAnnot(annot) || AppUtil.isEmpty(AppAnnotUtil.getAnnotUniqueID(annot)))
                return;
            List<AnnotNode> nodes = mPageNodesList.get(annot.getPage().getIndex());
            if (nodes == null) return;
            for (AnnotNode node : nodes) {
                if (node.getUID().equals(AppAnnotUtil.getAnnotUniqueID(annot))) {
                    if (annot.isMarkup())
                        node.setAuthor(((Markup) annot).getTitle());
                    node.setContent(annot.getContent());
                    node.setApplyRedaction(!annot.isEmpty() && annot.getType() == Annot.e_Redact);
                    String date = AppDmUtil.formatDocumentDate(AppDmUtil.FORMAT_MMM_DD_YYYY_HH_MM, annot.getModifiedDateTime());
                    if (date == null || date.equals(AppDmUtil.DEFAULT_MMM_DD_YYYY_HH_MM)) {
                        if (annot.isMarkup())
                            date = AppDmUtil.formatDocumentDate(AppDmUtil.FORMAT_MMM_DD_YYYY_HH_MM, ((Markup) annot).getCreationDateTime());
                    }

                    boolean isLocked = AppAnnotUtil.isLocked(annot);
                    boolean isReadOnly = AppAnnotUtil.isReadOnly(annot);
                    node.setLocked(isLocked);
                    node.setReadOnlyFlag(isReadOnly);
                    node.setEditable(!isReadOnly);
                    if (GroupManager.getInstance().isGrouped(mPdfViewCtrl, annot)) {
                        node.setDeletable(GroupManager.getInstance().canDelete(mPdfViewCtrl, annot));
                        node.setCanReply(GroupManager.getInstance().canReply(mPdfViewCtrl, annot));
                        node.setCanModifyContents(GroupManager.getInstance().contentsModifiable(mPdfViewCtrl, annot));
                        node.setCanComment(false);
                        node.setGroupHeaderNM(GroupManager.getInstance().getHeaderUniqueID(mPdfViewCtrl, annot));
                    } else {
                        node.setDeletable(AppAnnotUtil.isSupportDeleteAnnot(annot) && !(isLocked || isReadOnly));
                        node.setCanReply(AppAnnotUtil.isAnnotSupportReply(annot) && !isReadOnly);
                        node.setCanModifyContents(AppAnnotUtil.contentsModifiable(node.getType()));
                        node.setCanComment(!isLocked && !isReadOnly);
                        node.setGroupHeaderNM("");
                    }
                    DocumentManager dm = mUIExtensionsManager.getDocumentManager();
                    node.setWithDeletePermission(AnnotPermissionUtil.canDeleteAnnot(dm, annot));
                    node.setWithModificationPermission(AnnotPermissionUtil.canModifyAnnot(dm, annot));
                    node.setWithReplyPermission(AnnotPermissionUtil.canReplyAnnot(dm, annot));
                    node.setModifiedDate(date);
                    break;
                }
            }
            notifyDataSetChanged();
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    void removeNode(PDFPage page, String uid) {
        if (uid == null || uid.equals("")) return;

        try {
            List<AnnotNode> nodes = mPageNodesList.get(page.getIndex());
            if (nodes == null) return;
            for (int i = nodes.size() - 1; i >= 0; i--) {
                AnnotNode node = nodes.get(i);
                if (node.getUID().equals(uid)) {
                    removeNodeFromPageNode(node, nodes);
                    if (node.getParent() != null) {
                        node.getParent().removeChild(node);
                    } else {
                        node.removeChildren();
                    }
                    break;
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private void removeNodeFromPageNode(final AnnotNode node, final List<AnnotNode> pageNodes) {
        if (pageNodes == null || node == null || !pageNodes.contains(node))
            return;
        List<AnnotNode> children = node.getChildren();
        if (children != null && children.size() != 0) {
            for (AnnotNode child : children) {
                removeNodeFromPageNode(child, pageNodes);
            }
        }
        pageNodes.remove(node);
    }

    private void flattenNode(@NonNull AnnotNode node, final Event.Callback callback) {
        final List<AnnotNode> nodes = mPageNodesList.get(node.getPageIndex());
        if (nodes == null || !nodes.contains(node)) {
            if (callback != null) callback.result(null, true);
            return;
        }

        PDFPage page = getAnnotNodePage(node);
        if (page == null || page.isEmpty()) {
            if (callback != null) callback.result(null, false);
            return;
        }

        Annot annot = AppAnnotUtil.getAnnot(page, node.getUID());
        if (annot == null || annot.isEmpty()) {
            if (callback != null) {
                //first remove children from page nodes list.
                removeNodeFromPageNode(node, nodes);
                // clear children list.
                node.removeChildren();
                callback.result(null, true);

                establishNodeList();
                notifyDataSetChanged();
            }
            return;
        }
        mUIExtensionsManager.getDocumentManager().flattenAnnot(annot, callback);
    }

    void removeNode(@NonNull final AnnotNode node, final DeleteCallback callback) {
        final List<AnnotNode> nodes = mPageNodesList.get(node.getPageIndex());
        if (nodes == null || !nodes.contains(node)) {
            if (callback != null) {
                callback.result(true, node);
            }
            return;
        }

        PDFPage page = getAnnotNodePage(node);
        if (page == null || page.isEmpty())
            return;
        Annot annot = AppAnnotUtil.getAnnot(page, node.getUID());

        if (annot == null || annot.isEmpty()) {
            if (callback != null) {
                //first remove children from page nodes list.
                removeNodeFromPageNode(node, nodes);
                // clear children list.
                node.removeChildren();
                callback.result(true, node);

                establishNodeList();
                notifyDataSetChanged();
            }
            return;
        }

        mUIExtensionsManager.getDocumentManager().removeAnnot(annot, true, new Event.Callback() {
            @Override
            public void result(Event event, boolean success) {
                callback.result(success, node);
            }
        });
    }

    PDFPage getAnnotNodePage(@NonNull AnnotNode node) {
        try {
            return mPdfViewCtrl.getDoc().getPage(node.getPageIndex());
        } catch (PDFException e) {
            return null;
        }
    }

    @Override
    public void notifyUpdateData() {
        notifyDataSetChanged();
    }

    @Override
    public AnnotNode getDataItem(int position) {
        List<AnnotNode> itemList;
        if (getState() == SEARCH_STATE)
            itemList = mSearchItems;
        else
            itemList = mNodeList;
        if (position < 0 || position >= itemList.size()) return null;
        return itemList.get(position);
    }

    @NonNull
    @Override
    public SuperViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        return new AnnotationViewHolder(LayoutInflater.from(getContext()).inflate(R.layout.panel_annot_item, parent, false));
    }

    @Override
    public long getItemId(int position) {
        return position;
    }

    @Override
    public int getItemCount() {
        List<AnnotNode> itemList;
        if (getState() == SEARCH_STATE)
            itemList = mSearchItems;
        else
            itemList = mNodeList;
        return itemList.size();
    }

    public void dismissGroupList() {
        if (mGroupListDialog != null && mGroupListDialog.isShowing()) {
            mGroupListDialog.dismiss();
            mGroupListDialog = null;
        }
    }

    class AnnotationViewHolder extends SuperViewHolder {
        private View mMainLayout;
        private ImageView mIconImageView;
        private TextView mContentsTextView;
        private TextView mAuthorTextView;
        private TextView mDateTextView;
        private ImageView mItemMore;
        private CheckBox mCheckView;

        private View mPageLayout;
        private TextView mPageIndexTextView;
        private TextView mCounterTextView;
        private UIBtnImageView mExpandView;
        private View mItemView;
        private View mTopDivider;
        private View mBottomDivider;

        AnnotationViewHolder(View itemView) {
            super(itemView);
            mItemView = itemView;
            mPageLayout = itemView.findViewById(R.id.rv_panel_annot_item_page_layout);
            mPageIndexTextView = mPageLayout.findViewById(R.id.rv_panel_annot_item_page_tv);
            mCounterTextView = itemView.findViewById(R.id.rv_panel_annot_item_page_count_tv);
            mContentsTextView = itemView.findViewById(R.id.rv_panel_annot_item_contents_tv);
            mMainLayout = itemView.findViewById(R.id.rv_panel_annot_item_main_layout);
            mAuthorTextView = mMainLayout.findViewById(R.id.rv_panel_annot_item_author_tv);
            mDateTextView = mMainLayout.findViewById(R.id.rv_panel_annot_item_date_tv);
            mIconImageView = mMainLayout.findViewById(R.id.rv_panel_annot_item_icon_iv);
            mItemMore = mMainLayout.findViewById(R.id.rd_panel_annot_item_more);
            mCheckView = mMainLayout.findViewById(R.id.rd_annot_item_checkbox);
            mExpandView = mMainLayout.findViewById(R.id.panel_annot_reply_expand_iv);
            mTopDivider = itemView.findViewById(R.id.rv_panel_annot_item_top_divider);
            mBottomDivider = itemView.findViewById(R.id.rv_panel_annot_item_bootom_divider);
            mTopDivider.setBackgroundColor(AppResource.getColor(mContext, R.color.p1));
            mBottomDivider.setBackgroundColor(AppResource.getColor(mContext, R.color.p1));
            mPageLayout.setBackgroundColor(AppResource.getColor(mContext, R.color.b2));
            mItemMore.setOnClickListener(this);
            mContentsTextView.setOnClickListener(this);
            mExpandView.setOnClickListener(this);
            mMainLayout.setOnClickListener(this);
        }

        @Override
        public void bind(BaseBean data, int position) {
            final AnnotNode node = (AnnotNode) data;
            mTopDivider.setBackgroundColor(AppResource.getColor(mContext, R.color.p1));
            mBottomDivider.setBackgroundColor(AppResource.getColor(mContext, R.color.p1));
            mPageLayout.setBackgroundColor(AppResource.getColor(mContext, R.color.b2));
            if (node.isPageDivider()) {
                mMainLayout.setVisibility(View.GONE);
                mPageLayout.setVisibility(View.VISIBLE);
                mCounterTextView.setText(String.valueOf(node.counter));
                if (node.getFlag() == AnnotNode.FLAG_TOTAL_COUNT)
                    mPageIndexTextView.setText(AppResource.getString(getContext(), R.string.search_find_number));
                else
                    mPageIndexTextView.setText(AppResource.getString(getContext(), R.string.attachment_page_tab,
                            node.getPageIndex() + 1));
                return;
            } else {
                mMainLayout.setVisibility(View.VISIBLE);
                mPageLayout.setVisibility(View.GONE);
            }
            mContentsTextView.setVisibility(AppUtil.isEmpty(node.getContent()) ? View.GONE : View.VISIBLE);
            if (getState() == SEARCH_STATE) {
                mItemMore.setVisibility(View.GONE);
                mCheckView.setVisibility(View.GONE);
                mContentsTextView.setClickable(false);

                if (AppUtil.isEmpty(node.getAuthor())) {
                    mAuthorTextView.setText("");
                    mAuthorTextView.setVisibility(View.GONE);
                } else {
                    String content = node.getAuthor();

                    Object text = mAuthorTextView.getText();
                    SpannableStringBuilder spannable;
                    if (text instanceof SpannableStringBuilder) {
                        spannable = (SpannableStringBuilder) text;
                        spannable.clearSpans();
                        spannable.clear();
                        spannable.append(content);
                    } else {
                        spannable = new SpannableStringBuilder(content);
                    }

                    if (!AppUtil.isEmpty(mSearchPattern)) {
                        int start = content.toLowerCase().indexOf(mSearchPattern, -1);
                        if (start != -1) {
                            spannable.setSpan(new ForegroundColorSpan(ThemeConfig.getInstance(getContext()).getPrimaryColor()),
                                    start, start + mSearchPattern.length(),
                                    Spannable.SPAN_EXCLUSIVE_INCLUSIVE
                            );
                        }
                    }
                    mAuthorTextView.setVisibility(View.VISIBLE);
                    mAuthorTextView.setText(spannable);
                }

                if (!AppUtil.isEmpty(node.getContent())) {
                    String content = node.getContent().toString();
                    Object text = mContentsTextView.getText();
                    SpannableStringBuilder spannable;
                    if (text instanceof SpannableStringBuilder) {
                        spannable = (SpannableStringBuilder) text;
                        spannable.clearSpans();
                        spannable.clear();
                        spannable.append(content);
                    } else {
                        spannable = new SpannableStringBuilder(content);
                    }

                    if (!AppUtil.isEmpty(mSearchPattern)) {
                        int start = content.toLowerCase().indexOf(mSearchPattern, -1);
                        if (start != -1) {
                            spannable.setSpan(new ForegroundColorSpan(ThemeConfig.getInstance(getContext()).getPrimaryColor()),
                                    start, start + mSearchPattern.length(),
                                    Spannable.SPAN_EXCLUSIVE_INCLUSIVE
                            );
                        }
                    }
                    mContentsTextView.setText(spannable);
                }
            } else {
                if (getState() == EDIT_STATE) {
                    mItemMore.setVisibility(View.GONE);
                    mContentsTextView.setClickable(false);
                    boolean canDelete = node.canDelete() && node.isWithDeletePermission();
                    int visibility = canDelete ? View.VISIBLE : View.INVISIBLE;
                    mCheckView.setVisibility(visibility);
                    mCheckView.setChecked(node.isChecked());
                    ThemeUtil.setTintList(mCheckView, ThemeUtil.getCheckboxColor(getContext()));
                } else {
                    mItemMore.setVisibility(View.VISIBLE);
                    mCheckView.setVisibility(View.GONE);
                    mContentsTextView.setClickable(AppAnnotUtil.contentsModifiable(node.getType()));
                }

                if (AppUtil.isEmpty(node.getAuthor())) {
                    mAuthorTextView.setVisibility(View.GONE);
                    mAuthorTextView.setText("");
                } else {
                    mAuthorTextView.setVisibility(View.VISIBLE);
                    mAuthorTextView.setText(node.getAuthor());
                }
                mContentsTextView.setText(node.getContent());
            }

            int level = node.getLevel();
            mDateTextView.setText(node.getModifiedDate());
            if (node.isRootNode()) {
                int drawable = AppAnnotUtil.getIconId(node.getType());
                mIconImageView.setImageResource(drawable);
                mExpandView.setVisibility(View.GONE);
            } else if (level == 1 && !node.isLeafNode()) {
                if (getState() == SEARCH_STATE) {
                    mExpandView.setVisibility(View.GONE);
                    mIconImageView.setImageResource(R.drawable.panel_annot_reply_arrow_large);
                } else {
                    mExpandView.setVisibility(View.VISIBLE);
                    if (node.isExpanded())
                        mExpandView.setImageResource(R.drawable.rd_collapse_icon);
                    else
                        mExpandView.setImageResource(R.drawable.rd_expand_icon);
                    mIconImageView.setImageResource(R.drawable.panel_annot_reply_arrow_small);
                }
            } else {
                if (getState() == SEARCH_STATE) {
                    mExpandView.setVisibility(View.GONE);
                    mIconImageView.setImageResource(R.drawable.panel_annot_reply_arrow_large);
                } else {
                    mExpandView.setVisibility(View.INVISIBLE);
                    mIconImageView.setImageResource(R.drawable.panel_annot_reply_arrow_small);
                }
            }

            if (mPdfViewCtrl.getDoc() != null) {
                boolean moreMenuEnabled;
                if (mUIExtensionsManager.getDocumentManager().canAddAnnot()&& mUIExtensionsManager.isEnableModification()) {
                    if (getState() == GROUP_LIST_STATE || AppUtil.isEmpty(node.getGroupHeaderNM())) {
                        moreMenuEnabled = (node.isCanModifyContents() && node.canComment() && node.isWithModificationPermission())
                                || (node.canDelete() && node.isWithDeletePermission())
                                || (node.canReply() && node.isWithReplyPermission());
                    } else {
                        moreMenuEnabled = (node.isCanModifyContents() && node.isWithModificationPermission())
                                || (node.canDelete() && node.isWithDeletePermission())
                                || (node.canReply() && node.isWithReplyPermission());
                    }
                } else {
                    boolean isNodeWithPermission = node.isWithModificationPermission() || node.isWithDeletePermission() || node.isWithReplyPermission();
                    moreMenuEnabled = node.isCanModifyContents() && isNodeWithPermission;
                }
                mItemMore.setEnabled(moreMenuEnabled);
                if (!mItemMore.isEnabled()) {
                    mItemMore.setVisibility(View.GONE);
                }
            }

            if (mSelectedGroupNodeList.contains(node) || (!mIsPad && mSelectedPosition == position)) {
                mMainLayout.setSelected(true);
            } else {
                mMainLayout.setSelected(false);
            }

            mItemView.setBackground(AppResource.getDrawable(mContext, R.drawable.rd_list_item_bg));
            mPageIndexTextView.setTextColor(AppResource.getColor(mContext, R.color.t3));
            mCounterTextView.setTextColor(AppResource.getColor(mContext, R.color.t3));
            mMainLayout.setBackground(AppResource.getDrawable(mContext, R.drawable.rd_list_item_bg));
            ThemeUtil.setTintList(mExpandView, ThemeUtil.getPrimaryIconColor(mContext));
            ThemeUtil.setTintList(mIconImageView, ThemeUtil.getEnableIconColor(mContext));
            mAuthorTextView.setTextColor(AppResource.getColor(mContext, R.color.t4));
            mDateTextView.setTextColor(AppResource.getColor(mContext, R.color.t3));
            ThemeUtil.setTintList(mItemMore, ThemeUtil.getPrimaryIconColor(mContext));
            mContentsTextView.setTextColor(AppResource.getColor(mContext, R.color.t4));

        }

        @Override
        public void onClick(final View v) {
            if (AppUtil.isFastDoubleClick()) return;

            int id = v.getId();
            final int position = getAdapterPosition();
            if (position == -1) return;

            final AnnotNode node;
            if (getState() == SEARCH_STATE)
                node = mSearchItems.get(position);
            else
                node = mNodeList.get(position);
            if (id == R.id.rd_panel_annot_item_more) {
                List<Integer> items = new ArrayList<>();

                mSelectedGroupNodeList.clear();
                if (getState() != GROUP_LIST_STATE) {
                    if (!AppUtil.isEmpty(node.getGroupHeaderNM())) {
                        List<AnnotNode> nodes = mPageNodesList.get(node.getPageIndex());
                        if (nodes != null) {
                            for (AnnotNode annotNode : nodes) {
                                if (node.getGroupHeaderNM().equals(annotNode.getGroupHeaderNM())) {
                                    mSelectedGroupNodeList.add(annotNode);
                                }
                            }
                        }

                        if (mSelectedGroupNodeList.size() > 0) {
                            notifyDataSetChanged();
                            if (node.isWithModificationPermission() || node.isWithDeletePermission() || node.isWithReplyPermission())
                                items.add(ISheetMenu.GROUP_LIST_MENU);
                        }
                    }

                    mSelectedPosition = position;
                    if (mSelectedGroupNodeList.size() == 0)
                        notifyItemChanged(mSelectedPosition);
                } else {
                    mSelectedPosition = position;
                    if (mSelectedGroupNodeList.size() == 0)
                        notifyItemChanged(mSelectedPosition);
                }

                mSheetMenu = UISheetMenu.newInstance((FragmentActivity) mUIExtensionsManager.getAttachedActivity());
                if (mIsPad)
                    mSheetMenu.setWidth(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_pad_more_menu_width));
                if (!mUIExtensionsManager.getDocumentManager().canAddAnnot()|| !mUIExtensionsManager.isEnableModification() && node.isCanModifyContents()) {
                    if (node.isWithModificationPermission())
                        items.add(ISheetMenu.COMMENT_MENU);
                } else {
                    if (node.canReply() && node.isWithReplyPermission())
                        items.add(ISheetMenu.REPLY_MENU);
                    if (node.isCanModifyContents() && node.isWithModificationPermission()) {
                        if (node.canComment())
                            items.add(ISheetMenu.COMMENT_MENU);
                    }
                    if (node.getLevel() == 0
                            && node.isWithModificationPermission()
                            && mUIExtensionsManager.getDocumentManager().withDeletePermission())
                        items.add(ISheetMenu.FLATTEN_MENU);
                    if (node.canDelete() && node.isWithDeletePermission())
                        items.add(ISheetMenu.DELETE_MENU);
                }
                mSheetMenu.setSheetItems(items);
                mSheetMenu.setSheetItemClickListener(new UIBottomSheetMenu.OnSheetItemClickListener() {
                    @Override
                    public void onItemClick(int type) {
                        if (ISheetMenu.COMMENT_MENU == type) {
                            onCommentListener(position, node);
                        } else if (ISheetMenu.REPLY_MENU == type) {
                            onReplyListener(node);
                        } else if (ISheetMenu.DELETE_MENU == type) {
                            onDeleteListener(node);
                        } else if (ISheetMenu.FLATTEN_MENU == type) {
                            if (mUIExtensionsManager.getPermissionProvider() != null) {
                                mUIExtensionsManager.getPermissionProvider().checkPermission(
                                        IPermissionProvider.FUNCTION_FLATTEN, new IPermissionProvider.IPermissionState() {
                                            @Override
                                            public void onState(int state) {
                                                if (state == IPermissionProvider.PERMISSION_STATE_SHOW) {
                                                    onFlattenListener(node);
                                                }
                                            }
                                        }
                                );
                            } else {
                                onFlattenListener(node);
                            }
                        } else if (ISheetMenu.GROUP_LIST_MENU == type) {
                            final List<AnnotNode> groupList = new ArrayList<>(mSelectedGroupNodeList);
                            AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
                                @Override
                                public void run() {
                                    showGroupListDialog(groupList);
                                }
                            });
                        }
                        dismissSheetMenu();
                    }
                });
                mSheetMenu.setOnSheetDismissListener(new ISheetMenu.OnSheetDismissListener() {
                    @Override
                    public void onDismiss(ISheetMenu sheetMenu) {
                        if (mSelectedGroupNodeList.size() > 0) {
                            clearGroupNodes();
                        } else {
                            if (mSelectedPosition != -1) {
                                int tempPosition = mSelectedPosition;
                                mSelectedPosition = -1;
                                if (tempPosition >= 0)
                                    notifyItemChanged(tempPosition);
                            }
                        }
                    }
                });
                showSheetMenu(v);
            } else if (id == R.id.rv_panel_annot_item_contents_tv) {
                PDFPage page = getAnnotNodePage(node);
                if (page == null) return;
                final Annot annot = AppAnnotUtil.getAnnot(page, node.getUID());
                if (annot == null || annot.isEmpty()) return;

                if (AppAnnotUtil.isAnnotSupportReply(annot) && !node.isReadOnly()) {
                    boolean canEdit = mUIExtensionsManager.getDocumentManager().canAddAnnot()&& mUIExtensionsManager.isEnableModification()
                            && !(node.isReadOnly() || node.isLocked());
                    canEdit = canEdit && node.isWithModificationPermission();
                    UIAnnotReply.replyToAnnot(mPdfViewCtrl, mUIExtensionsManager.getRootView(),
                            canEdit, UIAnnotReply.TYPE_COMMENT, new UIAnnotReply.ReplyCallback() {
                                @Override
                                public void result(final String content) {
                                    if (!AppAnnotUtil.isSupportEditAnnot(annot))
                                        return;

                                    mUIExtensionsManager.getDocumentManager().modifyAnnot(annot,
                                            new UIAnnotReply.CommentContent(annot, content), true, new Event.Callback() {
                                                @Override
                                                public void result(Event event, boolean success) {
                                                    if (success) {
                                                        node.setContent(content);
                                                        try {
                                                            node.setModifiedDate(AppDmUtil.formatDocumentDate(AppDmUtil.FORMAT_MMM_DD_YYYY_HH_MM, annot.getModifiedDateTime()));
                                                        } catch (PDFException e) {
                                                            e.printStackTrace();
                                                        }
                                                        notifyItemChanged(position);
                                                    }
                                                }
                                            });
                                }

                                @Override
                                public String getContent() {
                                    return node.getContent().toString();
                                }
                            });
                } else {
                    if (!node.isPageDivider() && node.isRootNode()) {
                        if (!AppUtil.isEmpty(node.getUID())) {
                            mPdfViewCtrl.gotoPage(node.getPageIndex(), 0, 0);
                            if (mUIExtensionsManager.getPanelManager().isShowingPanel()) {
                                mUIExtensionsManager.getPanelManager().hidePanel();
                            }
                        }
                    }
                }
            } else if (id == R.id.panel_annot_reply_expand_iv) {
                node.setExpanded(!node.isExpanded());
                establishReplyList(node);
                notifyUpdateData();
            } else if (id == R.id.rv_panel_annot_item_main_layout) {
                if (getState() == EDIT_STATE) {
                    if (node.isRedundant()
                            || !mUIExtensionsManager.getDocumentManager().canAddAnnot()
                            || !mUIExtensionsManager.isEnableModification()
                            || !node.canDelete() || !node.isWithDeletePermission()) return;

                    boolean checked = !node.isChecked();
                    node.setChecked(checked);
                    if (checked) {
                        if (!mEditItems.contains(node))
                            mEditItems.add(node);
                    } else {
                        mEditItems.remove(node);
                    }
                    notifyItemChanged(position);
                } else {
                    if (jumpToPage(node)) {
                        if (mUIExtensionsManager.getPanelManager().isShowingPanel())
                            mUIExtensionsManager.getPanelManager().hidePanel();
                    }
                }

                if (mItemListener != null)
                    mItemListener.onItemClick(position, node);
            }
        }

        private void onCommentListener(final int position, final AnnotNode node) {
            if (!mUIExtensionsManager.getDocumentManager().canAddAnnot()|| !mUIExtensionsManager.isEnableModification()) {
                UIAnnotReply.replyToAnnot(mPdfViewCtrl, mUIExtensionsManager.getRootView(),
                        false, UIAnnotReply.TYPE_COMMENT, new UIAnnotReply.ReplyCallback() {
                            @Override
                            public void result(final String contents) {
                            }

                            @Override
                            public String getContent() {
                                return (String) node.getContent();
                            }
                        });
            } else {
                UIAnnotReply.replyToAnnot(mPdfViewCtrl, mUIExtensionsManager.getRootView(),
                        true, UIAnnotReply.TYPE_COMMENT, new UIAnnotReply.ReplyCallback() {
                            @Override
                            public void result(final String content) {
                                PDFPage page = getAnnotNodePage(node);
                                if (page == null) return;
                                final Annot annot = AppAnnotUtil.getAnnot(page, node.getUID());
                                if (!AppAnnotUtil.isSupportEditAnnot(annot))
                                    return;

                                mUIExtensionsManager.getDocumentManager().modifyAnnot(annot,
                                        new UIAnnotReply.CommentContent(annot, content), true, new Event.Callback() {
                                            @Override
                                            public void result(Event event, boolean success) {
                                                if (success) {
                                                    node.setContent(content);
                                                    try {
                                                        node.setModifiedDate(AppDmUtil.formatDocumentDate(AppDmUtil.FORMAT_MMM_DD_YYYY_HH_MM, annot.getModifiedDateTime()));
                                                    } catch (PDFException e) {
                                                        e.printStackTrace();
                                                    }
                                                    notifyItemChanged(position);
                                                }
                                            }
                                        });
                            }

                            @Override
                            public String getContent() {
                                return node.getContent().toString();
                            }
                        });
            }
        }

        private void onReplyListener(final AnnotNode node) {
            UIAnnotReply.replyToAnnot(mPdfViewCtrl, mUIExtensionsManager.getRootView(), true, UIAnnotReply.TYPE_REPLY, new UIAnnotReply.ReplyCallback() {
                @Override
                public void result(final String contents) {
                    PDFPage page = getAnnotNodePage(node);
                    if (page == null) return;
                    Annot annot = AppAnnotUtil.getAnnot(page, node.getUID());
                    if (annot == null) return;

                    boolean isGroupAnnot = false;
                    try {
                        if (GroupManager.getInstance().isGrouped(mPdfViewCtrl, annot)) {
                            isGroupAnnot = true;
                            annot = AppAnnotUtil.createAnnot(((Markup) annot).getGroupHeader());
                        }
                        if (annot == null) return;

                    } catch (PDFException e) {
                        e.printStackTrace();
                    }
                    final String uid = AppDmUtil.randomUUID(null);
                    final boolean finalIsGroupAnnot = isGroupAnnot;
                    final Annot finalAnnot = annot;
                    UIAnnotReply.addReplyAnnot(mPdfViewCtrl, annot, page, uid, contents, new Event.Callback() {
                        @Override
                        public void result(Event event, boolean success) {
                            if (finalIsGroupAnnot
                                    && mFilterTypes != null
                                    && mFilterTypes.size() > 0
                                    && !mFilterTypes.contains(AppAnnotUtil.getTypeString(finalAnnot))) {
                                UIToast.getInstance(getContext()).show(AppResource.getString(mContext, R.string.fx_add_reply_success_tips));
                            }
                            if (mMenuClickListener != null)
                                mMenuClickListener.onSheetMenuClick(success, ISheetMenu.REPLY_MENU, node);
                        }
                    });
                }

                @Override
                public String getContent() {
                    return null;
                }
            });
        }

        private void onDeleteListener(AnnotNode node) {
            ToolHandler currentToolHandler = mUIExtensionsManager.getCurrentToolHandler();
            if (currentToolHandler != null
                    && currentToolHandler.getType().equals(ToolHandler.TH_TYPE_SELECT_ANNOTATIONS)) {
                MultiSelectToolHandler selectToolHander = (MultiSelectToolHandler) currentToolHandler;
                List<Annot> annots = selectToolHander.getSelectAnnots();
                for (Annot annot : annots) {
                    if (AppAnnotUtil.getAnnotUniqueID(annot).equals(node.getUID())) {
                        mUIExtensionsManager.setCurrentToolHandler(null);
                        break;
                    }
                }
            }

            showProgressDlg(AppResource.getString(mContext, R.string.rv_panel_annot_deleting));
            removeNode(node, new DeleteCallback() {
                @Override
                public void result(boolean success, AnnotNode node) {
                    dismissProgressDlg();
                    if (success)
                        notifyDataSetChanged();
                    if (mMenuClickListener != null)
                        mMenuClickListener.onSheetMenuClick(success, ISheetMenu.DELETE_MENU, node);
                }
            });
        }

        private void onFlattenListener(final AnnotNode node) {
            flattenNode(node, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success)
                        notifyDataSetChanged();
                    if (mMenuClickListener != null)
                        mMenuClickListener.onSheetMenuClick(success, ISheetMenu.FLATTEN_MENU, node);
                }
            });
        }

        private void showGroupListDialog(List<AnnotNode> groupList) {
            Activity context = mUIExtensionsManager.getAttachedActivity();
            if (context == null) return;
            SystemUiHelper.getInstance().showSystemUI(context);

            mGroupListDialog = new GroupListDialog(context, mPdfViewCtrl);
            mGroupListDialog.resetWH();
            mGroupListDialog.initData(mPageNodesList, groupList);
            mGroupListDialog.showDialog();
        }
    }

    private void clearGroupNodes() {
        mSelectedPosition = -1;
        if (mSelectedGroupNodeList.size() > 0) {
            mSelectedGroupNodeList.clear();
            notifyDataSetChanged();
        }
    }

    void onPagesRemoved(boolean success, int[] pageIndexes) {
        if (!success)
            return;
        try {
            for (int i = 0; i < pageIndexes.length; i++) {
                int pageIndex = pageIndexes[i] - i;
                mPageNodesList.remove(pageIndex);
            }

            updateNodePageIndex(findMin(pageIndexes), mPageNodesList.size());
            establishNodeList();
            mDateChanged = true;
        } catch (Throwable e) {
            e.printStackTrace();
        }
    }

    public int findMin(int[] arr) {
        int min = Integer.MAX_VALUE;
        for (int num : arr) {
            if (num < min) {
                min = num;
            }
        }
        return min;
    }

    void updateNodePageIndex(int start, int end) {
        for (int i = start; i < end; i++) {
            List<AnnotNode> nodes = mPageNodesList.get(i);
            if (nodes != null) {
                for (AnnotNode node : nodes) {
                    node.setPageIndex(i);
                }
            }
        }
    }

    void onPageMoved(boolean success, int index, int dstIndex) {
        if (!success) return;
        if (index < dstIndex) {
            for (int i = index; i < dstIndex; i++) {
                Collections.swap(mPageNodesList, i, i + 1);
            }
        } else {
            for (int i = index; i > dstIndex; i--) {
                Collections.swap(mPageNodesList, i, i - 1);
            }
        }
        updateNodePageIndex(Math.min(index, dstIndex), Math.max(index, dstIndex) + 1);
        establishNodeList();
        mDateChanged = true;
    }

    void onPagesInsert(boolean success, int dstIndex, int[] range) {
        if (success) {
            for (int i = 0; i < range.length / 2; i++) {
                for (int index = 0; index < range[2 * i + 1]; index++) {
                    mPageNodesList.add(dstIndex, null);
                    dstIndex++;
                }
            }
            updateNodePageIndex(dstIndex, mPageNodesList.size());
            establishNodeList();
            mDateChanged = true;
        }
    }

    private int getSelectedCount() {
        int count = 0;
        for (int i = 0; i < mPageNodesList.size(); i++) {
            List<AnnotNode> nodes = mPageNodesList.get(i);
            if (nodes != null) {
                for (AnnotNode node : nodes) {
                    if (node.isChecked()) {
                        count++;
                    }
                }
            }
        }
        return count;
    }

    private void showSheetMenu(View view) {
        if (getState() == GROUP_LIST_STATE) {
            Rect rect = new Rect();
            view.getGlobalVisibleRect(rect);

            int[] location = new int[2];
            view.getLocationOnScreen(location);
            int screen_x = location[0];
            int screen_y = location[1];
            if (AppDevice.isChromeOs(mUIExtensionsManager.getAttachedActivity())) {
                mUIExtensionsManager.getRootView().getLocationOnScreen(location);
                screen_x -= location[0];
                screen_y -= location[1];
                rect.set(screen_x, screen_y, screen_x + rect.width(), screen_y + rect.height());
            } else {
                view.getLocationInWindow(location);
                int window_x = location[0];
                int window_y = location[1];
                rect.offset(screen_x - window_x, screen_y - window_y);
            }
            mSheetMenu.show(mUIExtensionsManager.getRootView(), rect);
        } else {
            Rect rect = new Rect();
            view.getGlobalVisibleRect(rect);
            mSheetMenu.show(mUIExtensionsManager.getRootView(), rect);
        }
    }

    private void dismissSheetMenu() {
        if (mSheetMenu != null && mSheetMenu.isShowing()) {
            mSheetMenu.dismiss();
        }
    }

    void onConfigurationChanged(final LinearLayoutManager layoutManager) {
        if (mGroupListDialog != null && mGroupListDialog.isShowing()) {
            mGroupListDialog.resetWH();
            mGroupListDialog.showDialog();
            mGroupListDialog.setOnShowListener(new DialogInterface.OnShowListener() {
                @Override
                public void onShow(DialogInterface dialog) {
                    updateSheetMenu(layoutManager);
                }
            });
        } else {
            updateSheetMenu(layoutManager);
        }
    }

    private void updateSheetMenu(final LinearLayoutManager layoutManager) {
        if (mSheetMenu != null && mSheetMenu.isShowing()) {
            new Handler().postDelayed(new Runnable() {
                @Override
                public void run() {
                    if (mIsPad) {
                        View view = layoutManager.findViewByPosition(mSelectedPosition);
                        if (view != null) {
                            View moreView = view.findViewById(R.id.rd_panel_annot_item_more);
                            showSheetMenu(moreView);
                        } else {
                            mbUpdateMenu = true;

                            int scrollPosition = Math.min(getItemCount() - 1, mSelectedPosition + 1);
                            layoutManager.scrollToPosition(scrollPosition);
                        }
                    } else {
                        showSheetMenu(mUIExtensionsManager.getRootView());
                    }
                }
            }, 380);
        }
    }

    private boolean mbUpdateMenu = false;

    void onScrolled(LinearLayoutManager layoutManager) {
        if (mSelectedPosition == -1 || !mbUpdateMenu) return;

        mbUpdateMenu = false;
        View view = layoutManager.findViewByPosition(mSelectedPosition);
        if (view != null) {
            View moreView = view.findViewById(R.id.rd_panel_annot_item_more);
            showSheetMenu(moreView);
        } else {
            dismissSheetMenu();
        }
    }

    private void showProgressDlg(String msg) {
        if (mProgressDialog == null) {
            Context context = mUIExtensionsManager.getAttachedActivity();
            mProgressDialog = new ProgressDialog(context);
            mProgressDialog.setProgressStyle(ProgressDialog.STYLE_SPINNER);
            mProgressDialog.setCancelable(false);
            mProgressDialog.setIndeterminate(false);
        }

        if (mProgressDialog != null && !mProgressDialog.isShowing()) {
            mProgressDialog.setMessage(msg);
            AppDialogManager.getInstance().showAllowManager(mProgressDialog, null);
        }
    }

    private void dismissProgressDlg() {
        if (mProgressDialog != null && mProgressDialog.isShowing()) {
            AppDialogManager.getInstance().dismiss(mProgressDialog);
            mProgressDialog = null;
        }
    }

    AnnotNode getAnnotNode(PDFPage page, String uid) {
        if (uid == null || uid.equals("")) return null;

        try {
            List<AnnotNode> nodes = mPageNodesList.get(page.getIndex());
            if (nodes == null) return null;
            for (AnnotNode node : nodes) {
                if (node.getUID().equals(uid)) {
                    return node;
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return null;
    }

    boolean canEdit(@NonNull AnnotNode node) {
        if (node.isRootNode()) {
            return node.canEdit();
        } else {
            return node.canEdit() && canEdit(node.getParent());
        }
    }

    boolean canDelete(@NonNull AnnotNode node) {
        if (node.isRootNode()) {
            return node.canDelete() && node.isWithDeletePermission();
        } else {
            return node.canDelete() && node.isWithDeletePermission() && canDelete(node.getParent());
        }
    }

    boolean canComments(@NonNull AnnotNode node) {
        if (node.isRootNode()) {
            return node.canComment();
        } else {
            return node.canComment() && canComments(node.getParent());
        }
    }

    boolean canReply(@NonNull AnnotNode node) {
        if (node.isRootNode()) {
            return node.canReply();
        } else {
            return node.canReply() && canReply(node.getParent());
        }
    }

    private boolean jumpToPage(final AnnotNode node) {
        if (node != null && !node.isPageDivider() && node.isRootNode() && !AppUtil.isEmpty(node.getUID())) {
            Task.CallBack callBack = new Task.CallBack() {
                @Override
                public void result(Task task) {
                    try {
                        PDFPage page = mPdfViewCtrl.getDoc().getPage(node.getPageIndex());
                        Annot annot = AppAnnotUtil.getAnnot(page, node.getUID());
                        if (annot == null || annot.isEmpty()) return;
                        RectF rect = AppUtil.toRectF(annot.getRect());
                        RectF rectPageView = new RectF();

                        //Covert rect from the PDF coordinate system to the page view coordinate system,
                        // and show the annotation to the middle of the screen as possible.
                        if (mPdfViewCtrl.convertPdfRectToPageViewRect(rect, rectPageView, node.getPageIndex())) {
                            RectF pageRect = new RectF(0, 0, page.getWidth(), page.getHeight());
                            RectF pageViewRect = new RectF();
                            mPdfViewCtrl.convertPdfRectToPageViewRect(pageRect, pageViewRect, node.getPageIndex());

                            float devX = 0;
                            float devY = 0;
                            boolean intersect = rectPageView.intersect(pageViewRect.left, pageViewRect.top, pageViewRect.right, pageViewRect.bottom);
                            if (intersect) {
                                devX = rectPageView.left - (mPdfViewCtrl.getWidth() - rectPageView.width()) / 2;
                                devY = rectPageView.top - (mPdfViewCtrl.getHeight() - rectPageView.height()) / 2;
                            }
                            mPdfViewCtrl.gotoPage(node.getPageIndex(), devX, devY);
                        } else {
                            mPdfViewCtrl.gotoPage(node.getPageIndex(), new PointF(rect.left, rect.top));
                        }
//                        if (mUIExtensionsManager.getCurrentToolHandler() != null) {
//                            mUIExtensionsManager.setCurrentToolHandler(null);
//                        }
                        mUIExtensionsManager.getDocumentManager().setCurrentAnnot(annot);
                    } catch (PDFException e) {
                        e.printStackTrace();
                    }
                }
            };

            mPdfViewCtrl.addTask(new Task(callBack) {
                @Override
                protected void execute() {
                }
            });
            return true;
        }
        return false;
    }

    void setItemClickListener(OnItemClickListener listener) {
        this.mItemListener = listener;
    }

    void setSheetMenuClickListener(OnSheetMenuClickListener listener) {
        this.mMenuClickListener = listener;
    }

    interface OnItemClickListener {
        void onItemClick(int position, AnnotNode node);
    }

    interface OnSheetMenuClickListener {
        void onSheetMenuClick(boolean success, int type, AnnotNode node);
    }

}
