/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.panel;

import com.foxit.uiextensions.controls.panel.PanelHost;
import com.foxit.uiextensions.controls.panel.PanelSpec;

/**
 *  The interface to handle the creation, display, hiding, layout, and interaction of panels.
 */
public interface IPanelManager {

    /**
     * Listener for monitoring events about panels.
     */
    interface OnPanelEventListener {
        /**
         * Called when the panel has settled in a completely open state.
         */
        void onPanelOpened();

        /**
         * Called when the panel has settled in a completely closed state.
         */
        void onPanelClosed();
    }

    /**
     * Get the panel host
     *
     * @return The panel host
     */
    PanelHost getPanelHost();

    /**
     * Display the default or the current panel window.
     */
    void showPanel();

    /**
     * Display the panel window by the specified panel type.
     *
     * @param panelType The panel type, start with {@link PanelSpec#BOOKMARKS},
     */
    void showPanel(int panelType);

    /**
     * @return Whether the panel is currently showing.
     */
    boolean isShowingPanel();

    /**
     * Dismiss the panel and its window.
     */
    void hidePanel();

    /**
     * <p>Adds a child panel view. If the panel type{@link PanelSpec#getPanelType()} already exists, will not be added again.
     * reference{@link PanelSpec#BOOKMARKS}</p>
     *
     * @param panelSpec the child panel to add
     */
    void addPanel(PanelSpec panelSpec);

    /**
     * <p>Adds a child panel view. If the panel type{@link PanelSpec#getPanelType()} already exists, will not be added again.
     * reference{@link PanelSpec#BOOKMARKS}</p>
     *
     * @param index     the position at which to add the child
     * @param panelSpec the child panel to add
     */
    void addPanel(int index, PanelSpec panelSpec);

    /**
     * Remove the child panel view.
     *
     * @param panelSpec the child panel view.
     */
    void removePanel(PanelSpec panelSpec);

    /**
     * Remove the child panel view at the specified position in the group.
     *
     * @param panelType the position in the group of the panel to remove
     */
    void removePanel(int panelType);

    /**
     * Adds the specified listener to the list of listeners that will be notified of panel events.
     *
     * @param listener Listener to notify when panel events occur.
     */
    public void registerPanelEventListener(OnPanelEventListener listener);

    /**
     * Removes the specified listener from the list of listeners that will be notified of panel
     * events.
     *
     * @param listener Listener to remove from being notified of panel events
     */
    public void unregisterPanelEventListener(OnPanelEventListener listener);

}
