/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules;

import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.RectF;
import android.view.View;
import android.view.ViewGroup;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.controls.toolbar.IBaseItem;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.impl.BaseItemImpl;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.modules.panel.IPanelManager;
import com.foxit.uiextensions.pdfreader.ILayoutChangeListener;
import com.foxit.uiextensions.pdfreader.IStateChangeListener;
import com.foxit.uiextensions.pdfreader.config.ReadStateConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.SystemUiHelper;

import java.util.List;

/**
 * Search the text on the PDF document, and provides the information of all the found results.
 */
public class SearchModule implements Module {
    private Context mContext = null;
    private PDFViewCtrl mPdfViewCtrl = null;
    private ViewGroup mParent = null;
    private SearchView mSearchView = null;
    private PDFViewCtrl.UIExtensionsManager mUiExtensionsManager;

    private IBaseItem mSearchButtonItem;
    private int mLastReadState = ReadStateConfig.STATE_NORMAL;

    public SearchModule(Context context, ViewGroup parent, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        if (context == null || parent == null || pdfViewCtrl == null) {
            throw new NullPointerException();
        }
        mContext = context;
        mParent = parent;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = uiExtensionsManager;
    }

    @Override
    public boolean loadModule() {
        mSearchView = new SearchView(mContext, mParent, mPdfViewCtrl);

        mPdfViewCtrl.registerDocEventListener(mDocEventListener);
        mPdfViewCtrl.registerDrawEventListener(mDrawEventListener);

        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            ((UIExtensionsManager) mUiExtensionsManager).registerModule(this);
            ((UIExtensionsManager) mUiExtensionsManager).registerStateChangeListener(mStateChangeListener);
            ((UIExtensionsManager) mUiExtensionsManager).registerLayoutChangeListener(mLayoutChangeListener);
            ((UIExtensionsManager) mUiExtensionsManager).registerThemeEventListener(mThemeEventListener);
            initSearchItem();
        }
        return true;
    }


    private void initSearchItem() {
        final UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
        mSearchView.setSearchCancelListener(new SearchView.SearchCancelListener() {
            @Override
            public void onSearchCancel() {
                SystemUiHelper.getInstance().setAllowedHideSystemUI(uiExtensionsManager.getAttachedActivity(), true);

                uiExtensionsManager.getMainFrame().showToolbars();
                uiExtensionsManager.changeState(mSearchView.isRedactSearch() ? ReadStateConfig.STATE_REDACT : mLastReadState);
            }
        });

        mSearchButtonItem = new BaseItemImpl(mContext, R.drawable.toobar_search_icon);
        mSearchButtonItem.setImageTintList(ThemeUtil.getTopBarIconColorStateList(mContext));
        mSearchButtonItem.setTag(ToolbarItemConfig.ITEM_TOPBAR_SEARCH);
        mSearchButtonItem.setId(R.id.id_top_bar_search);
        mSearchButtonItem.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (AppUtil.isFastDoubleClick()) {
                    return;
                }
                uiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_Search);
                mLastReadState = ((UIExtensionsManager) mUiExtensionsManager).getState();
                mSearchView.launchSearchView();
                mSearchView.show();
            }
        });
        uiExtensionsManager.getMainFrame().getTopToolbar().addView(mSearchButtonItem, BaseBar.TB_Position.Position_RB);
    }

    public void startSearch(String text, boolean Consecutive, boolean Word) {
        mLastReadState = ((UIExtensionsManager) mUiExtensionsManager).getState();
        mSearchView.launchSearchView();
        mSearchView.setFlag(Consecutive, Word);
        mSearchView.show();
        mSearchView.startSearch(text);
    }

    @Override
    public String getName() {
        return Module.MODULE_NAME_SEARCH;
    }

    @Override
    public boolean unloadModule() {
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            ((UIExtensionsManager) mUiExtensionsManager).unregisterStateChangeListener(mStateChangeListener);
            ((UIExtensionsManager) mUiExtensionsManager).unregisterLayoutChangeListener(mLayoutChangeListener);
            ((UIExtensionsManager) mUiExtensionsManager).unregisterThemeEventListener(mThemeEventListener);
        }
        mPdfViewCtrl.unregisterDocEventListener(mDocEventListener);
        mPdfViewCtrl.unregisterDrawEventListener(mDrawEventListener);
        mDocEventListener = null;
        mDrawEventListener = null;
        return true;
    }

    public SearchView getSearchView() {
        return mSearchView;
    }

    private PDFViewCtrl.IDrawEventListener mDrawEventListener = new PDFViewCtrl.IDrawEventListener() {


        @Override
        public void onDraw(int pageIndex, Canvas canvas) {
            if (mSearchView.mIsCancel) {
                return;
            }

            if (mSearchView.mRect == null || mSearchView.mPageIndex == -1) {
                return;
            }

            if (mSearchView.mPageIndex == pageIndex) {
                if (mSearchView.mRect.size() > 0) {
                    Paint paint = new Paint();
                    paint.setColor(AppResource.getColor(mContext, R.color.ux_search_highlight_color));
                    for (int i = 0; i < mSearchView.mRect.size(); i++) {
                        RectF rectF = new RectF(mSearchView.mRect.get(i));
                        RectF deviceRect = new RectF();
                        if (mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, deviceRect, mSearchView.mPageIndex)) {
                            canvas.drawRect(deviceRect, paint);
                        }
                    }
                }
                int listCount = mSearchView.mSearchFoundRectList.size();
                if (listCount > 0) {
                    List<RectF> rectFList;
                    Paint paint = new Paint();
                    RectF rectF;
                    RectF deviceRect;
                    paint.setARGB(150, 255, 238, 162);
                    for (int i = 0; i < listCount; i++) {
                        rectFList = mSearchView.mSearchFoundRectList.get(i);
                        for (int j = 0; j < rectFList.size(); j++) {
                            rectF = new RectF(rectFList.get(j));
                            deviceRect = new RectF();
                            if (mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, deviceRect, pageIndex)) {
                                canvas.drawRect(deviceRect, paint);
                            }
                        }
                    }
                }
            }
        }
    };

    private PDFViewCtrl.IDocEventListener mDocEventListener = new DocEventListener() {
        @Override
        public void onDocWillOpen() {
        }

        @Override
        public void onDocOpened(PDFDoc pdfDoc, int i) {
            if (i == Constants.e_ErrSuccess) {
                if (mPdfViewCtrl.isDynamicXFA()) {
                    mSearchButtonItem.setEnable(false);
                } else {
                    mSearchButtonItem.setEnable(true);
                }
            }
        }

        @Override
        public void onDocWillClose(PDFDoc pdfDoc) {
        }

        @Override
        public void onDocClosed(PDFDoc pdfDoc, int i) {
            if (mSearchView.getView() == null) return;
            mSearchView.onDocumentClosed();
        }

    };

    public boolean onKeyBack() {
        if (mSearchView.getView() != null && mSearchView.getView().getVisibility() == View.VISIBLE) {
            mSearchView.onKeyBack();
            return true;

        }
        return false;
    }

    private ILayoutChangeListener mLayoutChangeListener = new ILayoutChangeListener() {
        @Override
        public void onLayoutChange(View v, int newWidth, int newHeight, int oldWidth, int oldHeight) {
            if (mSearchView.getView() == null) return;
            if (mSearchView.getView().getVisibility() == View.VISIBLE) {
                mSearchView.onLayoutChange(v, newWidth, newHeight, oldWidth, oldHeight);
            }
        }
    };

    private IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            if (mSearchView.getView() == null) return;
            mSearchView.onThemeChange(type, color);
        }
    };

    private IStateChangeListener mStateChangeListener = new IStateChangeListener() {

        @Override
        public void onStateChanged(int oldState, int newState) {
            if (mSearchView.getView() == null) return;
            if (newState == ReadStateConfig.STATE_SEARCH && oldState != ReadStateConfig.STATE_SEARCH) {
                if (mSearchView.mIsCancel) {
                    mSearchView.launchSearchView();
                    if (mSearchView.getView().getVisibility() != View.VISIBLE)
                        mSearchView.getView().setVisibility(View.VISIBLE);
                }

                UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
                SystemUiHelper.getInstance().setAllowedHideSystemUI(uiExtensionsManager.getAttachedActivity(), false);
                uiExtensionsManager.triggerDismissMenuEvent();
                IPanelManager panelManager = uiExtensionsManager.getPanelManager();
                if (panelManager != null && panelManager.isShowingPanel())
                    panelManager.hidePanel();
                uiExtensionsManager.getMainFrame().hideToolbars();
            } else if (oldState == ReadStateConfig.STATE_SEARCH && newState != ReadStateConfig.STATE_SEARCH) {
                mSearchView.cleanSearch();
                UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
                SystemUiHelper.getInstance().setAllowedHideSystemUI(uiExtensionsManager.getAttachedActivity(), true);
            }
        }
    };

}
