/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.graphics.Rect;
import android.os.Build;
import android.view.KeyEvent;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.uiextensions.IInteractionEventListener;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.controls.dialog.CircleProgressBar;
import com.foxit.uiextensions.controls.propertybar.IViewSettingsWindow;
import com.foxit.uiextensions.controls.toolbar.drag.UIDragView;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.event.PageEventListener;
import com.foxit.uiextensions.pdfreader.ILifecycleEventListener;
import com.foxit.uiextensions.pdfreader.IMainFrame;
import com.foxit.uiextensions.pdfreader.IStateChangeListener;
import com.foxit.uiextensions.pdfreader.config.ReadStateConfig;
import com.foxit.uiextensions.pdfreader.impl.LifecycleEventListener;
import com.foxit.uiextensions.pdfreader.impl.MainFrame;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

public class AutoFlipModule implements Module {
    private Context mContext;
    private ViewGroup mParent;
    private PDFViewCtrl mPdfViewCtrl;
    private PDFViewCtrl.UIExtensionsManager mUiExtensionsManager;

    private View mTopBarContentView;
    private View mBottomBarContentView;
    private CircleProgressBar mProgressBar;

    private IViewSettingsWindow mSettingWindow;

    private int mAutoFlipTime = 0;
    private int mStartTime;
    private boolean mIsRunThread;
    private ImageView mIvAdd, mIvReduce;
    private TextView mTvBack;
    private ImageView mIvQuit;

    public AutoFlipModule(Context context, ViewGroup parent, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        mContext = context;
        mParent = parent;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = uiExtensionsManager;
    }

    @Override
    public String getName() {
        return MODULE_NAME_AUTOFLIP;
    }

    @Override
    public boolean loadModule() {
        ((UIExtensionsManager) mUiExtensionsManager).registerModule(this);
        ((UIExtensionsManager) mUiExtensionsManager).registerStateChangeListener(mStateChangeListener);
        ((UIExtensionsManager) mUiExtensionsManager).registerInteractionListener(mInteractionListener);
        ((UIExtensionsManager) mUiExtensionsManager).registerThemeEventListener(mThemeEventListener);
        if (AppDisplay.isPad())
            ((UIExtensionsManager) mUiExtensionsManager).registerLifecycleListener(mLifecycleEventListener);

        mPdfViewCtrl.registerDocEventListener(mDocEventListener);
        mPdfViewCtrl.registerPageEventListener(mPageEventListener);
        return true;
    }

    @Override
    public boolean unloadModule() {
        ((UIExtensionsManager) mUiExtensionsManager).unregisterStateChangeListener(mStateChangeListener);
        ((UIExtensionsManager) mUiExtensionsManager).unregisterInteractionListener(mInteractionListener);
        ((UIExtensionsManager) mUiExtensionsManager).unregisterThemeEventListener(mThemeEventListener);
        if (AppDisplay.isPad())
            ((UIExtensionsManager) mUiExtensionsManager).unregisterLifecycleListener(mLifecycleEventListener);

        mPdfViewCtrl.unregisterDocEventListener(mDocEventListener);
        mPdfViewCtrl.unregisterPageEventListener(mPageEventListener);
        mSettingWindow = null;
        mParent = null;
        return true;
    }

    private void initToolBar() {
        if (!AppDisplay.isPad()) {
            //top bar
            mTopBarContentView = View.inflate(mContext, R.layout.auto_flip_topbar_layout, null);
            mTopBarContentView.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
            mTvBack = mTopBarContentView.findViewById(R.id.auto_flip_close);

            int color = ThemeConfig.getInstance(mContext).getPrimaryColor();
            mTvBack.setTextColor(color);
            mTvBack.setOnClickListener(mOnClickListener);

            mTopBarContentView.setOnTouchListener(new View.OnTouchListener() {
                @SuppressLint("ClickableViewAccessibility")
                @Override
                public boolean onTouch(View v, MotionEvent event) {
                    return true;
                }
            });
        }

        //bottom bar
        mBottomBarContentView = View.inflate(mContext, R.layout.auto_flip_bottom_bar, null);
        mBottomBarContentView.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
        if (!AppDisplay.isPad()){
            mBottomBarContentView.setOnTouchListener(new View.OnTouchListener() {
                @SuppressLint("ClickableViewAccessibility")
                @Override
                public boolean onTouch(View v, MotionEvent event) {
                    return true;
                }
            });
        }
        mIvAdd = mBottomBarContentView.findViewById(R.id.ml_iv_autoflip_add);
        mIvReduce = mBottomBarContentView.findViewById(R.id.ml_iv_autoflip_reduce);
        mIvQuit = mBottomBarContentView.findViewById(R.id.iv_auto_flip_quit);
        mProgressBar = mBottomBarContentView.findViewById(R.id.auto_flip_progressBar);
        mProgressBar.setTextColor(AppResource.getColor(mContext, R.color.t4));
        mProgressBar.setSecondaryColor(ThemeConfig.getInstance(mContext).getPrimaryColor());
        ThemeUtil.setTintList(mIvAdd, ThemeUtil.getPrimaryIconColor(mContext));
        ThemeUtil.setTintList(mIvReduce, ThemeUtil.getPrimaryIconColor(mContext));
        ThemeUtil.setTintList(mIvQuit, ThemeUtil.getPrimaryIconColor(mContext));
        mIvAdd.setOnClickListener(mOnClickListener);
        mIvReduce.setOnClickListener(mOnClickListener);
        mIvQuit.setOnClickListener(mOnClickListener);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            mIvAdd.setForceDarkAllowed(false);
            mIvReduce.setForceDarkAllowed(false);
            mProgressBar.setForceDarkAllowed(false);
            mIvQuit.setForceDarkAllowed(false);
        }
        if (!AppDisplay.isPad()) {
            RelativeLayout.LayoutParams ttsTopLp = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
            ttsTopLp.addRule(RelativeLayout.ALIGN_PARENT_TOP);
            mParent.addView(mTopBarContentView, ttsTopLp);

            RelativeLayout.LayoutParams ttsBottomLp = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
            ttsBottomLp.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
            mParent.addView(mBottomBarContentView, ttsBottomLp);

            mTopBarContentView.setVisibility(View.INVISIBLE);
            mBottomBarContentView.setVisibility(View.INVISIBLE);
        } else {
            mIvQuit.setVisibility(View.VISIBLE);
        }
    }

    private View.OnClickListener mOnClickListener = new View.OnClickListener() {
        @Override
        public void onClick(View v) {
            int id = v.getId();
            if (id == R.id.auto_flip_close || id == R.id.iv_auto_flip_quit) {
                if (mLastReadState == ReadStateConfig.STATE_FILLSIGN)
                    ((UIExtensionsManager) mUiExtensionsManager).changeState(ReadStateConfig.STATE_FILLSIGN);
                else
                    ((UIExtensionsManager) mUiExtensionsManager).changeState(ReadStateConfig.STATE_NORMAL);
            } else if (id == R.id.ml_iv_autoflip_add) {
                if (mAutoFlipTime <= 100000 && mAutoFlipTime > 20000) {
                    float i = ((mAutoFlipTime - 10000) * 1.0f) / (mAutoFlipTime * 1.0f);
                    mStartTime = (int) (mStartTime * i);
                    mAutoFlipTime = mAutoFlipTime - 10000;
                    if (mAutoFlipTime == 20000) {
                        mIvAdd.setEnabled(false);
                    }
                    mIvReduce.setEnabled(true);
                    mProgressBar.setMax(mAutoFlipTime);
                    mProgressBar.setProgress(mStartTime);
                }
            } else if (id == R.id.ml_iv_autoflip_reduce) {
                if (mAutoFlipTime < 100000 && mAutoFlipTime >= 20000) {
                    float i = ((mAutoFlipTime + 10000) * 1.0f) / (mAutoFlipTime * 1.0f);
                    mStartTime = (int) (mStartTime * i);
                    mAutoFlipTime = mAutoFlipTime + 10000;
                    if (mAutoFlipTime == 100000) {
                        mIvReduce.setEnabled(false);
                    }
                    mIvAdd.setEnabled(true);
                    mProgressBar.setMax(mAutoFlipTime);
                    mProgressBar.setProgress(mStartTime);
                }
            }
        }
    };

    private boolean mIsAutoFlip = false;
    private int mLastReadState;
    private IViewSettingsWindow.IValueChangeListener mValueChangeListener = new IViewSettingsWindow.IValueChangeListener() {
        @Override
        public void onValueChanged(int type, Object value) {
            if (!(value instanceof Boolean)) return;
            if (type == IViewSettingsWindow.TYPE_AUTO_FLIP) {
                if ((boolean) value) {
                    mIsAutoFlip = true;
                    mLastReadState = ((UIExtensionsManager) mUiExtensionsManager).getState();
                    ((UIExtensionsManager) mUiExtensionsManager).changeState(ReadStateConfig.STATE_AUTOFLIP);
                    if (!((UIExtensionsManager) mUiExtensionsManager).getMainFrame().isToolbarsVisible())
                        ((UIExtensionsManager) mUiExtensionsManager).getMainFrame().showToolbars();

                    ((UIExtensionsManager) mUiExtensionsManager).getMainFrame().hideSettingWindow();

                    if (mAutoFlipTime == 0) {
                        mAutoFlipTime = 60000;
                        mProgressBar.setMax(mAutoFlipTime);
                        mIvAdd.setEnabled(true);
                        mIvReduce.setEnabled(true);
                    }
                    setProgressBar();
                } else {
                    if (mLastReadState == ReadStateConfig.STATE_FILLSIGN)
                        ((UIExtensionsManager) mUiExtensionsManager).changeState(ReadStateConfig.STATE_FILLSIGN);
                    else
                        ((UIExtensionsManager) mUiExtensionsManager).changeState(ReadStateConfig.STATE_NORMAL);
                }
            }
        }

        @Override
        public int getType() {
            return IViewSettingsWindow.TYPE_AUTO_FLIP;
        }
    };

    private UIDragView mDragView;
    private IStateChangeListener mStateChangeListener = new IStateChangeListener() {
        @Override
        public void onStateChanged(int oldState, int newState) {
            if (mPdfViewCtrl.getDoc() == null) return;
            if (mBottomBarContentView == null)
                initToolBar();
            MainFrame mainFrame = (MainFrame) ((UIExtensionsManager) mUiExtensionsManager).getMainFrame();
            if (((UIExtensionsManager) mUiExtensionsManager).getState() == ReadStateConfig.STATE_AUTOFLIP) {
                boolean isToolbarsVisible = mainFrame.isToolbarsVisible();
                if (AppDisplay.isPad()) {
                    int width = Math.min(AppDisplay.getScreenWidth(), AppDisplay.getScreenHeight()) / 3;
                    if (mDragView == null)
                        mDragView = new UIDragView(mBottomBarContentView, (UIExtensionsManager) mUiExtensionsManager);
                    mDragView.setVisible(!mainFrame.isShowFullScreenUI());
                } else {
                    mTopBarContentView.setVisibility(isToolbarsVisible ? View.VISIBLE : View.INVISIBLE);
                    mBottomBarContentView.setVisibility(mTopBarContentView.getVisibility());
                    mPdfViewCtrl.offsetScrollBoundary(0, isToolbarsVisible ? mTopBarContentView.getHeight() : 0,
                            0,  isToolbarsVisible ? -mBottomBarContentView.getHeight() : 0);
                    mPdfViewCtrl.postPageContainer();
                }

                if (!mIsAutoFlip) {
                    mIsAutoFlip = true;
                    if (mainFrame.getSettingWindow() != null) {
                        mainFrame.getSettingWindow().setProperty(IViewSettingsWindow.TYPE_AUTO_FLIP, true);
                    }
                    if (!mainFrame.isToolbarsVisible())
                        mainFrame.showToolbars();
                    if (mAutoFlipTime == 0) {
                        mAutoFlipTime = 60000;
                        mProgressBar.setMax(mAutoFlipTime);
                        mIvAdd.setEnabled(true);
                        mIvReduce.setEnabled(true);
                    }
                    setProgressBar();
                }
            } else {
                if (oldState == ReadStateConfig.STATE_AUTOFLIP) {
                    if (AppDisplay.isPad()) {
                        if (mDragView != null) {
                            mDragView.removeDragView();
                            mDragView = null;
                        }
                    } else {
                        mTopBarContentView.setVisibility(View.INVISIBLE);
                        mBottomBarContentView.setVisibility(View.INVISIBLE);
                    }

                    mProgressBar.setProgress(0);
                    mIsRunThread = false;
                    mIsAutoFlip = false;
                    if (mainFrame.getSettingWindow() != null) {
                        mainFrame.getSettingWindow().setProperty(IViewSettingsWindow.TYPE_AUTO_FLIP, false);
                    }
                }
            }
        }
    };

    private PDFViewCtrl.IDocEventListener mDocEventListener = new DocEventListener() {

        @Override
        public void onDocOpened(PDFDoc document, int errCode) {
            if (errCode != Constants.e_ErrSuccess) return;
            IMainFrame mainFrame = ((UIExtensionsManager) mUiExtensionsManager).getMainFrame();
            if (mainFrame == null) return;
            mSettingWindow = mainFrame.getSettingWindow();
            mSettingWindow.registerListener(mValueChangeListener);

            if (mUiExtensionsManager != null && ((UIExtensionsManager) mUiExtensionsManager).getState() == ReadStateConfig.STATE_AUTOFLIP) {
                ((UIExtensionsManager) mUiExtensionsManager).changeState(ReadStateConfig.STATE_NORMAL);
            }
        }

        @Override
        public void onDocClosed(PDFDoc document, int errCode) {
            if (mSettingWindow != null)
                mSettingWindow.unRegisterListener(mValueChangeListener);
        }

    };

    private final PDFViewCtrl.IPageEventListener mPageEventListener = new PageEventListener() {

        @Override
        public void onPageChanged(int oldPageIndex, int curPageIndex) {
            if (oldPageIndex != curPageIndex) {
                mStartTime = 0;
            }
        }

    };

    private void setProgressBar() {
        mIsRunThread = true;
        Thread mThread = new Thread(new Runnable() {
            @Override
            public void run() {
                while (mIsRunThread) {
                    for (mStartTime = 0; mStartTime < mAutoFlipTime; mStartTime++) {
                        if (mIsRunThread) {
                            if (mPdfViewCtrl.getDoc() == null) {
                                mProgressBar.setProgress(0);
                            } else if (mPdfViewCtrl.isContinuous()
                                    && (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_FACING
                                    || mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_COVER)) {

                                if (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_COVER && mPdfViewCtrl.getCurrentPage() == 0) {
                                    if (mPdfViewCtrl.getPageCount() >= 4) {
                                        mProgressBar.setProgress(mStartTime);
                                    } else if (mPdfViewCtrl.getPageCount() >= 2) {
                                        Rect nextLeftPageRect = mPdfViewCtrl.getPageViewRect(mPdfViewCtrl.getCurrentPage() + 1);
                                        Rect nextRightPageRect = mPdfViewCtrl.getPageViewRect(mPdfViewCtrl.getCurrentPage() + 2);
                                        int height = Math.max(nextLeftPageRect.height(), nextRightPageRect.height());
                                        if (height > mParent.getHeight() / 2) {
                                            mProgressBar.setProgress(mStartTime);
                                        } else {
                                            mProgressBar.setProgress(0);
                                        }
                                    } else {
                                        mProgressBar.setProgress(0);
                                    }
                                } else {
                                    if (mPdfViewCtrl.getCurrentPage() + 4 >= mPdfViewCtrl.getPageCount()) {
                                        Rect nextLeftPageRect = mPdfViewCtrl.getPageViewRect(mPdfViewCtrl.getCurrentPage() + 2);
                                        Rect nextRightPageRect = mPdfViewCtrl.getPageViewRect(mPdfViewCtrl.getCurrentPage() + 3);
                                        int height = Math.max(nextLeftPageRect.height(), nextRightPageRect.height());
                                        if (height > mParent.getHeight() / 2) {
                                            mProgressBar.setProgress(mStartTime);
                                        } else {
                                            mProgressBar.setProgress(0);
                                        }
                                    } else {
                                        mProgressBar.setProgress(mStartTime);
                                    }
                                }
                            } else if (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_FACING
                                    || (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_COVER
                                    && mPdfViewCtrl.getPageCount() > 2)) {
                                if (mPdfViewCtrl.getCurrentPage() + 2 >= mPdfViewCtrl.getPageCount()) {
                                    mProgressBar.setProgress(0);
                                } else {
                                    mProgressBar.setProgress(mStartTime);
                                }
                            } else {
                                if (mPdfViewCtrl.getCurrentPage() + 1 == mPdfViewCtrl.getPageCount()) {
                                    mProgressBar.setProgress(0);
                                } else {
                                    mProgressBar.setProgress(mStartTime);
                                }
                            }

                            try {
                                Thread.sleep(1);
                            } catch (InterruptedException e) {
                                e.printStackTrace();
                            }
                        } else {
                            break;
                        }
                    }
                    AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
                        @Override
                        public void run() {
                            if (mIsRunThread) {
                                mPdfViewCtrl.gotoNextPage();
                            }
                        }
                    });
                }
            }
        });
        mThread.start();
    }

    private IInteractionEventListener mInteractionListener = new IInteractionEventListener() {
        @Override
        public boolean onKeyDown(Activity act, int keyCode, KeyEvent event) {
            if (keyCode == KeyEvent.KEYCODE_BACK && event.getRepeatCount() == 0 && mIsAutoFlip) {
                mIsAutoFlip = false;
                if (mLastReadState == ReadStateConfig.STATE_FILLSIGN)
                    ((UIExtensionsManager) mUiExtensionsManager).changeState(ReadStateConfig.STATE_FILLSIGN);
                else
                    ((UIExtensionsManager) mUiExtensionsManager).changeState(ReadStateConfig.STATE_NORMAL);
                return true;
            }
            return false;
        }
    };

    private ILifecycleEventListener mLifecycleEventListener = new LifecycleEventListener() {
        @Override
        public void onHiddenChanged(boolean hidden) {
            if (hidden) {
                UIExtensionsManager extensionsManager = (UIExtensionsManager) mUiExtensionsManager;
                mInteractionListener.onKeyDown(extensionsManager.getAttachedActivity(),
                        KeyEvent.KEYCODE_BACK, new KeyEvent(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_BACK));
            }
        }
    };

    private IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            if (mTopBarContentView == null) return;
            if (!AppDisplay.isPad())
                mParent.removeView(mTopBarContentView);
            mParent.removeView(mBottomBarContentView);

            boolean addItemEnabled = mIvAdd.isEnabled();
            boolean reduceItemEnabled = mIvReduce.isEnabled();
            int progress = mProgressBar.getProgress();
            int max = mProgressBar.getMax();
            initToolBar();

            mProgressBar.setMax(max);
            mProgressBar.setProgress(progress);
            mIvAdd.setEnabled(addItemEnabled);
            mIvReduce.setEnabled(reduceItemEnabled);
            mProgressBar.setPrimaryColor(AppResource.getColor(mContext, R.color.i3));
            mProgressBar.setTextColor(AppResource.getColor(mContext, R.color.t4));
            mProgressBar.setSecondaryColor(ThemeConfig.getInstance(mContext).getPrimaryColor());
            ThemeUtil.setTintList(mIvAdd, ThemeUtil.getPrimaryIconColor(mContext));
            ThemeUtil.setTintList(mIvReduce, ThemeUtil.getPrimaryIconColor(mContext));
            ThemeUtil.setTintList(mIvQuit, ThemeUtil.getPrimaryIconColor(mContext));
            if (((UIExtensionsManager) mUiExtensionsManager).getState() == ReadStateConfig.STATE_AUTOFLIP) {
                if (AppDisplay.isPad()) {
                    if (mDragView != null) {
                        mDragView.removeDragView();
                        mDragView = null;
                    }
                    mDragView = new UIDragView(mBottomBarContentView, (UIExtensionsManager) mUiExtensionsManager);
                } else {
                    mTopBarContentView.setVisibility(View.VISIBLE);
                    mBottomBarContentView.setVisibility(View.VISIBLE);
                }
            } else {
                if (AppDisplay.isPad()) {
                    if (mDragView != null) {
                        mDragView.removeDragView();
                        mDragView = null;
                    }
                } else {
                    mTopBarContentView.setVisibility(View.INVISIBLE);
                    mBottomBarContentView.setVisibility(View.INVISIBLE);
                }
            }
        }
    };

    public void setDragToolbarVisible(boolean visible) {
        if (mDragView != null)
            mDragView.setVisible(visible);
    }

}
