/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.view;

import android.content.Context;
import android.content.res.TypedArray;
import android.os.Build;
import android.util.AttributeSet;
import android.view.View;
import android.widget.RelativeLayout;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.view.ViewCompat;
import androidx.core.view.WindowInsetsCompat;

import com.foxit.uiextensions.R;

public class E2ERelativeLayout extends RelativeLayout {

    private View topImmersiveView;

    public E2ERelativeLayout(@NonNull Context context) {
        super(context);
        init(null);
    }

    public E2ERelativeLayout(@NonNull Context context, @Nullable AttributeSet attrs) {
        super(context, attrs);
        init(attrs);
    }

    public E2ERelativeLayout(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init(attrs);
    }

    private void init(AttributeSet attrs) {
        topImmersiveView = new View(getContext());
        topImmersiveView.setId(View.generateViewId());
        RelativeLayout.LayoutParams topParams = new RelativeLayout.LayoutParams(LayoutParams.MATCH_PARENT, 0);
        topParams.addRule(RelativeLayout.ALIGN_PARENT_TOP);
        topImmersiveView.setLayoutParams(topParams);
        addView(topImmersiveView);

        if (attrs != null) {
            TypedArray a = getContext().obtainStyledAttributes(attrs, R.styleable.ImmersiveLayout);
            int topImmersiveColor = a.getColor(R.styleable.ImmersiveLayout_topImmersiveColor,
                    getResources().getColor(R.color.ui_color_top_bar_main));
            topImmersiveView.setBackgroundColor(topImmersiveColor);
            a.recycle();
        } else {
            topImmersiveView.setBackgroundColor(getResources().getColor(R.color.ui_color_top_bar_main));
        }

        ViewCompat.setOnApplyWindowInsetsListener(this, new androidx.core.view.OnApplyWindowInsetsListener() {
            @Override
            public WindowInsetsCompat onApplyWindowInsets(View v, WindowInsetsCompat windowInsets) {
                androidx.core.graphics.Insets insets;
                if (Build.VERSION.SDK_INT >= 30) {
                    insets = windowInsets.getInsets(WindowInsetsCompat.Type.systemBars() | WindowInsetsCompat.Type.displayCutout());
                } else {
                    insets = windowInsets.getInsets(WindowInsetsCompat.Type.displayCutout());
                }
                topImmersiveView.getLayoutParams().height = insets.top;

                E2ERelativeLayout.this.setPadding(insets.left, 0, insets.right, insets.bottom);
                return WindowInsetsCompat.CONSUMED;
            }
        });
    }

    public void setTopImmersiveColor(int color) {
        if (topImmersiveView != null) {
            topImmersiveView.setBackgroundColor(color);
        }
    }

    @Override
    protected void onFinishInflate() {
        super.onFinishInflate();

        int childCount = getChildCount();
        for (int i = 0; i < childCount; i++) {
            View child = getChildAt(i);
            if (child.getId() != topImmersiveView.getId()) {
                bindContentView(child);
                break;
            }
        }
    }

    private void bindContentView(View contentView) {
        RelativeLayout.LayoutParams params = (LayoutParams) contentView.getLayoutParams();
        params.addRule(RelativeLayout.BELOW, topImmersiveView.getId());
        contentView.setLayoutParams(params);
    }

}

