/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.toolbar.impl;

import android.content.Context;
import android.graphics.Point;
import android.graphics.PorterDuff;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewParent;
import android.widget.FrameLayout;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.controls.toolbar.IBaseItem;
import com.foxit.uiextensions.theme.ColorPair;
import com.foxit.uiextensions.theme.DynamicColorProvider;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.UIToast;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;

import androidx.annotation.ColorInt;

public class BaseBarImpl implements BaseBar {
    protected ArrayList<IBaseItem> mLT_Items;
    protected ArrayList<IBaseItem> mCenter_Items;
    protected ArrayList<IBaseItem> mRB_Items;

    protected BarFrameLayoutImpl mRootLayout;
    protected LinearLayout mLTLayout;
    protected LinearLayout mCenterLayout;
    protected LinearLayout mRBLayout;

    protected LinearLayout.LayoutParams mStartItemLayoutParams;
    protected LinearLayout.LayoutParams mCenterItemLayoutParams;
    protected LinearLayout.LayoutParams mEndItemLayoutParams;
    protected LinearLayout.LayoutParams mLastItemLayoutParams;

    private boolean mIsRefreshLayout = false;
    protected View mContentView;

    protected boolean mUniformSpace;
    protected boolean mAutoCompressItemsInterval;
    protected boolean mMiddleButtonCenter = true;

    protected ComparatorTBItemByIndex mItemComparator;
    protected Context mContext = null;

    protected int mBackgroundColor;
    protected int mWidth;
    protected int mHeight;
    protected int mOrientation;
    protected int mInterval;
    protected Integer mStartInterval = null;
    protected Integer mCenterInterval = null;
    protected Integer mEndInterval = null;

    protected int mStartMargin;
    protected int mEndMargin;

    public BaseBarImpl(Context context) {
        this(context, HORIZONTAL, 0, 0, false);
    }

    protected BaseBarImpl(Context context, int orientation) {
        this(context, orientation, 0, 0, false);
    }

    protected BaseBarImpl(Context context, int orientation, boolean uniformSpace) {
        this(context, orientation, 0, 0, uniformSpace);
    }

    /**
     * if this is an interval bar,length and wide must use px
     *
     * @param orientation BaseBarImpl.HORIZONTAL or BaseBarImpl.VERTICAL<br/>
     * @param width       The default wide (level), or high (vertical).(<code>ViewGroup.LayoutParams.MATCH_PARENT<code/>,<code>ViewGroup.LayoutParams.WRAP_CONTENT<code/> or dp).
     * @param height      The default high (level), or wide (vertical).(<code>ViewGroup.LayoutParams.MATCH_PARENT<code/>,<code>ViewGroup.LayoutParams.WRAP_CONTENT<code/> or dp).
     */
    protected BaseBarImpl(Context context, int orientation, int width, int height, boolean uniformSpace) {
        mContext = context;

        mInterval = 0;
        mStartMargin = mEndMargin = AppResource.getDimensionPixelSize(context, R.dimen.ux_margin_4dp);
        mOrientation = orientation;
        mUniformSpace = uniformSpace;

        if (height != 0) {
            mHeight = height;
        } else {
            if (orientation == HORIZONTAL) {
                mHeight = AppResource.getDimensionPixelSize(context, R.dimen.ux_bottombar_height);
            } else {
                mHeight = ViewGroup.LayoutParams.WRAP_CONTENT;
            }
        }

        if (width != 0) {
            mWidth = width;
        } else {
            if (orientation == VERTICAL) {
                mWidth = AppResource.getDimensionPixelSize(context, R.dimen.ux_bottombar_height);
            } else {
                mWidth = ViewGroup.LayoutParams.MATCH_PARENT;
            }
        }

        mRootLayout = new BarFrameLayoutImpl(context, this);
        mBackgroundColor = ThemeConfig.getInstance(context).getB2();
        mRootLayout.setBackgroundColor(mBackgroundColor);
        if (mOrientation == HORIZONTAL) {
            mRootLayout.setPadding(mStartMargin, 0, mEndMargin, 0);
        } else {
            mRootLayout.setPadding(0, mStartMargin, 0, mEndMargin);
        }

        mLTLayout = new LinearLayout(context);
        mLTLayout.setId(R.id.toolbar_lt_layout_id);

        mCenterLayout = new LinearLayout(context);
        mCenterLayout.setId(R.id.toolbar_center_layout_id);

        mRBLayout = new LinearLayout(context);
        mRBLayout.setId(R.id.toolbar_rb_layout_id);

        mLTLayout.setVisibility(View.GONE);
        mCenterLayout.setVisibility(View.GONE);
        mRBLayout.setVisibility(View.GONE);

        mRootLayout.addView(mLTLayout);
        mRootLayout.addView(mCenterLayout);
        mRootLayout.addView(mRBLayout);

        mLT_Items = new ArrayList<IBaseItem>();
        mCenter_Items = new ArrayList<IBaseItem>();
        mRB_Items = new ArrayList<IBaseItem>();

        mIsRefreshLayout = true;
        mItemComparator = new ComparatorTBItemByIndex();

        initOrientation(orientation);
    }

    @Override
    public boolean addView(IBaseItem item, TB_Position position) {
        try {
            return addView(item, position, IBaseItem.SortType.Sort_By_Tag, -1);
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }

    @Override
    public boolean addView(IBaseItem item, TB_Position position, int index) {
        try {
            return addView(item, position, IBaseItem.SortType.Sort_By_Index, index);
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }

    private boolean addView(IBaseItem item, TB_Position position, IBaseItem.SortType sortType, int index) {
        try {
            if (item == null) {
                return false;
            }
            if (mContentView != null) {
                mRootLayout.removeView(mContentView);
                mContentView = null;
                resetMargin();
            }

            if (mUniformSpace) {
                if (mCenter_Items.contains(item)) {
                    return false;
                }
                sortItems(item, mCenter_Items);
                resetItemsLayout(mCenter_Items, mCenterLayout, mCenterItemLayoutParams);
            } else {
                if (TB_Position.Position_LT.equals(position)) {
                    if (mLT_Items.contains(item)) {
                        return false;
                    }

                    if (IBaseItem.SortType.Sort_By_Tag.equals(sortType)) {
                        sortItems(item, mLT_Items);
                    } else {
                        mLT_Items.add(index, item);
                    }
                    resetItemsLayout(mLT_Items, mLTLayout, mStartItemLayoutParams);
                } else if (TB_Position.Position_CENTER.equals(position)) {
                    if (mCenter_Items.contains(item)) {
                        return false;
                    }
                    if (IBaseItem.SortType.Sort_By_Tag.equals(sortType)) {
                        sortItems(item, mCenter_Items);
                    } else {
                        mCenter_Items.add(index, item);
                    }
                    resetItemsLayout(mCenter_Items, mCenterLayout, mCenterItemLayoutParams);
                } else if (TB_Position.Position_RB.equals(position)) {
                    if (mRB_Items.contains(item)) {
                        return false;
                    }

                    if (IBaseItem.SortType.Sort_By_Tag.equals(sortType)) {
                        sortItems(item, mRB_Items);
                    } else {
                        mRB_Items.add(index, item);
                    }
                    resetItemsLayout(mRB_Items, mRBLayout, mEndItemLayoutParams);
                }
            }

            if (autoCompressItemsInterval(mCenter_Items, mCenterItemLayoutParams)) {
                resetItemsLayout(mCenter_Items, mCenterLayout, mCenterItemLayoutParams);
            }
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }

    public boolean autoCompressCenterItems() {
        if (autoCompressItemsInterval(mCenter_Items, mCenterItemLayoutParams)) {
            resetItemsLayout(mCenter_Items, mCenterLayout, mCenterItemLayoutParams);
            return true;
        }
        return false;
    }

    private boolean autoCompressItemsInterval(ArrayList<IBaseItem> items, LinearLayout.LayoutParams lp) {
        if (!mAutoCompressItemsInterval || items.size() == 0) {
            return false;
        }

        if (mOrientation == HORIZONTAL) {
            int maxH = 0;
            int maxW = 0;
            int itemTotalWidth = 0;

            for (IBaseItem item : items) {
                item.getContentView().measure(0, 0);
                if (item.getContentView().getMeasuredHeight() > maxH) {
                    maxH = item.getContentView().getMeasuredHeight();
                }
                int measureWidth = item.getContentView().getMeasuredWidth();
                if (measureWidth > maxW) {
                    maxW = measureWidth;
                }
                itemTotalWidth += item.getContentView().getMeasuredWidth();
            }

            int itemSize = items.size();
            if (itemSize > 1) {
                //int totalWidth = maxW * itemSize + mInterval * (itemSize - 1);
                int totalWidth = itemTotalWidth + mInterval * (itemSize - 1);
                int screenWidth = Math.min(AppDisplay.getActivityWidth(), AppDisplay.getActivityHeight());
                int barContentSpace = screenWidth - mRootLayout.getPaddingLeft() - mRootLayout.getPaddingRight();
                if (totalWidth > barContentSpace) {
                    //mInterval = (mInterval * (itemSize - 1) - (totalWidth - barContentSpace)) / (itemSize - 1);
                    mInterval = (barContentSpace - itemTotalWidth) / (itemSize - 1);
                    if (mOrientation == HORIZONTAL) {
                        lp.setMargins(0, 0, mInterval, 0);
                    } else {
                        lp.setMargins(0, 0, 0, mInterval);
                    }
                    return true;
                }
            }
        }
        return false;
    }

    private void sortItems(IBaseItem item, ArrayList<IBaseItem> items) {
        items.add(item);
        Collections.sort(items, mItemComparator);
    }

    private void resetItemsLayout(ArrayList<IBaseItem> items, final LinearLayout layout, LinearLayout.LayoutParams itemParams) {
        resetItemsLayout(items, layout, itemParams, mLastItemLayoutParams);
    }

    private void resetItemsLayout(ArrayList<IBaseItem> items, final LinearLayout layout, LinearLayout.LayoutParams itemParams, LinearLayout.LayoutParams endParams) {
        resetItemsLayout(items, layout, itemParams, itemParams, endParams);
    }

    private void resetItemsLayout(ArrayList<IBaseItem> items, final LinearLayout layout,
                                  LinearLayout.LayoutParams startParams,
                                  LinearLayout.LayoutParams itemParams,
                                  LinearLayout.LayoutParams endParams) {
        if (items == null || items.isEmpty() || layout == null) {
            return;
        }
        layout.removeAllViews();

        for (int i = 0; i < items.size(); i++) {
            AppUtil.removeViewFromParent(items.get(i).getContentView());
            if (i == 0) {
                layout.addView(items.get(i).getContentView(), startParams);
                continue;
            }
            if (i == items.size() - 1) {
                layout.addView(items.get(i).getContentView(), endParams);
                continue;
            }
            layout.addView(items.get(i).getContentView(), itemParams);
        }
        if (layout.getVisibility() != View.VISIBLE)
            layout.setVisibility(View.VISIBLE);
    }

    @Override
    public boolean removeItemByTag(int tag) {
        try {
            if (!mCenter_Items.isEmpty()) {
                for (IBaseItem item : mCenter_Items) {
                    if (item.getTag() == tag) {
                        mCenterLayout.removeView(item.getContentView());
                        boolean isRemoveSuccess = mCenter_Items.remove(item);
                        if (isRemoveSuccess) {
                            resetLayout(TB_Position.Position_CENTER);
                        }
                        return isRemoveSuccess;
                    }
                }
            }

            if (!mLT_Items.isEmpty()) {
                for (IBaseItem item : mLT_Items) {
                    if (item.getTag() == tag) {
                        mLTLayout.removeView(item.getContentView());
                        boolean isRemoveSuccess = mLT_Items.remove(item);
                        if (isRemoveSuccess) {
                            resetLayout(TB_Position.Position_LT);
                        }
                        return isRemoveSuccess;
                    }
                }
            }
            if (!mRB_Items.isEmpty()) {
                for (IBaseItem item : mRB_Items) {
                    if (item.getTag() == tag) {
                        mRBLayout.removeView(item.getContentView());
                        boolean isRemoveSuccess = mRB_Items.remove(item);
                        if (isRemoveSuccess) {
                            resetLayout(TB_Position.Position_RB);
                        }
                        return isRemoveSuccess;
                    }
                }
            }
            return false;
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }

    @Override
    public boolean removeItemByIndex(TB_Position position, int index) {
        try {
            if (TB_Position.Position_CENTER.equals(position)) {
                IBaseItem item = mCenter_Items.get(index);
                mCenterLayout.removeView(item.getContentView());
                boolean isRemoveSuccess = mCenter_Items.remove(item);
                if (isRemoveSuccess) {
                    resetLayout(TB_Position.Position_CENTER);
                }
                return isRemoveSuccess;
            }

            if (TB_Position.Position_LT.equals(position)) {
                IBaseItem item = mLT_Items.get(index);
                mLTLayout.removeView(item.getContentView());
                boolean isRemoveSuccess = mLT_Items.remove(item);
                if (isRemoveSuccess) {
                    resetLayout(TB_Position.Position_LT);
                }
                return isRemoveSuccess;
            }

            if (TB_Position.Position_RB.equals(position)) {
                IBaseItem item = mRB_Items.get(index);
                mRBLayout.removeView(item.getContentView());
                boolean isRemoveSuccess = mRB_Items.remove(item);
                if (isRemoveSuccess) {
                    resetLayout(TB_Position.Position_RB);
                }
                return isRemoveSuccess;
            }
            return false;
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }

    @Override
    public boolean removeItem(IBaseItem item) {
        try {
            if (mCenter_Items.contains(item)) {
                mCenterLayout.removeView(item.getContentView());
                boolean isRemoveSuccess = mCenter_Items.remove(item);
                if (isRemoveSuccess) {
                    resetLayout(TB_Position.Position_CENTER);
                }
                return isRemoveSuccess;
            }

            if (mLT_Items.contains(item)) {
                mLTLayout.removeView(item.getContentView());
                boolean isRemoveSuccess = mLT_Items.remove(item);
                if (isRemoveSuccess) {
                    resetLayout(TB_Position.Position_LT);
                }
                return isRemoveSuccess;
            }

            if (mRB_Items.contains(item)) {
                mRBLayout.removeView(item.getContentView());
                boolean isRemoveSuccess = mRB_Items.remove(item);
                if (isRemoveSuccess) {
                    resetLayout(TB_Position.Position_RB);
                }
                return isRemoveSuccess;
            }
            return false;
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }

    protected void resetLayout(TB_Position position) {
        if (TB_Position.Position_LT.equals(position)) {
            resetItemsLayout(mLT_Items, mLTLayout, mStartItemLayoutParams);
        } else if (TB_Position.Position_CENTER.equals(position)) {
            resetItemsLayout(mCenter_Items, mCenterLayout, mCenterItemLayoutParams);
        } else {
            resetItemsLayout(mRB_Items, mRBLayout, mEndItemLayoutParams);
        }
    }

    @Override
    public void removeAllItems() {
        mLTLayout.removeAllViews();
        mCenterLayout.removeAllViews();
        mRBLayout.removeAllViews();

        mLT_Items.clear();
        mCenter_Items.clear();
        mRB_Items.clear();

        if (mContentView != null) {
            mRootLayout.removeView(mContentView);
            mContentView = null;
        }
    }

    @Override
    public void removeItems(TB_Position position) {
        try {
            if (TB_Position.Position_LT == position) {
                mLTLayout.removeAllViews();
                mLT_Items.clear();
            } else if (TB_Position.Position_CENTER == position) {
                mCenterLayout.removeAllViews();
                mCenter_Items.clear();
            } else if (TB_Position.Position_RB == position) {
                mRBLayout.removeAllViews();
                mRB_Items.clear();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void setVisible(boolean visible) {
        View rootView = getContentView();
        if (rootView == null)
            return;
        if (visible) {
            rootView.setVisibility(View.VISIBLE);
        } else {
            rootView.setVisibility(View.INVISIBLE);
        }
    }

    @Override
    public View getContentView() {
        return mRootLayout;
    }

    @Override
    public IBaseItem getItem(TB_Position location, int tag) {
        if (TB_Position.Position_LT.equals(location)) {
            return getItem(mLT_Items, tag);
        } else if (TB_Position.Position_CENTER.equals(location)) {
            return getItem(mCenter_Items, tag);
        } else if (TB_Position.Position_RB.equals(location)) {
            return getItem(mRB_Items, tag);
        }
        return null;
    }

    @Override
    public IBaseItem getItemByIndex(TB_Position location, int index) {
        if (TB_Position.Position_LT.equals(location)) {
            return getItemByIndex(mLT_Items, index);
        } else if (TB_Position.Position_CENTER.equals(location)) {
            return getItemByIndex(mCenter_Items, index);
        } else if (TB_Position.Position_RB.equals(location)) {
            return getItemByIndex(mRB_Items, index);
        }
        return null;
    }

    @Override
    public int getItemVisibility(TB_Position location, int tag) {
        ArrayList<IBaseItem> baseItems = null;
        if (TB_Position.Position_LT.equals(location)) {
            baseItems = mLT_Items;
        } else if (TB_Position.Position_CENTER.equals(location)) {
            baseItems = mCenter_Items;
        } else if (TB_Position.Position_RB.equals(location)) {
            baseItems = mRB_Items;
        }

        if (baseItems == null || baseItems.isEmpty()) {
//            UIToast.getInstance(mContext).show(mContext.getApplicationContext().getString(R.string.fx_find_item_error));
            return -1;
        }

        IBaseItem item = getItem(baseItems, tag);
        if (item == null) {
//            UIToast.getInstance(mContext).show(mContext.getApplicationContext().getString(R.string.fx_find_item_error));
            return -1;
        }
        return item.getContentView().getVisibility();
    }

    @Override
    public int getItemVisibilityByIndex(TB_Position location, int index) {
        ArrayList<IBaseItem> baseItems = null;
        if (TB_Position.Position_LT.equals(location)) {
            baseItems = mLT_Items;
        } else if (TB_Position.Position_CENTER.equals(location)) {
            baseItems = mCenter_Items;
        } else if (TB_Position.Position_RB.equals(location)) {
            baseItems = mRB_Items;
        }

        if (baseItems == null || baseItems.isEmpty()) {
//            UIToast.getInstance(mContext).show(mContext.getApplicationContext().getString(R.string.fx_find_item_error));
            return -1;
        }

        IBaseItem item = getItemByIndex(baseItems, index);
        if (item == null) {
//            UIToast.getInstance(mContext).show(mContext.getApplicationContext().getString(R.string.fx_find_item_error));
            return -1;
        }
        return item.getContentView().getVisibility();
    }

    @Override
    public void setItemVisibility(TB_Position location, int tag, int visibility) {
        ArrayList<IBaseItem> baseItems = null;
        if (TB_Position.Position_LT.equals(location)) {
            baseItems = mLT_Items;
        } else if (TB_Position.Position_CENTER.equals(location)) {
            baseItems = mCenter_Items;
        } else if (TB_Position.Position_RB.equals(location)) {
            baseItems = mRB_Items;
        }

        if (baseItems == null || baseItems.isEmpty()) {
//            UIToast.getInstance(mContext).show(mContext.getApplicationContext().getString(R.string.fx_find_item_error));
            return;
        }

        IBaseItem item = getItem(baseItems, tag);
        if (item == null) {
//            UIToast.getInstance(mContext).show(mContext.getApplicationContext().getString(R.string.fx_find_item_error));
            return;
        }
        item.getContentView().setVisibility(visibility);
    }

    @Override
    public void setItemVisibilityByIndex(TB_Position location, int index, int visibility) {
        ArrayList<IBaseItem> baseItems = null;
        if (TB_Position.Position_LT.equals(location)) {
            baseItems = mLT_Items;
        } else if (TB_Position.Position_CENTER.equals(location)) {
            baseItems = mCenter_Items;
        } else if (TB_Position.Position_RB.equals(location)) {
            baseItems = mRB_Items;
        }

        if (baseItems == null || baseItems.isEmpty()) {
//            UIToast.getInstance(mContext).show(mContext.getApplicationContext().getString(R.string.fx_find_item_error));
            return;
        }

        IBaseItem item = getItemByIndex(baseItems, index);
        if (item == null) {
//            UIToast.getInstance(mContext).show(mContext.getApplicationContext().getString(R.string.fx_find_item_error));
            return;
        }
        item.getContentView().setVisibility(visibility);
    }

    private IBaseItem getItem(ArrayList<IBaseItem> items, int tag) {
        for (IBaseItem item : items) {
            if (item.getTag() == tag) {
                return item;
            }
        }
        return null;
    }

    private IBaseItem getItemByIndex(ArrayList<IBaseItem> items, int index) {
        return items.get(index);
    }

    @Override
    public int getItemsCount(TB_Position location) {
        if (TB_Position.Position_LT.equals(location)) {
            return mLT_Items.size();
        } else if (TB_Position.Position_CENTER.equals(location)) {
            return mCenter_Items.size();
        } else if (TB_Position.Position_RB.equals(location)) {
            return mRB_Items.size();
        }
        return 0;
    }

    @Override
    public void setOrientation(int orientation) {
        mIsRefreshLayout = true;
        initOrientation(orientation);
    }

    @Override
    public void setOrientation(int orientation, int width, int height) {
        mWidth = width;
        mHeight = height;
        mIsRefreshLayout = true;
        initOrientation(orientation);
    }

    @Override
    public void setWidth(int width) {
        mWidth = width;
        mRootLayout.getLayoutParams().width = width;
    }

    @Override
    public void setHeight(int height) {
        mHeight = height;
        mRootLayout.getLayoutParams().height = height;
    }

    public Point measureSize() {
        int xSpec = View.MeasureSpec.makeMeasureSpec(0, View.MeasureSpec.UNSPECIFIED);
        int ySpec = View.MeasureSpec.makeMeasureSpec(0, View.MeasureSpec.UNSPECIFIED);

        mLTLayout.measure(xSpec, ySpec);
        mCenterLayout.measure(xSpec, ySpec);
        mRBLayout.measure(xSpec, ySpec);

        int width = mLTLayout.getMeasuredWidth();
        width += mCenterLayout.getMeasuredWidth();
        width += mRBLayout.getMeasuredWidth();

        int height = mLTLayout.getMeasuredHeight();
        height += mCenterLayout.getMeasuredHeight();
        height += mRBLayout.getMeasuredHeight();

        if (mOrientation == HORIZONTAL)
            return new Point(width, mHeight);
        else
            return new Point(mWidth, height);
    }

    protected void initOrientation(int orientation, int width, int height) {
        mWidth = width;
        mHeight = height;
        mIsRefreshLayout = true;
        initOrientation(orientation);
    }

    protected void initOrientation(int orientation) {
        mOrientation = orientation;
        if (mIsRefreshLayout) {
            ViewParent parentView = mRootLayout.getParent();
            if (parentView == null) {
                RelativeLayout.LayoutParams rootLp = new RelativeLayout.LayoutParams(mWidth, mHeight);
                mRootLayout.setLayoutParams(rootLp);
            } else if (parentView instanceof RelativeLayout) {
                RelativeLayout.LayoutParams rootLp = new RelativeLayout.LayoutParams(mWidth, mHeight);
                mRootLayout.setLayoutParams(rootLp);
            } else if (parentView instanceof LinearLayout) {
                LinearLayout.LayoutParams rootLp = new LinearLayout.LayoutParams(mWidth, mHeight);
                mRootLayout.setLayoutParams(rootLp);
            }
        }

        if (orientation == HORIZONTAL) {
            if (mIsRefreshLayout) {
                FrameLayout.LayoutParams ltLp = new FrameLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, mHeight);
                ltLp.gravity = Gravity.START | Gravity.CENTER_VERTICAL;
                mLTLayout.setLayoutParams(ltLp);
                FrameLayout.LayoutParams rbLp = new FrameLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, mHeight);
                rbLp.gravity = Gravity.END | Gravity.CENTER_VERTICAL;
                mRBLayout.setLayoutParams(rbLp);
                FrameLayout.LayoutParams centerLp = new FrameLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, mHeight);
                //todo support case that mMiddleButtonCenter set to false for center layout
                centerLp.gravity = Gravity.CENTER;
                mCenterLayout.setLayoutParams(centerLp);

                mLTLayout.setOrientation(LinearLayout.HORIZONTAL);
                mLTLayout.setGravity(Gravity.CENTER_VERTICAL);

                mCenterLayout.setOrientation(LinearLayout.HORIZONTAL);
                mCenterLayout.setGravity(Gravity.CENTER);

                mRBLayout.setOrientation(LinearLayout.HORIZONTAL);
                mRBLayout.setGravity(Gravity.CENTER_VERTICAL);
            }

            if (mStartItemLayoutParams == null || mIsRefreshLayout) {
                mStartItemLayoutParams = new LinearLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, mHeight);
                mStartItemLayoutParams.setMargins(0, 0, mStartInterval != null ? mStartInterval : mInterval, 0);

                mCenterItemLayoutParams = new LinearLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, mHeight);
                mCenterItemLayoutParams.setMargins(0, 0, mCenterInterval != null ? mCenterInterval : mInterval, 0);

                mEndItemLayoutParams = new LinearLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, mHeight);
                mEndItemLayoutParams.setMargins(0, 0, mEndInterval != null ? mEndInterval : mInterval, 0);
            }

            mLastItemLayoutParams = new LinearLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, mHeight);
        } else {
            if (mIsRefreshLayout) {
                FrameLayout.LayoutParams ltLp = new FrameLayout.LayoutParams(mWidth, ViewGroup.LayoutParams.WRAP_CONTENT);
                ltLp.gravity = Gravity.TOP | Gravity.CENTER_HORIZONTAL;
                mLTLayout.setLayoutParams(ltLp);
                FrameLayout.LayoutParams rbLp = new FrameLayout.LayoutParams(mWidth, ViewGroup.LayoutParams.WRAP_CONTENT);
                rbLp.gravity = Gravity.BOTTOM | Gravity.CENTER_HORIZONTAL;
                mRBLayout.setLayoutParams(rbLp);
                FrameLayout.LayoutParams centerLp = new FrameLayout.LayoutParams(mWidth, ViewGroup.LayoutParams.MATCH_PARENT);
                //todo support case that mMiddleButtonCenter set to false for center layout
                centerLp.gravity = Gravity.CENTER;
                mCenterLayout.setLayoutParams(centerLp);

                mLTLayout.setOrientation(LinearLayout.VERTICAL);
                mLTLayout.setGravity(Gravity.CENTER_HORIZONTAL);

                mCenterLayout.setOrientation(LinearLayout.VERTICAL);
                mCenterLayout.setGravity(Gravity.CENTER);

                mRBLayout.setOrientation(LinearLayout.VERTICAL);
                mRBLayout.setGravity(Gravity.CENTER_HORIZONTAL);
            }

            if (mStartItemLayoutParams == null || mIsRefreshLayout) {
                mStartItemLayoutParams = new LinearLayout.LayoutParams(mWidth, ViewGroup.LayoutParams.WRAP_CONTENT);
                mStartItemLayoutParams.setMargins(0, 0, 0, mStartInterval != null ? mStartInterval : mInterval);

                mCenterItemLayoutParams = new LinearLayout.LayoutParams(mWidth, ViewGroup.LayoutParams.WRAP_CONTENT);
                mCenterItemLayoutParams.setMargins(0, 0, 0, mCenterInterval != null ? mCenterInterval : mInterval);

                mEndItemLayoutParams = new LinearLayout.LayoutParams(mWidth, ViewGroup.LayoutParams.WRAP_CONTENT);
                mEndItemLayoutParams.setMargins(0, 0, 0, mEndInterval != null ? mEndInterval : mInterval);
            }

            mLastItemLayoutParams = new LinearLayout.LayoutParams(mWidth, ViewGroup.LayoutParams.WRAP_CONTENT);
        }

        resetItemsLayout(mLT_Items, mLTLayout, mStartItemLayoutParams);
        resetItemsLayout(mCenter_Items, mCenterLayout, mCenterItemLayoutParams);
        resetItemsLayout(mRB_Items, mRBLayout, mEndItemLayoutParams);
        mIsRefreshLayout = false;
    }

    @Override
    public void setBackgroundColor(int color) {
        if (mRootLayout != null) {
            mBackgroundColor = color;
            mRootLayout.setBackgroundColor(color);
        }
    }

    public void setBackGroundColorFilter(int color, PorterDuff.Mode mode){
        if (mRootLayout != null && mRootLayout.getBackground() != null) {
            mRootLayout.getBackground().setColorFilter(color, mode);
        }
    }

    @Override
    public void setBackgroundResource(int res) {
        if (mRootLayout != null) {
            mRootLayout.setBackgroundResource(res);
        }
    }

    @Override
    public void setItemInterval(int space) {
        mInterval = space;
        if (mStartInterval != null)
            mStartInterval = space;
        if (mCenterInterval != null)
            mCenterInterval = space;
        if (mEndInterval != null)
            mEndInterval = space;

        if (mOrientation == HORIZONTAL) {
            mStartItemLayoutParams.setMargins(0, 0, mInterval, 0);
            mCenterItemLayoutParams.setMargins(0, 0, mInterval, 0);
            mEndItemLayoutParams.setMargins(0, 0, mInterval, 0);
        } else {
            mStartItemLayoutParams.setMargins(0, 0, 0, mInterval);
            mCenterItemLayoutParams.setMargins(0, 0, 0, mInterval);
            mEndItemLayoutParams.setMargins(0, 0, 0, mInterval);
        }

        resetItemsLayout(mLT_Items, mLTLayout, mStartItemLayoutParams);
        resetItemsLayout(mCenter_Items, mCenterLayout, mCenterItemLayoutParams);
        resetItemsLayout(mRB_Items, mRBLayout, mEndItemLayoutParams);
    }

    @Override
    public void setStartItemInterval(int space) {
        mStartInterval = space;
        if (mOrientation == HORIZONTAL) {
            mStartItemLayoutParams.setMargins(0, 0, mStartInterval, 0);
        } else {
            mStartItemLayoutParams.setMargins(0, 0, 0, mStartInterval);
        }
        resetItemsLayout(mLT_Items, mLTLayout, mStartItemLayoutParams);
    }

    @Override
    public void setCenterItemInterval(int space) {
        mCenterInterval = space;
        if (mOrientation == HORIZONTAL) {
            mCenterItemLayoutParams.setMargins(0, 0, mCenterInterval, 0);
        } else {
            mCenterItemLayoutParams.setMargins(0, 0, 0, mCenterInterval);
        }
        resetItemsLayout(mCenter_Items, mCenterLayout, mCenterItemLayoutParams);
    }

    @Override
    public void setEndItemiInterval(int space) {
        mEndInterval = space;
        if (mOrientation == HORIZONTAL) {
            mEndItemLayoutParams.setMargins(0, 0, mEndInterval, 0);
        } else {
            mEndItemLayoutParams.setMargins(0, 0, 0, mEndInterval);
        }
        resetItemsLayout(mRB_Items, mRBLayout, mEndItemLayoutParams);
    }

    @Override
    public void setContentView(View v) {
        removeAllItems();
        mRootLayout.setPadding(0, 0, 0, 0);
        mRootLayout.addView(v);
        mContentView = v;
    }

    @Override
    public void setInterceptTouch(boolean isInterceptTouch) {
        mRootLayout.setInterceptTouch(isInterceptTouch);
    }

    @Override
    public void setAutoCompressItemsInterval(boolean needResetItemInterval) {
        mAutoCompressItemsInterval = needResetItemInterval;
    }

    @Override
    public void setUniformSpace(boolean uniformSpace) {
        mUniformSpace = uniformSpace;

        if (uniformSpace) {
            mLTLayout.setVisibility(View.GONE);
            mRBLayout.setVisibility(View.GONE);
        } else {
            mLTLayout.setVisibility(View.VISIBLE);
            mRBLayout.setVisibility(View.VISIBLE);
        }
    }

    public void layout(int l, int t, int r, int b) {
        if (mUniformSpace) {
            int w = Math.abs(r - l);
            int h = Math.abs(b - t);
            if (mOrientation == HORIZONTAL) {
                resetUniformItems(w);
            } else {
                resetUniformItems(h);
            }
        }
    }

    private void resetUniformItems(int newLength) {
        int interval = calcUniformSpaceMargin(mOrientation, newLength);
        if (mInterval == interval)
            return;
        mInterval = interval;
        if (mOrientation == HORIZONTAL) {
            mCenterLayout.setGravity(Gravity.CENTER);
            mCenterItemLayoutParams.setMargins(0, 0, mInterval, 0);
        } else {
            mCenterLayout.setGravity(Gravity.CENTER);
            mCenterItemLayoutParams.setMargins(0, 0, 0, mInterval);
        }
        resetItemsLayout(mCenter_Items, mCenterLayout, mCenterItemLayoutParams);
    }

    private int calcUniformSpaceMargin(int orientation, int newLength) {
        int itemInterval = 0;
        int itemsTotalLength = 0;
        if (mCenter_Items.size() >= 2) {
            for (int i = 0; i < mCenter_Items.size(); i++) {
                mCenter_Items.get(i).getContentView().measure(0, 0);
                if (orientation == HORIZONTAL) {
                    itemsTotalLength += mCenter_Items.get(i).getContentView().getMeasuredWidth();
                } else {
                    itemsTotalLength += mCenter_Items.get(i).getContentView().getMeasuredHeight();
                }
                itemInterval = (newLength - mStartMargin - mEndMargin - itemsTotalLength) / (mCenter_Items.size() + 1);
            }
        }
        return itemInterval;
    }

    @Override
    public void updateLayout() {
        if (mLT_Items.size() > 0) {
            resetLayout(TB_Position.Position_LT);
        }
        if (mCenter_Items.size() > 0) {
            resetLayout(TB_Position.Position_CENTER);
        }
        if (mRB_Items.size() > 0) {
            resetLayout(TB_Position.Position_RB);
        }
    }

    @Override
    public int getHeight() {
        return mHeight;
    }

    void resetMargin() {
        if (mOrientation == HORIZONTAL) {
            mRootLayout.setPadding(mStartMargin, 0, mEndMargin, 0);
        } else {
            mRootLayout.setPadding(0, mStartMargin, 0, mEndMargin);
        }
    }

    public void resetMargin(int startMargin, int endMargin) {
        mStartMargin = startMargin;
        mEndMargin = endMargin;
        if (mOrientation == HORIZONTAL) {
            mRootLayout.setPadding(mStartMargin, 0, mEndMargin, 0);
        } else {
            mRootLayout.setPadding(0, mStartMargin, 0, mEndMargin);
        }
    }

    @Override
    public void setStartMargin(int margin) {
        mStartMargin = margin;
        resetMargin();
    }

    @Override
    public int getStartMargin() {
        return mStartMargin;
    }

    @Override
    public void setEndMargin(int margin) {
        mEndMargin = margin;
        resetMargin();
    }

    @Override
    public int getEndMargin() {
        return mEndMargin;
    }

    @Override
    public void setMiddleButtonCenter(boolean center) {
        boolean changed = mMiddleButtonCenter != center;
        mMiddleButtonCenter = center;
        mIsRefreshLayout = true;
        if (changed)
            initOrientation(mOrientation);
    }

    private class ComparatorTBItemByIndex implements Comparator<Object> {
        @Override
        public int compare(Object lhs, Object rhs) {
            if (lhs instanceof IBaseItem && rhs instanceof IBaseItem) {
                IBaseItem lItem = (IBaseItem) lhs;
                IBaseItem rItem = (IBaseItem) rhs;
                return lItem.getTag() - rItem.getTag();
            } else {
                return 0;
            }
        }
    }

    protected int dip2px(int dip) {
        if (dip <= 0) {
            return dip;
        } else {
            return AppDisplay.dp2px(dip);
        }
    }

    protected int dip2px_fromDimens(int dip) {
        return dip;
    }

    @Override
    public BaseBar cloneBar() {
        BaseBarImpl baseBar;
        if (this instanceof TopBarImpl) {
            baseBar = new TopBarImpl(mContext);
        } else if (this instanceof BottomBarImpl) {
            baseBar = new BottomBarImpl(mContext);
        } else {
            baseBar = new BaseBarImpl(mContext);
        }
        baseBar.mAutoCompressItemsInterval = mAutoCompressItemsInterval;
        baseBar.mMiddleButtonCenter = mMiddleButtonCenter;
        baseBar.mStartMargin = mStartMargin;
        baseBar.mEndMargin = mEndMargin;
        baseBar.mStartInterval = mStartInterval;
        baseBar.mCenterInterval = mCenterInterval;
        baseBar.mEndInterval = mEndInterval;
        baseBar.setBackgroundColor(mBackgroundColor);

        for (IBaseItem leftItem: mLT_Items){
            baseBar.addView(leftItem.cloneItem(), TB_Position.Position_LT);
        }
        for (IBaseItem centerItem: mCenter_Items){
            baseBar.addView(centerItem.cloneItem(), TB_Position.Position_CENTER);
        }
        for (IBaseItem rightItem: mRB_Items){
            baseBar.addView(rightItem.cloneItem(), TB_Position.Position_RB);
        }
        return baseBar;
    }

    public void updateThemeColor(){
        setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
        for (IBaseItem leftItem: mLT_Items){
           leftItem.updateThemeColor();
        }
        for (IBaseItem centerItem: mCenter_Items){
            centerItem.updateThemeColor();
        }
        for (IBaseItem rightItem: mRB_Items){
            rightItem.updateThemeColor();
        }
    }

    private DynamicColorProvider colorProvider;

    public void setBackgroundColorProvider(DynamicColorProvider provider) {
        this.colorProvider = provider;
        updateBackgroundColor();
    }

    public void setBackgroundColor(@ColorInt int lightColor, @ColorInt int darkColor){
        colorProvider = new ColorPair(lightColor, darkColor);
        updateBackgroundColor();
    }

    public void updateBackgroundColor() {
        int color = this.mBackgroundColor;
        if (colorProvider != null) {
            color = colorProvider.getColor(AppUtil.isDarkMode(mContext));
        }
        mRootLayout.setBackgroundColor(color);
    }

}
