/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.toolbar.drag;

import android.app.Activity;
import android.content.Context;
import android.content.res.ColorStateList;
import android.graphics.Color;
import android.graphics.Rect;
import android.util.SparseIntArray;
import android.view.View;
import android.widget.ImageView;
import android.widget.PopupWindow;
import android.widget.RelativeLayout;
import android.widget.TextView;

import androidx.fragment.app.FragmentActivity;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Renderer;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.common.UIFillView;
import com.foxit.uiextensions.config.Config;
import com.foxit.uiextensions.controls.dialog.UIPopoverFrag;
import com.foxit.uiextensions.controls.dialog.UIPopoverWin;
import com.foxit.uiextensions.controls.propertybar.IViewSettingsWindow;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.propertybar.imp.SimpleColorView;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.controls.toolbar.IBaseItem;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolItemBean;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.UIToolView;
import com.foxit.uiextensions.controls.toolbar.impl.UIColorItem;
import com.foxit.uiextensions.controls.toolbar.tabmenu.SortToolsDialog;
import com.foxit.uiextensions.pdfreader.MainCenterItemBean;
import com.foxit.uiextensions.pdfreader.config.ReadStateConfig;
import com.foxit.uiextensions.pdfreader.impl.MainFrame;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class ViewToolBar extends UIHSToolBar implements IViewSettingsWindow {

    private SparseIntArray mDrawableMap = new SparseIntArray();
    private SparseIntArray mItemTagMap = new SparseIntArray();
    private HashMap<Integer, UIColorItem> mItems = new HashMap<>();
    private Map<Integer, IValueChangeListener> mViewValueListeners = new HashMap<Integer, IValueChangeListener>();
    private IBaseItem mCurItem = null;

    private @PageColorMode
    int mPageColorMode = IViewSettingsWindow.DAY;

    private UIDragToolBar mUiToolbar;
    private boolean mbNightMode = false;
    private Integer mCustomPageColor;
    private SortToolsDialog mDialog;

    private boolean mInitPageColor = false;
    private UIExtensionsManager mUIExtensionsManager;

    public ViewToolBar(Context context, UIExtensionsManager uiExtensionsManager, UIDragToolBar uiToolbar) {
        this(context, UIToolBaseBar.HORIZONTAL, uiExtensionsManager, uiToolbar, null);
    }

    public ViewToolBar(Context context, int orientation, UIExtensionsManager uiExtensionsManager, UIDragToolBar uiToolbar, MainCenterItemBean bean) {
        this(context, orientation, false, uiExtensionsManager, uiToolbar, bean);
    }

    protected ViewToolBar(Context context, int orientation, boolean uniformSpace, UIExtensionsManager uiExtensionsManager, UIDragToolBar uiToolbar, MainCenterItemBean bean) {
        this(context, orientation, 0, 0, uniformSpace, uiExtensionsManager, uiToolbar, bean);
    }

    protected ViewToolBar(Context context, int orientation, int width, int height, boolean uniformSpace, UIExtensionsManager uiExtensionsManager, UIDragToolBar uiToolbar, MainCenterItemBean bean) {
        super(context, orientation, width, height, uniformSpace, uiExtensionsManager, uiToolbar);
        mItemsBean = bean;
        mUIExtensionsManager = uiExtensionsManager;
        initialize(uiToolbar);
    }

    private MainCenterItemBean mItemsBean;

    private void initialize(UIDragToolBar uiToolbar) {
        mInitPageColor = false;

        mUsageType = USAGE_TYPE_VIEW;
        mUiToolbar = uiToolbar;

        //if (pdfViewCtrl.isNightMode() && pdfViewCtrl.getColorMode() == Renderer.e_ColorModeMappingGray) {
        Config config = mUIExtensionsManager.getConfig();
        if (config.uiSettings.colorMode.equals("Night")) {
            mPageColorMode = IViewSettingsWindow.NIGHT;
        } else if (config.uiSettings.colorMode.equals("Map")) {
            mPageColorMode = IViewSettingsWindow.CUSTOM_COLOR;
        } else {
            mPageColorMode = IViewSettingsWindow.DAY;
        }

        if (mUIExtensionsManager.getPDFViewCtrl().getColorMode() == Renderer.e_ColorModeMappingGray
                || mUIExtensionsManager.getPDFViewCtrl().getColorMode() == Renderer.e_ColorModeMapping) {
            if (isNightMode()) {
                mPageColorMode = IViewSettingsWindow.NIGHT;
            } else {
                mPageColorMode = IViewSettingsWindow.CUSTOM_COLOR;
                mCustomPageColor = mUIExtensionsManager.getPDFViewCtrl().getMappingModeBackgroundColor();
            }
        }


        initializeDrawMap();
        initializeItems();
        initializeViewSetting();
        //initializePsageColor();

    }

    @Override
    public void addOptionItems(List<ToolItemBean> toolList) {
        super.addOptionItems(toolList);
        initSortItemView(toolList);
    }

    private void initSortItemView(final List<ToolItemBean> toolList) {
        setSortItemPosition(TB_Position.Position_RB);
        mSortItem.setOnItemClickListener(new IBaseItem.OnItemClickListener() {
            @Override
            public void onClick(IBaseItem item, View v) {
                if(mDialog ==null) {
                    mDialog = new SortToolsDialog(mUIExtensionsManager.getAttachedActivity(), mUIExtensionsManager, ToolbarItemConfig.ITEM_VIEW_TAB);
                }
                if(isNightMode() != mDialog.getViewNight()){
                    mDialog.setViewNight(isNightMode());
                }
                mDialog.setData(toolList);
                mDialog.showDialog();
            }
        });

    }
    private void initializeDrawMap() {
        mItemTagMap.put(0, IViewSettingsWindow.TYPE_SINGLE_PAGE);
        mItemTagMap.put(1, IViewSettingsWindow.TYPE_DAY);
        mItemTagMap.put(2, IViewSettingsWindow.TYPE_PAGE_COLOR);
        mItemTagMap.put(3, IViewSettingsWindow.TYPE_FIT_PAGE);
        mItemTagMap.put(4, IViewSettingsWindow.TYPE_FIT_WIDTH);
        mItemTagMap.put(5, IViewSettingsWindow.TYPE_REFLOW);
        mItemTagMap.put(6, IViewSettingsWindow.TYPE_CROP);
        mItemTagMap.put(7, IViewSettingsWindow.TYPE_TTS);
        mItemTagMap.put(8, IViewSettingsWindow.TYPE_AUTO_FLIP);
        mItemTagMap.put(9, IViewSettingsWindow.TYPE_ROTATE_VIEW);
        mItemTagMap.put(10, IViewSettingsWindow.TYPE_PAN_ZOOM);

        mDrawableMap.put(IViewSettingsWindow.TYPE_SINGLE_PAGE, R.drawable.view_setting_page_setting_pad);
        if (isNightMode()) {
            mDrawableMap.put(IViewSettingsWindow.TYPE_DAY, R.drawable.view_setting_night_ipad);
        }else{
            mDrawableMap.put(IViewSettingsWindow.TYPE_DAY, R.drawable.view_setting_day_ipad);
        }
        mDrawableMap.put(IViewSettingsWindow.TYPE_PAGE_COLOR, R.drawable.view_setting_page_color_ipad);
        mDrawableMap.put(IViewSettingsWindow.TYPE_FIT_PAGE, R.drawable.view_settings_fit_page);
        mDrawableMap.put(IViewSettingsWindow.TYPE_FIT_WIDTH, R.drawable.view_settings_fit_width);
        mDrawableMap.put(IViewSettingsWindow.TYPE_REFLOW, R.drawable.view_settings_reflow);
        mDrawableMap.put(IViewSettingsWindow.TYPE_CROP, R.drawable.view_settings_crop);
        mDrawableMap.put(IViewSettingsWindow.TYPE_TTS, R.drawable.view_settings_speak);
        mDrawableMap.put(IViewSettingsWindow.TYPE_AUTO_FLIP, R.drawable.view_settings_auto_flip);
        mDrawableMap.put(IViewSettingsWindow.TYPE_ROTATE_VIEW, R.drawable.view_settings_rotate);
        mDrawableMap.put(IViewSettingsWindow.TYPE_PAN_ZOOM, R.drawable.view_settings_pan_zoom);
    }

    private String _getItemString(int tag) {
       String str = "";
       switch (tag) {
           case IViewSettingsWindow.TYPE_SINGLE_PAGE:
               str = AppUtil.getToolbarItemString(mContext, ToolConstants.TypeSinglePage);
               break;
           case IViewSettingsWindow.TYPE_DAY:
               if (mDrawableMap.get(IViewSettingsWindow.TYPE_DAY) == R.drawable.view_setting_day_ipad)
                   str = AppUtil.getToolbarItemString(mContext, ToolConstants.TypeDay);
               else
                   str = AppUtil.getToolbarItemString(mContext, ToolConstants.TypeNight);
               break;
           case IViewSettingsWindow.TYPE_PAGE_COLOR:
               str = AppUtil.getToolbarItemString(mContext, ToolConstants.TypePageColor);
               break;
           case IViewSettingsWindow.TYPE_FIT_PAGE:
               str = AppUtil.getToolbarItemString(mContext, ToolConstants.TypeFitPage);
               break;
           case IViewSettingsWindow.TYPE_FIT_WIDTH:
               str = AppUtil.getToolbarItemString(mContext, ToolConstants.TypeFitWidth);
               break;
           case IViewSettingsWindow.TYPE_REFLOW:
               str = AppUtil.getToolbarItemString(mContext, ToolConstants.TypeReflow);
               break;
           case IViewSettingsWindow.TYPE_CROP:
               str = AppUtil.getToolbarItemString(mContext, ToolConstants.TypeCrop);
               break;
           case IViewSettingsWindow.TYPE_TTS:
               str = AppUtil.getToolbarItemString(mContext, ToolConstants.TypeTts);
               break;
           case IViewSettingsWindow.TYPE_AUTO_FLIP:
               str = AppUtil.getToolbarItemString(mContext, ToolConstants.TypeAutoFlip);
               break;
           case IViewSettingsWindow.TYPE_ROTATE_VIEW:
               str = AppUtil.getToolbarItemString(mContext, ToolConstants.TypeRotateView);
               break;
           case IViewSettingsWindow.TYPE_PAN_ZOOM:
               str = AppUtil.getToolbarItemString(mContext, ToolConstants.TypePanZoom);
               break;

       }
       return str;
    }

    private void initializeItems() {
        if(mItemsBean == null){
            return;
        }
         List<ToolItemBean> tempToolItems = new ArrayList<>();
         List<ToolItemBean> toolItems = new ArrayList<>();
        for (int i = 0; i < mDrawableMap.size(); i++) {
            UIColorItem item = new UIColorItem(mContext);
            item.setImageResource(mDrawableMap.get(mItemTagMap.get(i)));
            item.setImageTintList(ThemeUtil.getTextDesImageColorStateList(mContext));
            item.setText(_getItemString(mItemTagMap.get(i)));
            item.setTextColor(ThemeUtil.getTextDesTextColorStateList(mContext));
            item.setForceDarkAllowed(true);

            item.setOnItemClickListener(mItemClickListener);
            item.setTag(mItemTagMap.get(i));
            mItems.put(mItemTagMap.get(i), item);

            if (mItemsBean != null) {
                ToolItemBean toolItemBean = new ToolItemBean();
                toolItemBean.itemStyle = ToolItemBean.CUSTOM_STYLE;
                toolItemBean.toolItem = item;
                toolItemBean.type = getType(item.getTag());
                tempToolItems.add(toolItemBean);
            }
        }

        if (mItemsBean != null && mItemsBean.toolItems!=null &&  mItemsBean.toolItems.size() > 0) {
            for (int i = 0; i < mItemsBean.toolItems.size(); i++) {
                for (ToolItemBean itemBean:tempToolItems){
                    if(mItemsBean.toolItems.get(i).type == itemBean.type){
                        toolItems.add(itemBean);
                    }
                }
            }
            mItemsBean.toolItems = toolItems;
        }else{
            mItemsBean.toolItems = tempToolItems;
        }
    }

    private int getType(int tag){
        int type = 0;
        if(tag == TYPE_SINGLE_PAGE){
            type = ToolConstants.TypeSinglePage;
        }else if(tag == TYPE_DAY){
            type = ToolConstants.TypeDay;
        }else if(tag == TYPE_PAGE_COLOR){
            type = ToolConstants.TypePageColor;
        }else if(tag == TYPE_FIT_PAGE){
            type = ToolConstants.TypeFitPage;
        }else if(tag == TYPE_FIT_WIDTH){
            type = ToolConstants.TypeFitWidth;
        }else if(tag == TYPE_REFLOW){
            type = ToolConstants.TypeReflow;
        }else if(tag == TYPE_CROP){
            type = ToolConstants.TypeCrop;
        }else if(tag == TYPE_TTS){
            type = ToolConstants.TypeTts;
        }else if(tag == TYPE_AUTO_FLIP){
            type = ToolConstants.TypeAutoFlip;
        }else if(tag == TYPE_ROTATE_VIEW){
            type = ToolConstants.TypeRotateView;
        }else if(tag == TYPE_PAN_ZOOM){
            type = ToolConstants.TypePanZoom;
        }
        return  type;
    }

    IBaseItem.OnItemClickListener mItemClickListener = new IBaseItem.OnItemClickListener() {
        @Override
        public void onClick(IBaseItem item, View v) {
            //fit page/width
            if (item.getTag() == IViewSettingsWindow.TYPE_FIT_PAGE) {
                mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_FitPage);
                if (mViewValueListeners.get(item.getTag()) != null) {
                    mViewValueListeners.get(item.getTag()).onValueChanged(item.getTag(), PDFViewCtrl.ZOOMMODE_FITPAGE);
                }
                return;
            } else if (item.getTag() == IViewSettingsWindow.TYPE_FIT_WIDTH) {
                mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_FitWidth);
                if (mViewValueListeners.get(item.getTag()) != null) {
                    mViewValueListeners.get(item.getTag()).onValueChanged(item.getTag(), PDFViewCtrl.ZOOMMODE_FITWIDTH);
                }
                return;
            } else if (item.getTag() == IViewSettingsWindow.TYPE_DAY) {
                if (mPageColorMode == IViewSettingsWindow.DAY) {
                    mPageColorMode = IViewSettingsWindow.NIGHT;
                    mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_Night);
                } else if (mPageColorMode == IViewSettingsWindow.NIGHT) {
                    mPageColorMode = IViewSettingsWindow.DAY;
                    mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_Day);
                } else {
                    mPageColorMode = mPageColorMode == IViewSettingsWindow.CUSTOM_COLOR ? IViewSettingsWindow.NIGHT : IViewSettingsWindow.DAY;
                }
                mViewValueListeners.get(item.getTag()).onValueChanged(item.getTag(), mPageColorMode);
                if (mUiExtensionsManager.getState() == ReadStateConfig.STATE_PANZOOM) {
                    mViewValueListeners.get(IViewSettingsWindow.TYPE_PAN_ZOOM).onValueChanged(item.getTag(), mPageColorMode);
                }
                return;
            } else if (item.getTag() == IViewSettingsWindow.TYPE_PAGE_COLOR) {
                mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_PageColor);
                showPopupWindow(item.getTag(), v);
                return;

            } else if (item.getTag() == IViewSettingsWindow.TYPE_REFLOW) {
                mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_Reflow);
            } else if (item.getTag() == IViewSettingsWindow.TYPE_CROP) {
                mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_CropMode);
            } else if (item.getTag() == IViewSettingsWindow.TYPE_TTS) {
                mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_Speak);
            } else if (item.getTag() == IViewSettingsWindow.TYPE_AUTO_FLIP) {
                mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_AutoFlip);
            } else if (item.getTag() == IViewSettingsWindow.TYPE_ROTATE_VIEW) {
                mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_RotateView);
            } else if (item.getTag() == IViewSettingsWindow.TYPE_PAN_ZOOM) {
                mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_PanZoom);
            }
            if (item.isSelected()
                    && item.getTag() != IViewSettingsWindow.TYPE_CROP) {
                mViewValueListeners.get(item.getTag()).onValueChanged(item.getTag(), false);
                if (mCurItem != null && mCurItem.getTag() == item.getTag()) {
                    mCurItem = null;
                }
                return;
            }

            //reset current item
            if (mCurItem != null) {
                mCurItem.setSelected(false);
                if (mViewValueListeners.get(mCurItem.getTag()) != null) {
                    mViewValueListeners.get(mCurItem.getTag()).onValueChanged(mCurItem.getTag(), false);//TODO
                }
                mCurItem = null;
            }

            if (item.getTag() == IViewSettingsWindow.TYPE_SINGLE_PAGE) {
                showPopupWindow(item.getTag(), v);
            } else {
                if (item.getTag() != IViewSettingsWindow.TYPE_ROTATE_VIEW
                        && item.getTag() != IViewSettingsWindow.TYPE_CROP) {
                    mCurItem = item;
                    item.setSelected(true);
                }

                if (mViewValueListeners.get(item.getTag()) != null) {
                    mViewValueListeners.get(item.getTag()).onValueChanged(item.getTag(), true);
                }
            }
        }
    };

    @Override
    public int getUsageType() {
        return USAGE_TYPE_VIEW;
    }

//    @Override
//    public void setOrientation(int orientation) {
//        super.setOrientation(orientation);
//        layout();
//    }
//
//    @Override
//    public void setOrientation(int orientation, int width, int height) {
//        super.setOrientation(orientation, width, height);
//        layout();
//    }

    private void layout() {
        for (Map.Entry<Integer, UIColorItem> entry : mItems.entrySet()) {
            if (entry.getKey() == IViewSettingsWindow.TYPE_SINGLE_PAGE ||
                    entry.getKey() == IViewSettingsWindow.TYPE_PAGE_COLOR) {
                if (mUiToolbar.getToolBarPosition() == UIDragToolBar.TOOLBAR_POS_RIGHT) {
                    entry.getValue().setCoverImageResource(R.drawable.tool_bar_drop_left);
                } else {
                    entry.getValue().setCoverImageResource(R.drawable.tool_bar_drop_right);
                }
            }
        }
    }

    @Override
    public void setProperty(int property, Object value) {
        UIColorItem newItem = mItems.get(property);
        if (newItem == null) return;

        if (value instanceof Boolean) {
            if (property == IViewSettingsWindow.TYPE_CROP) {
                IBaseItem panzoomItem = mItems.get(IViewSettingsWindow.TYPE_PAN_ZOOM);
                if (panzoomItem != null)
                    panzoomItem.setEnable(!mUiExtensionsManager.getPDFViewCtrl().isDynamicXFA() && !(Boolean) value);
            }

            newItem.setSelected((Boolean) value);
            if (mCurItem != null && mCurItem.getTag() == newItem.getTag() && !(Boolean) value) {
                mCurItem = null;
            } else if ((Boolean) value
                    && mCurItem == null
                    && property != IViewSettingsWindow.TYPE_CROP
                    && property != IViewSettingsWindow.TYPE_ROTATE_VIEW) {
                mCurItem = newItem;
            }
        } else if (value instanceof Integer) {
            if (property == IViewSettingsWindow.TYPE_SINGLE_PAGE) {
                this.mPageMode = (Integer) value;
                setSelectedButtonState(mUiExtensionsManager.getPDFViewCtrl().isContinuous(), mContinuousPageButtonLayout, mIvContinuousPage);

                boolean rtl = mUiExtensionsManager.getPDFViewCtrl().getPageBinding() == PDFViewCtrl.RIGHT_EDGE;
                setSelectedButtonState(rtl, mRTL_SwitchLayout, mRTL_SwitchIcon);

                if (this.mPageMode == PDFViewCtrl.PAGELAYOUTMODE_SINGLE) {
                    mIvSingle.setSelected(true);
                    mIvFacing.setSelected(false);
                    mIvCover.setSelected(false);
                } else if (this.mPageMode == PDFViewCtrl.PAGELAYOUTMODE_FACING) {
                    mIvSingle.setSelected(false);
                    mIvFacing.setSelected(true);
                    mIvCover.setSelected(false);
                } else if (this.mPageMode == PDFViewCtrl.PAGELAYOUTMODE_COVER) {
                    mIvSingle.setSelected(false);
                    mIvFacing.setSelected(false);
                    mIvCover.setSelected(true);
                }
            } else if (property == IViewSettingsWindow.TYPE_DAY) {
                this.mPageColorMode = (Integer) value;
                int colorMode = mUiExtensionsManager.getPDFViewCtrl().getColorMode();
//                UIToolView toolView = (UIToolView) newItem.getCustomView();
                if (this.mPageColorMode == IViewSettingsWindow.DAY) {
                    if (mColorView != null) {
                        mColorView.setSelectedColor(Color.TRANSPARENT);
                        mColorView.reset();
                    }

                    if (colorMode == Renderer.e_ColorModeNormal) {
                        mbNightMode = false;
                        newItem.setImageResource(R.drawable.view_setting_day_ipad);
                        newItem.setText(AppUtil.getToolbarItemString(mContext, ToolConstants.TypeDay));
//                        toolView.setFillBackgroundResource(R.drawable.view_setting_day_ipad);
//                        toolView.setBackgroundColorFilter(AppResource.getColor(mContext, R.color.i3));
                    }
                } else if (this.mPageColorMode == IViewSettingsWindow.CUSTOM_COLOR) {
                    if (mbNightMode) {
                        mbNightMode = false;
                        newItem.setImageResource(R.drawable.view_setting_day_ipad);
                        newItem.setText(AppUtil.getToolbarItemString(mContext, ToolConstants.TypeDay));
//                        toolView.setFillBackgroundResource(R.drawable.view_setting_day_ipad);
//                        toolView.setBackgroundColorFilter(AppResource.getColor(mContext, R.color.i3));
                    }
                } else if (this.mPageColorMode == IViewSettingsWindow.NIGHT) {
                    if (mColorView != null) {
                        mColorView.setSelectedColor(Color.TRANSPARENT);
                        mColorView.reset();
                    }

                    if (colorMode == Renderer.e_ColorModeMapping || colorMode == Renderer.e_ColorModeMappingGray) {
                        mbNightMode = true;
                        newItem.setImageResource(R.drawable.view_setting_night_ipad);
                        newItem.setText(AppUtil.getToolbarItemString(mContext, ToolConstants.TypeNight));
//                        toolView.setFillBackgroundResource(R.drawable.view_setting_night_ipad);
//                        toolView.setBackgroundColorFilter(AppResource.getColor(mContext, R.color.i3));
                    }
                }
            } else if (property == IViewSettingsWindow.TYPE_FIT_PAGE) {
                mItems.get(IViewSettingsWindow.TYPE_FIT_WIDTH).setSelected(false);
                if ((Integer) value == PDFViewCtrl.ZOOMMODE_FITPAGE) {
                    mItems.get(IViewSettingsWindow.TYPE_FIT_PAGE).setSelected(true);
                } else {
                    mItems.get(IViewSettingsWindow.TYPE_FIT_PAGE).setSelected(false);
                }
            } else if (property == IViewSettingsWindow.TYPE_FIT_WIDTH) {
                mItems.get(IViewSettingsWindow.TYPE_FIT_PAGE).setSelected(false);

                if ((Integer) value == PDFViewCtrl.ZOOMMODE_FITWIDTH) {
                    mItems.get(IViewSettingsWindow.TYPE_FIT_WIDTH).setSelected(true);
                } else {
                    mItems.get(IViewSettingsWindow.TYPE_FIT_WIDTH).setSelected(false);
                }
            } else if (property == IViewSettingsWindow.TYPE_AUTO_FLIP) {
                mItems.get(IViewSettingsWindow.TYPE_AUTO_FLIP).setSelected((Boolean) value);
            }
        }
    }

    @Override
    public boolean isShowing() {
        return false;
    }

    @Override
    public void show() {
    }

    @Override
    public boolean getVisible(int type) {
        if (type == IViewSettingsWindow.TYPE_SINGLE_PAGE) {
            return mViewSetting.findViewById(R.id.ll_single_page_pad).getVisibility() == View.VISIBLE;
        } else if (type == IViewSettingsWindow.TYPE_FACING_PAGE) {
            return mViewSetting.findViewById(R.id.ll_facing_page_pad).getVisibility() == View.VISIBLE;
        } else if (type == IViewSettingsWindow.TYPE_COVER_PAGE) {
            return mViewSetting.findViewById(R.id.ll_cover_page_pad).getVisibility() == View.VISIBLE;
        } else if (type == IViewSettingsWindow.TYPE_CONTINUOUS_PAGE) {
            return mViewSetting.findViewById(R.id.ll_continuous_page_pad).getVisibility() == View.VISIBLE;
        } else if (type == IViewSettingsWindow.TYPE_RIGHT_TO_LEFT) {
            return mViewSetting.findViewById(R.id.ll_rtl_pad).getVisibility() == View.VISIBLE;
        } else {
            UIColorItem baseItem = mItems.get(type);
            if (null == baseItem) return false;
            return baseItem.getContentView().getVisibility() == View.VISIBLE;
        }
    }

    @Override
    public void setVisible(int type, boolean visible) {
        int visibility = visible ? View.VISIBLE : View.GONE;
        if (type == IViewSettingsWindow.TYPE_SINGLE_PAGE) {
            mViewSetting.findViewById(R.id.ll_single_page_pad).setVisibility(visibility);
        } else if (type == IViewSettingsWindow.TYPE_FACING_PAGE) {
            mViewSetting.findViewById(R.id.ll_facing_page_pad).setVisibility(visibility);
        } else if (type == IViewSettingsWindow.TYPE_COVER_PAGE) {
            mViewSetting.findViewById(R.id.ll_cover_page_pad).setVisibility(visibility);
        } else if (type == IViewSettingsWindow.TYPE_CONTINUOUS_PAGE) {
            mViewSetting.findViewById(R.id.ll_continuous_page_pad).setVisibility(visibility);
        } else if (type == IViewSettingsWindow.TYPE_RIGHT_TO_LEFT) {
            mViewSetting.findViewById(R.id.ll_rtl_pad).setVisibility(visibility);
        } else {
            if (type == IViewSettingsWindow.TYPE_NIGHT)
                type = IViewSettingsWindow.TYPE_DAY;
            UIColorItem baseItem = mItems.get(type);
            if (null == baseItem) return;
            baseItem.getContentView().setVisibility(visibility);
        }
    }

    @Override
    public void dismiss() {

    }

    @Override
    public void registerListener(IValueChangeListener listener) {
        int type = listener.getType();
        if (!mViewValueListeners.containsKey(type)) {
            mViewValueListeners.put(type, listener);
        }
    }

    @Override
    public void unRegisterListener(IValueChangeListener listener) {
        mViewValueListeners.remove(listener.getType());
    }

    @Override
    public IValueChangeListener getValueChangeListener(int type) {
        return mViewValueListeners.get(type);
    }

    @Override
    public void enableBar(int property, boolean enable) {
        if (property == IViewSettingsWindow.TYPE_FACING_PAGE) {
            mIvFacing.setEnabled(enable);
            mTvFacing.setEnabled(enable);
        } else if (property == IViewSettingsWindow.TYPE_COVER_PAGE) {
            mIvCover.setEnabled(enable);
            mTvConver.setEnabled(enable);
        } else if (property == IViewSettingsWindow.TYPE_CONTINUOUS_PAGE) {
            mContinuousPageTitleIv.setEnabled(enable);
            mIvContinuousPageTitleTv.setEnabled(enable);
            mContinuousPageButtonLayout.setEnabled(enable);
        } else if (property == IViewSettingsWindow.TYPE_RIGHT_TO_LEFT) {
            mRTLTitleIv.setEnabled(enable);
            mRTLTitleTv.setEnabled(enable);
            mRTL_SwitchLayout.setEnabled(enable);
        } else {
            for (Map.Entry<Integer, UIColorItem> entry : mItems.entrySet()) {
                if (entry.getKey() == property) {
                    entry.getValue().setEnable(enable);
                }
            }
        }
    }

    @Override
    public void setOnDismissListener(OnDismissListener listener) {

    }

    @Override
    public IViewSettingsWindow updateTheme() {
        reInit();

        for (UIColorItem colorItem : mItems.values()) {
            if (colorItem.getCustomView() instanceof UIToolView) {
                UIToolView toolView = (UIToolView) colorItem.getCustomView();
                toolView.updateTheme();
            } else {
//                if (colorItem.getTag() == IViewSettingsWindow.TYPE_SINGLE_PAGE ||
//                        colorItem.getTag() == IViewSettingsWindow.TYPE_PAGE_COLOR) {
//                    colorItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
//                    colorItem.setCoverImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
//                }
                colorItem.updateThemeColor();
            }
        }
        if (mIvSingle != null) {
            setProperty(IViewSettingsWindow.TYPE_SINGLE_PAGE, mPageMode);
            ((MainFrame) mUiExtensionsManager.getMainFrame()).resetPageLayout();
        }

        int backgroundColor = (mUIToolbar.getBackgroundColorProvider().getColor(AppUtil.isDarkMode(mContext)));
        if (mToolsbar != null)
            mToolsbar.setBackgroundColor(Color.TRANSPARENT);
        if (mOptionsToolbar != null)
            mOptionsToolbar.setBackgroundColor(Color.TRANSPARENT);
        mRootView.setBackgroundColor(backgroundColor);
        setBackgroundColor(backgroundColor);
        return this;
    }

    @Override
    public int getColorMode() {
        return mPageColorMode;
    }

    @Override
    public boolean isSelected(int property) {
        UIColorItem colorItem = mItems.get(property);
        if (colorItem != null) {
            return colorItem.isSelected();
        }
        return false;
    }

    //For view setting
    private View mViewSetting;
    private UIFillView mIvSingle;
    private UIFillView mIvFacing;
    private UIFillView mIvCover;
    private TextView mTvFacing;
    private TextView mTvConver;

    private ImageView mContinuousPageTitleIv;
    private TextView mIvContinuousPageTitleTv;
    private RelativeLayout mContinuousPageButtonLayout;
    private ImageView mIvContinuousPage;
    private int mPageMode = PDFViewCtrl.PAGELAYOUTMODE_SINGLE;

    private ImageView mRTLTitleIv;
    private TextView mRTLTitleTv;
    private RelativeLayout mRTL_SwitchLayout;
    private ImageView mRTL_SwitchIcon;

    private void initializeViewSetting() {
        mViewSetting = View.inflate(mContext, R.layout.view_setting_layout_pad, null);
        mIvSingle = mViewSetting.findViewById(R.id.iv_single_page_icon_pad);
        mIvSingle.setOnClickListener(mClickListener);
        mIvSingle.setTintList(getBorderColorStateList(), ThemeUtil.getPrimaryIconColor(mContext, true));

        mTvFacing = mViewSetting.findViewById(R.id.tv_facing_page_icon_pad);
        ThemeUtil.setTintList(mTvFacing, ThemeUtil.getToolbarTextColor(mContext));
        mIvFacing = mViewSetting.findViewById(R.id.iv_facing_page_icon_pad);
        mIvFacing.setOnClickListener(mClickListener);
        mIvFacing.setTintList(getBorderColorStateList(), ThemeUtil.getPrimaryIconColor(mContext, true));

        mTvConver = mViewSetting.findViewById(R.id.tv_cover_page_icon_pad);
        ThemeUtil.setTintList(mTvConver, ThemeUtil.getToolbarTextColor(mContext));
        mIvCover = mViewSetting.findViewById(R.id.iv_cover_page_icon_pad);
        mIvCover.setOnClickListener(mClickListener);
        mIvCover.setTintList(getBorderColorStateList(), ThemeUtil.getPrimaryIconColor(mContext, true));

        mContinuousPageTitleIv = mViewSetting.findViewById(R.id.iv_continuous_page_icon_pad);
        ThemeUtil.setTintList(mContinuousPageTitleIv, ThemeUtil.getPrimaryIconColor(mContext));
        mIvContinuousPageTitleTv = mViewSetting.findViewById(R.id.tv_continuous_page_title_pad);
        mContinuousPageButtonLayout = mViewSetting.findViewById(R.id.ll_continuous_page_button_pad);
        mContinuousPageButtonLayout.setOnClickListener(mClickListener);
        ThemeUtil.setTintList(mIvContinuousPageTitleTv, ThemeUtil.getToolbarTextColor(mContext));
        ThemeUtil.setBackgroundTintList(mContinuousPageButtonLayout, getSelectedButtonColorStateList());
        mIvContinuousPage = mViewSetting.findViewById(R.id.continuous_page_button_pad);

        mRTLTitleIv = mViewSetting.findViewById(R.id.iv_rtl_icon_pad);
        ThemeUtil.setTintList(mRTLTitleIv, ThemeUtil.getPrimaryIconColor(mContext));
        mRTLTitleTv = mViewSetting.findViewById(R.id.tv_rtl_title_pad);
        ThemeUtil.setTintList(mRTLTitleTv, ThemeUtil.getToolbarTextColor(mContext));
        mRTL_SwitchLayout = mViewSetting.findViewById(R.id.ll_rtl_switch_layout_pad);
        mRTL_SwitchLayout.setOnClickListener(mClickListener);
        ThemeUtil.setBackgroundTintList(mRTL_SwitchLayout, getSelectedButtonColorStateList());
        mRTL_SwitchIcon = mViewSetting.findViewById(R.id.iv_rtl_switch_icon_pad);
    }

    private View.OnClickListener mClickListener = new View.OnClickListener() {
        @Override
        public void onClick(View v) {
            int id = v.getId();
            if (id == R.id.iv_single_page_icon_pad) {
                if (mViewValueListeners.get(IViewSettingsWindow.TYPE_SINGLE_PAGE) != null) {
                    if (mPageMode != PDFViewCtrl.PAGELAYOUTMODE_SINGLE) {
                        mPageMode = PDFViewCtrl.PAGELAYOUTMODE_SINGLE;
                        mViewValueListeners.get(IViewSettingsWindow.TYPE_SINGLE_PAGE).onValueChanged(IViewSettingsWindow.TYPE_SINGLE_PAGE, mPageMode);
                    }
                }

                mIvSingle.setSelected(true);
                mIvFacing.setSelected(false);
                mIvCover.setSelected(false);
                mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_SinglePage);
            } else if (id == R.id.iv_facing_page_icon_pad) {
                if (mViewValueListeners.get(IViewSettingsWindow.TYPE_SINGLE_PAGE) != null) {
                    if (mPageMode != PDFViewCtrl.PAGELAYOUTMODE_FACING) {
                        mPageMode = PDFViewCtrl.PAGELAYOUTMODE_FACING;
                        mViewValueListeners.get(IViewSettingsWindow.TYPE_SINGLE_PAGE).onValueChanged(IViewSettingsWindow.TYPE_SINGLE_PAGE, mPageMode);
                    }
                }

                mIvSingle.setSelected(false);
                mIvFacing.setSelected(true);
                mIvCover.setSelected(false);
                mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_Facing);
            } else if (id == R.id.iv_cover_page_icon_pad) {
                if (mViewValueListeners.get(IViewSettingsWindow.TYPE_SINGLE_PAGE) != null) {
                    if (mPageMode != PDFViewCtrl.PAGELAYOUTMODE_COVER) {
                        mPageMode = PDFViewCtrl.PAGELAYOUTMODE_COVER;
                        mViewValueListeners.get(IViewSettingsWindow.TYPE_SINGLE_PAGE).onValueChanged(IViewSettingsWindow.TYPE_SINGLE_PAGE, mPageMode);
                    }
                }
                mIvSingle.setSelected(false);
                mIvFacing.setSelected(false);
                mIvCover.setSelected(true);
                mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_CoverFacing);
            } else if (id == R.id.ll_continuous_page_button_pad) {
                if (mViewValueListeners.get(IViewSettingsWindow.TYPE_SINGLE_PAGE) != null) {
                    if (mContinuousPageButtonLayout.isSelected()) {
                        setSelectedButtonState(false, mContinuousPageButtonLayout, mIvContinuousPage);
                        mUiExtensionsManager.getPDFViewCtrl().setContinuous(false, true);
                    } else {
                        setSelectedButtonState(true, mContinuousPageButtonLayout, mIvContinuousPage);
                        mUiExtensionsManager.getPDFViewCtrl().setContinuous(true, true);
                    }
                    mPageMode = mUiExtensionsManager.getPDFViewCtrl().getPageLayoutMode();
                    mViewValueListeners.get(IViewSettingsWindow.TYPE_SINGLE_PAGE).onValueChanged(IViewSettingsWindow.TYPE_SINGLE_PAGE, mPageMode);
                }
                mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_ContinuousScrolling);
            } else if (id == R.id.ll_rtl_switch_layout_pad) {
                if (mViewValueListeners.get(IViewSettingsWindow.TYPE_RIGHT_TO_LEFT) != null) {
                    if (mRTL_SwitchLayout.isSelected()) {
                        setSelectedButtonState(false, mRTL_SwitchLayout, mRTL_SwitchIcon);
                        mUiExtensionsManager.getPDFViewCtrl().setPageBinding(PDFViewCtrl.LEFT_EDGE);
                    } else {
                        setSelectedButtonState(true, mRTL_SwitchLayout, mRTL_SwitchIcon);
                        mUiExtensionsManager.getPDFViewCtrl().setPageBinding(PDFViewCtrl.RIGHT_EDGE);
                    }
                    mViewValueListeners.get(IViewSettingsWindow.TYPE_RIGHT_TO_LEFT)
                            .onValueChanged(IViewSettingsWindow.TYPE_RIGHT_TO_LEFT, mUiExtensionsManager.getPDFViewCtrl().getPageBinding());
                }
                mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_Right_to_Left);
            }
        }
    };

    private ColorStateList getBorderColorStateList() {
        int disabled = AppResource.getColor(mContext, R.color.b2);
        int selected = ThemeConfig.getInstance(mContext).getPrimaryColor();
        int normal = AppResource.getColor(mContext, R.color.b2);
        return AppResource.createColorStateList(selected, disabled, normal);
    }

    private ColorStateList getSelectedButtonColorStateList() {
        int disabled = AppResource.getColor(mContext, R.color.p1);
        int selected = ThemeConfig.getInstance(mContext).getPrimaryColor();
        int normal = AppResource.getColor(mContext, R.color.p1);
        return AppResource.createColorStateList(selected, disabled, normal);
    }

    private void setSelectedButtonState(boolean isSelected, RelativeLayout switchLayout, ImageView switchIcon) {
        if (switchLayout == null || switchIcon == null) return;
        switchLayout.setSelected(isSelected);
        RelativeLayout.LayoutParams params = (RelativeLayout.LayoutParams) switchIcon.getLayoutParams();
        if (params == null) return;
        params.removeRule(isSelected ? RelativeLayout.ALIGN_PARENT_LEFT : RelativeLayout.ALIGN_PARENT_RIGHT);
        params.addRule(isSelected ? RelativeLayout.ALIGN_PARENT_RIGHT : RelativeLayout.ALIGN_PARENT_LEFT, RelativeLayout.TRUE);
        switchIcon.setLayoutParams(params);
    }

    private void showPageLayoutWindow(View btnView) {
        if (mViewSetting == null) {
            initializeViewSetting();
        }
        if (mPageMode != mUiExtensionsManager.getPDFViewCtrl().getPageLayoutMode()
                || mContinuousPageButtonLayout.isSelected() != mUiExtensionsManager.getPDFViewCtrl().isContinuous()) {
            mPageMode = mUiExtensionsManager.getPDFViewCtrl().getPageLayoutMode();
            setProperty(TYPE_SINGLE_PAGE, mPageMode);
        }
        createPopupWindow(0, btnView, mViewSetting);
    }

    private UIPopoverFrag mViewSettingWindow;
    private View mShowPopView;
    private View mShowContentView;
    private int mShowType;

    private void createPopupWindow(int type, View btnView, View contentView) {
        mUiExtensionsManager.stopHideToolbarsTimer();
        mShowPopView = btnView;
        mShowType = type;
        mShowContentView = contentView;
        Rect rect = new Rect();
        btnView.getGlobalVisibleRect(rect);
        contentView.measure(View.MeasureSpec.UNSPECIFIED, View.MeasureSpec.UNSPECIFIED);

        int width;
        if (type == 1) {
            width = Math.min(AppDisplay.getActivityWidth(), AppDisplay.getActivityHeight()) * 3 / 5;
            width = Math.max(width, mColorView.getMinWidth());
        } else {
            width = contentView.getMeasuredWidth() + 100;
        }

        int height = contentView.getMeasuredHeight();
        if (width < height) {
            width = Math.min(AppDisplay.getActivityWidth(), AppDisplay.getActivityHeight()) * 3 / 5;
        }
        int pos = UIPopoverWin.ARROW_AUTO;
        if (mOrientation == BaseBar.HORIZONTAL) {
            pos = UIPopoverWin.ARROW_TOP;
        } else {
            if (rect.left < AppDisplay.getActivityWidth() / 2) {
                pos = UIPopoverWin.ARROW_LEFT;
            } else {
                pos = UIPopoverWin.ARROW_RIGHT;
            }
        }
        initViewSettingWindow();
        mViewSettingWindow.showAtLocation(mUiExtensionsManager.getRootView(), rect, width, height, pos, 0);
    }

    //For page color
    private SimpleColorView mColorView;
    private static final int[] bgColors = new int[]{0xFFFFFFFF, 0xFFE8E8E8,
            0xFFF0E8D8, 0xFFCCC5B2,
            0xFF99CFA1, 0xFF2E5073,
            0xFF644F47, 0xFF9A9A9A,
    };

    private static final int[] textColors = {0xFF323232, 0xFF2B2B2B,
            0xFF323232, 0xFF5F3811,
            0xFF323232, 0xFF6D7E88,
            0xFF95938F, 0xFF000000};

    private void initializePageColor(Activity activity) {
        if (!mInitPageColor) {
            mInitPageColor = true;

            PDFViewCtrl pdfViewCtrl = mUiExtensionsManager.getPDFViewCtrl();
            if (pdfViewCtrl.getColorMode() == Renderer.e_ColorModeMappingGray
                    || pdfViewCtrl.getColorMode() == Renderer.e_ColorModeMapping) {
                if (isNightMode()) {
                    mPageColorMode = IViewSettingsWindow.NIGHT;
                } else {
                    mPageColorMode = IViewSettingsWindow.CUSTOM_COLOR;
                    mCustomPageColor = pdfViewCtrl.getMappingModeBackgroundColor();
                }
            }
        }

        int selectColor = 0xFFFFFFFF;
        if (mPageColorMode == IViewSettingsWindow.DAY || mPageColorMode == IViewSettingsWindow.NIGHT) {
            selectColor = Color.TRANSPARENT;
        } else if (mPageColorMode == IViewSettingsWindow.CUSTOM_COLOR) {
            if (mCustomPageColor != null)
                selectColor = mCustomPageColor;
        }
        mColorView = new SimpleColorView(activity, selectColor, bgColors);
        mColorView.getRootView().setBackgroundResource(R.drawable.dlg_bg_4circle_corner_10dp_grayf4f4f4);
        mColorView.setDragImageVisibility(View.INVISIBLE);
        mColorView.setPropertyChangeListener(mPageColorListener);
    }

    private PropertyBar.PropertyChangeListener mPageColorListener = new PropertyBar.PropertyChangeListener() {
        @Override
        public void onValueChanged(long property, int value) {
            mPageColorMode = IViewSettingsWindow.CUSTOM_COLOR;
            if (property == PropertyBar.PROPERTY_COLOR) {
                mCustomPageColor = value;
                PDFViewCtrl pdfViewCtrl = mUiExtensionsManager.getPDFViewCtrl();
                for (int i = 0; i < bgColors.length; i++) {
                    int color = bgColors[i];
                    if (color == value) {
                        pdfViewCtrl.setMappingModeForegroundColor(textColors[i]);
                        break;
                    }
                }
                pdfViewCtrl.setMappingModeBackgroundColor(value);
                pdfViewCtrl.setBackgroundColor(AppResource.getColor(mContext, R.color.ux_bg_color_docviewer));
                if (mCustomPageColor == Color.WHITE)
                    pdfViewCtrl.setColorMode(Renderer.e_ColorModeNormal);
                else
                    pdfViewCtrl.setColorMode(
                            mUIExtensionsManager.getPageColorMode() == UIExtensionsManager.NIGHTCOLORMODE_MAPPINGGRAY
                                    ? Renderer.e_ColorModeMappingGray : Renderer.e_ColorModeMapping);
            }
            mViewValueListeners.get(IViewSettingsWindow.TYPE_DAY).onValueChanged(IViewSettingsWindow.TYPE_DAY, mPageColorMode);
        }

        @Override
        public void onValueChanged(long property, float value) {

        }

        @Override
        public void onValueChanged(long property, String value) {

        }
    };

    public void setPageBgColor(long property, int value) {
        mPageColorMode = IViewSettingsWindow.CUSTOM_COLOR;
        if (property == PropertyBar.PROPERTY_COLOR) {
            mCustomPageColor = value;
            PDFViewCtrl pdfViewCtrl = mUiExtensionsManager.getPDFViewCtrl();
            for (int i = 0; i < bgColors.length; i++) {
                int color = bgColors[i];
                if (color == value) {
                    pdfViewCtrl.setMappingModeForegroundColor(textColors[i]);
                    break;
                }
            }
            pdfViewCtrl.setMappingModeBackgroundColor(value);
            pdfViewCtrl.setBackgroundColor(AppResource.getColor(mContext, R.color.ux_bg_color_docviewer));
            if (mCustomPageColor == Color.WHITE)
                pdfViewCtrl.setColorMode(Renderer.e_ColorModeNormal);
            else
                pdfViewCtrl.setColorMode(
                        mUIExtensionsManager.getPageColorMode() == UIExtensionsManager.NIGHTCOLORMODE_MAPPINGGRAY
                                ? Renderer.e_ColorModeMappingGray : Renderer.e_ColorModeMapping);
        }
        mViewValueListeners.get(IViewSettingsWindow.TYPE_DAY).onValueChanged(IViewSettingsWindow.TYPE_DAY, mPageColorMode);
    }

    private void showPageColorWindow(View btnView) {
        mUiExtensionsManager.stopHideToolbarsTimer();
        if (mColorView == null) {
            initializePageColor(mUIExtensionsManager.getAttachedActivity());
        }
        createPopupWindow(1, btnView, mColorView.getRootView());
    }


    private void showPopupWindow(int property, View btnView) {
        if (property == IViewSettingsWindow.TYPE_SINGLE_PAGE) {
            showPageLayoutWindow(btnView);
        } else if (property == IViewSettingsWindow.TYPE_PAGE_COLOR) {
            showPageColorWindow(btnView);
        }
    }

    public void reInit() {
        if (mViewSettingWindow != null) {
            mViewSettingWindow.dismiss();
            mViewSettingWindow = null;
        }
        if (mColorView != null) {
            mColorView = null;
        }
        if (mViewSetting != null) {
            int[] viewSettingIds = {R.id.ll_single_page_pad, R.id.ll_facing_page_pad, R.id.ll_cover_page_pad,
                    R.id.ll_continuous_page_pad, R.id.ll_rtl_pad};
            SparseIntArray pageViewVisibles = new SparseIntArray();
            for (int id : viewSettingIds) {
                pageViewVisibles.put(id, mViewSetting.findViewById(id).getVisibility());
            }

            initializeViewSetting();

            for (int i = 0; i < pageViewVisibles.size(); i++) {
                int id = pageViewVisibles.keyAt(i);
                int visibility = pageViewVisibles.valueAt(i);
                mViewSetting.findViewById(id).setVisibility(visibility);
            }
        }
    }

    public void onConfigurationChanged() {
        if (mColorView != null) {
            mColorView.reset();
        }
    }

    @Override
    public void onLayoutChange() {
        if (mViewSettingWindow != null && mViewSettingWindow.isShowing()) {
            if (AppDevice.isChromeOs(mUiExtensionsManager.getAttachedActivity()) && mShowType == 1)
                mColorView.reset();

            int delayMillis = mOrientation == BaseBar.HORIZONTAL ? 0 : 400;
            AppThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
                @Override
                public void run() {
                    updateSettingPop();
                }
            }, delayMillis);
        }
    }

    private void updateSettingPop() {
        Rect rect = new Rect();
        mShowPopView.getGlobalVisibleRect(rect);
        mShowContentView.measure(View.MeasureSpec.UNSPECIFIED, View.MeasureSpec.UNSPECIFIED);

        int width;
        if (mShowType == 1) {
            width = Math.min(AppDisplay.getActivityWidth(), AppDisplay.getActivityHeight()) * 3 / 5;
            width = Math.max(width, mColorView.getMinWidth());
        } else {
            width = mShowContentView.getMeasuredWidth() + 100;
        }
        int height = mShowContentView.getMeasuredHeight();
        if (width < height) {
            width = Math.min(AppDisplay.getActivityWidth(), AppDisplay.getActivityHeight()) * 3 / 5;
        }
        initViewSettingWindow();
        mViewSettingWindow.update(mUiExtensionsManager.getRootView(), rect, width, height);
    }

//    @Override
//    public void setToolItems(MainCenterItemBean item) {
//        //todo
//    }
//
//    @Override
//    public MainCenterItemBean getToolItems() {
//        //todo
//        return null;
//    }
//
//    @Override
//    public void setToolBarEnabled(boolean enabled) {
//        //todo
//    }
//
//    @Override
//    public boolean isToolBarEnabled() {
//        //todo
//        return false;
//    }
//
//    @Override
//    public void onThemeColorChanged() {
//    }

    private boolean isNightMode() {
        PDFViewCtrl mPdfView = mUiExtensionsManager.getPDFViewCtrl();
        if (mPdfView.getColorMode() == Renderer.e_ColorModeMappingGray || mPdfView.getColorMode() == Renderer.e_ColorModeMapping) {
            if (mPdfView.getMappingModeBackgroundColor() == IViewSettingsWindow.DEFAULT_NIGHT_BG_COLOR
                    && mPdfView.getMappingModeForegroundColor() == IViewSettingsWindow.DEFAULT_NIGHT_FG_COLOR) {
                return true;
            }
        }
        return false;
    }

    private void initViewSettingWindow() {
        if (mViewSettingWindow != null) return;
        mViewSettingWindow = UIPopoverFrag.create((FragmentActivity) mUiExtensionsManager.getAttachedActivity(), mShowContentView, false, false);
        if (0 == mShowType)//page layout
            mViewSettingWindow.setArrowColor(AppResource.getColor(mContext, R.color.b1));
        else //page color
            mViewSettingWindow.setArrowColor(AppResource.getColor(mContext, R.color.b2));
        mViewSettingWindow.setOnDismissListener(new PopupWindow.OnDismissListener() {
            @Override
            public void onDismiss() {
                mUiExtensionsManager.startHideToolbarsTimer();
                AppUtil.removeViewFromParent(mShowContentView);
                mViewSettingWindow = null;
            }
        });
    }
}
