/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.toolbar.drag;

import android.content.Context;
import android.content.res.ColorStateList;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffColorFilter;
import android.graphics.RectF;
import android.view.Gravity;
import android.widget.ImageView;
import android.widget.LinearLayout;

import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.common.UIFillView;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.controls.toolbar.IBaseItem;
import com.foxit.uiextensions.controls.toolbar.IToolBar;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.UIToolView;
import com.foxit.uiextensions.controls.toolbar.impl.BaseBarImpl;
import com.foxit.uiextensions.controls.toolbar.impl.BaseItemImpl;
import com.foxit.uiextensions.controls.toolbar.impl.UIColorItem;
import com.foxit.uiextensions.modules.UndoModule;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

import androidx.annotation.IntDef;
import androidx.appcompat.widget.AppCompatImageView;

public abstract class UIToolBaseBar extends BaseBarImpl implements IToolBar {

    public static final int USAGE_TYPE_HOME = 0;
    public static final int USAGE_TYPE_EDIT = 1;
    public static final int USAGE_TYPE_COMMENT = 2;
    public static final int USAGE_TYPE_DRAW = 3;
    public static final int USAGE_TYPE_VIEW = 4;
    public static final int USAGE_TYPE_FORM = 5;
    public static final int USAGE_TYPE_SIGN = 6;
    public static final int USAGE_TYPE_PROTECT = 7;

    @IntDef({
            USAGE_TYPE_HOME,
            USAGE_TYPE_EDIT,
            USAGE_TYPE_COMMENT,
            USAGE_TYPE_DRAW,
            USAGE_TYPE_VIEW,
            USAGE_TYPE_FORM,
            USAGE_TYPE_SIGN,
            USAGE_TYPE_PROTECT
    })
    @Retention(RetentionPolicy.SOURCE)
    public @interface UsageType {
    }

    UIExtensionsManager mUiExtensionsManager;

    int mUsageType = USAGE_TYPE_HOME;
    boolean mIsSmallPad = false;
    float mDividerHeight;
    float mDividerWidth;

    //divider
    IBaseItem mDividerItem;
    ImageView mDividerView;
    IBaseItem mDragItem;
    UIColorItem mPropItem;
    UIFillView mPropView;
    LinearLayout mPropViewContainer;
    UIColorItem mUndoItem;
    UIColorItem mRedoItem;
    UIColorItem mAddItem;
    public UIColorItem mSortItem;
    UIToolView mMultiToolView;
    UIColorItem mMultiSelectItem;
    boolean mEnabled = false;

    int mLayoutWidth;
    int mLayoutHeight;

    public UIToolBaseBar(Context context, UIExtensionsManager uiExtensionsManager) {
        this(context, BaseBar.HORIZONTAL, uiExtensionsManager);
    }

    protected UIToolBaseBar(Context context, int orientation, UIExtensionsManager uiExtensionsManager) {
        this(context, orientation, false, uiExtensionsManager);
    }

    protected UIToolBaseBar(Context context, int orientation, boolean uniformSpace, UIExtensionsManager uiExtensionsManager) {
        this(context, orientation, 0, 0, uniformSpace, uiExtensionsManager);
    }

    protected UIToolBaseBar(Context context, int orientation, int width, int height, boolean uniformSpace, UIExtensionsManager uiExtensionsManager) {
        super(context, orientation, width, height, uniformSpace);
        mUiExtensionsManager = uiExtensionsManager;
        initialize(context, orientation);
    }

    private void initialize(Context context, int orientation) {
        if (orientation == BaseBar.HORIZONTAL) {
            mDividerHeight = AppDisplay.dp2px(20);
            mDividerWidth = AppDisplay.dp2px(1f);
        } else {
            mDividerHeight = AppDisplay.dp2px(1f);
            mDividerWidth = AppDisplay.dp2px(20);
        }

        if (AppDisplay.isPad()) {
            setStartMargin(AppResource.getDimensionPixelSize(context, R.dimen.ux_screen_margin_icon));
            setEndMargin(AppResource.getDimensionPixelSize(context, R.dimen.ux_screen_margin_icon));

            int scnWidth = AppDisplay.getActivityWidth(); //AppDisplay.getDisplayWidth(); for Chrome OS
            int scnHeight = AppDisplay.getActivityHeight(); //AppDisplay.getDisplayHeight(); for Chrome OS
            int length = Math.min(scnWidth, scnHeight);
            int minLength = AppDisplay.dp2px(768); // 16 * 40 = 640
            if (length < minLength) {
                mIsSmallPad = true;
            }
        } else {
            setStartMargin(AppResource.getDimensionPixelSize(context, R.dimen.ux_screen_margin_text));
            setEndMargin(AppResource.getDimensionPixelSize(context, R.dimen.ux_screen_margin_text));
            setItemInterval(AppResource.getDimensionPixelSize(context, R.dimen.ux_bottombar_button_space_phone));
            setAutoCompressItemsInterval(true);
        }
    }

    @UsageType
    public abstract int getUsageType();

    @Override
    public void setOrientation(int orientation) {
        super.setOrientation(orientation);
        layout(orientation, mLayoutWidth, mLayoutHeight);
    }

    @Override
    public void setOrientation(int orientation, int width, int height) {
        super.setOrientation(orientation, width, height);
        layout(orientation, width, height);
    }

    protected void layout(int orientation, int width, int height) {
        mLayoutWidth = width;
        mLayoutHeight = height;

        if (orientation == BaseBar.HORIZONTAL) {
            mDividerHeight = AppDisplay.dp2px(20);
            mDividerWidth = AppDisplay.dp2px(1f);
        } else {
            mDividerHeight = AppDisplay.dp2px(1f);
            mDividerWidth = AppDisplay.dp2px(20);
        }

        if (mDividerView != null) {
            mDividerView.setMinimumWidth((int) (mDividerWidth + 2));
            mDividerView.setMinimumHeight((int) (mDividerHeight + 2));
            mDividerView.invalidate();
        }
    }

    IBaseItem createDividerItem() {
        if (mDividerView == null) {
            mDividerView = new AppCompatImageView(mContext) {
                Paint mPaint = new Paint();
                RectF mRect = new RectF();

                @Override
                protected void onDraw(Canvas canvas) {
                    super.onDraw(canvas);

                    mPaint.setDither(true);
                    mPaint.setAntiAlias(true);
                    mPaint.setColor(mContext.getResources().getColor(R.color.p3));
                    mPaint.setStyle(Paint.Style.FILL);

                    int width = getWidth();
                    int height = getHeight();

                    mRect.set((width - mDividerWidth) / 2, (height - mDividerHeight) / 2,
                            (width + mDividerWidth) / 2, (height + mDividerHeight) / 2);
                    canvas.drawRect(mRect, mPaint);
                }
            };

            mDividerView.setMinimumWidth((int) (mDividerWidth + 2));
            mDividerView.setMinimumHeight((int) (mDividerHeight + 2));
            mDividerItem = new BaseItemImpl(mContext, mDividerView);
            mDividerItem.setTag(ToolConstants.Divider);
        }
        return mDividerItem;
    }

    UIColorItem createPropButtonItem() {
        if (mPropView == null) {
            mPropView = new UIFillView(mContext);
            mPropView.setForceDarkAllowed(false);
            mPropView.setFillResource(R.drawable.annot_prop_circle_border_bg);
            mPropView.setBorderResource(R.drawable.annot_prop_circle_border_bg);
            mPropView.setBorderWidth(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_item_border_unselected_width));
            mPropView.setBorderColor(AppResource.getColor(mContext, R.color.i2));
            mPropView.setFillColorFilter(AppResource.getColor(mContext, R.color.b2));

            mPropViewContainer = new LinearLayout(mContext);
            mPropViewContainer.setGravity(Gravity.CENTER);
            mPropViewContainer.addView(mPropView);
            mPropViewContainer.setBackgroundResource(R.drawable.tool_bar_drop_right);
            mPropItem = new UIColorItem(mContext, mPropViewContainer);
            mPropItem.setTag(ToolConstants.Property);
//            mPropItem.setBackgroundResource(R.drawable.tool_bar_drop_right);
            if (mPropViewContainer.getBackground() != null)
                mPropViewContainer.getBackground().setColorFilter(new PorterDuffColorFilter(
                        AppResource.getColor(mContext, R.color.i2), PorterDuff.Mode.SRC_IN));
        }
        return mPropItem;
    }

    UIColorItem createAddButtonItem() {
        if (mAddItem == null) {
            mAddItem = new UIColorItem(mContext,R.drawable.tool_add_normal);
            mAddItem.setId(R.id.id_at_additem);
            mAddItem.setTag(ToolConstants.Add);
            ColorStateList colorStateList = ColorStateList.valueOf(AppResource.getColor(mContext, R.color.i3));
            mAddItem.setImageTintList(colorStateList);
        }
        return mAddItem;
    }

    UIColorItem createSortButtonItem() {
        if (mSortItem == null) {
            mSortItem = new UIColorItem(mContext,R.drawable.tool_sort_normal);
            mSortItem.setId(R.id.id_at_sortitem);
            mSortItem.setTag(ToolConstants.sort);
            ColorStateList colorStateList = ColorStateList.valueOf(AppResource.getColor(mContext, R.color.i3));
            mSortItem.setImageTintList(colorStateList);
        }
        return mSortItem;
    }

    UIColorItem createUndoButtonItem() {
        if (mUndoItem == null) {
            UndoModule undoModule = (UndoModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_UNDO);
            if (undoModule != null)
                mUndoItem = undoModule.createUndoButtonItem();
        }
        return mUndoItem;
    }

    UIColorItem createRedoButtonItem() {
        if (mRedoItem == null) {
            UndoModule undoModule = (UndoModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_UNDO);
            mRedoItem = undoModule.createRedoButtonItem();
        }
        return mRedoItem;
    }

    UIColorItem createMultiSelectItem() {
        if (mMultiSelectItem == null) {
            mMultiToolView = new UIToolView(mContext);
            mMultiToolView.setFillBackgroundResource(R.drawable.common_tool_multi_select);
            mMultiToolView.setBackgroundColorFilter(AppResource.getColor(mContext, R.color.i3));
            mMultiToolView.setForceDarkAllowed(false);
            mMultiSelectItem = new UIColorItem(mContext, mMultiToolView);
            mMultiSelectItem.setTag(ToolConstants.Multi_Select);
            mMultiSelectItem.setId(R.id.id_comment_multiselect);
        }
        return mMultiSelectItem;
    }

    public void addDragItem() {
    }

    @Override
    public void setEnabled(boolean enabled) {
        mEnabled = enabled;
    }

    @Override
    public boolean isEnabled() {
        return mEnabled;
    }

    public void onLayoutChange() {
    }
    public UIFillView getPropView(){
        return mPropView;
    }
}