/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.toolbar.drag;


import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.content.res.ColorStateList;
import android.graphics.Color;
import android.graphics.Point;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffColorFilter;
import android.graphics.Rect;
import android.graphics.RectF;
import android.os.Build;
import android.view.Gravity;
import android.view.KeyEvent;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewParent;
import android.widget.FrameLayout;
import android.widget.HorizontalScrollView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.ScrollView;

import androidx.annotation.NonNull;

import com.foxit.uiextensions.IInteractionEventListener;
import com.foxit.uiextensions.IPermissionProvider;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.multiselect.MultiSelectModule;
import com.foxit.uiextensions.annots.multiselect.MultiSelectToolHandler;
import com.foxit.uiextensions.annots.stamp.StampConstants;
import com.foxit.uiextensions.annots.stamp.StampUtil;
import com.foxit.uiextensions.config.uisettings.UISettingsConfig;
import com.foxit.uiextensions.controls.dialog.UIPopoverWin;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.propertybar.imp.PropertyBarImpl;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.controls.toolbar.IBaseItem;
import com.foxit.uiextensions.controls.toolbar.IToolSupply;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolItemBean;
import com.foxit.uiextensions.controls.toolbar.ToolItemsManager;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.UIToolView;
import com.foxit.uiextensions.controls.toolbar.impl.BaseBarImpl;
import com.foxit.uiextensions.controls.toolbar.impl.UIColorItem;
import com.foxit.uiextensions.pdfreader.MainCenterItemBean;
import com.foxit.uiextensions.pdfreader.impl.MainFrame;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppSharedPreferences;
import com.foxit.uiextensions.utils.AppUtil;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

public abstract class UIHSToolBar extends UIToolBaseBar {
    static final String SP_TOOLS_LIST = "SP_TOOLS_LIST";

    MainCenterItemBean mCenterItemBean;
    List<ToolItemBean> mToolsList;
    AppSharedPreferences mSp;

    View mRootView;
    LinearLayout mOptionToolsView;
    LinearLayout mHSToolsView;
    HorizontalScrollView mHSView;
    ScrollView mVSView;
    LinearLayout mVSToolsView;

    Context mContext;
    UIDragToolBar mUIToolbar;
    BaseBarImpl mToolsbar;
    BaseBarImpl mOptionsToolbar;

    private PropertyBar mPropertyBar;
    private IToolSupply mCurToolApply;
    private ToolItemBean mCurToolBean;
    UIColorItem mCurItem;
    UIExtensionsManager mUIExtensionsManager;

    private View mHSViewStartMask;
    private View mHSViewEndMask;
    private View mVSViewStartMask;
    private View mVSViewEndMask;

    private int mScreenOrientation;

    public UIHSToolBar(Context context, UIExtensionsManager uiExtensionsManager, UIDragToolBar uiToolbar) {
        this(context, UIToolBaseBar.HORIZONTAL, uiExtensionsManager, uiToolbar);
    }

    protected UIHSToolBar(Context context, int orientation, UIExtensionsManager uiExtensionsManager, UIDragToolBar uiToolbar) {
        this(context, orientation, false, uiExtensionsManager, uiToolbar);
    }

    protected UIHSToolBar(Context context, int orientation, boolean uniformSpace, UIExtensionsManager uiExtensionsManager, UIDragToolBar uiToolbar) {
        this(context, orientation, 0, 0, uniformSpace, uiExtensionsManager, uiToolbar);
    }

    protected UIHSToolBar(Context context, int orientation, int width, int height, boolean uniformSpace, UIExtensionsManager uiExtensionsManager, UIDragToolBar uiToolbar) {
        super(context, orientation, width, height, uniformSpace, uiExtensionsManager);
        mUIExtensionsManager = uiExtensionsManager;
        initialize(context, orientation, uiToolbar);
    }

    void initialize(Context context, int orientation, UIDragToolBar uiToolbar) {
        mUsageType = getUsageType();
        mUIToolbar = uiToolbar;
        mOrientation = orientation;
        mScreenOrientation = context.getResources().getConfiguration().orientation;
        mContext = context;
        mToolsList = new ArrayList<>();
        mSp = AppSharedPreferences.getInstance(mContext);
        mUiExtensionsManager.registerToolHandlerChangedListener(mToolHandlerChangedListener);
        mUiExtensionsManager.registerInteractionListener(mInteractionEventListener);
        mUIExtensionsManager.registerThemeEventListener(mThemeEventListener);

        int backgroundColor = mUIToolbar.getBackgroundColorProvider().getColor(AppUtil.isDarkMode(mContext));
        mRootView = View.inflate(mContext, R.layout.toolbar_layout, null);
        mRootView.setBackgroundColor(backgroundColor);
        mHSView = mRootView.findViewById(R.id.toolbar_hs_view);
        mHSViewStartMask = mRootView.findViewById(R.id.hs_view_start_mask);
        mHSViewEndMask = mRootView.findViewById(R.id.hs_view_end_mask);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            mHSViewEndMask.setForceDarkAllowed(false);
            mHSViewStartMask.setForceDarkAllowed(false);
        }
        ThemeUtil.setTintList(mHSViewEndMask, getScrollViewMaskColor());
        ThemeUtil.setTintList(mHSViewStartMask, getScrollViewMaskColor());

        mHSToolsView = mRootView.findViewById(R.id.tool_hs_container);
        mVSView = mRootView.findViewById(R.id.toolbar_vs_view);
        mVSViewStartMask = mRootView.findViewById(R.id.vs_view_start_mask);
        mVSViewEndMask = mRootView.findViewById(R.id.vs_view_end_mask);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            mVSViewStartMask.setForceDarkAllowed(false);
            mVSViewEndMask.setForceDarkAllowed(false);
        }
        ThemeUtil.setTintList(mVSViewStartMask, getScrollViewMaskColor());
        ThemeUtil.setTintList(mVSViewEndMask, getScrollViewMaskColor());

        mVSToolsView = mRootView.findViewById(R.id.tool_vs_container);
        mOptionToolsView = mRootView.findViewById(R.id.tool_right_container);

        mToolsbar = new BaseBarImpl(mContext);
        mToolsbar.setMiddleButtonCenter(true);
        mToolsbar.setAutoCompressItemsInterval(true);
        mToolsbar.setStartMargin(AppResource.getDimensionPixelSize(context, R.dimen.ux_margin_8dp));
        mToolsbar.setEndMargin(AppResource.getDimensionPixelSize(context, R.dimen.ux_margin_8dp));
        mToolsbar.setBackgroundColor(Color.TRANSPARENT);

        mOptionsToolbar = new BaseBarImpl(mContext);
        mOptionsToolbar.setAutoCompressItemsInterval(true);
        setOptionToolsViewHorizontalOffset(AppResource.getDimensionPixelSize(context, R.dimen.ux_margin_8dp));
        mOptionsToolbar.setStartMargin(0);
        mOptionsToolbar.setBackgroundColor(Color.TRANSPARENT);
        if (mOrientation == UIToolBaseBar.HORIZONTAL) {
            mToolsbar.setOrientation(BaseBar.HORIZONTAL);
            mVSView.setVisibility(View.GONE);
            mHSView.setVisibility(View.VISIBLE);
            mHSView.setOnTouchListener(mScrollTouchListener);

            RelativeLayout.LayoutParams optionParams = (RelativeLayout.LayoutParams) mOptionToolsView.getLayoutParams();
            optionParams.addRule(RelativeLayout.ALIGN_PARENT_END);
            mOptionToolsView.setLayoutParams(optionParams);
            mOptionToolsView.setGravity(Gravity.END);
            mOptionToolsView.setOrientation(LinearLayout.HORIZONTAL);
            mOptionToolsView.addView(mOptionsToolbar.getContentView());

            LinearLayout.LayoutParams layoutParams = new LinearLayout.LayoutParams(LinearLayout.LayoutParams.MATCH_PARENT, LinearLayout.LayoutParams.WRAP_CONTENT);
            mHSToolsView.addView(mToolsbar.getContentView(), layoutParams);
            mHSView.addOnLayoutChangeListener(mScrollViewLayoutChangeListener);
        } else {
            mOptionsToolbar.setEndMargin(0);
            mToolsbar.setOrientation(BaseBar.VERTICAL);
            mHSView.setVisibility(View.GONE);
            mVSView.setVisibility(View.VISIBLE);

            RelativeLayout.LayoutParams optionParams = (RelativeLayout.LayoutParams) mOptionToolsView.getLayoutParams();
            optionParams.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
            mOptionToolsView.setLayoutParams(optionParams);
            mOptionToolsView.setGravity(Gravity.BOTTOM);
            mOptionToolsView.setOrientation(LinearLayout.VERTICAL);
            mOptionToolsView.addView(mOptionsToolbar.getContentView());

            LinearLayout.LayoutParams layoutParams = new LinearLayout.LayoutParams(LinearLayout.LayoutParams.WRAP_CONTENT, LinearLayout.LayoutParams.MATCH_PARENT);
            mVSToolsView.addView(mToolsbar.getContentView(), layoutParams);
        }
        mVSView.setOnTouchListener(mScrollTouchListener);
        mVSView.addOnLayoutChangeListener(mScrollViewLayoutChangeListener);
    }

    private final View.OnLayoutChangeListener mScrollViewLayoutChangeListener = new View.OnLayoutChangeListener() {
        @Override
        public void onLayoutChange(View v, int left, int top, int right, int bottom, int oldLeft, int oldTop, int oldRight, int oldBottom) {
            final View scrollView = v;
            v.postDelayed(new Runnable() {
                @Override
                public void run() {
                    updateScrollViewMask(scrollView);
                }
            }, 16);
        }
    };

    @Override
    public void setOrientation(int orientation) {
        mOrientation = orientation;
        layout(orientation, mLayoutWidth, mLayoutHeight);
    }

    @Override
    public void setOrientation(int orientation, int width, int height) {
        mOrientation = orientation;
        layout(orientation, width, height);
    }

    @Override
    protected void layout(int orientation, int width, int height) {
        super.layout(orientation, width, height);

        if (mPropItem != null) {
            if (mUIToolbar.getToolBarPosition() == UIDragToolBar.TOOLBAR_POS_RIGHT)
                mPropViewContainer.setBackgroundResource(R.drawable.tool_bar_drop_left);
            else
                mPropViewContainer.setBackgroundResource(R.drawable.tool_bar_drop_right);
            if (mPropViewContainer.getBackground() != null)
                mPropViewContainer.getBackground().setColorFilter(new PorterDuffColorFilter(
                        AppResource.getColor(mContext, R.color.i2), PorterDuff.Mode.SRC_IN));
        }

        if (mAddItem != null) {
            if (mToolsList.size() == 0) {
                if (mUIToolbar.getToolBarPosition() == UIDragToolBar.TOOLBAR_POS_LEFT
                        || mUIToolbar.getToolBarPosition() == UIDragToolBar.TOOLBAR_POS_RIGHT)
                    mAddItem.setText("");
                else
                    mAddItem.setText(AppResource.getString(mContext, R.string.add_tools_title));
            } else {
                mAddItem.setText("");
            }
        }

        _changeTextStatus(orientation);

        if (orientation == UIToolBaseBar.HORIZONTAL) {
            mVSView.setVisibility(View.GONE);
            mHSView.setVisibility(View.VISIBLE);
            mToolsbar.setOrientation(BaseBar.HORIZONTAL, RelativeLayout.LayoutParams.MATCH_PARENT,
                    Math.max(height, AppResource.getDimensionPixelSize(mContext, R.dimen.ui_bottombar_height)));
            mOptionsToolbar.setOrientation(BaseBar.HORIZONTAL, RelativeLayout.LayoutParams.MATCH_PARENT,
                    Math.max(height, AppResource.getDimensionPixelSize(mContext, R.dimen.ui_bottombar_height)));

            ViewParent parentView = mRootView.getParent();
            if (parentView == null) {
                RelativeLayout.LayoutParams rootLp = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.MATCH_PARENT,
                        RelativeLayout.LayoutParams.WRAP_CONTENT);
                mRootView.setLayoutParams(rootLp);
            } else if (parentView instanceof RelativeLayout) {
                RelativeLayout.LayoutParams rootLp = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.MATCH_PARENT,
                        RelativeLayout.LayoutParams.WRAP_CONTENT);
                mRootView.setLayoutParams(rootLp);
            } else if (parentView instanceof LinearLayout) {
                LinearLayout.LayoutParams rootLp = new LinearLayout.LayoutParams(LinearLayout.LayoutParams.MATCH_PARENT,
                        LinearLayout.LayoutParams.WRAP_CONTENT);
                mRootView.setLayoutParams(rootLp);
            }

            RelativeLayout.LayoutParams optionParams = (RelativeLayout.LayoutParams) mOptionToolsView.getLayoutParams();
            optionParams.addRule(RelativeLayout.ALIGN_PARENT_END);
            mOptionToolsView.setLayoutParams(optionParams);
            mOptionToolsView.setGravity(Gravity.END);
            mOptionToolsView.setOrientation(LinearLayout.HORIZONTAL);

            Point centerSize = mToolsbar.measureSize();
            Point rightSize = mOptionsToolbar.measureSize();
//            if (AppDisplay.isPad()) {
            View containerView = mUiExtensionsManager.getRootView();
            int rootWidth;
            if (containerView == null || !containerView.isShown()) {
                rootWidth = AppDisplay.getActivityWidth();
            } else {
                rootWidth = containerView.getWidth();
            }
            RelativeLayout.LayoutParams hsParams = (RelativeLayout.LayoutParams) mHSView.getLayoutParams();
            hsParams.width = RelativeLayout.LayoutParams.WRAP_CONTENT;
            hsParams.height = RelativeLayout.LayoutParams.WRAP_CONTENT;
            if ((centerSize.x / 2 + rightSize.x + mOptionToolsViewHorizontalOffset) > rootWidth / 2) {
                hsParams.addRule(RelativeLayout.LEFT_OF, mOptionToolsView.getId());
                hsParams.removeRule(RelativeLayout.CENTER_HORIZONTAL);
            } else {
                hsParams.removeRule(RelativeLayout.LEFT_OF);
                hsParams.addRule(RelativeLayout.CENTER_HORIZONTAL);
            }
            mHSView.setLayoutParams(hsParams);
//            } else {
//                RelativeLayout.LayoutParams hsParams = (RelativeLayout.LayoutParams) mHSView.getLayoutParams();
//                hsParams.width = RelativeLayout.LayoutParams.MATCH_PARENT;
//                hsParams.height = RelativeLayout.LayoutParams.WRAP_CONTENT;
//                if (mToolsbar.getItemsCount(TB_Position.Position_CENTER) > 0) {
//                    FrameLayout.LayoutParams centerParams = (FrameLayout.LayoutParams) mHSToolsView.getLayoutParams();
//                    centerParams.gravity = Gravity.CENTER;
//                    mHSToolsView.setLayoutParams(centerParams);
//                    hsParams.removeRule(RelativeLayout.LEFT_OF);
//                } else {
//                    FrameLayout.LayoutParams centerParams = (FrameLayout.LayoutParams) mHSToolsView.getLayoutParams();
//                    centerParams.gravity = Gravity.START;
//                    mHSToolsView.setLayoutParams(centerParams);
//                    hsParams.addRule(RelativeLayout.LEFT_OF, R.id.tool_right_containter);
//                }
//                mHSView.setLayoutParams(hsParams);
//            }

            AppUtil.removeViewFromParent(mToolsbar.getContentView());
            mHSToolsView.removeAllViews();
            mHSToolsView.addView(mToolsbar.getContentView());

            RelativeLayout.LayoutParams rightParams = (RelativeLayout.LayoutParams) mOptionToolsView.getLayoutParams();
            rightParams.width = rightSize.x + mOptionToolsViewHorizontalOffset;
            rightParams.height = RelativeLayout.LayoutParams.WRAP_CONTENT;
            mOptionToolsView.setLayoutParams(rightParams);
        } else {
            mHSView.setVisibility(View.GONE);
            mHSViewStartMask.setVisibility(mHSView.getVisibility());
            mHSViewEndMask.setVisibility(mHSView.getVisibility());
            mVSView.setVisibility(View.VISIBLE);

            mToolsbar.setOrientation(BaseBar.VERTICAL, Math.max(width, AppResource.getDimensionPixelSize(mContext,
                    R.dimen.ui_bottombar_height)), RelativeLayout.LayoutParams.WRAP_CONTENT);
            mOptionsToolbar.setOrientation(BaseBar.VERTICAL, Math.max(width, AppResource.getDimensionPixelSize(mContext,
                    R.dimen.ui_bottombar_height)), RelativeLayout.LayoutParams.WRAP_CONTENT);

            ViewParent parentView = mRootView.getParent();
            if (parentView == null) {
                RelativeLayout.LayoutParams rootLp = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.WRAP_CONTENT,
                        RelativeLayout.LayoutParams.MATCH_PARENT);
                mRootView.setLayoutParams(rootLp);
            } else if (parentView instanceof RelativeLayout) {
                RelativeLayout.LayoutParams rootLp = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.WRAP_CONTENT,
                        RelativeLayout.LayoutParams.MATCH_PARENT);
                mRootView.setLayoutParams(rootLp);
            } else if (parentView instanceof LinearLayout) {
                LinearLayout.LayoutParams rootLp = new LinearLayout.LayoutParams(LinearLayout.LayoutParams.WRAP_CONTENT,
                        LinearLayout.LayoutParams.MATCH_PARENT);
                mRootView.setLayoutParams(rootLp);
            }

            Point centerSize = mToolsbar.measureSize();
            Point rightSize = mOptionsToolbar.measureSize();

            RelativeLayout.LayoutParams optionParams = (RelativeLayout.LayoutParams) mOptionToolsView.getLayoutParams();
            optionParams.width = RelativeLayout.LayoutParams.WRAP_CONTENT;
            optionParams.height = rightSize.y + mOptionToolsViewHorizontalOffset;
            optionParams.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
            mOptionToolsView.setLayoutParams(optionParams);
            mOptionToolsView.setGravity(Gravity.BOTTOM);
            mOptionToolsView.setOrientation(LinearLayout.VERTICAL);
            if (mOptionsToolbar.getEndMargin() != 0) {
                mOptionsToolbar.setEndMargin(0);
            }

            RelativeLayout.LayoutParams vsParams = (RelativeLayout.LayoutParams) mVSView.getLayoutParams();
            vsParams.width = RelativeLayout.LayoutParams.WRAP_CONTENT;
            vsParams.height = RelativeLayout.LayoutParams.WRAP_CONTENT;
            boolean isAlignCenter = false;
            if (centerSize.y > (getToolBarHeight() - rightSize.y - mOptionToolsViewHorizontalOffset)) {
                vsParams.addRule(RelativeLayout.ABOVE, mOptionToolsView.getId());
                vsParams.removeRule(RelativeLayout.ALIGN_PARENT_TOP);
            } else {
                vsParams.addRule(RelativeLayout.ABOVE, mOptionToolsView.getId());
                vsParams.height = RelativeLayout.LayoutParams.MATCH_PARENT;
                isAlignCenter = true;
            }
            mVSView.setLayoutParams(vsParams);

            AppUtil.removeViewFromParent(mToolsbar.getContentView());
            mVSToolsView.removeAllViews();
            mVSToolsView.addView(mToolsbar.getContentView());
            if (mVSToolsView.getLayoutParams() instanceof FrameLayout.LayoutParams) {
                ((FrameLayout.LayoutParams) mVSToolsView.getLayoutParams()).gravity = isAlignCenter ? Gravity.CENTER : -1;
            }
        }

        int newOrientation = mContext.getResources().getConfiguration().orientation;
        if (mScreenOrientation != newOrientation) {
            mScreenOrientation = newOrientation;
            if (mCurItem != null) {
                final View view = mCurItem.getContentView();
                if (mOrientation == BaseBar.HORIZONTAL) {
                    mHSView.post(new Runnable() {
                        @Override
                        public void run() {
                            mHSView.smoothScrollTo((int) view.getX(), 0);
                        }
                    });
                } else {
                    mVSView.post(new Runnable() {
                        @Override
                        public void run() {
                            mVSView.smoothScrollTo(0, (int) view.getY());
                        }
                    });
                }
            }
        }
    }

    private void _changeTextStatus(int orientation) {
        if (mCenterItemBean == null || mCenterItemBean.toolItems == null || mCenterItemBean.type == ToolbarItemConfig.ITEM_HOME_TAB) return;
        boolean btn = AppSharedPreferences.getInstance(mContext).getBoolean("Btn_TestDes_mode", "key_mode", true);
        for (ToolItemBean itemBean : mCenterItemBean.toolItems) {
            if (itemBean.toolItem != null) {
                if (itemBean.toolItem.getTextView() != null) {
                    itemBean.toolItem.getTextView().setVisibility(
                            orientation == UIToolBaseBar.HORIZONTAL && btn?
                                    View.VISIBLE :
                                    View.GONE);
                }
            }
        }
    }

    @Override
    public View getContentView() {
        return mRootView;
    }

    public void setUndoItemVisiable(boolean visiable) {
        if (visiable) {
            if (mUndoItem == null) {
                mUndoItem = createUndoButtonItem();
            }
            if (mUndoItem.getContentView().getVisibility() == View.GONE)
                mUndoItem.getContentView().setVisibility(View.VISIBLE);
            mOptionsToolbar.addView(mUndoItem, TB_Position.Position_RB);
        } else {
            if (mUndoItem != null)
                mUndoItem.getContentView().setVisibility(View.VISIBLE);
        }
    }

    public void setRedoItemVisiable(boolean visiable) {
        if (visiable) {
            if (mRedoItem == null)
                mRedoItem = createRedoButtonItem();
            if (mRedoItem.getContentView().getVisibility() == View.GONE)
                mRedoItem.getContentView().setVisibility(View.VISIBLE);
            mOptionsToolbar.addView(mRedoItem, TB_Position.Position_RB);
        } else {
            if (mRedoItem != null)
                mRedoItem.getContentView().setVisibility(View.GONE);
        }
    }

    public void setDividerItemVisible(boolean visible) {
        if (visible) {
            if (mDividerItem == null)
                mDividerItem = createDividerItem();
            if (mDividerItem.getContentView().getVisibility() == View.GONE)
                mDividerItem.getContentView().setVisibility(View.VISIBLE);
            mOptionsToolbar.addView(mDividerItem, TB_Position.Position_RB);
        } else {
            if (mDividerItem != null)
                mDividerItem.getContentView().setVisibility(View.GONE);
        }
    }

    public void setPropItemVisiable(boolean visiable) {
        if (visiable) {
            if (mPropItem == null) {
                mPropItem = createPropButtonItem();
                mPropItem.setId(R.id.id_comment_Property);
            }
            if (mPropItem.getContentView().getVisibility() == View.GONE)
                mPropItem.getContentView().setVisibility(View.VISIBLE);
            mPropItem.setOnItemClickListener(new IBaseItem.OnItemClickListener() {
                @Override
                public void onClick(IBaseItem item, final View v) {
                    if (mCurToolApply != null) {
                        mPropertyBar = mCurToolApply.getPropertyBar();

                        if (mPropertyBar != null) {
                            boolean needCustomPropertyChangedListener = !(mPropertyBar instanceof PropertyBarImpl) || ((PropertyBarImpl) mPropertyBar).needAutoSetCustomPropertyListener();
                            if (needCustomPropertyChangedListener)
                                mPropertyBar.setCustomPropertyChangeListener(new PropertyBar.PropertyChangeListener() {
                                    @Override
                                    public void onValueChanged(long property, int value) {
                                        if (property == PropertyBar.PROPERTY_COLOR || property == PropertyBar.PROPERTY_SELF_COLOR) {
                                            mPropView.setFillColorFilter(value);
                                        } else if (property == PropertyBar.PROPERTY_STAMP) {
                                            int stampType = StampUtil.getStampIconThumbnail(value);
                                            if (getUsageType() == USAGE_TYPE_HOME) {
                                                UIToolView toolView = (UIToolView) mCurItem.getCustomView();
                                                toolView.setFillBackgroundResource(stampType);
                                            }
                                            mPropView.setBorderResource(stampType);
                                            if (value == StampConstants.CUSTOM_IMAGE_STAMP
                                                    || value == StampConstants.CUSTOM_TEXT_STAMP) {
                                                mPropView.setBorderColor(AppResource.getColor(mContext, R.color.i3));
                                            }
                                        }
                                    }

                                    @Override
                                    public void onValueChanged(long property, float value) {
                                    }

                                    @Override
                                    public void onValueChanged(long property, String value) {
                                    }
                                });

                            Rect rect = new Rect();
                            mPropItem.getContentView().getGlobalVisibleRect(rect);
                            int pos;
                            if (mOrientation == BaseBar.HORIZONTAL) {
                                pos = UIPopoverWin.ARROW_TOP;
                            } else {
                                if (rect.left < AppDisplay.getActivityWidth() / 2)
                                    pos = UIPopoverWin.ARROW_LEFT;
                                else
                                    pos = UIPopoverWin.ARROW_RIGHT;
                            }
                            mPropertyBar.show(mUiExtensionsManager.getRootView(), new RectF(rect), true, pos);
                        }
                    }
                }
            });
            mPropItem.setItemLayoutListener(new IBaseItem.IItemLayoutListener() {
                @Override
                public void onItemLayout(int l, int t, int r, int b) {
                    updatePropertyBarPosition();
                }
            });
            mOptionsToolbar.addView(mPropItem, TB_Position.Position_RB);
        } else {
            if (mPropItem != null)
                mPropItem.getContentView().setVisibility(View.GONE);
        }
    }

    public void addDragItem() {
        if (mDragItem == null) {
            mDragItem = mUIToolbar.getDragItem();
        }
        if (mDragItem.getContentView().getVisibility() == View.GONE)
            mDragItem.getContentView().setVisibility(View.VISIBLE);
        mOptionsToolbar.addView(mDragItem, TB_Position.Position_RB);
    }

    public void setAddItemPosition(TB_Position position) {
        if (mAddItem == null)
            mAddItem = createAddButtonItem();
//        mAddItem.setRelation(IBaseItem.RELATION_RIGNT);
        mToolsbar.addView(mAddItem, TB_Position.Position_LT, mToolsbar.getItemsCount(TB_Position.Position_LT));
    }

    public void setSortItemPosition(TB_Position position) {
        if (mSortItem == null)
            mSortItem = createSortButtonItem();
//        mSortItem.setRelation(IBaseItem.RELATION_RIGNT);
        mToolsbar.addView(mSortItem, TB_Position.Position_LT, mToolsbar.getItemsCount(TB_Position.Position_LT));
    }

    public void setMultiSelectItemVisible(boolean visible) {
        final MultiSelectModule multiSelectModule = (MultiSelectModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_SELECT_ANNOTATIONS);
        if (multiSelectModule == null) return;
        if (visible) {
            if (mMultiSelectItem == null)
                mMultiSelectItem = createMultiSelectItem();
            if (mMultiSelectItem.getContentView().getVisibility() == View.GONE)
                mMultiSelectItem.getContentView().setVisibility(View.VISIBLE);

            mMultiSelectItem.setEnable(!mUIExtensionsManager.getPDFViewCtrl().isDynamicXFA());

            if (AppDisplay.isPad()) {
                mOptionsToolbar.addView(mMultiSelectItem, TB_Position.Position_RB);
            } else {
                mToolsbar.addView(mMultiSelectItem, TB_Position.Position_LT, 0);
            }

            mMultiSelectItem.setOnItemClickListener(new IBaseItem.OnItemClickListener() {
                @Override
                public void onClick(IBaseItem item, View v) {
                    mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_Toolbar_MultipleSelection);
                    if (mCurItem != null && mCurItem.isChecked())
                        resetCurItem();

                    mMultiSelectItem.setChecked(!mMultiSelectItem.isChecked());
                    if (mMultiSelectItem.isChecked()) {
                        mUiExtensionsManager.stopHideToolbarsTimer();
                        mUiExtensionsManager.setCurrentToolHandler(multiSelectModule.getToolHandler());
                    } else {
                        mUiExtensionsManager.startHideToolbarsTimer();
                        if (mUiExtensionsManager.getCurrentToolHandler() == multiSelectModule.getToolHandler()) {
                            mUiExtensionsManager.setCurrentToolHandler(null);
                        }
                    }
                }
            });
        } else {
            if (mMultiSelectItem != null)
                mMultiSelectItem.getContentView().setVisibility(View.GONE);
        }
    }

    void setPreferenceItems(List<Integer> types) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < types.size(); i++) {
            sb.append(types.get(i));
            if (i < types.size() - 1) {
                sb.append(",");
            }
        }
        mSp.setString(getSpName(), SP_TOOLS_LIST, sb.toString());
    }

    List<Integer> getPreferenceItems() {
        String str = mSp.getString(getSpName(), SP_TOOLS_LIST, null);
        if (AppUtil.isEmpty(str))
            return null;

        String[] strList = str.split(",");
        if (strList.length == 0)
            return null;

        List<Integer> types = new ArrayList<>();
        for (String type : strList) {
            int tag = Integer.parseInt(type);
            types.add(tag);
        }
        return types;
    }

    List<ToolItemBean> getToolBeans(String data) {
        String str = mSp.getString(getSpName(), SP_TOOLS_LIST, null);
        if (AppUtil.isEmpty(str))
            return null;

        mToolsList.clear();
        String[] list = data.split(";");
        for (String value : list) {
            ToolItemBean tool = new ToolItemBean();
            String[] item = value.split(":");
            tool.property.type = Integer.parseInt(item[0]);
            tool.property.color = Integer.parseInt(item[1]);
            tool.property.opacity = Integer.parseInt(item[2]);
            tool.property.style = Integer.parseInt(item[3]);
            tool.property.fontName = item[4];
            tool.property.fontSize = Float.parseFloat(item[5]);
            tool.property.lineWidth = Float.parseFloat(item[6]);
            mToolsList.add(tool);
        }
        return mToolsList;
    }

    private String getToolsString() {
        StringBuilder result = new StringBuilder();
        String st;
        for (ToolItemBean bean : mToolsList) {
            st = bean.property.type + ":"
                    + bean.property.color + ":"
                    + bean.property.opacity + ":"
                    + bean.property.style + ":"
                    + bean.property.fontName + ":"
                    + bean.property.fontSize + ":"
                    + bean.property.lineWidth + ";";
            result.append(st);
        }
        return result.toString();
    }


    private String getSpName() {
        return getToolBarName() + "_" + "tools_pref";
    }

    private String getToolBarName() {
        if (getUsageType() == UIToolBaseBar.USAGE_TYPE_HOME)
            return "home";
        else if (getUsageType() == UIToolBaseBar.USAGE_TYPE_COMMENT) {
            return "comment";
        }
        return "";
    }

    private UIExtensionsManager.ToolHandlerChangedListener mToolHandlerChangedListener = new UIExtensionsManager.ToolHandlerChangedListener() {
        @Override
        public void onToolHandlerChanged(ToolHandler oldToolHandler, ToolHandler newToolHandler) {
            if (mMultiSelectItem != null
                    && mMultiSelectItem.isChecked()
                    && !(newToolHandler instanceof MultiSelectToolHandler)) {
                mMultiSelectItem.performClick();
            }

            if (oldToolHandler != null && newToolHandler == null) {
                if (mCurItem != null) {
                    resetCurItem();
                }
            }
        }
    };

    private IInteractionEventListener mInteractionEventListener = new IInteractionEventListener() {
        @Override
        public boolean onKeyDown(Activity act, int keyCode, KeyEvent event) {
            if (keyCode == KeyEvent.KEYCODE_BACK) {
                if (mCurItem != null) {
                    if (!AppDisplay.isPad() && mUIExtensionsManager.getPanelManager().isShowingPanel()) {
                        return false;
                    }
                    mCurItem.performClick();
                    return true;
                }
            }
            return false;
        }
    };

    private void resetCurItem() {
        if (mCurItem != null) {
            mCurItem.setChecked(false);

            if (mCurToolApply != null) {
                mCurToolApply.onClick(mCurToolBean);
                mCurToolApply = null;
            }

            mCurToolBean = null;
            mCurItem = null;
            if (mPropView != null) {
                mPropView.setFillResource(R.drawable.annot_prop_circle_border_bg);
                mPropView.setBorderResource(R.drawable.annot_prop_circle_border_bg);
                mPropView.setBorderWidth(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_item_border_unselected_width));
                mPropView.setBorderColor(AppResource.getColor(mContext, R.color.i2));
                mPropView.setFillColorFilter(AppResource.getColor(mContext, R.color.b2));
                if (mPropViewContainer.getBackground() != null)
                    mPropViewContainer.getBackground().setColorFilter(new PorterDuffColorFilter(AppResource.getColor(mContext, R.color.i2), PorterDuff.Mode.SRC_IN));
            }
        }

        if (mPropertyBar != null)
            mPropertyBar.setCustomPropertyChangeListener(null);
    }

    @Override
    public void setToolItems(MainCenterItemBean centerItemBean) {
        if (centerItemBean == null) return;

        mCenterItemBean = centerItemBean;
        mToolsbar.removeAllItems();
        if (centerItemBean.toolItems == null) {
            List<ToolItemBean> tools = new ArrayList<>();
            ToolItemsManager toolManager = mUiExtensionsManager.getToolsManager();
            List<Integer> toolItems = getToolTypes(toolManager);
            if (toolItems != null) {
                if (getUsageType() != UIToolBaseBar.USAGE_TYPE_HOME)
                    Collections.sort(toolItems);
                int size = toolItems.size();
                for (int i = 0; i < size; i++) {
                    int type = toolItems.get(i);
                    final IToolSupply toolSupply = toolManager.getToolSupply(type);
                    if (toolSupply == null) continue;
                    final ToolItemBean toolItemBean = new ToolItemBean();
                    toolItemBean.itemStyle = ToolItemBean.DEFAULT_STYLE;
                    toolItemBean.type = type;
                    toolItemBean.property = toolSupply.createProperty(type, ToolConstants.DEFAULT_PROP);
                    UIColorItem toolItem = toolSupply.createToolItem(type, centerItemBean.type == ToolbarItemConfig.ITEM_HOME_TAB);
                    boolean btn = AppSharedPreferences.getInstance(mContext).getBoolean("Btn_TestDes_mode", "key_mode", true);
                    if (toolItem.getTextView() != null) {
                        toolItem.getTextView().setVisibility(mOrientation == HORIZONTAL && btn  && centerItemBean.type != ToolbarItemConfig.ITEM_HOME_TAB ?
                                View.VISIBLE : View.GONE);
                    }
                    toolItemBean.toolItem = toolItem;
                    if (toolItemBean.property != null)
                        toolItem.setAlphaColorBg(toolItemBean.property.color);
                    toolItem.setId(i);
                    toolItem.setTag(type);
                    toolItem.setOnItemClickListener(mOnItemClickListener);
                    mToolsbar.addView(toolItem, TB_Position.Position_LT, i);
                    tools.add(toolItemBean);
                    setIds(toolItem);
                }
            }
            centerItemBean.toolItems = tools;
            mToolsList = tools;
        } else {
            List<ToolItemBean> toolItems = centerItemBean.toolItems;
            mToolsList = centerItemBean.toolItems;

            for (int i = 0; i < toolItems.size(); i++) {
                ToolItemBean toolItemBean = toolItems.get(i);
                if (toolItemBean.itemStyle == ToolItemBean.CUSTOM_STYLE) {
                    toolItemBean.toolItem.setId(i);
                    mToolsbar.addView(toolItemBean.toolItem, TB_Position.Position_LT, i);
                } else {
                    ToolItemsManager toolManager = mUiExtensionsManager.getToolsManager();
                    final IToolSupply toolSupply = toolManager.getToolSupply(toolItemBean.type);
                    if (toolSupply == null) continue;
                    if (toolItemBean.property == null){
                        toolItemBean.property = toolSupply.createProperty(toolItemBean.type, ToolConstants.DEFAULT_PROP);
                    }else{
                        if(centerItemBean.type == ToolbarItemConfig.ITEM_COMMENT_TAB && mUiExtensionsManager.canUpdateAnnotDefaultProperties()){
                            if(toolItemBean.type == ToolConstants.Highlight
                                    ||toolItemBean.type == ToolConstants.Strikeout
                                    ||toolItemBean.type == ToolConstants.Squiggly
                                    ||toolItemBean.type == ToolConstants.Underline){
                                toolSupply.setPropertyBar(toolItemBean.property);
                            }
                        }
                    }
                    UIColorItem toolItem = (UIColorItem) toolItemBean.toolItem;
                    if (toolItemBean.toolItem == null) {
                        toolItem = toolSupply.createToolItem(toolItemBean.type,
                                centerItemBean.type ==ToolbarItemConfig.ITEM_HOME_TAB );
                        toolItemBean.toolItem = toolItem;

                        if (getUsageType() == USAGE_TYPE_HOME && toolItemBean.type == ToolConstants.Stamp) {
                            UIToolView toolView = (UIToolView) toolItem.getCustomView();
                            toolView.setFillBackgroundResource(StampUtil.getStampIconThumbnail(toolItemBean.property.style));
                            if (toolItemBean.property.style == StampConstants.CUSTOM_IMAGE_STAMP
                                    || toolItemBean.property.style == StampConstants.CUSTOM_TEXT_STAMP) {
                                toolView.setBackgroundColorFilter(AppResource.getColor(mContext, R.color.i3));
                            }
                        }
                    } else if (toolItem.isChecked() && mCurItem == null) {
                        mCurItem = toolItem;
                        mCurToolBean = toolItemBean;
                        mCurToolApply = toolSupply;
                    }
                    if (toolItemBean.property != null)
                        toolItem.setAlphaColorBg(toolItemBean.property.color);
                    toolItem.setId(i);
                    toolItem.setTag(toolItemBean.type);
                    toolItem.setOnItemClickListener(mOnItemClickListener);
                    mToolsbar.addView(toolItem, TB_Position.Position_LT, i);

                    setIds(toolItem);
                }
            }
        }

        if (centerItemBean.featureItems == null) {
            addOptionItems(mToolsList);

        } else {
            for (IBaseItem baseItem : centerItemBean.featureItems) {
                mOptionsToolbar.addView(baseItem, TB_Position.Position_RB);
            }
        }
//        layout(mOrientation, mLayoutWidth, mLayoutHeight);
    }

    private void setIds(IBaseItem toolItem) {

        switch (toolItem.getTag()) {
            // fill & sign
            case ToolConstants.FillSignTypeText:
                toolItem.setId(R.id.fill_sign_type_text);
                break;
            case ToolConstants.FillSignTypeComboText:
                toolItem.setId(R.id.fill_sign_type_combo_text);
                break;
            case ToolConstants.FillSignTypeProFile:
                toolItem.setId(R.id.fill_sign_type_pro_file);
                break;
            case ToolConstants.FillSignTypeCheckMark:
                toolItem.setId(R.id.fill_sign_type_check_mark);
                break;
            case ToolConstants.FillSignTypeCrossMark:
                toolItem.setId(R.id.fill_sign_type_cross_mark);
                break;
            case ToolConstants.FillSignTypeDot:
                toolItem.setId(R.id.fill_sign_type_dot);
                break;
            case ToolConstants.FillSignTypeLine:
                toolItem.setId(R.id.fill_sign_type_line);
                break;
            case ToolConstants.FillSignTypeRectangle:
                toolItem.setId(R.id.fill_sign_type_rect);
                break;

            //digital signature
            case ToolConstants.DigitalSignature:
                toolItem.setId(R.id.fill_sign_type_signature);

                break;
            // COMMENTS
            case ToolConstants.Note:
                toolItem.setId(R.id.id_comment_Note);
                break;
            case ToolConstants.Attachments:
                toolItem.setId(R.id.id_comment_Attachments);
                break;
            case ToolConstants.Stamp:
                toolItem.setId(R.id.id_comment_Stamp);
                break;
            case ToolConstants.Typewriter:
                toolItem.setId(R.id.id_comment_Typewriter);
                break;
            case ToolConstants.Callout:
                toolItem.setId(R.id.id_comment_Callout);
                break;
            case ToolConstants.Textbox:
                toolItem.setId(R.id.id_comment_Textbox);
                break;
            case ToolConstants.Highlight:
                toolItem.setId(R.id.id_comment_Highlight);
                break;
            case ToolConstants.Underline:
                toolItem.setId(R.id.id_comment_Underline);
                break;
            case ToolConstants.Squiggly:
                toolItem.setId(R.id.id_comment_Squiggly);
                break;
            case ToolConstants.Strikeout:
                toolItem.setId(R.id.id_comment_Strikeout);
                break;
            case ToolConstants.Replace_Text:
                toolItem.setId(R.id.id_comment_Replace_Text);
                break;
            case ToolConstants.Insert_Text:
                toolItem.setId(R.id.id_comment_Insert_Text);
                break;
            case ToolConstants.Area_Highlight:
                toolItem.setId(R.id.id_comment_Area_Highlight);
                break;
            case ToolConstants.Measure:
                toolItem.setId(R.id.id_comment_Measure);
                break;
            case ToolConstants.Image_Annotation:
                toolItem.setId(R.id.id_comment_Image_Annotation);
                break;

            // form
            case ToolConstants.TextField:
                toolItem.setId(R.id.id_at_form_toolbar_textfield);
                break;
            case ToolConstants.CheckBox:
                toolItem.setId(R.id.id_at_form_toolbar_checkbox);
                break;
            case ToolConstants.RadioButton:
                toolItem.setId(R.id.id_at_form_toolbar_radiobutton);
                break;
            case ToolConstants.ComboBox:
                toolItem.setId(R.id.id_at_form_toolbar_combobox);
                break;
            case ToolConstants.ListBox:
                toolItem.setId(R.id.id_at_form_toolbar_listbox);
                break;
            case ToolConstants.SignatureField:
                toolItem.setId(R.id.id_at_form_toolbar_signaturefield);
                break;
            case ToolConstants.ImageField:
                toolItem.setId(R.id.id_at_form_toolbar_signaturefield);
                break;
            // edit
            case ToolConstants.Text:
                toolItem.setId(R.id.id_at_edit_toolbar_text);
                break;
            case ToolConstants.Image:
                toolItem.setId(R.id.id_at_edit_toolbar_image);
                break;
            case ToolConstants.Audio:
                toolItem.setId(R.id.id_at_edit_toolbar_audio);
                break;
            case ToolConstants.Video:
                toolItem.setId(R.id.id_at_edit_toolbar_video);
                break;

            // drawing
            case ToolConstants.Link:
                toolItem.setId(R.id.id_at_edit_toolbar_link);
                break;
            case ToolConstants.Pencil:
                toolItem.setId(R.id.id_at_drawing_toolbar_pencil);
                break;
            case ToolConstants.Highlighter:
                toolItem.setId(R.id.id_at_drawing_toolbar_highlight);
                break;
            case ToolConstants.Eraser:
                toolItem.setId(R.id.id_at_drawing_toolbar_eraser);
                break;
            case ToolConstants.Rectangle:
                toolItem.setId(R.id.id_at_drawing_toolbar_rectangle);
                break;
            case ToolConstants.Oval:
                toolItem.setId(R.id.id_at_drawing_toolbar_oval);
                break;
            case ToolConstants.Line:
                toolItem.setId(R.id.id_at_drawing_toolbar_line);
                break;
            case ToolConstants.Arrow:
                toolItem.setId(R.id.id_at_drawing_toolbar_arrow);
                break;
            case ToolConstants.PolyLine:
                toolItem.setId(R.id.id_at_drawing_toolbar_polyline);
                break;
            case ToolConstants.Polygon:
                toolItem.setId(R.id.id_at_drawing_toolbar_polygon);
                break;
            case ToolConstants.Cloud:
                toolItem.setId(R.id.id_at_drawing_toolbar_cloud);
                break;
            //Esign
            case ToolConstants.CreateSignature:
                toolItem.setId(R.id.id_at_esign_toolbar_1);
                break;
            case ToolConstants.ManageSignatures:
                toolItem.setId(R.id.id_at_esign_toolbar_2);
                break;
            case ToolConstants.ApplyAllSignatures:
                toolItem.setId(R.id.id_at_esign_toolbar_3);
                break;
            case ToolConstants.RequestSignature:
                toolItem.setId(R.id.id_at_esign_toolbar_4);
                break;
            case ToolConstants.DocumentStaus:
                toolItem.setId(R.id.id_at_esign_toolbar_5);
                break;
            case ToolConstants.SwitchAccount:
                toolItem.setId(R.id.id_at_esign_toolbar_6);
                break;
        }
    }

    private void slideSelectedArea(IBaseItem item) {
        if (item.getContentView().getWidth() == 0) {
            return;
        }
        if (item.getTag() == ToolConstants.Pencil) {
            MainFrame mainFrame = (MainFrame) mUiExtensionsManager.getMainFrame();
            int widthScreen = mHSView.getWidth();
            int heightScreen = mVSView.getHeight();

            int widthChild = 0;
            int nbChildInScreen = 0;

            int positionLeftChild = 0;
            int scroll = 0;
            if (mainFrame.isShowFullScreenUI()) {
                widthChild = item.getContentView().getHeight();
                nbChildInScreen = heightScreen / widthChild;
                positionLeftChild = item.getContentView().getTop();
                scroll = positionLeftChild - ((nbChildInScreen * widthChild) / 2) + widthChild / 2;
                mVSView.smoothScrollTo(mVSView.getScrollX(), scroll);
            } else {
                widthChild = item.getContentView().getWidth();
                nbChildInScreen = widthScreen / widthChild;
                positionLeftChild = item.getContentView().getLeft();
                scroll = positionLeftChild - ((nbChildInScreen * widthChild) / 2) + widthChild / 2;
                mHSView.smoothScrollTo(scroll, mHSView.getScrollY());
            }
        }
    }

    private IBaseItem.OnItemClickListener mOnItemClickListener = new IBaseItem.OnItemClickListener() {
        @Override
        public void onClick(final IBaseItem item, View v) {
            MainFrame mainFrame = (MainFrame) mUIExtensionsManager.getMainFrame();
            if (mainFrame.isShowFullScreenUI()) {
                mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_FullScreen_Toolbar);
            }
            if (mCurItem != null && mCurItem != item) {
                mCurItem.setChecked(false);
                mCurItem = null;

                mCurToolApply.onClick(mCurToolBean);
            }

            for (final ToolItemBean itemBean : mToolsList) {
                if (itemBean.toolItem == item) {
                    final Runnable runnable = new Runnable() {
                        @Override
                        public void run() {
                            ToolItemsManager toolManager = mUiExtensionsManager.getToolsManager();
                            final IToolSupply toolSupply = toolManager.getToolSupply(itemBean.type);

                            if (itemBean.type < ToolConstants.ProtectTypeRedaction || itemBean.type > ToolConstants.ProtectTypeAIP)
                                item.setChecked(!item.isChecked());
                            toolSupply.onClick(itemBean);

                            if (item.isChecked()) {

                                slideSelectedArea(item);

                                mUiExtensionsManager.stopHideToolbarsTimer();

                                mCurItem = (UIColorItem) item;
                                mCurToolApply = toolSupply;
                                mCurToolBean = itemBean;

                                if (mPropView != null) {
                                    if (itemBean.property == null) {
                                        mPropView.setFillResource(R.drawable.annot_prop_circle_border_bg);
                                        mPropView.setBorderResource(R.drawable.annot_prop_circle_border_bg);
                                        mPropView.setBorderWidth(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_item_border_unselected_width));
                                        mPropView.setFillColorFilter(AppResource.getColor(mContext, R.color.b2));
                                        mPropView.setBorderColor(AppResource.getColor(mContext, R.color.i2));
                                        if (mPropViewContainer.getBackground() != null)
                                            mPropViewContainer.getBackground().setColorFilter(new PorterDuffColorFilter(
                                                    AppResource.getColor(mContext, R.color.i2), PorterDuff.Mode.SRC_IN));
                                    } else {
                                        if (itemBean.type == ToolConstants.Stamp) {
                                            mPropView.setFillDrawale(null);
                                            mPropView.setBorderWidth(0);
                                            mPropView.setBorderResource(StampUtil.getStampIconThumbnail(itemBean.property.style));
                                            if (itemBean.property.style == StampConstants.CUSTOM_IMAGE_STAMP
                                                    || itemBean.property.style == StampConstants.CUSTOM_TEXT_STAMP) {
                                                mPropView.setBorderColor(AppResource.getColor(mContext, R.color.i3));
                                            }
                                            mPropView.setFillColorFilter(R.color.ux_color_translucent);
                                        } else {
                                            mPropView.setFillResource(R.drawable.annot_prop_circle_border_bg);
                                            mPropView.setBorderResource(R.drawable.annot_prop_circle_border_bg);
                                            mPropView.setBorderWidth(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_item_border_unselected_width));
                                            mPropView.setFillColorFilter(itemBean.property.color);
                                            mPropView.setBorderColor(AppResource.getColor(mContext, R.color.i3));
                                        }
                                        if (mPropViewContainer.getBackground() != null)
                                            mPropViewContainer.getBackground().setColorFilter(new PorterDuffColorFilter(
                                                    AppResource.getColor(mContext, R.color.i3), PorterDuff.Mode.SRC_IN));
                                    }
                                }
                            } else {
                                mUiExtensionsManager.startHideToolbarsTimer();
                                mCurToolApply = null;
                                resetCurItem();
                            }
                        }
                    };

                    if (itemBean.type == ToolConstants.Text
                            || itemBean.type == ToolConstants.Image
                            || itemBean.type == ToolConstants.Link
                            || itemBean.type == ToolConstants.Audio
                            || itemBean.type == ToolConstants.Video
                    ) {
                        if (mUiExtensionsManager.getPermissionProvider() != null) {
                            mUiExtensionsManager.getPermissionProvider().checkPermission(
                                    IPermissionProvider.FUNCTION_EDIT, new IPermissionProvider.IPermissionState() {
                                        @Override
                                        public void onState(int state) {
                                            if (state == IPermissionProvider.PERMISSION_STATE_SHOW) {
                                                runnable.run();
                                            }
                                        }
                                    }
                            );
                        } else {
                            runnable.run();
                        }
                    } else {
                        runnable.run();
                    }

                    break;
                }
            }
        }
    };

    @Override
    public MainCenterItemBean getToolItems() {
        return mCenterItemBean;
    }

    @Override
    public Point measureSize() {
        if (mOrientation == HORIZONTAL)
            return new Point(getToolBarHeight(), Math.max(mLayoutHeight, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_bottombar_height)));
        else
            return new Point(Math.max(mLayoutWidth, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_bottombar_height)), getToolBarHeight());
    }

    private int getToolBarHeight() {
        int toolWidth;
        int toolHeight;
        if (AppDisplay.isPad()
                || mUIToolbar.getToolBarPosition() == UIDragToolBar.TOOLBAR_POS_TOP
                /*|| mUIToolbar.getToolBarPosition() == UIDragToolBar.TOOLBAR_POS_BOTTOM*/) {
            mUIToolbar.getRootView().measure(0, 0); // Optimize screen rotation speed
            toolWidth = mUIToolbar.getWidth();
            toolHeight = mUIToolbar.getHeight();
            return Math.min(toolWidth, toolHeight)
                    - AppResource.getDimensionPixelSize(mContext, R.dimen.ux_toolbar_side_margin) * 2;
        } else {
            int screenHeight = 0, screenWidth = 0;
            if (mUiExtensionsManager.getMainFrame() != null && mUiExtensionsManager.getMainFrame().getContentView().isShown()) { // Chrome OS
                screenHeight = mUiExtensionsManager.getMainFrame().getContentView().getHeight();
            }
            if (screenHeight == 0)
                screenHeight = AppDisplay.getActivityHeight();
            int topbarHeight = AppResource.getDimensionPixelSize(mContext, R.dimen.ui_bottombar_height);
            toolHeight = screenHeight
                    - topbarHeight * 2 // topbar and bottombar height
                    - AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_58dp) * 2 // margin
                    - mUIToolbar.mShadowLength * 2; // shadow
            return toolHeight;
        }
    }

    @Override
    public void setEnabled(boolean enabled) {
        super.setEnabled(enabled);
        for (int i = 0; i < mToolsList.size(); i++) {
            ToolItemBean toolItemBean = mToolsList.get(i);
            if (toolItemBean.itemStyle == ToolItemBean.DEFAULT_STYLE) {
                IBaseItem toolItem = toolItemBean.toolItem;
                if (toolItem != null) {
                    toolItem.setEnable(enabled);
                }
            }
        }
    }

    @Override
    public void onThemeColorChanged() {
        updateThemeColor();
    }

    public List<Integer> getToolTypes(@NonNull ToolItemsManager toolItemsManager) {
        return toolItemsManager.getToolItems(getUsageType());
    }

    public void addOptionItems(List<ToolItemBean> toolList) {
    }

    public void updatePropertyBarPosition() {
        if (mCurItem != null
                && mPropertyBar != null
                && mPropertyBar.isShowing()) {
            Rect rect = new Rect();
            mPropItem.getContentView().getGlobalVisibleRect(rect);
            int pos;
            if (mOrientation == BaseBar.HORIZONTAL) {
                pos = UIPopoverWin.ARROW_TOP;
            } else {
                if (rect.left < mUiExtensionsManager.getRootView().getWidth() / 2)
                    pos = UIPopoverWin.ARROW_LEFT;
                else
                    pos = UIPopoverWin.ARROW_RIGHT;
            }
            mPropertyBar.update(new RectF(rect), pos);
        }
    }

    @Override
    public void updateThemeColor() {
        for (int i = 0; i < mToolsList.size(); i++) {
            IBaseItem baseItem = mToolsList.get(i).toolItem;
            if (baseItem instanceof UIColorItem) {
                UIColorItem colorItem = (UIColorItem) baseItem;
                if (colorItem.getCustomView() instanceof UIToolView) {
                    UIToolView toolView = (UIToolView) colorItem.getCustomView();
                    if (colorItem.getTag() == ToolConstants.Stamp && getUsageType() == USAGE_TYPE_HOME) {
                        if (toolView.getFillBackgroundResource() == R.drawable.stamp_custom_text
                                || toolView.getFillBackgroundResource() == R.drawable.stamp_custom_image) {
                            toolView.updateTheme();
                        } else {
                            toolView.setSelectBgColor(R.drawable.tool_view_select_bg);
                        }
                    } else {
                        toolView.updateTheme();
                    }
                } else {
                    colorItem.updateThemeColor();
                }
//                if (colorItem.getTag() >= ToolConstants.ProtectTypeRedaction && colorItem.getTag() <= ToolConstants.ProtectTypeAIP) {
//                    colorItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
//                }
            }
        }
        if (mMultiToolView != null)
            mMultiToolView.updateTheme();

        if (mPropView != null) {
            int borderColor;
            if (mCurItem != null) {
                borderColor = AppResource.getColor(mContext, R.color.i3);
                if (mCurItem.getTag() == ToolConstants.Stamp) {
                    if (mPropView.getBorderResource() == R.drawable.stamp_custom_text
                            || mPropView.getBorderResource() == R.drawable.stamp_custom_image) {
                        mPropView.setBorderColor(AppResource.getColor(mContext, R.color.i3));
                    }
                } else {
                    mPropView.setBorderColor(borderColor);
                    mPropView.setFillColorFilter(mPropView.getFillColor());
                }
            } else {
                borderColor = AppResource.getColor(mContext, R.color.i2);
                mPropView.setBorderColor(borderColor);
                mPropView.setFillColorFilter(AppResource.getColor(mContext, R.color.b2));
            }
            if (mPropViewContainer.getBackground() != null)
                mPropViewContainer.getBackground().setColorFilter(new PorterDuffColorFilter(borderColor, PorterDuff.Mode.SRC_IN));
        }
        if (mAddItem != null) {
            ColorStateList colorStateList = ColorStateList.valueOf(AppResource.getColor(mContext, R.color.i3));
            mAddItem.setImageTintList(colorStateList);
            mAddItem.setTextColor(ColorStateList.valueOf(AppResource.getColor(mContext, R.color.fx_button_text_selector)));
        }
        if (mToolsbar != null)
            mToolsbar.setBackgroundColor(Color.TRANSPARENT);
        if (mOptionsToolbar != null)
            mOptionsToolbar.setBackgroundColor(Color.TRANSPARENT);
        mRootView.setBackgroundColor(mUIToolbar.getBackgroundColorProvider().getColor(AppUtil.isDarkMode(mContext)));
        ThemeUtil.setTintList(mHSViewStartMask, getScrollViewMaskColor());
        ThemeUtil.setTintList(mHSViewEndMask, getScrollViewMaskColor());
        ThemeUtil.setTintList(mVSViewStartMask, getScrollViewMaskColor());
        ThemeUtil.setTintList(mVSViewEndMask, getScrollViewMaskColor());
    }

    public UIColorItem getCurrentItem() {
        return mCurItem;
    }

    private final View.OnTouchListener mScrollTouchListener = new View.OnTouchListener() {
        @SuppressLint("ClickableViewAccessibility")
        @Override
        public boolean onTouch(View v, MotionEvent event) {
            switch (event.getActionMasked()) {
                case MotionEvent.ACTION_MOVE:
                    updateScrollViewMask(v);
                    mUiExtensionsManager.stopHideToolbarsTimer();
                    break;
                case MotionEvent.ACTION_UP:
                case MotionEvent.ACTION_CANCEL:
                    updateScrollViewMask(v);
                    mUiExtensionsManager.startHideToolbarsTimer();
                    break;
                default:
                    break;
            }
            return false;
        }
    };

    private int mOptionToolsViewHorizontalOffset = 30;

    public void setOptionToolsViewHorizontalOffset(int offset) {
        mOptionToolsViewHorizontalOffset = offset;
    }

    private void updateScrollViewMask(View view) {
        if (view == mHSView) {
            if (mHSViewStartMask != null) {
                mHSViewStartMask.setVisibility(mHSView.getScrollX() == 0 ? View.GONE : View.VISIBLE);
            }
            if (mHSViewEndMask != null) {
                mHSViewEndMask.setVisibility(mHSView.getScrollX() + mHSView.getWidth() == mHSToolsView.getWidth() ? View.GONE : View.VISIBLE);
            }
        } else if (view == mVSView) {
            if (mVSViewStartMask != null) {
                mVSViewStartMask.setVisibility(mVSView.getScrollY() == 0 ? View.GONE : View.VISIBLE);
            }
            if (mVSViewEndMask != null) {
                mVSViewEndMask.setVisibility(mVSView.getScrollY() + mVSView.getHeight() == mVSToolsView.getHeight() ? View.GONE : View.VISIBLE);
            }
        }
    }

    ColorStateList getScrollViewMaskColor() {
        int disabled = 0;
        int normal = AppResource.getColor(mContext, R.color.tool_items_edge_mask_color);
        return AppResource.createColorStateList(mContext, disabled, normal);
    }

    private final IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            ThemeUtil.setTintList(mHSViewStartMask, getScrollViewMaskColor());
            ThemeUtil.setTintList(mHSViewEndMask, getScrollViewMaskColor());
        }
    };

}
