/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.propertybar.imp;

import android.app.Activity;
import android.content.Context;
import android.content.res.ColorStateList;
import android.graphics.Color;
import android.graphics.drawable.Drawable;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;

public class SimpleColorView {
    private Activity mAct;
    private Context mContext;
    private int mSelectedColor;
    private int[] mColors;
    private View mRootView;
    private TextView mTvTitle;
    private ImageView mIvDragImage;
    private LinearLayout mColorContainer;
    private boolean mCanEdit = true;
    private PropertyBar.PropertyChangeListener mPropertyChangeListener;
    private int mMinWidth;

    public SimpleColorView(Activity context, int selectedColor, int[] colors) {
        mAct = context;
        mContext = context.getApplicationContext();
        mSelectedColor = selectedColor;
        mColors = colors;
        initialize();
    }

    private void initialize() {
        mRootView = LayoutInflater.from(mContext).inflate(R.layout.simple_color_layout, null);
        mTvTitle = mRootView.findViewById(R.id.simple_color_title);
        mIvDragImage = mRootView.findViewById(R.id.simple_drag_image);

        mColorContainer = mRootView.findViewById(R.id.simple_color_container);
        initializeColors(mColorContainer);

        if (!mCanEdit) {
            mRootView.setAlpha(PropertyBar.PB_ALPHA);
        }
    }

    private void initializeColors(LinearLayout parent) {
        parent.removeAllViews();
        int width = 0;
        int colorWidth = AppDisplay.dp2px(24.0f);
        final int padding = AppDisplay.dp2px(4.0f);
        boolean isColorSelected = false;
        int length;
        if (AppDisplay.isPad()) {
            width = Math.min(AppDisplay.getActivityWidth(), AppDisplay.getActivityHeight()) * 3 / 5;
            length = width - AppDisplay.dp2px(16.0f) * 2;
        } else {
            width = Math.min(AppDisplay.getActivityWidth(), AppDisplay.getActivityHeight());
            length = width - AppDisplay.dp2px(16.0f) * 2;
        }
        mMinWidth = width;

        int space = (length - (colorWidth + padding * 2) * mColors.length) / (mColors.length - 1);
        space = Math.max(0, space);
        parent.setOrientation(LinearLayout.HORIZONTAL);
        parent.setGravity(Gravity.CENTER_VERTICAL | Gravity.LEFT);

        for (int i = 0; i < mColors.length; i++) {
            CircleColorImageView circleColorImageView = new CircleColorImageView(mContext);
            circleColorImageView.setForceDarkAllowed(false);
            circleColorImageView.setImageColor(mColors[i]);
            LinearLayout.LayoutParams layoutParams;

            layoutParams = new LinearLayout.LayoutParams(colorWidth + padding * 2, colorWidth + padding * 2);
            if (i > 0) {
                layoutParams.leftMargin = space;
            } else {
                layoutParams.leftMargin = 0;
            }

//            layoutParams.gravity = Gravity.CENTER;
            circleColorImageView.getRootView().setLayoutParams(layoutParams);
            circleColorImageView.getRootView().setTag(i);
            parent.addView(circleColorImageView.getRootView());

            Drawable drawable = AppResource.getDrawable(mContext, R.drawable.pb_color_bg);
            ColorStateList colorStateList = ColorStateList.valueOf(ThemeConfig.getInstance(mContext).getPrimaryColor());
            drawable = AppResource.createTintedDrawable(drawable, colorStateList);

            if (!isColorSelected && mSelectedColor == mColors[i]) {
                isColorSelected = true;
                circleColorImageView.getRootView().setBackground(drawable);
            } else {
                circleColorImageView.getRootView().setBackgroundColor(Color.TRANSPARENT);
            }

            if (mCanEdit) {
                final Drawable finalDrawable = drawable;
                circleColorImageView.getRootView().setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        if (v instanceof LinearLayout) {
                            int tag = (Integer) v.getTag();
                            mSelectedColor = mColors[tag];
                            for (int j = 0; j < mColors.length; j++) {
                                if (j == tag) {
                                    v.setBackground(finalDrawable);
                                } else {
                                    LinearLayout colorRow = (LinearLayout) v.getParent();
                                    LinearLayout otherColor = (LinearLayout) colorRow.getChildAt(j);
                                    otherColor.setBackgroundColor(Color.TRANSPARENT);
                                }
                            }
                            if (mPropertyChangeListener != null) {
                                mPropertyChangeListener.onValueChanged(PropertyBar.PROPERTY_COLOR, mSelectedColor);
                            }
                        }
                    }
                });
            } else {
                circleColorImageView.getRootView().setEnabled(false);
            }
        }
    }

    public View getRootView() {
        return mRootView;
    }

    public void setTitle(String title) {
        mTvTitle.setText(title);
    }

    public void setSelectedColor(int color) {
        mSelectedColor = color;
    }

    public int getSelectedColor() {
        return mSelectedColor;
    }

    public void setColors(int[] colors) {
        mColors = colors;
    }

    public void setEditable(boolean canEdit) {
        this.mCanEdit = canEdit;
    }

    public void setPropertyChangeListener(PropertyBar.PropertyChangeListener listener) {
        this.mPropertyChangeListener = listener;
    }

    public void reset() {
        initializeColors(mColorContainer);
    }

    public void setDragImageVisibility(int visibility) {
        mIvDragImage.setVisibility(visibility);
    }

    public int getMinWidth() {
        return mMinWidth;
    }

}
