/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.propertybar.imp;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.LinearGradient;
import android.graphics.Paint;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.Shader;
import android.util.AttributeSet;
import android.view.GestureDetector;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.utils.AppDisplay;

import androidx.annotation.Nullable;

public class ColorPicker extends View implements GestureDetector.OnGestureListener {
    private Context mContext;
    private Paint mBitmapPaint;
    private Bitmap mBitmapNormal;
    private Bitmap mGradualChangedBitmap;
    private PointF mSelectPoint;
    private GestureDetector detector;
    private ViewGroup mParent;
    private OnColorChangedListener mColorChangedListener;

    private int mHeight;
    private int mSelfColorHeight;
    private int mWidth;
    private int mSelfColorWidth;
    private int mCurrentColor;
    private boolean mShow;
    private boolean canEdit = true;
    private float mBitmapRadius = 30;

    public ColorPicker(Context context) {
        this(context, null, 0);
    }

    public ColorPicker(Context context, @Nullable AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public ColorPicker(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        mContext = context;
        init();
    }

    public ColorPicker(Context context, ViewGroup parent) {
        this(context, null, parent);
    }

    public ColorPicker(Context context, AttributeSet attrs, ViewGroup parent) {
        super(context, attrs);
        mContext = context;
        mParent = parent;

        init();
    }

    @SuppressWarnings("deprecation")
    private void init() {
        if (mParent != null) {
            if (AppDisplay.isPad()) {
                mSelfColorWidth = dp2px(320) - dp2px((4 + 16) * 2 + 10 + 30);
                mSelfColorHeight = AppDisplay.dp2px(120.0f);
            } else {
                int tempWidth = mParent.getWidth();
                int tempHeight = mParent.getHeight();

                if (mContext.getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE) {
                    tempWidth = Math.max(tempWidth, tempHeight);
                    if (tempWidth == AppDisplay.getActivityWidth())
                        tempWidth -= AppDisplay.getRealNavBarHeight();
                    mSelfColorHeight = AppDisplay.dp2px(80.0f);
                } else {
                    tempWidth = Math.min(tempWidth, tempHeight);
                    mSelfColorHeight = AppDisplay.dp2px(120.0f);
                }
                mSelfColorWidth = tempWidth - dp2px(16 * 2 + 10 + 30);
            }
        } else {
            mSelfColorWidth = getWidth();
            mSelfColorHeight = getHeight();
        }

        mBitmapPaint = new Paint();
        mBitmapNormal = BitmapFactory.decodeResource(mContext.getResources(), R.drawable.pb_colorpicker_point_selected);
        mBitmapRadius = mBitmapNormal.getWidth() / 2f;
        mSelectPoint = new PointF(dp2px(100), dp2px(30));
        detector = new GestureDetector(this);
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        mWidth = mSelfColorWidth;
        mHeight = mSelfColorHeight;

        setMeasuredDimension(mWidth, mHeight);
    }

    @SuppressLint("DrawAllocation")
    @Override
    protected void onDraw(Canvas canvas) {
        canvas.drawBitmap(getGradual(), null, new Rect(0, 0, mWidth, mHeight), mBitmapPaint);
        if (mShow) {
            canvas.drawBitmap(mBitmapNormal, mSelectPoint.x - mBitmapRadius, mSelectPoint.y - mBitmapRadius, mBitmapPaint);
        }
        super.onDraw(canvas);
    }

    @Override
    public boolean onSingleTapUp(MotionEvent e) {
        if (!canEdit) return false;
        float x = e.getX();
        float y = e.getY();
        mShow = true;
        proofDisk(x, y);
        mCurrentColor = getSelectColor(mSelectPoint.x, mSelectPoint.y);
        invalidate();
        if (mColorChangedListener != null) {
            mColorChangedListener.onColorChanged(mCurrentColor);
        }
        return true;
    }

    @Override
    protected void onDetachedFromWindow() {
        if (mGradualChangedBitmap != null && mGradualChangedBitmap.isRecycled()) {
            mGradualChangedBitmap.recycle();
        }
        if (mBitmapNormal != null && mBitmapNormal.isRecycled()) {
            mBitmapNormal.recycle();
        }
        super.onDetachedFromWindow();
    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {
        return detector.onTouchEvent(event);
    }

    private void proofDisk(float x, float y) {
        if (x < 0) {
            mSelectPoint.x = 0;
        } else if (x > mWidth) {
            mSelectPoint.x = mWidth;
        } else {
            mSelectPoint.x = x;
        }
        if (y < 0) {
            mSelectPoint.y = 0;
        } else if (y > mHeight) {
            mSelectPoint.y = mHeight;
        } else {
            mSelectPoint.y = y;
        }
    }

    private int getSelectColor(float x, float y) {
        Bitmap temp = getGradual();
        int intX = (int) x;
        int intY = (int) y;
        if (intX >= temp.getWidth()) {
            intX = temp.getWidth() - 1;
        }
        if (intY >= temp.getHeight()) {
            intY = temp.getHeight() - 1;
        }
        return temp.getPixel(intX, intY);
    }

    private Bitmap getGradual() {
        if (mGradualChangedBitmap == null) {
            Paint paint = new Paint();
            paint.setStrokeWidth(1);
            mGradualChangedBitmap = Bitmap.createBitmap(mWidth, mHeight, Bitmap.Config.RGB_565);
            Canvas canvas = new Canvas(mGradualChangedBitmap);
            int bitmapWidth = mGradualChangedBitmap.getWidth();
            mWidth = bitmapWidth;
            int bitmapHeight = mGradualChangedBitmap.getHeight();
            int[] Colors = new int[]{0xFFFF0000, 0xFFFF9D00, 0xFFFFFE00, 0xFF5FFF00, 0xFF00EDF2, 0xFF007AFF, 0xFF1000FF, 0xFFF000FF, 0xFFFF0000};
            Shader mShader = new LinearGradient(0, 0, bitmapWidth, 0, Colors, null, Shader.TileMode.MIRROR);
            paint.setShader(mShader);
            canvas.drawRect(0, 0, bitmapWidth, bitmapHeight, paint);
        }
        return mGradualChangedBitmap;
    }

    public interface OnColorChangedListener {
        void onColorChanged(int color);
    }

    public void setOnColorChangedListener(OnColorChangedListener listener) {
        mColorChangedListener = listener;
    }

    public void setColor(int color) {
        mShow = false;
        mCurrentColor = color;
        invalidate();
    }

    public void setSelectPointVisible(boolean visible) {
        mShow = visible;
        invalidate();
    }

    public int getColor() {
        return mCurrentColor | 0xFF000000;
    }

    public void setWidth(int width) {
        mSelfColorWidth = width;
    }

    public void setHeight(int height) {
        mSelfColorHeight = height;
    }

    @Override
    public boolean onDown(MotionEvent e) {
        return true;
    }

    @Override
    public void onShowPress(MotionEvent e) {
    }

    @Override
    public boolean onScroll(MotionEvent e1, MotionEvent e2, float distanceX, float distanceY) {
        return false;
    }

    @Override
    public void onLongPress(MotionEvent e) {
    }

    @Override
    public boolean onFling(MotionEvent e1, MotionEvent e2, float velocityX, float velocityY) {
        return false;
    }

    private int dp2px(int dip) {
        return AppDisplay.dp2px(dip);
    }

    public void setEditable(boolean canEdit) {
        this.canEdit = canEdit;
    }
}
