/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.propertybar;

import android.view.View;

import com.foxit.uiextensions.controls.toolbar.IToolBar;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

import androidx.annotation.IntDef;

/**
 * Interface that defines the settings view.
 */
public interface IViewSettingsWindow extends IToolBar {
    /**
     * Interface definition for a callback to be invoked
     * when a value of a view settings is changed or the window is dismissed.
     */
    interface IValueChangeListener {
        /**
         * Called when a value of a view settings is changed.
         *
         * @param type  The type of the bar. {@link #getType()}.
         * @param value current value
         */
        public void onValueChanged(int type, Object value);

        /**
         * Return a type of the current view setting window.
         */
        public int getType();
    }

    /**
     * Interface used to allow the creator of a setting window to run some code when the
     * setting window is dismissed.
     */
    interface OnDismissListener {
        /**
         * This method will be invoked when the setting window is dismissed.
         */
        void onDismiss();
    }

    /** Page color mode: day */
    int DAY = 0;
    /** Page color mode: night */
    int NIGHT = 1;
    /** Page color mode: custom color */
    int CUSTOM_COLOR = 2;

    @IntDef({DAY, NIGHT, CUSTOM_COLOR})
    @Retention(RetentionPolicy.SOURCE)
    @interface PageColorMode {
    }

    /** The type of view settings window: Single Page mode */
    int TYPE_SINGLE_PAGE = 0x0001;
    /** The type of view settings window: Crop mode */
    int TYPE_FACING_PAGE = 0x0002;
    /** The type of view settings window: Cover mode */
    int TYPE_COVER_PAGE = 0x0004;
    /**
     * The type of view settings window: Day
     * Note: On the pad, day and night are the same item,
     * if you hide the day, night{@link #TYPE_NIGHT} will also be hidden. Similarly, if you hide night, day will also be hidden.
     */
    int TYPE_DAY = 0x0008;
    /** The type of view settings window: Page color */
    int TYPE_PAGE_COLOR = 0x0010;
    /**
     * The type of view settings window: night
     * Note: On the pad, day and night are the same item,
     * if you hide the day{@link #TYPE_DAY}, night will also be hidden. Similarly, if you hide night, day will also be hidden.
     */
    int TYPE_NIGHT = 0x0020;
    /** The type of view settings window: Continuous Page mode */
    int TYPE_CONTINUOUS_PAGE = 0x0040;
    /** The type of view settings window: Fit page */
    int TYPE_FIT_PAGE = 0x0080;
    /** The type of view settings window: Fit Width */
    int TYPE_FIT_WIDTH = 0x0100;
    /** The type of view settings window: Reflow mode */
    int TYPE_REFLOW = 0x0120;
    /** The type of view settings window: Crop mode */
    int TYPE_CROP = 0x0140;
    /** The type of view settings window: TextToSpeech */
    int TYPE_TTS = 0x0180;
    /** The type of view settings window: auto flip */
    int TYPE_AUTO_FLIP = 0x0200;
    /** The type of view settings window: Rotate View */
    int TYPE_ROTATE_VIEW = 0x0220;
    /** The type of view settings window: Pan&Zoom */
    int TYPE_PAN_ZOOM = 0x0240;
    /** The type of view settings window: Right to Left */
    int TYPE_RIGHT_TO_LEFT = 0x0260;

    int DEFAULT_NIGHT_BG_COLOR = 0xff0a0a0a;
    int DEFAULT_NIGHT_FG_COLOR = 0xff9b9b9b;


    /**
     * Sets value by property.
     *
     * @param property The type
     * @param value    the value to set.
     */
    public void setProperty(int property, Object value);

    /**
     * Whether this view settings window is showing.
     */
    public boolean isShowing();

    /**
     * Display this view settings window.
     */
    public void show();

    /**
     * Returns the visibility status for this view.
     *
     * @param type the modules tag,  Please refer to {@link #TYPE_SINGLE_PAGE #TYPE_XXX } values
     * @return The whether the type is visible
     */
    public boolean getVisible(int type);

    /**
     * Set the enabled state of this view.
     *
     * @param type    the modules tag,  Please refer to {@link #TYPE_SINGLE_PAGE #TYPE_XXX } values
     * @param visible true display the type view, false otherwise.
     */
    public void setVisible(int type, boolean visible);

    /**
     * Disposes of the view settings window.
     */
    public void dismiss();

    /** Return the content view of this view settings window. */
    public View getContentView();

    /**
     * Sets the listener to be called when the view settings window is dismissed or its value is changed.
     *
     * @param listener the listener
     */
    public void registerListener(IViewSettingsWindow.IValueChangeListener listener);

    /**
     * Unregister the specified listener.
     */
    public void unRegisterListener(IViewSettingsWindow.IValueChangeListener listener);

    /**
     * Get the value change listener based on type
     *
     * @param type the setting view type {@link #TYPE_SINGLE_PAGE}
     * @return
     */
    public IViewSettingsWindow.IValueChangeListener getValueChangeListener(int type);

    /**
     * Set the enabled state of the specified bar.
     * NOW: only for TYPE_CROP, TYPE_REFLOW and TYPE_PANZOOM
     *
     * @param property TYPE_CROP, TYPE_REFLOW or TYPE_PANZOOM
     * @param enable   True if this bar is enabled, false otherwise.
     */
    public void enableBar(int property, boolean enable);

    /**
     * Set a listener to be invoked when the dialog is dismissed.
     *
     * @param listener The {@link OnDismissListener} to use.
     */
    public void setOnDismissListener(OnDismissListener listener);

    /**
     * Update the theme when the theme color changes
     *
     * @return the setting window
     */
    public IViewSettingsWindow updateTheme();

    public int getColorMode();

    /**
     * Indicates the selection state of this property view.
     *
     * @return true if the view is selected, false otherwise
     */
    public boolean isSelected(int property);
}
