/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.menu;


import android.util.SparseArray;
import android.view.View;

import com.foxit.uiextensions.controls.menu.action.IActionMenuPresenter;
import com.foxit.uiextensions.controls.menu.action.IMenuPresenter;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

import androidx.annotation.IntDef;

/**
 * Use this menu view manager can obtain a menu view of a specific location, and then customize some operations using this view.
 */
public class MenuViewManager {

    /**
     * the extract menu of thumbnail dialog
     */
    public static final int POSITION_THUMBNAIL_EXTRACT = 1;
    /**
     * the add pages menu of thumbnail dialog
     */
    public static final int POSITION_THUMBNAIL_ADD = 2;
    /**
     * the more menu in the top bar
     */
    public static final int POSITION_MORE = 3;

    @IntDef({POSITION_THUMBNAIL_EXTRACT,
            POSITION_THUMBNAIL_ADD,
            POSITION_MORE})
    @Retention(RetentionPolicy.SOURCE)
    @interface MenuPosition {
    }

    private SparseArray<IActionMenuPresenter> mCustomViews;
    private SparseArray<IMenuPresenter> mCustomMenuViews;
    private SparseArray<OnItemClickListener> mCustomMenuClicks;

    public MenuViewManager() {
        mCustomViews = new SparseArray<>();
        mCustomMenuViews = new SparseArray<>();
        mCustomMenuClicks = new SparseArray<>();
    }

    /**
     *  Interface definition for a callback to be invoked when a view is clicked.
     */
    public interface OnItemClickListener {

        /**
         * Called when a view has been clicked.
         *
         * @param view The view that was clicked.
         * @return true if the event is handled, false otherwise.
         */
        boolean onClick(View view);
    }

    /**
     * Set the custom menu click event according to postion
     *
     * @param position      the menu position {@link MenuPosition}
     * @param clickListener the custom menu click
     */
    public void setActionCallback(@MenuPosition int position, OnItemClickListener clickListener) {
        mCustomMenuClicks.put(position, clickListener);
    }

    /**
     * get the custom menu click event according to position
     *
     * @param position the menu position {@link MenuPosition}
     * @return the custom menu click
     */
    public OnItemClickListener getActionCallback(@MenuPosition int position) {
        return mCustomMenuClicks.get(position);
    }

    /**
     * Set a custom view for the specified positon according to {@link IActionMenuPresenter}
     *
     * @param position   the menu position{@link MenuPosition}
     * @param actionView the custom view constructed by {@link IActionMenuPresenter}
     */
    public void setActionView(@MenuPosition int position, IActionMenuPresenter actionView) {
        mCustomViews.put(position, actionView);
    }

    /**
     * Get custom view according to positon
     *
     * @param position the menu position{@link MenuPosition}
     * @return the custom view constructed by {@link IActionMenuPresenter}
     */
    public IActionMenuPresenter getActionView(@MenuPosition int position) {
        return mCustomViews.get(position);
    }

    /**
     * Set a custom view for the specified positon according to {@link IMenuPresenter}
     *
     * @param position  the menu position{@link MenuPosition}
     * @param presenter the custom view constructed by {@link IMenuPresenter}
     */
    public void setMenuPresenter(@MenuPosition int position, IMenuPresenter presenter) {
        mCustomMenuViews.put(position, presenter);
    }

    /**
     * Get custom view according to positon
     *
     * @param position the menu position{@link MenuPosition}
     * @return the custom view constructed by {@link IMenuPresenter}
     */
    public IMenuPresenter getMenuPresenter(@MenuPosition int position) {
        return mCustomMenuViews.get(position);
    }

}
