/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.menu;

import android.content.Context;
import android.view.View;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.controls.filebrowser.imp.FileThumbnail;
import com.foxit.uiextensions.controls.toolbar.IBaseItem;
import com.foxit.uiextensions.controls.toolbar.impl.BaseItemImpl;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppUtil;

import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import static com.foxit.uiextensions.utils.AppDmUtil.DEFAULT_MMM_DD_YYYY;
import static com.foxit.uiextensions.utils.AppUtil.setThumbnailForImageView;

/**
 * Class <CODE>MenuViewImpl</CODE> represents the more menu
 * Now, the more menu only include file group, which contains a menu item "properties".
 */
public class MenuViewImpl implements IMenuView {
    private Context mContext;
    private View mRootView;
    private LinearLayout mMenuList_ly;
    private RelativeLayout mMenuTitleLayout;
    private IBaseItem mTitleItem;
    private IBaseItem mTitleLeftItem;
    private IBaseItem mTitleRightItem;
    private List<OnClosedListener> mCloseMenuCallbacks;
    private List<IMenuGroup> mMenuGroups;
    private TextView nameTextView;
    private TextView dateTextView;
    private View mViewDivider;
    private View mTitleView;
    private ImageView mTitleLocationIcon;

    public MenuViewImpl(Context context) {
        this(context, null);
    }

    public MenuViewImpl(Context context, OnClosedListener callback) {
        mContext = context;
        mRootView = View.inflate(mContext, R.layout.view_menu_more, null);
        mMenuList_ly = (LinearLayout) mRootView.findViewById(R.id.menu_more_content_ly);
        mMenuTitleLayout = (RelativeLayout) mRootView.findViewById(R.id.menu_more_title_ly);
        mMenuGroups = new ArrayList<>();
        mCloseMenuCallbacks = new ArrayList<>();
        if (callback != null)
            mCloseMenuCallbacks.add(callback);

        initTitleView();
    }

    private void initTitleView() {
        mMenuTitleLayout.removeAllViews();
        mTitleView = View.inflate(mContext, R.layout.view_menu_more_title, null);
        mTitleItem = new BaseItemImpl(mContext, mTitleView);
        mMenuTitleLayout.addView(mTitleItem.getContentView());
        mTitleLocationIcon = mTitleView.findViewById(R.id.fb_item_location);
    }

    public void setDarkModeColor() {
        mMenuList_ly.setBackgroundColor(mContext.getResources().getColor(R.color.b1));
        if (mTitleView != null) {
            mTitleView.setBackgroundColor(mContext.getResources().getColor(R.color.b1));
            if(nameTextView!=null) {
                nameTextView.setTextColor(mContext.getResources().getColor(R.color.t4));
                dateTextView.setTextColor(mContext.getResources().getColor(R.color.t3));
                mViewDivider.setBackgroundColor(mContext.getResources().getColor(R.color.p1));
            }
        }
        for (int i = 0; i < mMenuGroups.size(); i++) {
            if (mMenuGroups.get(i) instanceof MenuGroupImpl) {
                ((MenuGroupImpl) mMenuGroups.get(i)).setDarkModeColor();
            }
            List<IMenuItem> menuItems = mMenuGroups.get(i).getItems();
            for (int k = 0; k < menuItems.size(); k++) {
                if (menuItems.get(k) instanceof MenuItemImpl) {
                    ((MenuItemImpl) menuItems.get(k)).setDarkModeColor();
                }

            }
        }
    }

    @Override
    public void setTitle(String title) {
        mTitleItem.setText(title);
    }

    @Override
    public void setTitleBarVisible(boolean visible) {
        mMenuTitleLayout.setVisibility(visible ? View.VISIBLE : View.GONE);
    }

    @Override
    public void setPositiveTitle(CharSequence text, View.OnClickListener onClickListener) {
        if (mTitleRightItem == null) return;
        mTitleRightItem.setText(text);
        mTitleRightItem.getContentView().setVisibility(View.VISIBLE);
        if (onClickListener != null)
            mTitleRightItem.setOnClickListener(onClickListener);
    }

    @Override
    public void setNegativeTitle(CharSequence text, View.OnClickListener onClickListener) {
        if (mTitleLeftItem == null) return;
        mTitleLeftItem.setText(text);
        mTitleLeftItem.getContentView().setVisibility(View.VISIBLE);
        if (onClickListener != null)
            mTitleLeftItem.setOnClickListener(onClickListener);
    }

    @Override
    public void setFileLocationIcon(int id) {
        if (mTitleLocationIcon == null) return;
        if (id > 0) {
            mTitleLocationIcon.setVisibility(View.VISIBLE);
            mTitleLocationIcon.setImageResource(id);
        } else {
            mTitleLocationIcon.setVisibility(View.GONE);
        }
    }

    @Override
    public List<IMenuGroup> getGroups() {
        return mMenuGroups;
    }

    @Override
    public void close() {
        for (OnClosedListener closedListener : mCloseMenuCallbacks) {
            closedListener.onClosed();
        }
    }

    @Override
    public void addClosedListener(OnClosedListener closedListener) {
        mCloseMenuCallbacks.add(closedListener);
    }

    @Override
    public IMenuGroup addGroup(CharSequence title) {
        MenuGroupImpl menuGroup = new MenuGroupImpl(mContext, title);
        mMenuGroups.add(menuGroup);
        resetView();
        return menuGroup;
    }

    @Override
    public IMenuGroup addGroup(int groupId, CharSequence title) {
        MenuGroupImpl menuGroup = new MenuGroupImpl(mContext, groupId, title);
        mMenuGroups.add(menuGroup);
        resetView();
        return menuGroup;
    }

    @Override
    public IMenuGroup addGroupAt(int index, CharSequence title) {
        MenuGroupImpl menuGroup = new MenuGroupImpl(mContext, title);
        mMenuGroups.add(index, menuGroup);
        resetView();
        return menuGroup;
    }

    @Override
    public IMenuGroup addGroupAt(int index, int groupId, CharSequence title) {
        MenuGroupImpl menuGroup = new MenuGroupImpl(mContext, groupId, title);
        mMenuGroups.add(index, menuGroup);
        resetView();
        return menuGroup;
    }

    @Override
    public void removeGroup(int groupId) {
        for (IMenuGroup item : mMenuGroups) {
            if (item.getGroupId() != 0 && (item.getGroupId() == groupId)) {
                removeGroup(item);
                return;
            }
        }
    }

    @Override
    public void removeGroupAt(int index) {
        IMenuGroup group = mMenuGroups.get(index);
        mMenuGroups.remove(group);
        mMenuList_ly.removeView(group.getContentView());
    }

    @Override
    public void removeGroup(IMenuGroup group) {
        mMenuGroups.remove(group);
        mMenuList_ly.removeView(group.getContentView());
    }

    @Override
    public IMenuGroup getGroup(int groupId) {
        for (IMenuGroup item : mMenuGroups) {
            if (item.getGroupId() != 0 && (item.getGroupId() == groupId)) {
                return item;
            }
        }
        return null;
    }

    @Override
    public IMenuGroup getGroupAt(int index) {
        return mMenuGroups.get(index);
    }

    @Override
    public View getContentView() {
        return mRootView;
    }

    private void resetView() {
        mMenuList_ly.removeAllViews();
        for (IMenuGroup group : mMenuGroups) {
            mMenuList_ly.addView(group.getContentView());
        }
    }

    /**
     * set title info on title view
     *
     * @param filePath document file path
     */
    public void setTitleInfo(String filePath) {
        if (filePath == null) {
            ((ImageView) mMenuTitleLayout.findViewById(R.id.fb_item_icon)).setImageResource(R.drawable.fb_file_pdf);
            return;
        }
        String fileName = AppFileUtil.getFileNameWithoutExt(filePath);
        File file = new File(filePath);
        String formatFileSize = AppFileUtil.formatFileSize(file.length());
        String formatJavaDate = AppDmUtil.formatJavaDate(AppDmUtil.FORMAT_MMM_DD_YYYY, new Date(file.lastModified()));
        nameTextView = mTitleView.findViewById(R.id.fb_item_name);
        dateTextView = mTitleView.findViewById(R.id.fb_item_date);
        mViewDivider = mTitleView.findViewById(R.id.menu_more_title_divider);
        nameTextView.setText(fileName);
        String dateString = AppFileUtil.getFileExt(filePath).toUpperCase() +
                (DEFAULT_MMM_DD_YYYY.equals(formatJavaDate) ? "" : " · " + formatJavaDate) +
                (formatFileSize == null ? "" : " · " + formatFileSize);
        dateTextView.setText(dateString);
        setThumbnailForImageView(mContext, (ImageView) mMenuTitleLayout.findViewById(R.id.fb_item_icon), filePath, mThumbnailCallback);
        setDarkModeColor();
    }


    private FileThumbnail.ThumbnailCallback mThumbnailCallback = new FileThumbnail.ThumbnailCallback() {
        @Override
        public void result(boolean succeed, final String filePath) {
            if (succeed) {
                mMenuTitleLayout.post(new Runnable() {
                    @Override
                    public void run() {
                        setThumbnailForImageView(mContext, (ImageView) mMenuTitleLayout.findViewById(R.id.fb_item_icon), filePath, mThumbnailCallback);
                    }
                });
            }
        }
    };

    public void setOnTitleClickListener(View.OnClickListener listener) {
        if (null == listener || null == mTitleItem) return;
        mTitleItem.setOnClickListener(listener);
    }
}
