/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.menu;

import android.view.View;

import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.modules.more.MoreMenuConstants;
import com.foxit.uiextensions.modules.more.MoreMenuModule;

import java.util.List;

/**
 * Control {@link MoreMenuModule} group and group's submenus display and hide,
 * as well as add group ,or group submenus to the {@link MoreMenuModule}
 * <br/><br/>
 * You can use it through {@link MoreMenuModule#getMenuView()}, or {@link UIExtensionsManager#getMenuView()}
 */
public interface IMenuView {

    /**
     * The interface will be executed after the {@link #close()} method is called
     */
    interface OnClosedListener {
        /* This method will be invoked when the menu view is closed.*/
        void onClosed();
    }

    /** Close the menu view */
    void close();

    /**
     * Add a listener to be invoked when the menu view is closed.
     *
     * @param closedListener The {@link IMenuView.OnClosedListener} to use.
     */
    void addClosedListener(OnClosedListener closedListener);

    /**
     * Add a new group to the menu view. This group displays the given title for its
     * label.
     *
     * @param title the group title string.
     * @return The newly added menu group.
     */
    IMenuGroup addGroup(CharSequence title);

    /**
     * Add a new group to the menu view. This group displays the given title for its
     * label.
     *
     * @param groupId Unique group ID. This can be used to define groups of items for batch state
     *                changes. Please refer to existing groups {@link MoreMenuConstants#GROUP_ACTION_MENU_PRIMARY}
     * @param title   the group title string.
     * @return The newly added menu group.
     */
    IMenuGroup addGroup(int groupId, CharSequence title);

    /**
     * Add a new group to the menu view. This group displays the given title for its
     * label.
     *
     * @param index the position at which to add the group item
     * @param title the group title string.
     * @return The newly added menu group.
     * @throws IndexOutOfBoundsException when {@code index < 0 || >= size()}
     */
    IMenuGroup addGroupAt(int index, CharSequence title);

    /**
     * Add a new group to the menu view. This group displays the given title for its
     * label.
     *
     * @param index   the position at which to add the group item
     * @param groupId Unique group ID. This can be used to define groups of items for batch state
     *                changes. Please refer to existing groups {@link MoreMenuConstants#GROUP_ACTION_MENU_PRIMARY}
     * @param title   the group title string.
     * @return The newly added menu group.
     * @throws IndexOutOfBoundsException when {@code index < 0 || >= size()}
     */
    IMenuGroup addGroupAt(int index, int groupId, CharSequence title);

    /**
     * Remove all items in the given group, include the group item.
     *
     * @param groupId The group to be removed.  If there is no such group id, nothing happens.
     */
    void removeGroup(int groupId);

    /**
     * Remove all items in the specified position group, include the group item.
     *
     * @param index the position in the group of the menu to remove
     * @throws IndexOutOfBoundsException when {@code index < 0 || >= size()}
     */
    void removeGroupAt(int index);

    /**
     * Remove all items in the given group, include the group item.
     *
     * @param group The group to be removed.
     */
    void removeGroup(IMenuGroup group);

    /**
     * Gets the group item by the given id.
     *
     * @param groupId Unique group ID. This can be used to define groups of items for batch state
     *                changes. Please refer to existing groups {@link MoreMenuConstants#GROUP_ACTION_MENU_PRIMARY}
     * @return The group item.
     */
    IMenuGroup getGroup(int groupId);

    /**
     * Gets the group item at the given index.
     *
     * @param index The index of the group item to return.
     * @return The group item.
     * @throws IndexOutOfBoundsException when {@code index < 0 || >= size()}
     */
    IMenuGroup getGroupAt(int index);

    /**
     * @return the all groups under the current menu view
     */
    List<IMenuGroup> getGroups();

    /** the content view of the menu. */
    View getContentView();

    /**
     * Change the menu view title
     *
     * @param text The new text to be displayed.
     */
    void setTitle(String text);

    /**
     * Set whether to display the title bar
     *
     * @param visible true display the title bar
     */
    void setTitleBarVisible(boolean visible);

    /**
     * Set a listener to be invoked when the positive button of the title bar is pressed.
     *
     * @param text            The text to display in the positive button
     * @param onClickListener The {@link View.OnClickListener} to use.
     */
    void setPositiveTitle(CharSequence text, View.OnClickListener onClickListener);

    /**
     * Set a listener to be invoked when the negative button of the title bar is pressed.
     *
     * @param text            The resource id of the text to display in the negative button
     * @param onClickListener The {@link View.OnClickListener} to use.
     */
    void setNegativeTitle(CharSequence text, View.OnClickListener onClickListener);

    /**
     * Change the file location icon in the menu title
     *
     * @param id            The resource id of the icon to display in the title bar.
     */
    void setFileLocationIcon(int id);
}
