/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.menu;


import android.view.View;

import androidx.annotation.DrawableRes;

/**
 * An Item is returned by calling one of the {@link IMenuGroup#addItem}
 * methods.
 * <p>
 * For a feature set of specific menu types, see {@link IMenuView}.
 */
public interface IMenuItem {
    /**
     * Interface definition for a callback to be invoked when a menu view is clicked.
     */
    interface OnMenuItemClickListener {
        /**
         * Called while a menu item has been clicked.
         *
         * @param item the menu item was clicked.
         */
        void onClick(IMenuItem item);
    }

    /**
     * Register a callback to be invoked when this menu item is clicked. I
     *
     * @param listener The {@link OnMenuItemClickListener} to use.
     */
    void setOnMenuItemClickListener(OnMenuItemClickListener listener);

    /**
     * Set whether the divider is visible
     *
     * @param visible true display the divider, false otherwise.
     */
    void setDividerVisible(boolean visible);

    /**
     * Return the identifier for this menu item.  The identifier can not
     * be changed after the menu is created.
     *
     * @return The menu item's identifier.
     */
    int getItemId();

    /**
     * Return the group identifier that this menu item is part of. The group
     * identifier can not be changed after the menu is created.
     *
     * @return The menu item's group identifier.
     */
    int getGroupId();

    /**
     * @return Return the group that this menu item is part of.
     */
    IMenuGroup getMenuGroup();

    /**
     * Set the menu group for current menu item
     *
     * @param menuGroup The current item's menu group
     */
    void setMenuGroup(IMenuGroup menuGroup);

    /**
     * Get the root view of the menu item.
     *
     * @return The root view of the menu item.
     */
    View getContentView();

    /**
     * Set whether the item is visible
     *
     * @param visible true display the item, false otherwise.
     */
    void setVisible(boolean visible);

    /**
     * @return The whether the item is visible
     */
    boolean getVisible();

    /**
     * Sets whether the menu item is enabled.
     *
     * @param enabled If true then the item will be invokable; if false it is
     *                won't be invokable.
     */
    void setEnabled(boolean enabled);

    /**
     * Sets whether the menu item is selected.
     *
     * @param selected true if the view must be selected, false otherwise
     */
    void setSelected(boolean selected);

    /**
     * Set title for the item
     *
     * @param title the title string
     */
    void setTitle(CharSequence title);

    /**
     * @return The item title
     */
    CharSequence getTitle();

    /**
     * Set the icon of the item
     *
     * @param resId The resource identifier of the drawable
     */
    void setIcon(@DrawableRes int resId);
}
