/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.dialog.fileselect;

import android.content.Context;
import android.text.TextUtils;
import android.util.TypedValue;
import android.view.View;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

import androidx.appcompat.widget.AppCompatTextView;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.controls.dialog.MatchDialog;
import com.foxit.uiextensions.controls.dialog.UIMatchDialog;
import com.foxit.uiextensions.controls.filebrowser.FileDelegate;
import com.foxit.uiextensions.controls.filebrowser.imp.FileBrowserImpl;
import com.foxit.uiextensions.controls.filebrowser.imp.FileItem;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppStorageManager;
import com.foxit.uiextensions.utils.AppUtil;

import java.io.File;
import java.io.FileFilter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import io.reactivex.functions.Consumer;

public class UIFolderSelectDialog extends UIMatchDialog {
    private FileBrowserImpl mFileBrowser;
    private RelativeLayout mContentView;
    private List<FileItem> mFileItems = new ArrayList<>();
    private FileItem mCurrentItem;
    private LinearLayout mPathLayout;
    private Context mContext;
    private TextView mBackTextView;
    private AppCompatTextView mNoContentView;
    private ImageView mNoContentIv;
    private ImageView mRollBackIv;
    private RelativeLayout mFileBrowserView;

    public UIFolderSelectDialog(Context context) {
        super(context, true);
        mContext = context.getApplicationContext();
        onCreateView();
    }

    public View onCreateView() {
        mContentView = (RelativeLayout) View.inflate(mContext, R.layout.cloud_select_file, null);
        mFileBrowserView = mContentView.findViewById(R.id.select_file_file_browser);
        mRollBackIv = mContentView.findViewById(R.id.select_file_rollback_iv);
        mRollBackIv.setColorFilter(mContext.getResources().getColor(R.color.i3));
        mPathLayout = mContentView.findViewById(R.id.select_file_path);
        mBackTextView = mContentView.findViewById(R.id.select_file_rollback_tv);
        mNoContentView = mContentView.findViewById(R.id.tv_no_content_tips);
        mNoContentIv = mContentView.findViewById(R.id.select_file_no_info_iv);
        mNoContentIv.setColorFilter(ThemeConfig.getInstance(mContext).getPrimaryColor());
        mPathLayout.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                String path = mFileBrowser.getDisplayPath();
                if (path == null || path.length() == 0) {
                    return;
                }
                List<String> roots = AppStorageManager.getInstance(mContext).getVolumePaths();
                if (roots.contains(path)) {
                    setCurrentPath(null);
                    return;
                }
                int lastIndex = path.lastIndexOf(File.separator);
                if (lastIndex != -1) {
                    path = path.substring(0, lastIndex);
                    setCurrentPath(path);
                    return;
                }
                setCurrentPath(null);
            }
        });

        mFileBrowser = new FileBrowserImpl(mContext, new FileDelegate() {
            @Override
            public List<FileItem> getDataSource() {
                return mFileItems;
            }

            @Override
            public void onPathChanged(String path) {
                if (AppUtil.isEmpty(path)) {
                    setButtonEnable(false, MatchDialog.DIALOG_OK);
                    mPathLayout.setVisibility(View.GONE);
                    mFileItems.clear();
                    List<String> paths = AppStorageManager.getInstance(mContext).getVolumePaths();
                    for (String p : paths) {
                        File f = new File(p);
                        FileItem item = new FileItem();
                        item.parentPath = path;
                        item.path = f.getPath();
                        item.name = f.getName();
                        item.date = AppDmUtil.formatJavaDate(AppDmUtil.FORMAT_MMM_DD_YYYY, new Date(f.lastModified()));
                        item.lastModifyTime = f.lastModified();
                        item.type = FileItem.TYPE_TARGET_FOLDER;
                        File[] fs = f.listFiles(mFileFilter);
                        if (fs != null) {
                            item.fileCount = fs.length;
                        } else {
                            item.fileCount = 0;
                        }
                        if (AppStorageManager.getInstance(mContext).checkStorageCanWrite(f.getPath()))
                            mFileItems.add(item);
                    }
                    return;
                }
                setButtonEnable(true, MatchDialog.DIALOG_OK);
                mPathLayout.setVisibility(View.VISIBLE);
                mFileItems.clear();
                File file = new File(path);
                mNoContentView.setVisibility(View.VISIBLE);
                mNoContentIv.setVisibility(View.VISIBLE);
                if (!file.exists()) return;
                mBackTextView.setText(file.getName());
                File[] files =  AppFileUtil.listFiles(mContext, file, mFileFilter);
                mNoContentView.setVisibility(files.length == 0 ? View.VISIBLE : View.GONE);
                mNoContentIv.setVisibility(files.length == 0 ? View.VISIBLE : View.GONE);
                for (File f : files) {
                    FileItem item = new FileItem();
                    item.parentPath = path;
                    item.path = f.getPath();
                    item.name = f.getName();
                    item.size = AppFileUtil.formatFileSize(f.length());
                    item.date = AppDmUtil.formatJavaDate(AppDmUtil.FORMAT_MMM_DD_YYYY, new Date(f.lastModified()));
                    if (f.isFile()) {
                        item.type = FileItem.TYPE_TARGET_FILE;
                    } else {
                        item.type = FileItem.TYPE_TARGET_FOLDER;
                        File[] childFiles = f.listFiles(mFileFilter);
                        item.fileCount = childFiles == null ? 0 : childFiles.length;
                    }
                    item.length = f.length();
                    mFileItems.add(item);
                }
                Collections.sort(mFileItems, mFileBrowser.getComparator());
                if (AppFileUtil.needScopedStorageAdaptation() && mFileItems.size() > 0) {
                    if (mFileBrowser != null){
                        mFileBrowser.loadDocumentTreeChildCount(mFileItems, mFileFilter, new Consumer<List<Integer>>() {
                            @Override
                            public void accept(List<Integer> result) {
                                if (mFileBrowser != null && result != null && result.size() == mFileItems.size()){
                                    for (int i = 0; i < mFileItems.size(); i++) {
                                        mFileItems.get(i).fileCount = result.get(i);
                                    }
                                    mFileBrowser.updateDataSource(true);
                                }
                            }
                        });
                    }
                }
            }

            @Override
            public void onItemClicked(View view, FileItem item) {
                if ((item.type & FileItem.TYPE_FILE) > 0) return;
                if (!AppStorageManager.getInstance(mActivity).checkCallDocumentTreeUriPermission(item.path, true)){
                    return;
                }
                mCurrentItem = item;
                mFileBrowser.setPath(item.path);
            }

            @Override
            public void onItemsCheckedChanged(boolean isAllSelected, int folderCount, int fileCount) {

            }

        });
        mFileBrowserView.addView(mFileBrowser.getContentView());
        mCurrentItem = new FileItem();
        mFileBrowser.hideDivider();
        mFileBrowser.showFileCount(true);
        mCurrentItem.path = AppFileUtil.getSDPath();
        setContentView(mContentView);

        setBackButtonVisible(View.VISIBLE);
        setBackButtonStyle(TEXT_BACK);
        setRightButtonVisible(View.VISIBLE);
        setRightButtonText(AppResource.getString(mContext, R.string.fx_string_done));
        setTitlePosition(BaseBar.TB_Position.Position_CENTER);
        setTitleTextSize(TypedValue.COMPLEX_UNIT_PX, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_text_size_16sp));
        setTitle(AppResource.getString(mContext, R.string.fx_select_folder));
        setStyle(DLG_TITLE_STYLE_BG_DEFAULT);
        return mContentView;
    }

    private void setCurrentPath(String path) {
        if (mFileBrowser != null){
            if (mCurrentItem == null){
                mCurrentItem = new FileItem();
            }
            mCurrentItem.path = TextUtils.isEmpty(path) ? AppFileUtil.getSDPath() : path;
            mFileBrowser.setPath(path);
        }
    }

    public String getCurrentPath() {
        return mCurrentItem.path;
    }

    private FileFilter mFileFilter = new FileFilter() {
        @Override
        public boolean accept(File pathname) {
            return !pathname.isHidden() && AppFileUtil.canRead(pathname);
        }
    };

    public void setFileFilter(FileFilter fileFilter) {
        if (fileFilter != null)
            mFileFilter = fileFilter;
        setCurrentPath(AppFileUtil.getSDPath());
    }

    @Override
    public void showDialog(boolean showMask) {
        super.showDialog(showMask);
//        setDarkModeColor();
    }

    @Override
    public void showDialog() {
        super.showDialog();
//        setDarkModeColor();
    }

//    public void setDarkModeColor() {
//        super.setDarkModeColor();
//        mContentView.setBackgroundColor(mContext.getResources().getColor(R.color.b1));
//        mRollBackIv.setColorFilter(mContext.getResources().getColor(R.color.i3));
//        mBackTextView.setTextColor(mContext.getResources().getColor(R.color.t4));
//        mNoContentView.setTextColor(mContext.getResources().getColor(R.color.t2));
//        mFileBrowserView.setBackgroundColor(mContext.getResources().getColor(R.color.b1));
//        notifyDataSetChanged();
//    }

    public void notifyDataSetChanged() {
        mFileBrowser.notifyDataSetChanged();
    }

}
