/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.dialog.fileselect;

import android.content.Context;
import android.util.TypedValue;
import android.view.View;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

import androidx.appcompat.widget.AppCompatTextView;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.controls.dialog.MatchDialog;
import com.foxit.uiextensions.controls.dialog.UIMatchDialog;
import com.foxit.uiextensions.controls.filebrowser.FileBrowser;
import com.foxit.uiextensions.controls.filebrowser.FileDelegate;
import com.foxit.uiextensions.controls.filebrowser.imp.FileBrowserImpl;
import com.foxit.uiextensions.controls.filebrowser.imp.FileItem;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppStorageManager;
import com.foxit.uiextensions.utils.AppUtil;

import java.io.File;
import java.io.FileFilter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import io.reactivex.functions.Consumer;


public class UIFileSelectDialog extends UIMatchDialog {
    private FileBrowserImpl mFileBrowser;
    private RelativeLayout mContentView;
    private List<FileItem> mFileItems = new ArrayList<FileItem>();
    private FileItem mCurrentItem;
    private LinearLayout mPathLayout;
    private OnFileClickedListener mListener = null;
    private AppCompatTextView mNoContentView;
    private TextView mBackTextView;
    private RelativeLayout mFileBrowserView;
    private ImageView mBackIv;
    private ImageView mNoContentIv;


    public UIFileSelectDialog(Context context) {
        super(context, true);
        onCreateView(true);
    }

    public UIFileSelectDialog(Context context, OnFileClickedListener listener) {
        super(context, true);
        onCreateView(false);
        mListener = listener;
    }

    private View onCreateView(boolean editState) {
        mContentView = (RelativeLayout) View.inflate(mContext, R.layout.cloud_select_file, null);
        mFileBrowserView = mContentView.findViewById(R.id.select_file_file_browser);
        mPathLayout = (LinearLayout) mContentView.findViewById(R.id.select_file_path);
        mBackIv = mPathLayout.findViewById(R.id.select_file_rollback_iv);
        ThemeUtil.setTintList(mBackIv, ThemeUtil.getPrimaryIconColor(mContext));
        mBackTextView = mPathLayout.findViewById(R.id.select_file_rollback_tv);
        mNoContentView = mContentView.findViewById(R.id.tv_no_content_tips);
        mNoContentIv = mContentView.findViewById(R.id.select_file_no_info_iv);
        mNoContentIv.setColorFilter(ThemeConfig.getInstance(mContext).getPrimaryColor());
        mNoContentView.setText(R.string.menu_more_no_files_yet);
        mPathLayout.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                String path = mFileBrowser.getDisplayPath();
                if (path == null || path.length() == 0) return;
                List<String> roots = AppStorageManager.getInstance(mContext).getVolumePaths();
                if (roots.contains(path)) {
                    mFileBrowser.setPath(null);
                    return;
                }
                int lastIndex = path.lastIndexOf(File.separator);
                if (lastIndex != -1) {
                    path = path.substring(0, lastIndex);
                    mFileBrowser.setPath(path);
                    return;
                }
                mFileBrowser.setPath(null);
            }
        });

        mFileBrowser = new FileBrowserImpl(mContext, new FileDelegate() {
            @Override
            public List<FileItem> getDataSource() {
                return mFileItems;
            }

            @Override
            public void onPathChanged(String path) {
                if (!mFileBrowser.isOnlyOneSelect()) {
                    clearCheckedItems();
                    notifyDataSetChanged();
                }

                if (AppUtil.isEmpty(path)) {
                    mPathLayout.setVisibility(View.GONE);
                    mFileItems.clear();
                    List<String> paths = AppStorageManager.getInstance(mContext).getVolumePaths();
                    mNoContentView.setVisibility(paths.size() == 0 ? View.VISIBLE : View.GONE);
                    mNoContentIv.setVisibility(paths.size() == 0 ? View.VISIBLE : View.GONE);
                    for (String p : paths) {
                        File f = new File(p);
                        FileItem item = new FileItem();
                        item.parentPath = path;
                        item.path = f.getPath();
                        item.name = f.getName();
                        item.date = AppDmUtil.formatJavaDate(AppDmUtil.FORMAT_MMM_DD_YYYY, new Date(f.lastModified()));
                        item.lastModifyTime = f.lastModified();
                        item.type = FileItem.TYPE_FOLDER;
                        File[] fs = f.listFiles(mFileFilter);
                        if (fs != null) {
                            item.fileCount = fs.length;
                        } else {
                            item.fileCount = 0;
                        }
                        mFileItems.add(item);
                    }
                    return;
                }
                mPathLayout.setVisibility(View.VISIBLE);
                mFileItems.clear();
                File file = new File(path);
                mContentView.setVisibility(View.VISIBLE);
                if (!file.exists()) return;
                mBackTextView.setText(file.getName());
                File[] files = AppFileUtil.listFiles(mContext, file, mFileFilter);
                if (files == null) return;
                mNoContentView.setVisibility(files.length == 0 ? View.VISIBLE : View.GONE);
                mNoContentIv.setVisibility(files.length == 0 ? View.VISIBLE : View.GONE);
                for (File f : files) {
                    FileItem item = new FileItem();
                    item.parentPath = path;
                    item.path = f.getPath();
                    item.name = f.getName();
                    item.length = f.length();
                    item.size = AppFileUtil.formatFileSize(f.length());
                    item.date = AppDmUtil.formatJavaDate(AppDmUtil.FORMAT_MMM_DD_YYYY, new Date(f.lastModified()));
                    item.lastModifyTime = f.lastModified();
                    if (f.isFile()) {
                        item.type = FileItem.TYPE_FILE;
                        if (mFileBrowser.getCheckedItems().size() > 0) {
                            List<FileItem> checkItems = mFileBrowser.getCheckedItems();
                            for (int i = 0; i < checkItems.size(); i++) {
                                FileItem checkItem = checkItems.get(i);
                                if (item.path.equalsIgnoreCase(checkItem.path)) {
                                    item.checked = checkItem.checked;
                                    checkItems.remove(checkItem);
                                    checkItems.add(i, item);
                                    break;
                                }
                            }
                        }
                    } else {
                        item.type = FileItem.TYPE_FOLDER;
                        File[] childFiles = f.listFiles(mFileFilter);
                        item.fileCount = childFiles == null ? 0 : childFiles.length;
                    }
                    mFileItems.add(item);
                }
                Collections.sort(mFileItems, mFileBrowser.getComparator());
                if (AppFileUtil.needScopedStorageAdaptation() && mFileItems.size() > 0) {
                    if (mFileBrowser != null){
                        mFileBrowser.loadDocumentTreeChildCount(mFileItems, mFileFilter,new Consumer<List<Integer>>() {
                            @Override
                            public void accept(List<Integer> result) {
                                if (mFileBrowser != null && result != null && result.size() == mFileItems.size()){
                                    for (int i = 0; i < mFileItems.size(); i++) {
                                        mFileItems.get(i).fileCount = result.get(i);
                                    }
                                    mFileBrowser.updateDataSource(true);
                                }
                            }
                        });
                    }
                }
            }

            @Override
            public void onItemClicked(View view, FileItem item) {
                mCurrentItem = item;
                if (mListener != null && item.type == FileItem.TYPE_FILE) {
                    UIFileSelectDialog.this.dismiss();
                    mListener.onFileClicked(item.path);
                } else {
                    if (!AppStorageManager.getInstance(mActivity).checkCallDocumentTreeUriPermission(item.path, true)){
                        return;
                    }
                    mFileBrowser.setPath(item.path);
                }
            }

            @Override
            public void onItemsCheckedChanged(boolean isAllSelected, int folderCount, int fileCount) {
                if (fileCount > 0) {
                    setRightButtonEnable(true);
                    setButtonEnable(true, MatchDialog.DIALOG_OK);
                    setButtonEnable(true, MatchDialog.DIALOG_UPLOAD);
                    setTitle(AppResource.getString(mContext, R.string.fx_selected_count, fileCount));
                } else {
                    setRightButtonEnable(false);
                    setButtonEnable(false, MatchDialog.DIALOG_OK);
                    setButtonEnable(false, MatchDialog.DIALOG_UPLOAD);
                    setTitle(AppResource.getString(mContext, R.string.fx_select_file));
                }
            }
        });
        setButtonEnable(false, MatchDialog.DIALOG_OK);
        setButtonEnable(false, MatchDialog.DIALOG_UPLOAD);
        mFileBrowser.setEditState(editState);
        mFileBrowser.showCheckBox(editState);
        mFileBrowser.showFileCount(true);
        mFileBrowser.hideDivider();
        mFileBrowserView.addView(mFileBrowser.getContentView());
        mCurrentItem = new FileItem();
        mCurrentItem.path = null;
        setContentView(mContentView);

        setBackButtonVisible(View.VISIBLE);
        setBackButtonStyle(TEXT_BACK);
        setRightButtonEnable(false);
        setRightButtonText(AppResource.getString(mContext, R.string.fx_string_done));
        setRightButtonVisible(editState ? View.VISIBLE : View.GONE);
        setTitlePosition(BaseBar.TB_Position.Position_CENTER);
        setTitleTextSize(TypedValue.COMPLEX_UNIT_PX, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_text_size_18sp));
        setTitle(AppResource.getString(mContext, R.string.fx_select_file));
        setStyle(DLG_TITLE_STYLE_BG_DEFAULT);
        return mContentView;
    }
//    public void setDarkModeColor() {
//        super.setDarkModeColor();
//        mContentView.setBackgroundColor(mContext.getResources().getColor(R.color.b1));
//        mBackTextView.setTextColor(mContext.getResources().getColor(R.color.t4));
//        mNoContentView.setTextColor(mContext.getResources().getColor(R.color.t2));
//        ThemeUtil.setTintList(mBackIv, ThemeUtil.getPrimaryIconColor(mContext));
//        mFileBrowserView.setBackgroundColor(mContext.getResources().getColor(R.color.b1));
//        notifyDataSetChanged();
//    }

//    private void setTintForNoContentView() {
//        TextView noContentTextView = (TextView) mNoContentView;
//        Drawable drawable = AppResource.createTintedDrawable(AppResource.getDrawable(mContext,R.drawable.ic_no_file_found), ThemeUtil.getPrimaryTextColor(mContext));
//        if (drawable == null)return;
//        drawable.setBounds(0,0,drawable.getMinimumWidth(),drawable.getMinimumHeight());
//        TextViewCompat.setCompoundDrawablesRelative(noContentTextView,null, drawable, null, null);
//    }

    public void notifyDataSetChanged() {
        mFileBrowser.notifyDataSetChanged();
    }

    public List<FileItem> getSelectedFiles() {
        return mFileBrowser.getCheckedItems();
    }

    private FileFilter mFileFilter = new FileFilter() {
        @Override
        public boolean accept(File pathname) {
            return !pathname.isHidden() && AppFileUtil.canRead(pathname);
        }
    };

    @Override
    public void showDialog() {
//        setTintForNoContentView();
        if (mFileBrowser.isOnlyOneSelect() && mFileBrowser.getCheckedItems().size() > 0) {
            clearCheckedItems();
            notifyDataSetChanged();
        }
        super.showDialog();
//        setDarkModeColor();
    }

    @Override
    public void showDialog(boolean showMask) {
//        setTintForNoContentView();
        if (mFileBrowser.isOnlyOneSelect() && mFileBrowser.getCheckedItems().size() > 0) {
            clearCheckedItems();
            notifyDataSetChanged();
        }
        super.showDialog(showMask);

//        setDarkModeColor();
    }
    public void init(FileFilter fileFilter, boolean singleChoice) {
        if (fileFilter != null)
            mFileFilter = fileFilter;
        mFileBrowser.setPath(AppFileUtil.getSDPath());
        mFileBrowser.setOnlyOneSelect(singleChoice);
    }

    public void setFileClickedListener(OnFileClickedListener listener) {
        mListener = listener;
        if (mListener != null) {
            mFileBrowser.setEditState(false);
            mFileBrowser.showCheckBox(false);
            setRightButtonVisible(View.GONE);
        }
    }

    public void clearCheckedItems() {
        mFileBrowser.clearCheckedItems();
    }

    public void setFileLimitListener(FileBrowser.FileLimitListener limitListener) {
        mFileBrowser.setFileLimitListener(limitListener);
    }
}
