/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.dialog;


import android.app.Activity;
import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.text.Editable;
import android.text.TextWatcher;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.view.inputmethod.EditorInfo;
import android.widget.Button;
import android.widget.EditText;
import android.widget.TextView;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppTheme;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.SystemUiHelper;

public class UITextEditDialogV {

    Activity mAct;
    private Dialog mDialog;
    private IDialogListener mDialogListener;
    private ITextWatcher mTextChangedListener;

    private EditText mInputEditText;
    private TextView mTitleView;
    private Button mOkButton;
    private Button mCancelButton;
    private String mOriginContent;
    private boolean mCanEdit = true;

    public UITextEditDialogV(Context context) {
        mAct = (Activity) context;

        View contentView = View.inflate(context.getApplicationContext(), R.layout.rd_note_dialog_edit, null);
        mTitleView = (TextView) contentView.findViewById(R.id.rd_note_dialog_edit_title);
        mInputEditText = (EditText) contentView.findViewById(R.id.rd_note_dialog_edit);
        if (AppDisplay.isPad()) { // SDKRD-9313
            mInputEditText.setImeOptions(EditorInfo.IME_FLAG_NO_EXTRACT_UI);
        }
        mCancelButton = (Button) contentView.findViewById(R.id.rd_note_dialog_edit_cancel);
        mOkButton = (Button) contentView.findViewById(R.id.rd_note_dialog_edit_ok);
        mInputEditText.setMaxLines(10);
        mInputEditText.addTextChangedListener(new TextWatcher() {
            @Override
            public void beforeTextChanged(CharSequence s, int start, int count, int after) {
                if (mTextChangedListener != null)
                    mTextChangedListener.beforeTextChanged(s, start, count, after);
            }

            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {
                if (mTextChangedListener != null)
                    mTextChangedListener.onTextChanged(s, start, before, count);
            }

            @Override
            public void afterTextChanged(Editable s) {
                if (mTextChangedListener != null && mTextChangedListener.afterTextChanged(s))
                    return;
                mOkButton.setEnabled(!s.toString().equals(mOriginContent));
            }
        });

        contentView.setLayoutParams(new ViewGroup.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.WRAP_CONTENT));
        mDialog = new Dialog(context, AppTheme.getDialogTheme());
        mDialog.setContentView(contentView,
                new ViewGroup.LayoutParams(AppDisplay.getUITextEditDialogWidth(), ViewGroup.LayoutParams.WRAP_CONTENT));
        if (mDialog.getWindow() != null) {
            mDialog.getWindow().setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN, WindowManager.LayoutParams.FLAG_FULLSCREEN);
            mDialog.getWindow().setBackgroundDrawableResource(R.drawable.dlg_title_bg_4circle_corner_white);
        }
        mDialog.setOnDismissListener(new DialogInterface.OnDismissListener() {
            @Override
            public void onDismiss(DialogInterface dialog) {
                if (mDialogListener != null)
                    mDialogListener.onDismiss();
                if (SystemUiHelper.getInstance().isFullScreen()
                        && SystemUiHelper.getInstance().isAllowedHideSystemUI()){
                    if (AppDisplay.isPad())
                        SystemUiHelper.getInstance().hideSystemUI(mAct);
                    else
                        SystemUiHelper.getInstance().hideStatusBar(mAct);
                }
            }
        });

        mCancelButton.setOnClickListener(new View.OnClickListener() {

            @Override
            public void onClick(View v) {
                mDialog.dismiss();
                AppUtil.dismissInputSoft(mInputEditText);

                if (mDialogListener != null)
                    mDialogListener.onNegativeButtonCallback();
            }
        });

        mOkButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mDialog.dismiss();
                AppUtil.dismissInputSoft(mInputEditText);
                if (mDialogListener != null)
                    mDialogListener.onPositiveButtonCallback(mInputEditText.getText().toString());
            }
        });
    }

    public void setTitle(String title) {
        mTitleView.setText(title);
    }

    public void setContent(String content) {
        mOriginContent = content == null ? "" : content;
        mInputEditText.setText(content);
    }

    public void show() {
        if (mDialog != null && !mDialog.isShowing()) {
            if (!mCanEdit) {
                mInputEditText.setEnabled(false);
                mOkButton.setEnabled(false);
            } else {
                AppUtil.showSoftInput(mInputEditText);
            }
            mDialog.show();
        }
    }

    public boolean isShowing() {
        return mDialog != null && mDialog.isShowing();
    }

    public void dismiss() {
        if (mDialog != null && mDialog.isShowing()) {
            mDialog.dismiss();
        }
    }

    public void setDialogListener(IDialogListener dialogListener) {
        mDialogListener = dialogListener;
    }

    public EditText getInputEditText() {
        return mInputEditText;
    }

    public Button getPositiveButton() {
        return mOkButton;
    }

    public Button getNegativeButton() {
        return mCancelButton;
    }

    public void setCanEdit(boolean canEdit) {
        mCanEdit = canEdit;
    }

    public interface IDialogListener {
        void onDismiss();

        void onPositiveButtonCallback(String editContent);

        void onNegativeButtonCallback();
    }

    public void setTextChangedListener(ITextWatcher textChangedListener) {
        mTextChangedListener = textChangedListener;
    }

    public interface ITextWatcher {
        boolean beforeTextChanged(CharSequence s, int start, int count, int after);

        boolean onTextChanged(CharSequence s, int start, int before, int count);

        boolean afterTextChanged(Editable s);
    }

}
